﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Text;
using System.Windows.Forms;
using Peak.Can.Basic;
using Peak.Can.Ccp;

using TCCPHandle = System.UInt32;
using TPCANHandle = System.UInt16;

namespace CCPDemo
{
    public partial class Form1 : Form
    {
        private TCCPHandle m_PccpHandle;
        private TPCANHandle m_Channel;
        private TPCANBaudrate m_Baudrate;
        private TCCPSlaveData m_SlaveData;
        private TCCPExchangeData m_ExchangeData;

        public Form1()
        {
            InitializeComponent();

            m_PccpHandle = 0;

            // PCAN Hardware to use
            //
            m_Channel = PCANBasic.PCAN_USBBUS1;
            m_Baudrate = TPCANBaudrate.PCAN_BAUD_500K;

            // ECU Data
            //
            m_SlaveData = new TCCPSlaveData();
            m_SlaveData.EcuAddress = 0x10;
            m_SlaveData.IdCRO = 0x200;
            m_SlaveData.IdDTO = 0x100;
            m_SlaveData.IntelFormat = true;
        }

        private string GetErrorText(TCCPResult errorCode)
        {
            StringBuilder strText = new StringBuilder(256);

            if (CCPApi.GetErrorText(errorCode, strText) == TCCPResult.CCP_ERROR_ACKNOWLEDGE_OK)
                return strText.ToString();
            return String.Empty;
        }

        private void Form1_Load(object sender, EventArgs e)
        {
            TCCPResult result;

            result = CCPApi.InitializeChannel(m_Channel, m_Baudrate);
            if (result != TCCPResult.CCP_ERROR_ACKNOWLEDGE_OK)
            {
                MessageBox.Show("Error: " + GetErrorText(result), "Error");
                btnConnect.Enabled = false;
                btnTest.Enabled = false;
            }
        }

        private void Form1_FormClosing(object sender, FormClosingEventArgs e)
        {
            CCPApi.UninitializeChannel(m_Channel);
        }

        private void btnConnect_Click(object sender, EventArgs e)
        {
            TCCPResult result;
            bool bConnected;

            result = CCPApi.Connect(m_Channel, ref m_SlaveData, out m_PccpHandle, 1000);
            bConnected = (result == TCCPResult.CCP_ERROR_ACKNOWLEDGE_OK);

            btnConnect.Enabled = !bConnected;
            btnDisconnect.Enabled = bConnected;
            btnGetVersion.Enabled = bConnected;
            btnExchangeId.Enabled = bConnected;
            btnSlaveId.Enabled = false;

            if (!bConnected)
                MessageBox.Show("Error: " + GetErrorText(result), "Error");

        }

        private void btnDisconnect_Click(object sender, EventArgs e)
        {
            TCCPResult result;

            result = CCPApi.Disconnect(m_PccpHandle, false, 0);
            m_PccpHandle = 0;

            btnConnect.Enabled = true;
            btnDisconnect.Enabled = false;
            btnGetVersion.Enabled = false;
            btnExchangeId.Enabled = false;
            btnSlaveId.Enabled = false;

            if (result != TCCPResult.CCP_ERROR_ACKNOWLEDGE_OK)
                MessageBox.Show("Error: " + GetErrorText(result), "Error");

            laVersion.Text = "0.0";
            laSlaveId.Text = "_________";
            laSlaveInfo.Text = "_________";
        }

        private void btnTest_Click(object sender, EventArgs e)
        {
            TCCPResult result;

            result = CCPApi.Test(m_Channel, ref m_SlaveData, 0);

            if(result == TCCPResult.CCP_ERROR_ACKNOWLEDGE_OK)
                MessageBox.Show("ECU is available for connect");
            else if(result == TCCPResult.CCP_ERROR_INTERNAL_TIMEOUT)
                MessageBox.Show("ECU NOT available");
            else
                MessageBox.Show("Error: " + GetErrorText(result),"Error");
        }

        private void btnGetVersion_Click(object sender, EventArgs e)
        {
            TCCPResult result;
            byte main, release;

            main = release = 0;
            result = CCPApi.GetCcpVersion(m_PccpHandle, ref main, ref release, 0);

            if (result != TCCPResult.CCP_ERROR_ACKNOWLEDGE_OK)
                MessageBox.Show("Error: " + GetErrorText(result), "Error");
            else
                laVersion.Text = string.Format("{0}.{1}", main, release);
        }

        private void btnExchangeId_Click(object sender, EventArgs e)
        {
            TCCPResult result;
            byte[] masterData;

            masterData = new byte[4] { 0x78, 0x56, 0x34, 0x12 };
            result = CCPApi.ExchangeId(m_PccpHandle, ref m_ExchangeData, masterData, masterData.Length, 0);

            if (result != TCCPResult.CCP_ERROR_ACKNOWLEDGE_OK)
                MessageBox.Show("Error: " + GetErrorText(result), "Error");
            else
            {
                laSlaveInfo.Text = string.Format("ID Length {0}. Data Type {1}. Res. Mask {2}. Protec. Mask {3}.", new object[] { m_ExchangeData.IdLength, m_ExchangeData.DataType, m_ExchangeData.AvailabilityMask, m_ExchangeData.ProtectionMask });
                btnSlaveId.Enabled = true;
            }
        }

        private void btnSlaveId_Click(object sender, EventArgs e)
        {
            TCCPResult result;
            byte[] IdArray;

            IdArray = new byte[m_ExchangeData.IdLength];
            result = CCPApi.Upload(m_PccpHandle, m_ExchangeData.IdLength, IdArray, 0);

            if (result != TCCPResult.CCP_ERROR_ACKNOWLEDGE_OK)
                MessageBox.Show("Error: " + GetErrorText(result), "Error");
            else
                switch (m_ExchangeData.IdLength)
                {
                    case 1:
                        laSlaveId.Text = IdArray[0].ToString("X2");
                        break;
                    case 2:
                        laSlaveId.Text = BitConverter.ToUInt16(IdArray, 0).ToString("X4");                        
                        break;
                    case 4:
                        laSlaveId.Text = BitConverter.ToUInt32(IdArray, 0).ToString("X8");
                        break;
                    default:
                        laSlaveId.Text = "";
                        for (int i = 0; i < m_ExchangeData.IdLength; i++)
                            laSlaveId.Text = IdArray[i].ToString("X2") + laSlaveId.Text;
                        break;                        
                }
        }
    }
}
