﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;


using Peak.Can.IsoTp;
using cantp_timestamp = System.UInt64;  // Note : must repeat it even if already defined in PCAN-ISO-TP_2016.cs

// Note : this example uses unsafe mode, it means that 
// "allow unsafe mode" must be checked in the Visual Studio project,
// and that 
// "#define UNSAFE" must be uncommented at the beginning of the file "PCAN-ISO-TP_2016.cs", or that UNSAFE must be defined at the project level.

namespace _01_server_ISO15765_2_normal_addressing
{
    class Program
    {
        const UInt32 STMIN_600US = 0xF6;

        /// <summary>Entry point of the program, start a small CAN ISO TP read/write example</summary>
        static void Main(string[] args)
        {
            // Local variables
            cantp_status res = new cantp_status();
            cantp_status read_status = new cantp_status();
            StringBuilder buffer = new StringBuilder(500);
            UInt32 STmin;
            cantp_msg rx_msg = new cantp_msg();
            cantp_mapping mapping = new cantp_mapping();
            cantp_handle server_handle = new cantp_handle();

            // Initialize variables
            server_handle = cantp_handle.PCANTP_HANDLE_USBBUS2;  // TODO: modify the value according to your available PCAN devices.

            // Print version informations
            CanTpApi.GetValue_2016(cantp_handle.PCANTP_HANDLE_NONEBUS, cantp_parameter.PCANTP_PARAMETER_API_VERSION, buffer, 500);
            Console.WriteLine("PCAN-ISO-TP API Version : {0}", buffer);

            // Initialize channel: CAN2.0 - 500Kbit/s
            res = CanTpApi.Initialize_2016(server_handle, cantp_baudrate.PCANTP_BAUDRATE_500K);
            Console.WriteLine("Initialize : {0}", STATUS_OK_KO(res));

            // Create and set a receive event on receiver
            System.Threading.AutoResetEvent receive_event = new System.Threading.AutoResetEvent(false);
            if (IntPtr.Size == 4)
            {
                UInt32 iBuffer = Convert.ToUInt32(receive_event.SafeWaitHandle.DangerousGetHandle().ToInt32());
                res = CanTpApi.SetValue_2016(server_handle, cantp_parameter.PCANTP_PARAMETER_RECEIVE_EVENT, ref iBuffer, sizeof(UInt32));
            }
            else if (IntPtr.Size == 8)
            {
                Int64 iBuffer = receive_event.SafeWaitHandle.DangerousGetHandle().ToInt64();
                byte[] byteArray = BitConverter.GetBytes(iBuffer);
                res = CanTpApi.SetValue_2016(server_handle, cantp_parameter.PCANTP_PARAMETER_RECEIVE_EVENT, byteArray, sizeof(UInt64));
            }
            Console.WriteLine("Set receive event : {0}", STATUS_OK_KO(res));

            // Change STmin value to 600us
            STmin = STMIN_600US;
            res = CanTpApi.SetValue_2016(server_handle, cantp_parameter.PCANTP_PARAMETER_SEPARATION_TIME, ref STmin, sizeof(UInt32));
            Console.WriteLine("Set STMIN = 600us : {0}", STATUS_OK_KO(res));

            // Create a simple physical mapping: 
            //    - Source 0xF1 (client), target 0x01 (server), CAN id 0xA1, CAN ID flow control 0xA2
            //    - Diagnostic message in a classic format
            mapping.can_id = 0xA1;
            mapping.can_id_flow_ctrl = 0xA2;
            mapping.can_msgtype = cantp_can_msgtype.PCANTP_CAN_MSGTYPE_STANDARD;
            mapping.netaddrinfo.extension_addr = 0x00;
            mapping.netaddrinfo.format = cantp_isotp_format.PCANTP_ISOTP_FORMAT_NORMAL;
            mapping.netaddrinfo.msgtype = cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC;
            mapping.netaddrinfo.source_addr = 0xF1;
            mapping.netaddrinfo.target_addr = 0x01;
            mapping.netaddrinfo.target_type = cantp_isotp_addressing.PCANTP_ISOTP_ADDRESSING_PHYSICAL;


            // Add mapping on channel
            res = CanTpApi.AddMapping_2016(server_handle, ref mapping);
            Console.WriteLine("Add a simple mapping : {0}", STATUS_OK_KO(res));
            // NOTE: this is a one-way mapping, the server is only capable of 
            //	receiving segmented or unsegmented ISO-TP message using "0xA1" CAN ID
            //	to be able to send ISO-TP message, another mapping is required.

            // Read while user do not press Q
            Console.WriteLine("Start listening, press Q to quit.");
            bool stop = false;
            while (!stop)
            {
                // Wait a receive event on receiver
                //	note: timeout is used to check keyboard hit.
                bool wait_result = receive_event.WaitOne(1000);

                // If we receive something, read the message
                if (wait_result)
                {
                    do
                    {
                        // Allocate rx message
                        res = CanTpApi.MsgDataAlloc_2016(out rx_msg, cantp_msgtype.PCANTP_MSGTYPE_NONE);
                        Console.WriteLine("Allocate rx message : {0}", STATUS_OK_KO(res));

                        // Read first available message (no filtering based on message's type is set): 
                        cantp_timestamp ts;
                        read_status = CanTpApi.Read_2016(server_handle, out rx_msg, out ts);
                        Console.WriteLine("Try to read a message : {0}", STATUS_OK_KO(read_status));

                        // If we read something, print the message
                        if (CanTpApi.StatusIsOk_2016(read_status))
                        {
                            Console.WriteLine();
                            Console.WriteLine("Print received data in unsafe mode:");
                            print_data(rx_msg, true);
                            Console.WriteLine();
                            Console.WriteLine("Print received data in safe mode:");
                            print_data(rx_msg, false);
                            Console.WriteLine();
                        }

                        // Free message
                        res = CanTpApi.MsgDataFree_2016(ref rx_msg);
                        Console.WriteLine("Free rx message : {0}", STATUS_OK_KO(res));
                    }
                    while (!CanTpApi.StatusIsOk_2016(read_status, cantp_status.PCANTP_STATUS_NO_MESSAGE));
                }

                // Quit when user press Q
                if (Console.KeyAvailable)
                {
                    char keyboard_res = Console.ReadKey().KeyChar;
                    if (keyboard_res == 'Q' || keyboard_res == 'q')
                    {
                        stop = true;
                    }
                }
            }

            // Close receive event
            if (IntPtr.Size == 4)
            {
                UInt32 iBuffer = 0;
                res = CanTpApi.SetValue_2016(server_handle, cantp_parameter.PCANTP_PARAMETER_RECEIVE_EVENT, ref iBuffer, sizeof(UInt32));
            }
            else if (IntPtr.Size == 8)
            {
                Int64 iBuffer = 0;
                byte[] byteArray = BitConverter.GetBytes(iBuffer);
                res = CanTpApi.SetValue_2016(server_handle, cantp_parameter.PCANTP_PARAMETER_RECEIVE_EVENT, byteArray, sizeof(UInt64));
            }
            Console.WriteLine("Stop receive event  : {0}", STATUS_OK_KO(res));
            receive_event.Close();

            // Uninitialize
            res = CanTpApi.Uninitialize_2016(server_handle);
            Console.WriteLine("Uninitialize : {0}", STATUS_OK_KO(res));

            // Exit
            Console.WriteLine("Press any key to exit...");
            Console.In.Read();
        }

        static String OK_KO(bool test)
        {
            if (test)
                return "OK";
            return "KO";
        }

        static String STATUS_OK_KO(cantp_status test)
        {
            return OK_KO(CanTpApi.StatusIsOk_2016(test));
        }

        /// <summary>Helper: print message data</summary>
        /// <param name="msg">message to print</param>
        /// <param name="unsafeMode">safe or unsafe mode</param>
        unsafe static void print_data(cantp_msg msg, bool unsafeMode)
        {
            if (msg.Msgdata != IntPtr.Zero)
            {
                uint len = msg.Msgdata_any_Copy.length;
                if (len != 0)
                {
                    byte[] vals = new byte[len];
                    if (unsafeMode ?
                        CanTpApi.getData_unsafe_2016(ref msg, 0, vals, (int)len) :
                        CanTpApi.getData_2016(ref msg, 0, vals, (int)len))
                    {
                        Console.Write("\t Byte:");
                        for (uint i = 0; i < len; i++)
                        {
                            Console.Write(" 0x{0:X2}", vals[i]);
                        }
                        Console.WriteLine();
                        Console.Write("\t Ascii:\"");
                        for (uint i = 0; i < len; i++)
                        {
                            Console.Write("{0}", Convert.ToChar(vals[i]));
                        }
                        Console.WriteLine("\"");
                    }
                }
            }
        }
    }
}
