﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

using Peak.Can.IsoTp;
using cantp_timestamp = System.UInt64;  // Note : must repeat it even if already defined in PCAN-ISO-TP_2016.cs


namespace _04_client_ISO15765_2_fixed_normal_addressing
{
    class Program
    {
        const UInt32 STMIN_600US = 0xF6;
        const String DATA_UNSEGMENTED = "PEAK";
        const String DATA_SEGMENTED = "PEAK-System - PCAN-ISO-TP Sample";
        // TODO: Change MSG definition to perform UNSEGMENTED or SEGMENTED ISO-TP message
        const String MSG = DATA_UNSEGMENTED;
        /// <summary>
        /// Entry point of the program, start a small CAN ISO TP read/write example
        ///    - Extended 29bits CAN addressing
        ///    - No CANTP_AddMapping_2016() call (API automaticaly manage it)
        ///    - Set priority option 
        ///</summary>
        static void Main(string[] args)
        {
            // Local variables
            int MSG_SIZE = MSG.Length;
            cantp_status res = new cantp_status();
            StringBuilder buffer = new StringBuilder(500);
            UInt32 STmin;
            cantp_msg tx_msg = new cantp_msg();
            cantp_msg rx_msg = new cantp_msg();
            cantp_mapping mapping = new cantp_mapping();
            cantp_handle client_handle = new cantp_handle();

            // Initialize variables
            client_handle = cantp_handle.PCANTP_HANDLE_USBBUS1;  // TODO: modify the value according to your available PCAN devices.

            // Print version informations
            CanTpApi.GetValue_2016(cantp_handle.PCANTP_HANDLE_NONEBUS, cantp_parameter.PCANTP_PARAMETER_API_VERSION, buffer, 500);
            Console.WriteLine("PCAN-ISO-TP API Version : {0}", buffer);

            // Initialize channel: CAN2.0 - 500Kbit/s
            res = CanTpApi.Initialize_2016(client_handle, cantp_baudrate.PCANTP_BAUDRATE_500K);
            Console.WriteLine("Initialize : {0}", STATUS_OK_KO(res));

            // Change STmin value to 600us
            STmin = STMIN_600US;
            res = CanTpApi.SetValue_2016(client_handle, cantp_parameter.PCANTP_PARAMETER_SEPARATION_TIME, ref STmin, sizeof(UInt32));
            Console.WriteLine("Set STMIN = 600us : {0}", STATUS_OK_KO(res));

            // Allocate tx message
            res = CanTpApi.MsgDataAlloc_2016(out tx_msg, cantp_msgtype.PCANTP_MSGTYPE_ISOTP);
            Console.WriteLine("Allocate tx message : {0}", STATUS_OK_KO(res));

            // Create a simple physical mapping: 
            //    - Source 0xF1 (client), target 0x01 (server)
            //    - Diagnostic message in a fixed normal format
            mapping.can_msgtype = cantp_can_msgtype.PCANTP_CAN_MSGTYPE_EXTENDED;
            mapping.netaddrinfo.extension_addr = 0x00;
            mapping.netaddrinfo.format = cantp_isotp_format.PCANTP_ISOTP_FORMAT_FIXED_NORMAL;
            mapping.netaddrinfo.msgtype = cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC;
            mapping.netaddrinfo.source_addr = 0xF1;
            mapping.netaddrinfo.target_addr = 0x01;
            mapping.netaddrinfo.target_type = cantp_isotp_addressing.PCANTP_ISOTP_ADDRESSING_PHYSICAL;

            // No add mapping, because API automaticaly manage it 

            // Initialize Tx message
            byte[] bytes = Encoding.ASCII.GetBytes(MSG);
            res = CanTpApi.MsgDataInit_2016(out tx_msg, CanTpApi.PCANTP_CAN_ID_DEFINED_BY_NAI, mapping.can_msgtype, (uint)MSG_SIZE, bytes, ref mapping.netaddrinfo);
            Console.WriteLine("Initialize tx message : {0}", STATUS_OK_KO(res));

            // Set priority (add message options)
            res = CanTpApi.MsgDataInitOptions_2016(out tx_msg, 1);
            Console.WriteLine("Initialize message priority option : {0}", STATUS_OK_KO(res));
            if (CanTpApi.StatusIsOk_2016(res))
            {
                cantp_msgoption option = new cantp_msgoption();
                option.name = cantp_option.PCANTP_OPTION_J1939_PRIORITY;
                option.value = 0x02;
                if (CanTpApi.setOption_2016(ref tx_msg, 0, ref option))
                    Console.WriteLine("Set message priority option : {0}", "OK");
                else
                    Console.WriteLine("Set message priority option : {0}", "KO");
            }

            // Write message
            res = CanTpApi.Write_2016(client_handle, ref tx_msg);
            Console.WriteLine("Write \"{0}\" message : {1}", MSG, STATUS_OK_KO(res));

            // Sleep to let the transmission runs
            System.Threading.Thread.Sleep(100);

            if (MSG_SIZE > 7)
            {
                // Initialize Rx message
                res = CanTpApi.MsgDataAlloc_2016(out rx_msg, cantp_msgtype.PCANTP_MSGTYPE_NONE);
                Console.WriteLine("Initialize rx message : {0}", STATUS_OK_KO(res));
                // Read loopback indication ISO-TP message
                res = CanTpApi.Read_2016(client_handle, out rx_msg);
                Console.WriteLine("CANTP_Read_2016 (indication) : {0}, netstatus={1}", STATUS_OK_KO(res), getNetStatusAsHexaString(ref rx_msg));
                // Free message space
                res = CanTpApi.MsgDataFree_2016(ref rx_msg);
                Console.WriteLine("Free rx message : {0}", STATUS_OK_KO(res));
                // Sleep to let the transmission complete
                System.Threading.Thread.Sleep(1000);
            }

            // Initialize Rx message
            res = CanTpApi.MsgDataAlloc_2016(out rx_msg, cantp_msgtype.PCANTP_MSGTYPE_NONE);
            Console.WriteLine("Initialize rx message : {0}", STATUS_OK_KO(res));
            // Read ISO-TP message
            res = CanTpApi.Read_2016(client_handle, out rx_msg);
            Console.WriteLine("CANTP_Read_2016 (complete) : {0}, netstatus={1}", STATUS_OK_KO(res), getNetStatusAsHexaString(ref rx_msg));
            // Free message space
            res = CanTpApi.MsgDataFree_2016(ref rx_msg);
            Console.WriteLine("Free rx message : {0}", STATUS_OK_KO(res));

            // Free message space
            res = CanTpApi.MsgDataFree_2016(ref tx_msg);
            Console.WriteLine("Free tx message : {0}", STATUS_OK_KO(res));

            // Uninitialize
            res = CanTpApi.Uninitialize_2016(client_handle);
            Console.WriteLine("Uninitialize : {0}", STATUS_OK_KO(res));

            // Exit
            Console.WriteLine("Press any key to exit...");
            Console.In.Read();
        }

        static String OK_KO(bool test)
        {
            if (test)
                return "OK";
            return "KO";
        }

        static String STATUS_OK_KO(cantp_status test)
        {
            return OK_KO(CanTpApi.StatusIsOk_2016(test));
        }

        static String getNetStatusAsHexaString(ref cantp_msg msg)
        {
            cantp_netstatus netstatus = new cantp_netstatus();
            if (CanTpApi.getNetStatus_2016(ref msg, out netstatus))
                return netstatus.ToString("X");
            else
                return "Unknown";
        }
    }
}
