﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;


using Peak.Can.IsoTp;
using cantp_timestamp = System.UInt64;  // Note : must repeat it even if already defined in PCAN-ISO-TP_2016.cs

namespace _05_isotp_read_write
{
    class Program
    {
        const UInt32 STMIN_600US = 0xF6;
        /// <summary>Entry point of the program, start a small CAN ISO TP read/write example</summary> 
        static void Main(string[] args)
        {
            // Local variables
            cantp_status res = new cantp_status();
            StringBuilder buffer = new StringBuilder(500);
            UInt32 STmin;
            cantp_msg rx_msg = new cantp_msg();
            cantp_msg tx_msg = new cantp_msg();
            cantp_handle transmitter_handle = new cantp_handle();
            cantp_handle receiver_handle = new cantp_handle();
            cantp_mapping mapping = new cantp_mapping();

            // Initialize handles 
            transmitter_handle = cantp_handle.PCANTP_HANDLE_USBBUS1; // TODO: modify the value according to your available PCAN devices.
            receiver_handle = cantp_handle.PCANTP_HANDLE_USBBUS2; // TODO: modify the value according to your available PCAN devices.

            // Print version informations
            CanTpApi.GetValue_2016(cantp_handle.PCANTP_HANDLE_NONEBUS, cantp_parameter.PCANTP_PARAMETER_API_VERSION, buffer, 500);
            Console.WriteLine("PCAN-ISO-TP API Version : {0}", buffer);

            Console.WriteLine("\n");
            Console.WriteLine("WARNING: this sample is not suited for segmented communication.");
            Console.WriteLine("         Data must be contained in a Single ISO-TP frame only.");
            Console.WriteLine("         Please check sample \"06_isotp_segmented_read_write\" for standard segmented communications.");
            Console.WriteLine("\n");
            System.Threading.Thread.Sleep(2000);

            // Initialize channels: CAN2.0 - 500Kbit/s
            res = CanTpApi.Initialize_2016(transmitter_handle, cantp_baudrate.PCANTP_BAUDRATE_500K);
            Console.WriteLine("Initialize transmitter : {0}", STATUS_OK_KO(res));
            res = CanTpApi.Initialize_2016(receiver_handle, cantp_baudrate.PCANTP_BAUDRATE_500K);
            Console.WriteLine("Initialize receiver : {0}", STATUS_OK_KO(res));

            // Create and set a receive event on receiver
            System.Threading.AutoResetEvent receive_event = new System.Threading.AutoResetEvent(false);
            if (IntPtr.Size == 4)
            {
                UInt32 iBuffer = Convert.ToUInt32(receive_event.SafeWaitHandle.DangerousGetHandle().ToInt32());
                res = CanTpApi.SetValue_2016(receiver_handle, cantp_parameter.PCANTP_PARAMETER_RECEIVE_EVENT, ref iBuffer, sizeof(UInt32));
            }
            else if (IntPtr.Size == 8)
            {
                Int64 iBuffer = receive_event.SafeWaitHandle.DangerousGetHandle().ToInt64();
                byte[] byteArray = BitConverter.GetBytes(iBuffer);
                res = CanTpApi.SetValue_2016(receiver_handle, cantp_parameter.PCANTP_PARAMETER_RECEIVE_EVENT, byteArray, sizeof(UInt64));
            }
            Console.WriteLine("Set receive event on receiver : {0}", STATUS_OK_KO(res));

            // Change STmin value to 600us
            STmin = STMIN_600US;
            res = CanTpApi.SetValue_2016(transmitter_handle, cantp_parameter.PCANTP_PARAMETER_SEPARATION_TIME, ref STmin, sizeof(UInt32));
            Console.WriteLine("Set STMIN = 600us on transmitter : {0}", STATUS_OK_KO(res));
            res = CanTpApi.SetValue_2016(receiver_handle, cantp_parameter.PCANTP_PARAMETER_SEPARATION_TIME, ref STmin, sizeof(UInt32));
            Console.WriteLine("Set STMIN = 600us on receiver : {0}", STATUS_OK_KO(res));

            // Allocate tx message
            res = CanTpApi.MsgDataAlloc_2016(out tx_msg, cantp_msgtype.PCANTP_MSGTYPE_ISOTP);
            Console.WriteLine("Allocate tx message : {0}", STATUS_OK_KO(res));

            // Allocate rx message
            res = CanTpApi.MsgDataAlloc_2016(out rx_msg, cantp_msgtype.PCANTP_MSGTYPE_NONE);
            Console.WriteLine("Allocate rx message : {0}", STATUS_OK_KO(res));

            // Create a simple physical mapping: 
            //    - Source 0xF1 (transmitter), target 0x01 (receiver), CAN id 0xA1, CAN ID flow control 0xA2
            //    - Diagnostic message in a classic format
            mapping.can_id = 0xA1;
            mapping.can_id_flow_ctrl = 0xA2;
            mapping.can_msgtype = cantp_can_msgtype.PCANTP_CAN_MSGTYPE_STANDARD;
            mapping.netaddrinfo.extension_addr = 0x00;
            mapping.netaddrinfo.format = cantp_isotp_format.PCANTP_ISOTP_FORMAT_NORMAL;
            mapping.netaddrinfo.msgtype = cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC;
            mapping.netaddrinfo.source_addr = 0xF1;
            mapping.netaddrinfo.target_addr = 0x01;
            mapping.netaddrinfo.target_type = cantp_isotp_addressing.PCANTP_ISOTP_ADDRESSING_PHYSICAL;

            // Add mapping on channels
            res = CanTpApi.AddMapping_2016(transmitter_handle, ref mapping);
            Console.WriteLine("Add a simple mapping on transmitter : {0}", STATUS_OK_KO(res));
            res = CanTpApi.AddMapping_2016(receiver_handle, ref mapping);
            Console.WriteLine("Add a simple mapping on receiver : {0}", STATUS_OK_KO(res));

            // Initialize Tx message containing "42"
            byte[] data = { 0x42 };
            res = CanTpApi.MsgDataInit_2016(out tx_msg, CanTpApi.PCANTP_CAN_ID_DEFINED_BY_NAI, mapping.can_msgtype, 1, data, ref mapping.netaddrinfo);
            Console.WriteLine("Initialize tx message : {0}", STATUS_OK_KO(res));

            // Write "42" message
            res = CanTpApi.Write_2016(transmitter_handle, ref tx_msg);
            Console.WriteLine("Write \"42\" message : {0}", STATUS_OK_KO(res));

            // Wait a receive event on receiver
            bool wait_result = receive_event.WaitOne(100);
            Console.WriteLine("Wait a message on receiver : {0}", OK_KO(wait_result));

            // If we receive something on the receiver, read the message
            if (wait_result)
            {
                res = CanTpApi.Read_2016(receiver_handle, out rx_msg);
                Console.WriteLine("Read message on receiver : {0}", STATUS_OK_KO(res));
                byte val = 0;
                if (CanTpApi.getData_2016(ref rx_msg, 0, out val))
                    Console.WriteLine("Check if the message is \"42\" : {0}", OK_KO(val == 0x42));
                else
                    Console.WriteLine("Check if the message is \"42\" : NOK");
            }

            // Free messages
            res = CanTpApi.MsgDataFree_2016(ref rx_msg);
            Console.WriteLine("Free rx message : {0}", STATUS_OK_KO(res));
            res = CanTpApi.MsgDataFree_2016(ref tx_msg);
            Console.WriteLine("Free tx message : {0}", STATUS_OK_KO(res));

            // Close receive event
            if (IntPtr.Size == 4)
            {
                UInt32 iBuffer = 0;
                res = CanTpApi.SetValue_2016(receiver_handle, cantp_parameter.PCANTP_PARAMETER_RECEIVE_EVENT, ref iBuffer, sizeof(UInt32));
            }
            else if (IntPtr.Size == 8)
            {
                Int64 iBuffer = 0;
                byte[] byteArray = BitConverter.GetBytes(iBuffer);
                res = CanTpApi.SetValue_2016(receiver_handle, cantp_parameter.PCANTP_PARAMETER_RECEIVE_EVENT, byteArray, sizeof(UInt64));
            }
            Console.WriteLine("Stop receive event  : {0}", STATUS_OK_KO(res));
            receive_event.Close();

            // Uninitialize
            res = CanTpApi.Uninitialize_2016(transmitter_handle);
            Console.WriteLine("Uninitialize transmitter : {0}", STATUS_OK_KO(res));
            res = CanTpApi.Uninitialize_2016(receiver_handle);
            Console.WriteLine("Uninitialize receiver : {0}", STATUS_OK_KO(res));

            // Exit
            Console.WriteLine("Press any key to exit...");
            Console.In.Read();
        }

        static String OK_KO(bool test)
        {
            if (test)
                return "OK";
            return "KO";
        }

        static String STATUS_OK_KO(cantp_status test)
        {
            return OK_KO(CanTpApi.StatusIsOk_2016(test));
        }



    }
}
