﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.IO;


using Peak.Can.IsoTp;
using cantp_timestamp = System.UInt64;  // Note : must repeat it even if already defined in PCAN-ISO-TP_2016.cs

// Note : this example uses unsafe mode, it means that 
// "allow unsafe mode" must be checked in the Visual Studio project,
// and that 
// "#define UNSAFE" must be uncommented at the beginning of the file "PCAN-ISO-TP_2016.cs", or that UNSAFE must be defined at the project level.

namespace _06_isotp_segmented_read_write
{
    class Program
    {
        const UInt32 STMIN_600US = 0xF6;
        /// <summary>
        ///   Entry point of the program, start a small CAN ISO TP read/write example
        ///     -> Send a message containing Peak logo 
        ///     -> Receive message, wait complete reception (show progress)
        ///     -> Show loopback message and received message (Peak logo)
        /// </summary>
        static void Main(string[] args)
        {
            // Local variables
            cantp_status res = new cantp_status();
            StringBuilder buffer = new StringBuilder(500);
            UInt32 STmin;
            cantp_msg rx_msg = new cantp_msg();
            cantp_msg tx_msg = new cantp_msg();
            cantp_msg loopback_msg = new cantp_msg();
            cantp_handle transmitter_handle = new cantp_handle();
            cantp_handle receiver_handle = new cantp_handle();
            cantp_mapping mapping = new cantp_mapping();
            cantp_mapping reverse_mapping = new cantp_mapping();
            bool safeVersion = false; // TODO: change to true, to test the safe version of this example.

            // Initialize handles 
            transmitter_handle = cantp_handle.PCANTP_HANDLE_USBBUS1; // TODO: modify the value according to your available PCAN devices.
            receiver_handle = cantp_handle.PCANTP_HANDLE_USBBUS2; // TODO: modify the value according to your available PCAN devices.

            // Print version informations
            CanTpApi.GetValue_2016(cantp_handle.PCANTP_HANDLE_NONEBUS, cantp_parameter.PCANTP_PARAMETER_API_VERSION, buffer, 500);
            Console.WriteLine("PCAN-ISO-TP API Version : {0}", buffer);

            // Initialize channels: CAN2.0 - 500Kbit/s
            res = CanTpApi.Initialize_2016(transmitter_handle, cantp_baudrate.PCANTP_BAUDRATE_500K);
            Console.WriteLine("Initialize transmitter {0} : {1}", cantp_handle_toShortString(transmitter_handle), STATUS_OK_KO(res));
            res = CanTpApi.Initialize_2016(receiver_handle, cantp_baudrate.PCANTP_BAUDRATE_500K);
            Console.WriteLine("Initialize receiver {0} : {1}", cantp_handle_toShortString(receiver_handle), STATUS_OK_KO(res));

            // Change STmin value to 600us
            STmin = STMIN_600US;
            res = CanTpApi.SetValue_2016(transmitter_handle, cantp_parameter.PCANTP_PARAMETER_SEPARATION_TIME, ref STmin, sizeof(UInt32));
            Console.WriteLine("Set STMIN = 600us on transmitter {0} : {1}", cantp_handle_toShortString(transmitter_handle), STATUS_OK_KO(res));
            res = CanTpApi.SetValue_2016(receiver_handle, cantp_parameter.PCANTP_PARAMETER_SEPARATION_TIME, ref STmin, sizeof(UInt32));
            Console.WriteLine("Set STMIN = 600us on receiver {0} : {1}", cantp_handle_toShortString(receiver_handle), STATUS_OK_KO(res));

            // Create and set a receive event on transmitter
            System.Threading.AutoResetEvent transmitter_receive_event = new System.Threading.AutoResetEvent(false);
            if (IntPtr.Size == 4)
            {
                UInt32 iBuffer = Convert.ToUInt32(transmitter_receive_event.SafeWaitHandle.DangerousGetHandle().ToInt32());
                res = CanTpApi.SetValue_2016(transmitter_handle, cantp_parameter.PCANTP_PARAMETER_RECEIVE_EVENT, ref iBuffer, sizeof(UInt32));
            }
            else if (IntPtr.Size == 8)
            {
                Int64 iBuffer = transmitter_receive_event.SafeWaitHandle.DangerousGetHandle().ToInt64();
                byte[] byteArray = BitConverter.GetBytes(iBuffer);
                res = CanTpApi.SetValue_2016(transmitter_handle, cantp_parameter.PCANTP_PARAMETER_RECEIVE_EVENT, byteArray, sizeof(UInt64));
            }
            Console.WriteLine("Set receive event on transmitter {0} : {1}", cantp_handle_toShortString(transmitter_handle), STATUS_OK_KO(res));

            // Create and set a receive event on receiver
            System.Threading.AutoResetEvent receiver_receive_event = new System.Threading.AutoResetEvent(false);
            if (IntPtr.Size == 4)
            {
                UInt32 iBuffer = Convert.ToUInt32(receiver_receive_event.SafeWaitHandle.DangerousGetHandle().ToInt32());
                res = CanTpApi.SetValue_2016(receiver_handle, cantp_parameter.PCANTP_PARAMETER_RECEIVE_EVENT, ref iBuffer, sizeof(UInt32));
            }
            else if (IntPtr.Size == 8)
            {
                Int64 iBuffer = receiver_receive_event.SafeWaitHandle.DangerousGetHandle().ToInt64();
                byte[] byteArray = BitConverter.GetBytes(iBuffer);
                res = CanTpApi.SetValue_2016(receiver_handle, cantp_parameter.PCANTP_PARAMETER_RECEIVE_EVENT, byteArray, sizeof(UInt64));
            }
            Console.WriteLine("Set receive event on receiver {0} : {1}", cantp_handle_toShortString(receiver_handle), STATUS_OK_KO(res));

            // Allocate tx message
            res = CanTpApi.MsgDataAlloc_2016(out tx_msg, cantp_msgtype.PCANTP_MSGTYPE_ISOTP);
            Console.WriteLine("Allocate tx message : {0}", STATUS_OK_KO(res));
            res = CanTpApi.MsgDataAlloc_2016(out loopback_msg, cantp_msgtype.PCANTP_MSGTYPE_NONE);
            Console.WriteLine("Allocate loopback message : {0}", STATUS_OK_KO(res));
            res = CanTpApi.MsgDataAlloc_2016(out rx_msg, cantp_msgtype.PCANTP_MSGTYPE_NONE);
            Console.WriteLine("Allocate rx message : {0}", STATUS_OK_KO(res));

            // Create a simple physical mapping: 
            //    - Source 0xF1 (transmitter), target 0x01 (receiver), CAN id 0xA1, CAN ID flow control 0xA2
            //    - Diagnostic message in a classic format
            mapping.can_id = 0xA1;
            mapping.can_id_flow_ctrl = 0xA2;
            mapping.can_msgtype = cantp_can_msgtype.PCANTP_CAN_MSGTYPE_STANDARD;
            mapping.can_tx_dlc = 0x0;
            mapping.netaddrinfo.extension_addr = 0x00;
            mapping.netaddrinfo.format = cantp_isotp_format.PCANTP_ISOTP_FORMAT_NORMAL;
            mapping.netaddrinfo.msgtype = cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC;
            mapping.netaddrinfo.source_addr = 0xF1;
            mapping.netaddrinfo.target_addr = 0x01;
            mapping.netaddrinfo.target_type = cantp_isotp_addressing.PCANTP_ISOTP_ADDRESSING_PHYSICAL;

            // Create the associated reversed mapping: 
            reverse_mapping = mapping;
            reverse_mapping.can_id = mapping.can_id_flow_ctrl;
            reverse_mapping.can_id_flow_ctrl = mapping.can_id;
            reverse_mapping.netaddrinfo.source_addr = mapping.netaddrinfo.target_addr;
            reverse_mapping.netaddrinfo.target_addr = mapping.netaddrinfo.source_addr;

            // Add mapping on channels
            res = CanTpApi.AddMapping_2016(transmitter_handle, ref mapping);
            Console.WriteLine("Add a simple mapping on transmitter {0} : {1}", cantp_handle_toShortString(transmitter_handle), STATUS_OK_KO(res));
            res = CanTpApi.AddMapping_2016(transmitter_handle, ref reverse_mapping);
            Console.WriteLine("Add the reverse mapping on transmitter {0} : {1}", cantp_handle_toShortString(transmitter_handle), STATUS_OK_KO(res));
            res = CanTpApi.AddMapping_2016(receiver_handle, ref mapping);
            Console.WriteLine("Add a simple mapping on receiver {0} : {1}", cantp_handle_toShortString(receiver_handle), STATUS_OK_KO(res));
            res = CanTpApi.AddMapping_2016(receiver_handle, ref reverse_mapping);
            Console.WriteLine("Add the reverse mapping on receiver {0} : {1}", cantp_handle_toShortString(receiver_handle), STATUS_OK_KO(res));

            // Initialize Tx message containing a heavy data
            res = CanTpApi.MsgDataInit_2016(out tx_msg, CanTpApi.PCANTP_CAN_ID_DEFINED_BY_NAI, mapping.can_msgtype, heavy_data.HEAVY_DATA_SIZE, heavy_data.HEAVY_DATA, ref mapping.netaddrinfo);
            Console.WriteLine("Initialize tx message: {0}", STATUS_OK_KO(res));

            // Write message on transmitter
            res = CanTpApi.Write_2016(transmitter_handle, ref tx_msg);
            Console.WriteLine("Write message on transmitter {0} : {1}", cantp_handle_toShortString(transmitter_handle), STATUS_OK_KO(res));

            // Read loopback message and save it 
            res = wait_and_read_msg(safeVersion, transmitter_handle, transmitter_receive_event, ref loopback_msg);
            Console.WriteLine("Read loopback message on transmitter {0} : {1}", cantp_handle_toShortString(transmitter_handle), STATUS_OK_KO(res));
            bool fileTxOk = CanTpApi.StatusIsOk_2016(res);
            if (fileTxOk)
                write_binary_output_file("loopback_message.png", ref loopback_msg);

            // Read received message and save it
            res = wait_and_read_msg(safeVersion, receiver_handle, receiver_receive_event, ref rx_msg);
            Console.WriteLine("Read message on receiver {0} : {1}", cantp_handle_toShortString(receiver_handle), STATUS_OK_KO(res));
            bool fileRxOk = CanTpApi.StatusIsOk_2016(res);
            if (fileRxOk)
                write_binary_output_file("received_message.png", ref rx_msg);


            // Show & remove result: 
            if (fileTxOk)
            {
                Console.WriteLine(" >>> Opening transmitted message (PEAK logo).");
                if (File.Exists("loopback_message.png"))
                    System.Diagnostics.Process.Start("loopback_message.png");
            }
            else
                Console.WriteLine("\n >>> Failed to successfully transmit message (PEAK logo)!");

            if (fileRxOk)
            {
                Console.WriteLine(" >>> Opening received message (PEAK logo).");
                if (File.Exists("received_message.png"))
                    System.Diagnostics.Process.Start("received_message.png");
            }
            else
                Console.WriteLine("\n >>> Failed to successfully receive message (PEAK logo)!");

            // Free messages
            res = CanTpApi.MsgDataFree_2016(ref rx_msg);
            Console.WriteLine("Free rx message : {0}", STATUS_OK_KO(res));
            res = CanTpApi.MsgDataFree_2016(ref tx_msg);
            Console.WriteLine("Free tx message : {0}", STATUS_OK_KO(res));
            res = CanTpApi.MsgDataFree_2016(ref loopback_msg);
            Console.WriteLine("Free loopback message : {0}", STATUS_OK_KO(res));

            // Close receive event on transmitter
            if (IntPtr.Size == 4)
            {
                UInt32 iBuffer = 0;
                res = CanTpApi.SetValue_2016(transmitter_handle, cantp_parameter.PCANTP_PARAMETER_RECEIVE_EVENT, ref iBuffer, sizeof(UInt32));
            }
            else if (IntPtr.Size == 8)
            {
                Int64 iBuffer = 0;
                byte[] byteArray = BitConverter.GetBytes(iBuffer);
                res = CanTpApi.SetValue_2016(transmitter_handle, cantp_parameter.PCANTP_PARAMETER_RECEIVE_EVENT, byteArray, sizeof(UInt64));
            }
            Console.WriteLine("Stop receive event on transmitter {0} : {1}", cantp_handle_toShortString(transmitter_handle), STATUS_OK_KO(res));
            transmitter_receive_event.Close();

            // Close receive event on receiver
            if (IntPtr.Size == 4)
            {
                UInt32 iBuffer = 0;
                res = CanTpApi.SetValue_2016(receiver_handle, cantp_parameter.PCANTP_PARAMETER_RECEIVE_EVENT, ref iBuffer, sizeof(UInt32));
            }
            else if (IntPtr.Size == 8)
            {
                Int64 iBuffer = 0;
                byte[] byteArray = BitConverter.GetBytes(iBuffer);
                res = CanTpApi.SetValue_2016(receiver_handle, cantp_parameter.PCANTP_PARAMETER_RECEIVE_EVENT, byteArray, sizeof(UInt64));
            }
            Console.WriteLine("Stop receive event on receiver {0} : {1}", cantp_handle_toShortString(receiver_handle), STATUS_OK_KO(res));
            receiver_receive_event.Close();

            // Uninitialize channels
            res = CanTpApi.Uninitialize_2016(transmitter_handle);
            Console.WriteLine("Uninitialize transmitter {0} : {1}", cantp_handle_toShortString(transmitter_handle), STATUS_OK_KO(res));
            res = CanTpApi.Uninitialize_2016(receiver_handle);
            Console.WriteLine("Uninitialize receiver {0} : {1}", cantp_handle_toShortString(receiver_handle), STATUS_OK_KO(res));

            // Remove results and exit
            Console.WriteLine("Press any key to exit...");
            Console.In.Read();
            try
            {
                File.Delete("loopback_message.png");
                File.Delete("received_message.png");
            }
            catch (Exception)
            {

            }
        }

        static String OK_KO(bool test)
        {
            if (test)
                return "OK";
            return "KO";
        }

        static String STATUS_OK_KO(cantp_status test)
        {
            return OK_KO(CanTpApi.StatusIsOk_2016(test));
        }


        /// <summary>Wait and read single message</summary>
        /// <param name="safeVersion">Safe or unsafe version</param>
        /// <param name="handle">Channel handle</param>
        /// <param name="msg_buffer">Received message (out)</param>
        /// <param name="evt">Receive event associated to the channel</param>
        /// <returns>Status</returns>
        static cantp_status wait_and_read_msg(bool safeVersion, cantp_handle handle, System.Threading.AutoResetEvent evt, ref cantp_msg msg_buffer)
        {
            cantp_status res = cantp_status.PCANTP_STATUS_NO_MESSAGE;

            // Wait a receive event and get a message
            // If we receive something, read the message
            if (evt.WaitOne(100))
            {
                res = CanTpApi.Read_2016(handle, out msg_buffer);
                Console.WriteLine("Wait a message on {0} : {1}", cantp_handle_toShortString(handle), STATUS_OK_KO(res));

                // Check if the message is totaly received (only in ISOTP mode)
                if (CanTpApi.StatusIsOk_2016(res)
                    && (cantp_msgtype.PCANTP_MSGTYPE_ISOTP & msg_buffer.type) == cantp_msgtype.PCANTP_MSGTYPE_ISOTP)
                {
                    // get the net info of the message 
                    cantp_netaddrinfo info = new cantp_netaddrinfo();
                    bool infoOk = false;
                    if (safeVersion)
                        infoOk = CanTpApi.getNetaddrinfo_2016(ref msg_buffer, out info);
                    else
                        infoOk = CanTpApi.getNetaddrinfo_unsafe_2016(ref msg_buffer, out info);

                    if (infoOk)
                    {
                        // test the rx indication
                        if ((info.msgtype & cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_FLAG_INDICATION_RX) == cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_FLAG_INDICATION_RX)
                        {
                            // The message is being received, wait and show progress
                            if (safeVersion)
                                res = wait_and_show_progress_safe(handle, ref msg_buffer);
                            else
                                res = wait_and_show_progress_unsafe(handle, ref msg_buffer);
                        }
                    }
                }
            }

            return res;
        }


        /// <summary>Write message data in a binary file</summary>
        /// <param name="file_path">Output file path</param>
        /// <param name="msg">Message to write</param>
        static void write_binary_output_file(String file_path, ref cantp_msg msg)
        {
            if (msg.Msgdata != IntPtr.Zero)
            {
                uint len = msg.Msgdata_any_Copy.length;
                if (len != 0)
                {
                    byte[] data = new byte[len];
                    if (CanTpApi.getData_2016(ref msg, 0, data, (int)len))
                    {
                        try
                        {
                            FileStream stream = new FileStream(file_path, FileMode.Create);
                            BinaryWriter writer = new BinaryWriter(stream);
                            for (int i = 0; i < len; i += 2)
                            {
                                byte lsb = data[i];
                                byte msb = data[i + 1];
                                writer.Write(msb);
                                writer.Write(lsb);
                            }
                            writer.Close();
                            stream.Close();
                        }
                        catch (Exception)
                        {

                        }
                    }
                }
            }
        }

        /// <summary>Wait for a message to be received on a channel and show reception progress, in safe mode</summary>
        /// <param name="handle">Channel handle</param>
        /// <param name="msg_buffer">Received message</param>
        /// <returns>a status</returns>
        static cantp_status wait_and_show_progress_safe(cantp_handle handle, ref cantp_msg msg_buffer)
        {
            cantp_msgprogress progress = new cantp_msgprogress();

            // Set a buffer in order to receive a copy of the pending message
            cantp_msg msgpend = new cantp_msg();

            // Allocate a buffer in order to receive a copy of the pending message
            cantp_status res = CanTpApi.allocProgressBuffer_2016(ref progress, cantp_msgtype.PCANTP_MSGTYPE_ISOTP);
            Console.WriteLine("Allocate a progress buffer : {0}", STATUS_OK_KO(res));

            if (CanTpApi.StatusIsOk_2016(res))
            {
                do
                {
                    System.Threading.Thread.Sleep(300);
                    // Refresh progress structure
                    res = CanTpApi.GetMsgProgress_2016(handle, ref msg_buffer, cantp_msgdirection.PCANTP_MSGDIRECTION_RX, out progress);
                    // Refresh pending message
                    bool getProgress = CanTpApi.getProgressBuffer_2016(ref progress, ref msgpend);
                    // Get last bytes of pending message
                    byte[] vals = new byte[10];
                    uint lenpend = 0;
                    bool getvals = CanTpApi.getLength_2016(ref msgpend, out lenpend);
                    getvals &= CanTpApi.getData_2016(ref msgpend, (int)lenpend - 10, vals, 10);
                    // Show progress pencentage and last bytes
                    Console.WriteLine("RX Progress on {0} = {1}%, last bytes {2} : {3}",
                        cantp_handle_toShortString(handle),
                        progress.percentage,
                        getvals ? getBytesAsString(ref vals, 10) : "Unknown",
                        OK_KO(getProgress && CanTpApi.StatusIsOk_2016(res)));

                } while (progress.state == cantp_msgprogress_state.PCANTP_MSGPROGRESS_STATE_PROCESSING);

                CanTpApi.freeProgressBuffer_2016(ref progress);

                // The message is received, read it
                res = CanTpApi.Read_2016(handle, out msg_buffer);
            }

            return res;
        }

        /// <summary>Wait for a message to be received on a channel and show reception progress, in unsafe mode</summary>
        /// <param name="handle">Channel handle</param>
        /// <param name="msg_buffer">Received message</param>
        /// <returns>a status</returns>
        static cantp_status wait_and_show_progress_unsafe(cantp_handle handle, ref cantp_msg msg_buffer)
        {
            cantp_status res = cantp_status.PCANTP_STATUS_OK;
            cantp_msgprogress progress = new cantp_msgprogress();

            // Set a buffer in order to receive a copy of the pending message
            cantp_msg msgpend = new cantp_msg();

            // Attach progress to pending message
            CanTpApi.setProgressBuffer_unsafe_2016(ref progress, ref msgpend);

            do
            {
                System.Threading.Thread.Sleep(300);
                // Refresh progress structure and pending message
                res = CanTpApi.GetMsgProgress_2016(handle, ref msg_buffer, cantp_msgdirection.PCANTP_MSGDIRECTION_RX, out progress);
                // Get last bytes of pending message
                byte[] vals = new byte[10];
                uint lenpend = 0;
                bool getvals = CanTpApi.getLength_unsafe_2016(ref msgpend, out lenpend);
                getvals &= CanTpApi.getData_unsafe_2016(ref msgpend, (int)lenpend - 10, vals, 10);
                // Show progress pencentage and last bytes
                Console.WriteLine("RX Progress on {0} = {1}%, last bytes {2} : {3}",
                    cantp_handle_toShortString(handle),
                    progress.percentage,
                    getvals ? getBytesAsString(ref vals, 10) : "Unknown",
                    STATUS_OK_KO(res));

            } while (progress.state == cantp_msgprogress_state.PCANTP_MSGPROGRESS_STATE_PROCESSING);


            // The message is received, read it
            res = CanTpApi.Read_2016(handle, out msg_buffer);

            return res;
        }

        /// <summary>Return the name of the channel</summary>
        /// <param name="value">Channel handle</param>
        /// <returns>String containing the name of the channel</returns>
        static String cantp_handle_toShortString(cantp_handle value)
        {
            String res = "";
            switch (value)
            {
                case cantp_handle.PCANTP_HANDLE_NONEBUS: res = "NONEBUS"; break;
                case cantp_handle.PCANTP_HANDLE_ISABUS1: res = "ISA1"; break;
                case cantp_handle.PCANTP_HANDLE_ISABUS2: res = "ISA2"; break;
                case cantp_handle.PCANTP_HANDLE_ISABUS3: res = "ISA3"; break;
                case cantp_handle.PCANTP_HANDLE_ISABUS4: res = "ISA4"; break;
                case cantp_handle.PCANTP_HANDLE_ISABUS5: res = "ISA5"; break;
                case cantp_handle.PCANTP_HANDLE_ISABUS6: res = "ISA6"; break;
                case cantp_handle.PCANTP_HANDLE_ISABUS7: res = "ISA7"; break;
                case cantp_handle.PCANTP_HANDLE_ISABUS8: res = "ISA8"; break;
                case cantp_handle.PCANTP_HANDLE_DNGBUS1: res = "DNG1"; break;
                case cantp_handle.PCANTP_HANDLE_PCIBUS1: res = "PCI1"; break;
                case cantp_handle.PCANTP_HANDLE_PCIBUS2: res = "PCI2"; break;
                case cantp_handle.PCANTP_HANDLE_PCIBUS3: res = "PCI3"; break;
                case cantp_handle.PCANTP_HANDLE_PCIBUS4: res = "PCI4"; break;
                case cantp_handle.PCANTP_HANDLE_PCIBUS5: res = "PCI5"; break;
                case cantp_handle.PCANTP_HANDLE_PCIBUS6: res = "PCI6"; break;
                case cantp_handle.PCANTP_HANDLE_PCIBUS7: res = "PCI7"; break;
                case cantp_handle.PCANTP_HANDLE_PCIBUS8: res = "PCI8"; break;
                case cantp_handle.PCANTP_HANDLE_PCIBUS9: res = "PCI9"; break;
                case cantp_handle.PCANTP_HANDLE_PCIBUS10: res = "PCI10"; break;
                case cantp_handle.PCANTP_HANDLE_PCIBUS11: res = "PCI11"; break;
                case cantp_handle.PCANTP_HANDLE_PCIBUS12: res = "PCI12"; break;
                case cantp_handle.PCANTP_HANDLE_PCIBUS13: res = "PCI13"; break;
                case cantp_handle.PCANTP_HANDLE_PCIBUS14: res = "PCI14"; break;
                case cantp_handle.PCANTP_HANDLE_PCIBUS15: res = "PCI15"; break;
                case cantp_handle.PCANTP_HANDLE_PCIBUS16: res = "PCI16"; break;
                case cantp_handle.PCANTP_HANDLE_USBBUS1: res = "USB1"; break;
                case cantp_handle.PCANTP_HANDLE_USBBUS2: res = "USB2"; break;
                case cantp_handle.PCANTP_HANDLE_USBBUS3: res = "USB3"; break;
                case cantp_handle.PCANTP_HANDLE_USBBUS4: res = "USB4"; break;
                case cantp_handle.PCANTP_HANDLE_USBBUS5: res = "USB5"; break;
                case cantp_handle.PCANTP_HANDLE_USBBUS6: res = "USB6"; break;
                case cantp_handle.PCANTP_HANDLE_USBBUS7: res = "USB7"; break;
                case cantp_handle.PCANTP_HANDLE_USBBUS8: res = "USB8"; break;
                case cantp_handle.PCANTP_HANDLE_USBBUS9: res = "USB9"; break;
                case cantp_handle.PCANTP_HANDLE_USBBUS10: res = "USB10"; break;
                case cantp_handle.PCANTP_HANDLE_USBBUS11: res = "USB11"; break;
                case cantp_handle.PCANTP_HANDLE_USBBUS12: res = "USB12"; break;
                case cantp_handle.PCANTP_HANDLE_USBBUS13: res = "USB13"; break;
                case cantp_handle.PCANTP_HANDLE_USBBUS14: res = "USB14"; break;
                case cantp_handle.PCANTP_HANDLE_USBBUS15: res = "USB15"; break;
                case cantp_handle.PCANTP_HANDLE_USBBUS16: res = "USB16"; break;
                case cantp_handle.PCANTP_HANDLE_PCCBUS1: res = "PCC1"; break;
                case cantp_handle.PCANTP_HANDLE_PCCBUS2: res = "PCC2"; break;
                case cantp_handle.PCANTP_HANDLE_LANBUS1: res = "LAN1"; break;
                case cantp_handle.PCANTP_HANDLE_LANBUS2: res = "LAN2"; break;
                case cantp_handle.PCANTP_HANDLE_LANBUS3: res = "LAN3"; break;
                case cantp_handle.PCANTP_HANDLE_LANBUS4: res = "LAN4"; break;
                case cantp_handle.PCANTP_HANDLE_LANBUS5: res = "LAN5"; break;
                case cantp_handle.PCANTP_HANDLE_LANBUS6: res = "LAN6"; break;
                case cantp_handle.PCANTP_HANDLE_LANBUS7: res = "LAN7"; break;
                case cantp_handle.PCANTP_HANDLE_LANBUS8: res = "LAN8"; break;
                case cantp_handle.PCANTP_HANDLE_LANBUS9: res = "LAN9"; break;
                case cantp_handle.PCANTP_HANDLE_LANBUS10: res = "LAN10"; break;
                case cantp_handle.PCANTP_HANDLE_LANBUS11: res = "LAN11"; break;
                case cantp_handle.PCANTP_HANDLE_LANBUS12: res = "LAN12"; break;
                case cantp_handle.PCANTP_HANDLE_LANBUS13: res = "LAN13"; break;
                case cantp_handle.PCANTP_HANDLE_LANBUS14: res = "LAN14"; break;
                case cantp_handle.PCANTP_HANDLE_LANBUS15: res = "LAN15"; break;
                case cantp_handle.PCANTP_HANDLE_LANBUS16: res = "LAN16"; break;
                default: res = "UNKNOWN"; break;
            }
            return res;
        }

        /// <summary>Get a string printing values in hexa of an array of bytes</summary>
        /// <param name="vals">Array of bytes</param>
        /// <param name="len">Length of the array</param>
        /// <returns>The string</returns>
        static String getBytesAsString(ref byte[] vals, uint len)
        {
            StringBuilder str = new StringBuilder();
            for (uint i = 0; i < len; i++)
                str.AppendFormat("{0:X2}", vals[i]);
            return str.ToString();
        }
    }
}
