﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;


using Peak.Can.IsoTp;
using cantp_timestamp = System.UInt64;  // Note : must repeat it even if already defined in PCAN-ISO-TP_2016.cs

namespace _07_classic_can_read_write
{
    class Program
    {
        /// <summary>Entry point of the program, start a small ISO TP classic CAN read/write example</summary>
        static void Main(string[] args)
        {
            // Local variables
            cantp_status res = new cantp_status();
            StringBuilder buffer = new StringBuilder(500);
            cantp_msg rx_msg = new cantp_msg();
            cantp_msg tx_msg = new cantp_msg();
            cantp_handle transmitter_handle = new cantp_handle();
            cantp_handle receiver_handle = new cantp_handle();

            // Initialize handles 
            transmitter_handle = cantp_handle.PCANTP_HANDLE_USBBUS1; // TODO: modify the value according to your available PCAN devices.
            receiver_handle = cantp_handle.PCANTP_HANDLE_USBBUS2; // TODO: modify the value according to your available PCAN devices.

            // Print version informations
            CanTpApi.GetValue_2016(cantp_handle.PCANTP_HANDLE_NONEBUS, cantp_parameter.PCANTP_PARAMETER_API_VERSION, buffer, 500);
            Console.WriteLine("PCAN-ISO-TP API Version : {0}", buffer);

            // Initialize channels: CAN2.0 - 500Kbit/s
            res = CanTpApi.Initialize_2016(transmitter_handle, cantp_baudrate.PCANTP_BAUDRATE_500K);
            Console.WriteLine("Initialize transmitter : {0}", STATUS_OK_KO(res));
            res = CanTpApi.Initialize_2016(receiver_handle, cantp_baudrate.PCANTP_BAUDRATE_500K);
            Console.WriteLine("Initialize receiver : {0}", STATUS_OK_KO(res));

            // Create and set a receive event on receiver
            System.Threading.AutoResetEvent receive_event = new System.Threading.AutoResetEvent(false);
            if (IntPtr.Size == 4)
            {
                UInt32 iBuffer = Convert.ToUInt32(receive_event.SafeWaitHandle.DangerousGetHandle().ToInt32());
                res = CanTpApi.SetValue_2016(receiver_handle, cantp_parameter.PCANTP_PARAMETER_RECEIVE_EVENT, ref iBuffer, sizeof(UInt32));
            }
            else if (IntPtr.Size == 8)
            {
                Int64 iBuffer = receive_event.SafeWaitHandle.DangerousGetHandle().ToInt64();
                byte[] byteArray = BitConverter.GetBytes(iBuffer);
                res = CanTpApi.SetValue_2016(receiver_handle, cantp_parameter.PCANTP_PARAMETER_RECEIVE_EVENT, byteArray, sizeof(UInt64));
            }
            Console.WriteLine("Set receive event on receiver : {0}", STATUS_OK_KO(res));

            // Allocate tx CAN message
            res = CanTpApi.MsgDataAlloc_2016(out tx_msg, cantp_msgtype.PCANTP_MSGTYPE_CAN);
            Console.WriteLine("Allocate tx CAN message : {0}", STATUS_OK_KO(res));

            // Allocate rx message
            res = CanTpApi.MsgDataAlloc_2016(out rx_msg, cantp_msgtype.PCANTP_MSGTYPE_NONE);
            Console.WriteLine("Allocate rx message : {0}", STATUS_OK_KO(res));

            // Initialize Tx message containing "42"
            uint can_id = 0xA1;
            byte[] data = { 0x42 };
            res = CanTpApi.MsgDataInit_2016(out tx_msg, can_id, cantp_can_msgtype.PCANTP_CAN_MSGTYPE_STANDARD, 1, data);
            Console.WriteLine("Initialize tx message : {0}", STATUS_OK_KO(res));

            // Write "42" message
            res = CanTpApi.Write_2016(transmitter_handle, ref tx_msg);
            Console.WriteLine("Write \"42\" message : {0}", STATUS_OK_KO(res));

            // Wait a receive event on receiver
            bool wait_result = receive_event.WaitOne(100);
            Console.WriteLine("Wait a message on receiver : {0}", OK_KO(wait_result));

            // If we receive something on the receiver, read the message
            if (wait_result)
            {
                res = CanTpApi.Read_2016(receiver_handle, out rx_msg);
                Console.WriteLine("Read message on receiver : {0}", STATUS_OK_KO(res));
                byte val = 0;
                if (CanTpApi.getData_2016(ref rx_msg, 0, out val))
                    Console.WriteLine("Check if the message is \"42\" : {0}", OK_KO(val == 0x42));
                else
                    Console.WriteLine("Check if the message is \"42\" : NOK");
            }

            // Free messages space
            res = CanTpApi.MsgDataFree_2016(ref rx_msg);
            Console.WriteLine("Free rx message : {0}", STATUS_OK_KO(res));
            res = CanTpApi.MsgDataFree_2016(ref tx_msg);
            Console.WriteLine("Free tx message : {0}", STATUS_OK_KO(res));

            // Close receive event
            if (IntPtr.Size == 4)
            {
                UInt32 iBuffer = 0;
                res = CanTpApi.SetValue_2016(receiver_handle, cantp_parameter.PCANTP_PARAMETER_RECEIVE_EVENT, ref iBuffer, sizeof(UInt32));
            }
            else if (IntPtr.Size == 8)
            {
                Int64 iBuffer = 0;
                byte[] byteArray = BitConverter.GetBytes(iBuffer);
                res = CanTpApi.SetValue_2016(receiver_handle, cantp_parameter.PCANTP_PARAMETER_RECEIVE_EVENT, byteArray, sizeof(UInt64));
            }
            Console.WriteLine("Stop receive event  : {0}", STATUS_OK_KO(res));
            receive_event.Close();

            // Uninitialize channels
            res = CanTpApi.Uninitialize_2016(transmitter_handle);
            Console.WriteLine("Uninitialize transmitter : {0}", STATUS_OK_KO(res));
            res = CanTpApi.Uninitialize_2016(receiver_handle);
            Console.WriteLine("Uninitialize receiver : {0}", STATUS_OK_KO(res));

            // Exit
            Console.WriteLine("Press any key to exit...");
            Console.In.Read();
        }

        static String OK_KO(bool test)
        {
            if (test)
                return "OK";
            return "KO";
        }

        static String STATUS_OK_KO(cantp_status test)
        {
            return OK_KO(CanTpApi.StatusIsOk_2016(test));
        }



    }
}
