using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

using Peak.Can.IsoTp;
using cantp_timestamp = System.UInt64;  // See PCAN-ISO-TP_2016.cs
using cantp_bitrate = System.String;  // See PCAN-ISO-TP_2016.cs
using System.Runtime.InteropServices;


namespace _10_client_isotp
{
    class Program
    {
        const bool USE_CAN_FD = false;
        const byte CAN_TX_DL_CODE = 0x0D;
        const byte CAN_TX_DL_VALUE = 32;
        const byte N_SA = 0xF1;
        const byte N_TA_PHYS = 0x13;
        const byte N_TA_FUNC = 0x26;
        const byte N_RA = 0x52;
        const bool TEST_PHYS_TRANSMISSION = true;
        const bool TEST_FUNC_TRANSMISSION = true;
        const bool TEST_BURST_TRANSMISSION = true;
        const UInt32 MESSAGE_MAX_LENGTH = CanTpApi.PCANTP_MAX_LENGTH_ISOTP2004 + 1000;
        const bool USE_GETCH = false;

        // A global value to display timing information
        static DateTime g_timeReference;

        /// <summary>Main entry-point for this application.</summary>
        /// <param name="args">The args.</param>
        static void Main(string[] args)
        {
            StringBuilder buffer = new StringBuilder(500);
            cantp_handle channel = new cantp_handle();
            bool useCanFd = false;
            cantp_baudrate baudrate = new cantp_baudrate();
            cantp_bitrate bitrateFd = "";

            Console.WriteLine("\nUsage: PCTPClient.exe [TPCANTPHandle] [USE_CAN_FD:0|1] [BTR0BTR1|BITRATE]");
            Console.WriteLine(" * CAN: PCTPClient.exe 0x51 0 0x1C ");
            Console.WriteLine(" * CAN FD: PCTPClient.exe 0x51 1 \"f_clock=80000000, nom_brp=10, nom_tseg1=12, nom_tseg2=3, nom_sjw=1, data_brp=4, data_tseg1=7, data_tseg2=2, data_sjw=1\"");
            Console.WriteLine("\n---------------------------------------------");

            // clear gracefully ISO-TP if console window is killed
            AppDomain.CurrentDomain.UnhandledException += (sender, eventArgs) => consoleExit();
            SetConsoleCtrlHandler(new HandlerRoutine(ConsoleCtrlCheck), true);

            // Show version information
            cantp_status status = CanTpApi.GetValue_2016(cantp_handle.PCANTP_HANDLE_NONEBUS, cantp_parameter.PCANTP_PARAMETER_API_VERSION, buffer, 500);
            if (CanTpApi.StatusIsOk_2016(status))
                Console.WriteLine("PCAN-ISO-TP API Version : {0}", buffer);
            else
                Console.WriteLine("Error failed to get PCAN-ISO-TP API Version (sts=0x{0:X})", status);

            // 1st argument is the PCAN-channel to use (PCAN-USB channel 1)
            if (args.Length >= 1)
                channel = convertToHandle(args[0]);
            else
                channel = cantp_handle.PCANTP_HANDLE_USBBUS1;
            // 2nd argument is CAN FD
            if (args.Length >= 2)
                useCanFd = convertToBool(args[1]);
            else
                useCanFd = USE_CAN_FD;
            // 3rd argument is bitrate (either CAN or CAN FD)
            baudrate = cantp_baudrate.PCANTP_BAUDRATE_500K;
            bitrateFd = "f_clock=80000000, nom_brp=10, nom_tseg1=12, nom_tseg2=3, nom_sjw=1, data_brp=4, data_tseg1=7, data_tseg2=2, data_sjw=1";
            if (args.Length >= 3)
            {
                if (useCanFd)
                    bitrateFd = args[2];
                else
                    baudrate = convertToBaudRate(args[2]);
            }

            // Initializing of the ISO-TP Communication session
            Console.WriteLine("Connecting to channel 0x{0:X}...", channel);
            if (!useCanFd)
            {
                Console.WriteLine(" * btr0btr1 = 0x{0:X}...", baudrate);
                status = CanTpApi.Initialize_2016(channel, baudrate);
            }
            else
            {
                Console.WriteLine(" * bitrateFd = \"{0}\"...", bitrateFd);
                status = CanTpApi.InitializeFD_2016(channel, bitrateFd);
            }
            Console.WriteLine(" -> Initialize CANTP: 0x{0:X}", status);
            // will start the reading loop only if successfully initialized
            if (!CanTpApi.StatusIsOk_2016(status))
            {
                Console.WriteLine(" -> Initialization failed, exiting...");
                CanTpApi.Uninitialize_2016(channel);
                Console.WriteLine("\n\nPress <Enter> to quit...");
                Console.In.Read();
                return;
            }

            // configure the test session to use can data padding with a specific padding value
            Byte[] param = new Byte[1];
            param[0] = CanTpApi.PCANTP_CAN_DATA_PADDING_ON;
            status = CanTpApi.SetValue_2016(channel, cantp_parameter.PCANTP_PARAMETER_CAN_DATA_PADDING, param, 1);
            if (!CanTpApi.StatusIsOk_2016(status))
            {
                Console.WriteLine("Failed to configure parameter 'PCANTP_PARAMETER_CAN_DATA_PADDING' (sts=0x{0:X}).", status);
            }
            param[0] = 0x99;
            status = CanTpApi.SetValue_2016(channel, cantp_parameter.PCANTP_PARAMETER_CAN_PADDING_VALUE, param, 1);
            if (!CanTpApi.StatusIsOk_2016(status))
            {
                Console.WriteLine("Failed to configure parameter 'PCANTP_PARAMETER_CAN_PADDING_VALUE' (sts=0x{0:X}).", status);
            }
            // set the default TX_DL size (i.e. the default CAN-FD DLC)
            param[0] = CAN_TX_DL_CODE;
            status = CanTpApi.SetValue_2016(channel, cantp_parameter.PCANTP_PARAMETER_CAN_TX_DL, param, 1);
            if (!CanTpApi.StatusIsOk_2016(status))
            {
                Console.WriteLine("Failed to configure parameter 'PCANTP_PARAMETER_CAN_TX_DL' (sts=0x{0:X}).", status);
            }
            // configure server - enable enhanced addressing to be compatible with PCTPClient
            param[0] = CanTpApi.PCANTP_VALUE_PARAMETER_ON;
            status = CanTpApi.SetValue_2016(channel, cantp_parameter.PCANTP_PARAMETER_SUPPORT_29B_ENHANCED, param, 1);
            if (!CanTpApi.StatusIsOk_2016(status))
            {
                Console.WriteLine("Failed to configure parameter 'PCANTP_PARAMETER_SUPPORT_29B_ENHANCED' (sts=0x{0:X}).", status);
            }
            // log CAN frames
            param[0] = CanTpApi.PCANTP_DEBUG_WARNING;
            status = CanTpApi.SetValue_2016(channel, cantp_parameter.PCANTP_PARAMETER_DEBUG, param, 1);
            if (!CanTpApi.StatusIsOk_2016(status))
            {
                Console.WriteLine("Failed to configure parameter 'PCANTP_PARAMETER_DEBUG' (sts=0x{0:X}).", status);
            }
            // configure ISO-TP mappings
            if (!initialize_mappings(channel))
            {
                Console.WriteLine(" -> Mappings configuration failed, exiting...");
                CanTpApi.Uninitialize_2016(channel);
                Console.WriteLine("\n\nPress <Enter> to quit...");
                Console.In.Read();
                return;
            }

            bool doCanFd = false;
            int nbErr = 0;
            for (int i = 0; i < 2; i++)
            {
                switch (i)
                {
                    case 0:
                        // first loop will test CAN-FD frames even if it was not initialized with Fd
                        doCanFd = true;
                        break;
                    case 1:
                        // second loop tests standard CAN frames
                        doCanFd = false;
                        break;
                }

                // Transmit with 29 Bit CAN identifiers (and no CAN ID mapping is required)
                nbErr += transmit(channel, cantp_can_msgtype.PCANTP_CAN_MSGTYPE_EXTENDED, cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, cantp_isotp_format.PCANTP_ISOTP_FORMAT_FIXED_NORMAL, doCanFd, useCanFd);
                nbErr += transmit(channel, cantp_can_msgtype.PCANTP_CAN_MSGTYPE_EXTENDED, cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_REMOTE_DIAGNOSTIC, cantp_isotp_format.PCANTP_ISOTP_FORMAT_MIXED, doCanFd, useCanFd);

                // Transmit with enhanced diagnostics 29 bit CAN identifiers (ISO-15765-3 8.3)
                nbErr += transmit(channel, cantp_can_msgtype.PCANTP_CAN_MSGTYPE_EXTENDED, cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, cantp_isotp_format.PCANTP_ISOTP_FORMAT_ENHANCED, doCanFd, useCanFd);

                // Transmit with 11 Bit CAN identifiers
                // those transmissions require a Mapping between CAN ID and Network Addressing Information
                // for each transmission, 3 mappings are added :
                //	- 2 for physical addressing, one defining SA->TA and the other TA->SA
                //	- 1 for functional addressing defining SA->TA
                //

                // 11 bit can ID, normal format addressing (diagnostic message mandatory)		
                nbErr += transmit(channel, cantp_can_msgtype.PCANTP_CAN_MSGTYPE_STANDARD, cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, cantp_isotp_format.PCANTP_ISOTP_FORMAT_NORMAL, doCanFd, useCanFd);
                // 11 bit can ID, extended format addressing (diagnostic message mandatory)		
                nbErr += transmit(channel, cantp_can_msgtype.PCANTP_CAN_MSGTYPE_STANDARD, cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, cantp_isotp_format.PCANTP_ISOTP_FORMAT_EXTENDED, doCanFd, useCanFd);
                // 11 bit can ID, mixed format addressing (remote diagnostic message mandatory)
                nbErr += transmit(channel, cantp_can_msgtype.PCANTP_CAN_MSGTYPE_STANDARD, cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_REMOTE_DIAGNOSTIC, cantp_isotp_format.PCANTP_ISOTP_FORMAT_MIXED, doCanFd, useCanFd);

                // Transmit with 29 Bit CAN identifiers
                // those transmissions require a Mapping between CAN ID and Network Addressing Information
                // for each transmission, 3 mappings are added 2 for physical addressing and 
                // the other for functional addressing.
                //

                // 29 bit can ID, normal format addressing (diagnostic message mandatory)
                nbErr += transmit(channel, cantp_can_msgtype.PCANTP_CAN_MSGTYPE_EXTENDED, cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, cantp_isotp_format.PCANTP_ISOTP_FORMAT_NORMAL, doCanFd, useCanFd);
                // 29 bit can ID, extended format addressing (diagnostic message mandatory)
                nbErr += transmit(channel, cantp_can_msgtype.PCANTP_CAN_MSGTYPE_EXTENDED, cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, cantp_isotp_format.PCANTP_ISOTP_FORMAT_EXTENDED, doCanFd, useCanFd);
            }

            // release CAN-TP
            if (nbErr > 0)
            {
                Console.WriteLine("\nERROR : a total of {0} errors occured.\n", nbErr);
                Console.WriteLine("\n\nPress <Enter> to quit...");
                Console.In.Read();
            }
            else
            {
                Console.WriteLine("\nALL Transmissions succeeded !\n");
            }

            CanTpApi.Uninitialize_2016(channel);
        }

        // Callback to handle closure of the console window
        static bool ConsoleCtrlCheck(CtrlTypes ctrlType)
        {
            bool isClosing = false;
            switch (ctrlType)
            {
                case CtrlTypes.CTRL_C_EVENT:
                case CtrlTypes.CTRL_BREAK_EVENT:
                case CtrlTypes.CTRL_CLOSE_EVENT:
                case CtrlTypes.CTRL_LOGOFF_EVENT:
                case CtrlTypes.CTRL_SHUTDOWN_EVENT:
                    isClosing = true;
                    break;

                default:
                    break;
            }
            if (isClosing)
                consoleExit();
            return false;
        }


        // Declare the SetConsoleCtrlHandler function as external and receiving a delegate.
        [DllImport("Kernel32")]
        public static extern bool SetConsoleCtrlHandler(HandlerRoutine Handler, bool Add);
        // A delegate type to be used as the handler routine for SetConsoleCtrlHandler.
        public delegate bool HandlerRoutine(CtrlTypes CtrlType);
        // An enumerated type for the control messages sent to the handler routine.
        public enum CtrlTypes
        {
            CTRL_C_EVENT = 0,
            CTRL_BREAK_EVENT,
            CTRL_CLOSE_EVENT,
            CTRL_LOGOFF_EVENT = 5,
            CTRL_SHUTDOWN_EVENT
        }

        // Function called to clean opened ISO-TP channels
        static void consoleExit()
        {
            CanTpApi.Uninitialize_2016(cantp_handle.PCANTP_HANDLE_NONEBUS);
        }

        static cantp_handle convertToHandle(String s)
        {
            cantp_handle res = cantp_handle.PCANTP_HANDLE_NONEBUS;
            try
            {
                res = (cantp_handle)Convert.ToUInt32(s);
            }
            catch (Exception)
            {
                // maybe the string is prefixed by 0x so try hexa conversion
                try
                {
                    res = (cantp_handle)Convert.ToUInt32(s, 16);
                }
                catch (Exception)
                {
                    res = cantp_handle.PCANTP_HANDLE_NONEBUS;
                }
            }
            return res;
        }

        static cantp_baudrate convertToBaudRate(String s)
        {
            cantp_baudrate res = (cantp_baudrate)0;
            try
            {
                res = (cantp_baudrate)Convert.ToUInt32(s);
            }
            catch (Exception)
            {
                // maybe the string is prefixed by 0x so try hexa conversion
                try
                {
                    res = (cantp_baudrate)Convert.ToUInt32(s, 16);
                }
                catch (Exception)
                {
                    res = (cantp_baudrate)0;
                }
            }
            return res;
        }

        static bool convertToBool(String s)
        {
            bool res = false;
            try
            {
                res = Convert.ToBoolean(s);
            }
            catch (Exception)
            {
                // maybe the string is 0 or 1 so try int conversion
                try
                {
                    byte b = Convert.ToByte(s);
                    if (b == 0)
                        res = false;
                    else if (b == 1)
                        res = true;
                    else
                        res = false;
                }
                catch (Exception)
                {
                    res = false;
                }
            }
            return res;
        }

        /// <summary>Initializes mappings compatible with PCTPClient/PCTPServer sample</summary>
        /// <param name="channel">CAN channel to add the mappings to.</param>
        /// <return>true on successful configuration</return>
        static bool initialize_mappings(cantp_handle channel)
        {
            bool result = true;

            // 11 bit can ID, normal format addressing, diagnostic message (mandatory)
            result &= initialize_mappings(channel, cantp_can_msgtype.PCANTP_CAN_MSGTYPE_STANDARD, N_SA, N_TA_PHYS, 0x00, N_TA_FUNC, 0xA1, 0xA2, 0xA5, cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, cantp_isotp_format.PCANTP_ISOTP_FORMAT_NORMAL);
            // 11 bit can ID, extended format addressing, diagnostic message (mandatory)
            result &= initialize_mappings(channel, cantp_can_msgtype.PCANTP_CAN_MSGTYPE_STANDARD, N_SA, N_TA_PHYS, 0x00, N_TA_FUNC, 0xB1, 0xB2, 0xB5, cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, cantp_isotp_format.PCANTP_ISOTP_FORMAT_EXTENDED);
            result &= initialize_mappings(channel, cantp_can_msgtype.PCANTP_CAN_MSGTYPE_STANDARD, N_SA, N_TA_PHYS, 0x00, N_TA_FUNC, 0xC1, 0xC2, 0xC5, cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, cantp_isotp_format.PCANTP_ISOTP_FORMAT_EXTENDED);


            // 11 bit can ID, mixed format addressing, remote diagnostic message (mandatory)
            result &= initialize_mappings(channel, cantp_can_msgtype.PCANTP_CAN_MSGTYPE_STANDARD, N_SA, N_TA_PHYS, N_RA, N_TA_FUNC, 0xD1, 0xD2, 0xD5, cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_REMOTE_DIAGNOSTIC, cantp_isotp_format.PCANTP_ISOTP_FORMAT_MIXED);

            // 29 bit can ID, normal format addressing, diagnostic message (mandatory)
            result &= initialize_mappings(channel, cantp_can_msgtype.PCANTP_CAN_MSGTYPE_EXTENDED, N_SA, N_TA_PHYS, 0x00, N_TA_FUNC, 0xA123A1, 0xA123A2, 0xA123A5, cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, cantp_isotp_format.PCANTP_ISOTP_FORMAT_NORMAL);
            // 29 bit can ID, extended format addressing, diagnostic message
            result &= initialize_mappings(channel, cantp_can_msgtype.PCANTP_CAN_MSGTYPE_EXTENDED, N_SA, N_TA_PHYS, 0x00, N_TA_FUNC, 0xA123C1, 0xA123C2, 0xA123C5, cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, cantp_isotp_format.PCANTP_ISOTP_FORMAT_EXTENDED);
            result &= initialize_mappings(channel, cantp_can_msgtype.PCANTP_CAN_MSGTYPE_EXTENDED, N_SA, N_TA_PHYS, 0x00, N_TA_FUNC, 0xA123D1, 0xA123D2, 0xA123D5, cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, cantp_isotp_format.PCANTP_ISOTP_FORMAT_EXTENDED);

            return result;
        }

        /// <summary>Configure mappings to handle a two-way communication and a functional communication (broadcast).</summary>
        /// <param name="channel">CAN channel to add </param>
        /// <param name="can_msgtype">CAN frame </param>
        /// <param name="source_addr">ISO-TP source address</param>
        /// <param name="target_addr">ISO-TP target address</param>
        /// <param name="extension_addr">ISO-TP extension address</param>
        /// <param name="target_addr_func">ISO-TP target address for functional addressing</param>
        /// <param name="can_id">CAN ID used to communicate for physical addressing</param>
        /// <param name="can_id_flow_ctrl">Flow Control CAN ID used during physical addressing communication</param>
        /// <param name="can_id_func">CAN ID used to communicate for functionnal addressing</param>
        /// <param name="isotp_msgtype">ISOTP message's type</param>
        /// <param name="isotp_format">ISOTP message's format</param>
        /// <return>true on successful configuration</return>
        static bool initialize_mappings(cantp_handle channel, cantp_can_msgtype can_msgtype, UInt16 source_addr, UInt16 target_addr, byte extension_addr, UInt16 target_addr_func, UInt32 can_id, UInt32 can_id_flow_ctrl, UInt32 can_id_func, cantp_isotp_msgtype isotp_msgtype, cantp_isotp_format isotp_format)
        {
            bool result = true;
            // C# "new" allows to clean variables (it is common to leave can_tx_dl uninitialized which can lead to invalid 0xCC values)
            cantp_mapping mapping_phys_rx = new cantp_mapping();
            cantp_mapping mapping_phys_tx = new cantp_mapping();
            cantp_mapping mapping_func = new cantp_mapping();
            cantp_status status;

            // configure a mapping to transmit physical message
            mapping_phys_tx.can_id = can_id;
            mapping_phys_tx.can_id_flow_ctrl = can_id_flow_ctrl;
            mapping_phys_tx.can_msgtype = can_msgtype;
            mapping_phys_tx.netaddrinfo.format = isotp_format;
            mapping_phys_tx.netaddrinfo.msgtype = isotp_msgtype;
            mapping_phys_tx.netaddrinfo.target_type = cantp_isotp_addressing.PCANTP_ISOTP_ADDRESSING_PHYSICAL;
            mapping_phys_tx.netaddrinfo.source_addr = source_addr;
            mapping_phys_tx.netaddrinfo.target_addr = target_addr;
            mapping_phys_tx.netaddrinfo.extension_addr = extension_addr;
            status = CanTpApi.AddMapping_2016(channel, ref mapping_phys_tx);
            if (!CanTpApi.StatusIsOk_2016(status))
            {
                Console.WriteLine("Failed to configure mapping (can_id=0x{0:X}) (sts=0x{1:X})", mapping_phys_tx.can_id, status);
                result = false;
            }
            // configure another mapping to receive physical message (invert source and target in previous mapping)
            mapping_phys_rx = mapping_phys_tx;
            mapping_phys_rx.can_id = mapping_phys_tx.can_id_flow_ctrl;
            mapping_phys_rx.can_id_flow_ctrl = mapping_phys_tx.can_id;
            mapping_phys_rx.netaddrinfo.source_addr = mapping_phys_tx.netaddrinfo.target_addr;
            mapping_phys_rx.netaddrinfo.target_addr = mapping_phys_tx.netaddrinfo.source_addr;
            status = CanTpApi.AddMapping_2016(channel, ref mapping_phys_rx);
            if (!CanTpApi.StatusIsOk_2016(status))
            {
                Console.WriteLine("Failed to configure mapping (can_id=0x{0:X}) (sts=0x{1:X})", mapping_phys_rx.can_id, status);
                result = false;
            }
            if (can_id_func != 0xFFFFFFFF)
            {
                // configure another mapping to transmit functional message
                mapping_func = mapping_phys_tx;
                mapping_func.can_id = can_id_func;
                mapping_func.can_id_flow_ctrl = CanTpApi.PCANTP_MAPPING_FLOW_CTRL_NONE;
                mapping_func.netaddrinfo.target_type = cantp_isotp_addressing.PCANTP_ISOTP_ADDRESSING_FUNCTIONAL;
                mapping_func.netaddrinfo.target_addr = target_addr_func;
                status = CanTpApi.AddMapping_2016(channel, ref mapping_func);
                if (!CanTpApi.StatusIsOk_2016(status))
                {
                    Console.WriteLine("Failed to configure mapping (can_id=0x{0:X}) (sts=0x{1:X})", mapping_func.can_id, status);
                    result = false;
                }
            }
            return result;
        }

        /// <summary>Outputs a summary of a received message.</summary>
        /// <param name="message">The message.</param>
        /// <param name="timestamp">The timestamp of the message.</param>
        /// <param name="status">status of the Read function.</param>
        /// <returns>1 on success (2 if extra information is available within status), 0 if no message and -1 on error</returns>
        static int display_message(ref cantp_msg message, cantp_timestamp timestamp, cantp_status status)
        {
            String strLoopback = "";
            String strMsgType = "";
            UInt32 displayedDataLength = 0;
            int result;

            // check status
            result = CanTpApi.StatusIsOk_2016(status) ? 1 : -1;
            if (result != 1)
            {
                if (CanTpApi.StatusIsOk_2016(status, cantp_status.PCANTP_STATUS_NO_MESSAGE))
                {
                    // no message received ignore
                    result = 0;
                }
                else
                {
                    Console.WriteLine("Failed to read message (sts=0x{0:X}).", status);
                }
                return result;
            }
            // check bus errors within status
            if (!CanTpApi.StatusIsOk_2016(status, cantp_status.PCANTP_STATUS_OK, true))
            {
                Console.WriteLine("Status returned information/warnings (sts=0x{0:X}).", status);
                result = 2;
            }

            // get items of message
            cantp_msgflag flags = new cantp_msgflag();
            if (!CanTpApi.getFlags_2016(ref message, out flags))
                Console.WriteLine("Failed to get flags of message");
            cantp_netstatus netsta = new cantp_netstatus();
            if (!CanTpApi.getNetStatus_2016(ref message, out netsta))
                Console.WriteLine("Failed to get net status of message");
            uint len = 0;
            byte[] vals = null;
            if (!CanTpApi.getLength_2016(ref message, out len))
                Console.WriteLine("Failed to get length of message");
            else if (len > 0)
            {
                vals = new byte[len];
                if (!CanTpApi.getData_2016(ref message, 0, vals, (int)len))
                    Console.WriteLine("Failed to get data of message");
            }
            cantp_netaddrinfo netaddr = new cantp_netaddrinfo();
            if (!CanTpApi.getNetaddrinfo_2016(ref message, out netaddr))
                Console.WriteLine("Failed to get net addr info of message");

            // check if message has the loopback flag
            if ((flags & cantp_msgflag.PCANTP_MSGFLAG_LOOPBACK) != 0)
            {
                strLoopback = "loopback ";
            }

            // format message's type
            switch (message.type)
            {
                case cantp_msgtype.PCANTP_MSGTYPE_CAN:
                    strMsgType = "PCANTP_MSGTYPE_CAN";
                    break;
                case cantp_msgtype.PCANTP_MSGTYPE_CANFD:
                    strMsgType = "PCANTP_MSGTYPE_CANFD";
                    break;
                case cantp_msgtype.PCANTP_MSGTYPE_ISOTP:
                    strMsgType = "PCANTP_MSGTYPE_ISOTP";
                    break;
                case cantp_msgtype.PCANTP_MSGTYPE_NONE:
                    strMsgType = "PCANTP_MSGTYPE_NONE";
                    break;
                default:
                    strMsgType = "Unknown";
                    break;
            }
            // Only display data if network result is OK
            if (netsta == cantp_netstatus.PCANTP_NETSTATUS_OK)
            {
                displayedDataLength = len;
            }

            // Display generic information
            Console.WriteLine("\n{0} - Received {1}{2} message : canid=0x{3:X}, length={4} - result: 0x{5:X} - {6}",
                timestamp,
                strLoopback,
                strMsgType,
                message.can_info.can_id,
                len,
                netsta,
               netsta != cantp_netstatus.PCANTP_NETSTATUS_OK ? "ERROR !!!" : "OK !");

            if (message.type == cantp_msgtype.PCANTP_MSGTYPE_ISOTP)
            {
                String strPending = "Completed";
                // Limit displayed data if message is pending
                if (displayedDataLength > 0 && (netaddr.msgtype & cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_FLAG_INDICATION) != 0)
                {
                    displayedDataLength = (len > 7) ? 7 : len;
                    strPending = "/!\\ Pending";
                }
                // Display iso-tp message's information
                Console.WriteLine("\t{0} message from 0x{1:X} (to 0x{2:X}, with RA 0x{3:X})",
                    strPending,
                    netaddr.source_addr,
                    netaddr.target_addr,
                    netaddr.extension_addr);
            }

            if (displayedDataLength > 0)
            {
                StringBuilder buffer = new StringBuilder();
                // display data
                buffer.AppendFormat("\t\\-> Length: {0}, Data= ", len);
                for (uint i = 0; i < displayedDataLength; i++)
                {
                    buffer.AppendFormat("{0:X} ", vals[i]);
                }
                if (displayedDataLength != len)
                {
                    buffer.Append("...");
                }
                Console.WriteLine(buffer);
            }
            return result;
        }

        /// <summary>A function to transmit CAN-TP messages</summary>
        /// <param name="channel">channel</param>
        /// <param name="canIdType">canIdType</param>
        /// <param name="msgType">CAN-TP message Type (PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC)</param>
        /// <param name="formatType">Format addressing type (see PCANTP_ISOTP_FORMAT_xxx)</param>
        /// <returns>number of transmission errors</returns>
        static int transmit(cantp_handle channel,
            cantp_can_msgtype canIdType,
            cantp_isotp_msgtype msgType,
            cantp_isotp_format formatType,
            bool doCanFd, bool canFdInitialized)
        {
            UInt32 index;
            cantp_status status;
            cantp_msg message = new cantp_msg();
            cantp_can_msgtype can_msgtype = cantp_can_msgtype.PCANTP_CAN_MSGTYPE_STANDARD;
            cantp_netaddrinfo isotp_nai = new cantp_netaddrinfo();
            byte lMaxSize = 0;
            int nbErrPhys = 0;
            int nbErrFunc = 0;
            int sleepTime = 0;
            int nbMsgRead = 0;
            byte[] data = new byte[MESSAGE_MAX_LENGTH];

            // Output information message
            String lStr = "??";
            switch (formatType)
            {
                case cantp_isotp_format.PCANTP_ISOTP_FORMAT_ENHANCED:
                    lStr = "Enhanced";
                    break;
                case cantp_isotp_format.PCANTP_ISOTP_FORMAT_EXTENDED:
                    lStr = "Extended";
                    break;
                case cantp_isotp_format.PCANTP_ISOTP_FORMAT_FIXED_NORMAL:
                    lStr = "Fixed Normal";
                    break;
                case cantp_isotp_format.PCANTP_ISOTP_FORMAT_MIXED:
                    lStr = "Mixed";
                    break;
                case cantp_isotp_format.PCANTP_ISOTP_FORMAT_NORMAL:
                    lStr = "Normal";
                    break;
            }
            Console.WriteLine("\n\nTransmission of {0}, {1} message, {2} addressing format",
                ((canIdType & cantp_can_msgtype.PCANTP_CAN_MSGTYPE_EXTENDED) != 0) ? "29 BIT CAN ID" : "11 BIT CAN ID",
                msgType == cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC ? "Diagnostic" : "Remote diagnostic",
                lStr);
            if (USE_GETCH)
            {
                Console.WriteLine("\nPress <Enter> to continue...");
                Console.In.Read();
            }

            // initialize message
            isotp_nai.source_addr = N_SA;
            if (msgType == cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_REMOTE_DIAGNOSTIC)
                isotp_nai.extension_addr = N_RA;
            can_msgtype = canIdType;
            if (doCanFd)
            {
                can_msgtype |= cantp_can_msgtype.PCANTP_CAN_MSGTYPE_FD;
                can_msgtype |= cantp_can_msgtype.PCANTP_CAN_MSGTYPE_BRS;
            }
            isotp_nai.msgtype = msgType;
            isotp_nai.format = formatType;
            // prepare a buffer that will be used as a reference to initialize the data to transmit
            for (int i = 0; i < MESSAGE_MAX_LENGTH; ++i)
                data[i] = (byte)i;

            if (TEST_PHYS_TRANSMISSION)
            {
                // Transmitting data using Physical addressing
                Console.WriteLine("\nTransmitting data using Physical addressing\n");
                g_timeReference = DateTime.Now;
                isotp_nai.target_addr = N_TA_PHYS;
                isotp_nai.target_type = cantp_isotp_addressing.PCANTP_ISOTP_ADDRESSING_PHYSICAL;
                // each loop transmits a message with a data length incremented by 1 and reads its loppbacl confirmation
                for (index = 1; index <= MESSAGE_MAX_LENGTH; index++)
                {
                    // allocate a new ISOTP message
                    status = CanTpApi.MsgDataAlloc_2016(out message, cantp_msgtype.PCANTP_MSGTYPE_ISOTP);
                    if (CanTpApi.StatusIsOk_2016(status))
                    {
                        // initialize ISOTP message
                        status = CanTpApi.MsgDataInit_2016(out message, CanTpApi.PCANTP_CAN_ID_DEFINED_BY_NAI, can_msgtype, index, data, ref isotp_nai);
                        if (CanTpApi.StatusIsOk_2016(status))
                        {
                            // write message
                            status = CanTpApi.Write_2016(channel, ref message);
                            if (CanTpApi.StatusIsOk_2016(status))
                            {
                                Console.WriteLine("\nSuccessfully queued ISOTP message: Length {0} (sts=0x{1:X}).", index, (uint)status);
                                // try to read loopback confirmation 
                                nbMsgRead = 0;
                                nbErrPhys += read_segmented_message(channel, ref nbMsgRead);
                                if (nbMsgRead != 1)
                                {
                                    Console.WriteLine("Received unexpected messages ({0} instead of 1)", nbMsgRead);
                                }
                            }
                            else
                            {
                                Console.WriteLine("\nFailed to write ISOTP message: Length {0} (sts=0x{1:X}).", index, (uint)status);
                                nbErrPhys++;
                            }
                        }
                        else
                        {
                            Console.WriteLine("\nFailed to initialize ISOTP message: Length {0} (sts=0x{1:X}).", index, (uint)status);
                            nbErrPhys++;
                        }
                        // release message
                        status = CanTpApi.MsgDataFree_2016(ref message);
                        if (!CanTpApi.StatusIsOk_2016(status))
                        {
                            Console.WriteLine("Failed to deallocate message (sts=0x{0:X}).", status);
                            nbErrPhys++;
                        }

                    }
                    else
                    {
                        Console.WriteLine("Failed to allocate message (sts=0x{0:X}).", status);
                        nbErrPhys++;
                    }

                    System.Threading.Thread.Sleep(sleepTime);

                    if (index == 100)
                    {
                        // skip and jump to maximum data length
                        index = MESSAGE_MAX_LENGTH - 1;
                    }
                }
            }

            if (TEST_BURST_TRANSMISSION)
            {
                Console.WriteLine("\nTransmitting data using Physical addressing in 'BURST' mode\n");

                g_timeReference = DateTime.Now;
                isotp_nai.target_addr = N_TA_PHYS;
                isotp_nai.target_type = cantp_isotp_addressing.PCANTP_ISOTP_ADDRESSING_PHYSICAL;
                // Transmitting data using Physical addressing WITHOUT waiting for response confirmation
                // each loop transmits a message with a data length incremented by 1
                int nbMsgSent = 0;
                for (index = 1; index <= MESSAGE_MAX_LENGTH; index++)
                {
                    // allocate a new ISOTP message
                    status = CanTpApi.MsgDataAlloc_2016(out message, cantp_msgtype.PCANTP_MSGTYPE_ISOTP);
                    if (CanTpApi.StatusIsOk_2016(status))
                    {
                        // initialize ISOTP message
                        status = CanTpApi.MsgDataInit_2016(out message, CanTpApi.PCANTP_CAN_ID_DEFINED_BY_NAI, can_msgtype, index, data, ref isotp_nai);
                        if (CanTpApi.StatusIsOk_2016(status))
                        {
                            // write message
                            status = CanTpApi.Write_2016(channel, ref message);
                            if (CanTpApi.StatusIsOk_2016(status))
                            {
                                nbMsgSent++;
                                Console.WriteLine("\nSuccessfully queued ISOTP message: Length {0} (sts=0x{1:X}).", index, (uint)status);
                            }
                            else
                            {
                                Console.WriteLine("\nFailed to write ISOTP message: Length {0} (sts=0x{1:X}).", index, (uint)status);
                                nbErrPhys++;
                            }
                        }
                        else
                        {
                            Console.WriteLine("\nFailed to initialize ISOTP message: Length {0} (sts=0x{1:X}).", index, (uint)status);
                            nbErrPhys++;
                        }
                        // release message
                        status = CanTpApi.MsgDataFree_2016(ref message);
                        if (!CanTpApi.StatusIsOk_2016(status))
                        {
                            Console.WriteLine("Failed to deallocate message (sts=0x{0:X}).", status);
                            nbErrPhys++;
                        }
                    }
                    else
                    {
                        Console.WriteLine("Failed to allocate message (sts=0x{0:X}).", status);
                        nbErrPhys++;
                    }


                    if (index == 10)
                    {  // skip and try bigger LEN
                        index = 2000;
                    }
                    if (index == 2005)
                    { // skip
                        break;
                    }
                }
                Console.WriteLine("\nReading confirmation for each transmitted messages in 'BURST' mode\n");


                for (nbMsgRead = 0; nbMsgRead < nbMsgSent;)
                {
                    // try to read loopback confirmation 
                    nbErrPhys += read_segmented_message(channel, ref nbMsgRead);
                }
                CanTpApi.Reset_2016(channel);
            }

            if (TEST_FUNC_TRANSMISSION)
            {
                // Transmitting data using Functional addressing 
                Console.WriteLine("\nTransmitting data using Functional addressing\n");


                g_timeReference = DateTime.Now;
                isotp_nai.target_addr = N_TA_FUNC;
                isotp_nai.target_type = cantp_isotp_addressing.PCANTP_ISOTP_ADDRESSING_FUNCTIONAL;
                // Reminder: Functional addressing shall only be supported 
                // for Single Frame communication.
                // Thus maximum size depends on Format Addressing Type.
                switch (formatType)
                {
                    case cantp_isotp_format.PCANTP_ISOTP_FORMAT_EXTENDED:
                    case cantp_isotp_format.PCANTP_ISOTP_FORMAT_MIXED:
                        lMaxSize = (byte)(((doCanFd && canFdInitialized) ? CAN_TX_DL_VALUE : CanTpApi.PCANTP_MAX_LENGTH_CAN_STANDARD) - 2);
                        break;
                    default:
                        lMaxSize = (byte)(((doCanFd && canFdInitialized) ? CAN_TX_DL_VALUE : CanTpApi.PCANTP_MAX_LENGTH_CAN_STANDARD) - 1);
                        break;
                }
                // each loop transmits a message with a data length incremented by 1 and reads its loppbacl confirmation
                for (index = 1; index <= lMaxSize; index++)
                {
                    // allocate a new ISOTP message
                    status = CanTpApi.MsgDataAlloc_2016(out message, cantp_msgtype.PCANTP_MSGTYPE_ISOTP);
                    if (CanTpApi.StatusIsOk_2016(status))
                    {
                        // initialize ISOTP message
                        status = CanTpApi.MsgDataInit_2016(out message, CanTpApi.PCANTP_CAN_ID_DEFINED_BY_NAI, can_msgtype, index, data, ref isotp_nai);
                        if (CanTpApi.StatusIsOk_2016(status))
                        {
                            // write message
                            status = CanTpApi.Write_2016(channel, ref message);
                            if (CanTpApi.StatusIsOk_2016(status))
                            {
                                Console.WriteLine("\nSuccessfully queued ISOTP message: Length {0} (sts=0x{1:X}).", index, (uint)status);
                                // try to read loopback confirmation 
                                nbMsgRead = 0;
                                nbErrFunc += read_segmented_message(channel, ref nbMsgRead);
                                if (nbMsgRead != 1)
                                {
                                    Console.WriteLine("Received unexpected messages ({0} instead of 1)", nbMsgRead);
                                }
                            }
                            else
                            {
                                Console.WriteLine("\nFailed to write ISOTP message: Length {0} (sts=0x{1:X}).", index, (uint)status);
                                nbErrFunc++;
                            }
                        }
                        else
                        {
                            Console.WriteLine("\nFailed to initialize ISOTP message: Length {0} (sts=0x{1:X}).", index, (uint)status);
                            nbErrFunc++;
                        }
                        // release message
                        status = CanTpApi.MsgDataFree_2016(ref message);
                        if (!CanTpApi.StatusIsOk_2016(status))
                        {
                            Console.WriteLine("Failed to deallocate message (sts=0x{0:X}).", status);
                            nbErrFunc++;
                        }

                    }
                    else
                    {
                        Console.WriteLine("Failed to allocate message (sts=0x{0:X}).", status);
                        nbErrFunc++;
                    }

                    System.Threading.Thread.Sleep(sleepTime);
                }
            }

            if (nbErrPhys > 0 || nbErrFunc > 0)
            {
                if (nbErrPhys > 0)
                {
                    Console.WriteLine("\nERROR : {0} errors occured.\n", nbErrPhys);
                }
                if (nbErrFunc > 0)
                {
                    Console.WriteLine("\nERROR : {0} errors occured.\n", nbErrFunc);
                }
                if (USE_GETCH)
                {
                    Console.WriteLine("\nPress <Enter> to continue...");
                    Console.ReadKey();
                }
            }
            else
            {
                Console.WriteLine("\nTransmissions successful !\n");
            }

            return nbErrFunc + nbErrPhys;
        }

        /// <summary>Reads a message.</summary>
        /// <param name="channel">The channel to read the message from.</param>
        /// <param name="nbMsgRead">Buffer to store the number of message actually read.</param>
        /// <returns>number of errors</returns>
        static int read_segmented_message(cantp_handle channel, ref int nbMsgRead)
        {
            cantp_msg message;
            cantp_status status;
            cantp_timestamp timestamp;
            int doLoop;

            int nbErr = 0;

            do
            {
                doLoop = 0;
                // initialize message
                status = CanTpApi.MsgDataAlloc_2016(out message, cantp_msgtype.PCANTP_MSGTYPE_NONE);
                if (!CanTpApi.StatusIsOk_2016(status))
                {
                    Console.WriteLine("Failed to allocate message (sts=0x{0X}).", status);
                    ++nbErr;
                    continue;
                }
                // retrieve any message from Rx queue
                status = CanTpApi.Read_2016(channel, out message, out timestamp);
                // display_message returns true when a message is available
                doLoop = display_message(ref message, timestamp, status);
                if (doLoop > 0)
                {
                    // get items of message                  
                    cantp_netstatus netsta = new cantp_netstatus();
                    if (!CanTpApi.getNetStatus_2016(ref message, out netsta))
                        Console.WriteLine("Failed to get net status of message");
                    cantp_netaddrinfo netaddr = new cantp_netaddrinfo();
                    if (!CanTpApi.getNetaddrinfo_2016(ref message, out netaddr))
                        Console.WriteLine("Failed to get net addr info of message");

                    // update error counter if a message was received but a network error occured
                    if (netsta != cantp_netstatus.PCANTP_NETSTATUS_OK)
                    {
                        nbErr++;
                    }
                    if (message.type == cantp_msgtype.PCANTP_MSGTYPE_ISOTP &&
                        (netaddr.msgtype & cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_FLAG_INDICATION) != 0)
                    {
                        // wait and read the full segmented message
                        nbErr += read_segmented_message(channel, ref nbMsgRead);
                    }
                    else
                        nbMsgRead++;
                }
                else
                {
                    if (doLoop == 0)
                    {
                        // no message read, wait for it
                        System.Threading.Thread.Sleep(10);
                    }
                }
                // release message
                status = CanTpApi.MsgDataFree_2016(ref message);
                if (!CanTpApi.StatusIsOk_2016(status))
                {
                    Console.WriteLine("Failed to deallocate message (sts=0x{0:X}).", status);
                }
            } while (doLoop == 0);

            return nbErr;
        }

    }
}
