// server_ISO15765-2_normal_addressing.cpp: Defines the entry point for the console application.
//

#include "stdafx.h"

#include "PCAN-ISO-TP_2016.h"

#define STMIN_600US 0xF6
#define OK_KO(test) (test)?"OK":"KO"
#define STATUS_OK_KO(test) OK_KO(CANTP_StatusIsOk_2016(test, PCANTP_STATUS_OK, false))

/// <summary>Helper: print memory</summary>
/// <param name="buf">Pointer on the memory to print</param>
/// <param name="len">Number of byte to print</param>
void print_data(const uint8_t* buf, int len)
{
	printf("\t Byte:");
	for (int i = 0; i < len; i++)
	{
		printf(" 0x%02X", buf[i]);
	}
	printf("\n\t Ascii: \"");
	for (int i = 0; i < len; i++)
	{
		printf("%c", buf[i]);
	}
	printf("\"\n");
}

/// <summary>Entry point of the program, start a small CAN ISO TP read/write example</summary>
/// <returns>By convention, return success.</returns>
int main()
{
	// Local variables
	cantp_status res;
	cantp_status read_status;
	char buffer[500];
	uint32_t STmin;
	cantp_msg rx_msg;
	cantp_mapping mapping;
	HANDLE receive_event;
	uint64_t null_handle = 0;
	int wait_result;
	BOOL res_b;
	cantp_handle server_handle;
	BOOL stop;
	int keyboard_res;

	// Initialize variables
	server_handle = PCANTP_HANDLE_USBBUS2;	// TODO: modify the value according to your available PCAN devices.
	memset(&rx_msg, 0, sizeof(cantp_msg));
	memset(&mapping, 0, sizeof(cantp_mapping));

	// Print version informations
	CANTP_GetValue_2016(PCANTP_HANDLE_NONEBUS, PCANTP_PARAMETER_API_VERSION, buffer, 500);
	printf("PCAN-ISO-TP API Version: %s\n", buffer);

	// Initialize channel: CAN2.0 - 500Kbit/s
	res = CANTP_Initialize_2016(server_handle, PCANTP_BAUDRATE_500K, (cantp_hwtype)0, 0, 0);
	printf("Initialize: %s\n", STATUS_OK_KO(res));

	// Create a receive event on receiver
	receive_event = CreateEvent(NULL, FALSE, FALSE, NULL);
	res = CANTP_SetValue_2016(server_handle, PCANTP_PARAMETER_RECEIVE_EVENT, &receive_event, sizeof(receive_event));
	printf("Create a receive event: %s\n", STATUS_OK_KO(res));

	// Change STmin value to 600us
	STmin = STMIN_600US;
	res = CANTP_SetValue_2016(server_handle, PCANTP_PARAMETER_SEPARATION_TIME, &STmin, sizeof(STmin));
	printf("Set STMIN = 600us: %s\n", STATUS_OK_KO(res));

	// Create a simple physical mapping: 
	//    - Source 0xF1 (client), target 0x01 (server), CAN id 0xA1, CAN ID flow control 0xA2
	//    - Diagnostic message in a classic format
	mapping.can_id = 0xA1;
	mapping.can_id_flow_ctrl = 0xA2;
	mapping.can_msgtype = PCANTP_CAN_MSGTYPE_STANDARD;
	mapping.netaddrinfo.extension_addr = 0x00;
	mapping.netaddrinfo.format = PCANTP_ISOTP_FORMAT_NORMAL;
	mapping.netaddrinfo.msgtype = PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC;
	mapping.netaddrinfo.source_addr = 0xF1;
	mapping.netaddrinfo.target_addr = 0x01;
	mapping.netaddrinfo.target_type = PCANTP_ISOTP_ADDRESSING_PHYSICAL;

	// Add mapping on channel
	res = CANTP_AddMapping_2016(server_handle, &mapping);
	printf("Add a simple mapping: %s\n", STATUS_OK_KO(res));
	// NOTE: this is a one-way mapping, the server is only capable of 
	//	receiving segmented or unsegmented ISO-TP message using "0xA1" CAN ID
	//	to be able to send ISO-TP message, another mapping is required.

	// Read while user do not press Q
	printf("Start listening, press Q to quit.\n");
	stop = false;
	while (!stop) {
		// Wait a receive event on receiver
		//	note: timeout is used to check keyboard hit.
		wait_result = WaitForSingleObject(receive_event, 1000);
		// If we receive something, read the message
		if (wait_result == WAIT_OBJECT_0) {

			do {
				// Allocate rx message
				res = CANTP_MsgDataAlloc_2016(&rx_msg, PCANTP_MSGTYPE_NONE);
				printf("Allocate rx message: %s\n", STATUS_OK_KO(res));

				// Read first available message (no filtering based on message's type is set): 
				read_status = CANTP_Read_2016(server_handle, &rx_msg, NULL, PCANTP_MSGTYPE_NONE);
				printf("Try to read a message: %s\n", STATUS_OK_KO(read_status));

				// If we read something, print the message
				if (CANTP_StatusIsOk_2016(read_status, PCANTP_STATUS_OK, false)) {
					printf("\nPrint received data:\n");
					print_data(rx_msg.msgdata.any->data, rx_msg.msgdata.any->length);
					printf("\n");
				}

				// Free message
				res = CANTP_MsgDataFree_2016(&rx_msg);
				printf("Free rx message: %s\n", STATUS_OK_KO(res));
			} while (read_status != PCANTP_STATUS_NO_MESSAGE);
		}

		// Quit when user press Q
		if (_kbhit()) {
			keyboard_res = _getch();
			if (keyboard_res == 'Q' || keyboard_res == 'q') {
				stop = true;
			}
		}
	}

	// Close receive event
	res = CANTP_SetValue_2016(server_handle, PCANTP_PARAMETER_RECEIVE_EVENT, &null_handle, sizeof(HANDLE));
	printf("Stop receive event: %s\n", STATUS_OK_KO(res));
	res_b = CloseHandle(receive_event);
	printf("Close receive event: %s\n", OK_KO(res_b));

	// Uninitialize
	res = CANTP_Uninitialize_2016(server_handle);
	printf("Uninitialize: %s\n", STATUS_OK_KO(res));

	// Exit
	system("PAUSE");
	return EXIT_SUCCESS;
}

