// isotp_segmented_read_write.cpp: Defines the entry point for the console application.
//

#include "stdafx.h"

#include "PCAN-ISO-TP_2016.h"
#include "heavy_data.h"

#define STMIN_600US 0xF6
#define OK_KO(test) (test)?"OK":"KO"
#define STATUS_OK_KO(test) OK_KO(CANTP_StatusIsOk_2016(test, PCANTP_STATUS_OK, false))

/// <summary>Return the name of the channel</summary>
/// <param name="value">Channel handle</param>
/// <returns>String containing the name of the channel</returns>
const char* cantp_handle_toShortString(cantp_handle value)
{
	const char *res;
	switch (value) {
	case PCANTP_HANDLE_NONEBUS: res = "NONEBUS"; break;
	case PCANTP_HANDLE_ISABUS1: res = "ISA1"; break;
	case PCANTP_HANDLE_ISABUS2: res = "ISA2"; break;
	case PCANTP_HANDLE_ISABUS3: res = "ISA3"; break;
	case PCANTP_HANDLE_ISABUS4: res = "ISA4"; break;
	case PCANTP_HANDLE_ISABUS5: res = "ISA5"; break;
	case PCANTP_HANDLE_ISABUS6: res = "ISA6"; break;
	case PCANTP_HANDLE_ISABUS7: res = "ISA7"; break;
	case PCANTP_HANDLE_ISABUS8: res = "ISA8"; break;
	case PCANTP_HANDLE_DNGBUS1: res = "DNG1"; break;
	case PCANTP_HANDLE_PCIBUS1: res = "PCI1"; break;
	case PCANTP_HANDLE_PCIBUS2: res = "PCI2"; break;
	case PCANTP_HANDLE_PCIBUS3: res = "PCI3"; break;
	case PCANTP_HANDLE_PCIBUS4: res = "PCI4"; break;
	case PCANTP_HANDLE_PCIBUS5: res = "PCI5"; break;
	case PCANTP_HANDLE_PCIBUS6: res = "PCI6"; break;
	case PCANTP_HANDLE_PCIBUS7: res = "PCI7"; break;
	case PCANTP_HANDLE_PCIBUS8: res = "PCI8"; break;
	case PCANTP_HANDLE_PCIBUS9: res = "PCI9"; break;
	case PCANTP_HANDLE_PCIBUS10: res = "PCI10"; break;
	case PCANTP_HANDLE_PCIBUS11: res = "PCI11"; break;
	case PCANTP_HANDLE_PCIBUS12: res = "PCI12"; break;
	case PCANTP_HANDLE_PCIBUS13: res = "PCI13"; break;
	case PCANTP_HANDLE_PCIBUS14: res = "PCI14"; break;
	case PCANTP_HANDLE_PCIBUS15: res = "PCI15"; break;
	case PCANTP_HANDLE_PCIBUS16: res = "PCI16"; break;
	case PCANTP_HANDLE_USBBUS1: res = "USB1"; break;
	case PCANTP_HANDLE_USBBUS2: res = "USB2"; break;
	case PCANTP_HANDLE_USBBUS3: res = "USB3"; break;
	case PCANTP_HANDLE_USBBUS4: res = "USB4"; break;
	case PCANTP_HANDLE_USBBUS5: res = "USB5"; break;
	case PCANTP_HANDLE_USBBUS6: res = "USB6"; break;
	case PCANTP_HANDLE_USBBUS7: res = "USB7"; break;
	case PCANTP_HANDLE_USBBUS8: res = "USB8"; break;
	case PCANTP_HANDLE_USBBUS9: res = "USB9"; break;
	case PCANTP_HANDLE_USBBUS10: res = "USB10"; break;
	case PCANTP_HANDLE_USBBUS11: res = "USB11"; break;
	case PCANTP_HANDLE_USBBUS12: res = "USB12"; break;
	case PCANTP_HANDLE_USBBUS13: res = "USB13"; break;
	case PCANTP_HANDLE_USBBUS14: res = "USB14"; break;
	case PCANTP_HANDLE_USBBUS15: res = "USB15"; break;
	case PCANTP_HANDLE_USBBUS16: res = "USB16"; break;
	case PCANTP_HANDLE_PCCBUS1: res = "PCC1"; break;
	case PCANTP_HANDLE_PCCBUS2: res = "PCC2"; break;
	case PCANTP_HANDLE_LANBUS1: res = "LAN1"; break;
	case PCANTP_HANDLE_LANBUS2: res = "LAN2"; break;
	case PCANTP_HANDLE_LANBUS3: res = "LAN3"; break;
	case PCANTP_HANDLE_LANBUS4: res = "LAN4"; break;
	case PCANTP_HANDLE_LANBUS5: res = "LAN5"; break;
	case PCANTP_HANDLE_LANBUS6: res = "LAN6"; break;
	case PCANTP_HANDLE_LANBUS7: res = "LAN7"; break;
	case PCANTP_HANDLE_LANBUS8: res = "LAN8"; break;
	case PCANTP_HANDLE_LANBUS9: res = "LAN9"; break;
	case PCANTP_HANDLE_LANBUS10: res = "LAN10"; break;
	case PCANTP_HANDLE_LANBUS11: res = "LAN11"; break;
	case PCANTP_HANDLE_LANBUS12: res = "LAN12"; break;
	case PCANTP_HANDLE_LANBUS13: res = "LAN13"; break;
	case PCANTP_HANDLE_LANBUS14: res = "LAN14"; break;
	case PCANTP_HANDLE_LANBUS15: res = "LAN15"; break;
	case PCANTP_HANDLE_LANBUS16: res = "LAN16"; break;
	default: res = "UNKNOWN"; break;
	}
	return res;
}

/// <summary>Write a uint8_t tab in a binary file</summary>
/// <param name="file_path">Output file path</param>
/// <param name="data">Data to write</param>
/// <param name="len">Data length (in byte)</param>
void write_binary_output_file(const char *file_path, const uint8_t *data, size_t len) {
	size_t i;
	FILE *file;
	uint8_t lsb;
	uint8_t msb;

	fopen_s(&file, file_path, "wb");
	if (file != NULL) {
		for (i = 0; i < len; i += 2) {
			lsb = data[i];
			msb = data[i + 1];
			fwrite(&msb, 1, 1, file);
			fwrite(&lsb, 1, 1, file);
		}
		fclose(file);
	}
}

/// <summary>Wait and read single message</summary>
/// <param name="handle">Channel handle</param>
/// <param name="msg_buffer">Received message (out)</param>
/// <returns>Status</returns>
cantp_status wait_and_read_msg(cantp_handle handle, cantp_msg *msg_buffer) {
	int wait_result;
	cantp_status res;
	HANDLE receive_event;
	cantp_msgprogress progress;

	// Initialize local variables 
	receive_event = NULL;
	memset(&progress, 0, sizeof(cantp_msgprogress));

	// Get receive event if any
	res = CANTP_GetValue_2016(handle, PCANTP_PARAMETER_RECEIVE_EVENT, &receive_event, sizeof(receive_event));
	printf("Get receive event on %s: %s\n", cantp_handle_toShortString(handle), STATUS_OK_KO(res));

	// Wait receive event and get a message
	if (CANTP_StatusIsOk_2016(res, PCANTP_STATUS_OK, false) && receive_event != NULL) {
		res = PCANTP_STATUS_NO_MESSAGE;
		wait_result = WaitForSingleObject(receive_event, 100);
		// If we receive something, read the message
		if (wait_result == WAIT_OBJECT_0) {
			res = CANTP_Read_2016(handle, msg_buffer, NULL, PCANTP_MSGTYPE_NONE);
			printf("Wait a message on %s: %s\n", cantp_handle_toShortString(handle), STATUS_OK_KO(res));

			// Check if the message is totaly received (only in ISOTP mode)
			if (CANTP_StatusIsOk_2016(res, PCANTP_STATUS_OK, false)
				&& ((PCANTP_MSGTYPE_ISOTP & msg_buffer->type) == PCANTP_MSGTYPE_ISOTP)
				&& ((msg_buffer->msgdata.isotp->netaddrinfo.msgtype & PCANTP_ISOTP_MSGTYPE_FLAG_INDICATION_RX) == PCANTP_ISOTP_MSGTYPE_FLAG_INDICATION_RX)) {

				// The message is being received, wait and show progress
				do {
					Sleep(300);
					res = CANTP_GetMsgProgress_2016(handle, msg_buffer, PCANTP_MSGDIRECTION_RX, &progress);
					printf("RX Progress on %s = %d%%: %s\n", cantp_handle_toShortString(handle), progress.percentage, STATUS_OK_KO(res));
				} while (progress.state == PCANTP_MSGPROGRESS_STATE_PROCESSING);

				// The message is received, read it
				res = CANTP_Read_2016(handle, msg_buffer, NULL, PCANTP_MSGTYPE_NONE);
			}
		}
	}

	return res;
}

/// <summary>
///   Entry point of the program, start a small CAN ISO TP read/write example
///     -> Send a message containing Peak logo 
///     -> Receive message, wait complete reception (show progress)
///     -> Show loopback message and received message (Peak logo)
/// </summary>
/// <returns>By convention, return success.</returns>
int main()
{
	// Local variables
	cantp_handle transmitter_handle, receiver_handle;
	cantp_status res;
	char buffer[500];
	uint32_t STmin;
	cantp_msg tx_msg, loopback_msg, rx_msg;
	cantp_mapping mapping, reverse_mapping;
	HANDLE transmitter_receive_event;
	HANDLE receiver_receive_event;
	uint64_t null_handle;
	bool res_b;
	bool fileTxOk, fileRxOk;

	// Initialize structs 
	memset(&tx_msg, 0, sizeof(cantp_msg));
	memset(&loopback_msg, 0, sizeof(cantp_msg));
	memset(&rx_msg, 0, sizeof(cantp_msg));
	memset(&mapping, 0, sizeof(cantp_mapping));
	memset(&reverse_mapping, 0, sizeof(cantp_mapping));
	null_handle = 0;

	// Define TX and RX channels
	transmitter_handle = PCANTP_HANDLE_USBBUS1; // TODO: modify the value according to your available PCAN devices.
	receiver_handle = PCANTP_HANDLE_USBBUS2; // TODO: modify the value according to your available PCAN devices.

	// Print version informations
	CANTP_GetValue_2016(PCANTP_HANDLE_NONEBUS, PCANTP_PARAMETER_API_VERSION, buffer, 500);
	printf("PCAN-ISO-TP API Version: %s\n", buffer);

	// Initialize channels: CAN2.0 - 500Kbit/s
	res = CANTP_Initialize_2016(transmitter_handle, PCANTP_BAUDRATE_500K, (cantp_hwtype)0, 0, 0);
	printf("Initialize channel %s: %s\n", cantp_handle_toShortString(transmitter_handle), STATUS_OK_KO(res));
	res = CANTP_Initialize_2016(receiver_handle, PCANTP_BAUDRATE_500K, (cantp_hwtype)0, 0, 0);
	printf("Initialize channel %s: %s\n", cantp_handle_toShortString(receiver_handle), STATUS_OK_KO(res));

	// Change STmin value to 600us
	STmin = STMIN_600US;
	res = CANTP_SetValue_2016(transmitter_handle, PCANTP_PARAMETER_SEPARATION_TIME, &STmin, sizeof(STmin));
	printf("Set STMIN = 600us on channel %s: %s\n", cantp_handle_toShortString(transmitter_handle), STATUS_OK_KO(res));
	res = CANTP_SetValue_2016(receiver_handle, PCANTP_PARAMETER_SEPARATION_TIME, &STmin, sizeof(STmin));
	printf("Set STMIN = 600us on channel %s: %s\n", cantp_handle_toShortString(receiver_handle), STATUS_OK_KO(res));

	// Create a receive event on transmitter (for loopback message)
	transmitter_receive_event = CreateEvent(NULL, FALSE, FALSE, NULL);
	res = CANTP_SetValue_2016(transmitter_handle, PCANTP_PARAMETER_RECEIVE_EVENT, &transmitter_receive_event, sizeof(transmitter_receive_event));
	printf("Create a receive event on channel %s: %s\n", cantp_handle_toShortString(transmitter_handle), STATUS_OK_KO(res));

	// Create a receive event on receiver
	receiver_receive_event = CreateEvent(NULL, FALSE, FALSE, NULL);
	res = CANTP_SetValue_2016(receiver_handle, PCANTP_PARAMETER_RECEIVE_EVENT, &receiver_receive_event, sizeof(receiver_receive_event));
	printf("Create a receive event on channel %s: %s\n", cantp_handle_toShortString(receiver_handle), STATUS_OK_KO(res));

	// Allocate messages
	res = CANTP_MsgDataAlloc_2016(&tx_msg, PCANTP_MSGTYPE_ISOTP);
	printf("Allocate tx message: %s\n", STATUS_OK_KO(res));
	res = CANTP_MsgDataAlloc_2016(&loopback_msg, PCANTP_MSGTYPE_NONE);
	printf("Allocate loopback message: %s\n", STATUS_OK_KO(res));
	res = CANTP_MsgDataAlloc_2016(&rx_msg, PCANTP_MSGTYPE_NONE);
	printf("Allocate rx message: %s\n", STATUS_OK_KO(res));

	// Create a simple physical mapping: 
	//    - Source 0xF1 (transmitter), target 0x01 (receiver), CAN id 0xA1, CAN ID flow control 0xA2
	//    - Diagnostic message in a classic format
	mapping.can_id = 0xA1;
	mapping.can_id_flow_ctrl = 0xA2;
	mapping.can_msgtype = PCANTP_CAN_MSGTYPE_STANDARD;
	mapping.can_tx_dlc = 0x0;
	mapping.netaddrinfo.extension_addr = 0x00;
	mapping.netaddrinfo.format = PCANTP_ISOTP_FORMAT_NORMAL;
	mapping.netaddrinfo.msgtype = PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC;
	mapping.netaddrinfo.source_addr = 0xF1;
	mapping.netaddrinfo.target_addr = 0x01;
	mapping.netaddrinfo.target_type = PCANTP_ISOTP_ADDRESSING_PHYSICAL;

	// Create the associated reversed mapping: 
	reverse_mapping = mapping;
	reverse_mapping.can_id = mapping.can_id_flow_ctrl;
	reverse_mapping.can_id_flow_ctrl = mapping.can_id;
	reverse_mapping.netaddrinfo.source_addr = mapping.netaddrinfo.target_addr;
	reverse_mapping.netaddrinfo.target_addr = mapping.netaddrinfo.source_addr;

	// Add mappings on channels
	res = CANTP_AddMapping_2016(transmitter_handle, &mapping);
	printf("Add a simple mapping on channel %s: %s\n", cantp_handle_toShortString(transmitter_handle), STATUS_OK_KO(res));
	res = CANTP_AddMapping_2016(transmitter_handle, &reverse_mapping);
	printf("Add the reverse mapping on channel %s: %s\n", cantp_handle_toShortString(transmitter_handle), STATUS_OK_KO(res));
	res = CANTP_AddMapping_2016(receiver_handle, &mapping);
	printf("Add a simple mapping on channel %s: %s\n", cantp_handle_toShortString(receiver_handle), STATUS_OK_KO(res));
	res = CANTP_AddMapping_2016(receiver_handle, &reverse_mapping);
	printf("Add the reverse mapping on channel %s: %s\n", cantp_handle_toShortString(receiver_handle), STATUS_OK_KO(res));

	// Initialize Tx message containing a heavy data
	res = CANTP_MsgDataInit_2016(&tx_msg, PCANTP_CAN_ID_DEFINED_BY_NAI, mapping.can_msgtype, HEAVY_DATA_SIZE, HEAVY_DATA, &mapping.netaddrinfo);
	printf("Initialize tx message: %s\n", STATUS_OK_KO(res));

	// Write message
	res = CANTP_Write_2016(transmitter_handle, &tx_msg);
	printf("Write message on channel %s: %s\n", cantp_handle_toShortString(transmitter_handle), STATUS_OK_KO(res));

	// Read loopback message and save it 
	res = wait_and_read_msg(transmitter_handle, &loopback_msg);

	printf("Read loopback message on channel %s: %s\n", cantp_handle_toShortString(transmitter_handle), STATUS_OK_KO(res));
	fileTxOk = CANTP_StatusIsOk_2016(res, PCANTP_STATUS_OK, false);
	if (fileTxOk) {
		write_binary_output_file("loopback_message.png", loopback_msg.msgdata.any->data, loopback_msg.msgdata.any->length);
	}

	// Read received message and save it
	res = wait_and_read_msg(receiver_handle, &rx_msg);
	printf("Read message on channel %s: %s\n", cantp_handle_toShortString(receiver_handle), STATUS_OK_KO(res));
	fileRxOk = CANTP_StatusIsOk_2016(res, PCANTP_STATUS_OK, false);
	if (fileRxOk) {
		write_binary_output_file("received_message.png", rx_msg.msgdata.any->data, rx_msg.msgdata.any->length);
	}

	// Show & remove result: 
	if (fileTxOk) {
		printf("\n >>> Opening transmitted message (PEAK logo).\nClose opening application to continue...\n\n");
		system("loopback_message.png");
	}
	else {
		printf("\n >>> Failed to successfully transmit message (PEAK logo)!\n\n");
	}
	if (fileRxOk) {
		printf("\n >>> Opening received message (PEAK logo).\nClose opening application to continue...\n\n");
		system("received_message.png");
	}
	else {
		printf("\n >>> Failed to successfully receive message (PEAK logo)!\n\n");
	}

	// Free messages space
	res = CANTP_MsgDataFree_2016(&tx_msg);
	printf("Free tx message: %s\n", STATUS_OK_KO(res));
	res = CANTP_MsgDataFree_2016(&loopback_msg);
	printf("Free loopback message: %s\n", STATUS_OK_KO(res));
	res = CANTP_MsgDataFree_2016(&rx_msg);
	printf("Free rx message: %s\n", STATUS_OK_KO(res));

	// Close receive event on transmitter
	res = CANTP_SetValue_2016(transmitter_handle, PCANTP_PARAMETER_RECEIVE_EVENT, &null_handle, sizeof(HANDLE));
	printf("Stop receive event on transmitter: %s\n", STATUS_OK_KO(res));
	res_b = CloseHandle(transmitter_receive_event) == TRUE;
	printf("Close receive event: %s\n", OK_KO(res_b != false));

	// Close receive event on receiver
	res = CANTP_SetValue_2016(receiver_handle, PCANTP_PARAMETER_RECEIVE_EVENT, &null_handle, sizeof(HANDLE));
	printf("Stop receive event on receiver: %s\n", STATUS_OK_KO(res));
	res_b = CloseHandle(receiver_receive_event) == TRUE;
	printf("Close receive event: %s\n", OK_KO(res_b != false));

	// Uninitialize channels
	res = CANTP_Uninitialize_2016(transmitter_handle);
	printf("Uninitialize channel %s: %s\n", cantp_handle_toShortString(transmitter_handle), STATUS_OK_KO(res));
	res = CANTP_Uninitialize_2016(receiver_handle);
	printf("Uninitialize channel %s: %s\n", cantp_handle_toShortString(receiver_handle), STATUS_OK_KO(res));

	// Remove results and exit
	system("PAUSE");
	system("del loopback_message.png");
	system("del received_message.png");
	return EXIT_SUCCESS;
}