// pctp_server.cpp : Defines the entry point for the console application.
//

#include "stdafx.h"
#include <conio.h>

#include "PCAN-ISO-TP_2016.h"

#define USE_CAN_FD false
#define USE_EVENT false

#define N_SA		((uint8_t)0xF1)
#define N_TA_PHYS	((uint8_t)0x13)
#define N_TA_FUNC	((uint8_t)0x26)
#define N_RA		((uint8_t)0x52)

/// <summary>Configure mappings to handle a two-way communication and a functional communication (broadcast).</summary>
/// <param name="channel">CAN channel to add </param>
/// <param name="can_msgtype">CAN frame </param>
/// <param name="source_addr">ISO-TP source address</param>
/// <param name="target_addr">ISO-TP target address</param>
/// <param name="extension_addr">ISO-TP extension address</param>
/// <param name="target_addr_func">ISO-TP target address for functional addressing</param>
/// <param name="can_id">CAN ID used to communicate for physical addressing</param>
/// <param name="can_id_flow_ctrl">Flow Control CAN ID used during physical addressing communication</param>
/// <param name="can_id_func">CAN ID used to communicate for functionnal addressing</param>
/// <param name="isotp_msgtype">ISOTP message's type</param>
/// <param name="isotp_format">ISOTP message's format</param>
/// <return>true on successful configuration</return>
static bool initialize_mappings(cantp_handle channel, cantp_can_msgtype can_msgtype, uint16_t source_addr, uint16_t target_addr, uint8_t extension_addr, uint16_t target_addr_func, uint32_t can_id, uint32_t can_id_flow_ctrl, uint32_t can_id_func, cantp_isotp_msgtype isotp_msgtype, cantp_isotp_format isotp_format) {
	bool result = true;
	cantp_mapping mapping_phys_rx;
	cantp_mapping mapping_phys_tx;
	cantp_mapping mapping_func;
	cantp_status status;

	// clean variables (it is common to leave can_tx_dl uninitialized which can lead to invalid 0xCC values)
	memset(&mapping_phys_rx, 0, sizeof(mapping_phys_rx));
	memset(&mapping_phys_tx, 0, sizeof(mapping_phys_tx));
	memset(&mapping_func, 0, sizeof(mapping_func));
	// configure a mapping to transmit physical message
	mapping_phys_tx.can_id = can_id;
	mapping_phys_tx.can_id_flow_ctrl = can_id_flow_ctrl;
	mapping_phys_tx.can_msgtype = can_msgtype;
	mapping_phys_tx.netaddrinfo.format = isotp_format;
	mapping_phys_tx.netaddrinfo.msgtype = isotp_msgtype;
	mapping_phys_tx.netaddrinfo.target_type = PCANTP_ISOTP_ADDRESSING_PHYSICAL;
	mapping_phys_tx.netaddrinfo.source_addr = source_addr;
	mapping_phys_tx.netaddrinfo.target_addr = target_addr;
	mapping_phys_tx.netaddrinfo.extension_addr = extension_addr;
	status = CANTP_AddMapping_2016(channel, &mapping_phys_tx);
	if (!CANTP_StatusIsOk_2016(status)) {
		printf("Failed to configure mapping (can_id=0x%04X) (sts=0x%04X)\n", mapping_phys_tx.can_id, status);
		result = false;
	}
	// configure another mapping to receive physical message (invert source and target in previous mapping)
	mapping_phys_rx = mapping_phys_tx;
	mapping_phys_rx.can_id = mapping_phys_tx.can_id_flow_ctrl;
	mapping_phys_rx.can_id_flow_ctrl = mapping_phys_tx.can_id;
	mapping_phys_rx.netaddrinfo.source_addr = mapping_phys_tx.netaddrinfo.target_addr;
	mapping_phys_rx.netaddrinfo.target_addr = mapping_phys_tx.netaddrinfo.source_addr;
	status = CANTP_AddMapping_2016(channel, &mapping_phys_rx);
	if (!CANTP_StatusIsOk_2016(status)) {
		printf("Failed to configure mapping (can_id=0x%04X) (sts=0x%04X)\n", mapping_phys_rx.can_id, status);
		result = false;
	}
	if (can_id_func != -1) {
		// configure another mapping to transmit functional message
		mapping_func = mapping_phys_tx;
		mapping_func.can_id = can_id_func;
		mapping_func.can_id_flow_ctrl = PCANTP_MAPPING_FLOW_CTRL_NONE;
		mapping_func.netaddrinfo.target_type = PCANTP_ISOTP_ADDRESSING_FUNCTIONAL;
		mapping_func.netaddrinfo.target_addr = target_addr_func;
		status = CANTP_AddMapping_2016(channel, &mapping_func);
		if (!CANTP_StatusIsOk_2016(status)) {
			printf("Failed to configure mapping (can_id=0x%04X) (sts=0x%04X)\n", mapping_func.can_id, status);
			result = false;
		}
	}
	return result;
}

/// <summary>Initializes mappings compatible with PCTPClient/PCTPServer sample</summary>
/// <param name="channel">CAN channel to add the mappings to.</param>
/// <return>true on successful configuration</return>
static bool initialize_mappings(cantp_handle channel) {
	bool result = true;

	// 11 bit can ID, normal format addressing, diagnostic message (mandatory)
	result &= initialize_mappings(channel, PCANTP_CAN_MSGTYPE_STANDARD, N_SA, N_TA_PHYS, 0x00, N_TA_FUNC, 0xA1, 0xA2, 0xA5, PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, PCANTP_ISOTP_FORMAT_NORMAL);
	// 11 bit can ID, extended format addressing, diagnostic message (mandatory)
	result &= initialize_mappings(channel, PCANTP_CAN_MSGTYPE_STANDARD, N_SA, N_TA_PHYS, 0x00, N_TA_FUNC, 0xB1, 0xB2, 0xB5, PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, PCANTP_ISOTP_FORMAT_EXTENDED);
	result &= initialize_mappings(channel, PCANTP_CAN_MSGTYPE_STANDARD, N_SA, N_TA_PHYS, 0x00, N_TA_FUNC, 0xC1, 0xC2, 0xC5, PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, PCANTP_ISOTP_FORMAT_EXTENDED);


	// 11 bit can ID, mixed format addressing, remote diagnostic message (mandatory)
	result &= initialize_mappings(channel, PCANTP_CAN_MSGTYPE_STANDARD, N_SA, N_TA_PHYS, N_RA, N_TA_FUNC, 0xD1, 0xD2, 0xD5, PCANTP_ISOTP_MSGTYPE_REMOTE_DIAGNOSTIC, PCANTP_ISOTP_FORMAT_MIXED);

	// 29 bit can ID, normal format addressing, diagnostic message (mandatory)
	result &= initialize_mappings(channel, PCANTP_CAN_MSGTYPE_EXTENDED, N_SA, N_TA_PHYS, 0x00, N_TA_FUNC, 0xA123A1, 0xA123A2, 0xA123A5, PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, PCANTP_ISOTP_FORMAT_NORMAL);
	// 29 bit can ID, extended format addressing, diagnostic message
	result &= initialize_mappings(channel, PCANTP_CAN_MSGTYPE_EXTENDED, N_SA, N_TA_PHYS, 0x00, N_TA_FUNC, 0xA123C1, 0xA123C2, 0xA123C5, PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, PCANTP_ISOTP_FORMAT_EXTENDED);
	result &= initialize_mappings(channel, PCANTP_CAN_MSGTYPE_EXTENDED, N_SA, N_TA_PHYS, 0x00, N_TA_FUNC, 0xA123D1, 0xA123D2, 0xA123D5, PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, PCANTP_ISOTP_FORMAT_EXTENDED);

	return result;
}

/// <summary>Outputs a summary of a received message.</summary>
/// <param name="message">The message.</param>
/// <param name="timestamp">The timestamp of the message.</param>
/// <param name="status">status of the Read function.</param>
/// <returns>1 on success (2 if extra information is available within status), 0 if no message and -1 on error</returns>
static int display_message(cantp_msg& message, cantp_timestamp timestamp, cantp_status status) {
	char* strLoopback = "";
	char* strMsgType = "";
	uint32_t displayedDataLength = 0;
	int result;

	// check status
	result = CANTP_StatusIsOk_2016(status) ? 1 : -1;
	if (result != 1) {
		if (CANTP_StatusIsOk_2016(status, PCANTP_STATUS_NO_MESSAGE)) {
			// no message received ignore
			result = 0;
		}
		else {
			printf("Failed to read message (sts=0x%04X).\n", status);
		}
		return result;
	}
	// check bus errors within status
	if (!CANTP_StatusIsOk_2016(status, PCANTP_STATUS_OK, true)) {
		printf("Status returned information/warnings (sts=0x%04X).\n", status);
		result = 2;
	}

	// check if message has the loopback flag
	if (message.msgdata.any->flags & PCANTP_MSGFLAG_LOOPBACK) {
		strLoopback = "loopback ";
	}
	// format message's type
	switch (message.type) {
	case PCANTP_MSGTYPE_CAN:
		strMsgType = "PCANTP_MSGTYPE_CAN";
		break;
	case PCANTP_MSGTYPE_CANFD:
		strMsgType = "PCANTP_MSGTYPE_CANFD";
		break;
	case PCANTP_MSGTYPE_ISOTP:
		strMsgType = "PCANTP_MSGTYPE_ISOTP";
		break;
	case PCANTP_MSGTYPE_NONE:
		strMsgType = "PCANTP_MSGTYPE_NONE";
		break;
	default:
		strMsgType = "Unknown";
		break;
	}
	// Only display data if network result is OK
	if (message.msgdata.any->netstatus == PCANTP_NETSTATUS_OK) {
		displayedDataLength = message.msgdata.any->length;
	}

	// Display generic information
	printf("\n%010llu - Received %s%s message : canid=0x%04X, length=%d - result: 0x%02x - %s\n",
		timestamp,
		strLoopback,
		strMsgType,
		(int)message.can_info.can_id,
		(int)message.msgdata.any->length,
		(int)message.msgdata.any->netstatus,
		message.msgdata.any->netstatus != PCANTP_NETSTATUS_OK ? "ERROR !!!" : "OK !");

	if (message.type == PCANTP_MSGTYPE_ISOTP) {
		char* strPending = "Completed";
		// Limit displayed data if message is pending
		if (displayedDataLength > 0 && message.msgdata.isotp->netaddrinfo.msgtype & PCANTP_ISOTP_MSGTYPE_FLAG_INDICATION) {
			displayedDataLength = (message.msgdata.any->length > 7) ? 7 : message.msgdata.any->length;
			strPending = "/!\\ Pending";
		}
		// Display iso-tp message's information
		printf("\t%s message from 0x%02x (to 0x%02x, with RA 0x%02x)\n",
			strPending,
			(int)message.msgdata.isotp->netaddrinfo.source_addr,
			(int)message.msgdata.isotp->netaddrinfo.target_addr,
			(int)message.msgdata.isotp->netaddrinfo.extension_addr);
	}

	if (displayedDataLength > 0) {
		char buffer[500];
		// display data
		setvbuf(stdout, buffer, _IOLBF, BUFSIZ);
		printf("\t\\-> Length: %i, Data= ", (int)message.msgdata.any->length);
		for (uint32_t i = 0; i < displayedDataLength; i++) {
			printf("%02x ", (int)message.msgdata.any->data[i]);
			fflush(stdout);
		}
		if (displayedDataLength != message.msgdata.any->length) {
			printf("...");
			fflush(stdout);
		}

		printf("\n");
		setvbuf(stdout, NULL, _IONBF, 0);
	}
	return result;
}

// Function called to clean opened ISO-TP channels
static void consoleExit(void) {
	CANTP_Uninitialize_2016(PCANTP_HANDLE_NONEBUS);
}
// Callback to handle closure of the console window
static BOOL WINAPI HandlerRoutine(_In_ DWORD dwCtrlType) {
	consoleExit();
	return FALSE;
}

/// <summary>Main entry-point for this application.</summary>
/// <param name="argc">The argc.</param>
/// <param name="argv">[in,out] If non-null, the argv.</param>
/// <returns>Exit status</returns>
int main(int argc, char* argv[])
{
	cantp_status status;
	cantp_handle channel;
	WORD usValue;
	char buffer[500];
	BYTE param;
	int nbErr = 0;	
	bool useCanFd;
	bool useEvent;
	bool bStop = false;
	cantp_baudrate baudrate;
	cantp_bitrate bitrateFd;
	HANDLE eventRcv = NULL;

	printf("\nUsage: pctp_server.exe [cantp_handle] [USE_CAN_FD:0|1] [BTR0BTR1|BITRATE] [USE_EVENT:0|1]\n");
	printf(" * CAN: pctp_server.exe 0x51 0 0x1C \n");
	printf(" * CAN FD: pctp_server.exe 0x51 1 \"f_clock=80000000, nom_brp=10, nom_tseg1=12, nom_tseg2=3, nom_sjw=1, data_brp=4, data_tseg1=7, data_tseg2=2, data_sjw=1\" \n");
	printf("\n---------------------------------------------\n");

	// clear gracefully ISO-TP if console window is killed
	SetConsoleCtrlHandler(HandlerRoutine, TRUE);
	atexit(consoleExit);

	// Show version information
	status = CANTP_GetValue_2016(PCANTP_HANDLE_NONEBUS, PCANTP_PARAMETER_API_VERSION, &buffer, 500);
	if (CANTP_StatusIsOk_2016(status))
		printf("PCAN-ISO-TP API Version : %s\n", buffer);
	else 
		printf("Error failed to get PCAN-ISO-TP API Version (sts=0x%04X)\n", status);

	
	// 1st argument is the PCAN-channel to use (PCAN-USB channel 2)
	if (argc > 1)
		channel = (cantp_handle)strtol(argv[1], NULL, 0);
	else
		channel = PCANTP_HANDLE_USBBUS2;
	// 2nd argument is CAN FD
	if (argc > 2)
		useCanFd = strtol(argv[2], NULL, 0) == 1;
	else
		useCanFd = USE_CAN_FD;
	// 3rd argument is bitrate (either CAN or CAN FD)
	baudrate = PCANTP_BAUDRATE_500K;
	bitrateFd = "f_clock=80000000, nom_brp=10, nom_tseg1=12, nom_tseg2=3, nom_sjw=1, data_brp=4, data_tseg1=7, data_tseg2=2, data_sjw=1";
	if (argc > 3) {
		if (useCanFd)
			bitrateFd = argv[3];
		else 
			baudrate = (cantp_baudrate)strtoul(argv[3], NULL, 0);
	}
	// 4th argument is USE EVENT
	if (argc > 4)
		useEvent = strtol(argv[4], NULL, 0) == 1;
	else
		useEvent = USE_EVENT;

	// Initializing of the ISO-TP Communication session
	printf("Connecting to channel 0x%02X...\n", channel);
	if (!useCanFd) {
		printf(" * btr0btr1 = 0x%02X...\n", baudrate);
		status = CANTP_Initialize_2016(channel, baudrate, (cantp_hwtype)0, 0, 0);
	}
	else {
		printf(" * bitrateFd = \"%s\"...\n", bitrateFd);
		status = CANTP_InitializeFD_2016(channel, bitrateFd);
	}
	printf(" -> Initialize CANTP: 0x%04X\n", (int)status);
	// will start the reading loop only if successfully initialized
	if (!CANTP_StatusIsOk_2016(status)) {
		bStop = true;		
		nbErr++;
		printf(" -> Initialization failed, exiting...\n");
	}
	if (!bStop) {
		bool result;

		// configure the test session to log debug information 
		usValue = PCANTP_DEBUG_INFO;
		status = CANTP_SetValue_2016(channel, PCANTP_PARAMETER_DEBUG, &usValue, 1);
		if (!CANTP_StatusIsOk_2016(status)) {
			printf("Failed to configure parameter 'PCANTP_PARAMETER_DEBUG' (sts=0x%04X).\n", status);
		}
		// configure server - block size, 0 = unlimited
		param = 20;		
		status = CANTP_SetValue_2016(channel, PCANTP_PARAMETER_BLOCK_SIZE, &param, 1);
		if (!CANTP_StatusIsOk_2016(status)) {
			printf("Failed to configure parameter 'PCANTP_PARAMETER_BLOCK_SIZE' (sts=0x%04X).\n", status);
		}
		// configure server - wait Xms between segmented frames transmission
		param = 2;
		status = CANTP_SetValue_2016(channel, PCANTP_PARAMETER_SEPARATION_TIME, &param, 1);
		if (!CANTP_StatusIsOk_2016(status)) {
			printf("Failed to configure parameter 'PCANTP_PARAMETER_SEPARATION_TIME' (sts=0x%04X).\n", status);
		}
		// configure server - enable enhanced addressing to be compatible with PCTPClient
		param = PCANTP_VALUE_PARAMETER_ON;
		status = CANTP_SetValue_2016(channel, PCANTP_PARAMETER_SUPPORT_29B_ENHANCED, &param, 1);
		if (!CANTP_StatusIsOk_2016(status)) {
			printf("Failed to configure parameter 'PCANTP_PARAMETER_SUPPORT_29B_ENHANCED' (sts=0x%04X).\n", status);
		}
		// configure receive event
		if (useEvent) {
			printf(" -> Creating receive event\n");
			eventRcv = CreateEvent(NULL, false, false, NULL);
			status = CANTP_SetValue_2016(channel, PCANTP_PARAMETER_RECEIVE_EVENT, &eventRcv, sizeof(eventRcv));
			if (!CANTP_StatusIsOk_2016(status)) {
				nbErr++;
				bStop = true;
				printf("Failed to configure parameter 'PCANTP_PARAMETER_RECEIVE_EVENT' (sts=0x%04X).\n", status);
			}
		}

		// configure ISO-TP mappings
		result = initialize_mappings(channel);
		if (!result) {
			nbErr++;
			bStop = true;
			printf(" -> Mappings configuration failed, exiting...\n");
		}
	}

	// Output available commands
	if (!bStop) {
		printf("\n---------------------------------------------\n\n");
		printf("\nNote: press 'c' or 'C' to clear screen,");
		printf("\nNote: press 'q', 'Q' or '<Escape>' to quit...\n");
		printf("\n---------------------------------------------\n\n");
		printf("\n -> Waiting for messages...\n");
	}
	// Reading loop
	cantp_msg message;
	cantp_timestamp timestamp;
	DWORD result;
	while(!bStop)
	{
		if (useEvent)
			result = WaitForSingleObject(eventRcv, 1);
		else {
			Sleep(1);
			result = WAIT_OBJECT_0;
		}
		if (result == WAIT_OBJECT_0)
		{
			bool doLoop;
			// loop until no more message is available
			do {
				doLoop = false;
				// initialize message
				status = CANTP_MsgDataAlloc_2016(&message, PCANTP_MSGTYPE_NONE);
				if (!CANTP_StatusIsOk_2016(status)) {
					printf("Failed to allocate message (sts=0x%04X).\n", status);
					continue;
				}
				// retrieve any message from Rx queue
				status = CANTP_Read_2016(channel, &message, &timestamp);
				doLoop = display_message(message, timestamp, status) > 0;
				// update error counter if a message was received but a network error occured
				if (doLoop && message.msgdata.any->netstatus != PCANTP_NETSTATUS_OK) {
					nbErr++;
				}
				// release message
				status = CANTP_MsgDataFree_2016(&message);
				if (!CANTP_StatusIsOk_2016(status)) {
					printf("Failed to deallocate message (sts=0x%04X).\n", status);
				}
			} while (doLoop);
		}
		//Sleep(1);		
		// check exit request
		if (_kbhit()) {
			switch(_getch()) {
				case 'q':
				case 'Q':
				case 27:	//Escape
					bStop = true;
					break;
				case 0x08:	// backspace
					CANTP_Reset_2016(channel);
					// no break do clear...
				case 'c':
				case 'C':					
					system("cls");
					printf("\nNote: press 'c' or 'C' to clear screen,");
					printf("\nNote: press 'q', 'Q' or '<Escape>' to quit...\n\n");
					break;
			}
		}
	}
	
	// Display a small report
	if (nbErr > 0) {
		printf("\nERROR : %d errors occured.\n\n", nbErr);		
	} 
	else {		
		printf("\nALL Transmissions succeeded !\n\n");
	}
	printf("\n\nPress <Enter> to quit...");
	getchar();

	// release CAN-TP
	CANTP_Uninitialize_2016(channel);
	return 0;
}

