// pctp_client.cpp : Defines the entry point for the console application.
//
#include "stdafx.h"
#include "PCAN-ISO-TP_2016.h"

#define USE_CAN_FD false
#define USE_GETCH false

#define N_SA		((BYTE)0xF1)
#define N_TA_PHYS	((BYTE)0x13)
#define N_TA_FUNC	((BYTE)0x26)
#define N_RA		((BYTE)0x52)

#define TEST_PHYS_TRANSMISSION	1
#define TEST_FUNC_TRANSMISSION	1
#define TEST_BURST_TRANSMISSION 1
#define MESSAGE_MAX_LENGTH PCANTP_MAX_LENGTH_ISOTP2004 + 1000
#define CAN_TX_DL_CODE	0x0D
#define CAN_TX_DL_VALUE 32


// A global value to display timing information
DWORD g_timeReference;

/// <summary>Configure mappings to handle a two-way communication and a functional communication (broadcast).</summary>
/// <param name="channel">CAN channel to add </param>
/// <param name="can_msgtype">CAN frame </param>
/// <param name="source_addr">ISO-TP source address</param>
/// <param name="target_addr">ISO-TP target address</param>
/// <param name="extension_addr">ISO-TP extension address</param>
/// <param name="target_addr_func">ISO-TP target address for functional addressing</param>
/// <param name="can_id">CAN ID used to communicate for physical addressing</param>
/// <param name="can_id_flow_ctrl">Flow Control CAN ID used during physical addressing communication</param>
/// <param name="can_id_func">CAN ID used to communicate for functionnal addressing</param>
/// <param name="isotp_msgtype">ISOTP message's type</param>
/// <param name="isotp_format">ISOTP message's format</param>
/// <return>true on successful configuration</return>
static bool initialize_mappings(cantp_handle channel, cantp_can_msgtype can_msgtype, uint16_t source_addr, uint16_t target_addr, uint8_t extension_addr, uint16_t target_addr_func, uint32_t can_id, uint32_t can_id_flow_ctrl, uint32_t can_id_func, cantp_isotp_msgtype isotp_msgtype, cantp_isotp_format isotp_format) {
	bool result = true;
	cantp_mapping mapping_phys_rx;
	cantp_mapping mapping_phys_tx;
	cantp_mapping mapping_func;
	cantp_status status;

	// clean variables (it is common to leave can_tx_dl uninitialized which can lead to invalid 0xCC values)
	memset(&mapping_phys_rx, 0, sizeof(mapping_phys_rx));
	memset(&mapping_phys_tx, 0, sizeof(mapping_phys_tx));
	memset(&mapping_func, 0, sizeof(mapping_func));
	// configure a mapping to transmit physical message
	mapping_phys_tx.can_id = can_id;
	mapping_phys_tx.can_id_flow_ctrl = can_id_flow_ctrl;
	mapping_phys_tx.can_msgtype = can_msgtype;
	mapping_phys_tx.netaddrinfo.format = isotp_format;
	mapping_phys_tx.netaddrinfo.msgtype = isotp_msgtype;
	mapping_phys_tx.netaddrinfo.target_type = PCANTP_ISOTP_ADDRESSING_PHYSICAL;
	mapping_phys_tx.netaddrinfo.source_addr = source_addr;
	mapping_phys_tx.netaddrinfo.target_addr = target_addr;
	mapping_phys_tx.netaddrinfo.extension_addr = extension_addr;
	status = CANTP_AddMapping_2016(channel, &mapping_phys_tx);
	if (!CANTP_StatusIsOk_2016(status)) {
		printf("Failed to configure mapping (can_id=0x%04X) (sts=0x%04X)\n", mapping_phys_tx.can_id, status);
		result = false;
	}
	// configure another mapping to receive physical message (invert source and target in previous mapping)
	mapping_phys_rx = mapping_phys_tx;
	mapping_phys_rx.can_id = mapping_phys_tx.can_id_flow_ctrl;
	mapping_phys_rx.can_id_flow_ctrl = mapping_phys_tx.can_id;
	mapping_phys_rx.netaddrinfo.source_addr = mapping_phys_tx.netaddrinfo.target_addr;
	mapping_phys_rx.netaddrinfo.target_addr = mapping_phys_tx.netaddrinfo.source_addr;
	status = CANTP_AddMapping_2016(channel, &mapping_phys_rx);
	if (!CANTP_StatusIsOk_2016(status)) {
		printf("Failed to configure mapping (can_id=0x%04X) (sts=0x%04X)\n", mapping_phys_rx.can_id, status);
		result = false;
	}
	if (can_id_func != -1) {
		// configure another mapping to transmit functional message
		mapping_func = mapping_phys_tx;
		mapping_func.can_id = can_id_func;
		mapping_func.can_id_flow_ctrl = PCANTP_MAPPING_FLOW_CTRL_NONE;
		mapping_func.netaddrinfo.target_type = PCANTP_ISOTP_ADDRESSING_FUNCTIONAL;
		mapping_func.netaddrinfo.target_addr = target_addr_func;
		status = CANTP_AddMapping_2016(channel, &mapping_func);
		if (!CANTP_StatusIsOk_2016(status)) {
			printf("Failed to configure mapping (can_id=0x%04X) (sts=0x%04X)\n", mapping_func.can_id, status);
			result = false;
		}
	}
	return result;
}

/// <summary>Initializes mappings compatible with PCTPClient/PCTPServer sample</summary>
/// <param name="channel">CAN channel to add the mappings to.</param>
/// <return>true on successful configuration</return>
static bool initialize_mappings(cantp_handle channel) {
	bool result = true;

	// 11 bit can ID, normal format addressing, diagnostic message (mandatory)
	result &= initialize_mappings(channel, PCANTP_CAN_MSGTYPE_STANDARD, N_SA, N_TA_PHYS, 0x00, N_TA_FUNC, 0xA1, 0xA2, 0xA5, PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, PCANTP_ISOTP_FORMAT_NORMAL);
	// 11 bit can ID, extended format addressing, diagnostic message (mandatory)
	result &= initialize_mappings(channel, PCANTP_CAN_MSGTYPE_STANDARD, N_SA, N_TA_PHYS, 0x00, N_TA_FUNC, 0xB1, 0xB2, 0xB5, PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, PCANTP_ISOTP_FORMAT_EXTENDED);
	result &= initialize_mappings(channel, PCANTP_CAN_MSGTYPE_STANDARD, N_SA, N_TA_PHYS, 0x00, N_TA_FUNC, 0xC1, 0xC2, 0xC5, PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, PCANTP_ISOTP_FORMAT_EXTENDED);


	// 11 bit can ID, mixed format addressing, remote diagnostic message (mandatory)
	result &= initialize_mappings(channel, PCANTP_CAN_MSGTYPE_STANDARD, N_SA, N_TA_PHYS, N_RA, N_TA_FUNC, 0xD1, 0xD2, 0xD5, PCANTP_ISOTP_MSGTYPE_REMOTE_DIAGNOSTIC, PCANTP_ISOTP_FORMAT_MIXED);

	// 29 bit can ID, normal format addressing, diagnostic message (mandatory)
	result &= initialize_mappings(channel, PCANTP_CAN_MSGTYPE_EXTENDED, N_SA, N_TA_PHYS, 0x00, N_TA_FUNC, 0xA123A1, 0xA123A2, 0xA123A5, PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, PCANTP_ISOTP_FORMAT_NORMAL);
	// 29 bit can ID, extended format addressing, diagnostic message
	result &= initialize_mappings(channel, PCANTP_CAN_MSGTYPE_EXTENDED, N_SA, N_TA_PHYS, 0x00, N_TA_FUNC, 0xA123C1, 0xA123C2, 0xA123C5, PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, PCANTP_ISOTP_FORMAT_EXTENDED);
	result &= initialize_mappings(channel, PCANTP_CAN_MSGTYPE_EXTENDED, N_SA, N_TA_PHYS, 0x00, N_TA_FUNC, 0xA123D1, 0xA123D2, 0xA123D5, PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, PCANTP_ISOTP_FORMAT_EXTENDED);

	return result;
}

/// <summary>Outputs a summary of a received message.</summary>
/// <param name="message">The message.</param>
/// <param name="timestamp">The timestamp of the message.</param>
/// <param name="status">status of the Read function.</param>
/// <returns>1 on success (2 if extra information is available within status), 0 if no message and -1 on error</returns>
static int display_message(cantp_msg& message, cantp_timestamp timestamp, cantp_status status) {
	char* strLoopback = "";
	char* strMsgType = "";
	uint32_t displayedDataLength = 0;
	int result;

	// check status
	result = CANTP_StatusIsOk_2016(status) ? 1 : -1;
	if (result != 1) {
		if (CANTP_StatusIsOk_2016(status, PCANTP_STATUS_NO_MESSAGE)) {
			// no message received ignore
			result = 0;
		}
		else {
			printf("Failed to read message (sts=0x%04X).\n", status);
		}
		return result;
	}
	// check bus errors within status
	if (!CANTP_StatusIsOk_2016(status, PCANTP_STATUS_OK, true)) {
		printf("Status returned information/warnings (sts=0x%04X).\n", status);
		result = 2;
	}

	// check if message has the loopback flag
	if (message.msgdata.any->flags & PCANTP_MSGFLAG_LOOPBACK) {
		strLoopback = "loopback ";
	}
	// format message's type
	switch (message.type) {
	case PCANTP_MSGTYPE_CAN:
		strMsgType = "PCANTP_MSGTYPE_CAN";
		break;
	case PCANTP_MSGTYPE_CANFD:
		strMsgType = "PCANTP_MSGTYPE_CANFD";
		break;
	case PCANTP_MSGTYPE_ISOTP:
		strMsgType = "PCANTP_MSGTYPE_ISOTP";
		break;
	case PCANTP_MSGTYPE_NONE:
		strMsgType = "PCANTP_MSGTYPE_NONE";
		break;
	default:
		strMsgType = "Unknown";
		break;
	}
	// Only display data if network result is OK
	if (message.msgdata.any->netstatus == PCANTP_NETSTATUS_OK) {
		displayedDataLength = message.msgdata.any->length;
	}

	// Display generic information
	printf("\n%010llu - Received %s%s message : canid=0x%04X, length=%d - result: 0x%02x - %s\n",
		timestamp,
		strLoopback,
		strMsgType,
		(int)message.can_info.can_id,
		(int)message.msgdata.any->length,
		(int)message.msgdata.any->netstatus,
		message.msgdata.any->netstatus != PCANTP_NETSTATUS_OK ? "ERROR !!!" : "OK !");

	if (message.type == PCANTP_MSGTYPE_ISOTP) {
		char* strPending = "Completed";
		// Limit displayed data if message is pending
		if (displayedDataLength > 0 && message.msgdata.isotp->netaddrinfo.msgtype & PCANTP_ISOTP_MSGTYPE_FLAG_INDICATION) {
			displayedDataLength = (message.msgdata.any->length > 7) ? 7 : message.msgdata.any->length;
			strPending = "/!\\ Pending";
		}
		// Display iso-tp message's information
		printf("\t%s message from 0x%02x (to 0x%02x, with RA 0x%02x)\n",
			strPending,
			(int)message.msgdata.isotp->netaddrinfo.source_addr,
			(int)message.msgdata.isotp->netaddrinfo.target_addr,
			(int)message.msgdata.isotp->netaddrinfo.extension_addr);
	}

	if (displayedDataLength > 0) {
		char buffer[500];
		// display data
		setvbuf(stdout, buffer, _IOLBF, BUFSIZ);
		printf("\t\\-> Length: %i, Data= ", (int)message.msgdata.any->length);
		for (uint32_t i = 0; i < displayedDataLength; i++) {
			printf("%02x ", (int)message.msgdata.any->data[i]);
			fflush(stdout);
		}
		if (displayedDataLength != message.msgdata.any->length) {
			printf("...");
			fflush(stdout);
		}

		printf("\n");
		setvbuf(stdout, NULL, _IONBF, 0);
	}
	return result;
}

// Function called to clean opened ISO-TP channels
static void consoleExit(void) {
	CANTP_Uninitialize_2016(PCANTP_HANDLE_NONEBUS);
}
// Callback to handle closure of the console window
static BOOL WINAPI HandlerRoutine(_In_ DWORD dwCtrlType) {
	consoleExit();
	return FALSE;
}

/// <summary>Reads a message.</summary>
/// <param name="channel">The channel to read the message from.</param>
/// <param name="nbMsgRead">Buffer to store the number of message actually read.</param>
/// <returns>number of errors</returns>
static int read_segmented_message(cantp_handle channel, int& nbMsgRead) {
	cantp_msg message;
	cantp_status status;
	cantp_timestamp timestamp;
	int doLoop;

	int nbErr = 0;

	do {
		doLoop = false;
		// initialize message
		status = CANTP_MsgDataAlloc_2016(&message, PCANTP_MSGTYPE_NONE);
		if (!CANTP_StatusIsOk_2016(status)) {
			printf("Failed to allocate message (sts=0x%04X).\n", status);
			++nbErr;
			continue;
		}
		// retrieve any message from Rx queue
		status = CANTP_Read_2016(channel, &message, &timestamp);
		// display_message returns true when a message is available
		doLoop = display_message(message, timestamp, status);
		if (doLoop > 0) {
			// update error counter if a message was received but a network error occured
			if (message.msgdata.any->netstatus != PCANTP_NETSTATUS_OK) {
				nbErr++;
			}
			if (message.type == PCANTP_MSGTYPE_ISOTP &&
				message.msgdata.isotp->netaddrinfo.msgtype & PCANTP_ISOTP_MSGTYPE_FLAG_INDICATION) {
				// wait and read the full segmented message
				nbErr += read_segmented_message(channel, nbMsgRead);
			}
			else
				nbMsgRead++;
		}
		else {
			if (doLoop == 0) {
				// no message read, wait for it
				Sleep(10);
			}
		}
		// release message
		status = CANTP_MsgDataFree_2016(&message);
		if (!CANTP_StatusIsOk_2016(status)) {
			printf("Failed to deallocate message (sts=0x%04X).\n", status);
		}
	} while (doLoop == 0);

	return nbErr;
}

/// <summary>A function to transmit CAN-TP messages</summary>
/// <param name="channel">channel</param>
/// <param name="canIdType">canIdType</param>
/// <param name="msgType">CAN-TP message Type (PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC)</param>
/// <param name="formatType">Format addressing type (see PCANTP_ISOTP_FORMAT_xxx)</param>
/// <returns>number of transmission errors</returns>
static int transmit(cantp_handle channel,
	cantp_can_msgtype canIdType,
	cantp_isotp_msgtype msgType,
	cantp_isotp_format formatType,
	bool doCanFd, bool canFdInitialized)
{
	uint32_t index;
	cantp_status status;
	cantp_msg message = {};
	cantp_can_msgtype can_msgtype = PCANTP_CAN_MSGTYPE_STANDARD;
	cantp_netaddrinfo isotp_nai = {};
	BYTE lMaxSize = 0;
	int nbErrPhys = 0;
	int nbErrFunc = 0;
	DWORD sleepTime = 0;
	int nbMsgRead = 0;
	uint8_t data[MESSAGE_MAX_LENGTH] = {};

	// Output information message
	char lStr[50] = "??";
	switch (formatType)
	{
	case PCANTP_ISOTP_FORMAT_ENHANCED:
		strcpy_s(lStr, "Enhanced");
		break;
	case PCANTP_ISOTP_FORMAT_EXTENDED:
		strcpy_s(lStr, "Extended");
		break;
	case PCANTP_ISOTP_FORMAT_FIXED_NORMAL:
		strcpy_s(lStr, "Fixed Normal");
		break;
	case PCANTP_ISOTP_FORMAT_MIXED:
		strcpy_s(lStr, "Mixed");
		break;
	case PCANTP_ISOTP_FORMAT_NORMAL:
		strcpy_s(lStr, "Normal");
		break;
	}
	printf("\n\nTransmission of %s, %s message, %s addressing format",
		(canIdType & PCANTP_CAN_MSGTYPE_EXTENDED) ? "29 BIT CAN ID" : "11 BIT CAN ID",
		msgType == PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC ? "Diagnostic" : "Remote diagnostic",
		lStr);
	if (USE_GETCH) {
		printf("\nPress <Enter> to continue...");
		getchar();
	}

	// initialize message
	isotp_nai.source_addr = N_SA;
	if (msgType == PCANTP_ISOTP_MSGTYPE_REMOTE_DIAGNOSTIC)
		isotp_nai.extension_addr = N_RA;
	can_msgtype = canIdType;
	if (doCanFd) {
		can_msgtype |= PCANTP_CAN_MSGTYPE_FD;
		can_msgtype |= PCANTP_CAN_MSGTYPE_BRS;
	}
	isotp_nai.msgtype = msgType;
	isotp_nai.format = formatType;
	// prepare a buffer that will be used as a reference to initialize the data to transmit
	for (int i = 0; i < MESSAGE_MAX_LENGTH; ++i)
		data[i] = (uint8_t)i;

#if TEST_PHYS_TRANSMISSION
	// Transmitting data using Physical addressing
	printf("\nTransmitting data using Physical addressing\n\n");
	g_timeReference = GetTickCount();
	isotp_nai.target_addr = N_TA_PHYS;
	isotp_nai.target_type = PCANTP_ISOTP_ADDRESSING_PHYSICAL;
	// each loop transmits a message with a data length incremented by 1 and reads its loppbacl confirmation
	for (index = 1; index <= MESSAGE_MAX_LENGTH; index++)
	{
		// allocate a new ISOTP message
		status = CANTP_MsgDataAlloc_2016(&message, PCANTP_MSGTYPE_ISOTP);
		if (CANTP_StatusIsOk_2016(status)) {
			// initialize ISOTP message
			status = CANTP_MsgDataInit_2016(&message, PCANTP_CAN_ID_DEFINED_BY_NAI, can_msgtype, index, data, &isotp_nai);
			if (CANTP_StatusIsOk_2016(status)) {
				// write message
				status = CANTP_Write_2016(channel, &message);
				if (CANTP_StatusIsOk_2016(status)) {
					printf("\nSuccessfully queued ISOTP message: Length %i (sts=0x%04X).\n", (int)index, (int)status);
					// try to read loopback confirmation 
					nbMsgRead = 0;
					nbErrPhys += read_segmented_message(channel, nbMsgRead);
					if (nbMsgRead != 1) {
						printf("Received unexpected messages (%d instead of 1)", nbMsgRead);
					}
				}
				else {
					printf("\nFailed to write ISOTP message: Length %i (sts=0x%04X).\n", (int)index, (int)status);
					nbErrPhys++;
				}
			}
			else {
				printf("\nFailed to initialize ISOTP message: Length %i (sts=0x%04X).\n", (int)index, (int)status);
				nbErrPhys++;
			}
			// release message
			status = CANTP_MsgDataFree_2016(&message);
			if (!CANTP_StatusIsOk_2016(status)) {
				printf("Failed to deallocate message (sts=0x%04X).\n", status);
				nbErrPhys++;
			}

		}
		else {
			printf("Failed to allocate message (sts=0x%04X).\n", status);
			nbErrPhys++;
		}
		
		Sleep(sleepTime);
		
		if (index == 100) { 
			// skip and jump to maximum data length
			index = MESSAGE_MAX_LENGTH - 1;
		}
	}
#endif

#if TEST_BURST_TRANSMISSION
	printf("\nTransmitting data using Physical addressing in 'BURST' mode\n\n");
	g_timeReference = GetTickCount();
	isotp_nai.target_addr = N_TA_PHYS;
	isotp_nai.target_type = PCANTP_ISOTP_ADDRESSING_PHYSICAL;
	// Transmitting data using Physical addressing WITHOUT waiting for response confirmation
	// each loop transmits a message with a data length incremented by 1
	int nbMsgSent = 0;
	for (index = 1; index <= MESSAGE_MAX_LENGTH; index++)
	{
		// allocate a new ISOTP message
		status = CANTP_MsgDataAlloc_2016(&message, PCANTP_MSGTYPE_ISOTP);
		if (CANTP_StatusIsOk_2016(status)) {
			// initialize ISOTP message
			status = CANTP_MsgDataInit_2016(&message, PCANTP_CAN_ID_DEFINED_BY_NAI, can_msgtype, index, data, &isotp_nai);
			if (CANTP_StatusIsOk_2016(status)) {
				// write message
				status = CANTP_Write_2016(channel, &message);
				if (CANTP_StatusIsOk_2016(status)) {
					nbMsgSent++;
					printf("\nSuccessfully queued ISOTP message: Length %i (sts=0x%04X).\n", (int)index, (int)status);					
				}
				else {
					printf("\nFailed to write ISOTP message: Length %i (sts=0x%04X).\n", (int)index, (int)status);
					nbErrPhys++;
				}
			}
			else {
				printf("\nFailed to initialize ISOTP message: Length %i (sts=0x%04X).\n", (int)index, (int)status);
				nbErrPhys++;
			}
			// release message
			status = CANTP_MsgDataFree_2016(&message);
			if (!CANTP_StatusIsOk_2016(status)) {
				printf("Failed to deallocate message (sts=0x%04X).\n", status);
				nbErrPhys++;
			}
		}
		else {
			printf("Failed to allocate message (sts=0x%04X).\n", status);
			nbErrPhys++;
		}

	
		if (index == 10) {	// skip and try bigger LEN
			index = 2000;
		}
		if (index == 2005) { // skip
			break;
		}
	}
	printf("\nReading confirmation for each transmitted messages in 'BURST' mode\n\n");
	for (nbMsgRead = 0; nbMsgRead < nbMsgSent;)
	{
		// try to read loopback confirmation 
		nbErrPhys += read_segmented_message(channel, nbMsgRead);
	}
	CANTP_Reset_2016(channel);
#endif

#if TEST_FUNC_TRANSMISSION
	// Transmitting data using Functional addressing 
	printf("\nTransmitting data using Functional addressing\n\n");
	g_timeReference = GetTickCount();
	isotp_nai.target_addr = N_TA_FUNC;
	isotp_nai.target_type = PCANTP_ISOTP_ADDRESSING_FUNCTIONAL;
	// Reminder: Functional addressing shall only be supported 
	// for Single Frame communication.
	// Thus maximum size depends on Format Addressing Type.
	switch (formatType) {
	case PCANTP_ISOTP_FORMAT_EXTENDED:
	case PCANTP_ISOTP_FORMAT_MIXED:
		lMaxSize = ((doCanFd && canFdInitialized) ? CAN_TX_DL_VALUE : PCANTP_MAX_LENGTH_CAN_STANDARD) - 2;
		break;
	default:
		lMaxSize = ((doCanFd && canFdInitialized) ? CAN_TX_DL_VALUE : PCANTP_MAX_LENGTH_CAN_STANDARD) - 1;
	}
	// each loop transmits a message with a data length incremented by 1 and reads its loppbacl confirmation
	for (index = 1; index <= lMaxSize; index++)
	{
		// allocate a new ISOTP message
		status = CANTP_MsgDataAlloc_2016(&message, PCANTP_MSGTYPE_ISOTP);
		if (CANTP_StatusIsOk_2016(status)) {
			// initialize ISOTP message
			status = CANTP_MsgDataInit_2016(&message, PCANTP_CAN_ID_DEFINED_BY_NAI, can_msgtype, index, data, &isotp_nai);
			if (CANTP_StatusIsOk_2016(status)) {
				// write message
				status = CANTP_Write_2016(channel, &message);
				if (CANTP_StatusIsOk_2016(status)) {
					printf("\nSuccessfully queued ISOTP message: Length %i (sts=0x%04X).\n", (int)index, (int)status);
					// try to read loopback confirmation 
					nbMsgRead = 0;
					nbErrFunc += read_segmented_message(channel, nbMsgRead);
					if (nbMsgRead != 1) {
						printf("Received unexpected messages (%d instead of 1)", nbMsgRead);
					}
				}
				else {
					printf("\nFailed to write ISOTP message: Length %i (sts=0x%04X).\n", (int)index, (int)status);
					nbErrFunc++;
				}
			}
			else {
				printf("\nFailed to initialize ISOTP message: Length %i (sts=0x%04X).\n", (int)index, (int)status);
				nbErrFunc++;
			}
			// release message
			status = CANTP_MsgDataFree_2016(&message);
			if (!CANTP_StatusIsOk_2016(status)) {
				printf("Failed to deallocate message (sts=0x%04X).\n", status);
				nbErrFunc++;
			}

		}
		else {
			printf("Failed to allocate message (sts=0x%04X).\n", status);
			nbErrFunc++;
		}

		Sleep(sleepTime);
	}
#endif

	if (nbErrPhys > 0 || nbErrFunc > 0) {
		if (nbErrPhys > 0) {
			printf("\nERROR : %d errors occured.\n\n", nbErrPhys);
		}
		if (nbErrFunc > 0) {
			printf("\nERROR : %d errors occured.\n\n", nbErrFunc);
		}
		if (USE_GETCH) {
			printf("\nPress <Enter> to continue...");
			getchar();
		}
	}
	else {
		printf("\nTransmissions successful !\n\n");
	}

	return nbErrFunc + nbErrPhys;
}

/// <summary>	Main entry-point for this application. </summary>
/// <param name="argc">	The argc. </param>
/// <param name="argv">	[in,out] If non-null, the argv. </param>
/// <returns>Exit status</returns>
int main(int argc, char* argv[])
{
	cantp_handle channel;
	cantp_status status;
	cantp_baudrate baudrate;
	cantp_bitrate bitrateFd;
	bool useCanFd = false;
	bool doCanFd = false;
	BYTE param;
	int nbErr = 0;
	char buff[50] = {};

	printf("\nUsage: PCTPClient.exe [TPCANTPHandle] [USE_CAN_FD:0|1] [BTR0BTR1|BITRATE]\n");
	printf(" * CAN: PCTPClient.exe 0x51 0 0x1C \n");
	printf(" * CAN FD: PCTPClient.exe 0x51 1 \"f_clock=80000000, nom_brp=10, nom_tseg1=12, nom_tseg2=3, nom_sjw=1, data_brp=4, data_tseg1=7, data_tseg2=2, data_sjw=1\" \n");
	printf("\n---------------------------------------------\n\n");

	// clear gracefully ISO-TP if console window is killed
	SetConsoleCtrlHandler(HandlerRoutine, TRUE);
	atexit(consoleExit);

	// Show version information
	CANTP_GetValue_2016(PCANTP_HANDLE_NONEBUS, PCANTP_PARAMETER_API_VERSION, &buff, 50);
	printf("PCAN-ISO-TP API Version : %s\n", buff);

	// 1st argument is the PCAN-channel to use (PCAN-USB channel 1)
	if (argc > 1)
		channel = (cantp_handle)strtol(argv[1], NULL, 0);
	else
		channel = PCANTP_HANDLE_USBBUS1;
	// 2nd argument is CAN FD
	if (argc > 2)
		useCanFd = strtol(argv[2], NULL, 0) == 1;
	else
		useCanFd = USE_CAN_FD;
	// 3rd argument is bitrate (either CAN or CAN FD)
	baudrate = PCANTP_BAUDRATE_500K;
	bitrateFd = "f_clock=80000000, nom_brp=10, nom_tseg1=12, nom_tseg2=3, nom_sjw=1, data_brp=4, data_tseg1=7, data_tseg2=2, data_sjw=1";
	if (argc > 3) {
		if (useCanFd)
			bitrateFd = argv[3];
		else
			baudrate = (cantp_baudrate)strtoul(argv[3], NULL, 0);
	}

	// Initializing of the ISO-TP Communication session
	printf("Connecting to channel 0x%02X...\n", channel);
	if (!useCanFd) {
		printf(" * btr0btr1 = 0x%02X...\n", baudrate);
		status = CANTP_Initialize_2016(channel, baudrate, (cantp_hwtype)0, 0, 0);
	}
	else {
		printf(" * bitrateFd = \"%s\"...\n", bitrateFd);
		status = CANTP_InitializeFD_2016(channel, bitrateFd);
	}
	printf(" -> Initialize CANTP: 0x%04X\n", (int)status);
	// Start the write/read loop only if the channel was successfully initialized
	if (!CANTP_StatusIsOk_2016(status)) {
		// initialization failed: abort
		printf(" -> Initialization failed, exiting...\n");
		CANTP_Uninitialize_2016(channel);
		printf("\n\nPress <Enter> to quit...");
		getchar();
		return -1;
	}

	// configure the test session to use can data padding with a specific padding value
	param = PCANTP_CAN_DATA_PADDING_ON;
	status = CANTP_SetValue_2016(channel, PCANTP_PARAMETER_CAN_DATA_PADDING, &param, 1);
	if (!CANTP_StatusIsOk_2016(status)) {
		printf("Failed to configure parameter 'PCANTP_PARAMETER_CAN_DATA_PADDING' (sts=0x%04X).\n", status);
	}
	param = 0x99;
	status = CANTP_SetValue_2016(channel, PCANTP_PARAMETER_CAN_PADDING_VALUE, &param, 1);
	if (!CANTP_StatusIsOk_2016(status)) {
		printf("Failed to configure parameter 'PCANTP_PARAMETER_CAN_PADDING_VALUE' (sts=0x%04X).\n", status);
	}
	// set the default TX_DL size (i.e. the default CAN-FD DLC)
	param = CAN_TX_DL_CODE;
	status = CANTP_SetValue_2016(channel, PCANTP_PARAMETER_CAN_TX_DL, &param, 1);
	if (!CANTP_StatusIsOk_2016(status)) {
		printf("Failed to configure parameter 'PCANTP_PARAMETER_CAN_TX_DL' (sts=0x%04X).\n", status);
	}
	// configure server - enable enhanced addressing to be compatible with PCTPClient
	param = PCANTP_VALUE_PARAMETER_ON;
	status = CANTP_SetValue_2016(channel, PCANTP_PARAMETER_SUPPORT_29B_ENHANCED, &param, 1);
	if (!CANTP_StatusIsOk_2016(status)) {
		printf("Failed to configure parameter 'PCANTP_PARAMETER_SUPPORT_29B_ENHANCED' (sts=0x%04X).\n", status);
	}
	// log CAN frames
	param = PCANTP_DEBUG_WARNING;
	status = CANTP_SetValue_2016(channel, PCANTP_PARAMETER_DEBUG, &param, 1);
	if (!CANTP_StatusIsOk_2016(status)) {
		printf("Failed to configure parameter 'PCANTP_PARAMETER_DEBUG' (sts=0x%04X).\n", status);
	}
	// configure ISO-TP mappings
	if (!initialize_mappings(channel)) {
		printf(" -> Mappings configuration failed, exiting...\n");
		CANTP_Uninitialize_2016(channel);
		printf("\n\nPress <Enter> to quit...");
		getchar();
		return -1;
	}

	for (int i = 0; i < 2; i++) {
		switch (i) {
		case 0:
			// first loop will test CAN-FD frames even if it was not initialized with Fd
			doCanFd = true;
			break;
		case 1:
			// second loop tests standard CAN frames
			doCanFd = false;
			break;
		}

		// Transmit with 29 Bit CAN identifiers (and no CAN ID mapping is required)
		nbErr += transmit(channel, PCANTP_CAN_MSGTYPE_EXTENDED, PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, PCANTP_ISOTP_FORMAT_FIXED_NORMAL, doCanFd, useCanFd);
		nbErr += transmit(channel, PCANTP_CAN_MSGTYPE_EXTENDED, PCANTP_ISOTP_MSGTYPE_REMOTE_DIAGNOSTIC, PCANTP_ISOTP_FORMAT_MIXED, doCanFd, useCanFd);

		// Transmit with enhanced diagnostics 29 bit CAN identifiers (ISO-15765-3 8.3)
		nbErr += transmit(channel, PCANTP_CAN_MSGTYPE_EXTENDED, PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, PCANTP_ISOTP_FORMAT_ENHANCED, doCanFd, useCanFd);

		// Transmit with 11 Bit CAN identifiers
		// those transmissions require a Mapping between CAN ID and Network Addressing Information
		// for each transmission, 3 mappings are added :
		//	- 2 for physical addressing, one defining SA->TA and the other TA->SA
		//	- 1 for functional addressing defining SA->TA
		//

		// 11 bit can ID, normal format addressing (diagnostic message mandatory)		
		nbErr += transmit(channel, PCANTP_CAN_MSGTYPE_STANDARD, PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, PCANTP_ISOTP_FORMAT_NORMAL, doCanFd, useCanFd);
		// 11 bit can ID, extended format addressing (diagnostic message mandatory)		
		nbErr += transmit(channel, PCANTP_CAN_MSGTYPE_STANDARD, PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, PCANTP_ISOTP_FORMAT_EXTENDED, doCanFd, useCanFd);
		// 11 bit can ID, mixed format addressing (remote diagnostic message mandatory)
		nbErr += transmit(channel, PCANTP_CAN_MSGTYPE_STANDARD, PCANTP_ISOTP_MSGTYPE_REMOTE_DIAGNOSTIC, PCANTP_ISOTP_FORMAT_MIXED, doCanFd, useCanFd);

		// Transmit with 29 Bit CAN identifiers
		// those transmissions require a Mapping between CAN ID and Network Addressing Information
		// for each transmission, 3 mappings are added 2 for physical addressing and 
		// the other for functional addressing.
		//

		// 29 bit can ID, normal format addressing (diagnostic message mandatory)
		nbErr += transmit(channel, PCANTP_CAN_MSGTYPE_EXTENDED, PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, PCANTP_ISOTP_FORMAT_NORMAL, doCanFd, useCanFd);
		// 29 bit can ID, extended format addressing (diagnostic message mandatory)
		nbErr += transmit(channel, PCANTP_CAN_MSGTYPE_EXTENDED, PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, PCANTP_ISOTP_FORMAT_EXTENDED, doCanFd, useCanFd);
	}

	if (nbErr > 0) {
		printf("\nERROR : a total of %d errors occured.\n\n", nbErr);
		printf("\n\nPress <Enter> to quit...");
		getchar();
	}
	else {
		printf("\nALL Transmissions succeeded !\n\n");
	}

	CANTP_Uninitialize_2016(channel);
	return 0;
}