#include "PCANBasicCLR.h" 
#include "PCAN-ISO-TP-CLR_2016.h"

using namespace System;
using namespace System::IO;
using namespace Peak::Can::IsoTp;

#define STMIN_600US 0xF6

static String^ OK_KO(bool test)
{
	if (test)
		return "OK";
	return "KO";
}

static String^ STATUS_OK_KO(cantp_status test)
{
	return OK_KO(CanTpApi::StatusIsOk_2016(test));
}

/// <summary>Helper: print message data</summary>
/// <param name="msg">message to print</param>
static void print_data(cantp_msg msg)
{
	if (msg.msgdata.any != nullptr)
	{
		UInt32 len = msg.msgdata.any->length;

		Console::Write("\t Byte:");
		for (UInt32 i = 0; i < len; i++)
		{
			Console::Write(" 0x{0:X2}", msg.msgdata.any->data[i]);
		}
		Console::WriteLine();
		Console::Write("\t Ascii:\"");
		for (UInt32 i = 0; i < len; i++)
		{
			Console::Write("{0}", Convert::ToChar(msg.msgdata.any->data[i]));
		}
		Console::WriteLine("\"");
	}
}


/// <summary>Entry point of the program, start a small CAN ISO TP read/write example</summary>
int main(array<System::String ^> ^args)
{
	// Local variables
	cantp_status res;
	cantp_status read_status;
	StringBuilder ^buffer = gcnew StringBuilder(500);
	UInt32 STmin;
	cantp_msg rx_msg;
	cantp_mapping mapping;
	cantp_handle server_handle;

	// Initialize variables
	server_handle = cantp_handle::PCANTP_HANDLE_USBBUS2;  // TODO: modify the value according to your available PCAN devices.

	// Print version informations
	CanTpApi::GetValue_2016(cantp_handle::PCANTP_HANDLE_NONEBUS, cantp_parameter::PCANTP_PARAMETER_API_VERSION, buffer, 500);
	Console::WriteLine("PCAN-ISO-TP API Version : {0}", buffer);

	// Initialize channel: CAN2.0 - 500Kbit/s
	res = CanTpApi::Initialize_2016(server_handle, cantp_baudrate::PCANTP_BAUDRATE_500K);
	Console::WriteLine("Initialize : {0}", STATUS_OK_KO(res));
	// Create and set a receive event on receiver
	System::Threading::AutoResetEvent receive_event(false);
	if (IntPtr::Size == 4)
	{
		UInt32 iBuffer = Convert::ToUInt32(receive_event.SafeWaitHandle->DangerousGetHandle().ToInt32());
		res = CanTpApi::SetValue_2016(server_handle, cantp_parameter::PCANTP_PARAMETER_RECEIVE_EVENT, iBuffer, sizeof(UInt32));
	}
	else if (IntPtr::Size == 8)
	{
		Int64 iBuffer = receive_event.SafeWaitHandle->DangerousGetHandle().ToInt64();
		array<Byte> ^byteArray = BitConverter::GetBytes(iBuffer);
		res = CanTpApi::SetValue_2016(server_handle, cantp_parameter::PCANTP_PARAMETER_RECEIVE_EVENT, byteArray, sizeof(UInt64));
	}
	Console::WriteLine("Set receive event : {0}", STATUS_OK_KO(res));

	// Change STmin value to 600us
	STmin = STMIN_600US;
	res = CanTpApi::SetValue_2016(server_handle, cantp_parameter::PCANTP_PARAMETER_SEPARATION_TIME, STmin, sizeof(UInt32));
	Console::WriteLine("Set STMIN = 600us : {0}", STATUS_OK_KO(res));

	// Create a simple physical mapping: 
	//    - Source 0xF1 (client), target 0x01 (server), CAN id 0xA1, CAN ID flow control 0xA2
	//    - Diagnostic message in a classic format
	mapping.can_id = 0xA1;
	mapping.can_id_flow_ctrl = 0xA2;
	mapping.can_msgtype = cantp_can_msgtype::PCANTP_CAN_MSGTYPE_STANDARD;
	mapping.netaddrinfo.extension_addr = 0x00;
	mapping.netaddrinfo.format = cantp_isotp_format::PCANTP_ISOTP_FORMAT_NORMAL;
	mapping.netaddrinfo.msgtype = cantp_isotp_msgtype::PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC;
	mapping.netaddrinfo.source_addr = 0xF1;
	mapping.netaddrinfo.target_addr = 0x01;
	mapping.netaddrinfo.target_type = cantp_isotp_addressing::PCANTP_ISOTP_ADDRESSING_PHYSICAL;

	// Add mapping on channel
	res = CanTpApi::AddMapping_2016(server_handle, mapping);
	Console::WriteLine("Add a simple mapping : {0}", STATUS_OK_KO(res));
	// NOTE: this is a one-way mapping, the server is only capable of 
	//	receiving segmented or unsegmented ISO-TP message using "0xA1" CAN ID
	//	to be able to send ISO-TP message, another mapping is required.

	// Read while user do not press Q
	Console::WriteLine("Start listening, press Q to quit.");
	bool stop = false;
	while (!stop)
	{
		// Wait a receive event on receiver
		//	note: timeout is used to check keyboard hit.
		bool wait_result = receive_event.WaitOne(1000);

		// If we receive something, read the message
		if (wait_result)
		{
			do
			{
				// Allocate rx message
				res = CanTpApi::MsgDataAlloc_2016(rx_msg, cantp_msgtype::PCANTP_MSGTYPE_NONE);
				Console::WriteLine("Allocate rx message : {0}", STATUS_OK_KO(res));

				// Read first available message (no filtering based on message's type is set): 
				cantp_timestamp ts;
				read_status = CanTpApi::Read_2016(server_handle, rx_msg, ts);
				Console::WriteLine("Try to read a message : {0}", STATUS_OK_KO(read_status));

				// If we read something, print the message
				if (CanTpApi::StatusIsOk_2016(read_status))
				{
					Console::WriteLine();
					Console::WriteLine("Print received data :");
					print_data(rx_msg);
					Console::WriteLine();
				}

				// Free message
				res = CanTpApi::MsgDataFree_2016(rx_msg);
				Console::WriteLine("Free rx message : {0}", STATUS_OK_KO(res));

			} while (!CanTpApi::StatusIsOk_2016(read_status, cantp_status::PCANTP_STATUS_NO_MESSAGE));
		}

		// Quit when user press Q
		if (Console::KeyAvailable)
		{
			char keyboard_res = (char)Console::ReadKey().KeyChar;
			if (keyboard_res == 'Q' || keyboard_res == 'q')
			{
				stop = true;
			}
		}
	}

	// Close receive event
	if (IntPtr::Size == 4)
	{
		UInt32 iBuffer = 0;
		res = CanTpApi::SetValue_2016(server_handle, cantp_parameter::PCANTP_PARAMETER_RECEIVE_EVENT, iBuffer, sizeof(UInt32));
	}
	else if (IntPtr::Size == 8)
	{
		Int64 iBuffer = 0;
		array<Byte>^ byteArray = BitConverter::GetBytes(iBuffer);
		res = CanTpApi::SetValue_2016(server_handle, cantp_parameter::PCANTP_PARAMETER_RECEIVE_EVENT, byteArray, sizeof(UInt64));
	}
	Console::WriteLine("Stop receive event  : {0}", STATUS_OK_KO(res));
	receive_event.Close();

	// Uninitialize
	res = CanTpApi::Uninitialize_2016(server_handle);
	Console::WriteLine("Uninitialize : {0}", STATUS_OK_KO(res));

	// Exit
	Console::WriteLine("Press any key to exit...");
	Console::Read();
}


