#include "PCANBasicCLR.h" 
#include "PCAN-ISO-TP-CLR_2016.h"

using namespace System;
using namespace System::IO;
using namespace Peak::Can::IsoTp;

#define STMIN_600US 0xF6

static String^ OK_KO(bool test)
{
	if (test)
		return "OK";
	return "KO";
}

static String^ STATUS_OK_KO(cantp_status test)
{
	return OK_KO(CanTpApi::StatusIsOk_2016(test));
}

/// <summary>Entry point of the program, start a small CAN ISO TP read/write example</summary>
int main(array<System::String ^> ^args)
{
	// Local variables
	cantp_status res;
	StringBuilder ^buffer = gcnew StringBuilder(500);
	UInt32 STmin;
	cantp_msg tx_msg;
	cantp_msg rx_msg;
	cantp_mapping mapping;
	cantp_handle transmitter_handle;
	cantp_handle receiver_handle;

	// Initialize variables
	transmitter_handle = cantp_handle::PCANTP_HANDLE_USBBUS1;  // TODO: modify the value according to your available PCAN devices.
	receiver_handle = cantp_handle::PCANTP_HANDLE_USBBUS2;  // TODO: modify the value according to your available PCAN devices.

	// Print version informations
	CanTpApi::GetValue_2016(cantp_handle::PCANTP_HANDLE_NONEBUS, cantp_parameter::PCANTP_PARAMETER_API_VERSION, buffer, 500);
	Console::WriteLine("PCAN-ISO-TP API Version : {0}", buffer);

	Console::WriteLine("\n");
	Console::WriteLine("WARNING: this sample is not suited for segmented communication.");
	Console::WriteLine("         Data must be contained in a Single ISO-TP frame only.");
	Console::WriteLine("         Please check sample \"06_isotp_segmented_read_write\" for standard segmented communications.");
	Console::WriteLine("\n");
	System::Threading::Thread::Sleep(2000);

	// Initialize channel: CAN2.0 - 500Kbit/s
	res = CanTpApi::Initialize_2016(transmitter_handle, cantp_baudrate::PCANTP_BAUDRATE_500K);
	Console::WriteLine("Initialize : {0}", STATUS_OK_KO(res));
	res = CanTpApi::Initialize_2016(receiver_handle, cantp_baudrate::PCANTP_BAUDRATE_500K);
	Console::WriteLine("Initialize : {0}", STATUS_OK_KO(res));

	// Create and set a receive event on receiver
	System::Threading::AutoResetEvent receive_event(false);
	if (IntPtr::Size == 4)
	{
		UInt32 iBuffer = Convert::ToUInt32(receive_event.SafeWaitHandle->DangerousGetHandle().ToInt32());
		res = CanTpApi::SetValue_2016(receiver_handle, cantp_parameter::PCANTP_PARAMETER_RECEIVE_EVENT, iBuffer, sizeof(UInt32));
	}
	else if (IntPtr::Size == 8)
	{
		Int64 iBuffer = receive_event.SafeWaitHandle->DangerousGetHandle().ToInt64();
		array<Byte> ^byteArray = BitConverter::GetBytes(iBuffer);
		res = CanTpApi::SetValue_2016(receiver_handle, cantp_parameter::PCANTP_PARAMETER_RECEIVE_EVENT, byteArray, sizeof(UInt64));
	}
	Console::WriteLine("Set receive event on receiver : {0}", STATUS_OK_KO(res));

	// Change STmin value to 600us
	STmin = STMIN_600US;
	res = CanTpApi::SetValue_2016(transmitter_handle, cantp_parameter::PCANTP_PARAMETER_SEPARATION_TIME, STmin, sizeof(UInt32));
	Console::WriteLine("Set STMIN = 600us on transmitter : {0}", STATUS_OK_KO(res));
	res = CanTpApi::SetValue_2016(receiver_handle, cantp_parameter::PCANTP_PARAMETER_SEPARATION_TIME, STmin, sizeof(UInt32));
	Console::WriteLine("Set STMIN = 600us on receiver : {0}", STATUS_OK_KO(res));

	// Allocate tx message
	res = CanTpApi::MsgDataAlloc_2016(tx_msg, cantp_msgtype::PCANTP_MSGTYPE_ISOTP);
	Console::WriteLine("Allocate tx message : {0}", STATUS_OK_KO(res));
	// Allocate rx message
	res = CanTpApi::MsgDataAlloc_2016(rx_msg, cantp_msgtype::PCANTP_MSGTYPE_NONE);
	Console::WriteLine("Allocate rx message : {0}", STATUS_OK_KO(res));

	// Create a simple physical mapping: 
	//    - Source 0xF1 (transmitter), target 0x01 (receiver), CAN id 0xA1, CAN ID flow control 0xA2
	//    - Diagnostic message in a classic format
	mapping.can_id = 0xA1;
	mapping.can_id_flow_ctrl = 0xA2;
	mapping.can_msgtype = cantp_can_msgtype::PCANTP_CAN_MSGTYPE_STANDARD;
	mapping.netaddrinfo.extension_addr = 0x00;
	mapping.netaddrinfo.format = cantp_isotp_format::PCANTP_ISOTP_FORMAT_NORMAL;
	mapping.netaddrinfo.msgtype = cantp_isotp_msgtype::PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC;
	mapping.netaddrinfo.source_addr = 0xF1;
	mapping.netaddrinfo.target_addr = 0x01;
	mapping.netaddrinfo.target_type = cantp_isotp_addressing::PCANTP_ISOTP_ADDRESSING_PHYSICAL;

	// Add mapping on channels
	res = CanTpApi::AddMapping_2016(transmitter_handle, mapping);
	Console::WriteLine("Add a simple mapping on transmitter: {0}", STATUS_OK_KO(res));
	res = CanTpApi::AddMapping_2016(receiver_handle, mapping);
	Console::WriteLine("Add a simple mapping on receiver: {0}", STATUS_OK_KO(res));

	// Initialize Tx message containing "42"
	array<Byte>^ data = { 0x42 };
	res = CanTpApi::MsgDataInit_2016(tx_msg, CanTpApi::PCANTP_CAN_ID_DEFINED_BY_NAI, mapping.can_msgtype, 1, data, mapping.netaddrinfo);
	Console::WriteLine("Initialize tx message : {0}", STATUS_OK_KO(res));

	// Write "42" message
	res = CanTpApi::Write_2016(transmitter_handle, tx_msg);
	Console::WriteLine("Write \"42\" message : {0}", STATUS_OK_KO(res));

	// Wait a receive event on receiver
	bool wait_result = receive_event.WaitOne(100);
	Console::WriteLine("Wait a message on receiver : {0}", OK_KO(wait_result));

	// If we receive something on the receiver, read the message
	if (wait_result)
	{
		res = CanTpApi::Read_2016(receiver_handle, rx_msg);
		Console::WriteLine("Read message on receiver : {0}", STATUS_OK_KO(res));
		Byte val = rx_msg.msgdata.any->data[0];
		Console::WriteLine("Check if the message is \"42\" : {0}", OK_KO(val == 0x42));
	}

	// Free messages
	res = CanTpApi::MsgDataFree_2016(rx_msg);
	Console::WriteLine("Free rx message : {0}", STATUS_OK_KO(res));
	res = CanTpApi::MsgDataFree_2016(tx_msg);
	Console::WriteLine("Free tx message : {0}", STATUS_OK_KO(res));

	// Close receive event
	if (IntPtr::Size == 4)
	{
		UInt32 iBuffer = 0;
		res = CanTpApi::SetValue_2016(receiver_handle, cantp_parameter::PCANTP_PARAMETER_RECEIVE_EVENT, iBuffer, sizeof(UInt32));
	}
	else if (IntPtr::Size == 8)
	{
		Int64 iBuffer = 0;
		array<Byte>^ byteArray = BitConverter::GetBytes(iBuffer);
		res = CanTpApi::SetValue_2016(receiver_handle, cantp_parameter::PCANTP_PARAMETER_RECEIVE_EVENT, byteArray, sizeof(UInt64));
	}
	Console::WriteLine("Stop receive event  : {0}", STATUS_OK_KO(res));
	receive_event.Close();

	// Uninitialize
	res = CanTpApi::Uninitialize_2016(transmitter_handle);
	Console::WriteLine("Uninitialize transmitter : {0}", STATUS_OK_KO(res));
	res = CanTpApi::Uninitialize_2016(receiver_handle);
	Console::WriteLine("Uninitialize receiver : {0}", STATUS_OK_KO(res));

	// Exit
	Console::WriteLine("Press any key to exit...");
	Console::Read();
}


