#include "PCANBasicCLR.h" 
#include "PCAN-ISO-TP-CLR_2016.h"
#include "heavy_data.h"

using namespace System;
using namespace System::IO;
using namespace Peak::Can::IsoTp;

#define STMIN_600US 0xF6

static String^ OK_KO(bool test)
{
	if (test)
		return "OK";
	return "KO";
}

static String^ STATUS_OK_KO(cantp_status test)
{
	return OK_KO(CanTpApi::StatusIsOk_2016(test));
}

/// <summary>Return the name of the channel</summary>
/// <param name="value">Channel handle</param>
/// <returns>String containing the name of the channel</returns>
static String ^ cantp_handle_toShortString(cantp_handle value)
{
	String ^res = "";
	switch (value)
	{
	case cantp_handle::PCANTP_HANDLE_NONEBUS: res = "NONEBUS"; break;
	case cantp_handle::PCANTP_HANDLE_ISABUS1: res = "ISA1"; break;
	case cantp_handle::PCANTP_HANDLE_ISABUS2: res = "ISA2"; break;
	case cantp_handle::PCANTP_HANDLE_ISABUS3: res = "ISA3"; break;
	case cantp_handle::PCANTP_HANDLE_ISABUS4: res = "ISA4"; break;
	case cantp_handle::PCANTP_HANDLE_ISABUS5: res = "ISA5"; break;
	case cantp_handle::PCANTP_HANDLE_ISABUS6: res = "ISA6"; break;
	case cantp_handle::PCANTP_HANDLE_ISABUS7: res = "ISA7"; break;
	case cantp_handle::PCANTP_HANDLE_ISABUS8: res = "ISA8"; break;
	case cantp_handle::PCANTP_HANDLE_DNGBUS1: res = "DNG1"; break;
	case cantp_handle::PCANTP_HANDLE_PCIBUS1: res = "PCI1"; break;
	case cantp_handle::PCANTP_HANDLE_PCIBUS2: res = "PCI2"; break;
	case cantp_handle::PCANTP_HANDLE_PCIBUS3: res = "PCI3"; break;
	case cantp_handle::PCANTP_HANDLE_PCIBUS4: res = "PCI4"; break;
	case cantp_handle::PCANTP_HANDLE_PCIBUS5: res = "PCI5"; break;
	case cantp_handle::PCANTP_HANDLE_PCIBUS6: res = "PCI6"; break;
	case cantp_handle::PCANTP_HANDLE_PCIBUS7: res = "PCI7"; break;
	case cantp_handle::PCANTP_HANDLE_PCIBUS8: res = "PCI8"; break;
	case cantp_handle::PCANTP_HANDLE_PCIBUS9: res = "PCI9"; break;
	case cantp_handle::PCANTP_HANDLE_PCIBUS10: res = "PCI10"; break;
	case cantp_handle::PCANTP_HANDLE_PCIBUS11: res = "PCI11"; break;
	case cantp_handle::PCANTP_HANDLE_PCIBUS12: res = "PCI12"; break;
	case cantp_handle::PCANTP_HANDLE_PCIBUS13: res = "PCI13"; break;
	case cantp_handle::PCANTP_HANDLE_PCIBUS14: res = "PCI14"; break;
	case cantp_handle::PCANTP_HANDLE_PCIBUS15: res = "PCI15"; break;
	case cantp_handle::PCANTP_HANDLE_PCIBUS16: res = "PCI16"; break;
	case cantp_handle::PCANTP_HANDLE_USBBUS1: res = "USB1"; break;
	case cantp_handle::PCANTP_HANDLE_USBBUS2: res = "USB2"; break;
	case cantp_handle::PCANTP_HANDLE_USBBUS3: res = "USB3"; break;
	case cantp_handle::PCANTP_HANDLE_USBBUS4: res = "USB4"; break;
	case cantp_handle::PCANTP_HANDLE_USBBUS5: res = "USB5"; break;
	case cantp_handle::PCANTP_HANDLE_USBBUS6: res = "USB6"; break;
	case cantp_handle::PCANTP_HANDLE_USBBUS7: res = "USB7"; break;
	case cantp_handle::PCANTP_HANDLE_USBBUS8: res = "USB8"; break;
	case cantp_handle::PCANTP_HANDLE_USBBUS9: res = "USB9"; break;
	case cantp_handle::PCANTP_HANDLE_USBBUS10: res = "USB10"; break;
	case cantp_handle::PCANTP_HANDLE_USBBUS11: res = "USB11"; break;
	case cantp_handle::PCANTP_HANDLE_USBBUS12: res = "USB12"; break;
	case cantp_handle::PCANTP_HANDLE_USBBUS13: res = "USB13"; break;
	case cantp_handle::PCANTP_HANDLE_USBBUS14: res = "USB14"; break;
	case cantp_handle::PCANTP_HANDLE_USBBUS15: res = "USB15"; break;
	case cantp_handle::PCANTP_HANDLE_USBBUS16: res = "USB16"; break;
	case cantp_handle::PCANTP_HANDLE_PCCBUS1: res = "PCC1"; break;
	case cantp_handle::PCANTP_HANDLE_PCCBUS2: res = "PCC2"; break;
	case cantp_handle::PCANTP_HANDLE_LANBUS1: res = "LAN1"; break;
	case cantp_handle::PCANTP_HANDLE_LANBUS2: res = "LAN2"; break;
	case cantp_handle::PCANTP_HANDLE_LANBUS3: res = "LAN3"; break;
	case cantp_handle::PCANTP_HANDLE_LANBUS4: res = "LAN4"; break;
	case cantp_handle::PCANTP_HANDLE_LANBUS5: res = "LAN5"; break;
	case cantp_handle::PCANTP_HANDLE_LANBUS6: res = "LAN6"; break;
	case cantp_handle::PCANTP_HANDLE_LANBUS7: res = "LAN7"; break;
	case cantp_handle::PCANTP_HANDLE_LANBUS8: res = "LAN8"; break;
	case cantp_handle::PCANTP_HANDLE_LANBUS9: res = "LAN9"; break;
	case cantp_handle::PCANTP_HANDLE_LANBUS10: res = "LAN10"; break;
	case cantp_handle::PCANTP_HANDLE_LANBUS11: res = "LAN11"; break;
	case cantp_handle::PCANTP_HANDLE_LANBUS12: res = "LAN12"; break;
	case cantp_handle::PCANTP_HANDLE_LANBUS13: res = "LAN13"; break;
	case cantp_handle::PCANTP_HANDLE_LANBUS14: res = "LAN14"; break;
	case cantp_handle::PCANTP_HANDLE_LANBUS15: res = "LAN15"; break;
	case cantp_handle::PCANTP_HANDLE_LANBUS16: res = "LAN16"; break;
	default: res = "UNKNOWN"; break;
	}
	return res;
}

/// <summary>Wait for a message to be received on a channel and show reception progress, in safe mode</summary>
/// <param name="handle">Channel handle</param>
/// <param name="msg_buffer">Received message</param>
/// <returns>a status</returns>
static cantp_status wait_and_show_progress(cantp_handle handle, cantp_msg %msg_buffer)
{
	cantp_status res = cantp_status::PCANTP_STATUS_OK;
	cantp_msgprogress progress {};

	// Set a buffer in order to receive a copy of the pending message
	cantp_msg msgpend;
	progress.buffer = &msgpend;

	do
	{
		System::Threading::Thread::Sleep(300);

		// Refresh progress structure
		res = CanTpApi::GetMsgProgress_2016(handle, msg_buffer, cantp_msgdirection::PCANTP_MSGDIRECTION_RX, progress);

		// Show progress pencentage and last bytes
		Console::Write("RX Progress on {0} = {1}%, last bytes ",
			cantp_handle_toShortString(handle),
			progress.percentage);
		for (UInt32 i = msgpend.msgdata.any->length - 10; i < msgpend.msgdata.any->length; i++)
			Console::Write("{0:X} ", msgpend.msgdata.isotp->data[i]);
		Console::WriteLine("{0}", OK_KO(CanTpApi::StatusIsOk_2016(res)));

	} while (progress.state == cantp_msgprogress_state::PCANTP_MSGPROGRESS_STATE_PROCESSING);

	CanTpApi::MsgDataFree_2016(msgpend);

	// The message is received, read it
	res = CanTpApi::Read_2016(handle, msg_buffer);

	return res;
}

/// <summary>Wait and read single message</summary>
/// <param name="handle">Channel handle</param>
/// <param name="msg_buffer">Received message (out)</param>
/// <param name="evt">Receive event associated to the channel</param>
/// <returns>Status</returns>
static cantp_status wait_and_read_msg(cantp_handle handle, System::Threading::AutoResetEvent %evt, cantp_msg %msg_buffer)
{
	cantp_status res = cantp_status::PCANTP_STATUS_NO_MESSAGE;

	// Wait a receive event and get a message
	// If we receive something, read the message
	if (evt.WaitOne(100))
	{
		res = CanTpApi::Read_2016(handle, msg_buffer);

		// Check if the message is totally received (only in ISOTP mode)
		if (CanTpApi::StatusIsOk_2016(res)
			&& (cantp_msgtype::PCANTP_MSGTYPE_ISOTP & msg_buffer.type) == cantp_msgtype::PCANTP_MSGTYPE_ISOTP)
		{
			// test the rx indication
			if ((msg_buffer.msgdata.isotp->netaddrinfo.msgtype & cantp_isotp_msgtype::PCANTP_ISOTP_MSGTYPE_FLAG_INDICATION_RX) == cantp_isotp_msgtype::PCANTP_ISOTP_MSGTYPE_FLAG_INDICATION_RX)
			{
				// The message is being received, wait and show progress
				res = wait_and_show_progress(handle, msg_buffer);
			}
		}
	}

	return res;
}

/// <summary>Write message data in a binary file</summary>
/// <param name="file_path">Output file path</param>
/// <param name="msg">Message to write</param>
static void write_binary_output_file(String ^file_path, cantp_msg msg)
{
	if (msg.msgdata.any != nullptr)
	{
		UInt32 len = msg.msgdata.any->length;
		try
		{
			FileStream ^stream = gcnew FileStream(file_path, FileMode::Create);
			BinaryWriter ^writer = gcnew BinaryWriter(stream);
			for (UInt32 i = 0; i < len; i += 2)
			{
				Byte lsb = msg.msgdata.any->data[i];
				Byte msb = msg.msgdata.any->data[i + 1];
				writer->Write(msb);
				writer->Write(lsb);
			}
			writer->Close();
			stream->Close();
		}
		catch (Exception^)
		{

		}
	}
}

/// <summary>
///   Entry point of the program, start a small CAN ISO TP read/write example
///     -> Send a message containing Peak logo 
///     -> Receive message, wait complete reception (show progress)
///     -> Show loopback message and received message (Peak logo)
/// </summary>
int main(array<System::String ^> ^args)
{
	// Local variables
	cantp_status res;
	StringBuilder ^buffer = gcnew StringBuilder(500);
	UInt32 STmin;
	cantp_msg tx_msg;
	cantp_msg rx_msg;
	cantp_msg loopback_msg;
	cantp_mapping mapping;
	cantp_mapping reverse_mapping;
	cantp_handle transmitter_handle;
	cantp_handle receiver_handle;

	// Initialize variables
	transmitter_handle = cantp_handle::PCANTP_HANDLE_USBBUS1;  // TODO: modify the value according to your available PCAN devices.
	receiver_handle = cantp_handle::PCANTP_HANDLE_USBBUS2;  // TODO: modify the value according to your available PCAN devices.

	// Print version informations
	CanTpApi::GetValue_2016(cantp_handle::PCANTP_HANDLE_NONEBUS, cantp_parameter::PCANTP_PARAMETER_API_VERSION, buffer, 500);
	Console::WriteLine("PCAN-ISO-TP API Version : {0}", buffer);

	// Initialize channel: CAN2.0 - 500Kbit/s
	res = CanTpApi::Initialize_2016(transmitter_handle, cantp_baudrate::PCANTP_BAUDRATE_500K);
	Console::WriteLine("Initialize transmitter : {0}", STATUS_OK_KO(res));
	res = CanTpApi::Initialize_2016(receiver_handle, cantp_baudrate::PCANTP_BAUDRATE_500K);
	Console::WriteLine("Initialize receiver : {0}", STATUS_OK_KO(res));

	// Change STmin value to 600us
	STmin = STMIN_600US;
	res = CanTpApi::SetValue_2016(transmitter_handle, cantp_parameter::PCANTP_PARAMETER_SEPARATION_TIME, STmin, sizeof(UInt32));
	Console::WriteLine("Set STMIN = 600us on transmitter : {0}", STATUS_OK_KO(res));
	res = CanTpApi::SetValue_2016(receiver_handle, cantp_parameter::PCANTP_PARAMETER_SEPARATION_TIME, STmin, sizeof(UInt32));
	Console::WriteLine("Set STMIN = 600us on receiver : {0}", STATUS_OK_KO(res));

	// Create and set a receive event on transmitter
	System::Threading::AutoResetEvent transmitter_receive_event(false);
	if (IntPtr::Size == 4)
	{
		UInt32 iBuffer = Convert::ToUInt32(transmitter_receive_event.SafeWaitHandle->DangerousGetHandle().ToInt32());
		res = CanTpApi::SetValue_2016(transmitter_handle, cantp_parameter::PCANTP_PARAMETER_RECEIVE_EVENT, iBuffer, sizeof(UInt32));
	}
	else if (IntPtr::Size == 8)
	{
		Int64 iBuffer = transmitter_receive_event.SafeWaitHandle->DangerousGetHandle().ToInt64();
		array<Byte> ^byteArray = BitConverter::GetBytes(iBuffer);
		res = CanTpApi::SetValue_2016(transmitter_handle, cantp_parameter::PCANTP_PARAMETER_RECEIVE_EVENT, byteArray, sizeof(UInt64));
	}
	Console::WriteLine("Set receive event on transmitter : {0}", STATUS_OK_KO(res));

	// Create and set a receive event on receiver
	System::Threading::AutoResetEvent receiver_receive_event(false);
	if (IntPtr::Size == 4)
	{
		UInt32 iBuffer = Convert::ToUInt32(receiver_receive_event.SafeWaitHandle->DangerousGetHandle().ToInt32());
		res = CanTpApi::SetValue_2016(receiver_handle, cantp_parameter::PCANTP_PARAMETER_RECEIVE_EVENT, iBuffer, sizeof(UInt32));
	}
	else if (IntPtr::Size == 8)
	{
		Int64 iBuffer = receiver_receive_event.SafeWaitHandle->DangerousGetHandle().ToInt64();
		array<Byte> ^byteArray = BitConverter::GetBytes(iBuffer);
		res = CanTpApi::SetValue_2016(receiver_handle, cantp_parameter::PCANTP_PARAMETER_RECEIVE_EVENT, byteArray, sizeof(UInt64));
	}
	Console::WriteLine("Set receive event on receiver : {0}", STATUS_OK_KO(res));

	// Allocate messages
	res = CanTpApi::MsgDataAlloc_2016(tx_msg, cantp_msgtype::PCANTP_MSGTYPE_ISOTP);
	Console::WriteLine("Allocate tx message : {0}", STATUS_OK_KO(res));
	res = CanTpApi::MsgDataAlloc_2016(rx_msg, cantp_msgtype::PCANTP_MSGTYPE_NONE);
	Console::WriteLine("Allocate rx message : {0}", STATUS_OK_KO(res));
	res = CanTpApi::MsgDataAlloc_2016(loopback_msg, cantp_msgtype::PCANTP_MSGTYPE_NONE);
	Console::WriteLine("Allocate loopback message : {0}", STATUS_OK_KO(res));

	// Create a simple physical mapping: 
	//    - Source 0xF1 (transmitter), target 0x01 (receiver), CAN id 0xA1, CAN ID flow control 0xA2
	//    - Diagnostic message in a classic format
	mapping.can_id = 0xA1;
	mapping.can_id_flow_ctrl = 0xA2;
	mapping.can_msgtype = cantp_can_msgtype::PCANTP_CAN_MSGTYPE_STANDARD;
	mapping.can_tx_dlc = 0x0;
	mapping.netaddrinfo.extension_addr = 0x00;
	mapping.netaddrinfo.format = cantp_isotp_format::PCANTP_ISOTP_FORMAT_NORMAL;
	mapping.netaddrinfo.msgtype = cantp_isotp_msgtype::PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC;
	mapping.netaddrinfo.source_addr = 0xF1;
	mapping.netaddrinfo.target_addr = 0x01;
	mapping.netaddrinfo.target_type = cantp_isotp_addressing::PCANTP_ISOTP_ADDRESSING_PHYSICAL;

	// Create the associated reversed mapping: 
	reverse_mapping = mapping;
	reverse_mapping.can_id = mapping.can_id_flow_ctrl;
	reverse_mapping.can_id_flow_ctrl = mapping.can_id;
	reverse_mapping.netaddrinfo.source_addr = mapping.netaddrinfo.target_addr;
	reverse_mapping.netaddrinfo.target_addr = mapping.netaddrinfo.source_addr;

	// Add mapping on channels
	res = CanTpApi::AddMapping_2016(transmitter_handle, mapping);
	Console::WriteLine("Add a simple mapping on transmitter {0} : {1}", cantp_handle_toShortString(transmitter_handle), STATUS_OK_KO(res));
	res = CanTpApi::AddMapping_2016(transmitter_handle, reverse_mapping);
	Console::WriteLine("Add the reverse mapping on transmitter {0} : {1}", cantp_handle_toShortString(transmitter_handle), STATUS_OK_KO(res));
	res = CanTpApi::AddMapping_2016(receiver_handle, mapping);
	Console::WriteLine("Add a simple mapping on receiver {0} : {1}", cantp_handle_toShortString(receiver_handle), STATUS_OK_KO(res));
	res = CanTpApi::AddMapping_2016(receiver_handle, reverse_mapping);
	Console::WriteLine("Add the reverse mapping on receiver {0} : {1}", cantp_handle_toShortString(receiver_handle), STATUS_OK_KO(res));

	// Initialize Tx message containing a heavy data
	res = CanTpApi::MsgDataInit_2016(tx_msg, CanTpApi::PCANTP_CAN_ID_DEFINED_BY_NAI, mapping.can_msgtype, (UInt32)HEAVY_DATA_SIZE, heavy_data::HEAVY_DATA, mapping.netaddrinfo);
	Console::WriteLine("Initialize tx message: {0}", STATUS_OK_KO(res));

	// Write message on transmitter
	res = CanTpApi::Write_2016(transmitter_handle, tx_msg);
	Console::WriteLine("Write message on transmitter {0} : {1}", cantp_handle_toShortString(transmitter_handle), STATUS_OK_KO(res));

	// Read loopback message and save it 
	res = wait_and_read_msg(transmitter_handle, transmitter_receive_event, loopback_msg);
	Console::WriteLine("Read loopback message on transmitter {0} : {1}", cantp_handle_toShortString(transmitter_handle), STATUS_OK_KO(res));
	bool fileTxOk = CanTpApi::StatusIsOk_2016(res);
	if (fileTxOk)
		write_binary_output_file("loopback_message.png", loopback_msg);

	// Read received message and save it
	res = wait_and_read_msg(receiver_handle, receiver_receive_event, rx_msg);
	Console::WriteLine("Read message on receiver {0} : {1}", cantp_handle_toShortString(receiver_handle), STATUS_OK_KO(res));
	bool fileRxOk = CanTpApi::StatusIsOk_2016(res);
	if (fileRxOk)
		write_binary_output_file("received_message.png", rx_msg);

	// Show & remove result: 
	if (fileTxOk)
	{
		Console::WriteLine(" >>> Opening transmitted message (PEAK logo).");
		if (File::Exists("loopback_message.png"))
		{
			System::Diagnostics::Process ^ proc = gcnew System::Diagnostics::Process();
			proc->StartInfo->FileName = "loopback_message.png";
			proc->Start();
			proc->Close();
		}
	}
	else
		Console::WriteLine("\n >>> Failed to successfully transmit message (PEAK logo)!");

	if (fileRxOk)
	{
		Console::WriteLine(" >>> Opening received message (PEAK logo).");
		if (File::Exists("received_message.png"))
		{
			System::Diagnostics::Process ^ proc = gcnew System::Diagnostics::Process();
			proc->StartInfo->FileName = "received_message.png";
			proc->Start();
			proc->Close();
		}
	}
	else
		Console::WriteLine("\n >>> Failed to successfully receive message (PEAK logo)!");

	// Free messages
	res = CanTpApi::MsgDataFree_2016(rx_msg);
	Console::WriteLine("Free rx message : {0}", STATUS_OK_KO(res));
	res = CanTpApi::MsgDataFree_2016(tx_msg);
	Console::WriteLine("Free tx message : {0}", STATUS_OK_KO(res));
	res = CanTpApi::MsgDataFree_2016(loopback_msg);
	Console::WriteLine("Free loopback message : {0}", STATUS_OK_KO(res));

	// Close receive event on transmitter
	if (IntPtr::Size == 4)
	{
		UInt32 iBuffer = 0;
		res = CanTpApi::SetValue_2016(transmitter_handle, cantp_parameter::PCANTP_PARAMETER_RECEIVE_EVENT, iBuffer, sizeof(UInt32));
	}
	else if (IntPtr::Size == 8)
	{
		Int64 iBuffer = 0;
		array<Byte>^ byteArray = BitConverter::GetBytes(iBuffer);
		res = CanTpApi::SetValue_2016(transmitter_handle, cantp_parameter::PCANTP_PARAMETER_RECEIVE_EVENT, byteArray, sizeof(UInt64));
	}
	Console::WriteLine("Stop receive event on transmitter {0} : {1}", cantp_handle_toShortString(transmitter_handle), STATUS_OK_KO(res));
	transmitter_receive_event.Close();

	// Close receive event on receiver
	if (IntPtr::Size == 4)
	{
		UInt32 iBuffer = 0;
		res = CanTpApi::SetValue_2016(receiver_handle, cantp_parameter::PCANTP_PARAMETER_RECEIVE_EVENT, iBuffer, sizeof(UInt32));
	}
	else if (IntPtr::Size == 8)
	{
		Int64 iBuffer = 0;
		array<Byte>^ byteArray = BitConverter::GetBytes(iBuffer);
		res = CanTpApi::SetValue_2016(receiver_handle, cantp_parameter::PCANTP_PARAMETER_RECEIVE_EVENT, byteArray, sizeof(UInt64));
	}
	Console::WriteLine("Stop receive event on receiver {0} : {1}", cantp_handle_toShortString(receiver_handle), STATUS_OK_KO(res));
	receiver_receive_event.Close();

	// Uninitialize channels
	res = CanTpApi::Uninitialize_2016(transmitter_handle);
	Console::WriteLine("Uninitialize transmitter {0} : {1}", cantp_handle_toShortString(transmitter_handle), STATUS_OK_KO(res));
	res = CanTpApi::Uninitialize_2016(receiver_handle);
	Console::WriteLine("Uninitialize receiver {0} : {1}", cantp_handle_toShortString(receiver_handle), STATUS_OK_KO(res));

	// Remove results and exit
	Console::WriteLine("Press any key to exit...");
	Console::Read();
	try
	{
		File::Delete("loopback_message.png");
		File::Delete("received_message.png");
	}
	catch (Exception^)
	{

	}
}


