#include "PCANBasicCLR.h" 
#include "PCAN-ISO-TP-CLR_2016.h"

using namespace System;
using namespace System::IO;
using namespace Peak::Can::IsoTp;

static String^ OK_KO(bool test)
{
	if (test)
		return "OK";
	return "KO";
}

static String^ STATUS_OK_KO(cantp_status test)
{
	return OK_KO(CanTpApi::StatusIsOk_2016(test));
}

/// <summary>Entry point of the program, start a small ISO TP classic CAN read/write example</summary>
int main(array<System::String ^> ^args)
{
	// Local variables
	cantp_status res;
	StringBuilder ^buffer = gcnew StringBuilder(500);
	cantp_msg tx_msg;
	cantp_msg rx_msg;
	cantp_handle transmitter_handle;
	cantp_handle receiver_handle;

	// Initialize variables
	transmitter_handle = cantp_handle::PCANTP_HANDLE_USBBUS1;  // TODO: modify the value according to your available PCAN devices.
	receiver_handle = cantp_handle::PCANTP_HANDLE_USBBUS2;  // TODO: modify the value according to your available PCAN devices.

	// Print version informations
	CanTpApi::GetValue_2016(cantp_handle::PCANTP_HANDLE_NONEBUS, cantp_parameter::PCANTP_PARAMETER_API_VERSION, buffer, 500);
	Console::WriteLine("PCAN-ISO-TP API Version : {0}", buffer);

	// Initialize channel: CAN2.0 - 500Kbit/s
	res = CanTpApi::Initialize_2016(transmitter_handle, cantp_baudrate::PCANTP_BAUDRATE_500K);
	Console::WriteLine("Initialize : {0}", STATUS_OK_KO(res));
	res = CanTpApi::Initialize_2016(receiver_handle, cantp_baudrate::PCANTP_BAUDRATE_500K);
	Console::WriteLine("Initialize : {0}", STATUS_OK_KO(res));

	// Create and set a receive event on receiver
	System::Threading::AutoResetEvent receive_event(false);
	if (IntPtr::Size == 4)
	{
		UInt32 iBuffer = Convert::ToUInt32(receive_event.SafeWaitHandle->DangerousGetHandle().ToInt32());
		res = CanTpApi::SetValue_2016(receiver_handle, cantp_parameter::PCANTP_PARAMETER_RECEIVE_EVENT, iBuffer, sizeof(UInt32));
	}
	else if (IntPtr::Size == 8)
	{
		Int64 iBuffer = receive_event.SafeWaitHandle->DangerousGetHandle().ToInt64();
		array<Byte> ^byteArray = BitConverter::GetBytes(iBuffer);
		res = CanTpApi::SetValue_2016(receiver_handle, cantp_parameter::PCANTP_PARAMETER_RECEIVE_EVENT, byteArray, sizeof(UInt64));
	}
	Console::WriteLine("Set receive event on receiver : {0}", STATUS_OK_KO(res));

	// Allocate tx CAN message
	res = CanTpApi::MsgDataAlloc_2016(tx_msg, cantp_msgtype::PCANTP_MSGTYPE_CAN);
	Console::WriteLine("Allocate tx message : {0}", STATUS_OK_KO(res));
	// Allocate rx message
	res = CanTpApi::MsgDataAlloc_2016(rx_msg, cantp_msgtype::PCANTP_MSGTYPE_NONE);
	Console::WriteLine("Allocate rx message : {0}", STATUS_OK_KO(res));

	// Initialize Tx message containing "42"
	UInt32 can_id = 0xA1;
	array<Byte>^ data = { 0x42 };
	res = CanTpApi::MsgDataInit_2016(tx_msg, can_id, cantp_can_msgtype::PCANTP_CAN_MSGTYPE_STANDARD, 1, data);
	Console::WriteLine("Initialize tx message : {0}", STATUS_OK_KO(res));

	// Write "42" message
	res = CanTpApi::Write_2016(transmitter_handle, tx_msg);
	Console::WriteLine("Write \"42\" message : {0}", STATUS_OK_KO(res));

	// Wait a receive event on receiver
	bool wait_result = receive_event.WaitOne(100);
	Console::WriteLine("Wait a message on receiver : {0}", OK_KO(wait_result));

	// If we receive something on the receiver, read the message
	if (wait_result)
	{
		res = CanTpApi::Read_2016(receiver_handle, rx_msg);
		Console::WriteLine("Read message on receiver : {0}", STATUS_OK_KO(res));
		Byte val = rx_msg.msgdata.any->data[0];
		Console::WriteLine("Check if the message is \"42\" : {0}", OK_KO(val == 0x42));
	}

	// Free messages
	res = CanTpApi::MsgDataFree_2016(rx_msg);
	Console::WriteLine("Free rx message : {0}", STATUS_OK_KO(res));
	res = CanTpApi::MsgDataFree_2016(tx_msg);
	Console::WriteLine("Free tx message : {0}", STATUS_OK_KO(res));

	// Close receive event
	if (IntPtr::Size == 4)
	{
		UInt32 iBuffer = 0;
		res = CanTpApi::SetValue_2016(receiver_handle, cantp_parameter::PCANTP_PARAMETER_RECEIVE_EVENT, iBuffer, sizeof(UInt32));
	}
	else if (IntPtr::Size == 8)
	{
		Int64 iBuffer = 0;
		array<Byte>^ byteArray = BitConverter::GetBytes(iBuffer);
		res = CanTpApi::SetValue_2016(receiver_handle, cantp_parameter::PCANTP_PARAMETER_RECEIVE_EVENT, byteArray, sizeof(UInt64));
	}
	Console::WriteLine("Stop receive event  : {0}", STATUS_OK_KO(res));
	receive_event.Close();

	// Uninitialize
	res = CanTpApi::Uninitialize_2016(transmitter_handle);
	Console::WriteLine("Uninitialize transmitter : {0}", STATUS_OK_KO(res));
	res = CanTpApi::Uninitialize_2016(receiver_handle);
	Console::WriteLine("Uninitialize receiver : {0}", STATUS_OK_KO(res));

	// Exit
	Console::WriteLine("Press any key to exit...");
	Console::Read();
}


