#include "PCANBasicCLR.h" 
#include "PCAN-ISO-TP-CLR_2016.h"

using namespace System;
using namespace System::IO;
using namespace Peak::Can::IsoTp;

#define USE_CAN_FD false
#define CAN_TX_DL_CODE 0x0D
#define CAN_TX_DL_VALUE 32
#define N_SA 0xF1
#define N_TA_PHYS 0x13
#define N_TA_FUNC 0x26
#define N_RA 0x52
#define TEST_PHYS_TRANSMISSION true
#define TEST_FUNC_TRANSMISSION true
#define TEST_BURST_TRANSMISSION true
const UInt32 MESSAGE_MAX_LENGTH = CanTpApi::PCANTP_MAX_LENGTH_ISOTP2004 + 1000;
#define USE_GETCH false

/// <summary>Outputs a summary of a received message.</summary>
/// <param name="message">The message.</param>
/// <param name="timestamp">The timestamp of the message.</param>
/// <param name="status">status of the Read function.</param>
/// <returns>1 on success (2 if extra information is available within status), 0 if no message and -1 on error</returns>
static int display_message(cantp_msg message, cantp_timestamp timestamp, cantp_status status)
{
	String^ strLoopback = "";
	String^ strMsgType = "";
	UInt32 displayedDataLength = 0;
	int result;

	// check status
	result = CanTpApi::StatusIsOk_2016(status) ? 1 : -1;
	if (result != 1)
	{
		if (CanTpApi::StatusIsOk_2016(status, cantp_status::PCANTP_STATUS_NO_MESSAGE))
		{
			// no message received ignore
			result = 0;
		}
		else
		{
			Console::WriteLine("Failed to read message (sts=0x{0:X}).", (UInt32)status);
		}
		return result;
	}

	// check bus errors within status
	if (!CanTpApi::StatusIsOk_2016(status, cantp_status::PCANTP_STATUS_OK, true))
	{
		Console::WriteLine("Status returned information/warnings (sts=0x{0:X}).", (UInt32)status);
		result = 2;
	}

	// check message
	if (message.msgdata.any == nullptr)
	{
		Console::WriteLine("Failed to get data of message");
		return -1;
	}

	// check if message has the loopback flag
	if ((message.msgdata.any->flags & cantp_msgflag::PCANTP_MSGFLAG_LOOPBACK) != 0)
	{
		strLoopback = "loopback ";
	}

	// format message's type
	switch (message.type)
	{
	case cantp_msgtype::PCANTP_MSGTYPE_CAN:
		strMsgType = "PCANTP_MSGTYPE_CAN";
		break;
	case cantp_msgtype::PCANTP_MSGTYPE_CANFD:
		strMsgType = "PCANTP_MSGTYPE_CANFD";
		break;
	case cantp_msgtype::PCANTP_MSGTYPE_ISOTP:
		strMsgType = "PCANTP_MSGTYPE_ISOTP";
		break;
	case cantp_msgtype::PCANTP_MSGTYPE_NONE:
		strMsgType = "PCANTP_MSGTYPE_NONE";
		break;
	default:
		strMsgType = "Unknown";
		break;
	}

	// Only display data if network result is OK
	if (message.msgdata.any->netstatus == cantp_netstatus::PCANTP_NETSTATUS_OK)
	{
		displayedDataLength = message.msgdata.any->length;
	}

	// Display generic information
	String ^strRes = message.msgdata.any->netstatus != cantp_netstatus::PCANTP_NETSTATUS_OK ? "ERROR !!!" : "OK !";
	Console::WriteLine("\n{0} - Received {1}{2} message : canid=0x{3:X}, length={4} - result: 0x{5:X} - {6}",
		timestamp,
		strLoopback,
		strMsgType,
		message.can_info.can_id,
		message.msgdata.any->length,
		message.msgdata.any->netstatus,
		strRes);

	if (message.type == cantp_msgtype::PCANTP_MSGTYPE_ISOTP)
	{
		if (message.msgdata.isotp == nullptr)
		{
			Console::WriteLine("Failed to get data of message");
			return -1;
		}
		String^ strPending = "Completed";
		// Limit displayed data if message is pending
		if (displayedDataLength > 0 && ((message.msgdata.isotp->netaddrinfo.msgtype & cantp_isotp_msgtype::PCANTP_ISOTP_MSGTYPE_FLAG_INDICATION) != 0))
		{
			displayedDataLength = (message.msgdata.any->length > 7) ? 7 : message.msgdata.any->length;
			strPending = "/!\\ Pending";
		}
		// Display iso-tp message's information
		Console::WriteLine("\t{0} message from 0x{1:X} (to 0x{2:X}, with RA 0x{3:X})",
			strPending,
			message.msgdata.isotp->netaddrinfo.source_addr,
			message.msgdata.isotp->netaddrinfo.target_addr,
			message.msgdata.isotp->netaddrinfo.extension_addr);
	}

	if (displayedDataLength > 0)
	{
		StringBuilder^ buffer = gcnew StringBuilder();
		// display data
		buffer->AppendFormat("\t\\-> Length: {0}, Data= ", message.msgdata.any->length);
		for (UInt32 i = 0; i < displayedDataLength; i++)
		{
			buffer->AppendFormat("{0:X} ", message.msgdata.any->data[i]);
		}
		if (displayedDataLength != message.msgdata.any->length)
		{
			buffer->Append("...");
		}
		Console::WriteLine(buffer);
	}
	return result;
}

// Function called to clean opened ISO-TP channels
static void consoleExit()
{
	CanTpApi::Uninitialize_2016(cantp_handle::PCANTP_HANDLE_NONEBUS);
}

// An enumerated type for the control messages sent to the handler routine.
public enum CtrlTypes
{
	CTRL_C_EVENT = 0,
	CTRL_BREAK_EVENT,
	CTRL_CLOSE_EVENT,
	CTRL_LOGOFF_EVENT = 5,
	CTRL_SHUTDOWN_EVENT
};
// Callback to handle closure of the console window
static bool ConsoleCtrlCheck(CtrlTypes ctrlType)
{
	bool isClosing = false;
	switch (ctrlType)
	{
	case CtrlTypes::CTRL_C_EVENT:
	case CtrlTypes::CTRL_BREAK_EVENT:
	case CtrlTypes::CTRL_CLOSE_EVENT:
	case CtrlTypes::CTRL_LOGOFF_EVENT:
	case CtrlTypes::CTRL_SHUTDOWN_EVENT:
		isClosing = true;
		break;

	default:
		break;
	}
	if (isClosing)
		consoleExit();
	return false;
}
// A delegate type to be used as the handler routine for SetConsoleCtrlHandler.
public delegate bool HandlerRoutine(CtrlTypes CtrlType);
// Declare the SetConsoleCtrlHandler function as external and receiving a delegate.
[DllImport("Kernel32")]
extern "C" bool SetConsoleCtrlHandler(HandlerRoutine^ Handler, bool Add);
// Handler for unhandled exceptions
static void MyHandler(Object^ sender, UnhandledExceptionEventArgs^ args)
{
	consoleExit();
}

static cantp_handle convertToHandle(String^ s)
{
	cantp_handle res = cantp_handle::PCANTP_HANDLE_NONEBUS;
	try
	{
		res = (cantp_handle)Convert::ToUInt32(s);
	}
	catch (Exception^)
	{
		// maybe the string is prefixed by 0x so try hexa conversion
		try
		{
			res = (cantp_handle)Convert::ToUInt32(s, 16);
		}
		catch (Exception^)
		{
			res = cantp_handle::PCANTP_HANDLE_NONEBUS;
		}
	}
	return res;
}

static cantp_baudrate convertToBaudRate(String^ s)
{
	cantp_baudrate res = (cantp_baudrate)0;
	try
	{
		res = (cantp_baudrate)Convert::ToUInt32(s);
	}
	catch (Exception^)
	{
		// maybe the string is prefixed by 0x so try hexa conversion
		try
		{
			res = (cantp_baudrate)Convert::ToUInt32(s, 16);
		}
		catch (Exception^)
		{
			res = (cantp_baudrate)0;
		}
	}
	return res;
}

static bool convertToBool(String^ s)
{
	bool res = false;
	try
	{
		res = Convert::ToBoolean(s);
	}
	catch (Exception^)
	{
		// maybe the string is 0 or 1 so try int conversion
		try
		{
			Byte b = Convert::ToByte(s);
			if (b == 0)
				res = false;
			else if (b == 1)
				res = true;
			else
				res = false;
		}
		catch (Exception^)
		{
			res = false;
		}
	}
	return res;
}

/// <summary>Configure mappings to handle a two-way communication and a functional communication (broadcast).</summary>
/// <param name="channel">CAN channel to add </param>
/// <param name="can_msgtype">CAN frame </param>
/// <param name="source_addr">ISO-TP source address</param>
/// <param name="target_addr">ISO-TP target address</param>
/// <param name="extension_addr">ISO-TP extension address</param>
/// <param name="target_addr_func">ISO-TP target address for functional addressing</param>
/// <param name="can_id">CAN ID used to communicate for physical addressing</param>
/// <param name="can_id_flow_ctrl">Flow Control CAN ID used during physical addressing communication</param>
/// <param name="can_id_func">CAN ID used to communicate for functionnal addressing</param>
/// <param name="isotp_msgtype">ISOTP message's type</param>
/// <param name="isotp_format">ISOTP message's format</param>
/// <return>true on successful configuration</return>
static bool initialize_mappings(cantp_handle channel, cantp_can_msgtype can_msgtype, UInt16 source_addr, UInt16 target_addr, Byte extension_addr, UInt16 target_addr_func, UInt32 can_id, UInt32 can_id_flow_ctrl, UInt32 can_id_func, cantp_isotp_msgtype isotp_msgtype, cantp_isotp_format isotp_format)
{
	bool result = true;
	// {} allows to clean variables (it is common to leave can_tx_dl uninitialized which can lead to invalid 0xCC values)
	cantp_mapping mapping_phys_rx {};
	cantp_mapping mapping_phys_tx {};
	cantp_mapping mapping_func {};
	cantp_status status;

	// configure a mapping to transmit physical message
	mapping_phys_tx.can_id = can_id;
	mapping_phys_tx.can_id_flow_ctrl = can_id_flow_ctrl;
	mapping_phys_tx.can_msgtype = can_msgtype;
	mapping_phys_tx.netaddrinfo.format = isotp_format;
	mapping_phys_tx.netaddrinfo.msgtype = isotp_msgtype;
	mapping_phys_tx.netaddrinfo.target_type = cantp_isotp_addressing::PCANTP_ISOTP_ADDRESSING_PHYSICAL;
	mapping_phys_tx.netaddrinfo.source_addr = source_addr;
	mapping_phys_tx.netaddrinfo.target_addr = target_addr;
	mapping_phys_tx.netaddrinfo.extension_addr = extension_addr;
	status = CanTpApi::AddMapping_2016(channel, mapping_phys_tx);
	if (!CanTpApi::StatusIsOk_2016(status))
	{
		Console::WriteLine("Failed to configure mapping (can_id=0x{0:X}) (sts=0x{1:X})", mapping_phys_tx.can_id, (UInt32)status);
		result = false;
	}
	// configure another mapping to receive physical message (invert source and target in previous mapping)
	mapping_phys_rx = mapping_phys_tx;
	mapping_phys_rx.can_id = mapping_phys_tx.can_id_flow_ctrl;
	mapping_phys_rx.can_id_flow_ctrl = mapping_phys_tx.can_id;
	mapping_phys_rx.netaddrinfo.source_addr = mapping_phys_tx.netaddrinfo.target_addr;
	mapping_phys_rx.netaddrinfo.target_addr = mapping_phys_tx.netaddrinfo.source_addr;
	status = CanTpApi::AddMapping_2016(channel, mapping_phys_rx);
	if (!CanTpApi::StatusIsOk_2016(status))
	{
		Console::WriteLine("Failed to configure mapping (can_id=0x{0:X}) (sts=0x{1:X})", mapping_phys_rx.can_id, (UInt32)status);
		result = false;
	}
	if (can_id_func != 0xFFFFFFFF)
	{
		// configure another mapping to transmit functional message
		mapping_func = mapping_phys_tx;
		mapping_func.can_id = can_id_func;
		mapping_func.can_id_flow_ctrl = CanTpApi::PCANTP_MAPPING_FLOW_CTRL_NONE;
		mapping_func.netaddrinfo.target_type = cantp_isotp_addressing::PCANTP_ISOTP_ADDRESSING_FUNCTIONAL;
		mapping_func.netaddrinfo.target_addr = target_addr_func;
		status = CanTpApi::AddMapping_2016(channel, mapping_func);
		if (!CanTpApi::StatusIsOk_2016(status))
		{
			Console::WriteLine("Failed to configure mapping (can_id=0x{0:X}) (sts=0x{1:X})", mapping_func.can_id, (UInt32)status);
			result = false;
		}
	}
	return result;
}

/// <summary>Initializes mappings compatible with PCTPClient/PCTPServer sample</summary>
/// <param name="channel">CAN channel to add the mappings to.</param>
/// <return>true on successful configuration</return>
static bool initialize_mappings(cantp_handle channel)
{
	bool result = true;

	// 11 bit can ID, normal format addressing, diagnostic message (mandatory)
	result &= initialize_mappings(channel, cantp_can_msgtype::PCANTP_CAN_MSGTYPE_STANDARD, N_SA, N_TA_PHYS, 0x00, N_TA_FUNC, 0xA1, 0xA2, 0xA5, cantp_isotp_msgtype::PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, cantp_isotp_format::PCANTP_ISOTP_FORMAT_NORMAL);
	// 11 bit can ID, extended format addressing, diagnostic message (mandatory)
	result &= initialize_mappings(channel, cantp_can_msgtype::PCANTP_CAN_MSGTYPE_STANDARD, N_SA, N_TA_PHYS, 0x00, N_TA_FUNC, 0xB1, 0xB2, 0xB5, cantp_isotp_msgtype::PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, cantp_isotp_format::PCANTP_ISOTP_FORMAT_EXTENDED);
	result &= initialize_mappings(channel, cantp_can_msgtype::PCANTP_CAN_MSGTYPE_STANDARD, N_SA, N_TA_PHYS, 0x00, N_TA_FUNC, 0xC1, 0xC2, 0xC5, cantp_isotp_msgtype::PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, cantp_isotp_format::PCANTP_ISOTP_FORMAT_EXTENDED);

	// 11 bit can ID, mixed format addressing, remote diagnostic message (mandatory)
	result &= initialize_mappings(channel, cantp_can_msgtype::PCANTP_CAN_MSGTYPE_STANDARD, N_SA, N_TA_PHYS, N_RA, N_TA_FUNC, 0xD1, 0xD2, 0xD5, cantp_isotp_msgtype::PCANTP_ISOTP_MSGTYPE_REMOTE_DIAGNOSTIC, cantp_isotp_format::PCANTP_ISOTP_FORMAT_MIXED);

	// 29 bit can ID, normal format addressing, diagnostic message (mandatory)
	result &= initialize_mappings(channel, cantp_can_msgtype::PCANTP_CAN_MSGTYPE_EXTENDED, N_SA, N_TA_PHYS, 0x00, N_TA_FUNC, 0xA123A1, 0xA123A2, 0xA123A5, cantp_isotp_msgtype::PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, cantp_isotp_format::PCANTP_ISOTP_FORMAT_NORMAL);
	// 29 bit can ID, extended format addressing, diagnostic message
	result &= initialize_mappings(channel, cantp_can_msgtype::PCANTP_CAN_MSGTYPE_EXTENDED, N_SA, N_TA_PHYS, 0x00, N_TA_FUNC, 0xA123C1, 0xA123C2, 0xA123C5, cantp_isotp_msgtype::PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, cantp_isotp_format::PCANTP_ISOTP_FORMAT_EXTENDED);
	result &= initialize_mappings(channel, cantp_can_msgtype::PCANTP_CAN_MSGTYPE_EXTENDED, N_SA, N_TA_PHYS, 0x00, N_TA_FUNC, 0xA123D1, 0xA123D2, 0xA123D5, cantp_isotp_msgtype::PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, cantp_isotp_format::PCANTP_ISOTP_FORMAT_EXTENDED);

	return result;
}

/// <summary>Reads a message.</summary>
/// <param name="channel">The channel to read the message from.</param>
/// <param name="nbMsgRead">Buffer to store the number of message actually read.</param>
/// <returns>number of errors</returns>
static int read_segmented_message(cantp_handle channel, int %nbMsgRead)
{
	cantp_msg message;
	cantp_status status;
	cantp_timestamp timestamp;
	int doLoop;
	int nbErr = 0;

	do
	{
		doLoop = 0;
		// initialize message
		status = CanTpApi::MsgDataAlloc_2016(message, cantp_msgtype::PCANTP_MSGTYPE_NONE);
		if (!CanTpApi::StatusIsOk_2016(status))
		{
			Console::WriteLine("Failed to allocate message (sts=0x{0X}).", (UInt32)status);
			++nbErr;
			continue;
		}
		// retrieve any message from Rx queue
		status = CanTpApi::Read_2016(channel, message, timestamp);
		// display_message returns true when a message is available
		doLoop = display_message(message, timestamp, status);
		if (doLoop > 0)
		{
			// check message                  
			if (message.msgdata.any == nullptr)
				nbErr++;
			else if (message.type == cantp_msgtype::PCANTP_MSGTYPE_ISOTP && message.msgdata.isotp == nullptr)
				nbErr++;
			else
			{
				// update error counter if a message was received but a network error occured
				if (message.msgdata.any->netstatus != cantp_netstatus::PCANTP_NETSTATUS_OK)
				{
					nbErr++;
				}
				if (message.type == cantp_msgtype::PCANTP_MSGTYPE_ISOTP &&
					(message.msgdata.isotp->netaddrinfo.msgtype & cantp_isotp_msgtype::PCANTP_ISOTP_MSGTYPE_FLAG_INDICATION) != 0)
				{
					// wait and read the full segmented message
					nbErr += read_segmented_message(channel, nbMsgRead);
				}
				else
					nbMsgRead++;
			}
		}
		else
		{
			if (doLoop == 0)
			{
				// no message read, wait for it
				System::Threading::Thread::Sleep(10);
			}
		}
		// release message
		status = CanTpApi::MsgDataFree_2016(message);
		if (!CanTpApi::StatusIsOk_2016(status))
		{
			Console::WriteLine("Failed to deallocate message (sts=0x{0:X}).", (UInt32)status);
		}
	} while (doLoop == 0);

	return nbErr;
}

/// <summary>A function to transmit CAN-TP messages</summary>
/// <param name="channel">channel</param>
/// <param name="canIdType">canIdType</param>
/// <param name="msgType">CAN-TP message Type (PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC)</param>
/// <param name="formatType">Format addressing type (see PCANTP_ISOTP_FORMAT_xxx)</param>
/// <returns>number of transmission errors</returns>
static int transmit(cantp_handle channel,
	cantp_can_msgtype canIdType,
	cantp_isotp_msgtype msgType,
	cantp_isotp_format formatType,
	bool doCanFd, bool canFdInitialized)
{
	UInt32 index;
	cantp_status status;
	cantp_msg message;
	cantp_can_msgtype can_msgtype = cantp_can_msgtype::PCANTP_CAN_MSGTYPE_STANDARD;
	cantp_netaddrinfo isotp_nai;
	Byte lMaxSize = 0;
	int nbErrPhys = 0;
	int nbErrFunc = 0;
	int sleepTime = 0;
	int nbMsgRead = 0;
	array<Byte>^ data = gcnew array<Byte>(MESSAGE_MAX_LENGTH);

	// Output information message
	String^ lStr = "??";
	switch (formatType)
	{
	case cantp_isotp_format::PCANTP_ISOTP_FORMAT_ENHANCED:
		lStr = "Enhanced";
		break;
	case cantp_isotp_format::PCANTP_ISOTP_FORMAT_EXTENDED:
		lStr = "Extended";
		break;
	case cantp_isotp_format::PCANTP_ISOTP_FORMAT_FIXED_NORMAL:
		lStr = "Fixed Normal";
		break;
	case cantp_isotp_format::PCANTP_ISOTP_FORMAT_MIXED:
		lStr = "Mixed";
		break;
	case cantp_isotp_format::PCANTP_ISOTP_FORMAT_NORMAL:
		lStr = "Normal";
		break;
	}
	Console::WriteLine("\n\nTransmission of {0}, {1} message, {2} addressing format",
		((canIdType & cantp_can_msgtype::PCANTP_CAN_MSGTYPE_EXTENDED) != 0) ? "29 BIT CAN ID" : "11 BIT CAN ID",
		msgType == cantp_isotp_msgtype::PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC ? "Diagnostic" : "Remote diagnostic",
		lStr);
	if (USE_GETCH)
	{
		Console::WriteLine("\nPress <Enter> to continue...");
		Console::Read();
	}

	// initialize message
	isotp_nai.source_addr = N_SA;
	if (msgType == cantp_isotp_msgtype::PCANTP_ISOTP_MSGTYPE_REMOTE_DIAGNOSTIC)
		isotp_nai.extension_addr = N_RA;
	can_msgtype = canIdType;
	if (doCanFd)
	{
		can_msgtype |= cantp_can_msgtype::PCANTP_CAN_MSGTYPE_FD;
		can_msgtype |= cantp_can_msgtype::PCANTP_CAN_MSGTYPE_BRS;
	}
	isotp_nai.msgtype = msgType;
	isotp_nai.format = formatType;
	// prepare a buffer that will be used as a reference to initialize the data to transmit
	for (int i = 0; i < MESSAGE_MAX_LENGTH; ++i)
		data[i] = (Byte)i;

	if (TEST_PHYS_TRANSMISSION)
	{
		// Transmitting data using Physical addressing
		Console::WriteLine("\nTransmitting data using Physical addressing\n");
		isotp_nai.target_addr = N_TA_PHYS;
		isotp_nai.target_type = cantp_isotp_addressing::PCANTP_ISOTP_ADDRESSING_PHYSICAL;
		// each loop transmits a message with a data length incremented by 1 and reads its loppbacl confirmation
		for (index = 1; index <= MESSAGE_MAX_LENGTH; index++)
		{
			// allocate a new ISOTP message
			status = CanTpApi::MsgDataAlloc_2016(message, cantp_msgtype::PCANTP_MSGTYPE_ISOTP);
			if (CanTpApi::StatusIsOk_2016(status))
			{
				// initialize ISOTP message
				status = CanTpApi::MsgDataInit_2016(message, CanTpApi::PCANTP_CAN_ID_DEFINED_BY_NAI, can_msgtype, index, data, isotp_nai);
				if (CanTpApi::StatusIsOk_2016(status))
				{
					// write message
					status = CanTpApi::Write_2016(channel, message);
					if (CanTpApi::StatusIsOk_2016(status))
					{
						Console::WriteLine("\nSuccessfully queued ISOTP message: Length {0} (sts=0x{1:X}).", index, (UInt32)status);
						// try to read loopback confirmation 
						nbMsgRead = 0;
						nbErrPhys += read_segmented_message(channel, nbMsgRead);
						if (nbMsgRead != 1)
						{
							Console::WriteLine("Received unexpected messages ({0} instead of 1)", nbMsgRead);
						}
					}
					else
					{
						Console::WriteLine("\nFailed to write ISOTP message: Length {0} (sts=0x{1:X}).", index, (UInt32)status);
						nbErrPhys++;
					}
				}
				else
				{
					Console::WriteLine("\nFailed to initialize ISOTP message: Length {0} (sts=0x{1:X}).", index, (UInt32)status);
					nbErrPhys++;
				}
				// release message
				status = CanTpApi::MsgDataFree_2016(message);
				if (!CanTpApi::StatusIsOk_2016(status))
				{
					Console::WriteLine("Failed to deallocate message (sts=0x{0:X}).", (UInt32)status);
					nbErrPhys++;
				}
			}
			else
			{
				Console::WriteLine("Failed to allocate message (sts=0x{0:X}).", (UInt32)status);
				nbErrPhys++;
			}

			System::Threading::Thread::Sleep(sleepTime);

			if (index == 100)
			{
				// skip and jump to maximum data length
				index = MESSAGE_MAX_LENGTH - 1;
			}
		}
	}

	if (TEST_BURST_TRANSMISSION)
	{
		Console::WriteLine("\nTransmitting data using Physical addressing in 'BURST' mode\n");

		isotp_nai.target_addr = N_TA_PHYS;
		isotp_nai.target_type = cantp_isotp_addressing::PCANTP_ISOTP_ADDRESSING_PHYSICAL;
		// Transmitting data using Physical addressing WITHOUT waiting for response confirmation
		// each loop transmits a message with a data length incremented by 1
		int nbMsgSent = 0;
		for (index = 1; index <= MESSAGE_MAX_LENGTH; index++)
		{
			// allocate a new ISOTP message
			status = CanTpApi::MsgDataAlloc_2016(message, cantp_msgtype::PCANTP_MSGTYPE_ISOTP);
			if (CanTpApi::StatusIsOk_2016(status))
			{
				// initialize ISOTP message
				status = CanTpApi::MsgDataInit_2016(message, CanTpApi::PCANTP_CAN_ID_DEFINED_BY_NAI, can_msgtype, index, data, isotp_nai);
				if (CanTpApi::StatusIsOk_2016(status))
				{
					// write message
					status = CanTpApi::Write_2016(channel, message);
					if (CanTpApi::StatusIsOk_2016(status))
					{
						nbMsgSent++;
						Console::WriteLine("\nSuccessfully queued ISOTP message: Length {0} (sts=0x{1:X}).", index, (UInt32)status);
					}
					else
					{
						Console::WriteLine("\nFailed to write ISOTP message: Length {0} (sts=0x{1:X}).", index, (UInt32)status);
						nbErrPhys++;
					}
				}
				else
				{
					Console::WriteLine("\nFailed to initialize ISOTP message: Length {0} (sts=0x{1:X}).", index, (UInt32)status);
					nbErrPhys++;
				}
				// release message
				status = CanTpApi::MsgDataFree_2016(message);
				if (!CanTpApi::StatusIsOk_2016(status))
				{
					Console::WriteLine("Failed to deallocate message (sts=0x{0:X}).", (UInt32)status);
					nbErrPhys++;
				}
			}
			else
			{
				Console::WriteLine("Failed to allocate message (sts=0x{0:X}).", (UInt32)status);
				nbErrPhys++;
			}


			if (index == 10)
			{  // skip and try bigger LEN
				index = 2000;
			}
			if (index == 2005)
			{ // skip
				break;
			}
		}
		Console::WriteLine("\nReading confirmation for each transmitted messages in 'BURST' mode\n");


		for (nbMsgRead = 0; nbMsgRead < nbMsgSent;)
		{
			// try to read loopback confirmation 
			nbErrPhys += read_segmented_message(channel, nbMsgRead);
		}
		CanTpApi::Reset_2016(channel);
	}

	if (TEST_FUNC_TRANSMISSION)
	{
		// Transmitting data using Functional addressing 
		Console::WriteLine("\nTransmitting data using Functional addressing\n");

		isotp_nai.target_addr = N_TA_FUNC;
		isotp_nai.target_type = cantp_isotp_addressing::PCANTP_ISOTP_ADDRESSING_FUNCTIONAL;
		// Reminder: Functional addressing shall only be supported 
		// for Single Frame communication.
		// Thus maximum size depends on Format Addressing Type.
		switch (formatType)
		{
		case cantp_isotp_format::PCANTP_ISOTP_FORMAT_EXTENDED:
		case cantp_isotp_format::PCANTP_ISOTP_FORMAT_MIXED:
			lMaxSize = (Byte)(((doCanFd && canFdInitialized) ? CAN_TX_DL_VALUE : CanTpApi::PCANTP_MAX_LENGTH_CAN_STANDARD) - 2);
			break;
		default:
			lMaxSize = (Byte)(((doCanFd && canFdInitialized) ? CAN_TX_DL_VALUE : CanTpApi::PCANTP_MAX_LENGTH_CAN_STANDARD) - 1);
			break;
		}
		// each loop transmits a message with a data length incremented by 1 and reads its loppbacl confirmation
		for (index = 1; index <= lMaxSize; index++)
		{
			// allocate a new ISOTP message
			status = CanTpApi::MsgDataAlloc_2016(message, cantp_msgtype::PCANTP_MSGTYPE_ISOTP);
			if (CanTpApi::StatusIsOk_2016(status))
			{
				// initialize ISOTP message
				status = CanTpApi::MsgDataInit_2016(message, CanTpApi::PCANTP_CAN_ID_DEFINED_BY_NAI, can_msgtype, index, data, isotp_nai);
				if (CanTpApi::StatusIsOk_2016(status))
				{
					// write message
					status = CanTpApi::Write_2016(channel, message);
					if (CanTpApi::StatusIsOk_2016(status))
					{
						Console::WriteLine("\nSuccessfully queued ISOTP message: Length {0} (sts=0x{1:X}).", index, (UInt32)status);
						// try to read loopback confirmation 
						nbMsgRead = 0;
						nbErrFunc += read_segmented_message(channel, nbMsgRead);
						if (nbMsgRead != 1)
						{
							Console::WriteLine("Received unexpected messages ({0} instead of 1)", nbMsgRead);
						}
					}
					else
					{
						Console::WriteLine("\nFailed to write ISOTP message: Length {0} (sts=0x{1:X}).", index, (UInt32)status);
						nbErrFunc++;
					}
				}
				else
				{
					Console::WriteLine("\nFailed to initialize ISOTP message: Length {0} (sts=0x{1:X}).", index, (UInt32)status);
					nbErrFunc++;
				}
				// release message
				status = CanTpApi::MsgDataFree_2016(message);
				if (!CanTpApi::StatusIsOk_2016(status))
				{
					Console::WriteLine("Failed to deallocate message (sts=0x{0:X}).", (UInt32)status);
					nbErrFunc++;
				}

			}
			else
			{
				Console::WriteLine("Failed to allocate message (sts=0x{0:X}).", (UInt32)status);
				nbErrFunc++;
			}

			System::Threading::Thread::Sleep(sleepTime);
		}
	}

	if (nbErrPhys > 0 || nbErrFunc > 0)
	{
		if (nbErrPhys > 0)
		{
			Console::WriteLine("\nERROR : {0} errors occured.\n", nbErrPhys);
		}
		if (nbErrFunc > 0)
		{
			Console::WriteLine("\nERROR : {0} errors occured.\n", nbErrFunc);
		}
		if (USE_GETCH)
		{
			Console::WriteLine("\nPress <Enter> to continue...");
			Console::ReadKey();
		}
	}
	else
	{
		Console::WriteLine("\nTransmissions successful !\n");
	}

	return nbErrFunc + nbErrPhys;
}

/// <summary>Main entry-point for this application.</summary>
/// <param name="args">The args.</param>
int main(array<System::String ^> ^args)
{
	StringBuilder ^buffer = gcnew StringBuilder(500);
	cantp_handle channel;
	bool useCanFd = false;
	cantp_baudrate baudrate;
	cantp_bitrate bitrateFd = "";

	Console::WriteLine("\nUsage: PCTPClient.exe [TPCANTPHandle] [USE_CAN_FD:0|1] [BTR0BTR1|BITRATE]");
	Console::WriteLine(" * CAN: PCTPClient.exe 0x51 0 0x1C ");
	Console::WriteLine(" * CAN FD: PCTPClient.exe 0x51 1 \"f_clock=80000000, nom_brp=10, nom_tseg1=12, nom_tseg2=3, nom_sjw=1, data_brp=4, data_tseg1=7, data_tseg2=2, data_sjw=1\"");
	Console::WriteLine("\n---------------------------------------------");

	// clear gracefully ISO-TP if console window is killed
	AppDomain::CurrentDomain->UnhandledException += gcnew UnhandledExceptionEventHandler(MyHandler);
	SetConsoleCtrlHandler(gcnew HandlerRoutine(ConsoleCtrlCheck), true);

	// Show version information
	cantp_status status = CanTpApi::GetValue_2016(cantp_handle::PCANTP_HANDLE_NONEBUS, cantp_parameter::PCANTP_PARAMETER_API_VERSION, buffer, 500);
	if (CanTpApi::StatusIsOk_2016(status))
		Console::WriteLine("PCAN-ISO-TP API Version : {0}", buffer);
	else
		Console::WriteLine("Error failed to get PCAN-ISO-TP API Version (sts=0x{0:X})", (UInt32)status);

	// 1st argument is the PCAN-channel to use (PCAN-USB channel 1)
	if (args->Length >= 1)
		channel = convertToHandle(args[0]);
	else
		channel = cantp_handle::PCANTP_HANDLE_USBBUS1;
	// 2nd argument is CAN FD
	if (args->Length >= 2)
		useCanFd = convertToBool(args[1]);
	else
		useCanFd = USE_CAN_FD;
	// 3rd argument is bitrate (either CAN or CAN FD)
	baudrate = cantp_baudrate::PCANTP_BAUDRATE_500K;
	bitrateFd = "f_clock=80000000, nom_brp=10, nom_tseg1=12, nom_tseg2=3, nom_sjw=1, data_brp=4, data_tseg1=7, data_tseg2=2, data_sjw=1";
	if (args->Length >= 3)
	{
		if (useCanFd)
			bitrateFd = args[2];
		else
			baudrate = convertToBaudRate(args[2]);
	}

	// Initializing of the ISO-TP Communication session
	Console::WriteLine("Connecting to channel 0x{0:X}...", (UInt32)channel);
	if (!useCanFd)
	{
		Console::WriteLine(" * btr0btr1 = 0x{0:X}...", (UInt32)baudrate);
		status = CanTpApi::Initialize_2016(channel, baudrate);
	}
	else
	{
		Console::WriteLine(" * bitrateFd = \"{0}\"...", bitrateFd);
		status = CanTpApi::InitializeFD_2016(channel, bitrateFd);
	}
	Console::WriteLine(" -> Initialize CANTP: 0x{0:X}", (UInt32)status);
	// will start the reading loop only if successfully initialized
	if (!CanTpApi::StatusIsOk_2016(status))
	{
		Console::WriteLine(" -> Initialization failed, exiting...");
		CanTpApi::Uninitialize_2016(channel);
		Console::WriteLine("\n\nPress <Enter> to quit...");
		Console::Read();
		return 0;
	}

	// configure the test session to use can data padding with a specific padding value
	array<Byte>^ param = gcnew array<Byte>(1);
	param[0] = CanTpApi::PCANTP_CAN_DATA_PADDING_ON;
	status = CanTpApi::SetValue_2016(channel, cantp_parameter::PCANTP_PARAMETER_CAN_DATA_PADDING, param, 1);
	if (!CanTpApi::StatusIsOk_2016(status))
	{
		Console::WriteLine("Failed to configure parameter 'PCANTP_PARAMETER_CAN_DATA_PADDING' (sts=0x{0:X}).", (UInt32)status);
	}
	param[0] = 0x99;
	status = CanTpApi::SetValue_2016(channel, cantp_parameter::PCANTP_PARAMETER_CAN_PADDING_VALUE, param, 1);
	if (!CanTpApi::StatusIsOk_2016(status))
	{
		Console::WriteLine("Failed to configure parameter 'PCANTP_PARAMETER_CAN_PADDING_VALUE' (sts=0x{0:X}).", (UInt32)status);
	}
	// set the default TX_DL size (i.e. the default CAN-FD DLC)
	param[0] = CAN_TX_DL_CODE;
	status = CanTpApi::SetValue_2016(channel, cantp_parameter::PCANTP_PARAMETER_CAN_TX_DL, param, 1);
	if (!CanTpApi::StatusIsOk_2016(status))
	{
		Console::WriteLine("Failed to configure parameter 'PCANTP_PARAMETER_CAN_TX_DL' (sts=0x{0:X}).", (UInt32)status);
	}
	// configure server - enable enhanced addressing to be compatible with PCTPClient
	param[0] = CanTpApi::PCANTP_VALUE_PARAMETER_ON;
	status = CanTpApi::SetValue_2016(channel, cantp_parameter::PCANTP_PARAMETER_SUPPORT_29B_ENHANCED, param, 1);
	if (!CanTpApi::StatusIsOk_2016(status))
	{
		Console::WriteLine("Failed to configure parameter 'PCANTP_PARAMETER_SUPPORT_29B_ENHANCED' (sts=0x{0:X}).", (UInt32)status);
	}
	// log CAN frames
	param[0] = CanTpApi::PCANTP_DEBUG_WARNING;
	status = CanTpApi::SetValue_2016(channel, cantp_parameter::PCANTP_PARAMETER_DEBUG, param, 1);
	if (!CanTpApi::StatusIsOk_2016(status))
	{
		Console::WriteLine("Failed to configure parameter 'PCANTP_PARAMETER_DEBUG' (sts=0x{0:X}).", (UInt32)status);
	}
	// configure ISO-TP mappings
	if (!initialize_mappings(channel))
	{
		Console::WriteLine(" -> Mappings configuration failed, exiting...");
		CanTpApi::Uninitialize_2016(channel);
		Console::WriteLine("\n\nPress <Enter> to quit...");
		Console::Read();
		return 0;
	}

	bool doCanFd = false;
	int nbErr = 0;
	for (int i = 0; i < 2; i++)
	{
		switch (i)
		{
		case 0:
			// first loop will test CAN-FD frames even if it was not initialized with Fd
			doCanFd = true;
			break;
		case 1:
			// second loop tests standard CAN frames
			doCanFd = false;
			break;
		}

		// Transmit with 29 Bit CAN identifiers (and no CAN ID mapping is required)
		nbErr += transmit(channel, cantp_can_msgtype::PCANTP_CAN_MSGTYPE_EXTENDED, cantp_isotp_msgtype::PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, cantp_isotp_format::PCANTP_ISOTP_FORMAT_FIXED_NORMAL, doCanFd, useCanFd);
		nbErr += transmit(channel, cantp_can_msgtype::PCANTP_CAN_MSGTYPE_EXTENDED, cantp_isotp_msgtype::PCANTP_ISOTP_MSGTYPE_REMOTE_DIAGNOSTIC, cantp_isotp_format::PCANTP_ISOTP_FORMAT_MIXED, doCanFd, useCanFd);

		// Transmit with enhanced diagnostics 29 bit CAN identifiers (ISO-15765-3 8.3)
		nbErr += transmit(channel, cantp_can_msgtype::PCANTP_CAN_MSGTYPE_EXTENDED, cantp_isotp_msgtype::PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, cantp_isotp_format::PCANTP_ISOTP_FORMAT_ENHANCED, doCanFd, useCanFd);

		// Transmit with 11 Bit CAN identifiers
		// those transmissions require a Mapping between CAN ID and Network Addressing Information
		// for each transmission, 3 mappings are added :
		//	- 2 for physical addressing, one defining SA->TA and the other TA->SA
		//	- 1 for functional addressing defining SA->TA
		//

		// 11 bit can ID, normal format addressing (diagnostic message mandatory)		
		nbErr += transmit(channel, cantp_can_msgtype::PCANTP_CAN_MSGTYPE_STANDARD, cantp_isotp_msgtype::PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, cantp_isotp_format::PCANTP_ISOTP_FORMAT_NORMAL, doCanFd, useCanFd);
		// 11 bit can ID, extended format addressing (diagnostic message mandatory)		
		nbErr += transmit(channel, cantp_can_msgtype::PCANTP_CAN_MSGTYPE_STANDARD, cantp_isotp_msgtype::PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, cantp_isotp_format::PCANTP_ISOTP_FORMAT_EXTENDED, doCanFd, useCanFd);
		// 11 bit can ID, mixed format addressing (remote diagnostic message mandatory)
		nbErr += transmit(channel, cantp_can_msgtype::PCANTP_CAN_MSGTYPE_STANDARD, cantp_isotp_msgtype::PCANTP_ISOTP_MSGTYPE_REMOTE_DIAGNOSTIC, cantp_isotp_format::PCANTP_ISOTP_FORMAT_MIXED, doCanFd, useCanFd);

		// Transmit with 29 Bit CAN identifiers
		// those transmissions require a Mapping between CAN ID and Network Addressing Information
		// for each transmission, 3 mappings are added 2 for physical addressing and 
		// the other for functional addressing.
		//

		// 29 bit can ID, normal format addressing (diagnostic message mandatory)
		nbErr += transmit(channel, cantp_can_msgtype::PCANTP_CAN_MSGTYPE_EXTENDED, cantp_isotp_msgtype::PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, cantp_isotp_format::PCANTP_ISOTP_FORMAT_NORMAL, doCanFd, useCanFd);
		// 29 bit can ID, extended format addressing (diagnostic message mandatory)
		nbErr += transmit(channel, cantp_can_msgtype::PCANTP_CAN_MSGTYPE_EXTENDED, cantp_isotp_msgtype::PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, cantp_isotp_format::PCANTP_ISOTP_FORMAT_EXTENDED, doCanFd, useCanFd);
	}

	// release CAN-TP
	if (nbErr > 0)
	{
		Console::WriteLine("\nERROR : a total of {0} errors occured.\n", nbErr);
		Console::WriteLine("\n\nPress <Enter> to quit...");
		Console::Read();
	}
	else
	{
		Console::WriteLine("\nALL Transmissions succeeded !\n");
	}

	CanTpApi::Uninitialize_2016(channel);
}


