program _01_server_ISO15765_2_normal_addressing;

{$APPTYPE CONSOLE}

uses
  SysUtils,
  ShellApi,
  windows,
  PCANBasic in 'PCANBasic.pas',
  PCANTP_2016 in 'PCANTP_2016.pas';

Function OK_KO(test: Boolean): String;
begin
  If test Then
    result := 'OK'
  else
    result := 'KO';
End;

Function STATUS_OK_KO(test: cantp_status): String;
begin
  result := OK_KO(TCantpApi.StatusIsOk_2016(test));
End;

Function KeyPressed: Boolean;
Const
  buffer_size = 20;
Var
  zone: Array [1 .. buffer_size] Of TInputRecord;
  is_waiting: Cardinal;
  i: Cardinal;
  han: THandle;
Begin
  han := GetStdHandle(STD_INPUT_HANDLE);
  PeekConsoleInput(han, zone[1], buffer_size, is_waiting);
  result := false;
  i := 1;
  While Not result And (i <= is_waiting) And (i <= buffer_size) Do
  Begin
    result := (zone[i].EventType = 1) And (zone[i].Event.KeyEvent.bKeyDown) And (zone[i].Event.KeyEvent.AsciiChar <> #0);
    inc(i);
  End;
  If (is_waiting <> 0) And Not result Then
    ReadConsoleInput(STD_INPUT_HANDLE, zone[1], is_waiting, i);
End;

Function ReadKey: AnsiChar;
Var
  zone: TInputRecord;
  nb_read: Cardinal;
  han: THandle;
Begin
  han := GetStdHandle(STD_INPUT_HANDLE);
  result := #0;
  Repeat
    ReadConsoleInput(han, zone, 1, nb_read);
    If (nb_read = 1) And (zone.EventType = 1) And (zone.Event.KeyEvent.bKeyDown) And (zone.Event.KeyEvent.AsciiChar <> #0) Then
      result := zone.Event.KeyEvent.AsciiChar;
  Until result <> #0;
End;

/// <summary>Helper: print message data</summary>
/// <param name="msg">message to print</param>
procedure print_data(const msg: Pcantp_msg);
var
  len: UInt32;
  i: UInt32;
begin
  if msg.msgdata_any <> nil then
  begin
    len := msg.msgdata_any.length;
    Write(#9, 'Byte:');
    for i := 0 to len - 1 do
      Write(Format(' 0x%x', [PByte(msg.msgdata_any^.data)[i]]));
    Writeln;
    Write(#9, 'Ascii: "');
    for i := 0 to len - 1 do
      Write(PAnsiChar(msg.msgdata_any^.data)[i]);
    Writeln('"');
  end;
end;

/// <summary>Entry point of the program, start a small CAN ISO TP read/write example</summary>

const
  STMIN_600US = $F6;

var
  // Local variables
  Res: cantp_status;
  read_status: cantp_status;
  Buffer: array [0 .. 499] of AnsiChar;
  STmin: UInt32;
  rx_msg: cantp_msg;
  mapping: cantp_mapping;
  server_handle: cantp_handle;
  receive_event: THandle;
  null_handle: THandle;
  stop: Boolean;
  wait_result: UInt32;
  ts: cantp_timestamp;
  keyboard_res: AnsiChar;
  dummy: char;

begin
  try

    // Initialize variables
    server_handle := cantp_handle.PCANTP_HANDLE_USBBUS2; // TODO: modify the value according to your available PCAN devices.
    FillChar(rx_msg, sizeof(rx_msg), 0);
    FillChar(mapping, sizeof(mapping), 0);

    // Print version informations
    TCantpApi.GetValue_2016(cantp_handle.PCANTP_HANDLE_NONEBUS, cantp_parameter.PCANTP_PARAMETER_API_VERSION, Buffer, 500);
    // Writeln(Format(' PCAN - ISO - TP API Version: %s ', [Buffer]));
    Writeln('PCAN - ISO - TP API Version : ', Buffer);

    // Initialize channel: CAN2.0 - 500Kbit/s
    Res := TCantpApi.Initialize_2016(server_handle, cantp_baudrate.PCANTP_BAUDRATE_500K);
    Writeln('Initialize : ', STATUS_OK_KO(Res));

    // Create and set a receive event on receiver
    receive_event := CreateEvent(nil, false, false, '');
    Res := TCantpApi.SetValue_2016(server_handle, PCANTP_PARAMETER_RECEIVE_EVENT, PByte(@receive_event), sizeof(receive_event));
    Writeln('Set receive event : ', STATUS_OK_KO(Res));

    // Change STmin value to 600us
    STmin := STMIN_600US;
    Res := TCantpApi.SetValue_2016(server_handle, cantp_parameter.PCANTP_PARAMETER_SEPARATION_TIME, PByte(@STmin), sizeof(STmin));
    Writeln('Set STMIN = 600us : ', STATUS_OK_KO(Res));

    // Create a simple physical mapping:
    // - Source 0xF1 (client), target 0x01 (server), CAN id 0xA1, CAN ID flow control 0xA2
    // - Diagnostic message in a classic format
    mapping.can_id := $A1;
    mapping.can_id_flow_ctrl := $A2;
    mapping.can_msgtype := cantp_can_msgtype.PCANTP_CAN_MSGTYPE_STANDARD;
    mapping.netaddrinfo.extension_addr := $00;
    mapping.netaddrinfo.Format := cantp_isotp_format.PCANTP_ISOTP_FORMAT_NORMAL;
    mapping.netaddrinfo.msgtype := cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC;
    mapping.netaddrinfo.source_addr := $F1;
    mapping.netaddrinfo.target_addr := $01;
    mapping.netaddrinfo.target_type := cantp_isotp_addressing.PCANTP_ISOTP_ADDRESSING_PHYSICAL;

    // Add mapping on channel
    Res := TCantpApi.AddMapping_2016(server_handle, @mapping);
    Writeln('Add a simple mapping : ', STATUS_OK_KO(Res));
    // NOTE: this is a one-way mapping, the server is only capable of
    // receiving segmented or unsegmented ISO-TP message using "0xA1" CAN ID
    // to be able to send ISO-TP message, another mapping is required.

    // Read while user do not press Q
    Writeln('Start listening, press Q to quit.');
    stop := false;
    while (not stop) do
    begin

      // Wait a receive event on receiver
      // note: timeout is used to check keyboard hit.
      wait_result := WaitForSingleObject(receive_event, 1000);

      // If we receive something, read the message
      if wait_result = WAIT_OBJECT_0 then
      begin
        repeat

          // Allocate rx message
          Res := TCantpApi.MsgDataAlloc_2016(rx_msg, cantp_msgtype.PCANTP_MSGTYPE_NONE);
          Writeln('Allocate rx message : ', STATUS_OK_KO(Res));

          // Read first available message (no filtering based on message's type is set):
          read_status := TCantpApi.Read_2016(server_handle, rx_msg, @ts);
          Writeln('Try to read a message : ', STATUS_OK_KO(read_status));

          // If we read something, print the message
          if TCantpApi.StatusIsOk_2016(read_status) then
          begin
            Writeln(sLineBreak + 'Print received data :');
            print_data(@rx_msg);
            Writeln;
          end;

          // Free message
          Res := TCantpApi.MsgDataFree_2016(rx_msg);
          Writeln('Free rx message : ', STATUS_OK_KO(Res));

        until (TCantpApi.StatusIsOk_2016(read_status, cantp_status.PCANTP_STATUS_NO_MESSAGE));
      end;

      // Quit when user press Q
      if KeyPressed() then
      begin
        keyboard_res := ReadKey();
        if (keyboard_res = 'Q') or (keyboard_res = 'q') then
          stop := true;
      end;
    end;

    // Close receive event
    null_handle := 0;
    Res := TCantpApi.SetValue_2016(server_handle, PCANTP_PARAMETER_RECEIVE_EVENT, PByte(@null_handle), sizeof(null_handle));
    Writeln('Stop receive event  : ', STATUS_OK_KO(Res));
    CloseHandle(receive_event);

    // Uninitialize
    Res := TCantpApi.Uninitialize_2016(server_handle);
    Writeln('Uninitialize : ', STATUS_OK_KO(Res));

    // Exit
    Writeln('Press any key to exit...');
    Readln(dummy);
  except
    on E: Exception do
      Writeln(E.Classname, ': ', E.Message);
  end;

end.
