program _02_client_ISO15765_2_normal_addressing;

{$APPTYPE CONSOLE}

uses
  SysUtils,
  ShellApi,
  windows,
  PCANBasic in 'PCANBasic.pas',
  PCANTP_2016 in 'PCANTP_2016.pas';

Function OK_KO(test: Boolean): String;
begin
  If test Then
    result := 'OK'
  else
    result := 'KO';
End;

Function STATUS_OK_KO(test: cantp_status): String;
begin
  result := OK_KO(TCantpApi.StatusIsOk_2016(test));
End;

function getNetStatusAsHexaString(const rx_msg: Pcantp_msg): String;
begin
  if rx_msg.msgdata_any <> nil then
    result := Format('%x', [UInt32(rx_msg.msgdata_any^.netstatus)])
  else
    result := 'Unknown';
end;

/// <summary>Entry point of the program, start a small CAN ISO TP read/write example</summary>

const
  STMIN_600US = $F6;
  DATA_UNSEGMENTED = 'PEAK';
  DATA_SEGMENTED = 'PEAK-System - PCAN-ISO-TP Sample';
  // TODO: Change MSG definition to perform UNSEGMENTED or SEGMENTED ISO-TP message
  msg = DATA_SEGMENTED;

var
  // Local variables
  Res: cantp_status;
  Buffer: array [0 .. 499] of AnsiChar;
  STmin: UInt32;
  rx_msg: cantp_msg;
  tx_msg: cantp_msg;
  msg_size: UInt32;
  strMsg: AnsiString;
  mapping: cantp_mapping;
  mappingSegmented: cantp_mapping;
  client_handle: cantp_handle;
  dummy: char;

begin
  try

    // Initialize variables
    strMsg := msg;
    msg_size := msg.length;
    client_handle := cantp_handle.PCANTP_HANDLE_USBBUS1; // TODO: modify the value according to your available PCAN devices.
    FillChar(tx_msg, sizeof(tx_msg), 0);
    FillChar(mapping, sizeof(mapping), 0);

    // Print version informations
    TCantpApi.GetValue_2016(cantp_handle.PCANTP_HANDLE_NONEBUS, cantp_parameter.PCANTP_PARAMETER_API_VERSION, Buffer, 500);
    // Writeln(Format(' PCAN - ISO - TP API Version: %s ', [Buffer]));
    Writeln('PCAN - ISO - TP API Version : ', Buffer);

    // Initialize channel: CAN2.0 - 500Kbit/s
    Res := TCantpApi.Initialize_2016(client_handle, cantp_baudrate.PCANTP_BAUDRATE_500K);
    Writeln('Initialize : ', STATUS_OK_KO(Res));

    // Change STmin value to 600us
    STmin := STMIN_600US;
    Res := TCantpApi.SetValue_2016(client_handle, cantp_parameter.PCANTP_PARAMETER_SEPARATION_TIME, PByte(@STmin), sizeof(STmin));
    Writeln('Set STMIN = 600us : ', STATUS_OK_KO(Res));

    // Allocate tx message
    Res := TCantpApi.MsgDataAlloc_2016(tx_msg, cantp_msgtype.PCANTP_MSGTYPE_ISOTP);
    Writeln('Allocate tx message : ', STATUS_OK_KO(Res));

    // Create a simple physical mapping:
    // - Source 0xF1 (client), target 0x01 (server), CAN id 0xA1, CAN ID flow control 0xA2
    // - Diagnostic message in a classic format
    mapping.can_id := $A1;
    mapping.can_id_flow_ctrl := $A2;
    mapping.can_msgtype := cantp_can_msgtype.PCANTP_CAN_MSGTYPE_STANDARD;
    mapping.netaddrinfo.extension_addr := $00;
    mapping.netaddrinfo.Format := cantp_isotp_format.PCANTP_ISOTP_FORMAT_NORMAL;
    mapping.netaddrinfo.msgtype := cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC;
    mapping.netaddrinfo.source_addr := $F1;
    mapping.netaddrinfo.target_addr := $01;
    mapping.netaddrinfo.target_type := cantp_isotp_addressing.PCANTP_ISOTP_ADDRESSING_PHYSICAL;

    // Add mapping on channel
    Res := TCantpApi.AddMapping_2016(client_handle, @mapping);
    Writeln('Add a simple mapping : ', STATUS_OK_KO(Res));

    if msg_size > 7 then
    begin
      // data is too big to fit in an ISO-TP "Single Frame",
      // Communication will be segmented a mapping to receive communications from the receiver:
      // - Source 0x01 (server), target 0xF1 (client), CAN id 0xA2, CAN ID flow control 0xA1
      // - Diagnostic message in a classic format
      FillChar(mappingSegmented, sizeof(mappingSegmented), 0);
      mappingSegmented.can_id := $A2;
      mappingSegmented.can_id_flow_ctrl := $A1;
      mappingSegmented.can_msgtype := cantp_can_msgtype.PCANTP_CAN_MSGTYPE_STANDARD;
      mappingSegmented.netaddrinfo.extension_addr := $00;
      mappingSegmented.netaddrinfo.Format := cantp_isotp_format.PCANTP_ISOTP_FORMAT_NORMAL;
      mappingSegmented.netaddrinfo.msgtype := cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC;
      mappingSegmented.netaddrinfo.source_addr := $01;
      mappingSegmented.netaddrinfo.target_addr := $F1;
      mappingSegmented.netaddrinfo.target_type := cantp_isotp_addressing.PCANTP_ISOTP_ADDRESSING_PHYSICAL;

      // Add mapping on channels
      Res := TCantpApi.AddMapping_2016(client_handle, @mappingSegmented);
      Writeln('Add a simple mapping : ', STATUS_OK_KO(Res));
    end;

    // Initialize Tx message
    Res := TCantpApi.MsgDataInit_2016(tx_msg, TCanTpApi.PCANTP_CAN_ID_DEFINED_BY_NAI, mapping.can_msgtype, msg_size, PByte(strMsg), @mapping.netaddrinfo);
    Writeln('Initialize tx message : ', STATUS_OK_KO(Res));

    // Write message
    Res := TCantpApi.Write_2016(client_handle, tx_msg);
    Writeln('Write "', MSG,'" message : ', STATUS_OK_KO(Res));

    // Sleep to let the transmission runs
    Sleep(100);

    if msg_size > 7 then
    begin
      // Initialize Rx message
      Res := TCantpApi.MsgDataAlloc_2016(rx_msg, cantp_msgtype.PCANTP_MSGTYPE_NONE);
      Writeln('Initialize rx message : ', STATUS_OK_KO(Res));
      // Read loopback indication ISO-TP message
      Res := TCantpApi.Read_2016(client_handle, rx_msg);
      Writeln('CANTP_Read_2016 (indication) : ', STATUS_OK_KO(Res), ', netstatus=', getNetStatusAsHexaString(@rx_msg));
      // Free message space
      Res := TCantpApi.MsgDataFree_2016(rx_msg);
      Writeln('Free rx message : ', STATUS_OK_KO(Res));
      // Sleep to let the transmission complete
      Sleep(1000);
    end;

    // Initialize Rx message
    Res := TCantpApi.MsgDataAlloc_2016(rx_msg, cantp_msgtype.PCANTP_MSGTYPE_NONE);
    Writeln('Initialize rx message : ', STATUS_OK_KO(Res));
    // Read ISO-TP message
    Res := TCantpApi.Read_2016(client_handle, rx_msg);
    Writeln('CANTP_Read_2016 (complete) : ', STATUS_OK_KO(Res), ', netstatus=', getNetStatusAsHexaString(@rx_msg));
    // Free message space
    Res := TCantpApi.MsgDataFree_2016(rx_msg);
    Writeln('Free rx message : ', STATUS_OK_KO(Res));

    // Free messages space
    Res := TCantpApi.MsgDataFree_2016(tx_msg);
    Writeln('Free tx message : ', STATUS_OK_KO(Res));

    // Uninitialize
    Res := TCantpApi.Uninitialize_2016(client_handle);
    Writeln('Uninitialize : ', STATUS_OK_KO(Res));

    // Exit
    Writeln('Press any key to exit...');
    Readln(dummy);

  except
    on E: Exception do
      Writeln(E.Classname, ': ', E.Message);
  end;

end.
