program _05_isotp_read_write;
{$APPTYPE CONSOLE}

uses
  SysUtils,
  ShellApi,
  windows,
  PCANBasic in 'PCANBasic.pas',
  PCANTP_2016 in 'PCANTP_2016.pas';

Function OK_KO(test: Boolean): String;
begin
  If test Then
    result := 'OK'
  else
    result := 'KO';
End;

Function STATUS_OK_KO(test: cantp_status): String;
begin
  result := OK_KO(TCantpApi.StatusIsOk_2016(test));
End;

/// <summary>Entry point of the program, start a small CAN ISO TP read/write example</summary>

const
  STMIN_600US = $F6;

var
  // Local variables
  Res: cantp_status;
  Buffer: array [0 .. 499] of AnsiChar;
  STmin: UInt32;
  rx_msg: cantp_msg;
  tx_msg: cantp_msg;
  val: Byte;
  mapping: cantp_mapping;
  transmitter_handle: cantp_handle;
  receiver_handle: cantp_handle;
  receive_event: THandle;
  null_handle: THandle;
  wait_result: UInt32;
  dummy: char;

begin
  try

    // Initialize variables
    transmitter_handle := cantp_handle.PCANTP_HANDLE_USBBUS1; // TODO: modify the value according to your available PCAN devices.
    receiver_handle := cantp_handle.PCANTP_HANDLE_USBBUS2; // TODO: modify the value according to your available PCAN devices.
    FillChar(tx_msg, sizeof(tx_msg), 0);
    FillChar(rx_msg, sizeof(rx_msg), 0);
    FillChar(mapping, sizeof(mapping), 0);

    // Print version informations
    TCantpApi.GetValue_2016(cantp_handle.PCANTP_HANDLE_NONEBUS, cantp_parameter.PCANTP_PARAMETER_API_VERSION, Buffer, 500);
    // Writeln(Format(' PCAN - ISO - TP API Version: %s ', [Buffer]));
    Writeln('PCAN - ISO - TP API Version : ', Buffer);

    Writeln(sLineBreak + sLineBreak + 'WARNING: this sample is not suited for segmented communication.');
    Writeln('         Data must be contained in a Single ISO-TP frame only.');
    Writeln('         Please check sample "06_isotp_segmented_read_write" for standard segmented communications.');
    Writeln(sLineBreak);
    Sleep(2000);

    // Initialize channels : CAN2.0 - 500Kbit/s
    Res := TCantpApi.Initialize_2016(transmitter_handle, cantp_baudrate.PCANTP_BAUDRATE_500K);
    Writeln('Initialize transmitter : ', STATUS_OK_KO(Res));
    Res := TCantpApi.Initialize_2016(receiver_handle, cantp_baudrate.PCANTP_BAUDRATE_500K);
    Writeln('Initialize receiver : ', STATUS_OK_KO(Res));

    // Create and set a receive event on receiver
    receive_event := CreateEvent(nil, false, false, '');
    Res := TCantpApi.SetValue_2016(receiver_handle, PCANTP_PARAMETER_RECEIVE_EVENT, PByte(@receive_event), sizeof(receive_event));
    Writeln('Set receive event : ', STATUS_OK_KO(Res));

    // Change STmin value to 600us
    STmin := STMIN_600US;
    Res := TCantpApi.SetValue_2016(transmitter_handle, cantp_parameter.PCANTP_PARAMETER_SEPARATION_TIME, PByte(@STmin), sizeof(STmin));
    Writeln('Set STMIN = 600us on transmitter : ', STATUS_OK_KO(Res));
    Res := TCantpApi.SetValue_2016(receiver_handle, cantp_parameter.PCANTP_PARAMETER_SEPARATION_TIME, PByte(@STmin), sizeof(STmin));
    Writeln('Set STMIN = 600us on receiver : ', STATUS_OK_KO(Res));

    // Allocate tx message
    Res := TCantpApi.MsgDataAlloc_2016(tx_msg, cantp_msgtype.PCANTP_MSGTYPE_ISOTP);
    Writeln('Allocate tx message : ', STATUS_OK_KO(Res));

    // Allocate rx message
    Res := TCantpApi.MsgDataAlloc_2016(rx_msg, cantp_msgtype.PCANTP_MSGTYPE_NONE);
    Writeln('Allocate rx message : ', STATUS_OK_KO(Res));

    // Create a simple physical mapping:
    // - Source 0xF1 (transmitter), target 0x01 (receiver), CAN id 0xA1, CAN ID flow control 0xA2
    // - Diagnostic message in a classic format
    mapping.can_id := $A1;
    mapping.can_id_flow_ctrl := $A2;
    mapping.can_msgtype := cantp_can_msgtype.PCANTP_CAN_MSGTYPE_STANDARD;
    mapping.netaddrinfo.extension_addr := $00;
    mapping.netaddrinfo.Format := cantp_isotp_format.PCANTP_ISOTP_FORMAT_NORMAL;
    mapping.netaddrinfo.msgtype := cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC;
    mapping.netaddrinfo.source_addr := $F1;
    mapping.netaddrinfo.target_addr := $01;
    mapping.netaddrinfo.target_type := cantp_isotp_addressing.PCANTP_ISOTP_ADDRESSING_PHYSICAL;

    // Add mapping on channels
    Res := TCantpApi.AddMapping_2016(transmitter_handle, @mapping);
    Writeln('Add a simple mapping on transmitter : ', STATUS_OK_KO(Res));
    Res := TCantpApi.AddMapping_2016(receiver_handle, @mapping);
    Writeln('Add a simple mapping on receiver : ', STATUS_OK_KO(Res));

    // Initialize Tx message containing "42"
    val := $42;
    Res := TCantpApi.MsgDataInit_2016(tx_msg, TCanTpApi.PCANTP_CAN_ID_DEFINED_BY_NAI, mapping.can_msgtype, 1, PByte(@val), @mapping.netaddrinfo);
    Writeln('Initialize tx message : ', STATUS_OK_KO(Res));

    // Write "42" message
    Res := TCantpApi.Write_2016(transmitter_handle, tx_msg);
    Writeln('Write "42" message : ', STATUS_OK_KO(Res));

    // Wait a receive event on receiver
    wait_result := WaitForSingleObject(receive_event, 100);
    Writeln('Wait a message on receiver : ', OK_KO(wait_result = WAIT_OBJECT_0));

    // If we receive something on the receiver, read the message
    if wait_result = WAIT_OBJECT_0 then
    begin
      Res := TCantpApi.Read_2016(receiver_handle, rx_msg);
      Writeln('Read message on receiver : ', STATUS_OK_KO(Res));
      if TCantpApi.StatusIsOk_2016(Res) and (rx_msg.msgdata_any^.length <> 0) then
      begin
        if PByte(rx_msg.msgdata_any^.data)[0] = $42 then
          Writeln('Check if the message is "42" : OK')
        else
          Writeln('Check if the message is "42" : NOK')
      end;
    end;

    // Free messages
    Res := TCantpApi.MsgDataFree_2016(rx_msg);
    Writeln('Free rx message : ', STATUS_OK_KO(Res));
    Res := TCantpApi.MsgDataFree_2016(tx_msg);
    Writeln('Free tx message : ', STATUS_OK_KO(Res));

    // Close receive event
    null_handle := 0;
    Res := TCantpApi.SetValue_2016(receiver_handle, PCANTP_PARAMETER_RECEIVE_EVENT, PByte(@null_handle), sizeof(null_handle));
    Writeln('Stop receive event  : ', STATUS_OK_KO(Res));
    CloseHandle(receive_event);

    // Uninitialize
    Res := TCantpApi.Uninitialize_2016(transmitter_handle);
    Writeln('Uninitialize transmitter : ', STATUS_OK_KO(Res));
    Res := TCantpApi.Uninitialize_2016(receiver_handle);
    Writeln('Uninitialize receiver : ', STATUS_OK_KO(Res));

    // Exit
    Writeln('Press any key to exit...');
    Readln(dummy);
  except
    on E: Exception do
      Writeln(E.Classname, ': ', E.Message);
  end;

end.
