program _06_isotp_segmented_read_write;

{$APPTYPE CONSOLE}

uses
  SysUtils,
  ShellApi,
  windows,
  PCANBasic in 'PCANBasic.pas',
  PCANTP_2016 in 'PCANTP_2016.pas',
  heavy_data_u in 'heavy_data_u.pas';

Function OK_KO(test: Boolean): String;
begin
  If test Then
    result := 'OK'
  else
    result := 'KO';
End;

Function STATUS_OK_KO(test: cantp_status): String;
begin
  result := OK_KO(TCantpApi.StatusIsOk_2016(test));
End;

/// <summary>Return the name of the channel</summary>
/// <param name="value">Channel handle</param>
/// <returns>String containing the name of the channel</returns>
Function cantp_handle_toShortString(value: cantp_handle): String;
var
  res: String;
begin
  case value of
    cantp_handle.PCANTP_HANDLE_NONEBUS:
      res := 'NONEBUS';
    cantp_handle.PCANTP_HANDLE_ISABUS1:
      res := 'ISA1';
    cantp_handle.PCANTP_HANDLE_ISABUS2:
      res := 'ISA2';
    cantp_handle.PCANTP_HANDLE_ISABUS3:
      res := 'ISA3';
    cantp_handle.PCANTP_HANDLE_ISABUS4:
      res := 'ISA4';
    cantp_handle.PCANTP_HANDLE_ISABUS5:
      res := 'ISA5';
    cantp_handle.PCANTP_HANDLE_ISABUS6:
      res := 'ISA6';
    cantp_handle.PCANTP_HANDLE_ISABUS7:
      res := 'ISA7';
    cantp_handle.PCANTP_HANDLE_ISABUS8:
      res := 'ISA8';
    cantp_handle.PCANTP_HANDLE_DNGBUS1:
      res := 'DNG1';
    cantp_handle.PCANTP_HANDLE_PCIBUS1:
      res := 'PCI1';
    cantp_handle.PCANTP_HANDLE_PCIBUS2:
      res := 'PCI2';
    cantp_handle.PCANTP_HANDLE_PCIBUS3:
      res := 'PCI3';
    cantp_handle.PCANTP_HANDLE_PCIBUS4:
      res := 'PCI4';
    cantp_handle.PCANTP_HANDLE_PCIBUS5:
      res := 'PCI5';
    cantp_handle.PCANTP_HANDLE_PCIBUS6:
      res := 'PCI6';
    cantp_handle.PCANTP_HANDLE_PCIBUS7:
      res := 'PCI7';
    cantp_handle.PCANTP_HANDLE_PCIBUS8:
      res := 'PCI8';
    cantp_handle.PCANTP_HANDLE_PCIBUS9:
      res := 'PCI9';
    cantp_handle.PCANTP_HANDLE_PCIBUS10:
      res := 'PCI10';
    cantp_handle.PCANTP_HANDLE_PCIBUS11:
      res := 'PCI11';
    cantp_handle.PCANTP_HANDLE_PCIBUS12:
      res := 'PCI12';
    cantp_handle.PCANTP_HANDLE_PCIBUS13:
      res := 'PCI13';
    cantp_handle.PCANTP_HANDLE_PCIBUS14:
      res := 'PCI14';
    cantp_handle.PCANTP_HANDLE_PCIBUS15:
      res := 'PCI15';
    cantp_handle.PCANTP_HANDLE_PCIBUS16:
      res := 'PCI16';
    cantp_handle.PCANTP_HANDLE_USBBUS1:
      res := 'USB1';
    cantp_handle.PCANTP_HANDLE_USBBUS2:
      res := 'USB2';
    cantp_handle.PCANTP_HANDLE_USBBUS3:
      res := 'USB3';
    cantp_handle.PCANTP_HANDLE_USBBUS4:
      res := 'USB4';
    cantp_handle.PCANTP_HANDLE_USBBUS5:
      res := 'USB5';
    cantp_handle.PCANTP_HANDLE_USBBUS6:
      res := 'USB6';
    cantp_handle.PCANTP_HANDLE_USBBUS7:
      res := 'USB7';
    cantp_handle.PCANTP_HANDLE_USBBUS8:
      res := 'USB8';
    cantp_handle.PCANTP_HANDLE_USBBUS9:
      res := 'USB9';
    cantp_handle.PCANTP_HANDLE_USBBUS10:
      res := 'USB10';
    cantp_handle.PCANTP_HANDLE_USBBUS11:
      res := 'USB11';
    cantp_handle.PCANTP_HANDLE_USBBUS12:
      res := 'USB12';
    cantp_handle.PCANTP_HANDLE_USBBUS13:
      res := 'USB13';
    cantp_handle.PCANTP_HANDLE_USBBUS14:
      res := 'USB14';
    cantp_handle.PCANTP_HANDLE_USBBUS15:
      res := 'USB15';
    cantp_handle.PCANTP_HANDLE_USBBUS16:
      res := 'USB16';
    cantp_handle.PCANTP_HANDLE_PCCBUS1:
      res := 'PCC1';
    cantp_handle.PCANTP_HANDLE_PCCBUS2:
      res := 'PCC2';
    cantp_handle.PCANTP_HANDLE_LANBUS1:
      res := 'LAN1';
    cantp_handle.PCANTP_HANDLE_LANBUS2:
      res := 'LAN2';
    cantp_handle.PCANTP_HANDLE_LANBUS3:
      res := 'LAN3';
    cantp_handle.PCANTP_HANDLE_LANBUS4:
      res := 'LAN4';
    cantp_handle.PCANTP_HANDLE_LANBUS5:
      res := 'LAN5';
    cantp_handle.PCANTP_HANDLE_LANBUS6:
      res := 'LAN6';
    cantp_handle.PCANTP_HANDLE_LANBUS7:
      res := 'LAN7';
    cantp_handle.PCANTP_HANDLE_LANBUS8:
      res := 'LAN8';
    cantp_handle.PCANTP_HANDLE_LANBUS9:
      res := 'LAN9';
    cantp_handle.PCANTP_HANDLE_LANBUS10:
      res := 'LAN10';
    cantp_handle.PCANTP_HANDLE_LANBUS11:
      res := 'LAN11';
    cantp_handle.PCANTP_HANDLE_LANBUS12:
      res := 'LAN12';
    cantp_handle.PCANTP_HANDLE_LANBUS13:
      res := 'LAN13';
    cantp_handle.PCANTP_HANDLE_LANBUS14:
      res := 'LAN14';
    cantp_handle.PCANTP_HANDLE_LANBUS15:
      res := 'LAN15';
    cantp_handle.PCANTP_HANDLE_LANBUS16:
      res := 'LAN16';
  else
    res := 'UNKNOWN';
  end;
  result := res;
end;

/// <summary>Wait for a message to be received on a channel and show reception progress</summary>
/// <param name="handle">Channel handle</param>
/// <param name="msg_buffer">Received message</param>
/// <returns>a status</returns>
Function wait_and_show_progress(handle: cantp_handle; var msg_buffer: cantp_msg): cantp_status;
var
  res: cantp_status;
  progress: cantp_msgprogress;
  msgpend: cantp_msg;
  i: Uint32;
begin
  // Set a buffer in order to receive a copy of the pending message
  FillChar(progress, sizeof(progress), 0);
  progress.buffer := @msgpend;

  repeat

    Sleep(300);
    // Refresh progress structure and pending message
    res := TCantpApi.GetMsgProgress_2016(handle, @msg_buffer, cantp_msgdirection.PCANTP_MSGDIRECTION_RX, progress);
    // Show progress pencentage and last bytes
    Write(' RX progress on ', cantp_handle_toShortString(handle), ' = ', progress.percentage, '%, last bytes ');
    for i := msgpend.msgdata_any^.length - 10 to msgpend.msgdata_any^.length - 1 do
      Write(Format('%x', [PByte(msgpend.msgdata_isotp^.data)[i]]));
    Writeln(' ', OK_KO(TCantpApi.StatusIsOk_2016(res)));

  until progress.state <> cantp_msgprogress_state.PCANTP_MSGPROGRESS_STATE_PROCESSING;

  // The message is received, read it
  res := TCantpApi.Read_2016(handle, msg_buffer);

  result := res;
end;

/// <summary>Wait and read single message</summary>
/// <param name="handle">Channel handle</param>
/// <param name="msg_buffer">Received message (out)</param>
/// <param name="evt">Receive event associated to the channel</param>
/// <returns>Status</returns>
function wait_and_read_msg(handle: cantp_handle; evt: THandle; var msg_buffer: cantp_msg): cantp_status;
var
  res: cantp_status;
  wait_result: Uint32;

begin
  res := cantp_status.PCANTP_STATUS_NO_MESSAGE;

  // Wait a receive event and get a message
  // If we receive something, read the message
  wait_result := WaitForSingleObject(evt, 100);
  if wait_result = WAIT_OBJECT_0 then
  begin
    res := TCantpApi.Read_2016(handle, msg_buffer);
    Writeln('Wait a message on ', cantp_handle_toShortString(handle), ' : ', STATUS_OK_KO(res));

    // Check if the message is totaly received (only in ISOTP mode)
    if TCantpApi.StatusIsOk_2016(res) and ((Uint32(cantp_msgtype.PCANTP_MSGTYPE_ISOTP) and Uint32(msg_buffer.typem)) = Uint32(cantp_msgtype.PCANTP_MSGTYPE_ISOTP)) then
    begin
      // test the rx indication
      if (Uint32(msg_buffer.msgdata_isotp^.netaddrinfo.msgtype) and Uint32(cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_FLAG_INDICATION_RX))
        = Uint32(cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_FLAG_INDICATION_RX) then
        // The message is being received, wait and show progress
        res := wait_and_show_progress(handle, msg_buffer);
    end;
  end;
  result := res;
end;

/// <summary>Write message data in a binary file</summary>
/// <param name="file_path">Output file path</param>
/// <param name="msg">Message to write</param>
procedure write_binary_output_file(file_path: string; const msg: Pcantp_msg);
var
  len: Uint32;
  myfile: File of Byte;
  i: Uint32;
  lsb: Byte;
  msb: Byte;
begin
  if msg.msgdata_any <> nil then
  begin
    len := msg.msgdata_any^.length;
    try
      try
        AssignFile(myfile, file_path);
        ReWrite(myfile);
        i := 0;
        while i < len do
        begin
          lsb := PByte(msg.msgdata_any^.data)[i];
          msb := PByte(msg.msgdata_any^.data)[i + 1];
          write(myfile, msb, lsb);
          i := i + 2;
        end;
      except
        on E: Exception do
          Writeln(E.Classname, ': ', E.Message);
      end;
    finally
      closefile(myfile);
    end;
  end;
end;

/// <summary>
/// Entry point of the program, start a small CAN ISO TP read/write example
/// -> Send a message containing Peak logo
/// -> Receive message, wait complete reception (show progress)
/// -> Show loopback message and received message (Peak logo)
/// </summary>

const
  STMIN_600US = $F6;

var
  // Local variables
  res: cantp_status;
  buffer: array [0 .. 499] of AnsiChar;
  STmin: Uint32;
  rx_msg: cantp_msg;
  tx_msg: cantp_msg;
  loopback_msg: cantp_msg;
  mapping: cantp_mapping;
  reverse_mapping: cantp_mapping;
  transmitter_handle: cantp_handle;
  receiver_handle: cantp_handle;
  transmitter_receive_event: THandle;
  receiver_receive_event: THandle;
  null_handle: THandle;
  dummy: char;
  fileTxOk: Boolean;
  fileRxOk: Boolean;

begin
  try

    // Initialize variables
    transmitter_handle := cantp_handle.PCANTP_HANDLE_USBBUS1; // TODO: modify the value according to your available PCAN devices.
    receiver_handle := cantp_handle.PCANTP_HANDLE_USBBUS2; // TODO: modify the value according to your available PCAN devices.
    FillChar(tx_msg, sizeof(tx_msg), 0);
    FillChar(rx_msg, sizeof(rx_msg), 0);
    FillChar(loopback_msg, sizeof(loopback_msg), 0);
    FillChar(mapping, sizeof(mapping), 0);
    FillChar(reverse_mapping, sizeof(reverse_mapping), 0);

    // Print version informations
    TCantpApi.GetValue_2016(cantp_handle.PCANTP_HANDLE_NONEBUS, cantp_parameter.PCANTP_PARAMETER_API_VERSION, buffer, 500);
    // Writeln(Format(' PCAN - ISO - TP API Version: %s ', [Buffer]));
    Writeln('PCAN - ISO - TP API Version : ', buffer);

    // Initialize channels : CAN2.0 - 500Kbit/s
    res := TCantpApi.Initialize_2016(transmitter_handle, cantp_baudrate.PCANTP_BAUDRATE_500K);
    Writeln('Initialize transmitter : ', STATUS_OK_KO(res));
    res := TCantpApi.Initialize_2016(receiver_handle, cantp_baudrate.PCANTP_BAUDRATE_500K);
    Writeln('Initialize receiver : ', STATUS_OK_KO(res));

    // Change STmin value to 600us
    STmin := STMIN_600US;
    res := TCantpApi.SetValue_2016(transmitter_handle, cantp_parameter.PCANTP_PARAMETER_SEPARATION_TIME, PByte(@STmin), sizeof(STmin));
    Writeln('Set STMIN = 600us on transmitter : ', STATUS_OK_KO(res));
    res := TCantpApi.SetValue_2016(receiver_handle, cantp_parameter.PCANTP_PARAMETER_SEPARATION_TIME, PByte(@STmin), sizeof(STmin));
    Writeln('Set STMIN = 600us on receiver : ', STATUS_OK_KO(res));

    // Create and set a receive event on transmitter
    transmitter_receive_event := CreateEvent(nil, false, false, '');
    res := TCantpApi.SetValue_2016(transmitter_handle, PCANTP_PARAMETER_RECEIVE_EVENT, PByte(@transmitter_receive_event), sizeof(transmitter_receive_event));
    Writeln('Set receive event on transmitter : ', STATUS_OK_KO(res));

    // Create and set a receive event on receiver
    receiver_receive_event := CreateEvent(nil, false, false, '');
    res := TCantpApi.SetValue_2016(receiver_handle, PCANTP_PARAMETER_RECEIVE_EVENT, PByte(@receiver_receive_event), sizeof(receiver_receive_event));
    Writeln('Set receive event on receiver : ', STATUS_OK_KO(res));

    // Allocate messages
    res := TCantpApi.MsgDataAlloc_2016(tx_msg, cantp_msgtype.PCANTP_MSGTYPE_ISOTP);
    Writeln('Allocate tx message : ', STATUS_OK_KO(res));
    res := TCantpApi.MsgDataAlloc_2016(loopback_msg, cantp_msgtype.PCANTP_MSGTYPE_NONE);
    Writeln('Allocate loopback message : ', STATUS_OK_KO(res));
    res := TCantpApi.MsgDataAlloc_2016(rx_msg, cantp_msgtype.PCANTP_MSGTYPE_NONE);
    Writeln('Allocate rx message : ', STATUS_OK_KO(res));

    // Create a simple physical mapping:
    // - Source 0xF1 (transmitter), target 0x01 (receiver), CAN id 0xA1, CAN ID flow control 0xA2
    // - Diagnostic message in a classic format
    mapping.can_id := $A1;
    mapping.can_id_flow_ctrl := $A2;
    mapping.can_msgtype := cantp_can_msgtype.PCANTP_CAN_MSGTYPE_STANDARD;
    mapping.can_tx_dlc := $0;
    mapping.netaddrinfo.extension_addr := $00;
    mapping.netaddrinfo.Format := cantp_isotp_format.PCANTP_ISOTP_FORMAT_NORMAL;
    mapping.netaddrinfo.msgtype := cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC;
    mapping.netaddrinfo.source_addr := $F1;
    mapping.netaddrinfo.target_addr := $01;
    mapping.netaddrinfo.target_type := cantp_isotp_addressing.PCANTP_ISOTP_ADDRESSING_PHYSICAL;

    // Create the associated reversed mapping:
    reverse_mapping := mapping;
    reverse_mapping.can_id := mapping.can_id_flow_ctrl;
    reverse_mapping.can_id_flow_ctrl := mapping.can_id;
    reverse_mapping.netaddrinfo.source_addr := mapping.netaddrinfo.target_addr;
    reverse_mapping.netaddrinfo.target_addr := mapping.netaddrinfo.source_addr;

    // Add mapping on channels
    res := TCantpApi.AddMapping_2016(transmitter_handle, @mapping);
    Writeln('Add a simple mapping on transmitter ', cantp_handle_toShortString(transmitter_handle), ' : ', STATUS_OK_KO(res));
    res := TCantpApi.AddMapping_2016(transmitter_handle, @reverse_mapping);
    Writeln('Add the reverse mapping on transmitter ', cantp_handle_toShortString(transmitter_handle), ' : ', STATUS_OK_KO(res));
    res := TCantpApi.AddMapping_2016(receiver_handle, @mapping);
    Writeln('Add a simple mapping on receiver ', cantp_handle_toShortString(receiver_handle), ' : ', STATUS_OK_KO(res));
    res := TCantpApi.AddMapping_2016(receiver_handle, @reverse_mapping);
    Writeln('Add the reverse mapping on receiver ', cantp_handle_toShortString(receiver_handle), ' : ', STATUS_OK_KO(res));

    // Initialize Tx message containing a heavy data
    res := TCantpApi.MsgDataInit_2016(tx_msg, TCanTpApi.PCANTP_CAN_ID_DEFINED_BY_NAI, mapping.can_msgtype, Uint32(HEAVY_DATA_SIZE), PByte(@HEAVY_DATA), @mapping.netaddrinfo);
    Writeln('Initialize tx message : ', STATUS_OK_KO(res));

    // Write message on transmitter
    res := TCantpApi.Write_2016(transmitter_handle, tx_msg);
    Writeln('Write message on transmitter ', cantp_handle_toShortString(transmitter_handle), ' : ', STATUS_OK_KO(res));

    // Read loopback message and save it
    res := wait_and_read_msg(transmitter_handle, transmitter_receive_event, loopback_msg);
    Writeln('Read loopback message on transmitter ', cantp_handle_toShortString(transmitter_handle), ' : ', STATUS_OK_KO(res));
    fileTxOk := TCantpApi.StatusIsOk_2016(res);
    if fileTxOk then
      write_binary_output_file('loopback_message.png', @loopback_msg);

    // Read received message and save it
    res := wait_and_read_msg(receiver_handle, receiver_receive_event, rx_msg);
    Writeln('Read message on receiver ', cantp_handle_toShortString(receiver_handle), ' : ', STATUS_OK_KO(res));
    fileRxOk := TCantpApi.StatusIsOk_2016(res);
    if fileRxOk then
      write_binary_output_file('received_message.png', @rx_msg);

    // Show & remove result:
    if fileTxOk then
    begin
      Writeln(' >>> Opening transmitted message (PEAK logo).');
      if FileExists('loopback_message.png') then
        ShellExecute(0, 'OPEN', 'loopback_message.png', nil, nil, 0);
    end
    else
      Writeln(' >>> Failed to successfully transmit message (PEAK logo)!');

    if (fileRxOk) then
    begin
      Writeln(' >>> Opening received message (PEAK logo).');
      if FileExists('received_message.png') then
        ShellExecute(0, 'OPEN', 'received_message.png', nil, nil, 0);
    end
    else
      Writeln(' >>> Failed to successfully receive message (PEAK logo)!');

    // Free messages
    res := TCantpApi.MsgDataFree_2016(rx_msg);
    Writeln('Free rx message : ', STATUS_OK_KO(res));
    res := TCantpApi.MsgDataFree_2016(tx_msg);
    Writeln('Free tx message : ', STATUS_OK_KO(res));
    res := TCantpApi.MsgDataFree_2016(loopback_msg);
    Writeln('Free loopback message : ', STATUS_OK_KO(res));

    // Close receive event on transmitter
    null_handle := 0;
    res := TCantpApi.SetValue_2016(transmitter_handle, cantp_parameter.PCANTP_PARAMETER_RECEIVE_EVENT, PByte(@null_handle), sizeof(null_handle));
    Writeln('Stop receive event on transmitter ', cantp_handle_toShortString(transmitter_handle), ' : ', STATUS_OK_KO(res));
    CloseHandle(transmitter_receive_event);

    // Close receive event on receiver
    null_handle := 0;
    res := TCantpApi.SetValue_2016(receiver_handle, cantp_parameter.PCANTP_PARAMETER_RECEIVE_EVENT, PByte(@null_handle), sizeof(null_handle));
    Writeln('Stop receive event on receiver ', cantp_handle_toShortString(receiver_handle), ' : ', STATUS_OK_KO(res));
    CloseHandle(receiver_receive_event);

    // Uninitialize channels
    res := TCantpApi.Uninitialize_2016(transmitter_handle);
    Writeln('Uninitialize transmitter ', cantp_handle_toShortString(transmitter_handle), ' : ', STATUS_OK_KO(res));
    res := TCantpApi.Uninitialize_2016(receiver_handle);
    Writeln('Uninitialize receiver ', cantp_handle_toShortString(receiver_handle), ' : ', STATUS_OK_KO(res));

    // Remove results and exit
    Writeln('Press any key to exit...');
    Readln(dummy);
    try
      DeleteFile('loopback_message.png');
      DeleteFile('received_message.png');
    except
    end;
  except
    on E: Exception do
      Writeln(E.Classname, ': ', E.Message);
  end;

end.
