program _08_canfd_read_write;

{$APPTYPE CONSOLE}

uses
  SysUtils,
  ShellApi,
  windows,
  PCANBasic in 'PCANBasic.pas',
  PCANTP_2016 in 'PCANTP_2016.pas';

Function OK_KO(test: Boolean): String;
begin
  If test Then
    result := 'OK'
  else
    result := 'KO';
End;

Function STATUS_OK_KO(test: cantp_status): String;
begin
  result := OK_KO(TCantpApi.StatusIsOk_2016(test));
End;

/// <summary>Entry point of the program, start a small ISO TP CAN FD read/write example</summary>
const
  // Define Bitrate: SAE J2284-4: High-Speed CAN (HSC) for Vehicle Applications at 500 kbps with CAN FD Data at 2 Mbps
  PCAN_BITRATE_SAE_J2284_4: cantp_bitrate = 'f_clock=80000000,nom_brp=2,nom_tseg1=63,nom_tseg2=16,nom_sjw=16,data_brp=2,data_tseg1=15,data_tseg2=4,data_sjw=4';

var
  // Local variables
  Res: cantp_status;
  Buffer: array [0 .. 499] of AnsiChar;
  rx_msg: cantp_msg;
  tx_msg: cantp_msg;
  val: Byte;
  mapping: cantp_mapping;
  transmitter_handle: cantp_handle;
  receiver_handle: cantp_handle;
  receive_event: THandle;
  null_handle: THandle;
  wait_result: UInt32;
  dummy: char;
  can_id: UInt32;

begin
  try

    // Initialize variables
    transmitter_handle := cantp_handle.PCANTP_HANDLE_USBBUS1; // TODO: modify the value according to your available PCAN devices.
    receiver_handle := cantp_handle.PCANTP_HANDLE_USBBUS2; // TODO: modify the value according to your available PCAN devices.
    FillChar(tx_msg, sizeof(tx_msg), 0);
    FillChar(rx_msg, sizeof(rx_msg), 0);

    // Print version informations
    TCantpApi.GetValue_2016(cantp_handle.PCANTP_HANDLE_NONEBUS, cantp_parameter.PCANTP_PARAMETER_API_VERSION, Buffer, 500);
    // Writeln(Format(' PCAN - ISO - TP API Version: %s ', [Buffer]));
    Writeln('PCAN - ISO - TP API Version : ', Buffer);

    // Initialize channels, bitrate: SAE J2284-4
    Res := TCantpApi.InitializeFD_2016(transmitter_handle, PCAN_BITRATE_SAE_J2284_4);
    Writeln('Initialize transmitter : ', STATUS_OK_KO(Res));
    Res := TCantpApi.InitializeFD_2016(receiver_handle, PCAN_BITRATE_SAE_J2284_4);
    Writeln('Initialize receiver : ', STATUS_OK_KO(Res));

    // Create and set a receive event on receiver
    receive_event := CreateEvent(nil, false, false, '');
    Res := TCantpApi.SetValue_2016(receiver_handle, PCANTP_PARAMETER_RECEIVE_EVENT, PByte(@receive_event), sizeof(receive_event));
    Writeln('Set receive event : ', STATUS_OK_KO(Res));

    // Allocate tx CAN-FD message
    Res := TCantpApi.MsgDataAlloc_2016(tx_msg, cantp_msgtype.PCANTP_MSGTYPE_CANFD);
    Writeln('Allocate tx message : ', STATUS_OK_KO(Res));

    // Allocate rx message
    Res := TCantpApi.MsgDataAlloc_2016(rx_msg, cantp_msgtype.PCANTP_MSGTYPE_NONE);
    Writeln('Allocate rx message : ', STATUS_OK_KO(Res));

    // Initialize Tx message containing "42"
    can_id := $A1;
    val := $42;
    Res := TCantpApi.MsgDataInit_2016(tx_msg, can_id, mapping.can_msgtype, 1, PByte(@val), @mapping.netaddrinfo);
    Writeln('Initialize tx message : ', STATUS_OK_KO(Res));

    // Write "42" message
    Res := TCantpApi.Write_2016(transmitter_handle, tx_msg);
    Writeln('Write "42" message : ', STATUS_OK_KO(Res));

    // Wait a receive event on receiver
    wait_result := WaitForSingleObject(receive_event, 100);
    Writeln('Wait a message on receiver : ', OK_KO(wait_result = WAIT_OBJECT_0));

    // If we receive something on the receiver, read the message
    if wait_result = WAIT_OBJECT_0 then
    begin
      Res := TCantpApi.Read_2016(receiver_handle, rx_msg);
      Writeln('Read message on receiver : ', STATUS_OK_KO(Res));
      if TCantpApi.StatusIsOk_2016(Res) and (rx_msg.msgdata_any^.length <> 0) then
      begin
        if PByte(rx_msg.msgdata_any^.data)[0] = $42 then
          Writeln('Check if the message is "42" : OK')
        else
          Writeln('Check if the message is "42" : NOK')
      end;
    end;

    // Free messages
    Res := TCantpApi.MsgDataFree_2016(rx_msg);
    Writeln('Free rx message : ', STATUS_OK_KO(Res));
    Res := TCantpApi.MsgDataFree_2016(tx_msg);
    Writeln('Free tx message : ', STATUS_OK_KO(Res));

    // Close receive event
    null_handle := 0;
    Res := TCantpApi.SetValue_2016(receiver_handle, PCANTP_PARAMETER_RECEIVE_EVENT, PByte(@null_handle), sizeof(null_handle));
    Writeln('Stop receive event  : ', STATUS_OK_KO(Res));
    CloseHandle(receive_event);

    // Uninitialize
    Res := TCantpApi.Uninitialize_2016(transmitter_handle);
    Writeln('Uninitialize transmitter : ', STATUS_OK_KO(Res));
    Res := TCantpApi.Uninitialize_2016(receiver_handle);
    Writeln('Uninitialize receiver : ', STATUS_OK_KO(Res));

    // Exit
    Writeln('Press any key to exit...');
    Readln(dummy);
  except
    on E: Exception do
      Writeln(E.Classname, ': ', E.Message);
  end;

end.
