program _09_server_isotp;

{$APPTYPE CONSOLE}

uses
  SysUtils,
  ShellApi,
  windows,
  PCANBasic in 'PCANBasic.pas',
  PCANTP_2016 in 'PCANTP_2016.pas';

const
  N_SA: Byte = $F1;
  N_TA_PHYS: Byte = $13;
  N_TA_FUNC: Byte = $26;
  N_RA: Byte = $52;
  USE_CAN_FD: Boolean = false;
  USE_EVENT: Boolean = false;

// Function called to clean opened ISO-TP channels
Procedure consoleExit();
begin
  TCanTpApi.Uninitialize_2016(cantp_handle.PCANTP_HANDLE_NONEBUS);
end;

// Callback to handle closure of the console window
Function ConsoleCtrlCheck(dwCtrlType: DWORD): Boolean; cdecl;
var
  isClosing: Boolean;
begin
  case dwCtrlType of
    CTRL_CLOSE_EVENT, CTRL_C_EVENT, CTRL_BREAK_EVENT, CTRL_LOGOFF_EVENT, CTRL_SHUTDOWN_EVENT:
      isClosing := true;
  else
    isClosing := false;
  end;
  if isClosing then
    consoleExit();
  result := false;
end;

Function convertToHandle(s: String): cantp_handle;
var
  res: cantp_handle;
begin
  try
    res := cantp_handle(strToInt(s));
  Except
    res := cantp_handle.PCANTP_HANDLE_NONEBUS;
  end;
  result := res;
end;

Function convertToBaudRate(s: String): cantp_baudrate;
var
  res: cantp_baudrate;
begin
  try
    res := cantp_baudrate(strToInt(s));
  Except
    res := cantp_baudrate(0);
  end;
  result := res;
end;

Function convertToBool(s: String): Boolean;
var
  res: Boolean;
begin
  try
    res := Boolean(strToInt(s));
  Except
    res := false;
  end;
  result := res;
end;

Function KeyPressed: Boolean;
Const
  buffer_size = 20;
Var
  zone: Array [1 .. buffer_size] Of TInputRecord;
  is_waiting: Cardinal;
  i: Cardinal;
  han: THandle;
Begin
  han := GetStdHandle(STD_INPUT_HANDLE);
  PeekConsoleInput(han, zone[1], buffer_size, is_waiting);
  result := false;
  i := 1;
  While Not result And (i <= is_waiting) And (i <= buffer_size) Do
  Begin
    result := (zone[i].EventType = 1) And (zone[i].Event.KeyEvent.bKeyDown) And (zone[i].Event.KeyEvent.AsciiChar <> #0);
    inc(i);
  End;
  If (is_waiting <> 0) And Not result Then
    ReadConsoleInput(STD_INPUT_HANDLE, zone[1], is_waiting, i);
End;

Function ReadKey: AnsiChar;
Var
  zone: TInputRecord;
  nb_read: Cardinal;
  han: THandle;
Begin
  han := GetStdHandle(STD_INPUT_HANDLE);
  result := #0;
  Repeat
    ReadConsoleInput(han, zone, 1, nb_read);
    If (nb_read = 1) And (zone.EventType = 1) And (zone.Event.KeyEvent.bKeyDown) And (zone.Event.KeyEvent.AsciiChar <> #0) Then
      result := zone.Event.KeyEvent.AsciiChar;
  Until result <> #0;
End;

procedure clearScreen();
const
  coordScreen: TCoord = (X: 0; Y: 0);

var
  hConsole: THandle;
  cCharsWritten: DWORD;
  csbi: TConsoleScreenBufferInfo;
  dwConSize: DWORD;

begin
  hConsole := GetStdHandle(STD_OUTPUT_HANDLE);
  if hConsole = INVALID_HANDLE_VALUE then
    exit;
  if not GetConsoleScreenBufferInfo(hConsole, csbi) then
    exit;
  dwConSize := csbi.dwSize.X * csbi.dwSize.Y;
  if not FillConsoleOutputCharacter(hConsole, #32, dwConSize, coordScreen, cCharsWritten) then
    exit;
  if not GetConsoleScreenBufferInfo(hConsole, csbi) then
    exit;
  if not FillConsoleOutputAttribute(hConsole, csbi.wAttributes, dwConSize, coordScreen, cCharsWritten) then
    exit;
  SetConsoleCursorPosition(hConsole, coordScreen);
end;

/// <summary>Configure mappings to handle a two-way communication and a functional communication (broadcast).</summary>
/// <param name="channel">CAN channel to add </param>
/// <param name="can_msgtype">CAN frame </param>
/// <param name="source_addr">ISO-TP source address</param>
/// <param name="target_addr">ISO-TP target address</param>
/// <param name="extension_addr">ISO-TP extension address</param>
/// <param name="target_addr_func">ISO-TP target address for functional addressing</param>
/// <param name="can_id">CAN ID used to communicate for physical addressing</param>
/// <param name="can_id_flow_ctrl">Flow Control CAN ID used during physical addressing communication</param>
/// <param name="can_id_func">CAN ID used to communicate for functionnal addressing</param>
/// <param name="isotp_msgtype">ISOTP message's type</param>
/// <param name="isotp_format">ISOTP message's format</param>
/// <return>true on successful configuration</return>
Function initialize_mappings(channel: cantp_handle; can_msgtype: cantp_can_msgtype; source_addr: UInt16; target_addr: UInt16; extension_addr: Byte; target_addr_func: UInt16; can_id: UInt32;
  can_id_flow_ctrl: UInt32; can_id_func: UInt32; isotp_msgtype: cantp_isotp_msgtype; isotp_format: cantp_isotp_format): Boolean; overload;
var
  res: Boolean;
  mapping_phys_rx: cantp_mapping;
  mapping_phys_tx: cantp_mapping;
  mapping_func: cantp_mapping;
  status: cantp_status;
begin
  res := true;
  // "FillChar" allows to clean variables (it is common to leave can_tx_dl uninitialized which can lead to invalid 0xCC values)
  FillChar(mapping_phys_rx, sizeof(cantp_mapping), 0);
  FillChar(mapping_phys_tx, sizeof(cantp_mapping), 0);
  FillChar(mapping_func, sizeof(cantp_mapping), 0);

  // configure a mapping to transmit physical message
  mapping_phys_tx.can_id := can_id;
  mapping_phys_tx.can_id_flow_ctrl := can_id_flow_ctrl;
  mapping_phys_tx.can_msgtype := can_msgtype;
  mapping_phys_tx.netaddrinfo.format := isotp_format;
  mapping_phys_tx.netaddrinfo.msgtype := isotp_msgtype;
  mapping_phys_tx.netaddrinfo.target_type := cantp_isotp_addressing.PCANTP_ISOTP_ADDRESSING_PHYSICAL;
  mapping_phys_tx.netaddrinfo.source_addr := source_addr;
  mapping_phys_tx.netaddrinfo.target_addr := target_addr;
  mapping_phys_tx.netaddrinfo.extension_addr := extension_addr;
  status := TCanTpApi.AddMapping_2016(channel, @mapping_phys_tx);
  if not TCanTpApi.StatusIsOk_2016(status) then
  begin
    Writeln(format('Failed to configure mapping (can_id=0x%x) (sts=0x%x)', [mapping_phys_tx.can_id, UInt32(status)]));
    res := false;
  end;
  // configure another mapping to receive physical message (invert source and target in previous mapping)
  mapping_phys_rx := mapping_phys_tx;
  mapping_phys_rx.can_id := mapping_phys_tx.can_id_flow_ctrl;
  mapping_phys_rx.can_id_flow_ctrl := mapping_phys_tx.can_id;
  mapping_phys_rx.netaddrinfo.source_addr := mapping_phys_tx.netaddrinfo.target_addr;
  mapping_phys_rx.netaddrinfo.target_addr := mapping_phys_tx.netaddrinfo.source_addr;
  status := TCanTpApi.AddMapping_2016(channel, @mapping_phys_rx);
  if not TCanTpApi.StatusIsOk_2016(status) then
  begin
    Writeln(format('Failed to configure mapping (can_id=0x%x) (sts=0x%x)', [mapping_phys_rx.can_id, UInt32(status)]));
    res := false;
  end;
  if can_id_func <> $FFFFFFFF then
  begin
    // configure another mapping to transmit functional message
    mapping_func := mapping_phys_tx;
    mapping_func.can_id := can_id_func;
    mapping_func.can_id_flow_ctrl := TCanTpApi.PCANTP_MAPPING_FLOW_CTRL_NONE;
    mapping_func.netaddrinfo.target_type := cantp_isotp_addressing.PCANTP_ISOTP_ADDRESSING_FUNCTIONAL;
    mapping_func.netaddrinfo.target_addr := target_addr_func;
    status := TCanTpApi.AddMapping_2016(channel, @mapping_func);
    if not TCanTpApi.StatusIsOk_2016(status) then
    begin
      Writeln(format('Failed to configure mapping (can_id=0x%x) (sts=0x%x)', [mapping_func.can_id, UInt32(status)]));
      res := false;
    end;
  end;
  result := res;
end;

/// <summary>Initializes mappings compatible with PCTPClient/PCTPServer sample</summary>
/// <param name="channel">CAN channel to add the mappings to.</param>
/// <return>true on successful configuration</return>
Function initialize_mappings(channel: cantp_handle): Boolean; overload;
var
  res: Boolean;
begin
  res := true;

  // 11 bit can ID, normal format addressing, diagnostic message (mandatory)
  res := res and initialize_mappings(channel, cantp_can_msgtype.PCANTP_CAN_MSGTYPE_STANDARD, N_SA, N_TA_PHYS, $00, N_TA_FUNC, $A1, $A2, $A5, cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC,
    cantp_isotp_format.PCANTP_ISOTP_FORMAT_NORMAL);
  // 11 bit can ID, extended format addressing, diagnostic message (mandatory)
  res := res and initialize_mappings(channel, cantp_can_msgtype.PCANTP_CAN_MSGTYPE_STANDARD, N_SA, N_TA_PHYS, $00, N_TA_FUNC, $B1, $B2, $B5, cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC,
    cantp_isotp_format.PCANTP_ISOTP_FORMAT_EXTENDED);
  res := res and initialize_mappings(channel, cantp_can_msgtype.PCANTP_CAN_MSGTYPE_STANDARD, N_SA, N_TA_PHYS, $00, N_TA_FUNC, $C1, $C2, $C5, cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC,
    cantp_isotp_format.PCANTP_ISOTP_FORMAT_EXTENDED);

  // 11 bit can ID, mixed format addressing, remote diagnostic message (mandatory)
  res := res and initialize_mappings(channel, cantp_can_msgtype.PCANTP_CAN_MSGTYPE_STANDARD, N_SA, N_TA_PHYS, N_RA, N_TA_FUNC, $D1, $D2, $D5,
    cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_REMOTE_DIAGNOSTIC, cantp_isotp_format.PCANTP_ISOTP_FORMAT_MIXED);

  // 29 bit can ID, normal format addressing, diagnostic message (mandatory)
  res := res and initialize_mappings(channel, cantp_can_msgtype.PCANTP_CAN_MSGTYPE_EXTENDED, N_SA, N_TA_PHYS, $00, N_TA_FUNC, $A123A1, $A123A2, $A123A5,
    cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, cantp_isotp_format.PCANTP_ISOTP_FORMAT_NORMAL);
  // 29 bit can ID, extended format addressing, diagnostic message
  res := res and initialize_mappings(channel, cantp_can_msgtype.PCANTP_CAN_MSGTYPE_EXTENDED, N_SA, N_TA_PHYS, $00, N_TA_FUNC, $A123C1, $A123C2, $A123C5,
    cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, cantp_isotp_format.PCANTP_ISOTP_FORMAT_EXTENDED);
  res := res and initialize_mappings(channel, cantp_can_msgtype.PCANTP_CAN_MSGTYPE_EXTENDED, N_SA, N_TA_PHYS, $00, N_TA_FUNC, $A123D1, $A123D2, $A123D5,
    cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, cantp_isotp_format.PCANTP_ISOTP_FORMAT_EXTENDED);

  result := res;
end;

/// <summary>Outputs a summary of a received message.</summary>
/// <param name="message">The message.</param>
/// <param name="timestamp">The timestamp of the message.</param>
/// <param name="status">status of the Read function.</param>
/// <returns>1 on success (2 if extra information is available within status), 0 if no message and -1 on error</returns>
Function display_message(const message: Pcantp_msg; timestamp: cantp_timestamp; status: cantp_status): integer;
var
  strLoopback: String;
  strMsgType: String;
  displayedDataLength: UInt32;
  res: integer;
  len: UInt32;
  strNetSta: String;
  strPending: String;
  buffer: String;
  i: UInt32;
begin
  strLoopback := '';
  strMsgType := '';
  displayedDataLength := 0;

  // check status
  if TCanTpApi.StatusIsOk_2016(status) then
    res := 1
  else
    res := -1;

  if res <> 1 then
  begin
    if TCanTpApi.StatusIsOk_2016(status, cantp_status.PCANTP_STATUS_NO_MESSAGE) then
      // no message received ignore
      res := 0
    else
      Writeln(format('Failed to read message (sts=0x%x).', [UInt32(status)]));
    result := res;
    exit;
  end;
  // check bus errors within status
  if not TCanTpApi.StatusIsOk_2016(status, cantp_status.PCANTP_STATUS_OK, true) then
  begin
    Writeln(format('Status returned information/warnings (sts=0x%x).', [UInt32(status)]));
    res := 2;
  end;

  if message.msgdata_any = nil then
  begin
    Writeln('Message null');
    res := -1;
    result := res;
    exit;
  end;

  len := message.msgdata_any^.Length;

  // check if message has the loopback flag
  if (UInt32(message.msgdata_any^.flags) and UInt32(cantp_msgflag.PCANTP_MSGFLAG_LOOPBACK)) <> 0 then
    strLoopback := 'loopback ';

  // format message's type
  Case message.typem of
    cantp_msgtype.PCANTP_MSGTYPE_CAN:
      strMsgType := 'PCANTP_MSGTYPE_CAN';
    cantp_msgtype.PCANTP_MSGTYPE_CANFD:
      strMsgType := 'PCANTP_MSGTYPE_CANFD';
    cantp_msgtype.PCANTP_MSGTYPE_ISOTP:
      strMsgType := 'PCANTP_MSGTYPE_ISOTP';
    cantp_msgtype.PCANTP_MSGTYPE_NONE:
      strMsgType := 'PCANTP_MSGTYPE_NONE';
  else
    strMsgType := 'Unknown';
  end;

  // Only display data if network result is OK
  if message.msgdata_any^.netstatus = cantp_netstatus.PCANTP_NETSTATUS_OK then
    displayedDataLength := len;

  // Display generic information
  strNetSta := 'OK !';
  if message.msgdata_any^.netstatus <> cantp_netstatus.PCANTP_NETSTATUS_OK then
    strNetSta := 'ERROR !!!';
  Writeln(sLineBreak + format('%d - Received %s%s message : canid=0x%x, length=%d - result: 0x%x - %s', [timestamp, strLoopback, strMsgType, message.can_info.can_id, len,
    UInt32(message.msgdata_any^.netstatus), strNetSta]));

  if message.typem = cantp_msgtype.PCANTP_MSGTYPE_ISOTP then
  begin
    strPending := 'Completed';
    // Limit displayed data if message is pending
    if (displayedDataLength > 0) and (((UInt32(message.msgdata_isotp^.netaddrinfo.msgtype) and UInt32(cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_FLAG_INDICATION)) <> 0)) then
    begin
      if len > 7 then
        displayedDataLength := 7
      else
        displayedDataLength := len;
      strPending := '/!\ Pending';
    end;
    // Display iso-tp message's information
    Writeln(#9 + format('%s message from 0x%x (to 0x%x, with RA 0x%x)', [strPending, message.msgdata_isotp^.netaddrinfo.source_addr, message.msgdata_isotp^.netaddrinfo.target_addr,
      message.msgdata_isotp^.netaddrinfo.extension_addr]));
  end;

  if displayedDataLength > 0 then
  begin
    buffer := #9;
    // display data
    buffer := buffer + format('\-> Length: %d, Data= ', [len]);
    for i := 0 to displayedDataLength - 1 do
      buffer := buffer + format('%x ', [PByte(message.msgdata_any^.data)[i]]);
    if displayedDataLength <> len then
      buffer := buffer + '...';
    Writeln(buffer);
  end;
  result := res;
end;

var
  status: cantp_status;
  buffer: array [0 .. 499] of AnsiChar;
  channel: cantp_handle;
  useCanFd: Boolean;
  baudrate: cantp_baudrate;
  bitrateFd: cantp_bitrate;
  useEvent: Boolean;
  nbErr: integer;
  bStop: Boolean;
  message: cantp_msg;
  timestamp: cantp_timestamp;
  result: Boolean;
  doLoop: Boolean;
  param: Byte;
  eventRcv: THandle;
  wait_result: UInt32;
  keyboard_res: AnsiChar;
  Dummy: char;

/// <summary>Main entry-point for this application.</summary>

begin
  try
    Writeln(sLineBreak + 'Usage: _09_server_isotp.exe [cantp_handle] [USE_CAN_FD:0|1] [BTR0BTR1|BITRATE] [USE_EVENT:0|1]');
    Writeln(' * CAN: _09_server_isotp.exe 0x51 0 0x1C ');
    Writeln(' * CAN FD: _09_server_isotp.exe 0x51 1 ''f_clock=80000000, nom_brp=10, nom_tseg1=12, nom_tseg2=3, nom_sjw=1, data_brp=4, data_tseg1=7, data_tseg2=2, data_sjw=1'' ');
    Writeln(sLineBreak + '---------------------------------------------');

    // clear gracefully ISO-TP if console window is killed
    SetConsoleCtrlHandler(@ConsoleCtrlCheck, true);

    // Show version information
    status := TCanTpApi.GetValue_2016(cantp_handle.PCANTP_HANDLE_NONEBUS, cantp_parameter.PCANTP_PARAMETER_API_VERSION, buffer, 500);
    if TCanTpApi.StatusIsOk_2016(status) then
      Writeln('PCAN-ISO-TP API Version : ', buffer)
    else
      Writeln(format('Error failed to get PCAN-ISO-TP API Version (sts=0x%x)', [UInt32(status)]));

    // 1st argument is the PCAN-channel to use (PCAN-USB channel 2)
    if (ParamCount >= 1) then
      channel := convertToHandle(ParamStr(1))
    else
      channel := cantp_handle.PCANTP_HANDLE_USBBUS2;
    // 2nd argument is CAN FD
    if (ParamCount >= 2) then
      useCanFd := convertToBool(ParamStr(2))
    else
      useCanFd := USE_CAN_FD;
    // 3rd argument is bitrate (either CAN or CAN FD)
    baudrate := cantp_baudrate.PCANTP_BAUDRATE_500K;
    bitrateFd := 'f_clock=80000000, nom_brp=10, nom_tseg1=12, nom_tseg2=3, nom_sjw=1, data_brp=4, data_tseg1=7, data_tseg2=2, data_sjw=1';
    if (ParamCount >= 3) then
    begin
      if useCanFd then
        bitrateFd := cantp_bitrate(AnsiString(ParamStr(3)))
      else
        baudrate := convertToBaudRate(ParamStr(3));
    end;
    // 4th argument is USE EVENT
    if (ParamCount >= 4) then
      useEvent := convertToBool(ParamStr(4))
    else
      useEvent := USE_EVENT;

    // Initializing of the ISO-TP Communication session
    Writeln(format('Connecting to channel 0x%x...', [UInt32(channel)]));
    if not useCanFd then
    begin
      Writeln(format(' * btr0btr1 = 0x%x...', [UInt32(baudrate)]));
      status := TCanTpApi.Initialize_2016(channel, baudrate);
    end
    else
    begin
      Writeln(format(' * bitrateFd = "%s"...', [bitrateFd]));
      status := TCanTpApi.InitializeFD_2016(channel, bitrateFd);
    end;
    Writeln(format(' -> Initialize CANTP: 0x%x', [UInt32(status)]));

    // will start the reading loop only if successfully initialized
    nbErr := 0;
    bStop := false;
    if not TCanTpApi.StatusIsOk_2016(status) then
    begin
      bStop := true;
      nbErr := nbErr + 1;
      Writeln(' -> Initialization failed, exiting...');
    end;
    if not bStop then
    begin
      // configure the test session to log debug information
      param := TCanTpApi.PCANTP_DEBUG_INFO;
      status := TCanTpApi.SetValue_2016(channel, cantp_parameter.PCANTP_PARAMETER_DEBUG, PByte(@param), 1);
      if not TCanTpApi.StatusIsOk_2016(status) then
        Writeln(format('Failed to configure parameter "PCANTP_PARAMETER_DEBUG" (sts=0x%x).', [UInt32(status)]));

      // configure server - block size, 0 = unlimited
      param := 20;
      status := TCanTpApi.SetValue_2016(channel, cantp_parameter.PCANTP_PARAMETER_BLOCK_SIZE, PByte(@param), 1);
      if not TCanTpApi.StatusIsOk_2016(status) then
        Writeln(format('Failed to configure parameter "PCANTP_PARAMETER_BLOCK_SIZE" (sts=0x%x).', [UInt32(status)]));

      // configure server - wait Xms between segmented frames transmission
      param := 2;
      status := TCanTpApi.SetValue_2016(channel, cantp_parameter.PCANTP_PARAMETER_SEPARATION_TIME, PByte(@param), 1);
      if not TCanTpApi.StatusIsOk_2016(status) then
        Writeln(format('Failed to configure parameter "PCANTP_PARAMETER_SEPARATION_TIME" (sts=0x%x).', [UInt32(status)]));

      // configure server - enable enhanced addressing to be compatible with PCTPClient
      param := TCanTpApi.PCANTP_VALUE_PARAMETER_ON;
      status := TCanTpApi.SetValue_2016(channel, cantp_parameter.PCANTP_PARAMETER_SUPPORT_29B_ENHANCED, PByte(@param), 1);
      if not TCanTpApi.StatusIsOk_2016(status) then
        Writeln(format('Failed to configure parameter "PCANTP_PARAMETER_SUPPORT_29B_ENHANCED" (sts=0x%x).', [UInt32(status)]));

      // configure receive event
      if useEvent then
      begin
        Writeln(' -> Creating receive event');
        eventRcv := CreateEvent(nil, false, false, '');
        status := TCanTpApi.SetValue_2016(channel, PCANTP_PARAMETER_RECEIVE_EVENT, PByte(@eventRcv), sizeof(eventRcv));
        if not TCanTpApi.StatusIsOk_2016(status) then
        begin
          nbErr := nbErr + 1;
          bStop := true;
          Writeln(format('Failed to configure parameter "PCANTP_PARAMETER_RECEIVE_EVENT" (sts=0x%x).', [UInt32(status)]));
        end;
      end;

      // configure ISO-TP mappings
      if not initialize_mappings(channel) then
      begin
        nbErr := nbErr + 1;
        bStop := true;
        Writeln(' -> Mappings configuration failed, exiting...');
      end;
    end;

    // Output available commands
    if not bStop then
    begin
      Writeln(sLineBreak + '---------------------------------------------');
      Writeln(sLineBreak + sLineBreak + 'Note: press ''c'' or ''C'' to clear screen,');
      Writeln(sLineBreak + 'Note: press ''q'', ''Q'' or ''<Escape>'' to quit...');
      Writeln(sLineBreak + '---------------------------------------------');
      Writeln(sLineBreak + sLineBreak + ' -> Waiting for messages...');
    end;

    // Reading loop

    while not bStop do
    begin
      if useEvent then
      begin
        wait_result := WaitForSingleObject(eventRcv, 1);
        result := (wait_result = WAIT_OBJECT_0);
      end
      else
      begin
        Sleep(1);
        result := true;
      end;
      if result then
      begin

        // loop until no more message is available
        repeat

          doLoop := false;
          // initialize message
          status := TCanTpApi.MsgDataAlloc_2016(message, cantp_msgtype.PCANTP_MSGTYPE_NONE);
          if not TCanTpApi.StatusIsOk_2016(status) then
          begin
            Writeln(format('Failed to allocate message (sts=0x%x).', [UInt32(status)]));
            continue;
          end;

          // retrieve any message from Rx queue
          status := TCanTpApi.Read_2016(channel, message, @timestamp);
          doLoop := (display_message(@message, timestamp, status) > 0);
          // update error counter if a message was received but a network error occured
          if doLoop then
          begin
            if message.msgdata_any <> nil then
            begin
              if message.msgdata_any^.netstatus <> cantp_netstatus.PCANTP_NETSTATUS_OK then
                nbErr := nbErr + 1;
            end
            else
              Writeln('Failed to get netstatus of message.');
          end;
          // release message
          status := TCanTpApi.MsgDataFree_2016(message);
          if not TCanTpApi.StatusIsOk_2016(status) then
            Writeln(format('Failed to deallocate message (sts=0x%x).', [UInt32(status)]));

        until not doLoop;
      end;

      // Sleep(1);
      // check exit request
      if KeyPressed() then
      begin
        keyboard_res := ReadKey();
        case keyboard_res of
          'q', 'Q', char(27): // 27 : escape
            bStop := true;
          char($08), 'c', 'C': // $08 : backspace
            begin
              if keyboard_res = char($08) then
                TCanTpApi.Reset_2016(channel);
              clearScreen();
              Writeln(sLineBreak + 'Note: press ''c'' or ''C'' to clear screen,');
              Writeln('Note: press ''q'', ''Q'' or ''<Escape>'' to quit...' + sLineBreak);
            end
        end;
      end;
    end;

    // Display a small report
    Writeln;
    if nbErr > 0 then
      Writeln(format('ERROR : %d errors occured.', [nbErr]))
    else
      Writeln('ALL Transmissions succeeded !');
    Writeln(sLineBreak + sLineBreak + sLineBreak + 'Press <Enter> to quit...');
    Readln(Dummy);

    // release CAN-TP
    TCanTpApi.Uninitialize_2016(channel);

  except
    on E: Exception do
      // clear gracefully ISO-TP
      consoleExit();
  end;

end.
