program _10_client_isotp;

{$APPTYPE CONSOLE}

uses
  SysUtils,
  ShellApi,
  windows,
  PCANBasic in 'PCANBasic.pas',
  PCANTP_2016 in 'PCANTP_2016.pas';

const
  N_SA: Byte = $F1;
  N_TA_PHYS: Byte = $13;
  N_TA_FUNC: Byte = $26;
  N_RA: Byte = $52;
  USE_CAN_FD: Boolean = false;
  USE_EVENT: Boolean = false;

  CAN_TX_DL_CODE: Byte = $0D;
  CAN_TX_DL_VALUE: Byte = 32;

  TEST_PHYS_TRANSMISSION: Boolean = true;
  TEST_FUNC_TRANSMISSION: Boolean = true;
  TEST_BURST_TRANSMISSION: Boolean = true;
  MESSAGE_MAX_LENGTH: UInt32 = TCanTpApi.PCANTP_MAX_LENGTH_ISOTP2004 + 1000;
  USE_GETCH: Boolean = false;

// Function called to clean opened ISO-TP channels
Procedure consoleExit();
begin
  TCanTpApi.Uninitialize_2016(cantp_handle.PCANTP_HANDLE_NONEBUS);
end;

// Callback to handle closure of the console window
Function ConsoleCtrlCheck(dwCtrlType: DWORD): Boolean; cdecl;
var
  isClosing: Boolean;
begin
  case dwCtrlType of
    CTRL_CLOSE_EVENT, CTRL_C_EVENT, CTRL_BREAK_EVENT, CTRL_LOGOFF_EVENT, CTRL_SHUTDOWN_EVENT:
      isClosing := true;
  else
    isClosing := false;
  end;
  if isClosing then
    consoleExit();
  result := false;
end;

Function convertToHandle(s: String): cantp_handle;
var
  res: cantp_handle;
begin
  try
    res := cantp_handle(strToInt(s));
  Except
    res := cantp_handle.PCANTP_HANDLE_NONEBUS;
  end;
  result := res;
end;

Function convertToBaudRate(s: String): cantp_baudrate;
var
  res: cantp_baudrate;
begin
  try
    res := cantp_baudrate(strToInt(s));
  Except
    res := cantp_baudrate(0);
  end;
  result := res;
end;

Function convertToBool(s: String): Boolean;
var
  res: Boolean;
begin
  try
    res := Boolean(strToInt(s));
  Except
    res := false;
  end;
  result := res;
end;

/// <summary>Configure mappings to handle a two-way communication and a functional communication (broadcast).</summary>
/// <param name="channel">CAN channel to add </param>
/// <param name="can_msgtype">CAN frame </param>
/// <param name="source_addr">ISO-TP source address</param>
/// <param name="target_addr">ISO-TP target address</param>
/// <param name="extension_addr">ISO-TP extension address</param>
/// <param name="target_addr_func">ISO-TP target address for functional addressing</param>
/// <param name="can_id">CAN ID used to communicate for physical addressing</param>
/// <param name="can_id_flow_ctrl">Flow Control CAN ID used during physical addressing communication</param>
/// <param name="can_id_func">CAN ID used to communicate for functionnal addressing</param>
/// <param name="isotp_msgtype">ISOTP message's type</param>
/// <param name="isotp_format">ISOTP message's format</param>
/// <return>true on successful configuration</return>
Function initialize_mappings(channel: cantp_handle; can_msgtype: cantp_can_msgtype; source_addr: UInt16; target_addr: UInt16; extension_addr: Byte; target_addr_func: UInt16; can_id: UInt32;
  can_id_flow_ctrl: UInt32; can_id_func: UInt32; isotp_msgtype: cantp_isotp_msgtype; isotp_format: cantp_isotp_format): Boolean; overload;
var
  res: Boolean;
  mapping_phys_rx: cantp_mapping;
  mapping_phys_tx: cantp_mapping;
  mapping_func: cantp_mapping;
  status: cantp_status;
begin
  res := true;
  // "FillChar" allows to clean variables (it is common to leave can_tx_dl uninitialized which can lead to invalid 0xCC values)
  FillChar(mapping_phys_rx, sizeof(cantp_mapping), 0);
  FillChar(mapping_phys_tx, sizeof(cantp_mapping), 0);
  FillChar(mapping_func, sizeof(cantp_mapping), 0);

  // configure a mapping to transmit physical message
  mapping_phys_tx.can_id := can_id;
  mapping_phys_tx.can_id_flow_ctrl := can_id_flow_ctrl;
  mapping_phys_tx.can_msgtype := can_msgtype;
  mapping_phys_tx.netaddrinfo.format := isotp_format;
  mapping_phys_tx.netaddrinfo.msgtype := isotp_msgtype;
  mapping_phys_tx.netaddrinfo.target_type := cantp_isotp_addressing.PCANTP_ISOTP_ADDRESSING_PHYSICAL;
  mapping_phys_tx.netaddrinfo.source_addr := source_addr;
  mapping_phys_tx.netaddrinfo.target_addr := target_addr;
  mapping_phys_tx.netaddrinfo.extension_addr := extension_addr;
  status := TCanTpApi.AddMapping_2016(channel, @mapping_phys_tx);
  if not TCanTpApi.StatusIsOk_2016(status) then
  begin
    Writeln(format('Failed to configure mapping (can_id=0x%x) (sts=0x%x)', [mapping_phys_tx.can_id, UInt32(status)]));
    res := false;
  end;
  // configure another mapping to receive physical message (invert source and target in previous mapping)
  mapping_phys_rx := mapping_phys_tx;
  mapping_phys_rx.can_id := mapping_phys_tx.can_id_flow_ctrl;
  mapping_phys_rx.can_id_flow_ctrl := mapping_phys_tx.can_id;
  mapping_phys_rx.netaddrinfo.source_addr := mapping_phys_tx.netaddrinfo.target_addr;
  mapping_phys_rx.netaddrinfo.target_addr := mapping_phys_tx.netaddrinfo.source_addr;
  status := TCanTpApi.AddMapping_2016(channel, @mapping_phys_rx);
  if not TCanTpApi.StatusIsOk_2016(status) then
  begin
    Writeln(format('Failed to configure mapping (can_id=0x%x) (sts=0x%x)', [mapping_phys_rx.can_id, UInt32(status)]));
    res := false;
  end;
  if can_id_func <> $FFFFFFFF then
  begin
    // configure another mapping to transmit functional message
    mapping_func := mapping_phys_tx;
    mapping_func.can_id := can_id_func;
    mapping_func.can_id_flow_ctrl := TCanTpApi.PCANTP_MAPPING_FLOW_CTRL_NONE;
    mapping_func.netaddrinfo.target_type := cantp_isotp_addressing.PCANTP_ISOTP_ADDRESSING_FUNCTIONAL;
    mapping_func.netaddrinfo.target_addr := target_addr_func;
    status := TCanTpApi.AddMapping_2016(channel, @mapping_func);
    if not TCanTpApi.StatusIsOk_2016(status) then
    begin
      Writeln(format('Failed to configure mapping (can_id=0x%x) (sts=0x%x)', [mapping_func.can_id, UInt32(status)]));
      res := false;
    end;
  end;
  result := res;
end;

/// <summary>Initializes mappings compatible with PCTPClient/PCTPServer sample</summary>
/// <param name="channel">CAN channel to add the mappings to.</param>
/// <return>true on successful configuration</return>
Function initialize_mappings(channel: cantp_handle): Boolean; overload;
var
  res: Boolean;
begin
  res := true;

  // 11 bit can ID, normal format addressing, diagnostic message (mandatory)
  res := res and initialize_mappings(channel, cantp_can_msgtype.PCANTP_CAN_MSGTYPE_STANDARD, N_SA, N_TA_PHYS, $00, N_TA_FUNC, $A1, $A2, $A5, cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC,
    cantp_isotp_format.PCANTP_ISOTP_FORMAT_NORMAL);
  // 11 bit can ID, extended format addressing, diagnostic message (mandatory)
  res := res and initialize_mappings(channel, cantp_can_msgtype.PCANTP_CAN_MSGTYPE_STANDARD, N_SA, N_TA_PHYS, $00, N_TA_FUNC, $B1, $B2, $B5, cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC,
    cantp_isotp_format.PCANTP_ISOTP_FORMAT_EXTENDED);
  res := res and initialize_mappings(channel, cantp_can_msgtype.PCANTP_CAN_MSGTYPE_STANDARD, N_SA, N_TA_PHYS, $00, N_TA_FUNC, $C1, $C2, $C5, cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC,
    cantp_isotp_format.PCANTP_ISOTP_FORMAT_EXTENDED);

  // 11 bit can ID, mixed format addressing, remote diagnostic message (mandatory)
  res := res and initialize_mappings(channel, cantp_can_msgtype.PCANTP_CAN_MSGTYPE_STANDARD, N_SA, N_TA_PHYS, N_RA, N_TA_FUNC, $D1, $D2, $D5,
    cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_REMOTE_DIAGNOSTIC, cantp_isotp_format.PCANTP_ISOTP_FORMAT_MIXED);

  // 29 bit can ID, normal format addressing, diagnostic message (mandatory)
  res := res and initialize_mappings(channel, cantp_can_msgtype.PCANTP_CAN_MSGTYPE_EXTENDED, N_SA, N_TA_PHYS, $00, N_TA_FUNC, $A123A1, $A123A2, $A123A5,
    cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, cantp_isotp_format.PCANTP_ISOTP_FORMAT_NORMAL);
  // 29 bit can ID, extended format addressing, diagnostic message
  res := res and initialize_mappings(channel, cantp_can_msgtype.PCANTP_CAN_MSGTYPE_EXTENDED, N_SA, N_TA_PHYS, $00, N_TA_FUNC, $A123C1, $A123C2, $A123C5,
    cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, cantp_isotp_format.PCANTP_ISOTP_FORMAT_EXTENDED);
  res := res and initialize_mappings(channel, cantp_can_msgtype.PCANTP_CAN_MSGTYPE_EXTENDED, N_SA, N_TA_PHYS, $00, N_TA_FUNC, $A123D1, $A123D2, $A123D5,
    cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, cantp_isotp_format.PCANTP_ISOTP_FORMAT_EXTENDED);

  result := res;
end;

/// <summary>Outputs a summary of a received message.</summary>
/// <param name="message">The message.</param>
/// <param name="timestamp">The timestamp of the message.</param>
/// <param name="status">status of the Read function.</param>
/// <returns>1 on success (2 if extra information is available within status), 0 if no message and -1 on error</returns>
Function display_message(const message: Pcantp_msg; timestamp: cantp_timestamp; status: cantp_status): integer;
var
  strLoopback: String;
  strMsgType: String;
  displayedDataLength: UInt32;
  res: integer;
  len: UInt32;
  strNetSta: String;
  strPending: String;
  buffer: String;
  i: UInt32;
begin
  strLoopback := '';
  strMsgType := '';
  displayedDataLength := 0;

  // check status
  if TCanTpApi.StatusIsOk_2016(status) then
    res := 1
  else
    res := -1;
  if res <> 1 then
  begin
    if TCanTpApi.StatusIsOk_2016(status, cantp_status.PCANTP_STATUS_NO_MESSAGE) then
      // no message received ignore
      res := 0
    else
      Writeln(format('Failed to read message (sts=0x%x).', [UInt32(status)]));
    result := res;
    exit;
  end;

  // check bus errors within status
  if not TCanTpApi.StatusIsOk_2016(status, cantp_status.PCANTP_STATUS_OK, true) then
  begin
    Writeln(format('Status returned information/warnings (sts=0x%x).', [UInt32(status)]));
    res := 2;
  end;

  if message.msgdata_any = nil then
  begin
    Writeln('Message null');
    res := -1;
    result := res;
    exit;
  end;

  len := message.msgdata_any^.Length;

  // check if message has the loopback flag
  if (UInt32(message.msgdata_any^.flags) and UInt32(cantp_msgflag.PCANTP_MSGFLAG_LOOPBACK)) <> 0 then
    strLoopback := 'loopback ';

  // format message's type
  Case message.typem of
    cantp_msgtype.PCANTP_MSGTYPE_CAN:
      strMsgType := 'PCANTP_MSGTYPE_CAN';
    cantp_msgtype.PCANTP_MSGTYPE_CANFD:
      strMsgType := 'PCANTP_MSGTYPE_CANFD';
    cantp_msgtype.PCANTP_MSGTYPE_ISOTP:
      strMsgType := 'PCANTP_MSGTYPE_ISOTP';
    cantp_msgtype.PCANTP_MSGTYPE_NONE:
      strMsgType := 'PCANTP_MSGTYPE_NONE';
  else
    strMsgType := 'Unknown';
  end;

  // Only display data if network result is OK
  if message.msgdata_any^.netstatus = cantp_netstatus.PCANTP_NETSTATUS_OK then
    displayedDataLength := len;

  // Display generic information
  strNetSta := 'OK !';
  if message.msgdata_any^.netstatus <> cantp_netstatus.PCANTP_NETSTATUS_OK then
    strNetSta := 'ERROR !!!';
  Writeln(format(sLineBreak + '%d - Received %s%s message : canid=0x%x, length=%d - result: 0x%x - %s', [timestamp, strLoopback, strMsgType, message.can_info.can_id, len,
    UInt32(message.msgdata_any^.netstatus), strNetSta]));

  if message.typem = cantp_msgtype.PCANTP_MSGTYPE_ISOTP then
  begin
    strPending := 'Completed';
    // Limit displayed data if message is pending
    if (displayedDataLength > 0) and (((UInt32(message.msgdata_isotp^.netaddrinfo.msgtype) and UInt32(cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_FLAG_INDICATION)) <> 0)) then
    begin
      if len > 7 then
        displayedDataLength := 7
      else
        displayedDataLength := len;
      strPending := '/!\ Pending';
    end;
    // Display iso-tp message's information
    Writeln(#9 + format('%s message from 0x%x (to 0x%x, with RA 0x%x)', [strPending, message.msgdata_isotp^.netaddrinfo.source_addr, message.msgdata_isotp^.netaddrinfo.target_addr,
      message.msgdata_isotp^.netaddrinfo.extension_addr]));
  end;

  if displayedDataLength > 0 then
  begin
    buffer := #9;
    // display data
    buffer := buffer + format('\-> Length: %d, Data= ', [len]);
    for i := 0 to displayedDataLength - 1 do
      buffer := buffer + format('%x ', [PByte(message.msgdata_any^.data)[i]]);
    if displayedDataLength <> len then
      buffer := buffer + '...';
    Writeln(buffer);
  end;
  result := res;
end;

/// <summary>Reads a message.</summary>
/// <param name="channel">The channel to read the message from.</param>
/// <param name="nbMsgRead">Buffer to store the number of message actually read.</param>
/// <returns>number of errors</returns>
Function read_segmented_message(channel: cantp_handle; var nbMsgRead: integer): integer;
var
  message: cantp_msg;
  status: cantp_status;
  timestamp: cantp_timestamp;
  doLoop: integer;
  nbErr: integer;
begin
  nbErr := 0;
  repeat
  begin
    doLoop := 0;
    // initialize message
    status := TCanTpApi.MsgDataAlloc_2016(message, cantp_msgtype.PCANTP_MSGTYPE_NONE);
    if not TCanTpApi.StatusIsOk_2016(status) then
    begin
      Writeln(format('Failed to allocate message (sts=0x%x).', [UInt32(status)]));
      nbErr := nbErr + 1;
      continue;
    end;
    // retrieve any message from Rx queue
    status := TCanTpApi.Read_2016(channel, message, @timestamp);
    // display_message returns true when a message is available
    doLoop := display_message(@message, timestamp, status);
    if doLoop > 0 then
    begin
      // update error counter if a message was received but a network error occured
      if message.msgdata_any^.netstatus <> cantp_netstatus.PCANTP_NETSTATUS_OK then
        nbErr := nbErr + 1;
      if message.typem = cantp_msgtype.PCANTP_MSGTYPE_ISOTP then
        if UInt32(message.msgdata_isotp^.netaddrinfo.msgtype) and UInt32(cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_FLAG_INDICATION) <> 0 then
          // wait and read the full segmented message
          nbErr := nbErr + read_segmented_message(channel, nbMsgRead)
        else
          nbMsgRead := nbMsgRead + 1
      else
        nbMsgRead := nbMsgRead + 1;
    end
    else
    begin
      if doLoop = 0 then
        // no message read, wait for it
        Sleep(10);
    end;

    // release message
    status := TCanTpApi.MsgDataFree_2016(message);
    if not TCanTpApi.StatusIsOk_2016(status) then
      Writeln(format('Failed to deallocate message (sts=0x%x).', [UInt32(status)]));

  end
  until doLoop <> 0;

  result := nbErr;
end;

/// <summary>A function to transmit CAN-TP messages</summary>
/// <param name="channel">channel</param>
/// <param name="canIdType">canIdType</param>
/// <param name="msgType">CAN-TP message Type (PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC)</param>
/// <param name="formatType">Format addressing type (see PCANTP_ISOTP_FORMAT_xxx)</param>
/// <returns>number of transmission errors</returns>
Function transmit(channel: cantp_handle; canIdType: cantp_can_msgtype; msgtype: cantp_isotp_msgtype; formatType: cantp_isotp_format; doCanFd: Boolean; canFdInitialized: Boolean): integer;
var
  status: cantp_status;
  message: cantp_msg;
  can_msgtype: cantp_can_msgtype;
  isotp_nai: cantp_netaddrinfo;
  lMaxSize: Byte;
  nbErrPhys: integer;
  nbErrFunc: integer;
  sleepTime: integer;
  nbMsgRead: integer;
  lStr: String;
  strCanIdType: String;
  strMsgType: String;
  data: array of Byte;
  i: integer;
  index: UInt32;
  nbMsgSent: integer;
  Dummy: char;

begin

  nbErrPhys := 0;
  nbErrFunc := 0;
  sleepTime := 0;
  nbMsgRead := 0;

  // Output information message
  lStr := '??';
  Case formatType of
    cantp_isotp_format.PCANTP_ISOTP_FORMAT_ENHANCED:
      lStr := 'Enhanced';
    cantp_isotp_format.PCANTP_ISOTP_FORMAT_EXTENDED:
      lStr := 'Extended';
    cantp_isotp_format.PCANTP_ISOTP_FORMAT_FIXED_NORMAL:
      lStr := 'Fixed Normal';
    cantp_isotp_format.PCANTP_ISOTP_FORMAT_MIXED:
      lStr := 'Mixed';
    cantp_isotp_format.PCANTP_ISOTP_FORMAT_NORMAL:
      lStr := 'Normal';
  end;

  Writeln(sLineBreak);
  if (UInt32(canIdType) and UInt32(cantp_can_msgtype.PCANTP_CAN_MSGTYPE_EXTENDED)) <> 0 then
    strCanIdType := '29 BIT CAN ID'
  else
    strCanIdType := '11 BIT CAN ID';
  if msgtype = cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC then
    strMsgType := 'Diagnostic'
  else
    strMsgType := 'Remote diagnostic';
  Writeln(format('Transmission of %s, %s message, %s addressing format', [strCanIdType, strMsgType, lStr]));
  if USE_GETCH then
  begin
    Writeln(sLineBreak + 'Press <Enter> to continue...');
    Readln(Dummy);
  end;

  // initialize message
  FillChar(message, sizeof(message), 0);
  FillChar(isotp_nai, sizeof(isotp_nai), 0);
  isotp_nai.source_addr := N_SA;
  if msgtype = cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_REMOTE_DIAGNOSTIC then
    isotp_nai.extension_addr := N_RA;
  can_msgtype := canIdType;
  if doCanFd then
  begin
    can_msgtype := cantp_can_msgtype(UInt32(can_msgtype) or UInt32(cantp_can_msgtype.PCANTP_CAN_MSGTYPE_FD));
    can_msgtype := cantp_can_msgtype(UInt32(can_msgtype) or UInt32(cantp_can_msgtype.PCANTP_CAN_MSGTYPE_BRS));
  end;
  isotp_nai.msgtype := msgtype;
  isotp_nai.format := formatType;

  // prepare a buffer that will be used as a reference to initialize the data to transmit
  setLength(data, MESSAGE_MAX_LENGTH);
  for i := 0 to MESSAGE_MAX_LENGTH - 1 do
    data[i] := i;

  if TEST_PHYS_TRANSMISSION then
  begin
    // Transmitting data using Physical addressing
    Writeln(sLineBreak + 'Transmitting data using Physical addressing' + sLineBreak);
    isotp_nai.target_addr := N_TA_PHYS;
    isotp_nai.target_type := cantp_isotp_addressing.PCANTP_ISOTP_ADDRESSING_PHYSICAL;
    // each loop transmits a message with a data length incremented by 1 and reads its loppbacl confirmation
    for index := 1 to MESSAGE_MAX_LENGTH do
      // if index is 100, skip and jump to maximum data length
      if (index <= 100) or (index = MESSAGE_MAX_LENGTH) then
      begin
        // allocate a new ISOTP message
        status := TCanTpApi.MsgDataAlloc_2016(message, cantp_msgtype.PCANTP_MSGTYPE_ISOTP);
        if TCanTpApi.StatusIsOk_2016(status) then
        begin
          // initialize ISOTP message
          status := TCanTpApi.MsgDataInit_2016(message, TCanTpApi.PCANTP_CAN_ID_DEFINED_BY_NAI, can_msgtype, index, PByte(data), @isotp_nai);
          if TCanTpApi.StatusIsOk_2016(status) then
          begin
            // write message
            status := TCanTpApi.Write_2016(channel, message);
            if TCanTpApi.StatusIsOk_2016(status) then
            begin
              Writeln(sLineBreak + format('Successfully queued ISOTP message: Length %d (sts=0x%x).', [index, UInt32(status)]));
              // try to read loopback confirmation
              nbMsgRead := 0;
              nbErrPhys := nbErrPhys + read_segmented_message(channel, nbMsgRead);
              if nbMsgRead <> 1 then
                Writeln(format('Received unexpected messages (%d instead of 1)', [nbMsgRead]));
            end
            else
            begin
              Writeln(format(sLineBreak + 'Failed to write ISOTP message: Length %d (sts=0x%x).', [index, UInt32(status)]));
              nbErrPhys := nbErrPhys + 1;
            end;
          end
          else
          begin
            Writeln(format(sLineBreak + 'Failed to initialize ISOTP message: Length %d (sts=0x%x).', [index, UInt32(status)]));
            nbErrPhys := nbErrPhys + 1;
          end;
          // release message
          status := TCanTpApi.MsgDataFree_2016(message);
          if not TCanTpApi.StatusIsOk_2016(status) then
          begin
            Writeln(format('Failed to deallocate message (sts=0x%x).', [UInt32(status)]));
            nbErrPhys := nbErrPhys + 1;
          end;
        end
        else
        begin
          Writeln(format('Failed to allocate message (sts=0x%x).', [UInt32(status)]));
          nbErrPhys := nbErrPhys + 1;
        end;

        Sleep(sleepTime);

      end;
  end;

  if TEST_BURST_TRANSMISSION then
  begin
    Writeln(sLineBreak + 'Transmitting data using Physical addressing in "BURST" mode' + sLineBreak);
    isotp_nai.target_addr := N_TA_PHYS;
    isotp_nai.target_type := cantp_isotp_addressing.PCANTP_ISOTP_ADDRESSING_PHYSICAL;
    // Transmitting data using Physical addressing WITHOUT waiting for response confirmation
    // each loop transmits a message with a data length incremented by 1
    nbMsgSent := 0;
    for index := 1 to MESSAGE_MAX_LENGTH do
      // if index is 10, skip and try bigger LEN
      if (index <= 10) or (index >= 2001) then
      begin
        // allocate a new ISOTP message
        status := TCanTpApi.MsgDataAlloc_2016(message, cantp_msgtype.PCANTP_MSGTYPE_ISOTP);
        if TCanTpApi.StatusIsOk_2016(status) then
        begin
          // initialize ISOTP message
          status := TCanTpApi.MsgDataInit_2016(message, TCanTpApi.PCANTP_CAN_ID_DEFINED_BY_NAI, can_msgtype, index, PByte(data), @isotp_nai);
          if TCanTpApi.StatusIsOk_2016(status) then
          begin
            // write message
            status := TCanTpApi.Write_2016(channel, message);
            if TCanTpApi.StatusIsOk_2016(status) then
            begin
              nbMsgSent := nbMsgSent + 1;
              Writeln(format(sLineBreak + 'Successfully queued ISOTP message: Length %d (sts=0x%x).', [index, UInt32(status)]));
            end
            else
            begin
              Writeln(format(sLineBreak + 'Failed to write ISOTP message: Length %d (sts=0x%x).', [index, UInt32(status)]));
              nbErrPhys := nbErrPhys + 1;
            end;
          end
          else
          begin
            Writeln(format(sLineBreak + 'Failed to initialize ISOTP message: Length %d (sts=0x%x).', [index, UInt32(status)]));
            nbErrPhys := nbErrPhys + 1;
          end;
          // release message
          status := TCanTpApi.MsgDataFree_2016(message);
          if not TCanTpApi.StatusIsOk_2016(status) then
          begin
            Writeln(format('Failed to deallocate message (sts=0x%x).', [UInt32(status)]));
            nbErrPhys := nbErrPhys + 1;
          end;
        end
        else
        begin
          Writeln(format('Failed to allocate message (sts=0x%x).', [UInt32(status)]));
          nbErrPhys := nbErrPhys + 1;
        end;

        if index = 2005 then
          // skip
          break;

      end;
    Writeln(sLineBreak + 'Reading confirmation for each transmitted messages in "BURST" mode' + sLineBreak);
    nbMsgRead := 0;
    while nbMsgRead < nbMsgSent do
      // try to read loopback confirmation
      nbErrPhys := nbErrPhys + read_segmented_message(channel, nbMsgRead);
    TCanTpApi.Reset_2016(channel);
  end;

  if TEST_FUNC_TRANSMISSION then
  begin
    // Transmitting data using Functional addressing
    Writeln(sLineBreak + 'Transmitting data using Functional addressing' + sLineBreak);
    isotp_nai.target_addr := N_TA_FUNC;
    isotp_nai.target_type := cantp_isotp_addressing.PCANTP_ISOTP_ADDRESSING_FUNCTIONAL;
    // Reminder: Functional addressing shall only be supported
    // for Single Frame communication.
    // Thus maximum size depends on Format Addressing Type.
    Case formatType of
      cantp_isotp_format.PCANTP_ISOTP_FORMAT_EXTENDED, cantp_isotp_format.PCANTP_ISOTP_FORMAT_MIXED:
        begin
          if (doCanFd and canFdInitialized) then
            lMaxSize := CAN_TX_DL_VALUE - 2
          else
            lMaxSize := TCanTpApi.PCANTP_MAX_LENGTH_CAN_STANDARD - 2;
        end;
    else
      begin
        if (doCanFd and canFdInitialized) then
          lMaxSize := CAN_TX_DL_VALUE - 1
        else
          lMaxSize := TCanTpApi.PCANTP_MAX_LENGTH_CAN_STANDARD - 1;
      end;
    end;

    // each loop transmits a message with a data length incremented by 1 and reads its loppbacl confirmation
    for index := 1 to lMaxSize do
    begin
      // allocate a new ISOTP message
      status := TCanTpApi.MsgDataAlloc_2016(message, cantp_msgtype.PCANTP_MSGTYPE_ISOTP);
      if TCanTpApi.StatusIsOk_2016(status) then
      begin
        // initialize ISOTP message
        status := TCanTpApi.MsgDataInit_2016(message, TCanTpApi.PCANTP_CAN_ID_DEFINED_BY_NAI, can_msgtype, index, PByte(data), @isotp_nai);
        if TCanTpApi.StatusIsOk_2016(status) then
        begin
          // write message
          status := TCanTpApi.Write_2016(channel, message);
          if TCanTpApi.StatusIsOk_2016(status) then
          begin
            Writeln(format(sLineBreak + 'Successfully queued ISOTP message: Length %d (sts=0x%x).', [index, UInt32(status)]));
            // try to read loopback confirmation
            nbMsgRead := 0;
            nbErrFunc := nbErrFunc + read_segmented_message(channel, nbMsgRead);
            if nbMsgRead <> 1 then
              Writeln(format('Received unexpected messages (%d instead of 1)', [nbMsgRead]));
          end
          else
          begin
            Writeln(format(sLineBreak + 'Failed to write ISOTP message: Length %d (sts=0x%x).', [index, UInt32(status)]));
            nbErrFunc := nbErrFunc + 1;
          end;
        end
        else
        begin
          Writeln(format(sLineBreak + 'Failed to initialize ISOTP message: Length %d (sts=0x%x).', [index, UInt32(status)]));
          nbErrFunc := nbErrFunc + 1;
        end;
        // release message
        status := TCanTpApi.MsgDataFree_2016(message);
        if not TCanTpApi.StatusIsOk_2016(status) then
        begin
          Writeln(format('Failed to deallocate message (sts=0x%x).', [UInt32(status)]));
          nbErrFunc := nbErrFunc + 1;
        end;
      end
      else
      begin
        Writeln(format('Failed to allocate message (sts=0x%x).', [UInt32(status)]));
        nbErrFunc := nbErrFunc + 1;
      end;
      Sleep(sleepTime);
    end;
  end;

  if (nbErrPhys > 0) or (nbErrFunc > 0) then
  begin
    if nbErrPhys > 0 then
      Writeln(format(sLineBreak + 'ERROR : %d errors occured.' + sLineBreak, [nbErrPhys]));
    if nbErrFunc > 0 then
      Writeln(format(sLineBreak + 'ERROR : %d errors occured.' + sLineBreak, [nbErrFunc]));
    if USE_GETCH then
    begin
      Writeln(sLineBreak + 'Press <Enter> to continue...');
      Readln(Dummy);
    end;
  end
  else
    Writeln(sLineBreak + 'Transmissions successful !' + sLineBreak);

  result := nbErrFunc + nbErrPhys;
end;

var
  status: cantp_status;
  buffer: array [0 .. 499] of AnsiChar;
  channel: cantp_handle;
  useCanFd: Boolean;
  doCanFd: Boolean;
  baudrate: cantp_baudrate;
  bitrateFd: cantp_bitrate;
  nbErr: integer;
  param: Byte;
  Dummy: char;
  i: integer;

/// <summary>Main entry-point for this application.</summary>

begin
  try

    useCanFd := false;
    bitrateFd := '';

    Writeln(sLineBreak + 'Usage: _10_client_isotp.exe [TPCANTPHandle] [USE_CAN_FD:0|1] [BTR0BTR1|BITRATE]');
    Writeln(' * CAN: _10_client_isotp.exe 0x51 0 0x1C ');
    Writeln(' * CAN FD: _10_client_isotp.exe 0x51 1 ''f_clock=80000000, nom_brp=10, nom_tseg1=12, nom_tseg2=3, nom_sjw=1, data_brp=4, data_tseg1=7, data_tseg2=2, data_sjw=1''');
    Writeln(sLineBreak + '---------------------------------------------');

    // clear gracefully ISO-TP if console window is killed
    SetConsoleCtrlHandler(@ConsoleCtrlCheck, true);

    // Show version information
    status := TCanTpApi.GetValue_2016(cantp_handle.PCANTP_HANDLE_NONEBUS, cantp_parameter.PCANTP_PARAMETER_API_VERSION, buffer, 500);
    if TCanTpApi.StatusIsOk_2016(status) then
      Writeln('PCAN-ISO-TP API Version : ', buffer)
    else
      Writeln(format('Error failed to get PCAN-ISO-TP API Version (sts=0x%x)', [UInt32(status)]));

    // 1st argument is the PCAN-channel to use (PCAN-USB channel 1)
    if (ParamCount >= 1) then
      channel := convertToHandle(ParamStr(1))
    else
      channel := cantp_handle.PCANTP_HANDLE_USBBUS1;
    // 2nd argument is CAN FD
    if (ParamCount >= 2) then
      useCanFd := convertToBool(ParamStr(2))
    else
      useCanFd := USE_CAN_FD;
    // 3rd argument is bitrate (either CAN or CAN FD)
    baudrate := cantp_baudrate.PCANTP_BAUDRATE_500K;
    bitrateFd := 'f_clock=80000000, nom_brp=10, nom_tseg1=12, nom_tseg2=3, nom_sjw=1, data_brp=4, data_tseg1=7, data_tseg2=2, data_sjw=1';
    if (ParamCount >= 3) then
    begin
      if useCanFd then
        bitrateFd := cantp_bitrate(AnsiString(ParamStr(3)))
      else
        baudrate := convertToBaudRate(ParamStr(3));
    end;

    // Initializing of the ISO-TP Communication session
    Writeln(format('Connecting to channel 0x%x...', [UInt32(channel)]));
    if not useCanFd then
    begin
      Writeln(format(' * btr0btr1 = 0x%x...', [UInt32(baudrate)]));
      status := TCanTpApi.Initialize_2016(channel, baudrate);
    end
    else
    begin
      Writeln(format(' * bitrateFd = "%s"...', [bitrateFd]));
      status := TCanTpApi.InitializeFD_2016(channel, bitrateFd);
    end;
    Writeln(format(' -> Initialize CANTP: 0x%x', [UInt32(status)]));

    // will start the reading loop only if successfully initialized
    if not TCanTpApi.StatusIsOk_2016(status) then
    begin
      Writeln(' -> Initialization failed, exiting...');
      TCanTpApi.Uninitialize_2016(channel);
      Writeln(sLineBreak + sLineBreak + 'Press <Enter> to quit...');
      Readln(Dummy);
      exit;
    end;

    // configure the test session to use can data padding with a specific padding value
    param := TCanTpApi.PCANTP_CAN_DATA_PADDING_ON;
    status := TCanTpApi.SetValue_2016(channel, cantp_parameter.PCANTP_PARAMETER_CAN_DATA_PADDING, PByte(@param), 1);
    if not TCanTpApi.StatusIsOk_2016(status) then
      Writeln(format('Failed to configure parameter "PCANTP_PARAMETER_CAN_DATA_PADDING" (sts=0x%x).', [UInt32(status)]));

    param := $99;
    status := TCanTpApi.SetValue_2016(channel, cantp_parameter.PCANTP_PARAMETER_CAN_PADDING_VALUE, PByte(@param), 1);
    if not TCanTpApi.StatusIsOk_2016(status) then
      Writeln(format('Failed to configure parameter "PCANTP_PARAMETER_CAN_PADDING_VALUE" (sts=0x%x).', [UInt32(status)]));

    // set the default TX_DL size (i.e. the default CAN-FD DLC)
    param := CAN_TX_DL_CODE;
    status := TCanTpApi.SetValue_2016(channel, cantp_parameter.PCANTP_PARAMETER_CAN_TX_DL, PByte(@param), 1);
    if not TCanTpApi.StatusIsOk_2016(status) then
      Writeln(format('Failed to configure parameter "PCANTP_PARAMETER_CAN_TX_DL" (sts=0x%x).', [UInt32(status)]));

    // configure server - enable enhanced addressing to be compatible with PCTPClient
    param := TCanTpApi.PCANTP_VALUE_PARAMETER_ON;
    status := TCanTpApi.SetValue_2016(channel, cantp_parameter.PCANTP_PARAMETER_SUPPORT_29B_ENHANCED, PByte(@param), 1);
    if not TCanTpApi.StatusIsOk_2016(status) then
      Writeln(format('Failed to configure parameter "PCANTP_PARAMETER_SUPPORT_29B_ENHANCED" (sts=0x%x).', [UInt32(status)]));

    // log CAN frames
    param := TCanTpApi.PCANTP_DEBUG_WARNING;
    status := TCanTpApi.SetValue_2016(channel, cantp_parameter.PCANTP_PARAMETER_DEBUG, PByte(@param), 1);
    if not TCanTpApi.StatusIsOk_2016(status) then
      Writeln(format('Failed to configure parameter "PCANTP_PARAMETER_DEBUG" (sts=0x%x).', [UInt32(status)]));

    // configure ISO-TP mappings
    if not initialize_mappings(channel) then
    begin
      Writeln(' -> Mappings configuration failed, exiting...');
      TCanTpApi.Uninitialize_2016(channel);
      Writeln(sLineBreak + sLineBreak + 'Press <Enter> to quit...');
      Readln(Dummy);
      exit;
    end;

    doCanFd := false;
    nbErr := 0;
    for i := 0 to 1 do
    begin
      case i of
        0:
          // first loop will test CAN-FD frames even if it was not initialized with Fd
          doCanFd := true;
        1:
          // second loop tests standard CAN frames
          doCanFd := false;
      end;

      // Transmit with 29 Bit CAN identifiers (and no CAN ID mapping is required)
      nbErr := nbErr + transmit(channel, cantp_can_msgtype.PCANTP_CAN_MSGTYPE_EXTENDED, cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, cantp_isotp_format.PCANTP_ISOTP_FORMAT_FIXED_NORMAL,
        doCanFd, useCanFd);
      nbErr := nbErr + transmit(channel, cantp_can_msgtype.PCANTP_CAN_MSGTYPE_EXTENDED, cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_REMOTE_DIAGNOSTIC, cantp_isotp_format.PCANTP_ISOTP_FORMAT_MIXED,
        doCanFd, useCanFd);

      // Transmit with enhanced diagnostics 29 bit CAN identifiers (ISO-15765-3 8.3)
      nbErr := nbErr + transmit(channel, cantp_can_msgtype.PCANTP_CAN_MSGTYPE_EXTENDED, cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, cantp_isotp_format.PCANTP_ISOTP_FORMAT_ENHANCED, doCanFd,
        useCanFd);

      // Transmit with 11 Bit CAN identifiers
      // those transmissions require a Mapping between CAN ID and Network Addressing Information
      // for each transmission, 3 mappings are added :
      // - 2 for physical addressing, one defining SA->TA and the other TA->SA
      // - 1 for functional addressing defining SA->TA
      //

      // 11 bit can ID, normal format addressing (diagnostic message mandatory)
      nbErr := nbErr + transmit(channel, cantp_can_msgtype.PCANTP_CAN_MSGTYPE_STANDARD, cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, cantp_isotp_format.PCANTP_ISOTP_FORMAT_NORMAL, doCanFd,
        useCanFd);
      // 11 bit can ID, extended format addressing (diagnostic message mandatory)
      nbErr := nbErr + transmit(channel, cantp_can_msgtype.PCANTP_CAN_MSGTYPE_STANDARD, cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, cantp_isotp_format.PCANTP_ISOTP_FORMAT_EXTENDED, doCanFd,
        useCanFd);
      // 11 bit can ID, mixed format addressing (remote diagnostic message mandatory)
      nbErr := nbErr + transmit(channel, cantp_can_msgtype.PCANTP_CAN_MSGTYPE_STANDARD, cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_REMOTE_DIAGNOSTIC, cantp_isotp_format.PCANTP_ISOTP_FORMAT_MIXED,
        doCanFd, useCanFd);

      // Transmit with 29 Bit CAN identifiers
      // those transmissions require a Mapping between CAN ID and Network Addressing Information
      // for each transmission, 3 mappings are added 2 for physical addressing and
      // the other for functional addressing.
      //

      // 29 bit can ID, normal format addressing (diagnostic message mandatory)
      nbErr := nbErr + transmit(channel, cantp_can_msgtype.PCANTP_CAN_MSGTYPE_EXTENDED, cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, cantp_isotp_format.PCANTP_ISOTP_FORMAT_NORMAL, doCanFd,
        useCanFd);
      // 29 bit can ID, extended format addressing (diagnostic message mandatory)
      nbErr := nbErr + transmit(channel, cantp_can_msgtype.PCANTP_CAN_MSGTYPE_EXTENDED, cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, cantp_isotp_format.PCANTP_ISOTP_FORMAT_EXTENDED, doCanFd,
        useCanFd);
    end;

    // release CAN-TP
    Writeln;
    if nbErr > 0 then
    begin
      Writeln(format('ERROR : a total of %d errors occured.', [nbErr]));
      Writeln(sLineBreak + sLineBreak + sLineBreak + 'Press <Enter> to quit...');
      Readln(Dummy);
    end
    else
      Writeln('ALL Transmissions succeeded !');

    TCanTpApi.Uninitialize_2016(channel);
  except
    on E: Exception do
      // clear gracefully ISO-TP
      consoleExit();
  end;

end.
