import string
import os
from PCAN_ISO_TP_2016 import *

IS_WINDOWS = platform.system() == 'Windows'

# Support events
if not IS_WINDOWS: 
	import select
	__LIBC_OBJ = cdll.LoadLibrary("libc.so.6")
    
WAIT_OBJECT_0 = 0
def NULL_HANDLE():
	return c_void_p(0) if IS_WINDOWS else c_int(0)

# Support keyboard
if IS_WINDOWS: 
	from msvcrt import getch, kbhit
	def install_keyboardHit():
		return 0
	def desinstall_keyboardHit(old_settings):
		pass
	def keyboardHit():
		return kbhit() != 0
	def getInput():
		key = getch()
		try:
			key_decode = key.decode('utf-8')
			return key_decode
		except UnicodeDecodeError:
			return key
else:
	import tty
	import termios
	def install_keyboardHit():
		old_settings = termios.tcgetattr(sys.stdin)
		tty.setcbreak(sys.stdin.fileno())
		return old_settings
	def desinstall_keyboardHit(old_settings):
		termios.tcsetattr(sys.stdin, termios.TCSADRAIN, old_settings)
	def keyboardHit():
		return select.select([sys.stdin], [], [], 0) == ([sys.stdin], [], [])
	def getInput():
		return sys.stdin.read(1)

# Isotp library
objPCANIsotp = PCAN_ISO_TP_2016()

# Help functions
def OK_KO(test):
    return "OK" if test else "KO" 

def STATUS_OK_KO(test):
    return OK_KO(objPCANIsotp.StatusIsOk_2016(test))

def print_data(buf, length):
	"""
	Helper: print memory

	parameters:
	 buf: Pointer on the memory to print
	 length: Number of byte to print
	"""	
	s = "\t Byte:"
	for i in range(length):
		s += "0x{x1} ".format(x1=format(buf[i], '02X'))
	print(s)
	s = "\t Ascii: \"";
	for i in range(length):
		if buf[i] != 0:
			s += "{x1}".format(x1=chr(buf[i]))
	s += "\""
	print(s)

# Definitions
STMIN_600US = 0xF6

#
# Main entry point of the program, start a small CAN ISO TP read/write example, server-side (reading)
#

try:
	old_settings = install_keyboardHit()

	# Initialize variables
	server_handle = PCANTP_HANDLE_USBBUS2	# TODO: modify the value according to your available PCAN devices.

	# Print version information
	buffer = create_string_buffer(500)
	objPCANIsotp.GetValue_2016(PCANTP_HANDLE_NONEBUS, PCANTP_PARAMETER_API_VERSION, buffer, 500)
	print("PCAN-ISO-TP API Version: ", buffer.value)

	# Initialize channel: CAN2.0 - 500Kbit/s
	res = objPCANIsotp.Initialize_2016(server_handle, PCANTP_BAUDRATE_500K, 0, 0, 0)
	print("Initialize: ", STATUS_OK_KO(res))

	#Create a receive event on receiver
	if IS_WINDOWS:
		receive_event = c_void_p(windll.kernel32.CreateEventA(None, 0,0,None))
		res = objPCANIsotp.SetValue_2016(server_handle, PCANTP_PARAMETER_RECEIVE_EVENT, receive_event, sizeof(receive_event))
		print("Create a receive event: ", STATUS_OK_KO(res))
	else:
		receive_event = NULL_HANDLE()
		res = objPCANIsotp.GetValue_2016(server_handle, PCANTP_PARAMETER_RECEIVE_EVENT, receive_event, sizeof(receive_event))
		print("Get receive event: ", STATUS_OK_KO(res))

	# Change STmin value to 600us
	STmin = c_uint32(STMIN_600US)
	res = objPCANIsotp.SetValue_2016(server_handle, PCANTP_PARAMETER_SEPARATION_TIME, STmin, sizeof(STmin))
	print("Set STMIN = 600us: ", STATUS_OK_KO(res))

	# Create a simple physical mapping: 
	#    - Source 0xF1 (client), target 0x01 (server), CAN id 0xA1, CAN ID flow control 0xA2
	#    - Diagnostic message in a classic format
	mapping = cantp_mapping()
	mapping.can_id = 0xA1
	mapping.can_id_flow_ctrl = 0xA2
	mapping.can_msgtype = PCANTP_CAN_MSGTYPE_STANDARD
	mapping.netaddrinfo.extension_addr = 0x00
	mapping.netaddrinfo.format = PCANTP_ISOTP_FORMAT_NORMAL
	mapping.netaddrinfo.msgtype = PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC
	mapping.netaddrinfo.source_addr = 0xF1
	mapping.netaddrinfo.target_addr = 0x01
	mapping.netaddrinfo.target_type = PCANTP_ISOTP_ADDRESSING_PHYSICAL

	# Add mapping on channel
	res = objPCANIsotp.AddMapping_2016(server_handle, mapping)
	print("Add a simple mapping: ", STATUS_OK_KO(res))
	# NOTE: this is a one-way mapping, the server is only capable of 
	#	receiving segmented or unsegmented ISO-TP message using "0xA1" CAN ID
	#	to be able to send ISO-TP message, another mapping is required.

	# Read while user do not press Q
	print("Start listening, press Q to quit.")
	rx_msg = cantp_msg()
	stop = False
	while not stop:

		# Wait a receive event on receiver
		#	note: timeout is used to check keyboard hit.
		if IS_WINDOWS:
			windows_wait_result = c_uint64(windll.kernel32.WaitForSingleObject(receive_event, 1000))
			wait_result = True if windows_wait_result.value == WAIT_OBJECT_0 else False
		else:
			readable, _, _ = select.select([receive_event.value], [], [], 1)
			wait_result = True if len(readable) > 0 else False
	
		# If we receive something, read the message
		if wait_result:
			read_status = PCANTP_STATUS_OK
			while read_status.value != PCANTP_STATUS_NO_MESSAGE.value:

				# Allocate rx message
				res = objPCANIsotp.MsgDataAlloc_2016(rx_msg, PCANTP_MSGTYPE_NONE)
				print("Allocate rx message: ", STATUS_OK_KO(res))

				# Read first available message (no filtering based on message's type is set): 
				read_status = objPCANIsotp.Read_2016(server_handle, rx_msg, None, PCANTP_MSGTYPE_NONE)
				print("Try to read a message: ", STATUS_OK_KO(read_status))

				# If we read something, print the message
				if objPCANIsotp.StatusIsOk_2016(read_status, PCANTP_STATUS_OK, False):
					print("\nPrint received data:")
					print_data(rx_msg.msgdata.any.contents.data, rx_msg.msgdata.any.contents.length)
					print("")
			
				# Free message
				res = objPCANIsotp.MsgDataFree_2016(rx_msg)
				print("Free rx message: ", STATUS_OK_KO(res))


		# Quit when user press Q
		if keyboardHit():
			keyboard_res = getInput()
			if keyboard_res == 'Q' or keyboard_res == 'q':
				stop = True

	# Close receive event
	if IS_WINDOWS:
		res = objPCANIsotp.SetValue_2016(server_handle, PCANTP_PARAMETER_RECEIVE_EVENT, NULL_HANDLE(), sizeof(NULL_HANDLE()))
		print("Stop receive event: ", STATUS_OK_KO(res))
		res_b = windll.kernel32.CloseHandle(receive_event)
		print("Close receive event: ", OK_KO(res_b))
	
	# Uninitialize
	res = objPCANIsotp.Uninitialize_2016(server_handle)
	print("Uninitialize: ", STATUS_OK_KO(res))

	# Exit
	print("Press <Enter> to close")
	getInput()

finally:
	desinstall_keyboardHit(old_settings)