import string
import os
import sys
from time import sleep
import copy
import io
import atexit
from PCAN_ISO_TP_2016 import *

IS_WINDOWS = platform.system() == 'Windows'

# Support events
if not IS_WINDOWS: 
	import select
	__LIBC_OBJ = cdll.LoadLibrary("libc.so.6")
	import signal

WAIT_OBJECT_0 = 0
def NULL_HANDLE():
	return c_void_p(0) if IS_WINDOWS else c_int(0)

# Support keyboard
if IS_WINDOWS: 
	from msvcrt import getch, kbhit
	def install_keyboardHit():
		return 0
	def desinstall_keyboardHit(old_settings):
		pass
	def keyboardHit():
		return kbhit() != 0
	def getInput():
		key = getch()
		try:
			key_decode = key.decode('utf-8')
			return key_decode
		except UnicodeDecodeError:
			return key
else:
	import tty
	import termios
	def install_keyboardHit():
		old_settings = termios.tcgetattr(sys.stdin)
		tty.setcbreak(sys.stdin.fileno())
		return old_settings
	def desinstall_keyboardHit(old_settings):
		termios.tcsetattr(sys.stdin, termios.TCSADRAIN, old_settings)
	def keyboardHit():
		return select.select([sys.stdin], [], [], 0) == ([sys.stdin], [], [])
	def getInput():
		return sys.stdin.read(1)

# Isotp library
objPCANIsotp = PCAN_ISO_TP_2016()

# Help functions
def strtoi(str):
	"""
	Convert string to int

	parameters:
	 str : string to convert
	returns:
	 int value
	"""
	base = 10
	if str.startswith("0x") or str.startswith("0X"):
		base = 16
	elif str.startswith("0"):
		base = 8
	return int(str, base)

def initialize_mappings(channel, can_msgtype, source_addr, target_addr, extension_addr, target_addr_func, can_id, can_id_flow_ctrl, can_id_func, isotp_msgtype, isotp_format):
	"""
	Configure mappings to handle a two-way communication and a functional communication (broadcast).

	parameters:
		channel: CAN channel to add 
		can_msgtype: CAN frame 
		source_addr: ISO-TP source address
		target_addr: ISO-TP target address
		extension_addr: ISO-TP extension address
		target_addr_func: ISO-TP target address for functional addressing
		can_id: CAN ID used to communicate for physical addressing
		can_id_flow_ctrl: Flow Control CAN ID used during physical addressing communication
		can_id_func: CAN ID used to communicate for functionnal addressing
		isotp_msgtype: ISOTP message's type
		isotp_format: ISOTP message's format

	returns:
		true on successful configuration
	"""

	result = True

	# configure a mapping to transmit physical message
	mapping_phys_tx = cantp_mapping()
	mapping_phys_tx.can_id = can_id
	mapping_phys_tx.can_id_flow_ctrl = can_id_flow_ctrl
	mapping_phys_tx.can_msgtype = can_msgtype
	mapping_phys_tx.netaddrinfo.format = isotp_format
	mapping_phys_tx.netaddrinfo.msgtype = isotp_msgtype
	mapping_phys_tx.netaddrinfo.target_type = PCANTP_ISOTP_ADDRESSING_PHYSICAL
	mapping_phys_tx.netaddrinfo.source_addr = source_addr
	mapping_phys_tx.netaddrinfo.target_addr = target_addr
	mapping_phys_tx.netaddrinfo.extension_addr = extension_addr
	status = objPCANIsotp.AddMapping_2016(channel, mapping_phys_tx)
	if not objPCANIsotp.StatusIsOk_2016(status):
		print("Failed to configure mapping (can_id=0x%04X) (sts=0x%04X)" %(mapping_phys_tx.can_id, status.value))
		result = False
	# configure another mapping to receive physical message (invert source and target in previous mapping)
	mapping_phys_rx = cantp_mapping()
	mapping_phys_rx.can_id = mapping_phys_tx.can_id_flow_ctrl
	mapping_phys_rx.can_id_flow_ctrl = mapping_phys_tx.can_id
	mapping_phys_rx.can_msgtype = mapping_phys_tx.can_msgtype
	mapping_phys_rx.netaddrinfo = copy.deepcopy(mapping_phys_tx.netaddrinfo) 
	mapping_phys_rx.netaddrinfo.source_addr = mapping_phys_tx.netaddrinfo.target_addr
	mapping_phys_rx.netaddrinfo.target_addr = mapping_phys_tx.netaddrinfo.source_addr
	status = objPCANIsotp.AddMapping_2016(channel, mapping_phys_rx)
	if not objPCANIsotp.StatusIsOk_2016(status):
		print("Failed to configure mapping (can_id=0x%04X) (sts=0x%04X)" %(mapping_phys_rx.can_id, status.value))
		result = False
	
	if can_id_func != -1:
		# configure another mapping to transmit functional message
		mapping_func = cantp_mapping()
		mapping_func.can_id = can_id_func
		mapping_func.can_id_flow_ctrl = PCANTP_MAPPING_FLOW_CTRL_NONE
		mapping_func.can_msgtype = mapping_phys_tx.can_msgtype
		mapping_func.netaddrinfo = copy.deepcopy(mapping_phys_tx.netaddrinfo) 
		mapping_func.netaddrinfo.target_type = PCANTP_ISOTP_ADDRESSING_FUNCTIONAL
		mapping_func.netaddrinfo.target_addr = target_addr_func
		status = objPCANIsotp.AddMapping_2016(channel, mapping_func)
		if not objPCANIsotp.StatusIsOk_2016(status):
			print("Failed to configure mapping (can_id=0x%04X) (sts=0x%04X)" %(mapping_func.can_id, status.value))
			result = False

	return result

def define_mappings(channel):
	"""
	Initializes mappings compatible with PCTPClient/PCTPServer sample

	parameters:
	 channel: CAN channel to add the mappings to.
	returns: true on successful configuration
	"""

	result = True

	# 11 bit can ID, normal format addressing, diagnostic message (mandatory)
	result &= initialize_mappings(channel, PCANTP_CAN_MSGTYPE_STANDARD, N_SA, N_TA_PHYS, 0x00, N_TA_FUNC, 0xA1, 0xA2, 0xA5, PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, PCANTP_ISOTP_FORMAT_NORMAL)
	# 11 bit can ID, extended format addressing, diagnostic message (mandatory)
	result &= initialize_mappings(channel, PCANTP_CAN_MSGTYPE_STANDARD, N_SA, N_TA_PHYS, 0x00, N_TA_FUNC, 0xB1, 0xB2, 0xB5, PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, PCANTP_ISOTP_FORMAT_EXTENDED)
	result &= initialize_mappings(channel, PCANTP_CAN_MSGTYPE_STANDARD, N_SA, N_TA_PHYS, 0x00, N_TA_FUNC, 0xC1, 0xC2, 0xC5, PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, PCANTP_ISOTP_FORMAT_EXTENDED)


	# 11 bit can ID, mixed format addressing, remote diagnostic message (mandatory)
	result &= initialize_mappings(channel, PCANTP_CAN_MSGTYPE_STANDARD, N_SA, N_TA_PHYS, N_RA, N_TA_FUNC, 0xD1, 0xD2, 0xD5, PCANTP_ISOTP_MSGTYPE_REMOTE_DIAGNOSTIC, PCANTP_ISOTP_FORMAT_MIXED)

	# 29 bit can ID, normal format addressing, diagnostic message (mandatory)
	result &= initialize_mappings(channel, PCANTP_CAN_MSGTYPE_EXTENDED, N_SA, N_TA_PHYS, 0x00, N_TA_FUNC, 0xA123A1, 0xA123A2, 0xA123A5, PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, PCANTP_ISOTP_FORMAT_NORMAL)
	# 29 bit can ID, extended format addressing, diagnostic message
	result &= initialize_mappings(channel, PCANTP_CAN_MSGTYPE_EXTENDED, N_SA, N_TA_PHYS, 0x00, N_TA_FUNC, 0xA123C1, 0xA123C2, 0xA123C5, PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, PCANTP_ISOTP_FORMAT_EXTENDED)
	result &= initialize_mappings(channel, PCANTP_CAN_MSGTYPE_EXTENDED, N_SA, N_TA_PHYS, 0x00, N_TA_FUNC, 0xA123D1, 0xA123D2, 0xA123D5, PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, PCANTP_ISOTP_FORMAT_EXTENDED)

	return result

def display_message(message, timestamp, status):
	"""
	Outputs a summary of a received message.

	parameters:
	 message: The message.
	 timestamp: The timestamp of the message.
	 status: status of the Read function.
	returns:
	 1 on success (2 if extra information is available within status), 0 if no message and -1 on error
	"""

	strLoopback = ""
	strMsgType = ""
	displayedDataLength = 0

	# check status
	result = 1 if objPCANIsotp.StatusIsOk_2016(status) else -1
	if result != 1:
		if objPCANIsotp.StatusIsOk_2016(status, PCANTP_STATUS_NO_MESSAGE):
			# no message received ignore
			result = 0
		else:
			print("Failed to read message (sts=0x%04X)." %(status.value))
		return result
	
	# check bus errors within status
	if not objPCANIsotp.StatusIsOk_2016(status, PCANTP_STATUS_OK, True):
		print("Status returned information/warnings (sts=0x%04X)." %(status.value))
		result = 2

	# check if message has the loopback flag
	if (message.msgdata.any.contents.flags & PCANTP_MSGFLAG_LOOPBACK.value) ==  PCANTP_MSGFLAG_LOOPBACK.value:
		strLoopback = "loopback "

	# format message's type
	if message.type == PCANTP_MSGTYPE_CAN.value:
		strMsgType = "PCANTP_MSGTYPE_CAN"
	elif message.type == PCANTP_MSGTYPE_CANFD.value:
		strMsgType = "PCANTP_MSGTYPE_CANFD"
	elif message.type == PCANTP_MSGTYPE_ISOTP.value:
		strMsgType = "PCANTP_MSGTYPE_ISOTP"
	elif message.type == PCANTP_MSGTYPE_NONE.value:
		strMsgType = "PCANTP_MSGTYPE_NONE"
	else:
		strMsgType = "Unknown"

	# Only display data if network result is OK
	if message.msgdata.any.contents.netstatus == PCANTP_NETSTATUS_OK.value:
		displayedDataLength = message.msgdata.any.contents.length

	# Display generic information
	print("\n%010u - Received %s%s message : canid=0x%04X, length=%d - result: 0x%02x - %s" %(
		timestamp.value,
		strLoopback,
		strMsgType,
		message.can_info.can_id,
		message.msgdata.any.contents.length,
		message.msgdata.any.contents.netstatus,
		"ERROR !!!" if message.msgdata.any.contents.netstatus !=  PCANTP_NETSTATUS_OK.value else "OK !"))

	if message.type == PCANTP_MSGTYPE_ISOTP.value:
		strPending = "Completed"
		# Limit displayed data if message is pending
		if displayedDataLength > 0 and (message.msgdata.isotp.contents.netaddrinfo.msgtype & PCANTP_ISOTP_MSGTYPE_FLAG_INDICATION.value) != 0:
			displayedDataLength = 7 if message.msgdata.any.contents.length > 7 else message.msgdata.any.contents.length
			strPending = "/!\\ Pending"

		# Display iso-tp message's information
		print("\t%s message from 0x%02x (to 0x%02x, with RA 0x%02x)" %(
			strPending,
			message.msgdata.isotp.contents.netaddrinfo.source_addr,
			message.msgdata.isotp.contents.netaddrinfo.target_addr,
			message.msgdata.isotp.contents.netaddrinfo.extension_addr))
	

	if displayedDataLength > 0:
		# display data
		buffer = "\t\\-> Length: {x1}, Data= ".format(x1=format(message.msgdata.any.contents.length, 'd'))
		for i in range(displayedDataLength):
			buffer += "{x1} ".format(x1=format(message.msgdata.any.contents.data[i], '02x'))	
		if displayedDataLength != message.msgdata.any.contents.length:
			buffer += "..."
		print(buffer)
	
	return result

def consoleExit():
	"""
	Function called to clean opened ISO-TP channels
	"""
	objPCANIsotp.Uninitialize_2016(PCANTP_HANDLE_NONEBUS)

# Callback to handle closure of the console window
if IS_WINDOWS:
	@CFUNCTYPE(c_int, c_uint32)
	def HandlerRoutine(dwCtrlType):
		consoleExit()
		return False
else:
	def HandlerRoutine(signal, frame):
		consoleExit()
		sys.exit(0)

# Definitions
USE_CAN_FD = False
USE_EVENT = False

N_SA = 0xF1
N_TA_PHYS = 0x13
N_TA_FUNC = 0x26
N_RA = 0x52

#
# Main entry-point for this application
#

bStop = False
nbErr = 0

print("\nUsage: py pctp_server.py [cantp_handle] [USE_CAN_FD:0|1] [BTR0BTR1|BITRATE] [USE_EVENT:0|1]")
print(" * CAN: py pctp_server.py 0x51 0 0x1C ")
print(" * CAN FD: py pctp_server.py 0x51 1 \"f_clock=80000000, nom_brp=10, nom_tseg1=12, nom_tseg2=3, nom_sjw=1, data_brp=4, data_tseg1=7, data_tseg2=2, data_sjw=1\" ")
print("\n---------------------------------------------")

# clear gracefully ISO-TP if console window is killed
if IS_WINDOWS:
	windll.kernel32.SetConsoleCtrlHandler(HandlerRoutine, True) 
else:
	signal.signal(signal.SIGINT, HandlerRoutine)
	signal.signal(signal.SIGTERM, HandlerRoutine)
	signal.signal(signal.SIGABRT, HandlerRoutine)

atexit.register(consoleExit)

# Show version information
buffer = create_string_buffer(500)
status = objPCANIsotp.GetValue_2016(PCANTP_HANDLE_NONEBUS, PCANTP_PARAMETER_API_VERSION, buffer, 500)
if objPCANIsotp.StatusIsOk_2016(status):
	print("PCAN-ISO-TP API Version : ", buffer.value)
else: 
	print("Error failed to get PCAN-ISO-TP API Version (sts=0x%04X)" %(status.value))

argc = len(sys.argv)	
# 1st argument is the PCAN-channel to use (PCAN-USB channel 2)
if argc > 1:
	channel = cantp_handle(strtoi(sys.argv[1]))
else:
	channel = PCANTP_HANDLE_USBBUS2

# 2nd argument is CAN FD
if argc > 2:
	useCanFd = strtoi(sys.argv[2]) == 1
else:
	useCanFd = USE_CAN_FD

# 3rd argument is bitrate (either CAN or CAN FD)
baudrate = PCANTP_BAUDRATE_500K
bitrateFd = create_string_buffer(b'f_clock=80000000, nom_brp=10, nom_tseg1=12, nom_tseg2=3, nom_sjw=1, data_brp=4, data_tseg1=7, data_tseg2=2, data_sjw=1')
if argc > 3: 
	if useCanFd:
		if sys.version_info.major >= 3:
			bufferFD = b''
			for i in range(len(sys.argv[3])):
				bufferFD += c_byte(ord(sys.argv[3][i]))
			bitrateFd = create_string_buffer(bufferFD)
		else:
			bitrateFd = create_string_buffer(sys.argv[3])
	else: 
		baudrate = cantp_baudrate(strtoi(sys.argv[3]))

# 4th argument is USE EVENT
if argc > 4:
	useEvent = strtoi(sys.argv[4]) == 1
else:
	useEvent = USE_EVENT

try:

	old_settings = install_keyboardHit()

	# Initializing of the ISO-TP Communication session
	print("Connecting to channel 0x%02X..." %(channel.value))
	if not useCanFd:
		print(" * btr0btr1 = 0x%02X..." %(baudrate.value))
		status = objPCANIsotp.Initialize_2016(channel, baudrate, 0, 0, 0)
	else:
		print(" * bitrateFd = \"%s\"..." %(bitrateFd.value))
		status = objPCANIsotp.InitializeFD_2016(channel, bitrateFd)
	print(" -> Initialize CANTP: 0x%04X" %(status.value))
	# will start the reading loop only if successfully initialized
	if not objPCANIsotp.StatusIsOk_2016(status):
		bStop = True		
		nbErr = nbErr + 1
		print(" -> Initialization failed, exiting...")

	if not bStop:

		# configure the test session to log debug information 
		usValue = c_uint8(PCANTP_DEBUG_INFO)
		status = objPCANIsotp.SetValue_2016(channel, PCANTP_PARAMETER_DEBUG, usValue, 1)
		if not objPCANIsotp.StatusIsOk_2016(status):
			print("Failed to configure parameter 'PCANTP_PARAMETER_DEBUG' (sts=0x%04X)." %(status.value))
		# configure server - block size, 0 = unlimited
		param = c_uint8(20)		
		status = objPCANIsotp.SetValue_2016(channel, PCANTP_PARAMETER_BLOCK_SIZE, param, 1)
		if not objPCANIsotp.StatusIsOk_2016(status):
			print("Failed to configure parameter 'PCANTP_PARAMETER_BLOCK_SIZE' (sts=0x%04X)." %(status.value))
		# configure server - wait Xms between segmented frames transmission
		param = c_uint8(2)
		status = objPCANIsotp.SetValue_2016(channel, PCANTP_PARAMETER_SEPARATION_TIME, param, 1)
		if not objPCANIsotp.StatusIsOk_2016(status):
			print("Failed to configure parameter 'PCANTP_PARAMETER_SEPARATION_TIME' (sts=0x%04X)." %(status.value))
		# configure server - enable enhanced addressing to be compatible with PCTPClient
		param = c_uint8(PCANTP_VALUE_PARAMETER_ON)
		status = objPCANIsotp.SetValue_2016(channel, PCANTP_PARAMETER_SUPPORT_29B_ENHANCED, param, 1)
		if not objPCANIsotp.StatusIsOk_2016(status):
			print("Failed to configure parameter 'PCANTP_PARAMETER_SUPPORT_29B_ENHANCED' (sts=0x%04X)." %(status.value))
		# configure receive event
		if useEvent:
			print(" -> Creating receive event")
			if IS_WINDOWS:
				eventRcv = c_void_p(windll.kernel32.CreateEventA(None, 0,0,None))
				status = objPCANIsotp.SetValue_2016(channel, PCANTP_PARAMETER_RECEIVE_EVENT, eventRcv, sizeof(eventRcv))
			else:
				eventRcv = NULL_HANDLE()
				status = objPCANIsotp.GetValue_2016(channel, PCANTP_PARAMETER_RECEIVE_EVENT, eventRcv, sizeof(eventRcv))
			if not objPCANIsotp.StatusIsOk_2016(status):
				nbErr = nbErr + 1
				bStop = True
				print("Failed to configure parameter 'PCANTP_PARAMETER_RECEIVE_EVENT' (sts=0x%04X)." %(status.value))

		# configure ISO-TP mappings
		result = define_mappings(channel)
		if not result:
			nbErr = nbErr + 1
			bStop = True
			print(" -> Mappings configuration failed, exiting...")

	# Output available commands
	if not bStop:
		print("\n---------------------------------------------\n")
		print("\nNote: press 'c' or 'C' to clear screen,")
		print("Note: press 'q', 'Q' or '<Escape>' to quit...")
		print("\n---------------------------------------------\n")
		print("\n -> Waiting for messages...")

	# Reading loop
	message = cantp_msg()
	timestamp = cantp_timestamp()

	while not bStop:
		if useEvent:
			if IS_WINDOWS:
				windows_wait_result = c_uint64(windll.kernel32.WaitForSingleObject(eventRcv, 1))
				wait_result = True if windows_wait_result.value == WAIT_OBJECT_0 else False
			else:
				readable, _, _ = select.select([eventRcv.value], [], [], 0.001)
				wait_result = True if len(readable) > 0 else False
		else:
			sleep(0.001)
			wait_result = True
		if wait_result:
			# loop until no more message is available
			doLoop = True
			while doLoop:
				doLoop = False
				# initialize message
				status = objPCANIsotp.MsgDataAlloc_2016(message, PCANTP_MSGTYPE_NONE)
				if not objPCANIsotp.StatusIsOk_2016(status):
					print("Failed to allocate message (sts=0x%04X)." %(status.value))
					continue
				# retrieve any message from Rx queue
				status = objPCANIsotp.Read_2016(channel, message, timestamp)
				doLoop = display_message(message, timestamp, status) > 0
				# update error counter if a message was received but a network error occured
				if doLoop and message.msgdata.any.contents.netstatus != PCANTP_NETSTATUS_OK.value:
					nbErr = nbErr + 1
				# release message
				status = objPCANIsotp.MsgDataFree_2016(message)
				if not objPCANIsotp.StatusIsOk_2016(status):
					print("Failed to deallocate message (sts=0x%04X)." %(status.value))
		#Sleep(1)		
		# check exit request
		if keyboardHit():
			key = getInput()			
			if key == 'q' or key == 'Q' or key == 27: # 27 = escape
				bStop = True
			if key == 0x08:	# backspace
				objPCANIsotp.Reset_2016(channel)
			if key == 'c' or key == 'C' or key == 0x08:	# 8 = backspace
				if IS_WINDOWS:
					os.system("cls")
				else:
					os.system("clear")
				print("\nNote: press 'c' or 'C' to clear screen,")
				print("Note: press 'q', 'Q' or '<Escape>' to quit...\n")

	# Display a small report
	if nbErr > 0:
		print("\nERROR : %d errors occured.\n" %(nbErr))		
	else:	
		print("\nALL Transmissions succeeded !\n")

	print("\n\nPress <Enter> to quit...")
	getInput()

	# release CAN-TP
	objPCANIsotp.Uninitialize_2016(channel)

except KeyboardInterrupt:
	pass
finally:
	desinstall_keyboardHit(old_settings)