import string
import os
import io
import atexit
import time
import copy
from PCAN_ISO_TP_2016 import *

IS_WINDOWS = platform.system() == 'Windows'

if not IS_WINDOWS:
	import signal

# Support keyboard
def getInput():
	sys.stdin.read(1)

# Isotp library
objPCANIsotp = PCAN_ISO_TP_2016()

# Help functions
def getTimeIndicationInMilliseconds():
	"""
	Get a time indication in milliseconds
	"""
	return time.time()*1000

def strtoi(str):
	"""
	Convert string to int

	parameters:
	 str : string to convert
	returns:
	 int value
	"""
	base = 10
	if str.startswith("0x") or str.startswith("0X"):
		base = 16
	elif str.startswith("0"):
		base = 8
	return int(str, base)

def initialize_mappings(channel, can_msgtype, source_addr, target_addr, extension_addr, target_addr_func, can_id, can_id_flow_ctrl, can_id_func, isotp_msgtype, isotp_format):
	"""
	Configure mappings to handle a two-way communication and a functional communication (broadcast).

	parameters:
		channel: CAN channel to add 
		can_msgtype: CAN frame 
		source_addr: ISO-TP source address
		target_addr: ISO-TP target address
		extension_addr: ISO-TP extension address
		target_addr_func: ISO-TP target address for functional addressing
		can_id: CAN ID used to communicate for physical addressing
		can_id_flow_ctrl: Flow Control CAN ID used during physical addressing communication
		can_id_func: CAN ID used to communicate for functionnal addressing
		isotp_msgtype: ISOTP message's type
		isotp_format: ISOTP message's format

	returns:
		true on successful configuration
	"""

	result = True

	# configure a mapping to transmit physical message
	mapping_phys_tx = cantp_mapping()
	mapping_phys_tx.can_id = can_id
	mapping_phys_tx.can_id_flow_ctrl = can_id_flow_ctrl
	mapping_phys_tx.can_msgtype = can_msgtype
	mapping_phys_tx.netaddrinfo.format = isotp_format
	mapping_phys_tx.netaddrinfo.msgtype = isotp_msgtype
	mapping_phys_tx.netaddrinfo.target_type = PCANTP_ISOTP_ADDRESSING_PHYSICAL
	mapping_phys_tx.netaddrinfo.source_addr = source_addr
	mapping_phys_tx.netaddrinfo.target_addr = target_addr
	mapping_phys_tx.netaddrinfo.extension_addr = extension_addr
	status = objPCANIsotp.AddMapping_2016(channel, mapping_phys_tx)
	if not objPCANIsotp.StatusIsOk_2016(status):
		print("Failed to configure mapping (can_id=0x%04X) (sts=0x%04X)" %(mapping_phys_tx.can_id, status.value))
		result = False
	# configure another mapping to receive physical message (invert source and target in previous mapping)
	mapping_phys_rx = cantp_mapping()
	mapping_phys_rx.can_id = mapping_phys_tx.can_id_flow_ctrl
	mapping_phys_rx.can_id_flow_ctrl = mapping_phys_tx.can_id
	mapping_phys_rx.can_msgtype = mapping_phys_tx.can_msgtype
	mapping_phys_rx.netaddrinfo = copy.deepcopy(mapping_phys_tx.netaddrinfo)
	mapping_phys_rx.netaddrinfo.source_addr = mapping_phys_tx.netaddrinfo.target_addr
	mapping_phys_rx.netaddrinfo.target_addr = mapping_phys_tx.netaddrinfo.source_addr
	status = objPCANIsotp.AddMapping_2016(channel, mapping_phys_rx)
	if not objPCANIsotp.StatusIsOk_2016(status):
		print("Failed to configure mapping (can_id=0x%04X) (sts=0x%04X)" %(mapping_phys_rx.can_id, status.value))
		result = False
	
	if can_id_func != -1:
		# configure another mapping to transmit functional message
		mapping_func = cantp_mapping()
		mapping_func.can_id = can_id_func
		mapping_func.can_id_flow_ctrl = PCANTP_MAPPING_FLOW_CTRL_NONE
		mapping_func.can_msgtype = mapping_phys_tx.can_msgtype
		mapping_func.netaddrinfo = copy.deepcopy(mapping_phys_tx.netaddrinfo) 
		mapping_func.netaddrinfo.target_type = PCANTP_ISOTP_ADDRESSING_FUNCTIONAL
		mapping_func.netaddrinfo.target_addr = target_addr_func
		status = objPCANIsotp.AddMapping_2016(channel, mapping_func)
		if not objPCANIsotp.StatusIsOk_2016(status):
			print("Failed to configure mapping (can_id=0x%04X) (sts=0x%04X)" %(mapping_func.can_id, status.value))
			result = False

	return result

def define_mappings(channel):
	"""
	Initializes mappings compatible with PCTPClient/PCTPServer sample

	parameters:
	 channel: CAN channel to add the mappings to.
	returns: true on successful configuration
	"""

	result = True

	# 11 bit can ID, normal format addressing, diagnostic message (mandatory)
	result &= initialize_mappings(channel, PCANTP_CAN_MSGTYPE_STANDARD, N_SA, N_TA_PHYS, 0x00, N_TA_FUNC, 0xA1, 0xA2, 0xA5, PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, PCANTP_ISOTP_FORMAT_NORMAL)
	# 11 bit can ID, extended format addressing, diagnostic message (mandatory)
	result &= initialize_mappings(channel, PCANTP_CAN_MSGTYPE_STANDARD, N_SA, N_TA_PHYS, 0x00, N_TA_FUNC, 0xB1, 0xB2, 0xB5, PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, PCANTP_ISOTP_FORMAT_EXTENDED)
	result &= initialize_mappings(channel, PCANTP_CAN_MSGTYPE_STANDARD, N_SA, N_TA_PHYS, 0x00, N_TA_FUNC, 0xC1, 0xC2, 0xC5, PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, PCANTP_ISOTP_FORMAT_EXTENDED)


	# 11 bit can ID, mixed format addressing, remote diagnostic message (mandatory)
	result &= initialize_mappings(channel, PCANTP_CAN_MSGTYPE_STANDARD, N_SA, N_TA_PHYS, N_RA, N_TA_FUNC, 0xD1, 0xD2, 0xD5, PCANTP_ISOTP_MSGTYPE_REMOTE_DIAGNOSTIC, PCANTP_ISOTP_FORMAT_MIXED)

	# 29 bit can ID, normal format addressing, diagnostic message (mandatory)
	result &= initialize_mappings(channel, PCANTP_CAN_MSGTYPE_EXTENDED, N_SA, N_TA_PHYS, 0x00, N_TA_FUNC, 0xA123A1, 0xA123A2, 0xA123A5, PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, PCANTP_ISOTP_FORMAT_NORMAL)
	# 29 bit can ID, extended format addressing, diagnostic message
	result &= initialize_mappings(channel, PCANTP_CAN_MSGTYPE_EXTENDED, N_SA, N_TA_PHYS, 0x00, N_TA_FUNC, 0xA123C1, 0xA123C2, 0xA123C5, PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, PCANTP_ISOTP_FORMAT_EXTENDED)
	result &= initialize_mappings(channel, PCANTP_CAN_MSGTYPE_EXTENDED, N_SA, N_TA_PHYS, 0x00, N_TA_FUNC, 0xA123D1, 0xA123D2, 0xA123D5, PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, PCANTP_ISOTP_FORMAT_EXTENDED)

	return result

def display_message(message, timestamp, status):
	"""
	Outputs a summary of a received message.

	parameters:
	 message: The message.
	 timestamp: The timestamp of the message.
	 status: status of the Read function.
	returns:
	 1 on success (2 if extra information is available within status), 0 if no message and -1 on error
	"""

	strLoopback = ""
	strMsgType = ""
	displayedDataLength = 0

	# check status
	result = 1 if objPCANIsotp.StatusIsOk_2016(status) else -1
	if result != 1:
		if objPCANIsotp.StatusIsOk_2016(status, PCANTP_STATUS_NO_MESSAGE):
			# no message received ignore
			result = 0
		else:
			print("Failed to read message (sts=0x%04X)." %(status.value))
		return result
	
	# check bus errors within status
	if not objPCANIsotp.StatusIsOk_2016(status, PCANTP_STATUS_OK, True):
		print("Status returned information/warnings (sts=0x%04X)." %(status.value))
		result = 2

	# check if message has the loopback flag
	if (message.msgdata.any.contents.flags & PCANTP_MSGFLAG_LOOPBACK.value) ==  PCANTP_MSGFLAG_LOOPBACK.value:
		strLoopback = "loopback "

	# format message's type
	if message.type == PCANTP_MSGTYPE_CAN.value:
		strMsgType = "PCANTP_MSGTYPE_CAN"
	elif message.type == PCANTP_MSGTYPE_CANFD.value:
		strMsgType = "PCANTP_MSGTYPE_CANFD"
	elif message.type == PCANTP_MSGTYPE_ISOTP.value:
		strMsgType = "PCANTP_MSGTYPE_ISOTP"
	elif message.type == PCANTP_MSGTYPE_NONE.value:
		strMsgType = "PCANTP_MSGTYPE_NONE"
	else:
		strMsgType = "Unknown"

	# Only display data if network result is OK
	if message.msgdata.any.contents.netstatus == PCANTP_NETSTATUS_OK.value:
		displayedDataLength = message.msgdata.any.contents.length

	# Display generic information
	print("\n%010u - Received %s%s message : canid=0x%04X, length=%d - result: 0x%02x - %s" %(
		timestamp.value,
		strLoopback,
		strMsgType,
		message.can_info.can_id,
		message.msgdata.any.contents.length,
		message.msgdata.any.contents.netstatus,
		"ERROR !!!" if message.msgdata.any.contents.netstatus !=  PCANTP_NETSTATUS_OK.value else "OK !"))

	if message.type == PCANTP_MSGTYPE_ISOTP.value:
		strPending = "Completed"
		# Limit displayed data if message is pending
		if displayedDataLength > 0 and (message.msgdata.isotp.contents.netaddrinfo.msgtype & PCANTP_ISOTP_MSGTYPE_FLAG_INDICATION.value) != 0:
			displayedDataLength = 7 if message.msgdata.any.contents.length > 7 else message.msgdata.any.contents.length
			strPending = "/!\\ Pending"

		# Display iso-tp message's information
		print("\t%s message from 0x%02x (to 0x%02x, with RA 0x%02x)" %(
			strPending,
			message.msgdata.isotp.contents.netaddrinfo.source_addr,
			message.msgdata.isotp.contents.netaddrinfo.target_addr,
			message.msgdata.isotp.contents.netaddrinfo.extension_addr))
	

	if displayedDataLength > 0:
		# display data
		buffer = "\t\\-> Length: {x1}, Data= ".format(x1=format(message.msgdata.any.contents.length, 'd'))
		for i in range(displayedDataLength):
			buffer += "{x1} ".format(x1=format(message.msgdata.any.contents.data[i], '02x'))	
		if displayedDataLength != message.msgdata.any.contents.length:
			buffer += "..."
		print(buffer)
	
	return result

def consoleExit():
	"""
	Function called to clean opened ISO-TP channels
	"""
	objPCANIsotp.Uninitialize_2016(PCANTP_HANDLE_NONEBUS)

# Callback to handle closure of the console window
if IS_WINDOWS:
	@CFUNCTYPE(c_int, c_uint32)
	def HandlerRoutine(dwCtrlType):
		consoleExit()
		return False
else:
	def HandlerRoutine(signal, frame):
		consoleExit()
		sys.exit(0)

def read_segmented_message(channel, nbMsgRead):
	"""
	Reads a message.

	parameters:
		channel: The channel to read the message from.
		nbMsgRead: List of one item, to store the number of message actually read.
	
	returns:
	 number of errors
	"""

	message = cantp_msg()
	timestamp = cantp_timestamp()
	nbErr = 0
	doLoop = 0

	while doLoop == 0:

		# initialize message
		status = objPCANIsotp.MsgDataAlloc_2016(message, PCANTP_MSGTYPE_NONE)
		if not objPCANIsotp.StatusIsOk_2016(status):
			print("Failed to allocate message (sts=0x%04X)." %(status.value))
			nbErr = nbErr + 1
			continue
		
		# retrieve any message from Rx queue
		status = objPCANIsotp.Read_2016(channel, message, timestamp)
		# display_message returns true when a message is available
		doLoop = display_message(message, timestamp, status)
		if doLoop > 0:
			# update error counter if a message was received but a network error occured
			if message.msgdata.any.contents.netstatus != PCANTP_NETSTATUS_OK.value:
				nbErr = nbErr + 1
			
			if message.type == PCANTP_MSGTYPE_ISOTP.value and (message.msgdata.isotp.contents.netaddrinfo.msgtype & PCANTP_ISOTP_MSGTYPE_FLAG_INDICATION.value) != 0:
				# wait and read the full segmented message
				nbErr = nbErr + read_segmented_message(channel, nbMsgRead)
			else:
				nbMsgRead[0] = nbMsgRead[0] + 1	
		else:
			if doLoop == 0:
				# no message read, wait for it
				time.sleep(0.01)

		# release message
		status = objPCANIsotp.MsgDataFree_2016(message)
		if not objPCANIsotp.StatusIsOk_2016(status):
			print("Failed to deallocate message (sts=0x%04X)." %(status.value))

	return nbErr

def transmit(channel, canIdType, msgType, formatType, doCanFd, canFdInitialized):
	"""
	A function to transmit CAN-TP messages
	
	parameters:
	 channel: channel
	 canIdType: canIdType
	 msgType: CAN-TP message Type (PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC)
	 formatType: Format addressing type (see PCANTP_ISOTP_FORMAT_xxx)
	returns:
	 number of transmission errors
	"""

	status = cantp_status()
	message = cantp_msg()
	can_msgtype = PCANTP_CAN_MSGTYPE_STANDARD
	isotp_nai = cantp_netaddrinfo()
	lMaxSize = 0
	nbErrPhys = 0
	nbErrFunc = 0
	sleepTimeInSeconds = 0.0
	nbMsgRead = [0]

	# Output information message
	lStr = "??"
	if formatType == PCANTP_ISOTP_FORMAT_ENHANCED:
		lStr = "Enhanced"
	elif formatType == PCANTP_ISOTP_FORMAT_EXTENDED:
		lStr = "Extended"
	elif formatType == PCANTP_ISOTP_FORMAT_FIXED_NORMAL:
		lStr = "Fixed Normal"
	elif formatType == PCANTP_ISOTP_FORMAT_MIXED:
		lStr = "Mixed"
	elif formatType == PCANTP_ISOTP_FORMAT_NORMAL:
		lStr = "Normal"

	print("\n\nTransmission of %s, %s message, %s addressing format" %(
		"29 BIT CAN ID" if (canIdType.value & PCANTP_CAN_MSGTYPE_EXTENDED.value) != 0 else "11 BIT CAN ID",
		"Diagnostic" if msgType == PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC else "Remote diagnostic",
		lStr))
	if USE_GETCH:
		print("Press <Enter> to continue...")
		getInput()
	

	# initialize message
	isotp_nai.source_addr = N_SA
	if msgType == PCANTP_ISOTP_MSGTYPE_REMOTE_DIAGNOSTIC:
		isotp_nai.extension_addr = N_RA
	can_msgtype = canIdType
	if doCanFd:
		can_msgtype = cantp_can_msgtype(can_msgtype.value | PCANTP_CAN_MSGTYPE_FD.value)
		can_msgtype = cantp_can_msgtype(can_msgtype.value | PCANTP_CAN_MSGTYPE_BRS.value)
	
	isotp_nai.msgtype = msgType
	isotp_nai.format = formatType
	# prepare a buffer that will be used as a reference to initialize the data to transmit
	data = create_string_buffer(MESSAGE_MAX_LENGTH)
	for i in range(MESSAGE_MAX_LENGTH):
		data[i] = c_ubyte(i%256).value if sys.version_info.major >= 3 else chr(i%256)
		
	if TEST_PHYS_TRANSMISSION:
		# Transmitting data using Physical addressing
		print("Transmitting data using Physical addressing\n")
		g_timeReference = getTimeIndicationInMilliseconds()
		isotp_nai.target_addr = N_TA_PHYS
		isotp_nai.target_type = PCANTP_ISOTP_ADDRESSING_PHYSICAL
		# each loop transmits a message with a data length incremented by 1 and reads its loppbacl confirmation
		index = 1
		while index <= MESSAGE_MAX_LENGTH:

			# allocate a new ISOTP message
			status = objPCANIsotp.MsgDataAlloc_2016(message, PCANTP_MSGTYPE_ISOTP)
			if objPCANIsotp.StatusIsOk_2016(status):
				# initialize ISOTP message
				status = objPCANIsotp.MsgDataInit_2016(message, PCANTP_CAN_ID_DEFINED_BY_NAI, can_msgtype, index, data, isotp_nai)
				if objPCANIsotp.StatusIsOk_2016(status):
					# write message
					status = objPCANIsotp.Write_2016(channel, message)
					if objPCANIsotp.StatusIsOk_2016(status):
						print("\nSuccessfully queued ISOTP message: Length %i (sts=0x%04X)." %(index, status.value))
						# try to read loopback confirmation 
						nbMsgRead = [0]
						nbErrPhys = nbErrPhys +  read_segmented_message(channel, nbMsgRead)
						if nbMsgRead[0] != 1:
							print("Received unexpected messages (%d instead of 1)" %(nbMsgRead[0]))
					else:
						print("\nFailed to write ISOTP message: Length %i (sts=0x%04X)." %(index, status.value))
						nbErrPhys = nbErrPhys + 1
				else:
					print("\nFailed to initialize ISOTP message: Length %i (sts=0x%04X)." %(index, status.value))
					nbErrPhys = nbErrPhys + 1
			
				# release message
				status = objPCANIsotp.MsgDataFree_2016(message)
				if not objPCANIsotp.StatusIsOk_2016(status):
					print("Failed to deallocate message (sts=0x%04X)." %(status.value))
					nbErrPhys = nbErrPhys + 1

			else:
				print("Failed to allocate message (sts=0x%04X)." %(status.value))
				nbErrPhys = nbErrPhys + 1
			
			time.sleep(sleepTimeInSeconds)
		
			if index == 100: 
				# skip and jump to maximum data length
				index = MESSAGE_MAX_LENGTH - 1
			index = index + 1

	# end of if TEST_PHYS_TRANSMISSION

	if TEST_BURST_TRANSMISSION:
		print("\nTransmitting data using Physical addressing in 'BURST' mode\n")
		g_timeReference = getTimeIndicationInMilliseconds()
		isotp_nai.target_addr = N_TA_PHYS
		isotp_nai.target_type = PCANTP_ISOTP_ADDRESSING_PHYSICAL
		# Transmitting data using Physical addressing WITHOUT waiting for response confirmation
		# each loop transmits a message with a data length incremented by 1
		nbMsgSent = 0
		index = 1
		while index <= MESSAGE_MAX_LENGTH:
	
			# allocate a new ISOTP message
			status = objPCANIsotp.MsgDataAlloc_2016(message, PCANTP_MSGTYPE_ISOTP)
			if objPCANIsotp.StatusIsOk_2016(status):
				# initialize ISOTP message
				status = objPCANIsotp.MsgDataInit_2016(message, PCANTP_CAN_ID_DEFINED_BY_NAI, can_msgtype, index, data, isotp_nai)
				if objPCANIsotp.StatusIsOk_2016(status):
					# write message
					status = objPCANIsotp.Write_2016(channel, message)
					if objPCANIsotp.StatusIsOk_2016(status):
						nbMsgSent = nbMsgSent + 1
						print("\nSuccessfully queued ISOTP message: Length %i (sts=0x%04X)." %(index, status.value))				
					else:
						print("\nFailed to write ISOTP message: Length %i (sts=0x%04X)." %(index, status.value))
						nbErrPhys = nbErrPhys + 1
				else:
					print("\nFailed to initialize ISOTP message: Length %i (sts=0x%04X)." %(index, status.value))
					nbErrPhys = nbErrPhys + 1
			
				# release message
				status = objPCANIsotp.MsgDataFree_2016(message)
				if not objPCANIsotp.StatusIsOk_2016(status):
					print("Failed to deallocate message (sts=0x%04X)." %(status.value))
					nbErrPhys = nbErrPhys + 1
			
			else:
				print("Failed to allocate message (sts=0x%04X)." %(status.value))
				nbErrPhys = nbErrPhys + 1

	
			if index == 10:	# skip and try bigger LEN
				index = 2000
		
			if index == 2005: # skip
				break

			index = index + 1
		
		print("\nReading confirmation for each transmitted messages in 'BURST' mode\n")
		nbMsgRead = [0]
		while nbMsgRead[0] < nbMsgSent:
			# try to read loopback confirmation 
			nbErrPhys = nbErrPhys +  read_segmented_message(channel, nbMsgRead)
	
		objPCANIsotp.Reset_2016(channel)
	# end of if TEST_BURST_TRANSMISSION

	if TEST_FUNC_TRANSMISSION:
		# Transmitting data using Functional addressing 
		print("\nTransmitting data using Functional addressing\n")
		g_timeReference = getTimeIndicationInMilliseconds()
		isotp_nai.target_addr = N_TA_FUNC
		isotp_nai.target_type = PCANTP_ISOTP_ADDRESSING_FUNCTIONAL
		# Reminder: Functional addressing shall only be supported 
		# for Single Frame communication.
		# Thus maximum size depends on Format Addressing Type.
		if formatType == PCANTP_ISOTP_FORMAT_EXTENDED or formatType == PCANTP_ISOTP_FORMAT_MIXED:
			lMaxSize = (CAN_TX_DL_VALUE if doCanFd and canFdInitialized else PCANTP_MAX_LENGTH_CAN_STANDARD) - 2
		else:
			lMaxSize = (CAN_TX_DL_VALUE if doCanFd and canFdInitialized else PCANTP_MAX_LENGTH_CAN_STANDARD) - 1
	
		# each loop transmits a message with a data length incremented by 1 and reads its loppbacl confirmation
		for index in range(1, lMaxSize+1, 1):
	
			# allocate a new ISOTP message
			status = objPCANIsotp.MsgDataAlloc_2016(message, PCANTP_MSGTYPE_ISOTP)
			if objPCANIsotp.StatusIsOk_2016(status):
				# initialize ISOTP message
				status = objPCANIsotp.MsgDataInit_2016(message, PCANTP_CAN_ID_DEFINED_BY_NAI, can_msgtype, index, data, isotp_nai)
				if objPCANIsotp.StatusIsOk_2016(status):
					# write message
					status = objPCANIsotp.Write_2016(channel, message)
					if objPCANIsotp.StatusIsOk_2016(status):
						print("\nSuccessfully queued ISOTP message: Length %i (sts=0x%04X)." %(index, status.value))
						# try to read loopback confirmation 
						nbMsgRead = [0]
						nbErrFunc = nbErrFunc + read_segmented_message(channel, nbMsgRead)
						if nbMsgRead[0] != 1:
							print("Received unexpected messages (%d instead of 1)", nbMsgRead[0])
					else:
						print("\nFailed to write ISOTP message: Length %i (sts=0x%04X)." %(index, status.value))
						nbErrFunc = nbErrFunc + 1
				else:
					print("\nFailed to initialize ISOTP message: Length %i (sts=0x%04X)." %(index, status.value))
					nbErrFunc = nbErrFunc + 1
			
				# release message
				status = objPCANIsotp.MsgDataFree_2016(message)
				if not objPCANIsotp.StatusIsOk_2016(status):
					print("Failed to deallocate message (sts=0x%04X)." %(status.value))
					nbErrFunc = nbErrFunc + 1
		
			else:
				print("Failed to allocate message (sts=0x%04X)." %(status.value))
				nbErrFunc = nbErrFunc + 1

			time.sleep(sleepTimeInSeconds)
	
	# end of if TEST_FUNC_TRANSMISSION

	if nbErrPhys > 0 or nbErrFunc > 0:
		if nbErrPhys > 0:
			print("\nERROR : %d errors occured.\n" %(nbErrPhys))
		
		if nbErrFunc > 0:
			print("\nERROR : %d errors occured.\n" %(nbErrFunc))
		
		if USE_GETCH:
			print("\nPress <Enter> to continue...")
			getInput()

	else:
		print("\nTransmissions successful !\n")


	return nbErrFunc + nbErrPhys

# Definitions
USE_CAN_FD = False
USE_GETCH = False

N_SA = 0xF1
N_TA_PHYS = 0x13
N_TA_FUNC = 0x26
N_RA = 0x52

TEST_PHYS_TRANSMISSION	= 1
TEST_FUNC_TRANSMISSION	= 1
TEST_BURST_TRANSMISSION = 1
MESSAGE_MAX_LENGTH = PCANTP_MAX_LENGTH_ISOTP2004 + 1000
CAN_TX_DL_CODE	= 0x0D
CAN_TX_DL_VALUE = 32

# A global value to display timing information
g_timeReference = 0

#
# Main entry-point for this application
#

useCanFd = False
doCanFd = False
nbErr = 0

print("\nUsage: py PCTPClient.py [TPCANTPHandle] [USE_CAN_FD:0|1] [BTR0BTR1|BITRATE]")
print(" * CAN: py PCTPClient.py 0x51 0 0x1C ")
print(" * CAN FD: py PCTPClient.py 0x51 1 \"f_clock=80000000, nom_brp=10, nom_tseg1=12, nom_tseg2=3, nom_sjw=1, data_brp=4, data_tseg1=7, data_tseg2=2, data_sjw=1\" ")
print("\n---------------------------------------------\n")

# clear gracefully ISO-TP if console window is killed
if IS_WINDOWS:
	windll.kernel32.SetConsoleCtrlHandler(HandlerRoutine, True)
else:
	signal.signal(signal.SIGINT, HandlerRoutine)
	signal.signal(signal.SIGTERM, HandlerRoutine)
	signal.signal(signal.SIGABRT, HandlerRoutine)


atexit.register(consoleExit)

# Show version information
buff = create_string_buffer(500)
status = objPCANIsotp.GetValue_2016(PCANTP_HANDLE_NONEBUS, PCANTP_PARAMETER_API_VERSION, buff, 500)
if objPCANIsotp.StatusIsOk_2016(status):
	print("PCAN-ISO-TP API Version : ", buff.value)
else: 
	print("Error failed to get PCAN-ISO-TP API Version (sts=0x%04X)" %(status.value))

argc = len(sys.argv)	
# 1st argument is the PCAN-channel to use (PCAN-USB channel 1)
if argc > 1:
	channel = cantp_handle(strtoi(sys.argv[1]))
else:
	channel = PCANTP_HANDLE_USBBUS1
# 2nd argument is CAN FD
if argc > 2:
	useCanFd = strtoi(sys.argv[2]) == 1
else:
	useCanFd = USE_CAN_FD
# 3rd argument is bitrate (either CAN or CAN FD)
baudrate = PCANTP_BAUDRATE_500K
bitrateFd = create_string_buffer(b'f_clock=80000000, nom_brp=10, nom_tseg1=12, nom_tseg2=3, nom_sjw=1, data_brp=4, data_tseg1=7, data_tseg2=2, data_sjw=1')
if argc > 3:
	if useCanFd:
		if sys.version_info.major >= 3:
			bufferFD = b''
			for i in range(len(sys.argv[3])):
				bufferFD += c_byte(ord(sys.argv[3][i]))
			bitrateFd = create_string_buffer(bufferFD)
		else:
			bitrateFd = create_string_buffer(sys.argv[3])
	else:
		baudrate = cantp_baudrate(strtoi(sys.argv[3]))

try:
	# Initializing of the ISO-TP Communication session
	print("Connecting to channel 0x%02X..." %(channel.value))
	if not useCanFd:
		print(" * btr0btr1 = 0x%02X..." %(baudrate.value))
		status = objPCANIsotp.Initialize_2016(channel, baudrate, 0, 0, 0)
	else:
		print(" * bitrateFd = \"%s\"..." %(bitrateFd.value))
		status = objPCANIsotp.InitializeFD_2016(channel, bitrateFd)
	
	print(" -> Initialize CANTP: 0x%04X" %(status.value))
	# Start the write/read loop only if the channel was successfully initialized
	if not objPCANIsotp.StatusIsOk_2016(status):
		# initialization failed: abort
		print(" -> Initialization failed, exiting...\n")
		objPCANIsotp.Uninitialize_2016(channel)
		print("\n\nPress <Enter> to quit...")
		getInput()
		exit()

	# configure the test session to use can data padding with a specific padding value
	param = c_uint8(PCANTP_CAN_DATA_PADDING_ON)
	status = objPCANIsotp.SetValue_2016(channel, PCANTP_PARAMETER_CAN_DATA_PADDING, param, 1)
	if not objPCANIsotp.StatusIsOk_2016(status):
		print("Failed to configure parameter 'PCANTP_PARAMETER_CAN_DATA_PADDING' (sts=0x%04X)." %(status.value))
	
	param = c_uint8(0x99)
	status = objPCANIsotp.SetValue_2016(channel, PCANTP_PARAMETER_CAN_PADDING_VALUE, param, 1)
	if not objPCANIsotp.StatusIsOk_2016(status):
		print("Failed to configure parameter 'PCANTP_PARAMETER_CAN_PADDING_VALUE' (sts=0x%04X)." %(status.value))
	
	# set the default TX_DL size (i.e. the default CAN-FD DLC)
	param = c_uint8(CAN_TX_DL_CODE)
	status = objPCANIsotp.SetValue_2016(channel, PCANTP_PARAMETER_CAN_TX_DL, param, 1)
	if not objPCANIsotp.StatusIsOk_2016(status):
		print("Failed to configure parameter 'PCANTP_PARAMETER_CAN_TX_DL' (sts=0x%04X)." %(status.value))
	
	# configure server - enable enhanced addressing to be compatible with PCTPClient
	param = c_uint8(PCANTP_VALUE_PARAMETER_ON)
	status = objPCANIsotp.SetValue_2016(channel, PCANTP_PARAMETER_SUPPORT_29B_ENHANCED, param, 1)
	if not objPCANIsotp.StatusIsOk_2016(status):
		print("Failed to configure parameter 'PCANTP_PARAMETER_SUPPORT_29B_ENHANCED' (sts=0x%04X)." %(status.value))
	
	# log CAN frames
	param = c_uint8(PCANTP_DEBUG_WARNING)
	status = objPCANIsotp.SetValue_2016(channel, PCANTP_PARAMETER_DEBUG, param, 1)
	if not objPCANIsotp.StatusIsOk_2016(status):
		print("Failed to configure parameter 'PCANTP_PARAMETER_DEBUG' (sts=0x%04X)." %(status.value))
	
	# configure ISO-TP mappings
	if not define_mappings(channel):
		print(" -> Mappings configuration failed, exiting...")
		objPCANIsotp.Uninitialize_2016(channel)
		print("\n\nPress <Enter> to quit...")
		getInput()
		exit()

	for i in range(2):
		if i == 0:
			# first loop will test CAN-FD frames even if it was not initialized with Fd
			doCanFd = True
		else:
			if i == 1:
				# second loop tests standard CAN frames
				doCanFd = False

		# Transmit with 29 Bit CAN identifiers (and no CAN ID mapping is required)
		nbErr = nbErr + transmit(channel, PCANTP_CAN_MSGTYPE_EXTENDED, PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, PCANTP_ISOTP_FORMAT_FIXED_NORMAL, doCanFd, useCanFd)
		nbErr = nbErr + transmit(channel, PCANTP_CAN_MSGTYPE_EXTENDED, PCANTP_ISOTP_MSGTYPE_REMOTE_DIAGNOSTIC, PCANTP_ISOTP_FORMAT_MIXED, doCanFd, useCanFd)

		# Transmit with enhanced diagnostics 29 bit CAN identifiers (ISO-15765-3 8.3)
		nbErr = nbErr + transmit(channel, PCANTP_CAN_MSGTYPE_EXTENDED, PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, PCANTP_ISOTP_FORMAT_ENHANCED, doCanFd, useCanFd)

		# Transmit with 11 Bit CAN identifiers
		# those transmissions require a Mapping between CAN ID and Network Addressing Information
		# for each transmission, 3 mappings are added :
		#	- 2 for physical addressing, one defining SA->TA and the other TA->SA
		#	- 1 for functional addressing defining SA->TA
		#

		# 11 bit can ID, normal format addressing (diagnostic message mandatory)		
		nbErr = nbErr + transmit(channel, PCANTP_CAN_MSGTYPE_STANDARD, PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, PCANTP_ISOTP_FORMAT_NORMAL, doCanFd, useCanFd)
		# 11 bit can ID, extended format addressing (diagnostic message mandatory)		
		nbErr = nbErr + transmit(channel, PCANTP_CAN_MSGTYPE_STANDARD, PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, PCANTP_ISOTP_FORMAT_EXTENDED, doCanFd, useCanFd)
		# 11 bit can ID, mixed format addressing (remote diagnostic message mandatory)
		nbErr = nbErr + transmit(channel, PCANTP_CAN_MSGTYPE_STANDARD, PCANTP_ISOTP_MSGTYPE_REMOTE_DIAGNOSTIC, PCANTP_ISOTP_FORMAT_MIXED, doCanFd, useCanFd)

		# Transmit with 29 Bit CAN identifiers
		# those transmissions require a Mapping between CAN ID and Network Addressing Information
		# for each transmission, 3 mappings are added 2 for physical addressing and 
		# the other for functional addressing.
		#

		# 29 bit can ID, normal format addressing (diagnostic message mandatory)
		nbErr = nbErr + transmit(channel, PCANTP_CAN_MSGTYPE_EXTENDED, PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, PCANTP_ISOTP_FORMAT_NORMAL, doCanFd, useCanFd)
		# 29 bit can ID, extended format addressing (diagnostic message mandatory)
		nbErr = nbErr + transmit(channel, PCANTP_CAN_MSGTYPE_EXTENDED, PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, PCANTP_ISOTP_FORMAT_EXTENDED, doCanFd, useCanFd)
	

	if nbErr > 0:
		print("\nERROR : a total of %d errors occured.\n" %(nbErr))
		print("\n\nPress <Enter> to quit...")
		getInput()
	else:
		print("\nALL Transmissions succeeded !\n")


	objPCANIsotp.Uninitialize_2016(channel)

except KeyboardInterrupt:
	pass