﻿Imports System
Imports System.Collections.Generic
Imports System.IO
Imports System.Linq
Imports System.Text
Imports System.Threading.Tasks


Imports Peak.Can.Basic
Imports Peak.Can.IsoTp

Imports cantp_timestamp = System.UInt64

Module Module1

    Const STMIN_600US As UInt32 = &HF6

    ''' <summary>
    '''   Entry point of the program, start a small CAN ISO TP read/write example
    '''     -> Send a message containing Peak logo 
    '''     -> Receive message, wait complete reception (show progress)
    '''     -> Show loopback message And received message (Peak logo)
    ''' </summary>
    Sub Main()
        ' Local variables
        Dim Res As cantp_status
        Dim Buffer As StringBuilder = New StringBuilder(500)
        Dim STmin As UInt32
        Dim rx_msg As cantp_msg
        Dim tx_msg As cantp_msg
        Dim loopback_msg As cantp_msg
        Dim mapping As cantp_mapping
        Dim reverse_mapping As cantp_mapping
        Dim receiver_handle As cantp_handle
        Dim transmitter_handle As cantp_handle

        ' Initialize handles
        transmitter_handle = cantp_handle.PCANTP_HANDLE_USBBUS1 ' TODO: modify the value according to your available PCAN devices.
        receiver_handle = cantp_handle.PCANTP_HANDLE_USBBUS2 ' TODO: modify the value according to your available PCAN devices.

        ' Print version informations
        CanTpApi.GetValue_2016(cantp_handle.PCANTP_HANDLE_NONEBUS, cantp_parameter.PCANTP_PARAMETER_API_VERSION, Buffer, 500)
        Console.WriteLine("PCAN-ISO-TP API Version : {0}", Buffer)

        ' Initialize channels : CAN2.0 - 500Kbit/s
        Res = CanTpApi.Initialize_2016(transmitter_handle, cantp_baudrate.PCANTP_BAUDRATE_500K)
        Console.WriteLine("Initialize transmitter : {0}", STATUS_OK_KO(Res))
        Res = CanTpApi.Initialize_2016(receiver_handle, cantp_baudrate.PCANTP_BAUDRATE_500K)
        Console.WriteLine("Initialize receiver : {0}", STATUS_OK_KO(Res))

        ' Change STmin value to 600us
        STmin = STMIN_600US
        Res = CanTpApi.SetValue_2016(receiver_handle, cantp_parameter.PCANTP_PARAMETER_SEPARATION_TIME, STmin, CType(System.Runtime.InteropServices.Marshal.SizeOf(STmin), UInt32))
        Console.WriteLine("Set STMIN = 600us : {0}", STATUS_OK_KO(Res))
        Res = CanTpApi.SetValue_2016(transmitter_handle, cantp_parameter.PCANTP_PARAMETER_SEPARATION_TIME, STmin, CType(System.Runtime.InteropServices.Marshal.SizeOf(STmin), UInt32))
        Console.WriteLine("Set STMIN = 600us : {0}", STATUS_OK_KO(Res))

        ' Create and set a receive event on transmitter
        Dim transmitter_receive_event As System.Threading.AutoResetEvent = New System.Threading.AutoResetEvent(False)
        If (IntPtr.Size = 4) Then
            Dim iBuffer As UInt32 = Convert.ToUInt32(transmitter_receive_event.SafeWaitHandle.DangerousGetHandle.ToInt32)
            Res = CanTpApi.SetValue_2016(transmitter_handle, cantp_parameter.PCANTP_PARAMETER_RECEIVE_EVENT, iBuffer, CType(System.Runtime.InteropServices.Marshal.SizeOf(iBuffer), UInt32))
        ElseIf (IntPtr.Size = 8) Then
            Dim iBuffer As Int64 = transmitter_receive_event.SafeWaitHandle.DangerousGetHandle.ToInt64
            Dim byteArray() As Byte = BitConverter.GetBytes(iBuffer)
            Res = CanTpApi.SetValue_2016(transmitter_handle, cantp_parameter.PCANTP_PARAMETER_RECEIVE_EVENT, byteArray, CType(System.Runtime.InteropServices.Marshal.SizeOf(iBuffer), UInt64))
        End If
        Console.WriteLine("Set receive event on transmitter: {0}", STATUS_OK_KO(Res))

        ' Create and set a receive event on receiver
        Dim receiver_receive_event As System.Threading.AutoResetEvent = New System.Threading.AutoResetEvent(False)
        If (IntPtr.Size = 4) Then
            Dim iBuffer As UInt32 = Convert.ToUInt32(receiver_receive_event.SafeWaitHandle.DangerousGetHandle.ToInt32)
            Res = CanTpApi.SetValue_2016(receiver_handle, cantp_parameter.PCANTP_PARAMETER_RECEIVE_EVENT, iBuffer, CType(System.Runtime.InteropServices.Marshal.SizeOf(iBuffer), UInt32))
        ElseIf (IntPtr.Size = 8) Then
            Dim iBuffer As Int64 = receiver_receive_event.SafeWaitHandle.DangerousGetHandle.ToInt64
            Dim byteArray() As Byte = BitConverter.GetBytes(iBuffer)
            Res = CanTpApi.SetValue_2016(receiver_handle, cantp_parameter.PCANTP_PARAMETER_RECEIVE_EVENT, byteArray, CType(System.Runtime.InteropServices.Marshal.SizeOf(iBuffer), UInt64))
        End If
        Console.WriteLine("Set receive event on receiver : {0}", STATUS_OK_KO(Res))

        ' Allocate tx message
        Res = CanTpApi.MsgDataAlloc_2016(tx_msg, cantp_msgtype.PCANTP_MSGTYPE_ISOTP)
        Console.WriteLine("Allocate tx message : {0}", STATUS_OK_KO(Res))
        ' Allocate loopback message
        Res = CanTpApi.MsgDataAlloc_2016(rx_msg, cantp_msgtype.PCANTP_MSGTYPE_NONE)
        Console.WriteLine("Allocate loopback message : {0}", STATUS_OK_KO(Res))
        ' Allocate rx message
        Res = CanTpApi.MsgDataAlloc_2016(rx_msg, cantp_msgtype.PCANTP_MSGTYPE_NONE)
        Console.WriteLine("Allocate rx message : {0}", STATUS_OK_KO(Res))

        ' Create a simple physical mapping 
        '    - Source 0xF1 (transmitter), target 0x01 (receiver), CAN id 0xA1, CAN ID flow control 0xA2
        '    - Diagnostic message in a classic format
        mapping.can_id = &HA1
        mapping.can_id_flow_ctrl = &HA2
        mapping.can_msgtype = cantp_can_msgtype.PCANTP_CAN_MSGTYPE_STANDARD
        mapping.can_tx_dlc = &H0
        mapping.netaddrinfo.extension_addr = &H0
        mapping.netaddrinfo.format = cantp_isotp_format.PCANTP_ISOTP_FORMAT_NORMAL
        mapping.netaddrinfo.msgtype = cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC
        mapping.netaddrinfo.source_addr = &HF1
        mapping.netaddrinfo.target_addr = &H1
        mapping.netaddrinfo.target_type = cantp_isotp_addressing.PCANTP_ISOTP_ADDRESSING_PHYSICAL

        ' Create the associated reversed mapping 
        reverse_mapping = mapping
        reverse_mapping.can_id = mapping.can_id_flow_ctrl
        reverse_mapping.can_id_flow_ctrl = mapping.can_id
        reverse_mapping.netaddrinfo.source_addr = mapping.netaddrinfo.target_addr
        reverse_mapping.netaddrinfo.target_addr = mapping.netaddrinfo.source_addr

        ' Add mapping on channels
        Res = CanTpApi.AddMapping_2016(transmitter_handle, mapping)
        Console.WriteLine("Add a simple mapping on transmitter {0} : {1}", cantp_handle_toShortString(transmitter_handle), STATUS_OK_KO(Res))
        Res = CanTpApi.AddMapping_2016(transmitter_handle, reverse_mapping)
        Console.WriteLine("Add the reverse mapping on transmitter {0} : {1}", cantp_handle_toShortString(transmitter_handle), STATUS_OK_KO(Res))
        Res = CanTpApi.AddMapping_2016(receiver_handle, mapping)
        Console.WriteLine("Add a simple mapping on receiver {0} : {1}", cantp_handle_toShortString(receiver_handle), STATUS_OK_KO(Res))
        Res = CanTpApi.AddMapping_2016(receiver_handle, reverse_mapping)
        Console.WriteLine("Add the reverse mapping on receiver {0} : {1}", cantp_handle_toShortString(receiver_handle), STATUS_OK_KO(Res))

        ' Initialize Tx message containing a heavy data
        Res = CanTpApi.MsgDataInit_2016(tx_msg, CanTpApi.PCANTP_CAN_ID_DEFINED_BY_NAI, mapping.can_msgtype, heavy_data.HEAVY_DATA_SIZE, heavy_data.HEAVY_DATA, mapping.netaddrinfo)
        Console.WriteLine("Initialize tx message: {0}", STATUS_OK_KO(Res))

        ' Write message on transmitter
        Res = CanTpApi.Write_2016(transmitter_handle, tx_msg)
        Console.WriteLine("Write message on transmitter {0} : {1}", cantp_handle_toShortString(transmitter_handle), STATUS_OK_KO(Res))

        ' Read loopback message And save it 
        Res = wait_and_read_msg(transmitter_handle, transmitter_receive_event, loopback_msg)
        Console.WriteLine("Read loopback message on transmitter {0} : {1}", cantp_handle_toShortString(transmitter_handle), STATUS_OK_KO(Res))
        Dim fileTxOk As Boolean = CanTpApi.StatusIsOk_2016(Res)
        If fileTxOk Then
            write_binary_output_file("loopback_message.png", loopback_msg)
        End If

        ' Read received message And save it
        Res = wait_and_read_msg(receiver_handle, receiver_receive_event, rx_msg)
        Console.WriteLine("Read message on receiver {0} : {1}", cantp_handle_toShortString(receiver_handle), STATUS_OK_KO(Res))
        Dim fileRxOk As Boolean = CanTpApi.StatusIsOk_2016(Res)
        If fileRxOk Then
            write_binary_output_file("received_message.png", rx_msg)
        End If

        ' Show & remove result 
        If fileTxOk Then
            Console.WriteLine(" >>> Opening transmitted message (PEAK logo).")
            If File.Exists("loopback_message.png") Then
                System.Diagnostics.Process.Start("loopback_message.png")
            End If
        Else
            Console.WriteLine(vbNewLine & " >>> Failed to successfully transmit message (PEAK logo)!")
        End If

        If fileRxOk Then
            Console.WriteLine(" >>> Opening received message (PEAK logo).")
            If File.Exists("received_message.png") Then
                System.Diagnostics.Process.Start("received_message.png")
            End If
        Else
            Console.WriteLine(vbNewLine & " >>> Failed to successfully receive message (PEAK logo)!")
        End If

        ' Free messages
        Res = CanTpApi.MsgDataFree_2016(rx_msg)
        Console.WriteLine("Free rx message : {0}", STATUS_OK_KO(Res))
        Res = CanTpApi.MsgDataFree_2016(tx_msg)
        Console.WriteLine("Free tx message : {0}", STATUS_OK_KO(Res))
        Res = CanTpApi.MsgDataFree_2016(loopback_msg)
        Console.WriteLine("Free loopback message : {0}", STATUS_OK_KO(Res))

        ' Close receive event on receiver
        If IntPtr.Size = 4 Then
            Dim iBuffer As UInt32 = 0
            Res = CanTpApi.SetValue_2016(receiver_handle, cantp_parameter.PCANTP_PARAMETER_RECEIVE_EVENT, iBuffer, CType(System.Runtime.InteropServices.Marshal.SizeOf(iBuffer), UInt32))
        ElseIf IntPtr.Size = 8 Then
            Dim iBuffer As Int64 = 0
            Dim byteArray As Byte() = BitConverter.GetBytes(iBuffer)
            Res = CanTpApi.SetValue_2016(receiver_handle, cantp_parameter.PCANTP_PARAMETER_RECEIVE_EVENT, byteArray, CType(System.Runtime.InteropServices.Marshal.SizeOf(iBuffer), UInt64))
        End If
        Console.WriteLine("Stop receive event  on receiver : {0}", STATUS_OK_KO(Res))
        receiver_receive_event.Close()

        ' Close receive event on transmitter
        If IntPtr.Size = 4 Then
            Dim iBuffer As UInt32 = 0
            Res = CanTpApi.SetValue_2016(transmitter_handle, cantp_parameter.PCANTP_PARAMETER_RECEIVE_EVENT, iBuffer, CType(System.Runtime.InteropServices.Marshal.SizeOf(iBuffer), UInt32))
        ElseIf IntPtr.Size = 8 Then
            Dim iBuffer As Int64 = 0
            Dim byteArray As Byte() = BitConverter.GetBytes(iBuffer)
            Res = CanTpApi.SetValue_2016(transmitter_handle, cantp_parameter.PCANTP_PARAMETER_RECEIVE_EVENT, byteArray, CType(System.Runtime.InteropServices.Marshal.SizeOf(iBuffer), UInt64))
        End If
        Console.WriteLine("Stop receive event  on transmitter : {0}", STATUS_OK_KO(Res))
        transmitter_receive_event.Close()

        ' Uninitialize channels
        Res = CanTpApi.Uninitialize_2016(transmitter_handle)
        Console.WriteLine("Uninitialize transmitter : {0}", STATUS_OK_KO(Res))
        Res = CanTpApi.Uninitialize_2016(receiver_handle)
        Console.WriteLine("Uninitialize receiver : {0}", STATUS_OK_KO(Res))

        '  Remove results And exit
        Console.WriteLine("Press any key to exit...")
        Console.In.Read()

        Try
            File.Delete("loopback_message.png")
            File.Delete("received_message.png")
        Catch
        End Try

    End Sub

    Private Function OK_KO(ByVal test As Boolean) As String
        If test Then
            Return "OK"
        End If
        Return "KO"
    End Function

    Private Function STATUS_OK_KO(ByVal test As cantp_status) As String
        Return OK_KO(CanTpApi.StatusIsOk_2016(test))
    End Function

    ''' <summary>Return the name of the channel</summary>
    ''' <param name="value">Channel handle</param>
    ''' <returns>String containing the name of the channel</returns>
    Private Function cantp_handle_toShortString(ByVal value As cantp_handle) As String
        Dim res As String = ""
        Select Case (value)
            Case cantp_handle.PCANTP_HANDLE_NONEBUS
                res = "NONEBUS"
            Case cantp_handle.PCANTP_HANDLE_ISABUS1
                res = "ISA1"
            Case cantp_handle.PCANTP_HANDLE_ISABUS2
                res = "ISA2"
            Case cantp_handle.PCANTP_HANDLE_ISABUS3
                res = "ISA3"
            Case cantp_handle.PCANTP_HANDLE_ISABUS4
                res = "ISA4"
            Case cantp_handle.PCANTP_HANDLE_ISABUS5
                res = "ISA5"
            Case cantp_handle.PCANTP_HANDLE_ISABUS6
                res = "ISA6"
            Case cantp_handle.PCANTP_HANDLE_ISABUS7
                res = "ISA7"
            Case cantp_handle.PCANTP_HANDLE_ISABUS8
                res = "ISA8"
            Case cantp_handle.PCANTP_HANDLE_DNGBUS1
                res = "DNG1"
            Case cantp_handle.PCANTP_HANDLE_PCIBUS1
                res = "PCI1"
            Case cantp_handle.PCANTP_HANDLE_PCIBUS2
                res = "PCI2"
            Case cantp_handle.PCANTP_HANDLE_PCIBUS3
                res = "PCI3"
            Case cantp_handle.PCANTP_HANDLE_PCIBUS4
                res = "PCI4"
            Case cantp_handle.PCANTP_HANDLE_PCIBUS5
                res = "PCI5"
            Case cantp_handle.PCANTP_HANDLE_PCIBUS6
                res = "PCI6"
            Case cantp_handle.PCANTP_HANDLE_PCIBUS7
                res = "PCI7"
            Case cantp_handle.PCANTP_HANDLE_PCIBUS8
                res = "PCI8"
            Case cantp_handle.PCANTP_HANDLE_PCIBUS9
                res = "PCI9"
            Case cantp_handle.PCANTP_HANDLE_PCIBUS10
                res = "PCI10"
            Case cantp_handle.PCANTP_HANDLE_PCIBUS11
                res = "PCI11"
            Case cantp_handle.PCANTP_HANDLE_PCIBUS12
                res = "PCI12"
            Case cantp_handle.PCANTP_HANDLE_PCIBUS13
                res = "PCI13"
            Case cantp_handle.PCANTP_HANDLE_PCIBUS14
                res = "PCI14"
            Case cantp_handle.PCANTP_HANDLE_PCIBUS15
                res = "PCI15"
            Case cantp_handle.PCANTP_HANDLE_PCIBUS16
                res = "PCI16"
            Case cantp_handle.PCANTP_HANDLE_USBBUS1
                res = "USB1"
            Case cantp_handle.PCANTP_HANDLE_USBBUS2
                res = "USB2"
            Case cantp_handle.PCANTP_HANDLE_USBBUS3
                res = "USB3"
            Case cantp_handle.PCANTP_HANDLE_USBBUS4
                res = "USB4"
            Case cantp_handle.PCANTP_HANDLE_USBBUS5
                res = "USB5"
            Case cantp_handle.PCANTP_HANDLE_USBBUS6
                res = "USB6"
            Case cantp_handle.PCANTP_HANDLE_USBBUS7
                res = "USB7"
            Case cantp_handle.PCANTP_HANDLE_USBBUS8
                res = "USB8"
            Case cantp_handle.PCANTP_HANDLE_USBBUS9
                res = "USB9"
            Case cantp_handle.PCANTP_HANDLE_USBBUS10
                res = "USB10"
            Case cantp_handle.PCANTP_HANDLE_USBBUS11
                res = "USB11"
            Case cantp_handle.PCANTP_HANDLE_USBBUS12
                res = "USB12"
            Case cantp_handle.PCANTP_HANDLE_USBBUS13
                res = "USB13"
            Case cantp_handle.PCANTP_HANDLE_USBBUS14
                res = "USB14"
            Case cantp_handle.PCANTP_HANDLE_USBBUS15
                res = "USB15"
            Case cantp_handle.PCANTP_HANDLE_USBBUS16
                res = "USB16"
            Case cantp_handle.PCANTP_HANDLE_PCCBUS1
                res = "PCC1"
            Case cantp_handle.PCANTP_HANDLE_PCCBUS2
                res = "PCC2"
            Case cantp_handle.PCANTP_HANDLE_LANBUS1
                res = "LAN1"
            Case cantp_handle.PCANTP_HANDLE_LANBUS2
                res = "LAN2"
            Case cantp_handle.PCANTP_HANDLE_LANBUS3
                res = "LAN3"
            Case cantp_handle.PCANTP_HANDLE_LANBUS4
                res = "LAN4"
            Case cantp_handle.PCANTP_HANDLE_LANBUS5
                res = "LAN5"
            Case cantp_handle.PCANTP_HANDLE_LANBUS6
                res = "LAN6"
            Case cantp_handle.PCANTP_HANDLE_LANBUS7
                res = "LAN7"
            Case cantp_handle.PCANTP_HANDLE_LANBUS8
                res = "LAN8"
            Case cantp_handle.PCANTP_HANDLE_LANBUS9
                res = "LAN9"
            Case cantp_handle.PCANTP_HANDLE_LANBUS10
                res = "LAN10"
            Case cantp_handle.PCANTP_HANDLE_LANBUS11
                res = "LAN11"
            Case cantp_handle.PCANTP_HANDLE_LANBUS12
                res = "LAN12"
            Case cantp_handle.PCANTP_HANDLE_LANBUS13
                res = "LAN13"
            Case cantp_handle.PCANTP_HANDLE_LANBUS14
                res = "LAN14"
            Case cantp_handle.PCANTP_HANDLE_LANBUS15
                res = "LAN15"
            Case cantp_handle.PCANTP_HANDLE_LANBUS16
                res = "LAN16"
            Case Else
                res = "UNKNOWN"
        End Select
        Return res
    End Function


    ''' <summary>Wait And read single message</summary>
    ''' <param name="handle">Channel handle</param>
    ''' <param name="msg_buffer">Received message (out)</param>
    ''' <param name="evt">Receive event associated to the channel</param>
    ''' <returns>Status</returns>
    Private Function wait_and_read_msg(ByVal handle As cantp_handle, ByVal evt As System.Threading.AutoResetEvent, ByRef msg_buffer As cantp_msg) As cantp_status
        Dim res As cantp_status = cantp_status.PCANTP_STATUS_NO_MESSAGE
        ' Wait a receive event And get a message
        ' If we receive something, read the message
        If (evt.WaitOne(100)) Then
            res = CanTpApi.Read_2016(handle, msg_buffer)
            Console.WriteLine("Wait a message on {0} : {1}", cantp_handle_toShortString(handle), STATUS_OK_KO(res))
            ' Check if the message Is totaly received (only in ISOTP mode)
            If CanTpApi.StatusIsOk_2016(res) And ((cantp_msgtype.PCANTP_MSGTYPE_ISOTP And msg_buffer.type) = cantp_msgtype.PCANTP_MSGTYPE_ISOTP) Then
                ' get the net info of the message 
                Dim info As cantp_netaddrinfo
                If CanTpApi.getNetaddrinfo_2016(msg_buffer, info) Then
                    ' test the rx indication
                    If (info.msgtype And cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_FLAG_INDICATION_RX) = cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_FLAG_INDICATION_RX Then
                        ' The message Is being received, wait And show progress
                        res = wait_and_show_progress(handle, msg_buffer)
                    End If
                End If
            End If
        End If
        Return res
    End Function


    ''' <summary>Write message data in a binary file</summary>
    ''' <param name="file_path">Output file path</param>
    ''' <param name="msg">Message to write</param>
    Private Sub write_binary_output_file(ByVal file_path As String, ByRef msg As cantp_msg)
        If msg.Msgdata <> IntPtr.Zero Then
            Dim len As UInt32 = msg.Msgdata_any_Copy.length
            If len <> 0 Then
                Dim data() As Byte = New Byte(len) {}
                If (CanTpApi.getData_2016(msg, 0, data, len)) Then
                    Try
                        Dim stream As FileStream = New FileStream(file_path, FileMode.Create)
                        Dim writer As BinaryWriter = New BinaryWriter(stream)
                        For i As Integer = 0 To len - 1
                            Dim lsb As Byte = data(i)
                            Dim msb As Byte = data(i + 1)
                            writer.Write(msb)
                            writer.Write(lsb)
                            i += 1
                        Next i
                        writer.Close()
                        stream.Close()
                    Catch
                    End Try
                End If
            End If
        End If
    End Sub

    ''' <summary>Wait for a message to be received on a channel And show reception progress</summary>
    ''' <param name="handle">Channel handle</param>
    ''' <param name="msg_buffer">Received message</param>
    ''' <returns>a status</returns>
    Private Function wait_and_show_progress(ByVal handle As cantp_handle, ByRef msg_buffer As cantp_msg) As cantp_status

        Dim progress As cantp_msgprogress

        ' Set a buffer in order to receive a copy of the pending message
        Dim msgpend As cantp_msg

        ' Allocate a buffer in order to receive a copy of the pending message
        Dim res As cantp_status = CanTpApi.allocProgressBuffer_2016(progress, cantp_msgtype.PCANTP_MSGTYPE_ISOTP)
        Console.WriteLine("Allocate a progress buffer : {0}", STATUS_OK_KO(res))

        If CanTpApi.StatusIsOk_2016(res) Then
            Do
                System.Threading.Thread.Sleep(300)
                ' Refresh progress structure
                res = CanTpApi.GetMsgProgress_2016(handle, msg_buffer, cantp_msgdirection.PCANTP_MSGDIRECTION_RX, progress)
                ' Refresh pending message
                Dim getProgress As Boolean = CanTpApi.getProgressBuffer_2016(progress, msgpend)
                ' Get last bytes of pending message
                Dim vals() As Byte = New Byte(10) {}
                Dim lenpend As UInt32 = 0
                Dim getvals As Boolean = CanTpApi.getLength_2016(msgpend, lenpend)
                getvals = getvals AndAlso CanTpApi.getData_2016(msgpend, lenpend - 10, vals, 10)
                ' Show progress pencentage And last bytes
                Dim strVals As String = "Unknown"
                If getvals Then
                    strVals = getBytesAsString(vals, 10)
                End If
                Console.WriteLine("RX Progress on {0} = {1}%, last bytes {2} : {3}",
                    cantp_handle_toShortString(handle),
                    progress.percentage,
                    strVals,
                    OK_KO(getProgress And CanTpApi.StatusIsOk_2016(res)))
            Loop While (progress.state = cantp_msgprogress_state.PCANTP_MSGPROGRESS_STATE_PROCESSING)
            CanTpApi.freeProgressBuffer_2016(progress)
            ' The message Is received, read it
            res = CanTpApi.Read_2016(handle, msg_buffer)
        End If
        Return res
    End Function

    ''' <summary>Get a string printing values in hexa of an array of bytes</summary>
    ''' <param name="vals">Array of bytes</param>
    ''' <param name="len">Length of the array</param>
    ''' <returns>The string</returns>
    Private Function getBytesAsString(ByRef vals() As Byte, ByVal len As UInt32) As String
        Dim str As StringBuilder = New StringBuilder("")
        For i As UInt32 = 0 To len - 1
            str.AppendFormat("{0:X2}", vals(i))
        Next i
        Return str.ToString()
    End Function

End Module
