﻿Imports System
Imports System.Collections.Generic
Imports System.Linq
Imports System.Text
Imports System.Threading.Tasks
Imports System.Runtime.InteropServices


Imports Peak.Can.Basic
Imports Peak.Can.IsoTp

Imports cantp_timestamp = System.UInt64
Imports cantp_bitrate = System.String

Module Module1
    Const N_SA As Byte = &HF1
    Const N_TA_PHYS As Byte = &H13
    Const N_TA_FUNC As Byte = &H26
    Const N_RA As Byte = &H52
    Const USE_CAN_FD As Boolean = False
    Const USE_EVENT As Boolean = False

    ''' <summary>Main entry-point for this application.</summary>
    ''' <param name="args">The args.</param>
    Sub Main(ByVal args() As String)
        Dim Buffer As StringBuilder = New StringBuilder(500)
        Dim channel As cantp_handle
        Dim useCanFd As Boolean = False
        Dim baudrate As cantp_baudrate
        Dim bitrateFd As cantp_bitrate = ""
        Dim useEvent As Boolean = False
        Dim eventRcv As System.Threading.AutoResetEvent

        Console.WriteLine(vbNewLine & "Usage: pctp_server.exe [cantp_handle] [USE_CAN_FD:0|1] [BTR0BTR1|BITRATE] [USE_EVENT:0|1]")
        Console.WriteLine(" * CAN: pctp_server.exe 0x51 0 0x1C ")
        Console.WriteLine(" * CAN FD: pctp_server.exe 0x51 1 ""f_clock=80000000, nom_brp=10, nom_tseg1=12, nom_tseg2=3, nom_sjw=1, data_brp=4, data_tseg1=7, data_tseg2=2, data_sjw=1""")
        Console.WriteLine(vbNewLine & "---------------------------------------------")

        ' clear gracefully ISO-TP if console window Is killed
        AddHandler AppDomain.CurrentDomain.UnhandledException, AddressOf consoleExit
        Dim handler As HandlerRoutine = AddressOf ConsoleCtrlCheck
        SetConsoleCtrlHandler(handler, True)

        ' Show version information
        Dim status As cantp_status = CanTpApi.GetValue_2016(cantp_handle.PCANTP_HANDLE_NONEBUS, cantp_parameter.PCANTP_PARAMETER_API_VERSION, Buffer, 500)
        If CanTpApi.StatusIsOk_2016(status) Then
            Console.WriteLine("PCAN-ISO-TP API Version : {0}", Buffer)
        Else
            Console.WriteLine("Error failed to get PCAN-ISO-TP API Version (sts=0x{0:X})", status)
        End If

        ' 1st argument Is the PCAN-channel to use (PCAN-USB channel 2)
        If args.Length >= 1 Then
            channel = convertToHandle(args(0))
        Else
            channel = cantp_handle.PCANTP_HANDLE_USBBUS2
        End If
        ' 2nd argument Is CAN FD
        If args.Length >= 2 Then
            useCanFd = convertToBool(args(1))
        Else
            useCanFd = USE_CAN_FD
        End If
        ' 3rd argument Is bitrate (either CAN Or CAN FD)
        baudrate = cantp_baudrate.PCANTP_BAUDRATE_500K
        bitrateFd = "f_clock=80000000, nom_brp=10, nom_tseg1=12, nom_tseg2=3, nom_sjw=1, data_brp=4, data_tseg1=7, data_tseg2=2, data_sjw=1"
        If args.Length >= 3 Then
            If (useCanFd) Then
                bitrateFd = args(2)
            Else
                baudrate = convertToBaudRate(args(2))
            End If
        End If
        ' 4th argument Is USE EVENT
        If args.Length >= 4 Then
            useEvent = convertToBool(args(3))
        Else
            useEvent = USE_EVENT
        End If

        ' Initializing of the ISO-TP Communication session
        Console.WriteLine("Connecting to channel 0x{0:X}...", channel)
        If Not useCanFd Then
            Console.WriteLine(" * btr0btr1 = 0x{0:X}...", baudrate)
            status = CanTpApi.Initialize_2016(channel, baudrate)
        Else
            Console.WriteLine(" * bitrateFd = ""{0}""...", bitrateFd)
            status = CanTpApi.InitializeFD_2016(channel, bitrateFd)
        End If
        Console.WriteLine(" -> Initialize CANTP: 0x{0:X}", status)

        ' will start the reading loop only if successfully initialized
        Dim nbErr As Int32 = 0
        Dim bStop As Boolean = False
        If Not CanTpApi.StatusIsOk_2016(status) Then
            bStop = True
            nbErr += 1
            Console.WriteLine(" -> Initialization failed, exiting...")
        End If
        If Not bStop Then
            ' configure the test session to log debug information 
            Dim param() As Byte = New Byte(1) {}
            param(0) = CanTpApi.PCANTP_DEBUG_INFO
            status = CanTpApi.SetValue_2016(channel, cantp_parameter.PCANTP_PARAMETER_DEBUG, param, 1)
            If Not CanTpApi.StatusIsOk_2016(status) Then
                Console.WriteLine("Failed to configure parameter 'PCANTP_PARAMETER_DEBUG' (sts=0x{0:X}).", status)
            End If
            ' configure server - block size, 0 = unlimited
            param(0) = 20
            status = CanTpApi.SetValue_2016(channel, cantp_parameter.PCANTP_PARAMETER_BLOCK_SIZE, param, 1)
            If Not CanTpApi.StatusIsOk_2016(status) Then
                Console.WriteLine("Failed to configure parameter 'PCANTP_PARAMETER_BLOCK_SIZE' (sts=0x{0:X}).", status)
            End If
            ' configure server - wait Xms between segmented frames transmission
            param(0) = 2
            status = CanTpApi.SetValue_2016(channel, cantp_parameter.PCANTP_PARAMETER_SEPARATION_TIME, param, 1)
            If Not CanTpApi.StatusIsOk_2016(status) Then
                Console.WriteLine("Failed to configure parameter 'PCANTP_PARAMETER_SEPARATION_TIME' (sts=0x{0:X}).", status)
            End If
            ' configure server - enable enhanced addressing to be compatible with PCTPClient
            param(0) = CanTpApi.PCANTP_VALUE_PARAMETER_ON
            status = CanTpApi.SetValue_2016(channel, cantp_parameter.PCANTP_PARAMETER_SUPPORT_29B_ENHANCED, param, 1)
            If Not CanTpApi.StatusIsOk_2016(status) Then
                Console.WriteLine("Failed to configure parameter 'PCANTP_PARAMETER_SUPPORT_29B_ENHANCED' (sts=0x{0:X}).", status)
            End If

            ' configure receive event
            If useEvent Then
                Console.WriteLine(" -> Creating receive event")
                eventRcv = New System.Threading.AutoResetEvent(False)
                If IntPtr.Size = 4 Then
                    Dim iBuffer As UInt32 = Convert.ToUInt32(eventRcv.SafeWaitHandle.DangerousGetHandle().ToInt32)
                    status = CanTpApi.SetValue_2016(channel, cantp_parameter.PCANTP_PARAMETER_RECEIVE_EVENT, iBuffer, CType(System.Runtime.InteropServices.Marshal.SizeOf(iBuffer), UInt32))
                ElseIf IntPtr.Size = 8 Then
                    Dim iBuffer As Int64 = eventRcv.SafeWaitHandle.DangerousGetHandle().ToInt64()
                    Dim byteArray() As Byte = BitConverter.GetBytes(iBuffer)
                    status = CanTpApi.SetValue_2016(channel, cantp_parameter.PCANTP_PARAMETER_RECEIVE_EVENT, byteArray, CType(System.Runtime.InteropServices.Marshal.SizeOf(iBuffer), UInt64))
                End If
                If Not CanTpApi.StatusIsOk_2016(status) Then
                    nbErr += 1
                    bStop = True
                    Console.WriteLine("Failed to configure parameter 'PCANTP_PARAMETER_RECEIVE_EVENT' (sts=0x{0:X}).", status)
                End If
            End If

            ' configure ISO-TP mappings
            If Not initialize_mappings(channel) Then
                nbErr += 1
                bStop = True
                Console.WriteLine(" -> Mappings configuration failed, exiting...")
            End If
        End If

        ' Output available commands
        If Not bStop Then
            Console.WriteLine(vbNewLine & "---------------------------------------------" & vbNewLine)
            Console.WriteLine(vbNewLine & "Note: press 'c' or 'C' to clear screen,")
            Console.WriteLine(vbNewLine & "Note: press 'q', 'Q' or '<Escape>' to quit...")
            Console.WriteLine(vbNewLine & "---------------------------------------------" & vbNewLine)
            Console.WriteLine(vbNewLine & " -> Waiting for messages...")
        End If

        ' Reading loop
        Dim message As cantp_msg
        Dim timestamp As cantp_timestamp
        Dim result As Boolean

        While Not bStop

            If useEvent Then
                result = eventRcv.WaitOne(1)
            Else
                System.Threading.Thread.Sleep(1)
                result = True
            End If

            If result Then
                Dim doLoop As Boolean
                ' loop until no more message Is available
                Do
                    doLoop = False
                    ' initialize message
                    status = CanTpApi.MsgDataAlloc_2016(message, cantp_msgtype.PCANTP_MSGTYPE_NONE)
                    If Not CanTpApi.StatusIsOk_2016(status) Then
                        Console.WriteLine("Failed to allocate message (sts=0x{0:X}).", status)
                        Continue Do
                    End If

                    ' retrieve any message from Rx queue
                    status = CanTpApi.Read_2016(channel, message, timestamp)
                    doLoop = display_message(message, timestamp, status) > 0

                    ' update error counter if a message was received but a network error occured
                    If doLoop Then
                        Dim netsta As cantp_netstatus
                        If (CanTpApi.getNetStatus_2016(message, netsta)) Then
                            If netsta <> cantp_netstatus.PCANTP_NETSTATUS_OK Then
                                nbErr += 1
                            End If
                        Else
                            Console.WriteLine("Failed to get netstatus of message.")
                        End If
                    End If

                    ' release message
                    status = CanTpApi.MsgDataFree_2016(message)
                    If Not CanTpApi.StatusIsOk_2016(status) Then
                        Console.WriteLine("Failed to deallocate message (sts=0x{0:X}).", status)
                    End If

                Loop While doLoop
            End If

            'Sleep(1)		
            ' check exit request
            If Console.KeyAvailable Then
                Dim keyboard_res As Char = Console.ReadKey().KeyChar
                Select Case (keyboard_res)
                    Case "q"c, "Q"c, Microsoft.VisualBasic.ChrW(27)    'Escape
                        bStop = True
                    Case Microsoft.VisualBasic.ChrW(8)    ' backspace
                        CanTpApi.Reset_2016(channel)
                        '  do clear...
                        Console.Clear()
                        Console.WriteLine(vbNewLine & "Note: press 'c' or 'C' to clear screen,")
                        Console.WriteLine(vbNewLine & "Note: press 'q', 'Q' or '<Escape>' to quit..." & vbNewLine)
                    Case "c"c, "C"c
                        Console.Clear()
                        Console.WriteLine(vbNewLine & "Note: press 'c' or 'C' to clear screen,")
                        Console.WriteLine(vbNewLine & "Note: press 'q', 'Q' or '<Escape>' to quit..." & vbNewLine)
                    Case Else
                End Select
            End If
        End While

        ' Display a small report
        If nbErr > 0 Then
            Console.WriteLine(vbNewLine & "ERROR : {0} errors occured." & vbNewLine, nbErr)
        Else
            Console.WriteLine(vbNewLine & "ALL Transmissions succeeded !" & vbNewLine)
        End If

        Console.WriteLine(vbNewLine & vbNewLine & "Press <Enter> to quit...")
        Console.In.Read()

        ' release CAN-TP
        CanTpApi.Uninitialize_2016(channel)


    End Sub

    ' Function called to clean opened ISO-TP channels
    Private Sub consoleExit()
        CanTpApi.Uninitialize_2016(cantp_handle.PCANTP_HANDLE_NONEBUS)
    End Sub

    ' Callback to handle closure of the console window
    Public Function ConsoleCtrlCheck(ByVal ctrlType As CtrlTypes) As Boolean
        Dim isClosing As Boolean = False
        Select Case (ctrlType)
            Case CtrlTypes.CTRL_C_EVENT
                isClosing = True
            Case CtrlTypes.CTRL_BREAK_EVENT
                isClosing = True
            Case CtrlTypes.CTRL_CLOSE_EVENT
                isClosing = True
            Case CtrlTypes.CTRL_LOGOFF_EVENT
                isClosing = True
            Case CtrlTypes.CTRL_SHUTDOWN_EVENT
                isClosing = True
            Case Else
        End Select
        If isClosing Then
            consoleExit()
        End If
        Return False
    End Function
    ' An enumerated type for the control messages sent to the handler routine.
    Public Enum CtrlTypes
        CTRL_C_EVENT = 0
        CTRL_BREAK_EVENT
        CTRL_CLOSE_EVENT
        CTRL_LOGOFF_EVENT = 5
        CTRL_SHUTDOWN_EVENT
    End Enum
    ' Declare the SetConsoleCtrlHandler function as external And receiving a delegate.  
    Private Declare Function SetConsoleCtrlHandler Lib "kernel32" (ByVal handlerRoutine As HandlerRoutine, ByVal add As Boolean) As Boolean
    ' A delegate type to be used as the handler routine for SetConsoleCtrlHandler.
    Public Delegate Function HandlerRoutine(ByVal MyEvent As CtrlTypes) As Boolean



    Private Function convertToHandle(ByVal s As String) As cantp_handle
        Dim res As cantp_handle = cantp_handle.PCANTP_HANDLE_NONEBUS
        Try
            res = Convert.ToUInt32(s)
        Catch
            ' maybe the string Is prefixed by 0x so try hexa conversion
            Try
                res = Convert.ToUInt32(s, 16)
            Catch
                res = cantp_handle.PCANTP_HANDLE_NONEBUS
            End Try
        End Try
        Return res
    End Function

    Private Function convertToBaudRate(ByVal s As String) As cantp_baudrate
        Dim res As cantp_baudrate = 0
        Try
            res = Convert.ToUInt32(s)
        Catch
            ' maybe the string Is prefixed by 0x so try hexa conversion
            Try
                res = Convert.ToUInt32(s, 16)
            Catch
                res = 0
            End Try
        End Try
        Return res
    End Function

    Private Function convertToBool(ByVal s As String) As Boolean
        Dim res As Boolean = False
        Dim b As Byte
        Try
            res = Convert.ToBoolean(s)
        Catch
            ' maybe the string Is 0 Or 1 so try int conversion
            Try
                b = Convert.ToByte(s)
                If b = 0 Then
                    res = False
                ElseIf b = 1 Then
                    res = True
                Else
                    res = False
                End If
            Catch
                res = False
            End Try
        End Try
        Return res
    End Function

    ''' <summary>Initializes mappings compatible with PCTPClient/PCTPServer sample</summary>
    ''' <param name="channel">CAN channel to add the mappings to.</param>
    ''' <return>true on successful configuration</return>
    Private Function initialize_mappings(ByVal channel As cantp_handle) As Boolean

        Dim result As Boolean = True

        ' 11 bit can ID, normal format addressing, diagnostic message (mandatory)
        result = result AndAlso initialize_mappings(channel, cantp_can_msgtype.PCANTP_CAN_MSGTYPE_STANDARD, N_SA, N_TA_PHYS, &H0, N_TA_FUNC, &HA1, &HA2, &HA5, cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, cantp_isotp_format.PCANTP_ISOTP_FORMAT_NORMAL)
        ' 11 bit can ID, extended format addressing, diagnostic message (mandatory)
        result = result AndAlso initialize_mappings(channel, cantp_can_msgtype.PCANTP_CAN_MSGTYPE_STANDARD, N_SA, N_TA_PHYS, &H0, N_TA_FUNC, &HB1, &HB2, &HB5, cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, cantp_isotp_format.PCANTP_ISOTP_FORMAT_EXTENDED)
        result = result AndAlso initialize_mappings(channel, cantp_can_msgtype.PCANTP_CAN_MSGTYPE_STANDARD, N_SA, N_TA_PHYS, &H0, N_TA_FUNC, &HC1, &HC2, &HC5, cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, cantp_isotp_format.PCANTP_ISOTP_FORMAT_EXTENDED)
        ' 11 bit can ID, mixed format addressing, remote diagnostic message (mandatory)
        result = result AndAlso initialize_mappings(channel, cantp_can_msgtype.PCANTP_CAN_MSGTYPE_STANDARD, N_SA, N_TA_PHYS, N_RA, N_TA_FUNC, &HD1, &HD2, &HD5, cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_REMOTE_DIAGNOSTIC, cantp_isotp_format.PCANTP_ISOTP_FORMAT_MIXED)

        ' 29 bit can ID, normal format addressing, diagnostic message (mandatory)
        result = result AndAlso initialize_mappings(channel, cantp_can_msgtype.PCANTP_CAN_MSGTYPE_EXTENDED, N_SA, N_TA_PHYS, &H0, N_TA_FUNC, &HA123A1, &HA123A2, &HA123A5, cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, cantp_isotp_format.PCANTP_ISOTP_FORMAT_NORMAL)
        ' 29 bit can ID, extended format addressing, diagnostic message
        result = result AndAlso initialize_mappings(channel, cantp_can_msgtype.PCANTP_CAN_MSGTYPE_EXTENDED, N_SA, N_TA_PHYS, &H0, N_TA_FUNC, &HA123C1, &HA123C2, &HA123C5, cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, cantp_isotp_format.PCANTP_ISOTP_FORMAT_EXTENDED)
        result = result AndAlso initialize_mappings(channel, cantp_can_msgtype.PCANTP_CAN_MSGTYPE_EXTENDED, N_SA, N_TA_PHYS, &H0, N_TA_FUNC, &HA123D1, &HA123D2, &HA123D5, cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, cantp_isotp_format.PCANTP_ISOTP_FORMAT_EXTENDED)

        Return result
    End Function

    ''' <summary>Configure mappings to handle a two-way communication And a functional communication (broadcast).</summary>
    ''' <param name="channel">CAN channel to add </param>
    ''' <param name="can_msgtype">CAN frame </param>
    ''' <param name="source_addr">ISO-TP source address</param>
    ''' <param name="target_addr">ISO-TP target address</param>
    ''' <param name="extension_addr">ISO-TP extension address</param>
    ''' <param name="target_addr_func">ISO-TP target address for functional addressing</param>
    ''' <param name="can_id">CAN ID used to communicate for physical addressing</param>
    ''' <param name="can_id_flow_ctrl">Flow Control CAN ID used during physical addressing communication</param>
    ''' <param name="can_id_func">CAN ID used to communicate for functionnal addressing</param>
    ''' <param name="isotp_msgtype">ISOTP message's type</param>
    ''' <param name="isotp_format">ISOTP message's format</param>
    ''' <return>true on successful configuration</return>
    Private Function initialize_mappings(ByVal channel As cantp_handle, ByVal can_msgtype As cantp_can_msgtype, ByVal source_addr As UInt16, ByVal target_addr As UInt16, ByVal extension_addr As Byte, ByVal target_addr_func As UInt16, ByVal can_id As UInt32, ByVal can_id_flow_ctrl As UInt32, ByVal can_id_func As UInt32, ByVal isotp_msgtype As cantp_isotp_msgtype, ByVal isotp_format As cantp_isotp_format) As Boolean

        Dim result As Boolean = True
        ' "new" allows to clean variables (it Is common to leave can_tx_dl uninitialized which can lead to invalid 0xCC values)
        Dim mapping_phys_rx As cantp_mapping = New cantp_mapping()
        Dim mapping_phys_tx As cantp_mapping = New cantp_mapping()
        Dim mapping_func As cantp_mapping = New cantp_mapping()
        Dim status As cantp_status

        ' configure a mapping to transmit physical message
        mapping_phys_tx.can_id = can_id
        mapping_phys_tx.can_id_flow_ctrl = can_id_flow_ctrl
        mapping_phys_tx.can_msgtype = can_msgtype
        mapping_phys_tx.netaddrinfo.format = isotp_format
        mapping_phys_tx.netaddrinfo.msgtype = isotp_msgtype
        mapping_phys_tx.netaddrinfo.target_type = cantp_isotp_addressing.PCANTP_ISOTP_ADDRESSING_PHYSICAL
        mapping_phys_tx.netaddrinfo.source_addr = source_addr
        mapping_phys_tx.netaddrinfo.target_addr = target_addr
        mapping_phys_tx.netaddrinfo.extension_addr = extension_addr
        status = CanTpApi.AddMapping_2016(channel, mapping_phys_tx)
        If Not CanTpApi.StatusIsOk_2016(status) Then
            Console.WriteLine("Failed to configure mapping (can_id=0x{0:X}) (sts=0x{1:X})", mapping_phys_tx.can_id, status)
            result = False
        End If
        ' configure another mapping to receive physical message (invert source And target in previous mapping)
        mapping_phys_rx = mapping_phys_tx
        mapping_phys_rx.can_id = mapping_phys_tx.can_id_flow_ctrl
        mapping_phys_rx.can_id_flow_ctrl = mapping_phys_tx.can_id
        mapping_phys_rx.netaddrinfo.source_addr = mapping_phys_tx.netaddrinfo.target_addr
        mapping_phys_rx.netaddrinfo.target_addr = mapping_phys_tx.netaddrinfo.source_addr
        status = CanTpApi.AddMapping_2016(channel, mapping_phys_rx)
        If Not CanTpApi.StatusIsOk_2016(status) Then
            Console.WriteLine("Failed to configure mapping (can_id=0x{0:X}) (sts=0x{1:X})", mapping_phys_rx.can_id, status)
            result = False
        End If
        If can_id_func <> &HFFFFFFFF Then
            ' configure another mapping to transmit functional message
            mapping_func = mapping_phys_tx
            mapping_func.can_id = can_id_func
            mapping_func.can_id_flow_ctrl = CanTpApi.PCANTP_MAPPING_FLOW_CTRL_NONE
            mapping_func.netaddrinfo.target_type = cantp_isotp_addressing.PCANTP_ISOTP_ADDRESSING_FUNCTIONAL
            mapping_func.netaddrinfo.target_addr = target_addr_func
            status = CanTpApi.AddMapping_2016(channel, mapping_func)
            If Not CanTpApi.StatusIsOk_2016(status) Then
                Console.WriteLine("Failed to configure mapping (can_id=0x{0:X}) (sts=0x{1:X})", mapping_func.can_id, status)
                result = False
            End If
        End If
        Return result
    End Function
    ''' <summary>Outputs a summary of a received message.</summary>
    ''' <param name="message">The message.</param>
    ''' <param name="timestamp">The timestamp of the message.</param>
    ''' <param name="status">status of the Read function.</param>
    ''' <returns>1 on success (2 if extra information Is available within status), 0 if no message And -1 on error</returns>
    Private Function display_message(ByRef message As cantp_msg, ByVal timestamp As cantp_timestamp, ByVal status As cantp_status) As Integer

        Dim strLoopback As String = ""
        Dim strMsgType As String = ""
        Dim displayedDataLength As UInt32 = 0
        Dim result As Integer

        ' check status
        result = If(CanTpApi.StatusIsOk_2016(status), 1, -1)
        If result <> 1 Then
            If (CanTpApi.StatusIsOk_2016(status, cantp_status.PCANTP_STATUS_NO_MESSAGE)) Then
                ' no message received ignore
                result = 0
            Else
                Console.WriteLine("Failed to read message (sts=0x{0:X}).", status)
            End If
            Return result
        End If

        ' check bus errors within status
        If Not CanTpApi.StatusIsOk_2016(status, cantp_status.PCANTP_STATUS_OK, True) Then
            Console.WriteLine("Status returned information/warnings (sts=0x{0:X}).", status)
            result = 2
        End If

        ' get items of message
        Dim flags As cantp_msgflag
        If Not CanTpApi.getFlags_2016(message, flags) Then
            Console.WriteLine("Failed to get flags of message")
        End If
        Dim netsta As cantp_netstatus
        If Not CanTpApi.getNetStatus_2016(message, netsta) Then
            Console.WriteLine("Failed to get net status of message")
        End If
        Dim len As UInt32 = 0
        Dim vals() As Byte
        If Not CanTpApi.getLength_2016(message, len) Then
            Console.WriteLine("Failed to get length of message")
        ElseIf (len > 0) Then
            vals = New Byte(len) {}
            If Not CanTpApi.getData_2016(message, 0, vals, len) Then
                Console.WriteLine("Failed to get data of message")
            End If
        End If
        Dim netaddr As cantp_netaddrinfo
        If Not CanTpApi.getNetaddrinfo_2016(message, netaddr) Then
            Console.WriteLine("Failed to get net addr info of message")
        End If

        ' check if message has the loopback flag
        If (flags And cantp_msgflag.PCANTP_MSGFLAG_LOOPBACK) <> 0 Then
            strLoopback = "loopback "
        End If

        ' format message's type
        Select Case (message.type)
            Case cantp_msgtype.PCANTP_MSGTYPE_CAN
                strMsgType = "PCANTP_MSGTYPE_CAN"
            Case cantp_msgtype.PCANTP_MSGTYPE_CANFD
                strMsgType = "PCANTP_MSGTYPE_CANFD"
            Case cantp_msgtype.PCANTP_MSGTYPE_ISOTP
                strMsgType = "PCANTP_MSGTYPE_ISOTP"
            Case cantp_msgtype.PCANTP_MSGTYPE_NONE
                strMsgType = "PCANTP_MSGTYPE_NONE"
            Case Else
                strMsgType = "Unknown"
        End Select

        ' Only display data if network result Is OK
        If netsta = cantp_netstatus.PCANTP_NETSTATUS_OK Then
            displayedDataLength = len
        End If

        ' Display generic information
        Dim strError As String = If(netsta <> cantp_netstatus.PCANTP_NETSTATUS_OK, "ERROR !!!", "OK !")
        Console.WriteLine(vbNewLine & "{0} - Received {1}{2} message : canid=0x{3:X}, length={4} - result: 0x{5:X} - {6}",
                timestamp,
                strLoopback,
                strMsgType,
                message.can_info.can_id,
                len,
                netsta,
                strError)

        If message.type = cantp_msgtype.PCANTP_MSGTYPE_ISOTP Then
            Dim strPending As String = "Completed"
            ' Limit displayed data if message Is pending
            If displayedDataLength > 0 AndAlso (netaddr.msgtype And cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_FLAG_INDICATION) <> 0 Then
                displayedDataLength = If(len > 7, 7, len)
                strPending = "/!\ Pending"
            End If
            ' Display iso-tp message's information
            Console.WriteLine(vbTab & "{0} message from 0x{1:X} (to 0x{2:X}, with RA 0x{3:X})",
                    strPending,
                    netaddr.source_addr,
                    netaddr.target_addr,
                    netaddr.extension_addr)
        End If

        If displayedDataLength > 0 Then
            Dim Buffer As StringBuilder = New StringBuilder("")
            ' display data
            Buffer.AppendFormat(vbTab & "\-> Length: {0}, Data= ", len)
            For i As UInt32 = 0 To displayedDataLength - 1
                Buffer.AppendFormat("{0:X} ", vals(i))
            Next i
            If (displayedDataLength <> len) Then
                Buffer.Append("...")
            End If
            Console.WriteLine(Buffer)
        End If

        Return result
    End Function

End Module
