﻿Imports System
Imports System.Collections.Generic
Imports System.Linq
Imports System.Text
Imports System.Threading.Tasks
Imports System.Runtime.InteropServices


Imports Peak.Can.Basic
Imports Peak.Can.IsoTp

Imports cantp_timestamp = System.UInt64
Imports cantp_bitrate = System.String

Module Module1
    Const CAN_TX_DL_CODE As Byte = &HD
    Const CAN_TX_DL_VALUE As Byte = 32
    Const TEST_PHYS_TRANSMISSION As Boolean = True
    Const TEST_FUNC_TRANSMISSION As Boolean = True
    Const TEST_BURST_TRANSMISSION As Boolean = True
    Const MESSAGE_MAX_LENGTH As UInt32 = CanTpApi.PCANTP_MAX_LENGTH_ISOTP2004 + 1000
    Const USE_GETCH As Boolean = False
    Const N_SA As Byte = &HF1
    Const N_TA_PHYS As Byte = &H13
    Const N_TA_FUNC As Byte = &H26
    Const N_RA As Byte = &H52
    Const USE_CAN_FD As Boolean = False

    ''' <summary>Main entry-point for this application.</summary>
    ''' <param name="args">The args.</param>
    Sub Main(ByVal args() As String)
        Dim Buffer As StringBuilder = New StringBuilder(500)
        Dim channel As cantp_handle
        Dim useCanFd As Boolean = False
        Dim baudrate As cantp_baudrate
        Dim bitrateFd As cantp_bitrate = ""
        Dim useEvent As Boolean = False

        Console.WriteLine(vbNewLine & "Usage: PCTPClient.exe [TPCANTPHandle] [USE_CAN_FD:0|1] [BTR0BTR1|BITRATE]")
        Console.WriteLine(" * CAN: PCTPClient.exe 0x51 0 0x1C ")
        Console.WriteLine(" * CAN FD: PCTPClient.exe 0x51 1 ""f_clock=80000000, nom_brp=10, nom_tseg1=12, nom_tseg2=3, nom_sjw=1, data_brp=4, data_tseg1=7, data_tseg2=2, data_sjw=1""")
        Console.WriteLine(vbNewLine & "---------------------------------------------")

        ' clear gracefully ISO-TP if console window Is killed
        AddHandler AppDomain.CurrentDomain.UnhandledException, AddressOf consoleExit
        Dim handler As HandlerRoutine = AddressOf ConsoleCtrlCheck
        SetConsoleCtrlHandler(handler, True)

        ' Show version information
        Dim status As cantp_status = CanTpApi.GetValue_2016(cantp_handle.PCANTP_HANDLE_NONEBUS, cantp_parameter.PCANTP_PARAMETER_API_VERSION, Buffer, 500)
        If CanTpApi.StatusIsOk_2016(status) Then
            Console.WriteLine("PCAN-ISO-TP API Version : {0}", Buffer)
        Else
            Console.WriteLine("Error failed to get PCAN-ISO-TP API Version (sts=0x{0:X})", status)
        End If


        ' 1st argument Is the PCAN-channel to use (PCAN-USB channel 1)
        If args.Length >= 1 Then
            channel = convertToHandle(args(0))
        Else
            channel = cantp_handle.PCANTP_HANDLE_USBBUS1
        End If
        ' 2nd argument Is CAN FD
        If args.Length >= 2 Then
            useCanFd = convertToBool(args(1))
        Else
            useCanFd = USE_CAN_FD
        End If
        ' 3rd argument Is bitrate (either CAN Or CAN FD)
        baudrate = cantp_baudrate.PCANTP_BAUDRATE_500K
        bitrateFd = "f_clock=80000000, nom_brp=10, nom_tseg1=12, nom_tseg2=3, nom_sjw=1, data_brp=4, data_tseg1=7, data_tseg2=2, data_sjw=1"
        If args.Length >= 3 Then
            If (useCanFd) Then
                bitrateFd = args(2)
            Else
                baudrate = convertToBaudRate(args(2))
            End If
        End If


        ' Initializing of the ISO-TP Communication session
        Console.WriteLine("Connecting to channel 0x{0:X}...", channel)
        If Not useCanFd Then
            Console.WriteLine(" * btr0btr1 = 0x{0:X}...", baudrate)
            status = CanTpApi.Initialize_2016(channel, baudrate)
        Else
            Console.WriteLine(" * bitrateFd = ""{0}""...", bitrateFd)
            status = CanTpApi.InitializeFD_2016(channel, bitrateFd)
        End If
        Console.WriteLine(" -> Initialize CANTP: 0x{0:X}", status)

        ' will start the reading loop only if successfully initialized
        If Not CanTpApi.StatusIsOk_2016(status) Then
            Console.WriteLine(" -> Initialization failed, exiting...")
            CanTpApi.Uninitialize_2016(channel)
            Console.WriteLine(vbNewLine & vbNewLine & "Press <Enter> to quit...")
            Console.In.Read()
            Return
        End If

        ' configure the test session to use can data padding with a specific padding value
        Dim param() As Byte = New Byte(1) {}
        param(0) = CanTpApi.PCANTP_CAN_DATA_PADDING_ON
        status = CanTpApi.SetValue_2016(channel, cantp_parameter.PCANTP_PARAMETER_CAN_DATA_PADDING, param, 1)
        If Not CanTpApi.StatusIsOk_2016(status) Then
            Console.WriteLine("Failed to configure parameter 'PCANTP_PARAMETER_CAN_DATA_PADDING' (sts=0x{0:X}).", status)
        End If
        param(0) = &H99
        status = CanTpApi.SetValue_2016(channel, cantp_parameter.PCANTP_PARAMETER_CAN_PADDING_VALUE, param, 1)
        If Not CanTpApi.StatusIsOk_2016(status) Then
            Console.WriteLine("Failed to configure parameter 'PCANTP_PARAMETER_CAN_PADDING_VALUE' (sts=0x{0:X}).", status)
        End If
        ' set the default TX_DL size (i.e. the default CAN-FD DLC)
        param(0) = CAN_TX_DL_CODE
        status = CanTpApi.SetValue_2016(channel, cantp_parameter.PCANTP_PARAMETER_CAN_TX_DL, param, 1)
        If Not CanTpApi.StatusIsOk_2016(status) Then
            Console.WriteLine("Failed to configure parameter 'PCANTP_PARAMETER_CAN_TX_DL' (sts=0x{0:X}).", status)
        End If
        ' configure server - enable enhanced addressing to be compatible with PCTPClient
        param(0) = CanTpApi.PCANTP_VALUE_PARAMETER_ON
        status = CanTpApi.SetValue_2016(channel, cantp_parameter.PCANTP_PARAMETER_SUPPORT_29B_ENHANCED, param, 1)
        If Not CanTpApi.StatusIsOk_2016(status) Then
            Console.WriteLine("Failed to configure parameter 'PCANTP_PARAMETER_SUPPORT_29B_ENHANCED' (sts=0x{0:X}).", status)
        End If
        ' log CAN frames
        param(0) = CanTpApi.PCANTP_DEBUG_WARNING
        status = CanTpApi.SetValue_2016(channel, cantp_parameter.PCANTP_PARAMETER_DEBUG, param, 1)
        If Not CanTpApi.StatusIsOk_2016(status) Then
            Console.WriteLine("Failed to configure parameter 'PCANTP_PARAMETER_DEBUG' (sts=0x{0:X}).", status)
        End If
        ' configure ISO-TP mappings
        If Not initialize_mappings(channel) Then
            Console.WriteLine(" -> Mappings configuration failed, exiting...")
            CanTpApi.Uninitialize_2016(channel)
            Console.WriteLine(vbNewLine & vbNewLine & "Press <Enter> to quit...")
            Console.In.Read()
            Return
        End If

        Dim doCanFd As Boolean = False
        Dim nbErr As Integer = 0
        For i As Integer = 0 To 1

            Select Case (i)
                Case 0
                    ' first loop will test CAN-FD frames even if it was Not initialized with Fd
                    doCanFd = True
                Case 1
                    ' second loop tests standard CAN frames
                    doCanFd = False
            End Select

            ' Transmit with 29 Bit CAN identifiers (And no CAN ID mapping Is required)
            nbErr += transmit(channel, cantp_can_msgtype.PCANTP_CAN_MSGTYPE_EXTENDED, cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, cantp_isotp_format.PCANTP_ISOTP_FORMAT_FIXED_NORMAL, doCanFd, useCanFd)
            nbErr += transmit(channel, cantp_can_msgtype.PCANTP_CAN_MSGTYPE_EXTENDED, cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_REMOTE_DIAGNOSTIC, cantp_isotp_format.PCANTP_ISOTP_FORMAT_MIXED, doCanFd, useCanFd)

            ' Transmit with enhanced diagnostics 29 bit CAN identifiers (ISO-15765-3 §8.3)
            nbErr += transmit(channel, cantp_can_msgtype.PCANTP_CAN_MSGTYPE_EXTENDED, cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, cantp_isotp_format.PCANTP_ISOTP_FORMAT_ENHANCED, doCanFd, useCanFd)

            ' Transmit with 11 Bit CAN identifiers
            ' those transmissions require a Mapping between CAN ID And Network Addressing Information
            ' for each transmission, 3 mappings are added :
            '	- 2 for physical addressing, one defining SA->TA And the other TA->SA
            '	- 1 for functional addressing defining SA->TA
            '

            ' 11 bit can ID, normal format addressing (diagnostic message mandatory)		
            nbErr += transmit(channel, cantp_can_msgtype.PCANTP_CAN_MSGTYPE_STANDARD, cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, cantp_isotp_format.PCANTP_ISOTP_FORMAT_NORMAL, doCanFd, useCanFd)
            ' 11 bit can ID, extended format addressing (diagnostic message mandatory)		
            nbErr += transmit(channel, cantp_can_msgtype.PCANTP_CAN_MSGTYPE_STANDARD, cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, cantp_isotp_format.PCANTP_ISOTP_FORMAT_EXTENDED, doCanFd, useCanFd)
            ' 11 bit can ID, mixed format addressing (remote diagnostic message mandatory)
            nbErr += transmit(channel, cantp_can_msgtype.PCANTP_CAN_MSGTYPE_STANDARD, cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_REMOTE_DIAGNOSTIC, cantp_isotp_format.PCANTP_ISOTP_FORMAT_MIXED, doCanFd, useCanFd)

            ' Transmit with 29 Bit CAN identifiers
            ' those transmissions require a Mapping between CAN ID And Network Addressing Information
            ' for each transmission, 3 mappings are added 2 for physical addressing And 
            ' the other for functional addressing.
            '

            ' 29 bit can ID, normal format addressing (diagnostic message mandatory)
            nbErr += transmit(channel, cantp_can_msgtype.PCANTP_CAN_MSGTYPE_EXTENDED, cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, cantp_isotp_format.PCANTP_ISOTP_FORMAT_NORMAL, doCanFd, useCanFd)
            ' 29 bit can ID, extended format addressing (diagnostic message mandatory)
            nbErr += transmit(channel, cantp_can_msgtype.PCANTP_CAN_MSGTYPE_EXTENDED, cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, cantp_isotp_format.PCANTP_ISOTP_FORMAT_EXTENDED, doCanFd, useCanFd)
        Next i

        ' release CAN-TP
        If nbErr > 0 Then
            Console.WriteLine(vbNewLine & "ERROR : a total of {0} errors occured." & vbNewLine, nbErr)
            Console.WriteLine(vbNewLine & vbNewLine & "Press <Enter> to quit...")
            Console.In.Read()
        Else
            Console.WriteLine(vbNewLine & "ALL Transmissions succeeded !" & vbNewLine)
        End If

        CanTpApi.Uninitialize_2016(channel)

    End Sub

    ' Function called to clean opened ISO-TP channels
    Private Sub consoleExit()
        CanTpApi.Uninitialize_2016(cantp_handle.PCANTP_HANDLE_NONEBUS)
    End Sub

    ' Callback to handle closure of the console window
    Public Function ConsoleCtrlCheck(ByVal ctrlType As CtrlTypes) As Boolean
        Dim isClosing As Boolean = False
        Select Case (ctrlType)
            Case CtrlTypes.CTRL_C_EVENT
                isClosing = True
            Case CtrlTypes.CTRL_BREAK_EVENT
                isClosing = True
            Case CtrlTypes.CTRL_CLOSE_EVENT
                isClosing = True
            Case CtrlTypes.CTRL_LOGOFF_EVENT
                isClosing = True
            Case CtrlTypes.CTRL_SHUTDOWN_EVENT
                isClosing = True
            Case Else
        End Select
        If isClosing Then
            consoleExit()
        End If
        Return False
    End Function
    ' An enumerated type for the control messages sent to the handler routine.
    Public Enum CtrlTypes
        CTRL_C_EVENT = 0
        CTRL_BREAK_EVENT
        CTRL_CLOSE_EVENT
        CTRL_LOGOFF_EVENT = 5
        CTRL_SHUTDOWN_EVENT
    End Enum
    ' Declare the SetConsoleCtrlHandler function as external And receiving a delegate.  
    Private Declare Function SetConsoleCtrlHandler Lib "kernel32" (ByVal handlerRoutine As HandlerRoutine, ByVal add As Boolean) As Boolean
    ' A delegate type to be used as the handler routine for SetConsoleCtrlHandler.
    Public Delegate Function HandlerRoutine(ByVal MyEvent As CtrlTypes) As Boolean

    Private Function convertToHandle(ByVal s As String) As cantp_handle
        Dim res As cantp_handle = cantp_handle.PCANTP_HANDLE_NONEBUS
        Try
            res = Convert.ToUInt32(s)
        Catch
            ' maybe the string Is prefixed by 0x so try hexa conversion
            Try
                res = Convert.ToUInt32(s, 16)
            Catch
                res = cantp_handle.PCANTP_HANDLE_NONEBUS
            End Try
        End Try
        Return res
    End Function

    Private Function convertToBaudRate(ByVal s As String) As cantp_baudrate
        Dim res As cantp_baudrate = 0
        Try
            res = Convert.ToUInt32(s)
        Catch
            ' maybe the string Is prefixed by 0x so try hexa conversion
            Try
                res = Convert.ToUInt32(s, 16)
            Catch
                res = 0
            End Try
        End Try
        Return res
    End Function

    Private Function convertToBool(ByVal s As String) As Boolean
        Dim res As Boolean = False
        Dim b As Byte
        Try
            res = Convert.ToBoolean(s)
        Catch
            ' maybe the string Is 0 Or 1 so try int conversion
            Try
                b = Convert.ToByte(s)
                If b = 0 Then
                    res = False
                ElseIf b = 1 Then
                    res = True
                Else
                    res = False
                End If
            Catch
                res = False
            End Try
        End Try
        Return res
    End Function

    ''' <summary>Initializes mappings compatible with PCTPClient/PCTPServer sample</summary>
    ''' <param name="channel">CAN channel to add the mappings to.</param>
    ''' <return>true on successful configuration</return>
    Private Function initialize_mappings(ByVal channel As cantp_handle) As Boolean

        Dim result As Boolean = True

        ' 11 bit can ID, normal format addressing, diagnostic message (mandatory)
        result = result AndAlso initialize_mappings(channel, cantp_can_msgtype.PCANTP_CAN_MSGTYPE_STANDARD, N_SA, N_TA_PHYS, &H0, N_TA_FUNC, &HA1, &HA2, &HA5, cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, cantp_isotp_format.PCANTP_ISOTP_FORMAT_NORMAL)
        ' 11 bit can ID, extended format addressing, diagnostic message (mandatory)
        result = result AndAlso initialize_mappings(channel, cantp_can_msgtype.PCANTP_CAN_MSGTYPE_STANDARD, N_SA, N_TA_PHYS, &H0, N_TA_FUNC, &HB1, &HB2, &HB5, cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, cantp_isotp_format.PCANTP_ISOTP_FORMAT_EXTENDED)
        result = result AndAlso initialize_mappings(channel, cantp_can_msgtype.PCANTP_CAN_MSGTYPE_STANDARD, N_SA, N_TA_PHYS, &H0, N_TA_FUNC, &HC1, &HC2, &HC5, cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, cantp_isotp_format.PCANTP_ISOTP_FORMAT_EXTENDED)
        ' 11 bit can ID, mixed format addressing, remote diagnostic message (mandatory)
        result = result AndAlso initialize_mappings(channel, cantp_can_msgtype.PCANTP_CAN_MSGTYPE_STANDARD, N_SA, N_TA_PHYS, N_RA, N_TA_FUNC, &HD1, &HD2, &HD5, cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_REMOTE_DIAGNOSTIC, cantp_isotp_format.PCANTP_ISOTP_FORMAT_MIXED)

        ' 29 bit can ID, normal format addressing, diagnostic message (mandatory)
        result = result AndAlso initialize_mappings(channel, cantp_can_msgtype.PCANTP_CAN_MSGTYPE_EXTENDED, N_SA, N_TA_PHYS, &H0, N_TA_FUNC, &HA123A1, &HA123A2, &HA123A5, cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, cantp_isotp_format.PCANTP_ISOTP_FORMAT_NORMAL)
        ' 29 bit can ID, extended format addressing, diagnostic message
        result = result AndAlso initialize_mappings(channel, cantp_can_msgtype.PCANTP_CAN_MSGTYPE_EXTENDED, N_SA, N_TA_PHYS, &H0, N_TA_FUNC, &HA123C1, &HA123C2, &HA123C5, cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, cantp_isotp_format.PCANTP_ISOTP_FORMAT_EXTENDED)
        result = result AndAlso initialize_mappings(channel, cantp_can_msgtype.PCANTP_CAN_MSGTYPE_EXTENDED, N_SA, N_TA_PHYS, &H0, N_TA_FUNC, &HA123D1, &HA123D2, &HA123D5, cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, cantp_isotp_format.PCANTP_ISOTP_FORMAT_EXTENDED)

        Return result
    End Function

    ''' <summary>Configure mappings to handle a two-way communication And a functional communication (broadcast).</summary>
    ''' <param name="channel">CAN channel to add </param>
    ''' <param name="can_msgtype">CAN frame </param>
    ''' <param name="source_addr">ISO-TP source address</param>
    ''' <param name="target_addr">ISO-TP target address</param>
    ''' <param name="extension_addr">ISO-TP extension address</param>
    ''' <param name="target_addr_func">ISO-TP target address for functional addressing</param>
    ''' <param name="can_id">CAN ID used to communicate for physical addressing</param>
    ''' <param name="can_id_flow_ctrl">Flow Control CAN ID used during physical addressing communication</param>
    ''' <param name="can_id_func">CAN ID used to communicate for functionnal addressing</param>
    ''' <param name="isotp_msgtype">ISOTP message's type</param>
    ''' <param name="isotp_format">ISOTP message's format</param>
    ''' <return>true on successful configuration</return>
    Private Function initialize_mappings(ByVal channel As cantp_handle, ByVal can_msgtype As cantp_can_msgtype, ByVal source_addr As UInt16, ByVal target_addr As UInt16, ByVal extension_addr As Byte, ByVal target_addr_func As UInt16, ByVal can_id As UInt32, ByVal can_id_flow_ctrl As UInt32, ByVal can_id_func As UInt32, ByVal isotp_msgtype As cantp_isotp_msgtype, ByVal isotp_format As cantp_isotp_format) As Boolean

        Dim result As Boolean = True
        ' "new" allows to clean variables (it Is common to leave can_tx_dl uninitialized which can lead to invalid 0xCC values)
        Dim mapping_phys_rx As cantp_mapping = New cantp_mapping()
        Dim mapping_phys_tx As cantp_mapping = New cantp_mapping()
        Dim mapping_func As cantp_mapping = New cantp_mapping()
        Dim status As cantp_status

        ' configure a mapping to transmit physical message
        mapping_phys_tx.can_id = can_id
        mapping_phys_tx.can_id_flow_ctrl = can_id_flow_ctrl
        mapping_phys_tx.can_msgtype = can_msgtype
        mapping_phys_tx.netaddrinfo.format = isotp_format
        mapping_phys_tx.netaddrinfo.msgtype = isotp_msgtype
        mapping_phys_tx.netaddrinfo.target_type = cantp_isotp_addressing.PCANTP_ISOTP_ADDRESSING_PHYSICAL
        mapping_phys_tx.netaddrinfo.source_addr = source_addr
        mapping_phys_tx.netaddrinfo.target_addr = target_addr
        mapping_phys_tx.netaddrinfo.extension_addr = extension_addr
        status = CanTpApi.AddMapping_2016(channel, mapping_phys_tx)
        If Not CanTpApi.StatusIsOk_2016(status) Then
            Console.WriteLine("Failed to configure mapping (can_id=0x{0:X}) (sts=0x{1:X})", mapping_phys_tx.can_id, status)
            result = False
        End If
        ' configure another mapping to receive physical message (invert source And target in previous mapping)
        mapping_phys_rx = mapping_phys_tx
        mapping_phys_rx.can_id = mapping_phys_tx.can_id_flow_ctrl
        mapping_phys_rx.can_id_flow_ctrl = mapping_phys_tx.can_id
        mapping_phys_rx.netaddrinfo.source_addr = mapping_phys_tx.netaddrinfo.target_addr
        mapping_phys_rx.netaddrinfo.target_addr = mapping_phys_tx.netaddrinfo.source_addr
        status = CanTpApi.AddMapping_2016(channel, mapping_phys_rx)
        If Not CanTpApi.StatusIsOk_2016(status) Then
            Console.WriteLine("Failed to configure mapping (can_id=0x{0:X}) (sts=0x{1:X})", mapping_phys_rx.can_id, status)
            result = False
        End If
        If can_id_func <> &HFFFFFFFF Then
            ' configure another mapping to transmit functional message
            mapping_func = mapping_phys_tx
            mapping_func.can_id = can_id_func
            mapping_func.can_id_flow_ctrl = CanTpApi.PCANTP_MAPPING_FLOW_CTRL_NONE
            mapping_func.netaddrinfo.target_type = cantp_isotp_addressing.PCANTP_ISOTP_ADDRESSING_FUNCTIONAL
            mapping_func.netaddrinfo.target_addr = target_addr_func
            status = CanTpApi.AddMapping_2016(channel, mapping_func)
            If Not CanTpApi.StatusIsOk_2016(status) Then
                Console.WriteLine("Failed to configure mapping (can_id=0x{0:X}) (sts=0x{1:X})", mapping_func.can_id, status)
                result = False
            End If
        End If
        Return result
    End Function
    ''' <summary>Outputs a summary of a received message.</summary>
    ''' <param name="message">The message.</param>
    ''' <param name="timestamp">The timestamp of the message.</param>
    ''' <param name="status">status of the Read function.</param>
    ''' <returns>1 on success (2 if extra information Is available within status), 0 if no message And -1 on error</returns>
    Private Function display_message(ByRef message As cantp_msg, ByVal timestamp As cantp_timestamp, ByVal status As cantp_status) As Integer

        Dim strLoopback As String = ""
        Dim strMsgType As String = ""
        Dim displayedDataLength As UInt32 = 0
        Dim result As Integer

        ' check status
        result = If(CanTpApi.StatusIsOk_2016(status), 1, -1)
        If result <> 1 Then
            If (CanTpApi.StatusIsOk_2016(status, cantp_status.PCANTP_STATUS_NO_MESSAGE)) Then
                ' no message received ignore
                result = 0
            Else
                Console.WriteLine("Failed to read message (sts=0x{0:X}).", status)
            End If
            Return result
        End If

        ' check bus errors within status
        If Not CanTpApi.StatusIsOk_2016(status, cantp_status.PCANTP_STATUS_OK, True) Then
            Console.WriteLine("Status returned information/warnings (sts=0x{0:X}).", status)
            result = 2
        End If

        ' get items of message
        Dim flags As cantp_msgflag
        If Not CanTpApi.getFlags_2016(message, flags) Then
            Console.WriteLine("Failed to get flags of message")
        End If
        Dim netsta As cantp_netstatus
        If Not CanTpApi.getNetStatus_2016(message, netsta) Then
            Console.WriteLine("Failed to get net status of message")
        End If
        Dim len As UInt32 = 0
        Dim vals() As Byte
        If Not CanTpApi.getLength_2016(message, len) Then
            Console.WriteLine("Failed to get length of message")
        ElseIf (len > 0) Then
            vals = New Byte(len) {}
            If Not CanTpApi.getData_2016(message, 0, vals, len) Then
                Console.WriteLine("Failed to get data of message")
            End If
        End If
        Dim netaddr As cantp_netaddrinfo
        If Not CanTpApi.getNetaddrinfo_2016(message, netaddr) Then
            Console.WriteLine("Failed to get net addr info of message")
        End If

        ' check if message has the loopback flag
        If (flags And cantp_msgflag.PCANTP_MSGFLAG_LOOPBACK) <> 0 Then
            strLoopback = "loopback "
        End If

        ' format message's type
        Select Case (message.type)
            Case cantp_msgtype.PCANTP_MSGTYPE_CAN
                strMsgType = "PCANTP_MSGTYPE_CAN"
            Case cantp_msgtype.PCANTP_MSGTYPE_CANFD
                strMsgType = "PCANTP_MSGTYPE_CANFD"
            Case cantp_msgtype.PCANTP_MSGTYPE_ISOTP
                strMsgType = "PCANTP_MSGTYPE_ISOTP"
            Case cantp_msgtype.PCANTP_MSGTYPE_NONE
                strMsgType = "PCANTP_MSGTYPE_NONE"
            Case Else
                strMsgType = "Unknown"
        End Select

        ' Only display data if network result Is OK
        If netsta = cantp_netstatus.PCANTP_NETSTATUS_OK Then
            displayedDataLength = len
        End If

        ' Display generic information
        Dim strError As String = If(netsta <> cantp_netstatus.PCANTP_NETSTATUS_OK, "ERROR !!!", "OK !")
        Console.WriteLine(vbNewLine & "{0} - Received {1}{2} message : canid=0x{3:X}, length={4} - result: 0x{5:X} - {6}",
                timestamp,
                strLoopback,
                strMsgType,
                message.can_info.can_id,
                len,
                netsta,
                strError)

        If message.type = cantp_msgtype.PCANTP_MSGTYPE_ISOTP Then
            Dim strPending As String = "Completed"
            ' Limit displayed data if message Is pending
            If displayedDataLength > 0 AndAlso (netaddr.msgtype And cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_FLAG_INDICATION) <> 0 Then
                displayedDataLength = If(len > 7, 7, len)
                strPending = "/!\ Pending"
            End If
            ' Display iso-tp message's information
            Console.WriteLine(vbTab & "{0} message from 0x{1:X} (to 0x{2:X}, with RA 0x{3:X})",
                    strPending,
                    netaddr.source_addr,
                    netaddr.target_addr,
                    netaddr.extension_addr)
        End If

        If displayedDataLength > 0 Then
            Dim Buffer As StringBuilder = New StringBuilder("")
            ' display data
            Buffer.AppendFormat(vbTab & "\-> Length: {0}, Data= ", len)
            For i As UInt32 = 0 To displayedDataLength - 1
                Buffer.AppendFormat("{0:X} ", vals(i))
            Next i
            If (displayedDataLength <> len) Then
                Buffer.Append("...")
            End If
            Console.WriteLine(Buffer)
        End If

        Return result
    End Function

    ''' <summary>A function to transmit CAN-TP messages</summary>
    ''' <param name="channel">channel</param>
    ''' <param name="canIdType">canIdType</param>
    ''' <param name="msgType">CAN-TP message Type (PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC)</param>
    ''' <param name="formatType">Format addressing type (see PCANTP_ISOTP_FORMAT_xxx)</param>
    ''' <returns>number of transmission errors</returns>
    Private Function transmit(ByVal channel As cantp_handle,
           ByVal canIdType As cantp_can_msgtype,
           ByVal msgType As cantp_isotp_msgtype,
           ByVal formatType As cantp_isotp_format,
           ByVal doCanFd As Boolean, ByVal canFdInitialized As Boolean) As Integer

        Dim index As UInt32
        Dim status As cantp_status
        Dim message As cantp_msg
        Dim can_msgtype As cantp_can_msgtype = cantp_can_msgtype.PCANTP_CAN_MSGTYPE_STANDARD
        Dim isotp_nai As cantp_netaddrinfo
        Dim lMaxSize As Byte = 0
        Dim nbErrPhys As Integer = 0
        Dim nbErrFunc As Integer = 0
        Dim sleepTime As Integer = 0
        Dim nbMsgRead As Integer = 0
        Dim data() As Byte = New Byte(MESSAGE_MAX_LENGTH) {}

        ' Output information message
        Dim lStr As String = "??"
        Select Case (formatType)
            Case cantp_isotp_format.PCANTP_ISOTP_FORMAT_ENHANCED
                lStr = "Enhanced"
            Case cantp_isotp_format.PCANTP_ISOTP_FORMAT_EXTENDED
                lStr = "Extended"
            Case cantp_isotp_format.PCANTP_ISOTP_FORMAT_FIXED_NORMAL
                lStr = "Fixed Normal"
            Case cantp_isotp_format.PCANTP_ISOTP_FORMAT_MIXED
                lStr = "Mixed"
            Case cantp_isotp_format.PCANTP_ISOTP_FORMAT_NORMAL
                lStr = "Normal"
        End Select

        Dim strCanIdType As String = If((canIdType And cantp_can_msgtype.PCANTP_CAN_MSGTYPE_EXTENDED) <> 0, "29 BIT CAN ID", "11 BIT CAN ID")
        Dim strMsgType As String = If(msgType = cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC, "Diagnostic", "Remote diagnostic")
        Console.WriteLine(vbNewLine & vbNewLine & "Transmission of {0}, {1} message, {2} addressing format",
                strCanIdType,
                strMsgType,
                lStr)

        If USE_GETCH Then
            Console.WriteLine(vbNewLine & "Press <Enter> to continue...")
            Console.In.Read()
        End If

        ' initialize message
        isotp_nai.source_addr = N_SA
        If msgType = cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_REMOTE_DIAGNOSTIC Then
            isotp_nai.extension_addr = N_RA
        End If
        can_msgtype = canIdType
        If doCanFd Then
            can_msgtype = can_msgtype Or cantp_can_msgtype.PCANTP_CAN_MSGTYPE_FD
            can_msgtype = can_msgtype Or cantp_can_msgtype.PCANTP_CAN_MSGTYPE_BRS
        End If
        isotp_nai.msgtype = msgType
        isotp_nai.format = formatType
        ' prepare a buffer that will be used as a reference to initialize the data to transmit
        For i As UInt32 = 0 To MESSAGE_MAX_LENGTH - 1
            data(i) = (i And &HFF)
        Next i

        If TEST_PHYS_TRANSMISSION Then

            ' Transmitting data using Physical addressing
            Console.WriteLine(vbNewLine & "Transmitting data using Physical addressing" & vbNewLine)
            isotp_nai.target_addr = N_TA_PHYS
            isotp_nai.target_type = cantp_isotp_addressing.PCANTP_ISOTP_ADDRESSING_PHYSICAL
            ' each loop transmits a message with a data length incremented by 1 And reads its loppbacl confirmation
            For index = 1 To MESSAGE_MAX_LENGTH
                ' allocate a New ISOTP message
                status = CanTpApi.MsgDataAlloc_2016(message, cantp_msgtype.PCANTP_MSGTYPE_ISOTP)
                If CanTpApi.StatusIsOk_2016(status) Then
                    ' initialize ISOTP message
                    status = CanTpApi.MsgDataInit_2016(message, CanTpApi.PCANTP_CAN_ID_DEFINED_BY_NAI, can_msgtype, index, data, isotp_nai)
                    If CanTpApi.StatusIsOk_2016(status) Then
                        ' write message
                        status = CanTpApi.Write_2016(channel, message)
                        If CanTpApi.StatusIsOk_2016(status) Then
                            Console.WriteLine(vbNewLine & "Successfully queued ISOTP message: Length {0} (sts=0x{1:X}).", index, status)
                            ' try to read loopback confirmation 
                            nbMsgRead = 0
                            nbErrPhys += read_segmented_message(channel, nbMsgRead)
                            If nbMsgRead <> 1 Then
                                Console.WriteLine("Received unexpected messages ({0} instead of 1)", nbMsgRead)
                            End If
                        Else
                            Console.WriteLine(vbNewLine & "Failed to write ISOTP message: Length {0} (sts=0x{1:X}).", index, status)
                            nbErrPhys += 1
                        End If
                    Else
                        Console.WriteLine(vbNewLine & "Failed to initialize ISOTP message: Length {0} (sts=0x{1:X}).", index, status)
                        nbErrPhys += 1
                    End If
                    ' release message
                    status = CanTpApi.MsgDataFree_2016(message)
                    If Not CanTpApi.StatusIsOk_2016(status) Then
                        Console.WriteLine("Failed to deallocate message (sts=0x{0:X}).", status)
                        nbErrPhys += 1
                    End If
                Else
                    Console.WriteLine("Failed to allocate message (sts=0x{0:X}).", status)
                    nbErrPhys += 1
                End If

                System.Threading.Thread.Sleep(sleepTime)

                If (index = 100) Then
                    ' skip And jump to maximum data length
                    index = MESSAGE_MAX_LENGTH - 1
                End If
            Next index
        End If

        If TEST_BURST_TRANSMISSION Then

            Console.WriteLine(vbNewLine & "Transmitting data using Physical addressing in 'BURST' mode" & vbNewLine)

            isotp_nai.target_addr = N_TA_PHYS
            isotp_nai.target_type = cantp_isotp_addressing.PCANTP_ISOTP_ADDRESSING_PHYSICAL
            ' Transmitting data using Physical addressing WITHOUT waiting for response confirmation
            ' each loop transmits a message with a data length incremented by 1
            Dim nbMsgSent As Integer = 0
            For index = 1 To MESSAGE_MAX_LENGTH
                ' allocate a New ISOTP message
                status = CanTpApi.MsgDataAlloc_2016(message, cantp_msgtype.PCANTP_MSGTYPE_ISOTP)
                If CanTpApi.StatusIsOk_2016(status) Then
                    ' initialize ISOTP message
                    status = CanTpApi.MsgDataInit_2016(message, CanTpApi.PCANTP_CAN_ID_DEFINED_BY_NAI, can_msgtype, index, data, isotp_nai)
                    If CanTpApi.StatusIsOk_2016(status) Then
                        ' write message
                        status = CanTpApi.Write_2016(channel, message)
                        If CanTpApi.StatusIsOk_2016(status) Then
                            nbMsgSent += 1
                            Console.WriteLine(vbNewLine & "Successfully queued ISOTP message: Length {0} (sts=0x{1:X}).", index, status)
                        Else
                            Console.WriteLine(vbNewLine & "Failed to write ISOTP message: Length {0} (sts=0x{1:X}).", index, status)
                            nbErrPhys += 1
                        End If
                    Else
                        Console.WriteLine(vbNewLine & "Failed to initialize ISOTP message: Length {0} (sts=0x{1:X}).", index, status)
                        nbErrPhys += 1
                    End If
                    ' release message
                    status = CanTpApi.MsgDataFree_2016(message)
                    If Not CanTpApi.StatusIsOk_2016(status) Then
                        Console.WriteLine("Failed to deallocate message (sts=0x{0:X}).", status)
                        nbErrPhys += 1
                    End If
                Else
                    Console.WriteLine("Failed to allocate message (sts=0x{0:X}).", status)
                    nbErrPhys += 1
                End If

                If index = 10 Then
                    ' skip And try bigger LEN
                    index = 2000
                End If
                If index = 2005 Then
                    ' skip
                    Exit For
                End If
            Next index

            Console.WriteLine(vbNewLine & "Reading confirmation for each transmitted messages in 'BURST' mode" & vbNewLine)
            nbMsgRead = 0
            While nbMsgRead < nbMsgSent
                ' try to read loopback confirmation 
                nbErrPhys += read_segmented_message(channel, nbMsgRead)
            End While
            CanTpApi.Reset_2016(channel)
        End If

        If TEST_FUNC_TRANSMISSION Then

            ' Transmitting data using Functional addressing 
            Console.WriteLine(vbNewLine & "Transmitting data using Functional addressing" & vbNewLine)

            isotp_nai.target_addr = N_TA_FUNC
            isotp_nai.target_type = cantp_isotp_addressing.PCANTP_ISOTP_ADDRESSING_FUNCTIONAL
            ' Reminder: Functional addressing shall only be supported 
            ' for Single Frame communication.
            ' Thus maximum size depends on Format Addressing Type.
            Select Case (formatType)
                Case cantp_isotp_format.PCANTP_ISOTP_FORMAT_EXTENDED
                    lMaxSize = (If(doCanFd AndAlso canFdInitialized, CAN_TX_DL_VALUE, CanTpApi.PCANTP_MAX_LENGTH_CAN_STANDARD) - 2) And &HFF
                Case cantp_isotp_format.PCANTP_ISOTP_FORMAT_MIXED
                    lMaxSize = (If(doCanFd AndAlso canFdInitialized, CAN_TX_DL_VALUE, CanTpApi.PCANTP_MAX_LENGTH_CAN_STANDARD) - 2) And &HFF
                Case Else
                    lMaxSize = (If(doCanFd AndAlso canFdInitialized, CAN_TX_DL_VALUE, CanTpApi.PCANTP_MAX_LENGTH_CAN_STANDARD) - 1) And &HFF
            End Select

            ' each loop transmits a message with a data length incremented by 1 And reads its loppbacl confirmation
            For index = 1 To lMaxSize
                ' allocate a New ISOTP message
                status = CanTpApi.MsgDataAlloc_2016(message, cantp_msgtype.PCANTP_MSGTYPE_ISOTP)
                If CanTpApi.StatusIsOk_2016(status) Then
                    ' initialize ISOTP message
                    status = CanTpApi.MsgDataInit_2016(message, CanTpApi.PCANTP_CAN_ID_DEFINED_BY_NAI, can_msgtype, index, data, isotp_nai)
                    If CanTpApi.StatusIsOk_2016(status) Then
                        ' write message
                        status = CanTpApi.Write_2016(channel, message)
                        If CanTpApi.StatusIsOk_2016(status) Then
                            Console.WriteLine(vbNewLine & "Successfully queued ISOTP message: Length {0} (sts=0x{1:X}).", index, status)
                            ' try to read loopback confirmation 
                            nbMsgRead = 0
                            nbErrFunc += read_segmented_message(channel, nbMsgRead)
                            If nbMsgRead <> 1 Then
                                Console.WriteLine("Received unexpected messages ({0} instead of 1)", nbMsgRead)
                            End If
                        Else
                            Console.WriteLine(vbNewLine & "Failed to write ISOTP message: Length {0} (sts=0x{1:X}).", index, status)
                            nbErrFunc += 1
                        End If
                    Else
                        Console.WriteLine(vbNewLine & "Failed to initialize ISOTP message: Length {0} (sts=0x{1:X}).", index, status)
                        nbErrFunc += 1
                    End If
                    ' release message
                    status = CanTpApi.MsgDataFree_2016(message)
                    If Not CanTpApi.StatusIsOk_2016(status) Then
                        Console.WriteLine("Failed to deallocate message (sts=0x{0:X}).", status)
                        nbErrFunc += 1
                    End If
                Else
                    Console.WriteLine("Failed to allocate message (sts=0x{0:X}).", status)
                    nbErrFunc += 1
                End If

                System.Threading.Thread.Sleep(sleepTime)
            Next index
        End If

        If nbErrPhys > 0 OrElse nbErrFunc > 0 Then
            If nbErrPhys > 0 Then
                Console.WriteLine(vbNewLine & "ERROR : {0} errors occured." & vbNewLine, nbErrPhys)
            End If
            If nbErrFunc > 0 Then
                Console.WriteLine(vbNewLine & "ERROR : {0} errors occured." & vbNewLine, nbErrFunc)
            End If
            If USE_GETCH Then
                Console.WriteLine(vbNewLine & "Press <Enter> to continue...")
                Console.ReadKey()
            End If
        Else
            Console.WriteLine(vbNewLine & "Transmissions successful !" & vbNewLine)
        End If

        Return nbErrFunc + nbErrPhys

    End Function

    ''' <summary>Reads a message.</summary>
    ''' <param name="channel">The channel to read the message from.</param>
    ''' <param name="nbMsgRead">Buffer to store the number of message actually read.</param>
    ''' <returns>number of errors</returns>
    Private Function read_segmented_message(ByVal channel As cantp_handle, ByRef nbMsgRead As Integer) As Integer

        Dim message As cantp_msg
        Dim status As cantp_status
        Dim timestamp As cantp_timestamp
        Dim doLoop As Integer
        Dim nbErr As Integer = 0

        Do
            doLoop = 0
            ' initialize message
            status = CanTpApi.MsgDataAlloc_2016(message, cantp_msgtype.PCANTP_MSGTYPE_NONE)
            If Not CanTpApi.StatusIsOk_2016(status) Then
                Console.WriteLine("Failed to allocate message (sts=0x{0X}).", status)
                nbErr += 1
                Continue Do
            End If
            ' retrieve any message from Rx queue
            status = CanTpApi.Read_2016(channel, message, timestamp)
            ' display_message returns true when a message Is available
            doLoop = display_message(message, timestamp, status)
            If doLoop > 0 Then
                ' get items of message                  
                Dim netsta As cantp_netstatus
                If Not CanTpApi.getNetStatus_2016(message, netsta) Then
                    Console.WriteLine("Failed to get net status of message")
                End If
                Dim netaddr As cantp_netaddrinfo
                If Not CanTpApi.getNetaddrinfo_2016(message, netaddr) Then
                    Console.WriteLine("Failed to get net addr info of message")
                End If
                ' update error counter if a message was received but a network error occured
                If netsta <> cantp_netstatus.PCANTP_NETSTATUS_OK Then
                    nbErr += 1
                End If
                If (message.type = cantp_msgtype.PCANTP_MSGTYPE_ISOTP) AndAlso
                    ((netaddr.msgtype And cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_FLAG_INDICATION) <> 0) Then
                    ' wait And read the full segmented message
                    nbErr += read_segmented_message(channel, nbMsgRead)
                Else
                    nbMsgRead += 1
                End If
            Else
                If doLoop = 0 Then
                    ' no message read, wait for it
                    System.Threading.Thread.Sleep(10)
                End If
            End If
            ' release message
            status = CanTpApi.MsgDataFree_2016(message)
            If Not CanTpApi.StatusIsOk_2016(status) Then
                Console.WriteLine("Failed to deallocate message (sts=0x{0:X}).", status)
            End If
        Loop While doLoop = 0

        Return nbErr

    End Function

End Module
