﻿//  PCAN-OBDII.cs
//
//  ~~~~~~~~~~~~
//
//  PCAN-OBDII API
//
//  ~~~~~~~~~~~~
//
//  ------------------------------------------------------------------
//  Author : Fabrice Vergnaud
//	Last changed by:	$Author: Fabrice $
//  Last changed date:	$Date: 2019-10-11 13:53:39 +0200 (Fri, 11 Oct 2019) $
//
//  Language: C#
//  ------------------------------------------------------------------
//
//  Copyright (C) 2015  PEAK-System Technik GmbH, Darmstadt
//  more Info at http://www.peak-system.com 
//
using System;
using System.Text;
using System.Runtime.InteropServices;

namespace Peak.Can.ObdII
{
    // Aliases definition
    //
    using TPOBDIICANHandle = System.UInt16;  // Represents a CAN hardware channel of the underlying CAN system
    using TPOBDIIPid = System.Byte;          // Represents the PID number
    using TPOBDIIOBDMid = System.Byte;       // Represents the OBDMID number
    using TPOBDIITid = System.Byte;          // Represents the TID number
    using TPOBDIIInfoType = System.Byte;     // Represents the InfoType number

    #region Enumerations
    /// <summary>
    /// OBDII baudrates
    /// </summary>
    public enum TPOBDIIBaudrateInfo : byte
    {
        /// <summary>
        /// Non legislated-OBD baudrate
        /// </summary>
        POBDII_BAUDRATE_NON_LEGISLATED = 0x00,
        /// <summary>
        /// Baudrate: 250 kBit/s
        /// </summary>
        POBDII_BAUDRATE_250K = 0x01,
        /// <summary>
        /// Baudrate: 500 kBit/s
        /// </summary>
        POBDII_BAUDRATE_500K = 0x02,
        /// <summary>
        /// Baudrate: Autodetect
        /// </summary>
        POBDII_BAUDRATE_AUTODETECT = 0xFF,
    }
    /// <summary>
    /// OBDII address definitions
    /// </summary>
    public enum TPOBDIIAddress : byte
    {
        /// <summary>
        /// ECU 1
        /// </summary>
        POBDII_ECU_1 = 0x01,
        /// <summary>
        /// ECU 2
        /// </summary>
        POBDII_ECU_2 = 0x02,
        /// <summary>
        /// ECU 3
        /// </summary>
        POBDII_ECU_3 = 0x03,
        /// <summary>
        /// ECU 4
        /// </summary>
        POBDII_ECU_4 = 0x04,
        /// <summary>
        /// ECU 5
        /// </summary>
        POBDII_ECU_5 = 0x05,
        /// <summary>
        /// ECU 6
        /// </summary>
        POBDII_ECU_6 = 0x06,
        /// <summary>
        /// ECU 7
        /// </summary>
        POBDII_ECU_7 = 0x07,
        /// <summary>
        /// ECU 8
        /// </summary>
        POBDII_ECU_8 = 0x08,
    }

    /// <summary>
    /// Represents the OBDII Service numbers
    /// </summary>
    public enum TPOBDIIService : byte
    {
        /// <summary>
        /// Service 01 — Request Current Powertrain Diagnostic Data
        /// </summary>
        POBDII_SERVICE_01 = 0x01,
        /// <summary>
        /// Service 02 — Request Powertrain Freeze Frame Data
        /// </summary>
        POBDII_SERVICE_02 = 0x02,
        /// <summary>
        /// Service 03 — Request Emission-Related Diagnostic Trouble Codes
        /// </summary>
        POBDII_SERVICE_03 = 0x03,
        /// <summary>
        /// Service 04 — Clear/Reset Emission-Related Diagnostic Information
        /// </summary>
        POBDII_SERVICE_04 = 0x04,
        /// <summary>
        /// Service 06 — Request On-Board Monitoring Test Results for Specific Monitored Systems
        /// </summary>
        POBDII_SERVICE_06 = 0x06,
        /// <summary>
        /// Service 07 — Request Emission-Related Diagnostic Trouble Codes Detected During Current or Last Completed Driving Cycle
        /// </summary>
        POBDII_SERVICE_07 = 0x07,
        /// <summary>
        /// Service 08 — Request Control of On-Board System, Test or Component
        /// </summary>
        POBDII_SERVICE_08 = 0x08,
        /// <summary>
        /// Service 09 — Request Vehicle Information
        /// </summary>
        POBDII_SERVICE_09 = 0x09,
        /// <summary>
        /// Service 0A — Request Emission-Related Diagnostic Trouble Codes with Permanent Status
        /// </summary>
        POBDII_SERVICE_0A = 0x0A,
    }

    /// <summary>
    /// Represents a OBDII parameter to be read or set
    /// </summary>
    public enum TPOBDIIParameter : byte
    { 
        /// <summary>
        /// 1 BYTE data describing the logging mode 
        /// </summary>
        POBDII_PARAM_LOGGING = 0xB1,
        /// <summary>
        /// 1 BYTE data describing the number of available ECUs 
        /// </summary>
        POBDII_PARAM_AVAILABLE_ECUS = 0xB2,  
        /// <summary>
        /// 256 Byte data describing the supported PIDs for Service 01: Current Data
        /// </summary>
        POBDII_PARAM_SUPPORTMASK_PIDS = 0xB3,
        /// <summary>
        /// 257 Byte data describing the supported PIDs of Frame 
        /// (identified by the first BYTE in the buffer) for Service 02: Freeze Frame Data 
        /// </summary>
        POBDII_PARAM_SUPPORTMASK_FFPIDS = 0xB4,
        /// <summary>
        /// 256 Byte data describing the supported OBDMIDs for Service 06: Test Results
        /// </summary>
        POBDII_PARAM_SUPPORTMASK_OBDMIDS = 0xB5,
        /// <summary>
        /// 256 Byte data describing the supported TIDs for Service 08: Control Operation
        /// </summary>
        POBDII_PARAM_SUPPORTMASK_TIDS = 0xB6,
        /// <summary>
        /// 256 Byte data describing the supported InfoTypes for Service 09: Vehicle Information
        /// </summary>
        POBDII_PARAM_SUPPORTMASK_INFOTYPES = 0xB7,
        /// <summary>
        /// API version parameter
        /// </summary>
        POBDII_PARAM_API_VERSION = 0xB8,
        /// <summary>
        /// 1 BYTE data describing baudrate value
        /// </summary>
        POBDII_PARAM_BAUDRATE = 0xB9,
        /// <summary>
        /// 1 BYTE data describing CAN identifier length
        /// </summary>
        POBDII_PARAM_CAN_ID = 0xBA,
        /// <summary>
        /// 1 BYTE data describing the debug mode 
        /// </summary>
        POBDII_PARAM_DEBUG = 0xE3,
        /// <summary>
        /// 1 Byte data describing the condition of a channel
        /// </summary>
        POBDII_PARAM_CHANNEL_CONDITION = 0xE4
    }

    /// <summary>
    /// Represents the type of CAN hardware to be initialized
    /// </summary>
    public enum TPOBDIIHwType : byte
    {
        /// <summary>
        /// PCAN-ISA 82C200
        /// </summary>
        POBDII_HWTYPE_ISA = 0x01,
        /// <summary>
        /// PCAN-ISA SJA1000
        /// </summary>
        POBDII_HWTYPE_ISA_SJA = 0x09,
        /// <summary>
        /// PHYTEC ISA 
        /// </summary>
        POBDII_HWTYPE_ISA_PHYTEC = 0x04,
        /// <summary>
        /// PCAN-Dongle 82C200
        /// </summary>
        POBDII_HWTYPE_DNG = 0x02,
        /// <summary>
        /// PCAN-Dongle EPP 82C200
        /// </summary>
        POBDII_HWTYPE_DNG_EPP = 0x03,
        /// <summary>
        /// PCAN-Dongle SJA1000
        /// </summary>
        POBDII_HWTYPE_DNG_SJA = 0x05,
        /// <summary>
        /// PCAN-Dongle EPP SJA1000
        /// </summary>
        POBDII_HWTYPE_DNG_SJA_EPP = 0x06,
    }

    /// <summary>
    ///  Represents a OBDII status/error code
    /// </summary>
    public enum TPOBDIIStatus : uint
    {
        /// <summary>
        /// No error 
        /// </summary>
        POBDII_ERROR_OK = 0x00000,
        /// <summary>
        /// Not Initialized
        /// </summary>
        POBDII_ERROR_NOT_INITIALIZED = 0x00001,
        /// <summary>
        /// Already Initialized
        /// </summary>
        POBDII_ERROR_ALREADY_INITIALIZED = 0x00002,
        /// <summary>
        /// Could not obtain memory
        /// </summary>
        POBDII_ERROR_NO_MEMORY = 0x00003,
        /// <summary>
        /// Input buffer overflow
        /// </summary>
        POBDII_ERROR_OVERFLOW = 0x00004,
        /// <summary>
        /// Timeout while accessing the UDS mutex
        /// </summary>
        POBDII_ERROR_TIMEOUT = 0x00006,
        /// <summary>
        /// No Message available
        /// </summary>
        POBDII_ERROR_NO_MESSAGE = 0x00007,
        /// <summary>
        /// Wrong parameters in query
        /// </summary>
        POBDII_ERROR_WRONG_PARAM = 0x00008,
        /// <summary>
        /// Buffer to parse contains response to Vehicle specific parameter
        /// </summary>
        POBDII_ERROR_NOT_SUPPORTED = 0x00009,
        /// <summary>
        /// Could not parse response
        /// </summary>
        POBDII_ERROR_PARSE_ERROR = 0x0000A,
        /// <summary>
        /// Channel is in BUS-LIGHT error state
        /// </summary>
        POBDII_ERROR_BUSLIGHT = 0x0000B,
        /// <summary>
        /// Channel is in BUS-HEAVY error state
        /// </summary>
        POBDII_ERROR_BUSHEAVY = 0x0000C,
        /// <summary>
        /// Channel is in BUS-OFF error state
        /// </summary>
        POBDII_ERROR_BUSOFF = 0x0000D,
        /// <summary>
        /// No connected ECUs, or ECU found is not supported
        /// </summary>
        POBDII_ERROR_UNSUPPORTED_ECUS = 0x0000E,
        /// <summary>
        /// 0x10000 .. 0x1ffff CAN System dependend (PCAN-Basic). The value must 
        /// be shifted 2 bytes to the right, then it is a PCAN-Basic Status value.
        /// </summary>
        POBDII_ERROR_CAN_ERROR = 0x10000,
    }

    /// <summary>
    /// Represents the OBD-II Response codes
    /// </summary>
    public enum TPOBDIIError : byte
    {
        /// <summary>
        /// No error, Positive response
        /// </summary>
        POBDII_R_NO_ERROR = 0,
        /// <summary>
        /// Server is busy
        /// </summary>
        POBDII_R_BUSY_REPEAT_REQUEST = 0x21,
        /// <summary>
        /// Conditions not correct
        /// </summary>
        POBDII_R_CONDITIONS_NOT_CORRECT = 0x22,
        /// <summary>
        /// server needs more time
        /// </summary>
        POBDII_R_RESPONSE_PENDING = 0x78,
        /// <summary>
        /// Not a response, invalid value
        /// </summary>
        POBDII_R_NOT_USED = 0xFF,
    }

    /// <summary>
    /// Represents the type of returned data
    /// </summary>
    public enum TPOBDIIInfoDataType : byte
    {
        /// <summary>
        /// WORD sized data
        /// </summary>
        POBDII_INFOTYPE_COUNTER = 0x00,
        /// <summary>
        /// 4 Byte long calibration data
        /// </summary>
        POBDII_INFOTYPE_CALDATA = 0x01,
        /// <summary>
        /// Char buffer containing up to 20 characters
        /// </summary>
        POBDII_INFOTYPE_STRING = 0x02,
        /// <summary>
        /// Data is not an InfoType (used to identify raw data bit encoded InfoType Support)
        /// </summary>
        POBDII_INFOTYPE_NONE = 0x03
    }
    #endregion

    #region Structures

    /// <summary>
    /// Represents the data returned by OBD service $04, $08
    /// </summary>
	[StructLayout(LayoutKind.Sequential, Pack = 8)]
    public struct TPOBDIIResponse
    {
        /// <summary>
        /// ECU number
        /// </summary>
        public byte SOURCE;
        /// <summary>
        /// Error number (POBDII_R_*)
        /// </summary>
        [MarshalAs(UnmanagedType.U1)]
        public TPOBDIIError ERRORNR;
        /// <summary>
        /// Raw data (without SID and Parameters)
        /// </summary>
        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 4095)]
        public byte[] DATA;
        /// <summary>
        /// Number of bytes stored in DATA
        /// </summary>
        public ushort LEN;
        /// <summary>
        /// Service ID
        /// </summary>
        public byte SID;
        /// <summary>
        /// Parameter ID (only with services $01, $02, $08, $09)
        /// </summary>
        public byte PID;
        /// <summary>
        /// Frame number (only with OBD Service $02)
        /// </summary>
        public byte FRAME;
    }

    /// <summary>
    /// Represents the data returned by OBD service $01, $02
    /// </summary>
	[StructLayout(LayoutKind.Sequential, Pack = 8)]
    public struct TPOBDIIParamData
    {
        /// <summary>
        /// Response definition data
        /// </summary>
        [MarshalAs(UnmanagedType.Struct)]
        public TPOBDIIResponse RESPONSE;
        /// <summary>
        /// Raw data (without SID, PID and Frame)
        /// </summary>
        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 41)]
        public byte[] BUFFER;
        /// <summary>
        /// Up to 10 double values
        /// </summary>
        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 10)]
        public double[] DOUBLES;
        /// <summary>
        /// Mask for bytes that are bit encoded: A(0) to J(9)
        /// </summary>
        public ushort BYTEMASK;
        /// <summary>
        /// Number of bytes stored in raw buffer (0 to 41)
        /// </summary>
        public byte BLEN;
        /// <summary>
        /// Number of doubles stored (0 to 10)
        /// </summary>
        public byte DLEN;
		/// <summary>
		/// A single DTC (incl. '\0')
		/// </summary>
		[MarshalAs(UnmanagedType.ByValTStr, SizeConst = 6)]
        public string DTC;
    }

    /// <summary>
    /// Represents a DTC as a string of 6 bytes
    /// </summary>
	[StructLayout(LayoutKind.Sequential, Pack = 8)]
    public struct TPOBDIIDTCText
    {
        [MarshalAs(UnmanagedType.ByValTStr, SizeConst = 6)]
        public string ErrorText;
    }

    /// <summary>
    /// Represents the data returned by OBD service $03, $07, $0A
    /// </summary>
	[StructLayout(LayoutKind.Sequential, Pack = 8)]
    public struct TPOBDIIDTCData
    {
        /// <summary>
        /// Response definition data
        /// </summary>
        [MarshalAs(UnmanagedType.Struct)]
        public TPOBDIIResponse RESPONSE;
        /// <summary>
        /// 10 DTCs each is 6 bytes (incl. '\0')
        /// </summary>
        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 10)]
        public TPOBDIIDTCText[] DTC;
        /// <summary>
        /// Number of DTCs stored (0 to 10)
        /// </summary>
        public byte DLEN;
    }

    /// <summary>
    /// Represents the data returned by OBD service $06
    /// </summary>
	[StructLayout(LayoutKind.Sequential, Pack = 8)]
    public struct TPOBDIIMonitorData
    {
        /// <summary>
        /// Response definition data
        /// </summary>
        [MarshalAs(UnmanagedType.Struct)]
        public TPOBDIIResponse RESPONSE;
        /// <summary>
        /// Test ID
        /// </summary>
        public byte TID;
        /// <summary>
        /// Unit And Scaling ID
        /// </summary>
        public byte UNITANDSCALING;
        /// <summary>
        /// Test value
        /// </summary>
        public double TESTVALUE;
        /// <summary>
        /// Minimum Test Limit
        /// </summary>
        public double MINLIMIT;
        /// <summary>
        /// Maximum Test Limit
        /// </summary>
        public double MAXLIMIT;
    }

    /// <summary>
    /// Represents the data returned by OBD service $09
    /// </summary>
	[StructLayout(LayoutKind.Sequential, Pack = 8)]
    public struct TPOBDIIInfoData
    {
        /// <summary>
        /// Response definition data
        /// </summary>
        [MarshalAs(UnmanagedType.Struct)]
        public TPOBDIIResponse RESPONSE;
        /// <summary>
        /// Index Number of the counter or the string
        /// </summary>
        public byte INDEXNR;
        /// <summary>
        /// Type of data available (POBDII_INFOTYPE_*)
        /// </summary>
        [MarshalAs(UnmanagedType.U1)]
        public TPOBDIIInfoDataType DATATYPE;
        /// <summary>
        /// Counter value
        /// </summary>
        public ushort COUNTER;
        /// <summary>
        /// Calibration data
        /// </summary>
        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 4)]
        public byte[] CALDATA;
        /// <summary>
        /// String value
        /// </summary>
        [MarshalAs(UnmanagedType.ByValTStr, SizeConst = 21)]
        public string TEXT;
    }

    /// <summary>
    /// Unit and Scaling definition
    /// </summary>
    [StructLayout(LayoutKind.Sequential, Pack = 8)]
    public struct TPOBDIIUnitAndScaling
    {
        /// <summary>
        /// Minimum value
        /// </summary>
        public double MIN;
        /// <summary>
        /// Maximum value
        /// </summary>
        public double MAX;
        /// <summary>
        /// Unit abbreviation
        /// </summary>
		[MarshalAs(UnmanagedType.ByValTStr, SizeConst = 16)]
        public String UNIT;
	};
    #endregion

    public static class OBDIIApi
    {
        #region PCAN-BUS Handles Definition
        /// <summary>
        /// Undefined/default value for a PCAN bus
        /// </summary>
        public const TPOBDIICANHandle POBDII_NONEBUS = 0x00;

        /// <summary>
        /// PCAN-ISA interface, channel 1
        /// </summary>
        public const TPOBDIICANHandle POBDII_ISABUS1 = 0x21;
        /// <summary>
        /// PCAN-ISA interface, channel 2
        /// </summary>
        public const TPOBDIICANHandle POBDII_ISABUS2 = 0x22;
        /// <summary>
        /// PCAN-ISA interface, channel 3
        /// </summary>
        public const TPOBDIICANHandle POBDII_ISABUS3 = 0x23;
        /// <summary>
        /// PCAN-ISA interface, channel 4
        /// </summary>
        public const TPOBDIICANHandle POBDII_ISABUS4 = 0x24;
        /// <summary>
        /// PCAN-ISA interface, channel 5
        /// </summary>
        public const TPOBDIICANHandle POBDII_ISABUS5 = 0x25;
        /// <summary>
        /// PCAN-ISA interface, channel 6
        /// </summary>
        public const TPOBDIICANHandle POBDII_ISABUS6 = 0x26;
        /// <summary>
        /// PCAN-ISA interface, channel 7
        /// </summary>
        public const TPOBDIICANHandle POBDII_ISABUS7 = 0x27;
        /// <summary>
        /// PCAN-ISA interface, channel 8
        /// </summary>
        public const TPOBDIICANHandle POBDII_ISABUS8 = 0x28;

        /// <summary>
        /// PPCAN-Dongle/LPT interface, channel 1 
        /// </summary>
        public const TPOBDIICANHandle POBDII_DNGBUS1 = 0x31;

        /// <summary>
        /// PCAN-PCI interface, channel 1
        /// </summary>
        public const TPOBDIICANHandle POBDII_PCIBUS1 = 0x41;
        /// <summary>
        /// PCAN-PCI interface, channel 2
        /// </summary>
        public const TPOBDIICANHandle POBDII_PCIBUS2 = 0x42;
        /// <summary>
        /// PCAN-PCI interface, channel 3
        /// </summary>
        public const TPOBDIICANHandle POBDII_PCIBUS3 = 0x43;
        /// <summary>
        /// PCAN-PCI interface, channel 4
        /// </summary>
        public const TPOBDIICANHandle POBDII_PCIBUS4 = 0x44;
        /// <summary>
        /// PCAN-PCI interface, channel 5
        /// </summary>
        public const TPOBDIICANHandle POBDII_PCIBUS5 = 0x45;
        /// <summary>
        /// PCAN-PCI interface, channel 6
        /// </summary>
        public const TPOBDIICANHandle POBDII_PCIBUS6 = 0x46;
        /// <summary>
        /// PCAN-PCI interface, channel 7
        /// </summary>
        public const TPOBDIICANHandle POBDII_PCIBUS7 = 0x47;
        /// <summary>
        /// PCAN-PCI interface, channel 8
        /// </summary>
        public const TPOBDIICANHandle POBDII_PCIBUS8 = 0x48;
        /// <summary>
        /// PCAN-PCI interface, channel 9
        /// </summary>
        public const TPOBDIICANHandle POBDII_PCIBUS9 = 0x409;
        /// <summary>
        /// PCAN-PCI interface, channel 10
        /// </summary>
        public const TPOBDIICANHandle POBDII_PCIBUS10 = 0x40A;
        /// <summary>
        /// PCAN-PCI interface, channel 11
        /// </summary>
        public const TPOBDIICANHandle POBDII_PCIBUS11 = 0x40B;
        /// <summary>
        /// PCAN-PCI interface, channel 12
        /// </summary>
        public const TPOBDIICANHandle POBDII_PCIBUS12 = 0x40C;
        /// <summary>
        /// PCAN-PCI interface, channel 13
        /// </summary>
        public const TPOBDIICANHandle POBDII_PCIBUS13 = 0x40D;
        /// <summary>
        /// PCAN-PCI interface, channel 14
        /// </summary>
        public const TPOBDIICANHandle POBDII_PCIBUS14 = 0x40E;
        /// <summary>
        /// PCAN-PCI interface, channel 15
        /// </summary>
        public const TPOBDIICANHandle POBDII_PCIBUS15 = 0x40F;
        /// <summary>
        /// PCAN-PCI interface, channel 16
        /// </summary>
        public const TPOBDIICANHandle POBDII_PCIBUS16 = 0x410;

        /// <summary>
        /// PCAN-USB interface, channel 1
        /// </summary>
        public const TPOBDIICANHandle POBDII_USBBUS1 = 0x51;
        /// <summary>
        /// PCAN-USB interface, channel 2
        /// </summary>
        public const TPOBDIICANHandle POBDII_USBBUS2 = 0x52;
        /// <summary>
        /// PCAN-USB interface, channel 3
        /// </summary>
        public const TPOBDIICANHandle POBDII_USBBUS3 = 0x53;
        /// <summary>
        /// PCAN-USB interface, channel 4
        /// </summary>
        public const TPOBDIICANHandle POBDII_USBBUS4 = 0x54;
        /// <summary>
        /// PCAN-USB interface, channel 5
        /// </summary>
        public const TPOBDIICANHandle POBDII_USBBUS5 = 0x55;
        /// <summary>
        /// PCAN-USB interface, channel 6
        /// </summary>
        public const TPOBDIICANHandle POBDII_USBBUS6 = 0x56;
        /// <summary>
        /// PCAN-USB interface, channel 7
        /// </summary>
        public const TPOBDIICANHandle POBDII_USBBUS7 = 0x57;
        /// <summary>
        /// PCAN-USB interface, channel 8
        /// </summary>
        public const TPOBDIICANHandle POBDII_USBBUS8 = 0x58;
        /// <summary>
        /// PCAN-USB interface, channel 9
        /// </summary>
        public const TPOBDIICANHandle POBDII_USBBUS9 = 0x509;
        /// <summary>
        /// PCAN-USB interface, channel 10
        /// </summary>
        public const TPOBDIICANHandle POBDII_USBBUS10 = 0x50A;
        /// <summary>
        /// PCAN-USB interface, channel 11
        /// </summary>
        public const TPOBDIICANHandle POBDII_USBBUS11 = 0x50B;
        /// <summary>
        /// PCAN-USB interface, channel 12
        /// </summary>
        public const TPOBDIICANHandle POBDII_USBBUS12 = 0x50C;
        /// <summary>
        /// PCAN-USB interface, channel 13
        /// </summary>
        public const TPOBDIICANHandle POBDII_USBBUS13 = 0x50D;
        /// <summary>
        /// PCAN-USB interface, channel 14
        /// </summary>
        public const TPOBDIICANHandle POBDII_USBBUS14 = 0x50E;
        /// <summary>
        /// PCAN-USB interface, channel 15
        /// </summary>
        public const TPOBDIICANHandle POBDII_USBBUS15 = 0x50F;
        /// <summary>
        /// PCAN-USB interface, channel 16
        /// </summary>
        public const TPOBDIICANHandle POBDII_USBBUS16 = 0x510;

        /// <summary>
        /// PCAN-PC Card interface, channel 1
        /// </summary>
        public const TPOBDIICANHandle POBDII_PCCBUS1 = 0x61;
        /// <summary>
        /// PCAN-PC Card interface, channel 2
        /// </summary>
        public const TPOBDIICANHandle POBDII_PCCBUS2 = 0x62;

        /// <summary>
        /// PCAN-LAN interface, channel 1
        /// </summary>
        public const TPOBDIICANHandle POBDII_LANBUS1 = 0x801;
        /// <summary>
        /// PCAN-LAN interface, channel 2
        /// </summary>
        public const TPOBDIICANHandle POBDII_LANBUS2 = 0x802;
        /// <summary>
        /// PCAN-LAN interface, channel 3
        /// </summary>
        public const TPOBDIICANHandle POBDII_LANBUS3 = 0x803;
        /// <summary>
        /// PCAN-LAN interface, channel 4
        /// </summary>
        public const TPOBDIICANHandle POBDII_LANBUS4 = 0x804;
        /// <summary>
        /// PCAN-LAN interface, channel 5
        /// </summary>
        public const TPOBDIICANHandle POBDII_LANBUS5 = 0x805;
        /// <summary>
        /// PCAN-LAN interface, channel 6
        /// </summary>
        public const TPOBDIICANHandle POBDII_LANBUS6 = 0x806;
        /// <summary>
        /// PCAN-LAN interface, channel 7
        /// </summary>
        public const TPOBDIICANHandle POBDII_LANBUS7 = 0x807;
        /// <summary>
        /// PCAN-LAN interface, channel 8
        /// </summary>
        public const TPOBDIICANHandle POBDII_LANBUS8 = 0x808;
        /// <summary>
        /// PCAN-LAN interface, channel 9
        /// </summary>
        public const TPOBDIICANHandle POBDII_LANBUS9 = 0x809;
        /// <summary>
        /// PCAN-LAN interface, channel 10
        /// </summary>
        public const TPOBDIICANHandle POBDII_LANBUS10 = 0x80A;
        /// <summary>
        /// PCAN-LAN interface, channel 11
        /// </summary>
        public const TPOBDIICANHandle POBDII_LANBUS11 = 0x80B;
        /// <summary>
        /// PCAN-LAN interface, channel 12
        /// </summary>
        public const TPOBDIICANHandle POBDII_LANBUS12 = 0x80C;
        /// <summary>
        /// PCAN-LAN interface, channel 13
        /// </summary>
        public const TPOBDIICANHandle POBDII_LANBUS13 = 0x80D;
        /// <summary>
        /// PCAN-LAN interface, channel 14
        /// </summary>
        public const TPOBDIICANHandle POBDII_LANBUS14 = 0x80E;
        /// <summary>
        /// PCAN-LAN interface, channel 15
        /// </summary>
        public const TPOBDIICANHandle POBDII_LANBUS15 = 0x80F;
        /// <summary>
        /// PCAN-LAN interface, channel 16
        /// </summary>
        public const TPOBDIICANHandle POBDII_LANBUS16 = 0x810;
        #endregion

        #region Parameter values definition
        /// <summary>
        /// No debug messages
        /// </summary>
        public const byte POBDII_DEBUG_NONE = 0;
        /// <summary>
        /// Puts CAN debug messages to stdout
        /// </summary>
        public const byte POBDII_DEBUG_CAN = 1;
        /// <summary>
        /// No logging
        /// </summary>
        public const byte POBDII_LOGGING_NONE = 0;
        /// <summary>
        /// Log function enter/exit data to file
        /// </summary>
        public const byte POBDII_LOGGING_TO_FILE = 0x01;
        /// <summary>
        /// Log function enter/exit data to stdout
        /// </summary>
        public const byte POBDII_LOGGING_TO_STDOUT = 0x02;
        /// <summary>
        /// The Channel is illegal or not available
        /// </summary>
        public const byte POBDII_CHANNEL_UNAVAILABLE = 0x00;   
        /// <summary>
        /// The Channel is available
        /// </summary>
        public const byte POBDII_CHANNEL_AVAILABLE = 0x01;
        /// <summary>
        /// The Channel is valid, and is being used
        /// </summary>
        public const byte POBDII_CHANNEL_OCCUPIED = 0x02;        
        /// <summary>
        /// 11 Bit CAN Identifier
        /// </summary>
        public const byte POBDII_CAN_ID_11BIT = 11;
        /// <summary>
        /// 29 Bit CAN Identifier
        /// </summary>
        public const byte POBDII_CAN_ID_29BIT = 29;
        #endregion

        #region POBDII API Implementation
        #region Initialization and Configuration
        /// <summary>
        /// Initializes an OBDII-Client based on a CAN-Channel
        /// </summary>
        /// <remarks>Only one OBDII-Client can be initialized per CAN-Channel</remarks>
        /// <param name="CanChannel">The PCAN-Basic channel to be used as OBDII client</param>
        /// <param name="HwType">NON PLUG&PLAY: The type of hardware and operation mode</param>
        /// <param name="IOPort">NON PLUG&PLAY: The I/O address for the parallel port</param>
        /// <param name="Interupt">NON PLUG&PLAY: Interrupt number of the parallel port</param>
        /// <returns>A TPOBDIIStatus code. POBDII_ERROR_OK is returned on success</returns>
        [DllImport("PCAN-OBDII.dll", EntryPoint = "OBDII_Initialize")]
        public static extern TPOBDIIStatus Initialize(
            [MarshalAs(UnmanagedType.U2)]
            TPOBDIICANHandle CanChannel,
            [MarshalAs(UnmanagedType.U1)]
			TPOBDIIBaudrateInfo Baudrate,
            [MarshalAs(UnmanagedType.U1)]
            TPOBDIIHwType HwType, 
            UInt32 IOPort, 
            UInt16 Interrupt);

        /// <summary>
        /// Initializes an OBDII-Client based on a CAN-Channel
        /// </summary>
        /// <remarks>Only one OBDII-Client can be initialized per CAN-Channel</remarks>
        /// <param name="CanChannel">The PCAN-Basic channel to be used as OBDII client</param>
        /// <returns>A TPOBDIIStatus code. POBDII_ERROR_OK is returned on success</returns>
        public static TPOBDIIStatus Initialize(
            TPOBDIICANHandle CanChannel)
        {
            return Initialize(CanChannel, TPOBDIIBaudrateInfo.POBDII_BAUDRATE_AUTODETECT, (TPOBDIIHwType)0, 0, 0);
        }

        /// <summary>
        /// Uninitializes an OBDII-Client initialized before
        /// </summary>
        /// <param name="CanChannel">An OBDII CAN-Channel representing a OBDII-Client</param>
        /// <returns>A TPOBDIIStatus code. POBDII_ERROR_OK is returned on success</returns>
        [DllImport("PCAN-OBDII.dll", EntryPoint = "OBDII_Uninitialize")]
        public static extern TPOBDIIStatus Uninitialize(
            [MarshalAs(UnmanagedType.U2)]
            TPOBDIICANHandle CanChannel);

        /// <summary>
        /// Resets the receive and transmit queues of an OBDII-Client 
        /// </summary>
        /// <param name="CanChannel">An OBDII CAN-Channel representing a OBDII-Client</param>
        /// <returns>A TPOBDIIStatus code. POBDII_ERROR_OK is returned on success</returns>
        [DllImport("PCAN-OBDII.dll", EntryPoint = "OBDII_Reset")]
        public static extern TPOBDIIStatus Reset(
            [MarshalAs(UnmanagedType.U2)]
            TPOBDIICANHandle CanChannel);

        /// <summary>
        /// Gets the initialization status of an OBDII CAN-Channel
        /// </summary>
        /// <param name="CanChannel">An OBDII CAN-Channel representing a OBDII-Client</param>
        /// <returns>A TPOBDIIStatus code. POBDII_ERROR_OK is returned on success</returns>
        [DllImport("PCAN-OBDII.dll", EntryPoint = "OBDII_GetStatus")]
        public static extern TPOBDIIStatus GetStatus(
            [MarshalAs(UnmanagedType.U2)]
            TPOBDIICANHandle CanChannel);

        /// <summary>
        /// Retrieves an OBDII-Client value
        /// </summary>
        /// <param name="CanChannel">An OBDII CAN-Channel representing a OBDII-Client</param>
        /// <param name="Parameter">The TPOBDIIParameter parameter to get</param>
        /// <param name="StringBuffer">Buffer for the parameter value</param>
        /// <param name="BufferLength">Size in bytes of the buffer</param>
        /// <returns>A TPOBDIIStatus code. POBDII_ERROR_OK is returned on success</returns>
        [DllImport("PCAN-OBDII.dll", EntryPoint = "OBDII_GetValue")]
        public static extern TPOBDIIStatus GetValue(
            [MarshalAs(UnmanagedType.U2)]
            TPOBDIICANHandle CanChannel,
            [MarshalAs(UnmanagedType.U1)]
            TPOBDIIParameter Parameter,
            StringBuilder StringBuffer,
            UInt32 BufferLength);
        /// <summary>
        /// Retrieves an OBDII-Client value
        /// </summary>
        /// <param name="CanChannel">An OBDII CAN-Channel representing a OBDII-Client</param>
        /// <param name="Parameter">The TPOBDIIParameter parameter to get</param>
        /// <param name="NumericBuffer">Buffer for the parameter value</param>
        /// <param name="BufferLength">Size in bytes of the buffer</param>
        /// <returns>A TPOBDIIStatus code. POBDII_ERROR_OK is returned on success</returns>
        [DllImport("PCAN-OBDII.dll", EntryPoint = "OBDII_GetValue")]
        public static extern TPOBDIIStatus GetValue(
            [MarshalAs(UnmanagedType.U2)]
            TPOBDIICANHandle CanChannel,
            [MarshalAs(UnmanagedType.U1)]
            TPOBDIIParameter Parameter,
            out UInt32 NumericBuffer,
            UInt32 BufferLength);
        /// <summary>
        /// Retrieves an OBDII-Client value
        /// </summary>
        /// <param name="CanChannel">An OBDII CAN-Channel representing a OBDII-Client</param>
        /// <param name="Parameter">The TPOBDIIParameter parameter to get</param>
        /// <param name="Buffer">Buffer for the parameter value</param>
        /// <param name="BufferLength">Size in bytes of the buffer</param>
        /// <returns>A TPOBDIIStatus code. POBDII_ERROR_OK is returned on success</returns>
        [DllImport("PCAN-OBDII.dll", EntryPoint = "OBDII_GetValue")]
        public static extern TPOBDIIStatus GetValue(
            [MarshalAs(UnmanagedType.U2)]
            TPOBDIICANHandle CanChannel,
            [MarshalAs(UnmanagedType.U1)]
            TPOBDIIParameter Parameter,
            [MarshalAs(UnmanagedType.LPArray)]
			[Out] Byte[] Buffer,
            UInt32 BufferLength);

        /// <summary>
        /// Configures or sets an OBDII-Client value
        /// </summary>
        /// <param name="CanChannel">An OBDII CAN-Channel representing a OBDII-Client</param>
        /// <param name="Parameter">The TPOBDIIParameter parameter to set</param>
        /// <param name="NumericBuffer">Buffer with the value to be set</param>
        /// <param name="BufferLength">Size in bytes of the buffer</param>
        /// <returns>A TPOBDIIStatus code. POBDII_ERROR_OK is returned on success</returns>
        [DllImport("PCAN-OBDII.dll", EntryPoint = "OBDII_SetValue")]
        public static extern TPOBDIIStatus SetValue(
            [MarshalAs(UnmanagedType.U2)]
            TPOBDIICANHandle CanChannel,
            [MarshalAs(UnmanagedType.U1)]
            TPOBDIIParameter Parameter,
            ref UInt32 NumericBuffer,
            UInt32 BufferLength);

        /// <summary>
        /// Returns the unit and scaling information for a specified Unit and Scaling ID
        /// </summary>
        /// <param name="id">Unit and Scaling ID</param>
        /// <param name="id">Structure to store the result</param>
        /// <returns>A TPOBDIIStatus code. POBDII_ERROR_OK is returned on success</returns>
        [DllImport("PCAN-OBDII.dll", EntryPoint = "OBDII_GetUnitAndScaling")]
        public static extern TPOBDIIStatus GetUnitAndScaling(
            Byte id, 
            out TPOBDIIUnitAndScaling unitAndScaling);
        #endregion

        #region Services Implementation
        #region Service 01h: RequestCurrentData
        /// <summary>
        /// Sends an OBDII Service $01 request into queue and waits for the received response
        /// </summary>
        /// <param name="CanChannel">An OBDII CAN-Channel representing a OBDII-Client</param>
        /// <param name="Pid">A Parameter Identifier</param>
        /// <param name="Data">A TPOBDIIParamData structure array to store the OBDII responses</param>
        /// <param name="DataLen">A length of provided Data array</param>
        /// <returns>A TPOBDIIStatus code. POBDII_ERROR_OK is returned on success</returns>
        [DllImport("PCAN-OBDII.dll", EntryPoint = "OBDII_RequestCurrentData")]
        public static extern TPOBDIIStatus RequestCurrentData(
            [MarshalAs(UnmanagedType.U2)]
            TPOBDIICANHandle CanChannel,
            [MarshalAs(UnmanagedType.U1)]
            TPOBDIIPid Pid,
            [In, Out]
            TPOBDIIParamData[] Data,
            byte DataLen);
        #endregion

        #region Service 02h: RequestFreezeFrameData
        /// <summary>
        /// Sends an OBDII Service $02 request into queue and waits for the received response
        /// </summary>
        /// <param name="CanChannel">An OBDII CAN-Channel representing a OBDII-Client</param>
        /// <param name="Pid">A Parameter Identifier</param>
        /// <param name="Frame">A Frame number</param>
        /// <param name="Data">A TPOBDIIParamData structure array to store the OBDII responses</param>
        /// <param name="DataLen">A length of provided Data array</param>
        /// <returns>A TPOBDIIStatus code. POBDII_ERROR_OK is returned on success</returns>
        [DllImport("PCAN-OBDII.dll", EntryPoint = "OBDII_RequestFreezeFrameData")]
        public static extern TPOBDIIStatus RequestFreezeFrameData(
            [MarshalAs(UnmanagedType.U2)]
            TPOBDIICANHandle CanChannel,
            [MarshalAs(UnmanagedType.U1)]
            TPOBDIIPid Pid,
            byte Frame,
            [In, Out]
            TPOBDIIParamData[] Data,
            byte DataLen);
        #endregion

        #region Service 03h: RequestStoredTroubleCodes
        /// <summary>
        /// Sends an OBDII Service $03 request into queue and waits for the received response
        /// </summary>
        /// <param name="CanChannel">An OBDII CAN-Channel representing a OBDII-Client</param>
        /// <param name="Data">A TPOBDIIDTCData structure array to store the OBDII responses</param>
        /// <param name="DataLen">A length of provided Data array</param>
        /// <returns>A TPOBDIIStatus code. POBDII_ERROR_OK is returned on success</returns>
        [DllImport("PCAN-OBDII.dll", EntryPoint = "OBDII_RequestStoredTroubleCodes")]
        public static extern TPOBDIIStatus RequestStoredTroubleCodes(
            [MarshalAs(UnmanagedType.U2)]
            TPOBDIICANHandle CanChannel,
            [In, Out]
            TPOBDIIDTCData[] Data,
            byte DataLen);
        #endregion

        #region Service 04h: ClearTroubleCodes
        /// <summary>
        /// Sends an OBDII Service $04 request into queue and waits for the received response
        /// </summary>
        /// <param name="CanChannel">An OBDII CAN-Channel representing a OBDII-Client</param>
        /// <param name="Data">A TPOBDIIResponse structure array to store the OBDII responses</param>
        /// <param name="DataLen">A length of provided Data array</param>
        /// <returns>A TPOBDIIStatus code. POBDII_ERROR_OK is returned on success</returns>
        [DllImport("PCAN-OBDII.dll", EntryPoint = "OBDII_ClearTroubleCodes")]
        public static extern TPOBDIIStatus ClearTroubleCodes(
            [MarshalAs(UnmanagedType.U2)]
            TPOBDIICANHandle CanChannel,
            [In, Out]
            TPOBDIIResponse[] Data,
            byte DataLen);
        #endregion

        #region Service 06h: RequestTestResults
        /// <summary>
        /// Sends an OBDII Service $06 request into queue and waits for the received response
        /// </summary>
        /// <param name="CanChannel">An OBDII CAN-Channel representing a OBDII-Client</param>
        /// <param name="OBDMid">An On-Board Diagnostic Monitor Identifier</param>
        /// <param name="Data">A TPOBDIIOBDMid structure array to store the OBDII responses</param>
        /// <param name="DataLen">A length of provided Data array</param>
        /// <returns>A TPOBDIIStatus code. POBDII_ERROR_OK is returned on success</returns>
        [DllImport("PCAN-OBDII.dll", EntryPoint = "OBDII_RequestTestResults")]
        public static extern TPOBDIIStatus RequestTestResults(
            [MarshalAs(UnmanagedType.U2)]
            TPOBDIICANHandle CanChannel,
            [MarshalAs(UnmanagedType.U1)]
            TPOBDIIOBDMid OBDMid,
            [In, Out]
            TPOBDIIMonitorData[] Data,
            byte DataLen);
        #endregion

        #region Service 07h: RequestPendingTroubleCodes
        /// <summary>
        /// Sends an OBDII Service $07 request into queue and waits for the received response
        /// </summary>
        /// <param name="CanChannel">An OBDII CAN-Channel representing a OBDII-Client</param>
        /// <param name="Data">A TPOBDIIDTCData structure array to store the OBDII responses</param>
        /// <param name="DataLen">A length of provided Data array</param>
        /// <returns>A TPOBDIIStatus code. POBDII_ERROR_OK is returned on success</returns>
        [DllImport("PCAN-OBDII.dll", EntryPoint = "OBDII_RequestPendingTroubleCodes")]
        public static extern TPOBDIIStatus RequestPendingTroubleCodes(
            [MarshalAs(UnmanagedType.U2)]
            TPOBDIICANHandle CanChannel,
            [In, Out]
            TPOBDIIDTCData[] Data,
            byte DataLen);
        #endregion

        #region Service 08h: RequestControlOperation
        /// <summary>
        /// Sends an OBDII Service $08 request into queue and waits for the received response
        /// </summary>
        /// <param name="CanChannel">An OBDII CAN-Channel representing a OBDII-Client</param>
        /// <param name="Tid">A Test Identifier</param>
        /// <param name="Data">A TPOBDIIResponse structure array to store the OBDII responses</param>
        /// <param name="DataLen">A length of provided Data array</param>
        /// <returns>A TPOBDIIStatus code. POBDII_ERROR_OK is returned on success</returns>
        [DllImport("PCAN-OBDII.dll", EntryPoint = "OBDII_RequestControlOperation")]
        public static extern TPOBDIIStatus RequestControlOperation(
            [MarshalAs(UnmanagedType.U2)]
            TPOBDIICANHandle CanChannel,
            [MarshalAs(UnmanagedType.U1)]
            TPOBDIITid Tid,
            [In, Out]
            TPOBDIIResponse[] Data,
            byte DataLen);
        #endregion

        #region Service 09h: RequestVehicleInformation
        /// <summary>
        /// Sends an OBDII Service $09 request into queue and waits for the received response
        /// </summary>
        /// <param name="CanChannel">An OBDII CAN-Channel representing a OBDII-Client</param>
        /// <param name="InfoType">An InfoType</param>
        /// <param name="Data">A TPOBDIIInfoData structure array to store the OBDII responses</param>
        /// <param name="DataLen">A length of provided Data array</param>
        /// <returns>A TPOBDIIStatus code. POBDII_ERROR_OK is returned on success</returns>
        [DllImport("PCAN-OBDII.dll", EntryPoint = "OBDII_RequestVehicleInformation")]
        public static extern TPOBDIIStatus RequestVehicleInformation(
            [MarshalAs(UnmanagedType.U2)]
            TPOBDIICANHandle CanChannel,
            [MarshalAs(UnmanagedType.U1)]
            TPOBDIIInfoType InfoType,
            [In, Out]
            TPOBDIIInfoData[] Data,
            byte DataLen);
        #endregion

        #region Service 0Ah: RequestPermanentTroubleCodes
        /// <summary>
        /// Sends an OBDII Service $0A request into queue and waits for the received response
        /// </summary>
        /// <param name="CanChannel">An OBDII CAN-Channel representing a OBDII-Client</param>
        /// <param name="Data">A TPOBDIIDTCData structure array to store the OBDII responses</param>
        /// <param name="DataLen">A length of provided Data array</param>
        /// <returns>A TPOBDIIStatus code. POBDII_ERROR_OK is returned on success</returns>
        [DllImport("PCAN-OBDII.dll", EntryPoint = "OBDII_RequestPermanentTroubleCodes")]
        public static extern TPOBDIIStatus RequestPermanentTroubleCodes(
            [MarshalAs(UnmanagedType.U2)]
            TPOBDIICANHandle CanChannel,
            [In, Out]
            TPOBDIIDTCData[] Data,
            byte DataLen);
        #endregion
        #endregion
        #endregion
    }
}