//  PCAN-OBDII.h
//
//  ~~~~~~~~~~~~
//
//  PCAN OBD-II API
//
//  ~~~~~~~~~~~~
//
//  ------------------------------------------------------------------
//  Author : Aleksandr Sarviro (ESS Embedded Systems Solutions GmbH)
//           Keneth Wagner-Rivas (PEAK-System technik GmbH)
//	Last changed by:	$Author: Fabrice $
//  Last changed date:	$Date: 2023-09-15 10:45:25 +0200 (Fri, 15 Sep 2023) $
//
//  Language: ANSI-C
//  ------------------------------------------------------------------
//
//  Copyright (C) 2015  PEAK-System Technik GmbH, Darmstadt
//  more Info at http://www.peak-system.com 
//
#ifndef __POBDII__
#define __POBDII__

////////////////////////////////////////////////////////////
// Value definitions
////////////////////////////////////////////////////////////

// Currently defined and supported OBDII channels
//
#define POBDII_NONEBUS             0x00U  // Undefined/default value for a PCAN bus

#define POBDII_ISABUS1             0x21U  // PCAN-ISA interface, channel 1
#define POBDII_ISABUS2             0x22U  // PCAN-ISA interface, channel 2
#define POBDII_ISABUS3             0x23U  // PCAN-ISA interface, channel 3
#define POBDII_ISABUS4             0x24U  // PCAN-ISA interface, channel 4
#define POBDII_ISABUS5             0x25U  // PCAN-ISA interface, channel 5
#define POBDII_ISABUS6             0x26U  // PCAN-ISA interface, channel 6
#define POBDII_ISABUS7             0x27U  // PCAN-ISA interface, channel 7
#define POBDII_ISABUS8             0x28U  // PCAN-ISA interface, channel 8

#define POBDII_DNGBUS1             0x31U  // PCAN-Dongle/LPT interface, channel 1

#define POBDII_PCIBUS1             0x41U  // PCAN-PCI interface, channel 1
#define POBDII_PCIBUS2             0x42U  // PCAN-PCI interface, channel 2
#define POBDII_PCIBUS3             0x43U  // PCAN-PCI interface, channel 3
#define POBDII_PCIBUS4             0x44U  // PCAN-PCI interface, channel 4
#define POBDII_PCIBUS5             0x45U  // PCAN-PCI interface, channel 5
#define POBDII_PCIBUS6	         0x46U  // PCAN-PCI interface, channel 6
#define POBDII_PCIBUS7	         0x47U  // PCAN-PCI interface, channel 7
#define POBDII_PCIBUS8	         0x48U  // PCAN-PCI interface, channel 8
#define POBDII_PCIBUS9             0x409U  // PCAN-PCI interface, channel 9
#define POBDII_PCIBUS10            0x40AU  // PCAN-PCI interface, channel 10
#define POBDII_PCIBUS11            0x40BU  // PCAN-PCI interface, channel 11
#define POBDII_PCIBUS12            0x40CU  // PCAN-PCI interface, channel 12
#define POBDII_PCIBUS13            0x40DU  // PCAN-PCI interface, channel 13
#define POBDII_PCIBUS14	         0x40EU  // PCAN-PCI interface, channel 14
#define POBDII_PCIBUS15	         0x40FU  // PCAN-PCI interface, channel 15
#define POBDII_PCIBUS16	         0x410U  // PCAN-PCI interface, channel 16

#define POBDII_USBBUS1             0x51U  // PCAN-USB interface, channel 1
#define POBDII_USBBUS2             0x52U  // PCAN-USB interface, channel 2
#define POBDII_USBBUS3             0x53U  // PCAN-USB interface, channel 3
#define POBDII_USBBUS4             0x54U  // PCAN-USB interface, channel 4
#define POBDII_USBBUS5             0x55U  // PCAN-USB interface, channel 5
#define POBDII_USBBUS6             0x56U  // PCAN-USB interface, channel 6
#define POBDII_USBBUS7             0x57U  // PCAN-USB interface, channel 7
#define POBDII_USBBUS8             0x58U  // PCAN-USB interface, channel 8
#define POBDII_USBBUS9             0x509U  // PCAN-USB interface, channel 9
#define POBDII_USBBUS10            0x50AU  // PCAN-USB interface, channel 10
#define POBDII_USBBUS11            0x50BU  // PCAN-USB interface, channel 11
#define POBDII_USBBUS12            0x50CU  // PCAN-USB interface, channel 12
#define POBDII_USBBUS13            0x50DU  // PCAN-USB interface, channel 13
#define POBDII_USBBUS14            0x50EU  // PCAN-USB interface, channel 14
#define POBDII_USBBUS15            0x50FU  // PCAN-USB interface, channel 15
#define POBDII_USBBUS16            0x510U  // PCAN-USB interface, channel 16

#define POBDII_PCCBUS1             0x61U  // PCAN-PC Card interface, channel 1
#define POBDII_PCCBUS2             0x62U  // PCAN-PC Card interface, channel 2

#define POBDII_LANBUS1             0x801U  // PCAN-LAN interface, channel 1
#define POBDII_LANBUS2             0x802U  // PCAN-LAN interface, channel 2
#define POBDII_LANBUS3             0x803U  // PCAN-LAN interface, channel 3
#define POBDII_LANBUS4             0x804U  // PCAN-LAN interface, channel 4
#define POBDII_LANBUS5             0x805U  // PCAN-LAN interface, channel 5
#define POBDII_LANBUS6             0x806U  // PCAN-LAN interface, channel 6
#define POBDII_LANBUS7             0x807U  // PCAN-LAN interface, channel 7
#define POBDII_LANBUS8             0x808U  // PCAN-LAN interface, channel 8
#define POBDII_LANBUS9             0x809U  // PCAN-LAN interface, channel 9
#define POBDII_LANBUS10            0x80AU  // PCAN-LAN interface, channel 10
#define POBDII_LANBUS11            0x80BU  // PCAN-LAN interface, channel 11
#define POBDII_LANBUS12            0x80CU  // PCAN-LAN interface, channel 12
#define POBDII_LANBUS13            0x80DU  // PCAN-LAN interface, channel 13
#define POBDII_LANBUS14            0x80EU  // PCAN-LAN interface, channel 14
#define POBDII_LANBUS15            0x80FU  // PCAN-LAN interface, channel 15
#define POBDII_LANBUS16            0x810U  // PCAN-LAN interface, channel 16

// PCAN hardware types (TPOBDIIHwType)
//
#define POBDII_HWTYPE_ISA            0x01  // PCAN-ISA 82C200
#define POBDII_HWTYPE_ISA_SJA        0x09  // PCAN-ISA SJA1000
#define POBDII_HWTYPE_ISA_PHYTEC     0x04  // PHYTEC ISA 
#define POBDII_HWTYPE_DNG            0x02  // PCAN-Dongle 82C200
#define POBDII_HWTYPE_DNG_EPP        0x03  // PCAN-Dongle EPP 82C200
#define POBDII_HWTYPE_DNG_SJA        0x05  // PCAN-Dongle SJA1000
#define POBDII_HWTYPE_DNG_SJA_EPP    0x06  // PCAN-Dongle EPP SJA1000

// OBD-II Address of ECUs
//
#define POBDII_ECU_1 0x01  // ECU #1
#define POBDII_ECU_2 0x02  // ECU #2
#define POBDII_ECU_3 0x03  // ECU #3
#define POBDII_ECU_4 0x04  // ECU #4
#define POBDII_ECU_5 0x05  // ECU #5
#define POBDII_ECU_6 0x06  // ECU #6
#define POBDII_ECU_7 0x07  // ECU #7
#define POBDII_ECU_8 0x08  // ECU #8
#define POBDII_ECU_9 0x09  // ECU #9

// OBD-II Services
//
#define POBDII_SERVICE_01 0x01  // OBDII Service $01
#define POBDII_SERVICE_02 0x02  // OBDII Service $02
#define POBDII_SERVICE_03 0x03  // OBDII Service $03
#define POBDII_SERVICE_04 0x04  // OBDII Service $04
#define POBDII_SERVICE_06 0x06  // OBDII Service $06
#define POBDII_SERVICE_07 0x07  // OBDII Service $07
#define POBDII_SERVICE_08 0x08  // OBDII Service $08
#define POBDII_SERVICE_09 0x09  // OBDII Service $09
#define POBDII_SERVICE_0A 0x0A  // OBDII Service $0A

// OBD-II Response codes
//
#define POBDII_R_NO_ERROR                            0x00     // element contains positive response
#define POBDII_R_BUSY_REPEAT_REQUEST                 0x21     // server is busy
#define POBDII_R_CONDITIONS_NOT_CORRECT              0x22     // conditions not correct
#define POBDII_R_RESPONSE_PENDING                    0x78     // server needs more time
#define POBDII_R_NOT_USED                            0xFF     // element is not used

// Represent the OBD-II error and status codes 
//
#define POBDII_ERROR_OK                              0x00000  // No error 
#define POBDII_ERROR_NOT_INITIALIZED                 0x00001  // Not Initialized
#define POBDII_ERROR_ALREADY_INITIALIZED			 0x00002  // Already Initialized
#define POBDII_ERROR_NO_MEMORY                       0x00003  // Could not obtain memory
#define POBDII_ERROR_OVERFLOW                        0x00004  // Input buffer overflow
#define POBDII_ERROR_TIMEOUT                         0x00006  // Timeout while accessing the mutex
#define POBDII_ERROR_NO_MESSAGE                      0x00007  // No Message available
#define POBDII_ERROR_WRONG_PARAM                     0x00008  // Wrong parameters in query
#define POBDII_ERROR_NOT_SUPPORTED                   0x00009  // Buffer to parse contains response to Vehicle specific parameter
#define POBDII_ERROR_PARSE_ERROR                     0x0000A  // Could not parse response	
#define POBDII_ERROR_BUSLIGHT                        0x0000B  // The Channel is in BUS-LIGHT error state
#define POBDII_ERROR_BUSHEAVY                        0x0000C  // The Channel is in BUS-HEAVY error state
#define POBDII_ERROR_BUSOFF                          0x0000D  // The Channel is in BUS-OFF error state
#define POBDII_ERROR_UNSUPPORTED_ECUS                0x0000E  // Connected ECU is not supported

#define POBDII_ERROR_CAN_ERROR                             0x10000  // 0x10000 .. 0x1ffff CAN System dependend

// OBD-II InfoType Data Types
//
#define POBDII_INFOTYPE_COUNTER						 0x00	  // WORD sized data
#define POBDII_INFOTYPE_CALDATA						 0x01	  // 4 Byte long calibration data
#define POBDII_INFOTYPE_STRING						 0x02	  // Char buffer containing up to 20 characters
#define POBDII_INFOTYPE_NONE						 0x03	  // Data is not an InfoType (used to identify raw data bit encoded InfoType Support)
						

// OBD-II parameters
//
#define POBDII_PARAM_LOGGING						 0xB1     // 1 BYTE data describing the logging mode 
#define POBDII_PARAM_AVAILABLE_ECUS  				 0xB2     // 1 BYTE data describing the number of available ECUs 
#define POBDII_PARAM_SUPPORTMASK_PIDS                0xB3     // 256 BYTE data describing the supported PIDs for Service 01: Current Data
#define POBDII_PARAM_SUPPORTMASK_FFPIDS              0xB4     // 257 BYTE data describing the supported PIDs for Service 02: Freeze Frame Data 
															  // 1st BYTE of buffer is used as a frame number indicator
#define POBDII_PARAM_SUPPORTMASK_OBDMIDS             0xB5     // 256 BYTE data describing the supported OBDMIDs for Service 06: Test Results
#define POBDII_PARAM_SUPPORTMASK_TIDS                0xB6     // 256 BYTE data describing the supported TIDs for Service 08: Control Operation
#define POBDII_PARAM_SUPPORTMASK_INFOTYPES           0xB7     // 256 BYTE data describing the supported InfoTypes for Service 09: Vehicle Information
#define POBDII_PARAM_API_VERSION					 0xB8	  // API version parameter	
#define POBDII_PARAM_BAUDRATE						 0xB9	  // 1 BYTE data describing baudrate value
#define POBDII_PARAM_CAN_ID							 0xBA	  // 1 BYTE data describing CAN identifier length
#define POBDII_PARAM_DEBUG							 0xE3     // 1 BYTE data describing the debug mode 
#define POBDII_PARAM_CHANNEL_CONDITION               0xE4     // 1 BYTE data describing the condition of a channel

// OBD-II parameter values
//
#define POBDII_DEBUG_NONE							0x00     // No debug messages
#define POBDII_DEBUG_CAN							0x01     // Puts CAN debug messages to stdout
#define POBDII_LOGGING_NONE							0x00     // No logging
#define POBDII_LOGGING_TO_FILE						0x01     // Log function enter/exit data to file
#define POBDII_LOGGING_TO_STDOUT					0x02     // Log function enter/exit data to stdout
#define POBDII_CHANNEL_UNAVAILABLE					0x00     // The Channel is illegal or not available
#define POBDII_CHANNEL_AVAILABLE					0x01     // The Channel is available
#define POBDII_CHANNEL_OCCUPIED						0x02     // The Channel is valid, and is being used
#define POBDII_BAUDRATE_NON_LEGISLATED				0x00	 // Non legislated-OBD baudrate
#define POBDII_BAUDRATE_250K						0x01	 // Baudrate: 250 kBit/s
#define POBDII_BAUDRATE_500K						0x02	 // Baudrate: 500 kBit/s
#define POBDII_BAUDRATE_AUTODETECT					0xFF	 // Baudrate: autodetection
#define POBDII_CAN_ID_11BIT							11		// 11 Bit CAN Identifier
#define POBDII_CAN_ID_29BIT							29		// 29 Bit CAN Identifier


////////////////////////////////////////////////////////////
// Type definitions
////////////////////////////////////////////////////////////

#define TPOBDIICANHandle           WORD        // Represents a CAN hardware channel of the underlying CAN system
#define TPOBDIIStatus              DWORD       // Represents an OBDII status/error code
#define TPOBDIIHWType              BYTE        // Represents the type of CAN hardware to be initialized
#define TPOBDIIBaudrateInfo        BYTE        // Represents the baudrate type
#define TPOBDIIAddress		       BYTE		   // Represents addresses of legislated OBD ECU 
#define TPOBDIIService		       BYTE		   // Represents OBD-II services
#define TPOBDIIParameter           BYTE        // Represents an OBDII parameter to be read or set
#define TPOBDIIError			   BYTE		   // Represents an OBDII response error code

#define TPOBDIIPid				   BYTE        // Represents the Parameter ID number
#define TPOBDIIOBDMid			   BYTE        // Represents the On-Board Diagnosctics Monitor ID number
#define TPOBDIITid				   BYTE        // Represents the Test ID number
#define TPOBDIIInfoType			   BYTE        // Represents the InfoType number
#define TPOBDIIInfoDataType		   BYTE        // Represents the type of returned data


////////////////////////////////////////////////////////////
// Data definitions
////////////////////////////////////////////////////////////

#pragma pack(push, 8)

// Represents the data returned by OBD service $04, $08
typedef struct tagTPOBDIIResponse
{
	BYTE			SOURCE;			// ECU number
	TPOBDIIError	ERRORNR;		// Error number (POBDII_R_*)
	BYTE			DATA[4095];		// Raw data (without SID and Parameters)
	WORD			LEN;			// Raw data length
	BYTE			SID;			// Service ID
	BYTE			PID;			// Parameter ID (only with services $01, $02, $08, $09)
	BYTE			FRAME;			// Frame number (only with OBD Service $02)
} TPOBDIIResponse;

// Represents the data returned by OBD service $01, $02
typedef struct tagTPOBDIIParamData
{
	TPOBDIIResponse	RESPONSE;		// Response definition data
	BYTE			BUFFER[41];		// Raw data (without SID, PID and Frame)
	double			DOUBLES[10];	// Up to 10 double values
	WORD			BYTEMASK;		// Mask for bytes that are bit encoded: A(0) to J(9)
	BYTE			BLEN;			// Number of bytes stored in raw buffer (0 to 41)
	BYTE			DLEN;			// Number of doubles stored (0 to 10)
	char			DTC[6];			// A single DTC (incl. '\0')
} TPOBDIIParamData;

// Represents the data returned by OBD service $03, $07, $0A
typedef struct tagTPOBDIIDTCData
{
	TPOBDIIResponse	RESPONSE;		// Response definition data
	char			DTC[10][6];		// 10 DTCs each is 6 bytes (incl. '\0')
	BYTE			DLEN;			// Number of DTCs stored (0 to 10)
} TPOBDIIDTCData;

// Represents the data returned by OBD service $06
typedef struct tagTPOBDIIMonitorData
{
	TPOBDIIResponse	RESPONSE;		// Response definition data
	BYTE			TID;			// Test ID
	BYTE			UNITANDSCALING;	// Unit And Scaling ID
	double			TESTVALUE;		// Test value
	double			MINLIMIT;		// Minimum Test Limit
	double			MAXLIMIT;		// Maximum Test Limit
} TPOBDIIMonitorData;

// Represents the data returned by OBD service $09
typedef struct tagTPOBDIIInfoData
{
	TPOBDIIResponse		RESPONSE;		// Response definition data
	BYTE				INDEXNR;		// Index Number of the counter or the string
	TPOBDIIInfoDataType	DATATYPE;		// Type of data available (POBDII_INFOTYPE_*)
	WORD				COUNTER;		// Counter value
	BYTE				CALDATA[4];     // Calibration data
	char				TEXT[21];		// String value
} TPOBDIIInfoData;

// Unit and Scaling definition
typedef struct tagTPOBDIIUnitAndScaling
{
	double			MIN;		// Min value
	double			MAX;		// Max value
	CHAR			UNIT[16];	// Unit name
} TPOBDIIUnitAndScaling;

#pragma pack(pop)


////////////////////////////////////////////////////////////
// PCAN OBD-II API function declarations
////////////////////////////////////////////////////////////

#ifdef __cplusplus
extern "C" {
#define _DEF_ARG =0
#define _DEF_ARG_BAUDRATE =POBDII_BAUDRATE_AUTODETECT
#else
#define _DEF_ARG
#define _DEF_ARG_BAUDRATE
#endif

/// <summary>
/// Initializes an OBDII-Client based on a CAN-Channel
/// </summary>
/// <remarks>Only one OBDII-Client can be initialized per CAN-Channel</remarks>
/// <param name="CanChannel">The PCAN-Basic channel to be used as OBDII client</param>
/// <param name="HwType">NON PLUG&PLAY: The type of hardware and operation mode</param>
/// <param name="IOPort">NON PLUG&PLAY: The I/O address for the parallel port</param>
/// <param name="Interupt">NON PLUG&PLAY: Interrupt number of the parallel port</param>
/// <returns>A TPOBDIIStatus code. POBDII_ERROR_OK is returned on success</returns>
TPOBDIIStatus __stdcall OBDII_Initialize(
        TPOBDIICANHandle CanChannel, 
        TPOBDIIBaudrateInfo Baudrate _DEF_ARG_BAUDRATE,
        TPOBDIIHWType HwType _DEF_ARG,
        DWORD IOPort _DEF_ARG, 
        WORD Interrupt _DEF_ARG);

/// <summary>
/// Uninitializes an OBDII-Client initialized before
/// </summary>
/// <param name="CanChannel">An OBDII CAN-Channel representing a OBDII-Client</param>
/// <returns>A TPOBDIIStatus code. POBDII_ERROR_OK is returned on success</returns>
TPOBDIIStatus __stdcall OBDII_Uninitialize(
        TPOBDIICANHandle CanChannel);

/// <summary>
/// Resets the receive and transmit queues of an OBDII-Client 
/// </summary>
/// <param name="CanChannel">An OBDII CAN-Channel representing a OBDII-Client</param>
/// <returns>A TPOBDIIStatus code. POBDII_ERROR_OK is returned on success</returns>
TPOBDIIStatus __stdcall OBDII_Reset(
        TPOBDIICANHandle CanChannel);

/// <summary>
/// Gets the initialization status of an OBDII CAN-Channel
/// </summary>
/// <param name="CanChannel">An OBDII CAN-Channel representing a OBDII-Client</param>
/// <returns>A TPOBDIIStatus code. POBDII_ERROR_OK is returned on success</returns>
TPOBDIIStatus __stdcall OBDII_GetStatus(
        TPOBDIICANHandle CanChannel);

/// <summary>
/// Retrieves an OBDII-Client value
/// </summary>
/// <param name="CanChannel">An OBDII CAN-Channel representing a OBDII-Client</param>
/// <param name="Parameter">The TPOBDIIParameter parameter to get</param>
/// <param name="Buffer">Buffer for the parameter value</param>
/// <param name="BufferLength">Size in bytes of the buffer</param>
/// <returns>A TPOBDIIStatus code. POBDII_ERROR_OK is returned on success</returns>
TPOBDIIStatus __stdcall OBDII_GetValue(
        TPOBDIICANHandle CanChannel, 
        TPOBDIIParameter Parameter, 
        void* Buffer, 
        DWORD BufferLength);

/// <summary>
/// Configures or sets an OBDII-Client value
/// </summary>
/// <param name="CanChannel">An OBDII CAN-Channel representing a OBDII-Client</param>
/// <param name="Parameter">The TPOBDIIParameter parameter to set</param>
/// <param name="Buffer">Buffer with the value to be set</param>
/// <param name="BufferLength">Size in bytes of the buffer</param>
/// <returns>A TPOBDIIStatus code. POBDII_ERROR_OK is returned on success</returns>
TPOBDIIStatus __stdcall OBDII_SetValue(
        TPOBDIICANHandle CanChannel,
        TPOBDIIParameter Parameter,
        void* Buffer,
		DWORD BufferLength);

/// <summary>
/// Sends an OBDII Service $01 request into queue and waits for the received response
/// </summary>
/// <param name="CanChannel">An OBDII CAN-Channel representing a OBDII-Client</param>
/// <param name="Pid">A Parameter Identifier</param>
/// <param name="Data">A TPOBDIIParamData structure array to store the OBDII responses</param>
/// <param name="DataLen">A length of provided Data array</param>
/// <returns>A TPOBDIIStatus code. POBDII_ERROR_OK is returned on success</returns>
TPOBDIIStatus __stdcall OBDII_RequestCurrentData(
        TPOBDIICANHandle CanChannel, 
		TPOBDIIPid Pid,
        TPOBDIIParamData* Data,
		BYTE DataLen);

/// <summary>
/// Sends an OBDII Service $02 request into queue and waits for the received response
/// </summary>
/// <param name="CanChannel">An OBDII CAN-Channel representing a OBDII-Client</param>
/// <param name="Pid">A Parameter Identifier</param>
/// <param name="Frame">A Frame number</param>
/// <param name="Data">A TPOBDIIParamData structure array to store the OBDII responses</param>
/// <param name="DataLen">A length of provided Data array</param>
/// <returns>A TPOBDIIStatus code. POBDII_ERROR_OK is returned on success</returns>
TPOBDIIStatus __stdcall OBDII_RequestFreezeFrameData(
        TPOBDIICANHandle CanChannel, 
		TPOBDIIPid Pid,
		BYTE Frame,
        TPOBDIIParamData* Data,
		BYTE DataLen);

/// <summary>
/// Sends an OBDII Service $03 request into queue and waits for the received response
/// </summary>
/// <param name="CanChannel">An OBDII CAN-Channel representing a OBDII-Client</param>
/// <param name="Data">A TPOBDIIDTCData structure array to store the OBDII responses</param>
/// <param name="DataLen">A length of provided Data array</param>
/// <returns>A TPOBDIIStatus code. POBDII_ERROR_OK is returned on success</returns>
TPOBDIIStatus __stdcall OBDII_RequestStoredTroubleCodes(
        TPOBDIICANHandle CanChannel, 
        TPOBDIIDTCData* Data,
		BYTE DataLen);

/// <summary>
/// Sends an OBDII Service $04 request into queue and waits for the received response
/// </summary>
/// <param name="CanChannel">An OBDII CAN-Channel representing a OBDII-Client</param>
/// <param name="Data">A TPOBDIIResponse structure array to store the OBDII responses</param>
/// <param name="DataLen">A length of provided Data array</param>
/// <returns>A TPOBDIIStatus code. POBDII_ERROR_OK is returned on success</returns>
TPOBDIIStatus __stdcall OBDII_ClearTroubleCodes(
        TPOBDIICANHandle CanChannel, 
        TPOBDIIResponse* Response,
        BYTE DataLen);

/// <summary>
/// Sends an OBDII Service $06 request into queue and waits for the received response
/// </summary>
/// <param name="CanChannel">An OBDII CAN-Channel representing a OBDII-Client</param>
/// <param name="OBDMid">An On-Board Diagnostic Monitor Identifier</param>
/// <param name="Data">A TPOBDIIOBDMid structure array to store the OBDII responses</param>
/// <param name="DataLen">A length of provided Data array</param>
/// <returns>A TPOBDIIStatus code. POBDII_ERROR_OK is returned on success</returns>
TPOBDIIStatus __stdcall OBDII_RequestTestResults(
        TPOBDIICANHandle CanChannel, 
        TPOBDIIOBDMid OBDMid, 
        TPOBDIIMonitorData* Data,
        BYTE DataLen);

/// <summary>
/// Sends an OBDII Service $07 request into queue and waits for the received response
/// </summary>
/// <param name="CanChannel">An OBDII CAN-Channel representing a OBDII-Client</param>
/// <param name="Data">A TPOBDIIDTCData structure array to store the OBDII responses</param>
/// <param name="DataLen">A length of provided Data array</param>
/// <returns>A TPOBDIIStatus code. POBDII_ERROR_OK is returned on success</returns>
TPOBDIIStatus __stdcall OBDII_RequestPendingTroubleCodes(
        TPOBDIICANHandle CanChannel, 
        TPOBDIIDTCData* Data,
		BYTE DataLen);

/// <summary>
/// Sends an OBDII Service $08 request into queue and waits for the received response
/// </summary>
/// <param name="CanChannel">An OBDII CAN-Channel representing a OBDII-Client</param>
/// <param name="Tid">A Test Identifier</param>
/// <param name="Data">A TPOBDIIResponse structure array to store the OBDII responses</param>
/// <param name="DataLen">A length of provided Data array</param>
/// <returns>A TPOBDIIStatus code. POBDII_ERROR_OK is returned on success</returns>
TPOBDIIStatus __stdcall OBDII_RequestControlOperation(
        TPOBDIICANHandle CanChannel, 
        TPOBDIITid Tid,
        TPOBDIIResponse* Response,
        BYTE DataLen);

/// <summary>
/// Sends an OBDII Service $09 request into queue and waits for the received response
/// </summary>
/// <param name="CanChannel">An OBDII CAN-Channel representing a OBDII-Client</param>
/// <param name="InfoType">An InfoType</param>
/// <param name="Data">A TPOBDIIInfoData structure array to store the OBDII responses</param>
/// <param name="DataLen">A length of provided Data array</param>
/// <returns>A TPOBDIIStatus code. POBDII_ERROR_OK is returned on success</returns>
TPOBDIIStatus __stdcall OBDII_RequestVehicleInformation(
        TPOBDIICANHandle CanChannel, 
		TPOBDIIInfoType InfoType, 
        TPOBDIIInfoData* Data,
		BYTE DataLen);

/// <summary>
/// Sends an OBDII Service $0A request into queue and waits for the received response
/// </summary>
/// <param name="CanChannel">An OBDII CAN-Channel representing a OBDII-Client</param>
/// <param name="Pid">A Parameter Identifier</param>
/// <param name="Data">A TPOBDIIDTCData structure array to store the OBDII responses</param>
/// <param name="DataLen">A length of provided Data array</param>
/// <returns>A TPOBDIIStatus code. POBDII_ERROR_OK is returned on success</returns>
TPOBDIIStatus __stdcall OBDII_RequestPermanentTroubleCodes(
        TPOBDIICANHandle CanChannel, 
        TPOBDIIDTCData* Data,
		BYTE DataLen);

/// <summary>
/// Returns the unit and scaling information for a specified Unit and Scaling ID
/// </summary>
/// <param name="id">Unit and Scaling ID</param>
/// <param name="Data">a TPOBDIIUnitAndScaling structure to store the unit information</param>
/// <returns>A TPOBDIIStatus code. POBDII_ERROR_OK is returned on success</returns>
TPOBDIIStatus __stdcall OBDII_GetUnitAndScaling(BYTE id, TPOBDIIUnitAndScaling * unitAndScaling);

#ifdef __cplusplus
}
#endif

#endif