﻿''  PCAN-OBDII.vb
''
''  ~~~~~~~~~~~~
''
''  PCAN-OBDII API
''
''  ~~~~~~~~~~~~
''
''  ------------------------------------------------------------------
''  Author : Fabrice Vergnaud
''	Last changed by:	$Author: Fabrice $
''  Last changed date:	$Date: 2019-10-11 13:53:39 +0200 (Fri, 11 Oct 2019) $
''
''  Language: VB.Net
''  ------------------------------------------------------------------
''
''  Copyright (C) 2015  PEAK-System Technik GmbH, Darmstadt
''  more Info at http:''www.peak-system.com 
''
Imports System
Imports System.Text
Imports System.Runtime.InteropServices

'' Aliases definition
''
Imports TPOBDIICANHandle = System.UInt16  '' Represents a CAN hardware channel of the underlying CAN system
Imports TPOBDIIPid = System.Byte          '' Represents the PID number
Imports TPOBDIIOBDMid = System.Byte       '' Represents the OBDMID number
Imports TPOBDIITid = System.Byte          '' Represents the TID number
Imports TPOBDIIInfoType = System.Byte     '' Represents the InfoType number

Namespace Peak.Can.ObdII

#Region "Enumerations"
    ''' <summary>
    ''' OBDII baudrates
    ''' </summary>
    Public Enum TPOBDIIBaudrateInfo As Byte
        ''' <summary>
        ''' Non legislated-OBD baudrate
        ''' </summary>
        POBDII_BAUDRATE_NON_LEGISLATED = &H0
        ''' <summary>
        ''' Baudrate: 250 kBit/s
        ''' </summary>
        POBDII_BAUDRATE_250K = &H1
        ''' <summary>
        ''' Baudrate: 500 kBit/s
        ''' </summary>
        POBDII_BAUDRATE_500K = &H2
        ''' <summary>
        ''' Baudrate: Autodetect
        ''' </summary>
        POBDII_BAUDRATE_AUTODETECT = &HFF
    End Enum

    ''' <summary>
    ''' OBDII address definitions
    ''' </summary>
    Public Enum TPOBDIIAddress As Byte
        ''' <summary>
        ''' ECU 1
        ''' </summary>
        POBDII_ECU_1 = &H1
        ''' <summary>
        ''' ECU 2
        ''' </summary>
        POBDII_ECU_2 = &H2
        ''' <summary>
        ''' ECU 3
        ''' </summary>
        POBDII_ECU_3 = &H3
        ''' <summary>
        ''' ECU 4
        ''' </summary>
        POBDII_ECU_4 = &H4
        ''' <summary>
        ''' ECU 5
        ''' </summary>
        POBDII_ECU_5 = &H5
        ''' <summary>
        ''' ECU 6
        ''' </summary>
        POBDII_ECU_6 = &H6
        ''' <summary>
        ''' ECU 7
        ''' </summary>
        POBDII_ECU_7 = &H7
        ''' <summary>
        ''' ECU 8
        ''' </summary>
        POBDII_ECU_8 = &H8
    End Enum

    ''' <summary>
    ''' Represents the OBDII Service numbers
    ''' </summary>
    Public Enum TPOBDIIService As Byte
        ''' <summary>
        ''' Service 01 — Request Current Powertrain Diagnostic Data
        ''' </summary>
        POBDII_SERVICE_01 = &H1
        ''' <summary>
        ''' Service 02 — Request Powertrain Freeze Frame Data
        ''' </summary>
        POBDII_SERVICE_02 = &H2
        ''' <summary>
        ''' Service 03 — Request Emission-Related Diagnostic Trouble Codes
        ''' </summary>
        POBDII_SERVICE_03 = &H3
        ''' <summary>
        ''' Service 04 — Clear/Reset Emission-Related Diagnostic Information
        ''' </summary>
        POBDII_SERVICE_04 = &H4
        ''' <summary>
        ''' Service 06 — Request On-Board Monitoring Test Results for Specific Monitored Systems
        ''' </summary>
        POBDII_SERVICE_06 = &H6
        ''' <summary>
        ''' Service 07 — Request Emission-Related Diagnostic Trouble Codes Detected During Current or Last Completed Driving Cycle
        ''' </summary>
        POBDII_SERVICE_07 = &H7
        ''' <summary>
        ''' Service 08 — Request Control of On-Board System, Test or Component
        ''' </summary>
        POBDII_SERVICE_08 = &H8
        ''' <summary>
        ''' Service 09 — Request Vehicle Information
        ''' </summary>
        POBDII_SERVICE_09 = &H9
        ''' <summary>
        ''' Service 0A — Request Emission-Related Diagnostic Trouble Codes with Permanent Status
        ''' </summary>
        POBDII_SERVICE_0A = &HA
    End Enum

    ''' <summary>
    ''' Represents a OBDII parameter to be read or set
    ''' </summary>
    Public Enum TPOBDIIParameter As Byte
        ''' <summary>
        ''' 1 BYTE data describing the logging mode 
        ''' </summary>
        POBDII_PARAM_LOGGING = &HB1
        ''' <summary>
        ''' 1 BYTE data describing the number of available ECUs 
        ''' </summary>
        POBDII_PARAM_AVAILABLE_ECUS = &HB2
        ''' <summary>
        ''' 256 Byte data describing the supported PIDs for Service 01: Current Data
        ''' </summary>
        POBDII_PARAM_SUPPORTMASK_PIDS = &HB3
        ''' <summary>
        ''' 257 Byte data describing the supported PIDs of Frame 
        ''' (identified by the first BYTE in the buffer) for Service 02: Freeze Frame Data 
        ''' </summary>
        POBDII_PARAM_SUPPORTMASK_FFPIDS = &HB4
        ''' <summary>
        ''' 256 Byte data describing the supported OBDMIDs for Service 06: Test Results
        ''' </summary>
        POBDII_PARAM_SUPPORTMASK_OBDMIDS = &HB5
        ''' <summary>
        ''' 256 Byte data describing the supported TIDs for Service 08: Control Operation
        ''' </summary>
        POBDII_PARAM_SUPPORTMASK_TIDS = &HB6
        ''' <summary>
        ''' 256 Byte data describing the supported InfoTypes for Service 09: Vehicle Information
        ''' </summary>
        POBDII_PARAM_SUPPORTMASK_INFOTYPES = &HB7
        ''' <summary>
        ''' 1 BYTE data describing baudrate value
        ''' </summary>
        POBDII_PARAM_BAUDRATE = &HB9
        ''' <summary>
        ''' 1 BYTE data describing CAN identifier length
        ''' </summary>
        POBDII_PARAM_CAN_ID = &HBA
        ''' <summary>
        ''' API version parameter
        ''' </summary>
        POBDII_PARAM_API_VERSION = &HB8
        ''' <summary>
        ''' 1 BYTE data describing the debug mode 
        ''' </summary>
        POBDII_PARAM_DEBUG = &HE3
        ''' <summary>
        ''' 1 Byte data describing the condition of a channel
        ''' </summary>
        POBDII_PARAM_CHANNEL_CONDITION = &HE4
    End Enum

    ''' <summary>
    ''' Represents the type of CAN hardware to be initialized
    ''' </summary>
    Public Enum TPOBDIIHwType As Byte

        ''' <summary>
        ''' PCAN-ISA 82C200
        ''' </summary>
        POBDII_HWTYPE_ISA = &H1
        ''' <summary>
        ''' PCAN-ISA SJA1000
        ''' </summary>
        POBDII_HWTYPE_ISA_SJA = &H9
        ''' <summary>
        ''' PHYTEC ISA 
        ''' </summary>
        POBDII_HWTYPE_ISA_PHYTEC = &H4
        ''' <summary>
        ''' PCAN-Dongle 82C200
        ''' </summary>
        POBDII_HWTYPE_DNG = &H2
        ''' <summary>
        ''' PCAN-Dongle EPP 82C200
        ''' </summary>
        POBDII_HWTYPE_DNG_EPP = &H3
        ''' <summary>
        ''' PCAN-Dongle SJA1000
        ''' </summary>
        POBDII_HWTYPE_DNG_SJA = &H5
        ''' <summary>
        ''' PCAN-Dongle EPP SJA1000
        ''' </summary>
        POBDII_HWTYPE_DNG_SJA_EPP = &H6
    End Enum

    ''' <summary>
    '''  Represents a OBDII status/error code
    ''' </summary>
    Public Enum TPOBDIIStatus : UInt32
        ''' <summary>
        ''' No error 
        ''' </summary>
        POBDII_ERROR_OK = &H0
        ''' <summary>
        ''' Not Initialized
        ''' </summary>
        POBDII_ERROR_NOT_INITIALIZED = &H1
        ''' <summary>
        ''' Already Initialized
        ''' </summary>
        POBDII_ERROR_ALREADY_INITIALIZED = &H2
        ''' <summary>
        ''' Could not obtain memory
        ''' </summary>
        POBDII_ERROR_NO_MEMORY = &H3
        ''' <summary>
        ''' Input buffer overflow
        ''' </summary>
        POBDII_ERROR_OVERFLOW = &H4
        ''' <summary>
        ''' Timeout while accessing the UDS mutex
        ''' </summary>
        POBDII_ERROR_TIMEOUT = &H6
        ''' <summary>
        ''' No Message available
        ''' </summary>
        POBDII_ERROR_NO_MESSAGE = &H7
        ''' <summary>
        ''' Wrong parameters in query
        ''' </summary>
        POBDII_ERROR_WRONG_PARAM = &H8
        ''' <summary>
        ''' Buffer to parse contains response to Vehicle specific parameter
        ''' </summary>
        POBDII_ERROR_NOT_SUPPORTED = &H9
        ''' <summary>
        ''' Could not parse response
        ''' </summary>
        POBDII_ERROR_PARSE_ERROR = &HA
        ''' <summary>
        ''' Channel is in BUS-LIGHT error state
        ''' </summary>
        POBDII_ERROR_BUSLIGHT = &HB
        ''' <summary>
        ''' Channel is in BUS-HEAVY error state
        ''' </summary>
        POBDII_ERROR_BUSHEAVY = &HC
        ''' <summary>
        ''' Channel is in BUS-OFF error state
        ''' </summary>
        POBDII_ERROR_BUSOFF = &HD
        ''' <summary>
        ''' No connected ECUs, or ECU found is not supported
        ''' </summary>
        POBDII_ERROR_UNSUPPORTED_ECUS = &HE
        ''' <summary>
        ''' 0x10000 .. 0x1ffff CAN System dependend (PCAN-Basic). The value must 
        ''' be shifted 2 bytes to the right, then it is a PCAN-Basic Status value.
        ''' </summary>
        POBDII_ERROR_CAN_ERROR = &H10000
    End Enum

    ''' <summary>
    ''' Represents the OBD-II Response codes
    ''' </summary>
    Public Enum TPOBDIIError As Byte
        ''' <summary>
        ''' No error, Positive response
        ''' </summary>
        POBDII_R_NO_ERROR = &H0
        ''' <summary>
        ''' Server is busy
        ''' </summary>
        POBDII_R_BUSY_REPEAT_REQUEST = &H21
        ''' <summary>
        ''' Conditions not correct
        ''' </summary>
        POBDII_R_CONDITIONS_NOT_CORRECT = &H22
        ''' <summary>
        ''' server needs more time
        ''' </summary>
        POBDII_R_RESPONSE_PENDING = &H78
        ''' <summary>
        ''' Not a response, invalid value
        ''' </summary>
        POBDII_R_NOT_USED = &HFF
    End Enum

    ''' <summary>
    ''' Represents the type of returned data
    ''' </summary>
    Public Enum TPOBDIIInfoDataType As Byte
        ''' <summary>
        ''' WORD sized data
        ''' </summary>
        POBDII_INFOTYPE_COUNTER = &H0
        ''' <summary>
        ''' 4 Byte long calibration data
        ''' </summary>
        POBDII_INFOTYPE_CALDATA = &H1
        ''' <summary>
        ''' Char buffer containing up to 20 characters
        ''' </summary>
        POBDII_INFOTYPE_STRING = &H2
        ''' <summary>
        ''' Data is not an InfoType (used to identify raw data bit encoded InfoType Support)
        ''' </summary>
        POBDII_INFOTYPE_NONE = &H3
    End Enum

#End Region

#Region "Structures"

    ''' <summary>
    ''' Represents the data returned by OBD service $04, $08
    ''' </summary>
    <StructLayout(LayoutKind.Sequential, Pack:=8)>
    Public Structure TPOBDIIResponse
        ''' <summary>
        ''' ECU number
        ''' </summary>
        Public SOURCE As Byte
        ''' <summary>
        ''' Error number (POBDII_R_*)
        ''' </summary>
        <MarshalAs(UnmanagedType.U1)> _
        Public ERRORNR As TPOBDIIError
        ''' <summary>
        ''' Raw data (without SID and Parameters)
        ''' </summary>
        <MarshalAs(UnmanagedType.ByValArray, SizeConst:=4095)> _
        Public DATA As Byte()
        ''' <summary>
        ''' Number of bytes stored in DATA
        ''' </summary>
        Public LEN As UShort
        ''' <summary>
        ''' Service ID
        ''' </summary>
        Public SID As Byte
        ''' <summary>
        ''' Parameter ID (only with services $01, $02, $08, $09)
        ''' </summary>
        Public PID As Byte
        ''' <summary>
        ''' Frame number (only with OBD Service $02)
        ''' </summary>
        Public FRAME As Byte
    End Structure

    ''' <summary>
    ''' Represents the data returned by OBD service $01, $02
    ''' </summary>
    <StructLayout(LayoutKind.Sequential, Pack:=8)>
    Public Structure TPOBDIIParamData
        ''' <summary>
        ''' Response definition data
        ''' </summary>
        <MarshalAs(UnmanagedType.Struct)> _
        Public RESPONSE As TPOBDIIResponse
        ''' <summary>
        ''' Raw data (without SID, PID and Frame)
        ''' </summary>
        <MarshalAs(UnmanagedType.ByValArray, SizeConst:=41)> _
        Public BUFFER As Byte()
        ''' <summary>
        ''' Up to 10 double values
        ''' </summary>
        <MarshalAs(UnmanagedType.ByValArray, SizeConst:=10)> _
        Public DOUBLES As Double()
        ''' <summary>
        ''' Mask for bytes that are bit encoded: A(0) to J(9)
        ''' </summary>
        Public BYTEMASK As UShort
        ''' <summary>
        ''' Number of bytes stored in raw buffer (0 to 41)
        ''' </summary>
        Public BLEN As Byte
        ''' <summary>
        ''' Number of doubles stored (0 to 10)
        ''' </summary>
        Public DLEN As Byte
        ''' <summary>
        ''' A single DTC (incl. '\0')
        ''' </summary>
        <MarshalAs(UnmanagedType.ByValTStr, SizeConst:=6)> _
        Public DTC As String
    End Structure

    ''' <summary>
    ''' Represents a DTC as a string of 6 bytes
    ''' </summary>
    <StructLayout(LayoutKind.Sequential, Pack:=8)>
    Public Structure TPOBDIIDTCText
        <MarshalAs(UnmanagedType.ByValTStr, SizeConst:=6)> _
        Public ErrorText As String
    End Structure

    ''' <summary>
    ''' Represents the data returned by OBD service $03, $07, $0A
    ''' </summary>
    <StructLayout(LayoutKind.Sequential, Pack:=8)>
    Public Structure TPOBDIIDTCData
        ''' <summary>
        ''' Response definition data
        ''' </summary>
        <MarshalAs(UnmanagedType.Struct)> _
        Public RESPONSE As TPOBDIIResponse
        ''' <summary>
        ''' 10 DTCs each is 6 bytes (incl. '\0')
        ''' </summary>
        <MarshalAs(UnmanagedType.ByValArray, SizeConst:=10)> _
        Public DTC As TPOBDIIDTCText()
        ''' <summary>
        ''' Number of DTCs stored (0 to 10)
        ''' </summary>
        Public DLEN As Byte
    End Structure

    ''' <summary>
    ''' Represents the data returned by OBD service $06
    ''' </summary>
    <StructLayout(LayoutKind.Sequential, Pack:=8)>
    Public Structure TPOBDIIMonitorData
        ''' <summary>
        ''' Response definition data
        ''' </summary>
        <MarshalAs(UnmanagedType.Struct)> _
        Public RESPONSE As TPOBDIIResponse
        ''' <summary>
        ''' Test ID
        ''' </summary>
        Public TID As Byte
        ''' <summary>
        ''' Unit And Scaling ID
        ''' </summary>
        Public UNITANDSCALING As Byte
        ''' <summary>
        ''' Test value
        ''' </summary>
        Public TESTVALUE As Double
        ''' <summary>
        ''' Minimum Test Limit
        ''' </summary>
        Public MINLIMIT As Double
        ''' <summary>
        ''' Maximum Test Limit
        ''' </summary>
        Public MAXLIMIT As Double
    End Structure

    ''' <summary>
    ''' Represents the data returned by OBD service $09
    ''' </summary>
    <StructLayout(LayoutKind.Sequential, Pack:=8)>
    Public Structure TPOBDIIInfoData
        ''' <summary>
        ''' Response definition data
        ''' </summary>
        <MarshalAs(UnmanagedType.Struct)> _
        Public RESPONSE As TPOBDIIResponse
        ''' <summary>
        ''' Index Number of the counter or the string
        ''' </summary>
        Public INDEXNR As Byte
        ''' <summary>
        ''' Type of data available (POBDII_INFOTYPE_*)
        ''' </summary>
        <MarshalAs(UnmanagedType.U1)> _
        Public DATATYPE As TPOBDIIInfoDataType
        ''' <summary>
        ''' Counter value
        ''' </summary>
        Public COUNTER As UShort
        ''' <summary>
        ''' Calibration data
        ''' </summary>
        <MarshalAs(UnmanagedType.ByValArray, SizeConst:=4)> _
        Public CALDATA As Byte()
        ''' <summary>
        ''' String value
        ''' </summary>
        <MarshalAs(UnmanagedType.ByValTStr, SizeConst:=21)> _
        Public TEXT As String
    End Structure

    ''' <summary>
    ''' Unit and Scaling definition
    ''' </summary>
    <StructLayout(LayoutKind.Sequential, Pack:=8)>
    Public Structure TPOBDIIUnitAndScaling
        ''' <summary>
        ''' Minimum value
        ''' </summary>
        Public MIN As Double
        ''' <summary>
        ''' Maximum value
        ''' </summary>
        Public MAX As Double
        ''' <summary>
        ''' Unit abbreviation
        ''' </summary>
        <MarshalAs(UnmanagedType.ByValTStr, SizeConst:=16)> _
        Public UNIT As String
    End Structure
#End Region

    Public NotInheritable Class OBDIIApi
#Region "PCAN-BUS Handles Definition"
        ''' <summary>
        ''' Undefined/default value for a PCAN bus
        ''' </summary>
        Public Const POBDII_NONEBUS As TPOBDIICANHandle = &H0

        ''' <summary>
        ''' PCAN-ISA interface, channel 1
        ''' </summary>
        Public Const POBDII_ISABUS1 As TPOBDIICANHandle = &H21
        ''' <summary>
        ''' PCAN-ISA interface, channel 2
        ''' </summary>
        Public Const POBDII_ISABUS2 As TPOBDIICANHandle = &H22
        ''' <summary>
        ''' PCAN-ISA interface, channel 3
        ''' </summary>
        Public Const POBDII_ISABUS3 As TPOBDIICANHandle = &H23
        ''' <summary>
        ''' PCAN-ISA interface, channel 4
        ''' </summary>
        Public Const POBDII_ISABUS4 As TPOBDIICANHandle = &H24
        ''' <summary>
        ''' PCAN-ISA interface, channel 5
        ''' </summary>
        Public Const POBDII_ISABUS5 As TPOBDIICANHandle = &H25
        ''' <summary>
        ''' PCAN-ISA interface, channel 6
        ''' </summary>
        Public Const POBDII_ISABUS6 As TPOBDIICANHandle = &H26
        ''' <summary>
        ''' PCAN-ISA interface, channel 7
        ''' </summary>
        Public Const POBDII_ISABUS7 As TPOBDIICANHandle = &H27
        ''' <summary>
        ''' PCAN-ISA interface, channel 8
        ''' </summary>
        Public Const POBDII_ISABUS8 As TPOBDIICANHandle = &H28

        ''' <summary>
        ''' PPCAN-Dongle/LPT interface, channel 1 
        ''' </summary>
        Public Const POBDII_DNGBUS1 As TPOBDIICANHandle = &H31

        ''' <summary>
        ''' PCAN-PCI interface, channel 1
        ''' </summary>
        Public Const POBDII_PCIBUS1 As TPOBDIICANHandle = &H41
        ''' <summary>
        ''' PCAN-PCI interface, channel 2
        ''' </summary>
        Public Const POBDII_PCIBUS2 As TPOBDIICANHandle = &H42
        ''' <summary>
        ''' PCAN-PCI interface, channel 3
        ''' </summary>
        Public Const POBDII_PCIBUS3 As TPOBDIICANHandle = &H43
        ''' <summary>
        ''' PCAN-PCI interface, channel 4
        ''' </summary>
        Public Const POBDII_PCIBUS4 As TPOBDIICANHandle = &H44
        ''' <summary>
        ''' PCAN-PCI interface, channel 5
        ''' </summary>
        Public Const POBDII_PCIBUS5 As TPOBDIICANHandle = &H45
        ''' <summary>
        ''' PCAN-PCI interface, channel 6
        ''' </summary>
        Public Const POBDII_PCIBUS6 As TPOBDIICANHandle = &H46
        ''' <summary>
        ''' PCAN-PCI interface, channel 7
        ''' </summary>
        Public Const POBDII_PCIBUS7 As TPOBDIICANHandle = &H47
        ''' <summary>
        ''' PCAN-PCI interface, channel 8
        ''' </summary>
        Public Const POBDII_PCIBUS8 As TPOBDIICANHandle = &H48
        ''' <summary>
        ''' PCAN-PCI interface, channel 9
        ''' </summary>
        Public Const POBDII_PCIBUS9 As TPOBDIICANHandle = &H409
        ''' <summary>
        ''' PCAN-PCI interface, channel 10
        ''' </summary>
        Public Const POBDII_PCIBUS10 As TPOBDIICANHandle = &H40A
        ''' <summary>
        ''' PCAN-PCI interface, channel 11
        ''' </summary>
        Public Const POBDII_PCIBUS11 As TPOBDIICANHandle = &H40B
        ''' <summary>
        ''' PCAN-PCI interface, channel 12
        ''' </summary>
        Public Const POBDII_PCIBUS12 As TPOBDIICANHandle = &H40C
        ''' <summary>
        ''' PCAN-PCI interface, channel 13
        ''' </summary>
        Public Const POBDII_PCIBUS13 As TPOBDIICANHandle = &H40D
        ''' <summary>
        ''' PCAN-PCI interface, channel 14
        ''' </summary>
        Public Const POBDII_PCIBUS14 As TPOBDIICANHandle = &H40E
        ''' <summary>
        ''' PCAN-PCI interface, channel 15
        ''' </summary>
        Public Const POBDII_PCIBUS15 As TPOBDIICANHandle = &H40F
        ''' <summary>
        ''' PCAN-PCI interface, channel 16
        ''' </summary>
        Public Const POBDII_PCIBUS16 As TPOBDIICANHandle = &H410

        ''' <summary>
        ''' PCAN-USB interface, channel 1
        ''' </summary>
        Public Const POBDII_USBBUS1 As TPOBDIICANHandle = &H51
        ''' <summary>
        ''' PCAN-USB interface, channel 2
        ''' </summary>
        Public Const POBDII_USBBUS2 As TPOBDIICANHandle = &H52
        ''' <summary>
        ''' PCAN-USB interface, channel 3
        ''' </summary>
        Public Const POBDII_USBBUS3 As TPOBDIICANHandle = &H53
        ''' <summary>
        ''' PCAN-USB interface, channel 4
        ''' </summary>
        Public Const POBDII_USBBUS4 As TPOBDIICANHandle = &H54
        ''' <summary>
        ''' PCAN-USB interface, channel 5
        ''' </summary>
        Public Const POBDII_USBBUS5 As TPOBDIICANHandle = &H55
        ''' <summary>
        ''' PCAN-USB interface, channel 6
        ''' </summary>
        Public Const POBDII_USBBUS6 As TPOBDIICANHandle = &H56
        ''' <summary>
        ''' PCAN-USB interface, channel 7
        ''' </summary>
        Public Const POBDII_USBBUS7 As TPOBDIICANHandle = &H57
        ''' <summary>
        ''' PCAN-USB interface, channel 8
        ''' </summary>
        Public Const POBDII_USBBUS8 As TPOBDIICANHandle = &H58
        ''' <summary>
        ''' PCAN-USB interface, channel 9
        ''' </summary>
        Public Const POBDII_USBBUS9 As TPOBDIICANHandle = &H509
        ''' <summary>
        ''' PCAN-USB interface, channel 10
        ''' </summary>
        Public Const POBDII_USBBUS10 As TPOBDIICANHandle = &H50A
        ''' <summary>
        ''' PCAN-USB interface, channel 11
        ''' </summary>
        Public Const POBDII_USBBUS11 As TPOBDIICANHandle = &H50B
        ''' <summary>
        ''' PCAN-USB interface, channel 12
        ''' </summary>
        Public Const POBDII_USBBUS12 As TPOBDIICANHandle = &H50C
        ''' <summary>
        ''' PCAN-USB interface, channel 13
        ''' </summary>
        Public Const POBDII_USBBUS13 As TPOBDIICANHandle = &H50D
        ''' <summary>
        ''' PCAN-USB interface, channel 14
        ''' </summary>
        Public Const POBDII_USBBUS14 As TPOBDIICANHandle = &H50E
        ''' <summary>
        ''' PCAN-USB interface, channel 15
        ''' </summary>
        Public Const POBDII_USBBUS15 As TPOBDIICANHandle = &H50F
        ''' <summary>
        ''' PCAN-USB interface, channel 16
        ''' </summary>
        Public Const POBDII_USBBUS16 As TPOBDIICANHandle = &H510

        ''' <summary>
        ''' PCAN-PC Card interface, channel 1
        ''' </summary>
        Public Const POBDII_PCCBUS1 As TPOBDIICANHandle = &H61
        ''' <summary>
        ''' PCAN-PC Card interface, channel 2
        ''' </summary>
        Public Const POBDII_PCCBUS2 As TPOBDIICANHandle = &H62

        ''' <summary>
        ''' PCAN-LAN interface, channel 1
        ''' </summary>
        Public Const POBDII_LANBUS1 As TPOBDIICANHandle = &H801
        ''' <summary>
        ''' PCAN-LAN interface, channel 2
        ''' </summary>
        Public Const POBDII_LANBUS2 As TPOBDIICANHandle = &H802
        ''' <summary>
        ''' PCAN-LAN interface, channel 3
        ''' </summary>
        Public Const POBDII_LANBUS3 As TPOBDIICANHandle = &H803
        ''' <summary>
        ''' PCAN-LAN interface, channel 4
        ''' </summary>
        Public Const POBDII_LANBUS4 As TPOBDIICANHandle = &H804
        ''' <summary>
        ''' PCAN-LAN interface, channel 5
        ''' </summary>
        Public Const POBDII_LANBUS5 As TPOBDIICANHandle = &H805
        ''' <summary>
        ''' PCAN-LAN interface, channel 6
        ''' </summary>
        Public Const POBDII_LANBUS6 As TPOBDIICANHandle = &H806
        ''' <summary>
        ''' PCAN-LAN interface, channel 7
        ''' </summary>
        Public Const POBDII_LANBUS7 As TPOBDIICANHandle = &H807
        ''' <summary>
        ''' PCAN-LAN interface, channel 8
        ''' </summary>
        Public Const POBDII_LANBUS8 As TPOBDIICANHandle = &H808
        ''' <summary>
        ''' PCAN-LAN interface, channel 9
        ''' </summary>
        Public Const POBDII_LANBUS9 As TPOBDIICANHandle = &H809
        ''' <summary>
        ''' PCAN-LAN interface, channel 10
        ''' </summary>
        Public Const POBDII_LANBUS10 As TPOBDIICANHandle = &H80A
        ''' <summary>
        ''' PCAN-LAN interface, channel 11
        ''' </summary>
        Public Const POBDII_LANBUS11 As TPOBDIICANHandle = &H80B
        ''' <summary>
        ''' PCAN-LAN interface, channel 12
        ''' </summary>
        Public Const POBDII_LANBUS12 As TPOBDIICANHandle = &H80C
        ''' <summary>
        ''' PCAN-LAN interface, channel 13
        ''' </summary>
        Public Const POBDII_LANBUS13 As TPOBDIICANHandle = &H80D
        ''' <summary>
        ''' PCAN-LAN interface, channel 14
        ''' </summary>
        Public Const POBDII_LANBUS14 As TPOBDIICANHandle = &H80E
        ''' <summary>
        ''' PCAN-LAN interface, channel 15
        ''' </summary>
        Public Const POBDII_LANBUS15 As TPOBDIICANHandle = &H80F
        ''' <summary>
        ''' PCAN-LAN interface, channel 16
        ''' </summary>
        Public Const POBDII_LANBUS16 As TPOBDIICANHandle = &H810
#End Region

#Region "Parameter values definition"
        ''' <summary>
        ''' No debug messages
        ''' </summary>
        Public Const POBDII_DEBUG_NONE As Byte = 0
        ''' <summary>
        ''' Puts CAN debug messages to stdout
        ''' </summary>
        Public Const POBDII_DEBUG_CAN As Byte = 1
        ''' <summary>
        ''' No logging
        ''' </summary>
        Public Const POBDII_LOGGING_NONE As Byte = &H0
        ''' <summary>
        ''' Log function enter/exit data to file
        ''' </summary>
        Public Const POBDII_LOGGING_TO_FILE As Byte = &H1
        ''' <summary>
        ''' Log function enter/exit data to stdout
        ''' </summary>
        Public Const POBDII_LOGGING_TO_STDOUT As Byte = &H2
        ''' <summary>
        ''' The Channel is illegal or not available
        ''' </summary>
        Public Const POBDII_CHANNEL_UNAVAILABLE As Byte = &H0
        ''' <summary>
        ''' The Channel is available
        ''' </summary>
        Public Const POBDII_CHANNEL_AVAILABLE As Byte = &H1
        ''' <summary>
        ''' The Channel is valid, and is being used
        ''' </summary>
        Public Const POBDII_CHANNEL_OCCUPIED As Byte = &H2
        ''' <summary>
        ''' 11 Bit CAN Identifier
        ''' </summary>
        Public Const POBDII_CAN_ID_11BIT As Byte = 11
        ''' <summary>
        ''' 29 Bit CAN Identifier
        ''' </summary>
        Public Const POBDII_CAN_ID_29BIT As Byte = 29
#End Region

#Region "POBDII API Implementation"
#Region "Initialization and Configuration"
        ''' <summary>
        ''' Initializes an OBDII-Client based on a CAN-Channel
        ''' </summary>
        ''' <remarks>Only one OBDII-Client can be initialized per CAN-Channel</remarks>
        ''' <param name="CanChannel">The PCAN-Basic channel to be used as OBDII client</param>
        ''' <param name="HwType">NON PLUG'n PLAY: The type of hardware and operation mode</param>
        ''' <param name="IOPort">NON PLUG'n PLAY: The I/O address for the parallel port</param>
        ''' <param name="Interrupt">NON PLUG'n PLAY: Interrupt number of the parallel port</param>
        ''' <returns>A TPOBDIIStatus code. POBDII_ERROR_OK is returned on success</returns>
        <DllImport("PCAN-OBDII.dll", EntryPoint:="OBDII_Initialize")> _
        Public Shared Function Initialize( _
            <MarshalAs(UnmanagedType.U2)> _
            ByVal CanChannel As TPOBDIICANHandle, _
            <MarshalAs(UnmanagedType.U1)> _
            ByVal Baudrate As TPOBDIIBaudrateInfo, _
            <MarshalAs(UnmanagedType.U1)> _
            ByVal HwType As TPOBDIIHwType, _
            ByVal IOPort As UInt32, _
            ByVal Interrupt As UInt16) As TPOBDIIStatus
        End Function

        ''' <summary>
        ''' Initializes an OBDII-Client based on a CAN-Channel
        ''' </summary>
        ''' <remarks>Only one OBDII-Client can be initialized per CAN-Channel</remarks>
        ''' <param name="CanChannel">The PCAN-Basic channel to be used as OBDII client</param>
        ''' <returns>A TPOBDIIStatus code. POBDII_ERROR_OK is returned on success</returns>
        Public Shared Function Initialize( _
            ByVal CanChannel As TPOBDIICANHandle) As TPOBDIIStatus
            Return Initialize(CanChannel, TPOBDIIBaudrateInfo.POBDII_BAUDRATE_AUTODETECT, 0, 0, 0)
        End Function

        ''' <summary>
        ''' Uninitializes an OBDII-Client initialized before
        ''' </summary>
        ''' <param name="CanChannel">An OBDII CAN-Channel representing a OBDII-Client</param>
        ''' <returns>A TPOBDIIStatus code. POBDII_ERROR_OK is returned on success</returns>
        <DllImport("PCAN-OBDII.dll", EntryPoint:="OBDII_Uninitialize")> _
        Public Shared Function Uninitialize( _
            <MarshalAs(UnmanagedType.U2)> _
            ByVal CanChannel As TPOBDIICANHandle) As TPOBDIIStatus
        End Function

        ''' <summary>
        ''' Resets the receive and transmit queues of an OBDII-Client 
        ''' </summary>
        ''' <param name="CanChannel">An OBDII CAN-Channel representing a OBDII-Client</param>
        ''' <returns>A TPOBDIIStatus code. POBDII_ERROR_OK is returned on success</returns>
        <DllImport("PCAN-OBDII.dll", EntryPoint:="OBDII_Reset")> _
        Public Shared Function Reset( _
            <MarshalAs(UnmanagedType.U2)> _
            ByVal CanChannel As TPOBDIICANHandle) As TPOBDIIStatus
        End Function

        ''' <summary>
        ''' Gets the initialization status of an OBDII CAN-Channel
        ''' </summary>
        ''' <param name="CanChannel">An OBDII CAN-Channel representing a OBDII-Client</param>
        ''' <returns>A TPOBDIIStatus code. POBDII_ERROR_OK is returned on success</returns>
        <DllImport("PCAN-OBDII.dll", EntryPoint:="OBDII_GetStatus")> _
        Public Shared Function GetStatus( _
            <MarshalAs(UnmanagedType.U2)> _
            ByVal CanChannel As TPOBDIICANHandle) As TPOBDIIStatus
        End Function

        ''' <summary>
        ''' Retrieves an OBDII-Client value
        ''' </summary>
        ''' <param name="CanChannel">An OBDII CAN-Channel representing a OBDII-Client</param>
        ''' <param name="Parameter">The TPOBDIIParameter parameter to get</param>
        ''' <param name="StringBuffer">Buffer for the parameter value</param>
        ''' <param name="BufferLength">Size in bytes of the buffer</param>
        ''' <returns>A TPOBDIIStatus code. POBDII_ERROR_OK is returned on success</returns>
        <DllImport("PCAN-OBDII.dll", EntryPoint:="OBDII_GetValue")> _
        Public Shared Function GetValue( _
            <MarshalAs(UnmanagedType.U2)> _
            ByVal CanChannel As TPOBDIICANHandle, _
            <MarshalAs(UnmanagedType.U1)> _
            ByVal Parameter As TPOBDIIParameter, _
            ByVal StringBuffer As StringBuilder, _
            ByVal BufferLength As UInt32) As TPOBDIIStatus
        End Function
        ''' <summary>
        ''' Retrieves an OBDII-Client value
        ''' </summary>
        ''' <param name="CanChannel">An OBDII CAN-Channel representing a OBDII-Client</param>
        ''' <param name="Parameter">The TPOBDIIParameter parameter to get</param>
        ''' <param name="NumericBuffer">Buffer for the parameter value</param>
        ''' <param name="BufferLength">Size in bytes of the buffer</param>
        ''' <returns>A TPOBDIIStatus code. POBDII_ERROR_OK is returned on success</returns>
        <DllImport("PCAN-OBDII.dll", EntryPoint:="OBDII_GetValue")> _
        Public Shared Function GetValue( _
            <MarshalAs(UnmanagedType.U2)> _
            ByVal CanChannel As TPOBDIICANHandle, _
            <MarshalAs(UnmanagedType.U1)> _
            ByVal Parameter As TPOBDIIParameter, _
            ByRef NumericBuffer As UInt32, _
            ByVal BufferLength As UInt32) As TPOBDIIStatus
        End Function
        ''' <summary>
        ''' Retrieves an OBDII-Client value
        ''' </summary>
        ''' <param name="CanChannel">An OBDII CAN-Channel representing a OBDII-Client</param>
        ''' <param name="Parameter">The TPOBDIIParameter parameter to get</param>
        ''' <param name="Buffer">Buffer for the parameter value</param>
        ''' <param name="BufferLength">Size in bytes of the buffer</param>
        ''' <returns>A TPOBDIIStatus code. POBDII_ERROR_OK is returned on success</returns>
        <DllImport("PCAN-OBDII.dll", EntryPoint:="OBDII_GetValue")> _
        Public Shared Function GetValue( _
            <MarshalAs(UnmanagedType.U2)> _
            ByVal CanChannel As TPOBDIICANHandle, _
            <MarshalAs(UnmanagedType.U1)> _
            ByVal Parameter As TPOBDIIParameter, _
            <MarshalAs(UnmanagedType.LPArray)> _
            ByVal Buffer As Byte(), _
            ByVal BufferLength As UInt32) As TPOBDIIStatus
        End Function

        ''' <summary>
        ''' Configures or sets an OBDII-Client value
        ''' </summary>
        ''' <param name="CanChannel">An OBDII CAN-Channel representing a OBDII-Client</param>
        ''' <param name="Parameter">The TPOBDIIParameter parameter to set</param>
        ''' <param name="NumericBuffer">Buffer with the value to be set</param>
        ''' <param name="BufferLength">Size in bytes of the buffer</param>
        ''' <returns>A TPOBDIIStatus code. POBDII_ERROR_OK is returned on success</returns>
        <DllImport("PCAN-OBDII.dll", EntryPoint:="OBDII_SetValue")> _
        Public Shared Function SetValue( _
            <MarshalAs(UnmanagedType.U2)> _
            ByVal CanChannel As TPOBDIICANHandle, _
            <MarshalAs(UnmanagedType.U1)> _
            ByVal Parameter As TPOBDIIParameter, _
            ByRef NumericBuffer As UInt32, _
            ByVal BufferLength As UInt32) As TPOBDIIStatus
        End Function

        ''' <summary>
        ''' Returns the unit and scaling information for a specified Unit and Scaling ID
        ''' </summary>
        ''' <param name="id">Unit and Scaling ID</param>
        <DllImport("PCAN-OBDII.dll", EntryPoint:="OBDII_GetUnitAndScaling")> _
        Public Shared Function GetUnitAndScaling( _
            ByVal id As Byte, _
            ByRef unitAndScaling As TPOBDIIUnitAndScaling) As TPOBDIIStatus
        End Function
#End Region

#Region "Services Implementation"
#Region "Service 01h: OBDII_RequestCurrentData"
        ''' <summary>
        ''' Sends an OBDII Service $01 request into queue and waits for the received response
        ''' </summary>
        ''' <param name="CanChannel">An OBDII CAN-Channel representing a OBDII-Client</param>
        ''' <param name="Pid">A Parameter Identifier</param>
        ''' <param name="Data">A TPOBDIIParamData structure array to store the OBDII responses</param>
        ''' <param name="DataLen">A length of provided Data array</param>
        ''' <returns>A TPOBDIIStatus code. POBDII_ERROR_OK is returned on success</returns>
        <DllImport("PCAN-OBDII.dll", EntryPoint:="OBDII_RequestCurrentData")> _
        Public Shared Function RequestCurrentData( _
            <MarshalAs(UnmanagedType.U2)> _
            ByVal CanChannel As TPOBDIICANHandle, _
            <MarshalAs(UnmanagedType.U1)> _
            ByVal Pid As TPOBDIIPid, _
            <[In](), Out()> ByVal Data As TPOBDIIParamData(), _
            ByVal DataLen As Byte) As TPOBDIIStatus
        End Function
#End Region

#Region "Service 02h: RequestFreezeFrameData"
        ''' <summary>
        ''' Sends an OBDII Service $02 request into queue and waits for the received response
        ''' </summary>
        ''' <param name="CanChannel">An OBDII CAN-Channel representing a OBDII-Client</param>
        ''' <param name="Pid">A Parameter Identifier</param>
        ''' <param name="Frame">A Frame number</param>
        ''' <param name="Data">A TPOBDIIParamData structure array to store the OBDII responses</param>
        ''' <param name="DataLen">A length of provided Data array</param>
        ''' <returns>A TPOBDIIStatus code. POBDII_ERROR_OK is returned on success</returns>
        <DllImport("PCAN-OBDII.dll", EntryPoint:="OBDII_RequestFreezeFrameData")> _
        Public Shared Function RequestFreezeFrameData( _
            <MarshalAs(UnmanagedType.U2)> _
            ByVal CanChannel As TPOBDIICANHandle, _
            <MarshalAs(UnmanagedType.U1)> _
            ByVal Pid As TPOBDIIPid, _
            ByVal Frame As Byte, _
            <[In](), Out()> ByVal Data As TPOBDIIParamData(), _
            ByVal DataLen As Byte) As TPOBDIIStatus
        End Function
#End Region

#Region "Service 03h: RequestStoredTroubleCodes"
        ''' <summary>
        ''' Sends an OBDII Service $03 request into queue and waits for the received response
        ''' </summary>
        ''' <param name="CanChannel">An OBDII CAN-Channel representing a OBDII-Client</param>
        ''' <param name="Data">A TPOBDIIDTCData structure array to store the OBDII responses</param>
        ''' <param name="DataLen">A length of provided Data array</param>
        ''' <returns>A TPOBDIIStatus code. POBDII_ERROR_OK is returned on success</returns>
        <DllImport("PCAN-OBDII.dll", EntryPoint:="OBDII_RequestStoredTroubleCodes")> _
        Public Shared Function RequestStoredTroubleCodes( _
            <MarshalAs(UnmanagedType.U2)> _
            ByVal CanChannel As TPOBDIICANHandle, _
            <[In](), Out()> ByVal Data As TPOBDIIDTCData(), _
           ByVal DataLen As Byte) As TPOBDIIStatus
        End Function
#End Region

#Region "Service 04h: ClearTroubleCodes"
        ''' <summary>
        ''' Sends an OBDII Service $04 request into queue and waits for the received response
        ''' </summary>
        ''' <param name="CanChannel">An OBDII CAN-Channel representing a OBDII-Client</param>
        ''' <param name="Data">A TPOBDIIResponse structure array to store the OBDII responses</param>
        ''' <param name="DataLen">A length of provided Data array</param>
        ''' <returns>A TPOBDIIStatus code. POBDII_ERROR_OK is returned on success</returns>
        <DllImport("PCAN-OBDII.dll", EntryPoint:="OBDII_ClearTroubleCodes")> _
        Public Shared Function ClearTroubleCodes( _
            <MarshalAs(UnmanagedType.U2)> _
            ByVal CanChannel As TPOBDIICANHandle, _
            <[In](), Out()> ByVal Data As TPOBDIIResponse(), _
            ByVal DataLen As Byte) As TPOBDIIStatus
        End Function
#End Region

#Region "Service 06h: RequestTestResults"
        ''' <summary>
        ''' Sends an OBDII Service $06 request into queue and waits for the received response
        ''' </summary>
        ''' <param name="CanChannel">An OBDII CAN-Channel representing a OBDII-Client</param>
        ''' <param name="OBDMid">An On-Board Diagnostic Monitor Identifier</param>
        ''' <param name="Data">A TPOBDIIOBDMid structure array to store the OBDII responses</param>
        ''' <param name="DataLen">A length of provided Data array</param>
        ''' <returns>A TPOBDIIStatus code. POBDII_ERROR_OK is returned on success</returns>
        <DllImport("PCAN-OBDII.dll", EntryPoint:="OBDII_RequestTestResults")> _
        Public Shared Function RequestTestResults( _
            <MarshalAs(UnmanagedType.U2)> _
            ByVal CanChannel As TPOBDIICANHandle, _
            <MarshalAs(UnmanagedType.U1)> _
            ByVal OBDMid As TPOBDIIOBDMid, _
            <[In](), Out()> ByVal Data As TPOBDIIMonitorData(), _
            ByVal DataLen As Byte) As TPOBDIIStatus
        End Function
#End Region

#Region "Service 07h: RequestPendingTroubleCodes"
        ''' <summary>
        ''' Sends an OBDII Service $07 request into queue and waits for the received response
        ''' </summary>
        ''' <param name="CanChannel">An OBDII CAN-Channel representing a OBDII-Client</param>
        ''' <param name="Data">A TPOBDIIDTCData structure array to store the OBDII responses</param>
        ''' <param name="DataLen">A length of provided Data array</param>
        ''' <returns>A TPOBDIIStatus code. POBDII_ERROR_OK is returned on success</returns>
        <DllImport("PCAN-OBDII.dll", EntryPoint:="OBDII_RequestPendingTroubleCodes")> _
        Public Shared Function RequestPendingTroubleCodes( _
            <MarshalAs(UnmanagedType.U2)> _
            ByVal CanChannel As TPOBDIICANHandle, _
            <[In](), Out()> ByVal Data As TPOBDIIDTCData(), _
            ByVal DataLen As Byte) As TPOBDIIStatus
        End Function
#End Region

#Region "Service 08h: RequestControlOperation"
        ''' <summary>
        ''' Sends an OBDII Service $08 request into queue and waits for the received response
        ''' </summary>
        ''' <param name="CanChannel">An OBDII CAN-Channel representing a OBDII-Client</param>
        ''' <param name="Tid">A Test Identifier</param>
        ''' <param name="Data">A TPOBDIIResponse structure array to store the OBDII responses</param>
        ''' <param name="DataLen">A length of provided Data array</param>
        ''' <returns>A TPOBDIIStatus code. POBDII_ERROR_OK is returned on success</returns>
        <DllImport("PCAN-OBDII.dll", EntryPoint:="OBDII_RequestControlOperation")> _
        Public Shared Function RequestControlOperation( _
            <MarshalAs(UnmanagedType.U2)> _
            ByVal CanChannel As TPOBDIICANHandle, _
            <MarshalAs(UnmanagedType.U1)> _
            ByVal Tid As TPOBDIITid, _
           <[In](), Out()> ByVal Data As TPOBDIIResponse(), _
            ByVal DataLen As Byte) As TPOBDIIStatus
        End Function
#End Region

#Region "Service 09h: RequestVehicleInformation"
        ''' <summary>
        ''' Sends an OBDII Service $09 request into queue and waits for the received response
        ''' </summary>
        ''' <param name="CanChannel">An OBDII CAN-Channel representing a OBDII-Client</param>
        ''' <param name="InfoType">An InfoType</param>
        ''' <param name="Data">A TPOBDIIInfoData structure array to store the OBDII responses</param>
        ''' <param name="DataLen">A length of provided Data array</param>
        ''' <returns>A TPOBDIIStatus code. POBDII_ERROR_OK is returned on success</returns>
        <DllImport("PCAN-OBDII.dll", EntryPoint:="OBDII_RequestVehicleInformation")> _
        Public Shared Function RequestVehicleInformation( _
            <MarshalAs(UnmanagedType.U2)> _
            ByVal CanChannel As TPOBDIICANHandle, _
            <MarshalAs(UnmanagedType.U1)> _
            ByVal InfoType As TPOBDIIInfoType, _
            <[In](), Out()> ByVal Data As TPOBDIIInfoData(), _
            ByVal DataLen As Byte) As TPOBDIIStatus
        End Function
#End Region

#Region "Service 0Ah: RequestPermanentTroubleCodes"
        ''' <summary>
        ''' Sends an OBDII Service $0A request into queue and waits for the received response
        ''' </summary>
        ''' <param name="CanChannel">An OBDII CAN-Channel representing a OBDII-Client</param>
        ''' <param name="Data">A TPOBDIIDTCData structure array to store the OBDII responses</param>
        ''' <param name="DataLen">A length of provided Data array</param>
        ''' <returns>A TPOBDIIStatus code. POBDII_ERROR_OK is returned on success</returns>
        <DllImport("PCAN-OBDII.dll", EntryPoint:="OBDII_RequestPermanentTroubleCodes")> _
        Public Shared Function RequestPermanentTroubleCodes( _
            <MarshalAs(UnmanagedType.U2)> _
            ByVal CanChannel As TPOBDIICANHandle, _
            <[In](), Out()> ByVal Data As TPOBDIIDTCData(), _
            ByVal DataLen As Byte) As TPOBDIIStatus
        End Function
#End Region
#End Region
#End Region
    End Class
End Namespace