//  POBDII.pas
//
//  ~~~~~~~~~~~~
//
//  PCAN-OBDII API
//
//  ~~~~~~~~~~~~
//
//  ------------------------------------------------------------------
//  Author : Fabrice Vergnaud
//	Last changed by:	$Author: Fabrice $
//  Last changed date:	$Date: 2019-10-11 13:53:39 +0200 (Fri, 11 Oct 2019) $
//
//  Language: Delphi
//  ------------------------------------------------------------------
//
//  Copyright (C) 2015  PEAK-System Technik GmbH, Darmstadt
//  more Info at http://www.peak-system.com 
//

unit POBDII;

interface

// Aliases definition
//
type
    TPOBDIICANHandle = Word;   // Represents a CAN hardware channel of the underlying CAN system
    TPOBDIIPid = Byte;         // Represents the PID number
    TPOBDIIOBDMid = Byte;      // Represents the OBDMID number
    TPOBDIITid = Byte;         // Represents the TID number
    TPOBDIIInfoType = Byte;    // Represents the InfoType number

{$Z1}
    /// <summary>
    /// OBDII baudrates
    /// </summary>
    TPOBDIIBaudrateInfo = (
        /// <summary>
        /// Non legislated-OBD baudrate
        /// </summary>
        POBDII_BAUDRATE_NON_LEGISLATED = $00,
        /// <summary>
        /// Baudrate: 250 kBit/s
        /// </summary>
        POBDII_BAUDRATE_250K = $01,
        /// <summary>
        /// Baudrate: 500 kBit/s
        /// </summary>
        POBDII_BAUDRATE_500K = $02,
        /// <summary>
        /// Baudrate: Autodetect
        /// </summary>
        POBDII_BAUDRATE_AUTODETECT = $FF
    );

{$Z1}
    /// <summary>
    /// OBDII address definitions
    /// </summary>
    TPOBDIIAddress = (
        /// <summary>
        /// ECU 1
        /// </summary>
        POBDII_ECU_1 = $01,
        /// <summary>
        /// ECU 2
        /// </summary>
        POBDII_ECU_2 = $02,
        /// <summary>
        /// ECU 3
        /// </summary>
        POBDII_ECU_3 = $03,
        /// <summary>
        /// ECU 4
        /// </summary>
        POBDII_ECU_4 = $04,
        /// <summary>
        /// ECU 5
        /// </summary>
        POBDII_ECU_5 = $05,
        /// <summary>
        /// ECU 6
        /// </summary>
        POBDII_ECU_6 = $06,
        /// <summary>
        /// ECU 7
        /// </summary>
        POBDII_ECU_7 = $07,
        /// <summary>
        /// ECU 8
        /// </summary>
        POBDII_ECU_8 = $08
    );

{$Z1}
    /// <summary>
    /// Represents the OBDII Service numbers
    /// </summary>
    TPOBDIIService = (
        /// <summary>
        /// Service 01  Request Current Powertrain Diagnostic Data
        /// </summary>
        POBDII_SERVICE_01 = $01,
        /// <summary>
        /// Service 02  Request Powertrain Freeze Frame Data
        /// </summary>
        POBDII_SERVICE_02 = $02,
        /// <summary>
        /// Service 03  Request Emission-Related Diagnostic Trouble Codes
        /// </summary>
        POBDII_SERVICE_03 = $03,
        /// <summary>
        /// Service 04  Clear/Reset Emission-Related Diagnostic Information
        /// </summary>
        POBDII_SERVICE_04 = $04,
        /// <summary>
        /// Service 06  Request On-Board Monitoring Test Results for Specific Monitored Systems
        /// </summary>
        POBDII_SERVICE_06 = $06,
        /// <summary>
        /// Service 07  Request Emission-Related Diagnostic Trouble Codes Detected During Current or Last Completed Driving Cycle
        /// </summary>
        POBDII_SERVICE_07 = $07,
        /// <summary>
        /// Service 08  Request Control of On-Board System, Test or Component
        /// </summary>
        POBDII_SERVICE_08 = $08,
        /// <summary>
        /// Service 09  Request Vehicle Information
        /// </summary>
        POBDII_SERVICE_09 = $09,
        /// <summary>
        /// Service 0A  Request Emission-Related Diagnostic Trouble Codes with Permanent Status
        /// </summary>
        POBDII_SERVICE_0A = $0A
    );

{$Z1}
    /// <summary>
    /// PCANTP parameters
    /// </summary>
    TPOBDIIParameter = (
        /// <summary>
        /// 1 BYTE data describing the logging mode 
        /// </summary>
        POBDII_PARAM_LOGGING = $B1,
        /// <summary>
        /// 1 BYTE data describing the number of available ECUs 
        /// </summary>
        POBDII_PARAM_AVAILABLE_ECUS = $B2,  
        /// <summary>
        /// 256 Byte data describing the supported PIDs for Service 01: Current Data
        /// </summary>
        POBDII_PARAM_SUPPORTMASK_PIDS = $B3,
        /// <summary>
		/// 257 Byte data describing the supported PIDs of Frame 
		/// (identified by the first BYTE in the buffer) for Service 02: Freeze Frame Data 
        /// </summary>
        POBDII_PARAM_SUPPORTMASK_FFPIDS = $B4,
        /// <summary>
        /// 256 Byte data describing the supported OBDMIDs for Service 06: Test Results
        /// </summary>
        POBDII_PARAM_SUPPORTMASK_OBDMIDS = $B5,
        /// <summary>
        /// 256 Byte data describing the supported TIDs for Service 08: Control Operation
        /// </summary>
        POBDII_PARAM_SUPPORTMASK_TIDS = $B6,
        /// <summary>
        /// 256 Byte data describing the supported InfoTypes for Service 09: Vehicle Information
        /// </summary>
        POBDII_PARAM_SUPPORTMASK_INFOTYPES = $B7,
        /// <summary>
        /// API version parameter
        /// </summary>
        POBDII_PARAM_API_VERSION = $B8,
        /// <summary>
        /// 1 BYTE data describing baudrate value
        /// </summary>
        POBDII_PARAM_BAUDRATE = $B9,
        /// <summary>
        /// 1 BYTE data describing CAN identifier length
        /// </summary>
        POBDII_PARAM_CAN_ID = $BA,             
        /// <summary>
        /// 1 BYTE data describing the debug mode 
        /// </summary>
        POBDII_PARAM_DEBUG = $E3,    
        /// <summary>
        /// 1 Byte data describing the condition of a channel
        /// </summary>
        POBDII_PARAM_CHANNEL_CONDITION =$E4
    );

{$Z1}
    /// <summary>
    /// Represents the type of CAN hardware to be initialized
    /// </summary>
    TPOBDIIHwType = (
        /// <summary>
        /// PCAN-ISA 82C200
        /// </summary>
        POBDII_HWTYPE_ISA = $01,
        /// <summary>
        /// PCAN-ISA SJA1000
        /// </summary>
        POBDII_HWTYPE_ISA_SJA = $09,
        /// <summary>
        /// PHYTEC ISA 
        /// </summary>
        POBDII_HWTYPE_ISA_PHYTEC = $04,
        /// <summary>
        /// PCAN-Dongle 82C200
        /// </summary>
        POBDII_HWTYPE_DNG = $02,
        /// <summary>
        /// PCAN-Dongle EPP 82C200
        /// </summary>
        POBDII_HWTYPE_DNG_EPP = $03,
        /// <summary>
        /// PCAN-Dongle SJA1000
        /// </summary>
        POBDII_HWTYPE_DNG_SJA = $05,
        /// <summary>
        /// PCAN-Dongle EPP SJA1000
        /// </summary>
        POBDII_HWTYPE_DNG_SJA_EPP = $06
    );

{$Z4}
    /// <summary>
    ///  Represents a OBDII status/error code
    /// </summary>
    TPOBDIIStatus = (
        /// <summary>
        /// No error 
        /// </summary>
        POBDII_ERROR_OK                             = $00000,
        /// <summary>
        /// Not Initialized
        /// </summary>
        POBDII_ERROR_NOT_INITIALIZED                      = $00001,
        /// <summary>
        /// Already Initialized
        /// </summary>
        POBDII_ERROR_ALREADY_INITIALIZED                  = $00002,
        /// <summary>
        /// Could not obtain memory
        /// </summary>
        POBDII_ERROR_NO_MEMORY                            = $00003,
        /// <summary>
        /// Input buffer overflow
        /// </summary>
        POBDII_ERROR_OVERFLOW                             = $00004,
        /// <summary>
        /// Timeout while accessing the UDS mutex
        /// </summary>
        POBDII_ERROR_TIMEOUT                            = $00006,
        /// <summary>
        /// No Message available
        /// </summary>
        POBDII_ERROR_NO_MESSAGE                         = $00007,
        /// <summary>
        /// Wrong parameters in query
        /// </summary>
        POBDII_ERROR_WRONG_PARAM                        = $00008,
        /// <summary>
        /// Buffer to parse contains response to Vehicle specific parameter
        /// </summary>
        POBDII_ERROR_NOT_SUPPORTED                      = $00009,
        /// <summary>
        /// Could not parse response
        /// </summary>
        POBDII_ERROR_PARSE_ERROR						= $0000A,
        /// <summary>
        /// Channel is in BUS-LIGHT error state
        /// </summary>
        POBDII_ERROR_BUSLIGHT							= $0000B,
        /// <summary>
        /// Channel is in BUS-HEAVY error state
        /// </summary>
        POBDII_ERROR_BUSHEAVY							= $0000C,
        /// <summary>
        /// Channel is in BUS-OFF error state
        /// </summary>
        POBDII_ERROR_BUSOFF								= $0000D,
        /// <summary>
        /// No connected ECUs, or ECU found is not supported
        /// </summary>
        POBDII_ERROR_UNSUPPORTED_ECUS                   = $0000E,
        /// <summary>
        /// $10000 .. $1ffff CAN System dependend (PCAN-Basic). The value must 
        /// be shifted 2 bytes to the right, then it is a PCAN-Basic Status value.
        /// </summary>
        POBDII_ERROR_CAN_ERROR							= $10000
    );

{$Z1}
    /// <summary>
    /// Represents the OBD-II Response codes
    /// </summary>
    TPOBDIIError = (
        /// <summary>
        /// No error, Positive response
        /// </summary>
        POBDII_R_NO_ERROR = 0,
        /// <summary>
        /// Server is busy
        /// </summary>
        POBDII_R_BUSY_REPEAT_REQUEST = $21,
        /// <summary>
        /// Conditions not correct
        /// </summary>
        POBDII_R_CONDITIONS_NOT_CORRECT = $22,
        /// <summary>
        /// server needs more time
        /// </summary>
        POBDII_R_RESPONSE_PENDING = $78,
        /// <summary>
        /// Not a response, invalid value
        /// </summary>
        POBDII_R_NOT_USED = $FF
    );

{$Z1}
    /// <summary>
    /// Represents the type of returned data
    /// </summary>
    TPOBDIIInfoDataType = (
        /// <summary>
        /// WORD sized data
        /// </summary>
        POBDII_INFOTYPE_COUNTER = $00,
        /// <summary>
        /// 4 Byte long calibration data
        /// </summary>
        POBDII_INFOTYPE_CALDATA = $01,
        /// <summary>
        /// Char buffer containing up to 20 characters
        /// </summary>
        POBDII_INFOTYPE_STRING = $02,
        /// <summary>
        /// Data is not an InfoType (used to identify raw data bit encoded InfoType Support)
        /// </summary>
        POBDII_INFOTYPE_NONE = $03
    );

	/// <summary>
    /// Represents the data returned by OBD service $04, $08
	/// </summary>
{$A8}
    TPOBDIIResponse = record
        /// <summary>
        /// ECU number
        /// </summary>
        SOURCE: Byte;
        /// <summary>
        /// Error number (POBDII_R_*)
        /// </summary>
        ERRORNR: TPOBDIIError;
        /// <summary>
        /// Raw data (without SID and Parameters)
        /// </summary>
        DATA: array[0..4094] of Byte;
        /// <summary>
        /// Number of bytes stored in DATA
        /// </summary>
        LEN: WORD;
        /// <summary>
        /// Service ID
        /// </summary>
        SID: Byte;
        /// <summary>
        /// Parameter ID (only with services $01, $02, $08, $09)
        /// </summary>
        PID: Byte;
        /// <summary>
        /// Frame number (only with OBD Service $02)
        /// </summary>
        FRAME: Byte;
    end;    
    PTPOBDIIResponse = ^TPOBDIIResponse;

	/// <summary>
    /// Represents the data returned by OBD service $01, $02
	/// </summary>
{$A8}
    TPOBDIIParamData = record
        /// <summary>
        /// Response definition data
        /// </summary>
        RESPONSE: TPOBDIIResponse;
        /// <summary>
        /// Raw data (without SID, PID and Frame)
        /// </summary>
        BUFFER: array[0..40] of Byte;
        /// <summary>
        /// Up to 10 double values
        /// </summary>
        DOUBLES: array[0..9] of Double;
        /// <summary>
        /// Mask for bytes that are bit encoded: A(0) to J(9)
        /// </summary>
        BYTEMASK: Word;
        /// <summary>
        /// Number of bytes stored in raw buffer (0 to 41)
        /// </summary>
        BLEN: Byte;
        /// <summary>
        /// Number of doubles stored (0 to 10)
        /// </summary>
        DLEN: Byte;
		/// <summary>
		/// A single DTC (incl. '\0')
		/// </summary>
        DTC: array[0..5] of Char;
    end;
    PTPOBDIIParamData = ^TPOBDIIParamData;

	/// <summary>
    /// Represents a DTC as a string of 6 bytes
	/// </summary>
    TPOBDIIDTCText = record
        ErrorText: String;
    end;

	/// <summary>
    /// Represents the data returned by OBD service $03, $07, $0A
	/// </summary>
{$A8}
    TPOBDIIDTCData = record
        /// <summary>
        /// Response definition data
        /// </summary>
        RESPONSE: TPOBDIIResponse;
        /// <summary>
        /// 10 DTCs each is 6 bytes (incl. '\0')
        /// </summary>
        DTC: array[0..9] of array[0..5] of Char;
        /// <summary>
        /// Number of DTCs stored (0 to 10)
        /// </summary>
        DLEN: Byte;
    end;      
    PTPOBDIIDTCData = ^TPOBDIIDTCData;

	/// <summary>
    /// Represents the data returned by OBD service $06
	/// </summary>
{$A8}
    TPOBDIIMonitorData = record
        /// <summary>
        /// Response definition data
        /// </summary>
        RESPONSE: TPOBDIIResponse;
        /// <summary>
        /// Test ID
        /// </summary>
        TID: Byte;
        /// <summary>
        /// Unit And Scaling ID
        /// </summary>
        UNITANDSCALING: Byte;
        /// <summary>
        /// Test value
        /// </summary>
        TESTVALUE: Double;
        /// <summary>
        /// Minimum Test Limit
        /// </summary>
        MINLIMIT: Double;
        /// <summary>
        /// Maximum Test Limit
        /// </summary>
        MAXLIMIT: Double;
    end;
    PTPOBDIIMonitorData = ^TPOBDIIMonitorData;

	/// <summary>
    /// Represents the data returned by OBD service $09
	/// </summary>
{$A8}
    TPOBDIIInfoData = record
        /// <summary>
        /// Response definition data
        /// </summary>
        RESPONSE: TPOBDIIResponse;
        /// <summary>
        /// Index Number of the counter or the string
        /// </summary>
        INDEXNR: Byte;
        /// <summary>
        /// Type of data available (POBDII_INFOTYPE_*)
        /// </summary>
        DATATYPE: TPOBDIIInfoDataType;
        /// <summary>
        /// Counter value
        /// </summary>
        COUNTER: Word;
        /// <summary>
        /// Calibration data
        /// </summary>
        CALDATA: array[0..3] of Byte;
        /// <summary>
        /// String value
        /// </summary>
        TEXT: array[0..20] of char;
    end;
    PTPOBDIIInfoData = ^TPOBDIIInfoData;    
            
    // Unit and Scaling definition
{$A8}
    TPOBDIIUnitAndScaling = record    
        /// <summary>
        /// Minimum value
        /// </summary>
        MIN: Double;
        /// <summary>
        /// Maximum value
        /// </summary>
        MAX: Double;
        /// <summary>
        /// Unit abbreviation
        /// </summary>
		UNIT_TXT: array[0..15] of Char;
    end;              
    PTPOBDIIUnitAndScaling = ^TPOBDIIUnitAndScaling;


////////////////////////////////////////////////////////////////////////////////
///    OBD-II API Class
///

    TObdiiApi = class
    public
        class var
        /// <summary>
        /// Undefined/default value for a PCAN bus
        /// </summary>
        const POBDII_NONEBUS: TPOBDIICANHandle = $00;

        /// <summary>
        /// PCAN-ISA interface, channel 1
        /// </summary>
        const POBDII_ISABUS1: TPOBDIICANHandle = $21;
        /// <summary>
        /// PCAN-ISA interface, channel 2
        /// </summary>
        const POBDII_ISABUS2: TPOBDIICANHandle = $22;
        /// <summary>
        /// PCAN-ISA interface, channel 3
        /// </summary>
        const POBDII_ISABUS3: TPOBDIICANHandle = $23;
        /// <summary>
        /// PCAN-ISA interface, channel 4
        /// </summary>
        const POBDII_ISABUS4: TPOBDIICANHandle = $24;
        /// <summary>
        /// PCAN-ISA interface, channel 5
        /// </summary>
        const POBDII_ISABUS5: TPOBDIICANHandle = $25;
        /// <summary>
        /// PCAN-ISA interface, channel 6
        /// </summary>
        const POBDII_ISABUS6: TPOBDIICANHandle = $26;
        /// <summary>
        /// PCAN-ISA interface, channel 7
        /// </summary>
        const POBDII_ISABUS7: TPOBDIICANHandle = $27;
        /// <summary>
        /// PCAN-ISA interface, channel 8
        /// </summary>
        const POBDII_ISABUS8: TPOBDIICANHandle = $28;

        /// <summary>
        /// PPCAN-Dongle/LPT interface, channel 1
        /// </summary>
        const POBDII_DNGBUS1: TPOBDIICANHandle = $31;

        /// <summary>
        /// PCAN-PCI interface, channel 1
        /// </summary>
        const POBDII_PCIBUS1: TPOBDIICANHandle = $41;
        /// <summary>
        /// PCAN-PCI interface, channel 2
        /// </summary>
        const POBDII_PCIBUS2: TPOBDIICANHandle = $42;
        /// <summary>
        /// PCAN-PCI interface, channel 3
        /// </summary>
        const POBDII_PCIBUS3: TPOBDIICANHandle = $43;
        /// <summary>
        /// PCAN-PCI interface, channel 4
        /// </summary>
        const POBDII_PCIBUS4: TPOBDIICANHandle = $44;
        /// <summary>
        /// PCAN-PCI interface, channel 5
        /// </summary>
        const POBDII_PCIBUS5: TPOBDIICANHandle = $45;
        /// <summary>
        /// PCAN-PCI interface, channel 6
        /// </summary>
        const POBDII_PCIBUS6: TPOBDIICANHandle = $46;
        /// <summary>
        /// PCAN-PCI interface, channel 7
        /// </summary>
        const POBDII_PCIBUS7: TPOBDIICANHandle = $47;
        /// <summary>
        /// PCAN-PCI interface, channel 8
        /// </summary>
        const POBDII_PCIBUS8: TPOBDIICANHandle = $48;
        /// <summary>
        /// PCAN-PCI interface, channel 9
        /// </summary>
        const POBDII_PCIBUS9: TPOBDIICANHandle = $409;
        /// <summary>
        /// PCAN-PCI interface, channel 10
        /// </summary>
        const POBDII_PCIBUS10: TPOBDIICANHandle = $40A;
        /// <summary>
        /// PCAN-PCI interface, channel 11
        /// </summary>
        const POBDII_PCIBUS11: TPOBDIICANHandle = $40B;
        /// <summary>
        /// PCAN-PCI interface, channel 12
        /// </summary>
        const POBDII_PCIBUS12: TPOBDIICANHandle = $40C;
        /// <summary>
        /// PCAN-PCI interface, channel 13
        /// </summary>
        const POBDII_PCIBUS13: TPOBDIICANHandle = $40D;
        /// <summary>
        /// PCAN-PCI interface, channel 14
        /// </summary>
        const POBDII_PCIBUS14: TPOBDIICANHandle = $40E;
        /// <summary>
        /// PCAN-PCI interface, channel 15
        /// </summary>
        const POBDII_PCIBUS15: TPOBDIICANHandle = $40F;
        /// <summary>
        /// PCAN-PCI interface, channel 16
        /// </summary>
        const POBDII_PCIBUS16: TPOBDIICANHandle = $410;

        /// <summary>
        /// PCAN-USB interface, channel 1
        /// </summary>
        const POBDII_USBBUS1: TPOBDIICANHandle = $51;
        /// <summary>
        /// PCAN-USB interface, channel 2
        /// </summary>
        const POBDII_USBBUS2: TPOBDIICANHandle = $52;
        /// <summary>
        /// PCAN-USB interface, channel 3
        /// </summary>
        const POBDII_USBBUS3: TPOBDIICANHandle = $53;
        /// <summary>
        /// PCAN-USB interface, channel 4
        /// </summary>
        const POBDII_USBBUS4: TPOBDIICANHandle = $54;
        /// <summary>
        /// PCAN-USB interface, channel 5
        /// </summary>
        const POBDII_USBBUS5: TPOBDIICANHandle = $55;
        /// <summary>
        /// PCAN-USB interface, channel 6
        /// </summary>
        const POBDII_USBBUS6: TPOBDIICANHandle = $56;
        /// <summary>
        /// PCAN-USB interface, channel 7
        /// </summary>
        const POBDII_USBBUS7: TPOBDIICANHandle = $57;
        /// <summary>
        /// PCAN-USB interface, channel 8
        /// </summary>
        const POBDII_USBBUS8: TPOBDIICANHandle = $58;
        /// <summary>
        /// PCAN-USB interface, channel 9
        /// </summary>
        const POBDII_USBBUS9: TPOBDIICANHandle = $509;
        /// <summary>
        /// PCAN-USB interface, channel 10
        /// </summary>
        const POBDII_USBBUS10: TPOBDIICANHandle = $50A;
        /// <summary>
        /// PCAN-USB interface, channel 11
        /// </summary>
        const POBDII_USBBUS11: TPOBDIICANHandle = $50B;
        /// <summary>
        /// PCAN-USB interface, channel 12
        /// </summary>
        const POBDII_USBBUS12: TPOBDIICANHandle = $50C;
        /// <summary>
        /// PCAN-USB interface, channel 13
        /// </summary>
        const POBDII_USBBUS13: TPOBDIICANHandle = $50D;
        /// <summary>
        /// PCAN-USB interface, channel 14
        /// </summary>
        const POBDII_USBBUS14: TPOBDIICANHandle = $50E;
        /// <summary>
        /// PCAN-USB interface, channel 15
        /// </summary>
        const POBDII_USBBUS15: TPOBDIICANHandle = $50F;
        /// <summary>
        /// PCAN-USB interface, channel 16
        /// </summary>
        const POBDII_USBBUS16: TPOBDIICANHandle = $510;

        /// <summary>
        /// PCAN-PC Card interface, channel 1
        /// </summary>
        const POBDII_PCCBUS1: TPOBDIICANHandle = $61;
        /// <summary>
        /// PCAN-PC Card interface, channel 2
        /// </summary>
        const POBDII_PCCBUS2: TPOBDIICANHandle = $62;

        /// <summary>
        /// PCAN-LAN interface, channel 1
        /// </summary>
        const POBDII_LANBUS1: TPOBDIICANHandle = $801;
        /// <summary>
        /// PCAN-LAN interface, channel 2
        /// </summary>
        const POBDII_LANBUS2: TPOBDIICANHandle = $802;
        /// <summary>
        /// PCAN-LAN interface, channel 3
        /// </summary>
        const POBDII_LANBUS3: TPOBDIICANHandle = $803;
        /// <summary>
        /// PCAN-LAN interface, channel 4
        /// </summary>
        const POBDII_LANBUS4: TPOBDIICANHandle = $804;
        /// <summary>
        /// PCAN-LAN interface, channel 5
        /// </summary>
        const POBDII_LANBUS5: TPOBDIICANHandle = $805;
        /// <summary>
        /// PCAN-LAN interface, channel 6
        /// </summary>
        const POBDII_LANBUS6: TPOBDIICANHandle = $806;
        /// <summary>
        /// PCAN-LAN interface, channel 7
        /// </summary>
        const POBDII_LANBUS7: TPOBDIICANHandle = $807;
        /// <summary>
        /// PCAN-LAN interface, channel 8
        /// </summary>
        const POBDII_LANBUS8: TPOBDIICANHandle = $808;
        /// <summary>
        /// PCAN-LAN interface, channel 9
        /// </summary>
        const POBDII_LANBUS9: TPOBDIICANHandle = $809;
        /// <summary>
        /// PCAN-LAN interface, channel 10
        /// </summary>
        const POBDII_LANBUS10: TPOBDIICANHandle = $80A;
        /// <summary>
        /// PCAN-LAN interface, channel 11
        /// </summary>
        const POBDII_LANBUS11: TPOBDIICANHandle = $80B;
        /// <summary>
        /// PCAN-LAN interface, channel 12
        /// </summary>
        const POBDII_LANBUS12: TPOBDIICANHandle = $80C;
        /// <summary>
        /// PCAN-LAN interface, channel 13
        /// </summary>
        const POBDII_LANBUS13: TPOBDIICANHandle = $80D;
        /// <summary>
        /// PCAN-LAN interface, channel 14
        /// </summary>
        const POBDII_LANBUS14: TPOBDIICANHandle = $80E;
        /// <summary>
        /// PCAN-LAN interface, channel 15
        /// </summary>
        const POBDII_LANBUS15: TPOBDIICANHandle = $80F;
        /// <summary>
        /// PCAN-LAN interface, channel 16
        /// </summary>
        const POBDII_LANBUS16: TPOBDIICANHandle = $810;


        /// <summary>
        /// No debug messages
        /// </summary>
        const POBDII_DEBUG_NONE: Byte = 0;
        /// <summary>
        /// Puts CAN debug messages to stdout
        /// </summary>
        const POBDII_DEBUG_CAN: Byte = 1;
        /// <summary>
        /// No logging
        /// </summary>
        const POBDII_LOGGING_NONE: Byte = 0;
        /// <summary>
        /// Log function enter/exit data to file
        /// </summary>
        const POBDII_LOGGING_TO_FILE: Byte = $01;
        /// <summary>
        /// Log function enter/exit data to stdout
        /// </summary>
        const POBDII_LOGGING_TO_STDOUT: Byte = $02;
        /// <summary>
        /// The Channel is illegal or not available
        /// </summary>
        const POBDII_CHANNEL_UNAVAILABLE: Byte = $00;
        /// <summary>
        /// The Channel is available
        /// </summary>
        const POBDII_CHANNEL_AVAILABLE: Byte = $01;
        /// <summary>
        /// The Channel is valid, and is being used
        /// </summary>
        const POBDII_CHANNEL_OCCUPIED: Byte = $02;
        /// <summary>
        /// 11 Bit CAN Identifier
        /// </summary>
        const POBDII_CAN_ID_11BIT: Byte = 11;
        /// <summary>
        /// 29 Bit CAN Identifier
        /// </summary>
        const POBDII_CAN_ID_29BIT: Byte = 29;

                     
        ////////////////////////////////////
        /// POBDII API Implementation
        /// Initialization and Configuration
        ///

        /// <summary>
        /// Initializes an OBDII-Client based on a CAN-Channel
        /// </summary>
        /// <remarks>Only one OBDII-Client can be initialized per CAN-Channel</remarks>
        /// <param name="CanChannel">The PCAN-Basic channel to be used as OBDII client</param>
        /// <param name="HwType">NON PLUG&PLAY: The type of hardware and operation mode</param>
        /// <param name="IOPort">NON PLUG&PLAY: The I/O address for the parallel port</param>
        /// <param name="Interupt">NON PLUG&PLAY: Interrupt number of the parallel port</param>
        /// <returns>A TPOBDIIStatus code. POBDII_ERROR_OK is returned on success</returns>
        class function Initialize(
            CanChannel: TPOBDIICANHandle;
            Baudrate: TPOBDIIBaudrateInfo;
            HwType: TPOBDIIHwType;
            IOPort: LongWord;
            Interrupt: Word): TPOBDIIStatus; overload;                                     
        /// <summary>
        /// Initializes an OBDII-Client based on a CAN-Channel
        /// </summary>
        /// <remarks>Only one OBDII-Client can be initialized per CAN-Channel</remarks>
        /// <param name="CanChannel">The PCAN-Basic channel to be used as OBDII client</param>
        /// <returns>A TPOBDIIStatus code. POBDII_ERROR_OK is returned on success</returns>
        class function Initialize(
            CanChannel: TPOBDIICANHandle): TPOBDIIStatus; overload;

        /// <summary>
        /// Uninitializes an OBDII-Client initialized before
        /// </summary>
        /// <param name="CanChannel">An OBDII CAN-Channel representing a OBDII-Client</param>
        /// <returns>A TPOBDIIStatus code. POBDII_ERROR_OK is returned on success</returns>
        class function Uninitialize(
            CanChannel: TPOBDIICANHandle): TPOBDIIStatus;

        /// <summary>
        /// Resets the receive and transmit queues of an OBDII-Client 
        /// </summary>
        /// <param name="CanChannel">An OBDII CAN-Channel representing a OBDII-Client</param>
        /// <returns>A TPOBDIIStatus code. POBDII_ERROR_OK is returned on success</returns>
        class function Reset(
            CanChannel: TPOBDIICANHandle): TPOBDIIStatus;

        /// <summary>
        /// Gets the initialization status of an OBDII CAN-Channel
        /// </summary>
        /// <param name="CanChannel">An OBDII CAN-Channel representing a OBDII-Client</param>
        /// <returns>A TPOBDIIStatus code. POBDII_ERROR_OK is returned on success</returns>
        class function GetStatus(
            CanChannel: TPOBDIICANHandle): TPOBDIIStatus;

        /// <summary>
        /// Retrieves an OBDII-Client value
        /// </summary>
        /// <param name="CanChannel">An OBDII CAN-Channel representing a OBDII-Client</param>
        /// <param name="Parameter">The TPOBDIIParameter parameter to get</param>
        /// <param name="StringBuffer">Buffer for the parameter value</param>
        /// <param name="BufferLength">Size in bytes of the buffer</param>
        /// <returns>A TPOBDIIStatus code. POBDII_ERROR_OK is returned on success</returns>
        class function GetValue(
            CanChannel: TPOBDIICANHandle;
            Parameter: TPOBDIIParameter;
            StringBuffer: PAnsiChar;
            BufferLength: LongWord): TPOBDIIStatus; overload;
        /// <summary>
        /// Retrieves an OBDII-Client value
        /// </summary>
        /// <param name="CanChannel">An OBDII CAN-Channel representing a OBDII-Client</param>
        /// <param name="Parameter">The TPOBDIIParameter parameter to get</param>
        /// <param name="NumericBuffer">Buffer for the parameter value</param>
        /// <param name="BufferLength">Size in bytes of the buffer</param>
        /// <returns>A TPOBDIIStatus code. POBDII_ERROR_OK is returned on success</returns>
        class function GetValue(            
            CanChannel: TPOBDIICANHandle;            
            Parameter: TPOBDIIParameter;
            NumericBuffer: PLongWord;
            BufferLength: LongWord): TPOBDIIStatus; overload;

        /// <summary>
        /// Configures or sets an OBDII-Client value
        /// </summary>
        /// <param name="CanChannel">An OBDII CAN-Channel representing a OBDII-Client</param>
        /// <param name="Parameter">The TPOBDIIParameter parameter to set</param>
        /// <param name="NumericBuffer">Buffer with the value to be set</param>
        /// <param name="BufferLength">Size in bytes of the buffer</param>
        /// <returns>A TPOBDIIStatus code. POBDII_ERROR_OK is returned on success</returns>
        class function SetValue(
            CanChannel: TPOBDIICANHandle;
            Parameter: TPOBDIIParameter;
            NumericBuffer: PLongWord;
            BufferLength: LongWord): TPOBDIIStatus; overload;


        ////////////////////////////////////
        /// Services Implementation
        ///

        /// <summary>
        /// Sends an OBDII Service $01 request into queue and waits for the received response
        /// </summary>
        /// <param name="CanChannel">An OBDII CAN-Channel representing a OBDII-Client</param>
        /// <param name="Pid">A Parameter Identifier</param>
        /// <param name="Data">A TPOBDIIParamData structure array to store the OBDII responses</param>
        /// <param name="DataLen">A length of provided Data array</param>
        /// <returns>A TPOBDIIStatus code. POBDII_ERROR_OK is returned on success</returns>
        class function RequestCurrentData(
            CanChannel: TPOBDIICANHandle;
            Pid: TPOBDIIPid;
            Data: PTPOBDIIParamData;
            DataLen: Byte): TPOBDIIStatus;
                                           
        /// <summary>
        /// Sends an OBDII Service $02 request into queue and waits for the received response
        /// </summary>
        /// <param name="CanChannel">An OBDII CAN-Channel representing a OBDII-Client</param>
        /// <param name="Pid">A Parameter Identifier</param>
        /// <param name="Frame">A Frame number</param>
        /// <param name="Data">A TPOBDIIParamData structure array to store the OBDII responses</param>
        /// <param name="DataLen">A length of provided Data array</param>
        /// <returns>A TPOBDIIStatus code. POBDII_ERROR_OK is returned on success</returns>
        class function RequestFreezeFrameData(
            CanChannel: TPOBDIICANHandle;
            Pid: TPOBDIIPid;
            Frame: Byte;
            Data: PTPOBDIIParamData;
            DataLen: Byte): TPOBDIIStatus;

        /// <summary>
        /// Sends an OBDII Service $03 request into queue and waits for the received response
        /// </summary>
        /// <param name="CanChannel">An OBDII CAN-Channel representing a OBDII-Client</param>
        /// <param name="Data">A TPOBDIIDTCData structure array to store the OBDII responses</param>
        /// <param name="DataLen">A length of provided Data array</param>
        /// <returns>A TPOBDIIStatus code. POBDII_ERROR_OK is returned on success</returns>
        class function RequestStoredTroubleCodes(
            CanChannel: TPOBDIICANHandle;
            Data: PTPOBDIIDTCData;
            DataLen: Byte): TPOBDIIStatus;
                             
        /// <summary>
        /// Sends an OBDII Service $04 request into queue and waits for the received response
        /// </summary>
        /// <param name="CanChannel">An OBDII CAN-Channel representing a OBDII-Client</param>
        /// <param name="Data">A TPOBDIIResponse structure array to store the OBDII responses</param>
        /// <param name="DataLen">A length of provided Data array</param>
        /// <returns>A TPOBDIIStatus code. POBDII_ERROR_OK is returned on success</returns>
        class function ClearTroubleCodes(
            CanChannel: TPOBDIICANHandle;
            Data: PTPOBDIIResponse;
            DataLen: Byte): TPOBDIIStatus;
                            
        /// <summary>
        /// Sends an OBDII Service $06 request into queue and waits for the received response
        /// </summary>
        /// <param name="CanChannel">An OBDII CAN-Channel representing a OBDII-Client</param>
        /// <param name="OBDMid">An On-Board Diagnostic Monitor Identifier</param>
        /// <param name="Data">A TPOBDIIOBDMid structure array to store the OBDII responses</param>
        /// <param name="DataLen">A length of provided Data array</param>
        /// <returns>A TPOBDIIStatus code. POBDII_ERROR_OK is returned on success</returns>
        class function RequestTestResults(
            CanChannel: TPOBDIICANHandle;
            OBDMid: TPOBDIIOBDMid;
            Data: PTPOBDIIMonitorData;
            DataLen: Byte): TPOBDIIStatus;
                                               
        /// <summary>
        /// Sends an OBDII Service $07 request into queue and waits for the received response
        /// </summary>
        /// <param name="CanChannel">An OBDII CAN-Channel representing a OBDII-Client</param>
        /// <param name="Data">A TPOBDIIDTCData structure array to store the OBDII responses</param>
        /// <param name="DataLen">A length of provided Data array</param>
        /// <returns>A TPOBDIIStatus code. POBDII_ERROR_OK is returned on success</returns> 
        class function RequestPendingTroubleCodes(
            CanChannel: TPOBDIICANHandle;
            Data: PTPOBDIIDTCData;
            DataLen: Byte): TPOBDIIStatus;
                   
        /// <summary>
        /// Sends an OBDII Service $08 request into queue and waits for the received response
        /// </summary>
        /// <param name="CanChannel">An OBDII CAN-Channel representing a OBDII-Client</param>
        /// <param name="Tid">A Test Identifier</param>
        /// <param name="Data">A TPOBDIIResponse structure array to store the OBDII responses</param>
        /// <param name="DataLen">A length of provided Data array</param>
        /// <returns>A TPOBDIIStatus code. POBDII_ERROR_OK is returned on success</returns>
        class function RequestControlOperation(
            CanChannel: TPOBDIICANHandle;
            Tid: TPOBDIITid;
            Data: PTPOBDIIResponse;
            DataLen: Byte): TPOBDIIStatus;
                           
        /// <summary>
        /// Sends an OBDII Service $09 request into queue and waits for the received response
        /// </summary>
        /// <param name="CanChannel">An OBDII CAN-Channel representing a OBDII-Client</param>
        /// <param name="InfoType">An InfoType</param>
        /// <param name="Data">A TPOBDIIInfoData structure array to store the OBDII responses</param>
        /// <param name="DataLen">A length of provided Data array</param>
        /// <returns>A TPOBDIIStatus code. POBDII_ERROR_OK is returned on success</returns>
        class function RequestVehicleInformation(
            CanChannel: TPOBDIICANHandle;
            InfoType: TPOBDIIInfoType;
            Data: PTPOBDIIInfoData;
            DataLen: Byte): TPOBDIIStatus;
                     
        /// <summary>
        /// Sends an OBDII Service $0A request into queue and waits for the received response
        /// </summary>
        /// <param name="CanChannel">An OBDII CAN-Channel representing a OBDII-Client</param>
        /// <param name="Data">A TPOBDIIDTCData structure array to store the OBDII responses</param>
        /// <param name="DataLen">A length of provided Data array</param>
        /// <returns>A TPOBDIIStatus code. POBDII_ERROR_OK is returned on success</returns>   
        class function RequestPermanentTroubleCodes(
            CanChannel: TPOBDIICANHandle;
            Data: PTPOBDIIDTCData;
            DataLen: Byte): TPOBDIIStatus;

        /// <summary>
        /// Returns the unit and scaling information for a specified Unit and Scaling ID
        /// </summary>
        /// <param name="id">Unit and Scaling ID</param>
        /// <param name="Data">a TPOBDIIUnitAndScaling structure to store the unit information</param>
        /// <returns>A TPOBDIIStatus code. POBDII_ERROR_OK is returned on success</returns>
        class function GetUnitAndScaling(
            id: Byte;
            unitAndScaling: PTPOBDIIUnitAndScaling): TPOBDIIStatus; stdcall;
    end;

    
implementation
uses SysUtils;
const DLL_Name = 'PCAN-OBDII.DLL';

function OBDII_Initialize(CanChannel: TPOBDIICANHandle; Baudrate: TPOBDIIBaudrateInfo; HwType: TPOBDIIHwType; IOPort: LongWord; Interrupt: Word): TPOBDIIStatus; stdcall;
external DLL_Name;
function OBDII_Uninitialize(CanChannel: TPOBDIICANHandle): TPOBDIIStatus; stdcall;
external DLL_Name;
function OBDII_Reset(CanChannel: TPOBDIICANHandle): TPOBDIIStatus; stdcall;
external DLL_Name;
function OBDII_GetStatus(CanChannel: TPOBDIICANHandle): TPOBDIIStatus; stdcall;
external DLL_Name;
function OBDII_GetValue(CanChannel: TPOBDIICANHandle; Parameter: TPOBDIIParameter; Buffer: Pointer; BufferLength: LongWord): TPOBDIIStatus;  stdcall;
external DLL_Name;
function OBDII_SetValue(CanChannel: TPOBDIICANHandle; Parameter: TPOBDIIParameter; Buffer: Pointer; BufferLength: LongWord): TPOBDIIStatus;  stdcall;
external DLL_Name;
function GetUnitAndScaling(id: Byte): TPOBDIIUnitAndScaling; stdcall;
external DLL_Name;
function OBDII_RequestCurrentData(CanChannel: TPOBDIICANHandle; Pid: TPOBDIIPid; Data: PTPOBDIIParamData; DataLen: Byte): TPOBDIIStatus; stdcall;
external DLL_Name;
function OBDII_RequestFreezeFrameData(CanChannel: TPOBDIICANHandle; Pid: TPOBDIIPid; Frame: Byte; Data: PTPOBDIIParamData; DataLen: Byte): TPOBDIIStatus; stdcall;
external DLL_Name;
function OBDII_RequestStoredTroubleCodes(CanChannel: TPOBDIICANHandle; Data: PTPOBDIIDTCData; DataLen: Byte): TPOBDIIStatus; stdcall;
external DLL_Name;
function OBDII_ClearTroubleCodes(CanChannel: TPOBDIICANHandle; Data: PTPOBDIIResponse; DataLen: Byte): TPOBDIIStatus; stdcall;
external DLL_Name;
function OBDII_RequestTestResults(CanChannel: TPOBDIICANHandle; OBDMid: TPOBDIIOBDMid; Data: PTPOBDIIMonitorData; DataLen: Byte): TPOBDIIStatus; stdcall;
external DLL_Name;
function OBDII_RequestPendingTroubleCodes(CanChannel: TPOBDIICANHandle; Data: PTPOBDIIDTCData; DataLen: Byte): TPOBDIIStatus; stdcall;
external DLL_Name;
function OBDII_RequestControlOperation(CanChannel: TPOBDIICANHandle; Tid: TPOBDIITid; Data: PTPOBDIIResponse; DataLen: Byte): TPOBDIIStatus; stdcall;
external DLL_Name;
function OBDII_RequestVehicleInformation(CanChannel: TPOBDIICANHandle; InfoType: TPOBDIIInfoType; Data: PTPOBDIIInfoData; DataLen: Byte): TPOBDIIStatus; stdcall;
external DLL_Name;
function OBDII_RequestPermanentTroubleCodes(CanChannel: TPOBDIICANHandle; Data: PTPOBDIIDTCData; DataLen: Byte): TPOBDIIStatus; stdcall;
external DLL_Name;   
function OBDII_GetUnitAndScaling(id: Byte; unitAndScaling: PTPOBDIIUnitAndScaling): TPOBDIIStatus; stdcall;
external DLL_Name;


class function TObdiiApi.Initialize(CanChannel: TPOBDIICANHandle; Baudrate: TPOBDIIBaudrateInfo; HwType: TPOBDIIHwType; IOPort: LongWord; Interrupt: Word): TPOBDIIStatus;
begin
    Result:= OBDII_Initialize(CanChannel, Baudrate, HwType, IOPort, Interrupt);
end;

class function TObdiiApi.Initialize(CanChannel: TPOBDIICANHandle): TPOBDIIStatus;
begin
    Result:= OBDII_Initialize(CanChannel, POBDII_BAUDRATE_AUTODETECT, TPOBDIIHwType(0), 0, 0);
end;

class function TObdiiApi.Uninitialize(CanChannel: TPOBDIICANHandle): TPOBDIIStatus;
begin
    Result:= OBDII_Uninitialize(CanChannel);
end;

class function TObdiiApi.Reset(CanChannel: TPOBDIICANHandle): TPOBDIIStatus;
begin
    Result:= OBDII_Reset(CanChannel);
end;

class function TObdiiApi.GetStatus(CanChannel: TPOBDIICANHandle): TPOBDIIStatus;
begin
    Result:= OBDII_GetStatus(CanChannel);
end;

class function TObdiiApi.GetValue(CanChannel: TPOBDIICANHandle; Parameter: TPOBDIIParameter; StringBuffer: PAnsiChar; BufferLength: LongWord): TPOBDIIStatus;
begin
    Result:= OBDII_GetValue(CanChannel, Parameter, StringBuffer, BufferLength);
end;
class function TObdiiApi.GetValue(CanChannel: TPOBDIICANHandle; Parameter: TPOBDIIParameter; NumericBuffer: PLongWord; BufferLength: LongWord): TPOBDIIStatus;
begin
    Result:= OBDII_GetValue(CanChannel, Parameter, NumericBuffer, BufferLength);
end;

class function TObdiiApi.SetValue(CanChannel: TPOBDIICANHandle; Parameter: TPOBDIIParameter; NumericBuffer: PLongWord; BufferLength: LongWord): TPOBDIIStatus;
begin
    Result:= OBDII_SetValue(CanChannel, Parameter, NumericBuffer, BufferLength);
end;



class function TObdiiApi.RequestCurrentData(CanChannel: TPOBDIICANHandle; Pid: TPOBDIIPid; Data: PTPOBDIIParamData; DataLen: Byte): TPOBDIIStatus;
begin
    Result:= OBDII_RequestCurrentData(CanChannel, Pid, Data, DataLen);
end;

class function TObdiiApi.RequestFreezeFrameData(CanChannel: TPOBDIICANHandle; Pid: TPOBDIIPid; Frame: Byte; Data: PTPOBDIIParamData; DataLen: Byte): TPOBDIIStatus;
begin
    Result:= OBDII_RequestFreezeFrameData(CanChannel, Pid, Frame, Data, DataLen);
end;

class function TObdiiApi.RequestStoredTroubleCodes(CanChannel: TPOBDIICANHandle; Data: PTPOBDIIDTCData; DataLen: Byte): TPOBDIIStatus;
begin
    Result:= OBDII_RequestStoredTroubleCodes(CanChannel, Data, DataLen);
end;

class function TObdiiApi.ClearTroubleCodes(CanChannel: TPOBDIICANHandle; Data: PTPOBDIIResponse; DataLen: Byte): TPOBDIIStatus;
begin
    Result:= OBDII_ClearTroubleCodes(CanChannel, Data, DataLen);
end;

class function TObdiiApi.RequestTestResults(CanChannel: TPOBDIICANHandle; OBDMid: TPOBDIIOBDMid; Data: PTPOBDIIMonitorData; DataLen: Byte): TPOBDIIStatus;
begin
    Result:= OBDII_RequestTestResults(CanChannel, OBDMid, Data, DataLen);
end;

class function TObdiiApi.RequestPendingTroubleCodes(CanChannel: TPOBDIICANHandle; Data: PTPOBDIIDTCData; DataLen: Byte): TPOBDIIStatus;
begin
    Result:= OBDII_RequestPendingTroubleCodes(CanChannel, Data, DataLen);
end;

class function TObdiiApi.RequestControlOperation(CanChannel: TPOBDIICANHandle; Tid: TPOBDIITid; Data: PTPOBDIIResponse; DataLen: Byte): TPOBDIIStatus;
begin
    Result:= OBDII_RequestControlOperation(CanChannel, Tid, Data, DataLen);
end;

class function TObdiiApi.RequestVehicleInformation(CanChannel: TPOBDIICANHandle; InfoType: TPOBDIIInfoType; Data: PTPOBDIIInfoData; DataLen: Byte): TPOBDIIStatus;
begin
    Result:= OBDII_RequestVehicleInformation(CanChannel, InfoType, Data, DataLen);
end;

class function TObdiiApi.RequestPermanentTroubleCodes(CanChannel: TPOBDIICANHandle; Data: PTPOBDIIDTCData; DataLen: Byte): TPOBDIIStatus;
begin
    Result:= OBDII_RequestPermanentTroubleCodes(CanChannel, Data, DataLen);
end;

class function TObdiiApi.GetUnitAndScaling(id: Byte; unitAndScaling: PTPOBDIIUnitAndScaling): TPOBDIIStatus;
begin
    Result:= OBDII_GetUnitAndScaling(id, unitAndScaling);
end;

end.

