﻿using System;
using System.Text;

using Peak.Can.IsoTp;
using Peak.Can.OBDonUDS;

using obd_DID_t = System.UInt16;  // See PCAN-OBDonUDS.cs

// Note : this example uses unsafe mode, it means that 
// "allow unsafe mode" must be checked in the Visual Studio project,
// and that 
// "#define UNSAFE" must be uncommented at the beginning of the file "PCAN-ISO-TP_2016.cs", or that UNSAFE must be defined at the project level.

namespace FirstSteps
{
    class Program
    {
        /// <summary>Entry point of the program</summary>
        static int Main(string[] args)
        {
            const int BUFFER_SIZE = 256;
            const int MAX_NB_RESPONSES = 16;
            int result = -1;

            // Get API version
            StringBuilder buffer = new StringBuilder(BUFFER_SIZE);
            obd_status status = OBDonUDSApi.GetValue(cantp_handle.PCANTP_HANDLE_NONEBUS, obd_parameter.POBDONUDS_PARAMETER_API_VERSION, buffer, BUFFER_SIZE);
            Console.WriteLine("Get API version ({0}): {1}", STATUS_OK_KO(status), buffer);

            // Initialize
            cantp_handle channel = cantp_handle.PCANTP_HANDLE_USBBUS1; // TODO: modify the value according to your available PCAN devices.
            Console.WriteLine("Channel : 0x{0:x}, Channel status : 0x{1:x}", channel, OBDonUDSApi.GetStatus(channel));
            Console.WriteLine("Connecting with automatic detection of baudrate...");
            status = OBDonUDSApi.Initialize(channel, (cantp_baudrate)obd_baudrate.OBD_BAUDRATE_AUTODETECT);
            Console.WriteLine("Initialize ({0})", STATUS_OK_KO(status));

            if (OBDonUDSApi.StatusIsOk(status))
            {
                result = 1;
                obd_baudrate baudrate = (obd_baudrate)0;
                status = OBDonUDSApi.GetValue(channel, obd_parameter.POBDONUDS_PARAMETER_BAUDRATE, out baudrate, sizeof(obd_baudrate));
                Console.WriteLine("-> Baudrate ({0}): {1}", STATUS_OK_KO(status), (baudrate == obd_baudrate.OBD_BAUDRATE_500K) ? "500 kbit/s" : (baudrate == obd_baudrate.OBD_BAUDRATE_250K) ? "250kbit/s" : "unknown");

                obd_msgprotocol canIdLen = (obd_msgprotocol)0;
                status = OBDonUDSApi.GetValue(channel, obd_parameter.POBDONUDS_PARAMETER_CAN_ID_LENGTH, out canIdLen, sizeof(obd_msgprotocol));
                Console.WriteLine("-> Can Id length ({0}): {1}", STATUS_OK_KO(status), (canIdLen == obd_msgprotocol.OBD_MSGPROTOCOL_11BIT) ? "11" : (canIdLen == obd_msgprotocol.OBD_MSGPROTOCOL_29BIT) ? "29" : "unknown");

                byte[] numberOfECU = new byte[1];
                status = OBDonUDSApi.GetValue(channel, obd_parameter.POBDONUDS_PARAMETER_AVAILABLE_ECUS, numberOfECU, sizeof(byte));
                Console.WriteLine("-> Number of OBDonUDS ECU detected ({0}): {1}", STATUS_OK_KO(status), numberOfECU[0]);

                // Request vehicle identification number (VIN) using functional addressing scheme
                obd_DID_t[] VINitid = new obd_DID_t[1] { 0xF802 };
                obd_msg msg_request = new obd_msg();
                status = OBDonUDSApi.RequestVehicleInformation(channel, canIdLen == obd_msgprotocol.OBD_MSGPROTOCOL_11BIT ? OBDonUDSApi.OBD_NAI_REQUEST_FUNCTIONAL_11B : OBDonUDSApi.OBD_NAI_REQUEST_FUNCTIONAL_29B,
                    out msg_request, VINitid, 1);
                Console.WriteLine("Request VIN ({0})", STATUS_OK_KO(status));
                if (OBDonUDSApi.StatusIsOk(status))
                {
                    result = 2;
                    // Wait for responses
                    obd_msg[] msg_responses = new obd_msg[MAX_NB_RESPONSES];
                    UInt32 nb_responses;
                    obd_msg msg_request_confirmation = new obd_msg();
                    status = OBDonUDSApi.WaitForServiceFunctional(channel, ref msg_request, MAX_NB_RESPONSES, false, msg_responses, out nb_responses, out msg_request_confirmation);
                    Console.WriteLine("Wait for service functional ({0})", STATUS_OK_KO(status));
                    if (OBDonUDSApi.StatusIsOk(status))
                    {
                        result = 3;
                        Console.WriteLine("->  Received {0} responses", nb_responses);
                        // Parse responses
                        for (UInt32 ir = 0; ir < nb_responses; ++ir)
                        {
                            obd_request_vehicle_information_response parsed_response = new obd_request_vehicle_information_response();
                            status = OBDonUDSApi.ParseResponse_RequestVehicleInformation(ref msg_responses[ir], out parsed_response);
                            Console.WriteLine("Parse response #{0} ({1}):", ir + 1, STATUS_OK_KO(status));
                            if (OBDonUDSApi.StatusIsOk(status))
                            {
                                if (parsed_response.nrc != 0)
                                {
                                    Console.WriteLine("-> Negative response code: 0x{0:X}", parsed_response.nrc);
                                }
                                else
                                {
                                    Console.Write("-> Vehicle Identification Number (Safe mode): ");
                                    print_data_safe(ref parsed_response);
                                    Console.WriteLine("");
                                    Console.Write("-> Vehicle Identification Number (Unsafe mode): ");
                                    print_data_unsafe(ref parsed_response);
                                    Console.WriteLine("");
                                    result = 0;
                                }
                            }
                            else
                            {
                                Console.WriteLine("-> failed status: 0x{0:X}", status);
                            }
                            // Free message
                            OBDonUDSApi.ParsedResponseFree(ref parsed_response);
                        }
                    }
                    else
                    {
                        Console.WriteLine("-> failed status: 0x{0:X}", status);
                    }

                    // Free messages
                    OBDonUDSApi.MsgFree(ref msg_request_confirmation);
                    for (UInt32 ir = 0; ir < nb_responses; ++ir)
                        OBDonUDSApi.MsgFree(ref msg_responses[ir]);
                }
                else
                {
                    Console.WriteLine("-> failed status: 0x{0:X}",status);
                }
                // Free messages
                OBDonUDSApi.MsgFree(ref msg_request);

                status = OBDonUDSApi.Uninitialize(channel);
                Console.WriteLine("Uninitialize ({0})", STATUS_OK_KO(status));
            }
            else
            {
                Console.WriteLine("-> failed status: 0x{0:X}", status);
            }
            return result;
        }

        static String OK_KO(bool test)
        {
            if (test)
                return "OK";
            return "KO";
        }

        static String STATUS_OK_KO(obd_status test)
        {
            return OK_KO(OBDonUDSApi.StatusIsOk(test));
        }

        static void print_data_safe(ref obd_request_vehicle_information_response parsed_response)
        {
            // Safe
            if (parsed_response.nb_elements != 0)
            {
                byte[] vals = new byte[parsed_response.nb_elements];
                if (OBDonUDSApi.GetData(ref parsed_response, vals, (Int32)parsed_response.nb_elements))
                    for (UInt32 i = 0; i < parsed_response.nb_elements; ++i)
                        Console.Write("{0}", (char)vals[i]);
            }
        }
        
        static unsafe void print_data_unsafe(ref obd_request_vehicle_information_response parsed_response)
        {
            // Unsafe
            if (parsed_response.nb_elements != 0)
            {
                byte[] vals = new byte[parsed_response.nb_elements];
                if (OBDonUDSApi.GetData_unsafe(ref parsed_response, vals, (Int32)parsed_response.nb_elements))
                    for (UInt32 i = 0; i < parsed_response.nb_elements; ++i)
                        Console.Write("{0}", (char)vals[i]);
            }
        }
    }
}
