﻿using System;
using System.Text;

using Peak.Can.IsoTp;
using Peak.Can.OBDonUDS;

using obd_DTC_t = System.UInt32;  // See PCAN-OBDonUDS.cs

// Note : this example uses unsafe mode, it means that 
// "allow unsafe mode" must be checked in the Visual Studio project,
// and that 
// "#define UNSAFE" must be uncommented at the beginning of the file "PCAN-ISO-TP_2016.cs", or that UNSAFE must be defined at the project level.

namespace ClearDTC
{
    class Program
    {
        /// <summary>Entry point of the program</summary>
        static int Main(string[] args)
        {
			const int BUFFER_SIZE = 256;
			const int MAX_NB_RESPONSES = 16;
			int result = -1;
            // Get API version
            StringBuilder buffer = new StringBuilder(BUFFER_SIZE);
            obd_status status = OBDonUDSApi.GetValue(cantp_handle.PCANTP_HANDLE_NONEBUS, obd_parameter.POBDONUDS_PARAMETER_API_VERSION, buffer, BUFFER_SIZE);
            Console.WriteLine("Get API version ({0}): {1}", STATUS_OK_KO(status), buffer);

            // Initialize
            cantp_handle channel = cantp_handle.PCANTP_HANDLE_USBBUS1; // TODO: modify the value according to your available PCAN devices.
            Console.WriteLine("Channel : 0x{0:x}, Channel status : 0x{1:x}", channel, OBDonUDSApi.GetStatus(channel));
            Console.WriteLine("Connecting with automatic detection of baudrate...");
            status = OBDonUDSApi.Initialize(channel, (cantp_baudrate)obd_baudrate.OBD_BAUDRATE_AUTODETECT);
            Console.WriteLine("Initialize ({0})", STATUS_OK_KO(status));

            if (OBDonUDSApi.StatusIsOk(status))
            {
                result = 1;
                obd_baudrate baudrate = (obd_baudrate)0;
                status = OBDonUDSApi.GetValue(channel, obd_parameter.POBDONUDS_PARAMETER_BAUDRATE, out baudrate, sizeof(obd_baudrate));
                Console.WriteLine("-> Baudrate ({0}): {1}", STATUS_OK_KO(status), (baudrate == obd_baudrate.OBD_BAUDRATE_500K) ? "500 kbit/s" : (baudrate == obd_baudrate.OBD_BAUDRATE_250K) ? "250kbit/s" : "unknown");

                obd_msgprotocol canIdLen = (obd_msgprotocol)0;
                status = OBDonUDSApi.GetValue(channel, obd_parameter.POBDONUDS_PARAMETER_CAN_ID_LENGTH, out canIdLen, sizeof(obd_msgprotocol));
                Console.WriteLine("-> Can Id length ({0}): {1}", STATUS_OK_KO(status), (canIdLen == obd_msgprotocol.OBD_MSGPROTOCOL_11BIT) ? "11" : (canIdLen == obd_msgprotocol.OBD_MSGPROTOCOL_29BIT) ? "29" : "unknown");

                byte[] numberOfECU = new byte[1];
                status = OBDonUDSApi.GetValue(channel, obd_parameter.POBDONUDS_PARAMETER_AVAILABLE_ECUS, numberOfECU, sizeof(byte));
                Console.WriteLine("-> Number of OBDonUDS ECU detected ({0}): {1}", STATUS_OK_KO(status), numberOfECU[0]);

				// Request to clear trouble codes using functional addressing scheme
				obd_DTC_t emissionSystemGroup = OBDonUDSApi.OBDONUDS_DTC_EMISSION_SYSTEM_GROUP;
				obd_msg msg_request = new obd_msg();
				status = OBDonUDSApi.ClearTroubleCodes(channel, canIdLen == obd_msgprotocol.OBD_MSGPROTOCOL_11BIT ? OBDonUDSApi.OBD_NAI_REQUEST_FUNCTIONAL_11B : OBDonUDSApi.OBD_NAI_REQUEST_FUNCTIONAL_29B,
					out msg_request, emissionSystemGroup);
				Console.WriteLine("Request Clear Emission-System Trouble Codes ({0})", STATUS_OK_KO(status));

				if (OBDonUDSApi.StatusIsOk(status))
				{
					result = 2;
					// Wait for responses
					obd_msg[] msg_responses = new obd_msg[MAX_NB_RESPONSES];
					UInt32 nb_responses;
					obd_msg msg_request_confirmation = new obd_msg();
					status = OBDonUDSApi.WaitForServiceFunctional(channel, ref msg_request, MAX_NB_RESPONSES, false, msg_responses, out nb_responses, out msg_request_confirmation);
					Console.WriteLine("Wait for service functional ({0}):", STATUS_OK_KO(status));
					if (OBDonUDSApi.StatusIsOk(status))
					{
						result = 3;
						Console.WriteLine("-> Received {0} responses", nb_responses);
						for (UInt32 i = 0; i < nb_responses; ++i)
						{
							Console.Write("Response number {0} from ECU {1} : service 0x{2:X} (safe) service 0x{3:X} (unsafe) ", i + 1, 
								msg_responses[i].nai.source_addr, 
								OBDonUDSApi.GetByte(msg_responses[i].links.service_id),
								OBDonUDSApi.GetByte_unsafe(msg_responses[i].links.service_id));					
							if (msg_responses[i].links.nrc != IntPtr.Zero)
							{
								Console.WriteLine("NRC 0x{0:X} (safe)", OBDonUDSApi.GetByte(msg_responses[i].links.nrc));
								Console.WriteLine("NRC 0x{0:X} (unsafe)", OBDonUDSApi.GetByte_unsafe(msg_responses[i].links.nrc));
							}
							else
							{
								Console.WriteLine("OK");
								result = 0;
							}
						}
					}
					else
					{
						Console.WriteLine("-> failed status: 0x{0:X}", status);
					}
					// Free messages
					OBDonUDSApi.MsgFree(ref msg_request_confirmation);
					for (UInt32 i = 0; i < nb_responses; ++i)
						OBDonUDSApi.MsgFree(ref msg_responses[i]);
				}
				else
				{
					Console.WriteLine("-> failed status: 0x{0:X}", status);
				}

				// Free messages
				OBDonUDSApi.MsgFree(ref msg_request);
			}
			else
            {
                Console.WriteLine("-> failed status: 0x{0:X}", status);
            }
			status = OBDonUDSApi.Uninitialize(channel);
			Console.WriteLine("Uninitialize ({0})", STATUS_OK_KO(status));
			return result;
        }

        static String OK_KO(bool test)
        {
            if (test)
                return "OK";
            return "KO";
        }

        static String STATUS_OK_KO(obd_status test)
        {
            return OK_KO(OBDonUDSApi.StatusIsOk(test));
        }
	}
}
