﻿using System;
using System.Text;

using Peak.Can.IsoTp;
using Peak.Can.Uds;
using Peak.Can.OBDonUDS;

using obd_DID_t = System.UInt16; // See PCAN-OBDonUDS.cs

// Note : this example uses unsafe mode, it means that 
// "allow unsafe mode" must be checked in the Visual Studio project,
// and that 
// "#define UNSAFE" must be uncommented at the beginning of the file "PCAN-ISO-TP_2016.cs", or that UNSAFE must be defined at the project level.

namespace CurrentData
{
	class Program
	{
		/// <summary>Entry point of the program</summary>
		static int Main(string[] args)
		{
			const int BUFFER_SIZE = 256;
			const int DID_SUPPORTED_SIZE = 256;
			const int DID_VALUE_SIZE = 5;
			int result = -1;

			// Get API version
			StringBuilder buffer = new StringBuilder(BUFFER_SIZE);
			obd_status status = OBDonUDSApi.GetValue(cantp_handle.PCANTP_HANDLE_NONEBUS, obd_parameter.POBDONUDS_PARAMETER_API_VERSION, buffer, BUFFER_SIZE);
			Console.WriteLine("Get API version ({0}): {1}", STATUS_OK_KO(status), buffer);

			// Configure logs to get general debug information
			byte[] debugValue = new byte[1] { OBDonUDSApi.POBDONUDS_DEBUG_LVL_DEBUG };
			status = OBDonUDSApi.SetValue(cantp_handle.PCANTP_HANDLE_NONEBUS, obd_parameter.POBDONUDS_PARAMETER_DEBUG, debugValue, 1);
			Console.WriteLine("Configure general logs ({0})", STATUS_OK_KO(status));

			// Initialize
			cantp_handle channel = cantp_handle.PCANTP_HANDLE_USBBUS1; // TODO: modify the value according to your available PCAN devices.
			Console.WriteLine("Channel : 0x{0:x}, Channel status : 0x{1:x}", channel, OBDonUDSApi.GetStatus(channel));
			Console.WriteLine("Connecting with automatic detection of baudrate...");
			status = OBDonUDSApi.Initialize(channel, (cantp_baudrate)obd_baudrate.OBD_BAUDRATE_AUTODETECT);
			Console.WriteLine("Initialize ({0})", STATUS_OK_KO(status));

			if (OBDonUDSApi.StatusIsOk(status))
			{
				result = 1;
				obd_baudrate baudrate = (obd_baudrate)0;
				status = OBDonUDSApi.GetValue(channel, obd_parameter.POBDONUDS_PARAMETER_BAUDRATE, out baudrate, sizeof(obd_baudrate));
				Console.WriteLine("-> Baudrate ({0}): {1}", STATUS_OK_KO(status), (baudrate == obd_baudrate.OBD_BAUDRATE_500K) ? "500 kbit/s" : (baudrate == obd_baudrate.OBD_BAUDRATE_250K) ? "250kbit/s" : "unknown");

				obd_msgprotocol canIdLen = (obd_msgprotocol)0;
				status = OBDonUDSApi.GetValue(channel, obd_parameter.POBDONUDS_PARAMETER_CAN_ID_LENGTH, out canIdLen, sizeof(obd_msgprotocol));
				Console.WriteLine("-> Can Id length ({0}): {1}", STATUS_OK_KO(status), (canIdLen == obd_msgprotocol.OBD_MSGPROTOCOL_11BIT) ? "11" : (canIdLen == obd_msgprotocol.OBD_MSGPROTOCOL_29BIT) ? "29" : "unknown");

				// Print supported DIDs F4XX for service $22 - Request current Data
				byte[] supportedDIDs_0xF4XX = new byte[DID_SUPPORTED_SIZE];
				status = OBDonUDSApi.GetValue(channel, obd_parameter.POBDONUDS_PARAMETER_SUPPORTMASK_DIDS_F4XX, supportedDIDs_0xF4XX, DID_SUPPORTED_SIZE);
				Console.WriteLine("Get supported DIDs 0xF4XX ({0}):", STATUS_OK_KO(status));
				if (OBDonUDSApi.StatusIsOk(status))
				{
					result = 2;
					for (int did = 0; did < DID_SUPPORTED_SIZE; ++did)
					{
						for (byte j = 0; j < 8; ++j)
						{
							if (((supportedDIDs_0xF4XX[did] >> j) & 0x01) != 0)
							{
								Console.WriteLine("-> DID 0x{0:X} is supported by ECU #{1}", 0xF400 + (did + 1), j + 1);
							}
						}
					}
				}
				else
				{
					Console.WriteLine("-> failed status: 0x{0:X}", status);
				}

				// Request current data values of 5 DID using physical address scheme (point to point)
				obd_DID_t[] DIDs = new obd_DID_t[DID_VALUE_SIZE] { 0xF415, 0xF401, 0xF405, 0xF403, 0xF40C }; // TODO: change the DIDs according to those supported
				obd_ecu myECU = obd_ecu.POBD_ECU_1; // TODO: change the ECU if wanted
				obd_netaddrinfo nai = new obd_netaddrinfo();
				nai.protocol = canIdLen;
				nai.target_type = obd_addressing.OBD_ADDRESSING_PHYSICAL;
				nai.source_addr = (UInt16)uds_address.PUDS_ADDRESS_ISO_15765_4_ADDR_TEST_EQUIPMENT;
				nai.target_addr = (UInt16)myECU;
				obd_msg msg_request = new obd_msg();
				status = OBDonUDSApi.RequestCurrentData(channel, nai, out msg_request, DIDs, DID_VALUE_SIZE);
				Console.WriteLine("Request Current Powertrain diagnostic data ({0})", STATUS_OK_KO(status));

				if (OBDonUDSApi.StatusIsOk(status))
				{
					result = 3;
					// Wait for responses
					obd_msg msg_response = new obd_msg();
					obd_msg msg_request_confirmation = new obd_msg();
					status = OBDonUDSApi.WaitForService(channel, ref msg_request, out msg_response, out msg_request_confirmation);
					Console.WriteLine("Wait for service physical ({0}):", STATUS_OK_KO(status));
					if (OBDonUDSApi.StatusIsOk(status))
					{
						result = 4;
						// Print raw (unparsed) response
						Console.WriteLine("-> Raw response : ");
						if (OBDonUDSApi.GetByte(msg_response.links.network_result) == (byte)cantp_netstatus.PCANTP_NETSTATUS_OK)
						{
							Console.WriteLine("Safe");
							print_cantp_data(ref msg_response.msg.msg, true);
							Console.WriteLine("Unsafe");
							print_cantp_data(ref msg_response.msg.msg, false);
						}
						else
						{
							Console.WriteLine("N/A, an error occured, check content of the internal UDS/ISOTP message...");
						}

						// Parse response
						obd_request_current_data_response parsed_response = new obd_request_current_data_response();
						status = OBDonUDSApi.ParseResponse_RequestCurrentData(ref msg_response, out parsed_response);
						Console.WriteLine("Parse response ({0}):", STATUS_OK_KO(status));
						if (OBDonUDSApi.StatusIsOk(status))
						{
							result = 0;
							// Print parsed response
							if (parsed_response.nrc != 0)
								Console.WriteLine("-> Negative response code 0x{0:X}", parsed_response.nrc);
							else
							{
								Console.WriteLine("Safe");
								print_data_safe(ref parsed_response);
								Console.WriteLine("Unsafe");
								print_data_unsafe(ref parsed_response);
							}
						}
						else
						{
							Console.WriteLine("-> failed status: 0x{0:X}", status);
						}
						// Free message
						OBDonUDSApi.ParsedResponseFree(ref parsed_response);
					}
					else
					{
						Console.WriteLine("-> failed status: 0x{0:X}", status);
					}
					// Free messages
					OBDonUDSApi.MsgFree(ref msg_request_confirmation);
					OBDonUDSApi.MsgFree(ref msg_response);
				}
				else
				{
					Console.WriteLine("-> failed status: 0x{0:X}", status);
				}
				// Free messages
				OBDonUDSApi.MsgFree(ref msg_request);
			}

			else
			{
				Console.WriteLine("-> failed status: 0x{0:X}", status);
			}
			status = OBDonUDSApi.Uninitialize(channel);
			Console.WriteLine("Uninitialize ({0})", STATUS_OK_KO(status));
			return result;
		}

		static String OK_KO(bool test)
		{
			if (test)
				return "OK";
			return "KO";
		}

		static String STATUS_OK_KO(obd_status test)
		{
			return OK_KO(OBDonUDSApi.StatusIsOk(test));
		}

		/// <summary>Helper: print PCAN_ISOTP message data</summary>
		/// <param name="msg">message to print</param>
		/// <param name="unsafeMode">safe or unsafe mode</param>
		unsafe static void print_cantp_data(ref cantp_msg msg, bool unsafeMode)
		{
			if (msg.Msgdata != IntPtr.Zero)
			{
				uint len = msg.Msgdata_any_Copy.length;
				if (len != 0)
				{
					byte[] vals = new byte[len];
					if (unsafeMode ?
						CanTpApi.getData_unsafe_2016(ref msg, 0, vals, (int)len) :
						CanTpApi.getData_2016(ref msg, 0, vals, (int)len))
					{
						Console.Write("\t[");
						for (uint i = 0; i < len; i++)
						{
							Console.Write(" {0:x2}", vals[i]);
						}
						Console.WriteLine("]");
					}
				}
			}
		}
		

		static void print_data_safe(ref obd_request_current_data_response parsed_response)
		{
			// Safe
			if (parsed_response.nb_elements != 0)
			{
				obd_did_object[] vals = new obd_did_object[parsed_response.nb_elements];
				if(OBDonUDSApi.GetData(ref parsed_response, vals, (Int32)parsed_response.nb_elements))
					for (UInt32 i = 0; i < parsed_response.nb_elements; ++i)
					{
						Console.Write("-> Element #{0} : ", i + 1);
						print_did_safe(ref vals[i]);
					}
			}
		}
		
		static void print_did_safe(ref obd_did_object didObject)
		{
			Console.Write("DID 0x{0:X}, {1} bytes: [", didObject.data_identifier, didObject.size);
			// Safe
			if (didObject.size != 0)
			{
				byte[] vals = new byte[didObject.size];
				if(OBDonUDSApi.GetData(ref didObject, vals, (Int32)didObject.size))
					for (UInt32 i = 0; i < didObject.size; ++i)
						Console.Write("{0:x2} ", vals[i]);
			}
			Console.WriteLine("]");
		}
		
		static unsafe void print_data_unsafe(ref obd_request_current_data_response parsed_response)
		{
			// Unsafe
			if (parsed_response.nb_elements != 0)
			{
				obd_did_object[] vals = new obd_did_object[parsed_response.nb_elements];
				if (OBDonUDSApi.GetData_unsafe(ref parsed_response, vals, (Int32)parsed_response.nb_elements))
				{
					for (UInt32 i = 0; i < parsed_response.nb_elements; ++i)
					{
						Console.Write("-> Element #{0} : ", i + 1);
						print_did_unsafe(ref vals[i]);
					}
				}
			}
		}
		
		static unsafe void print_did_unsafe(ref obd_did_object didObject)
		{
			Console.Write("DID 0x{0:X}, {1} bytes: [", didObject.data_identifier, didObject.size);
			// Unsafe
			if (didObject.size != 0)
			{
				byte[] vals = new byte[didObject.size];
				if (OBDonUDSApi.GetData_unsafe(ref didObject, vals, (Int32)didObject.size))
				{
					for (UInt32 i = 0; i < didObject.size; ++i)
						Console.Write("{0:x2} ", vals[i]);
				}
			}
			Console.WriteLine("]");
		}
	}
}
