﻿using System;
using System.Text;

using Peak.Can.IsoTp;
using Peak.Can.Uds;
using Peak.Can.OBDonUDS;

using obd_DID_t = System.UInt16;  // See PCAN-OBDonUDS.cs
using obd_DTC_t = System.UInt32;  // See PCAN-OBDonUDS.cs

// Note : this example uses unsafe mode, it means that 
// "allow unsafe mode" must be checked in the Visual Studio project,
// and that 
// "#define UNSAFE" must be uncommented at the beginning of the file "PCAN-ISO-TP_2016.cs", or that UNSAFE must be defined at the project level.


namespace Viewer
{
    class Program
    {
        const int DID_SUPPORTED_SIZE = 256;
        const int BUFFER_SIZE = 256;
        const int MAX_NB_RESPONSES = 16;
        const byte MAX_NB_ECU = 8;
        const int MAX_DID_PER_PHYSICAL_REQUEST = 6;
        const int MAX_NB_EXTENDED_DTC_RECORD = 16;

        /// <summary>Entry point of the program</summary>
        static int Main(string[] args)
        {
            int nbErr = 0;

            // Get API version
            StringBuilder buffer = new StringBuilder(BUFFER_SIZE);
            obd_status status = OBDonUDSApi.GetValue(cantp_handle.PCANTP_HANDLE_NONEBUS, obd_parameter.POBDONUDS_PARAMETER_API_VERSION, buffer, BUFFER_SIZE);
            Console.WriteLine("Get API version ({0}): {1}", STATUS_OK_KO(status), buffer);
#if (DODEBUG)
            // Configure logs to get general debug information
            byte[] debugValue = new byte[1] { OBDonUDSApi.POBDONUDS_DEBUG_LVL_DEBUG };
            status = OBDonUDSApi.SetValue(cantp_handle.PCANTP_HANDLE_NONEBUS, obd_parameter.POBDONUDS_PARAMETER_DEBUG, debugValue, 1);
            Console.WriteLine("Configure general logs ({0})", STATUS_OK_KO(status));
#endif
            // Initialize
            cantp_handle channel = cantp_handle.PCANTP_HANDLE_USBBUS1; // TODO: modify the value according to your available PCAN devices.
            Console.WriteLine("Channel : 0x{0:x}, Channel status : 0x{1:x}", channel, OBDonUDSApi.GetStatus(channel));
            Console.WriteLine("Connecting with automatic detection of baudrate...");
            status = OBDonUDSApi.Initialize(channel, (cantp_baudrate)obd_baudrate.OBD_BAUDRATE_AUTODETECT);
            Console.WriteLine("Initialize ({0})", STATUS_OK_KO(status));

            if (OBDonUDSApi.StatusIsOk(status))
            {
                obd_baudrate baudrate = (obd_baudrate)0;
                status = OBDonUDSApi.GetValue(channel, obd_parameter.POBDONUDS_PARAMETER_BAUDRATE, out baudrate, sizeof(obd_baudrate));
                Console.WriteLine("-> Baudrate ({0}): {1}", STATUS_OK_KO(status), (baudrate == obd_baudrate.OBD_BAUDRATE_500K) ? "500 kbit/s" : (baudrate == obd_baudrate.OBD_BAUDRATE_250K) ? "250kbit/s" : "unknown");

                obd_msgprotocol canIdLen = (obd_msgprotocol)0;
                status = OBDonUDSApi.GetValue(channel, obd_parameter.POBDONUDS_PARAMETER_CAN_ID_LENGTH, out canIdLen, sizeof(obd_msgprotocol));
                Console.WriteLine("-> Can Id length ({0}): {1}", STATUS_OK_KO(status), (canIdLen == obd_msgprotocol.OBD_MSGPROTOCOL_11BIT) ? "11" : (canIdLen == obd_msgprotocol.OBD_MSGPROTOCOL_29BIT) ? "29" : "unknown");

#if (DODEBUG)
		// Configure logs to get channel debug information
            byte[] debugValue = new byte[1] { OBDonUDSApi.POBDONUDS_DEBUG_LVL_DEBUG };
            status = OBDonUDSApi.SetValue(channel, obd_parameter.POBDONUDS_PARAMETER_DEBUG, debugValue, 1);
            Console.WriteLine("Configure channel logs ({0})", STATUS_OK_KO(status));
#endif

                // Request and print Vehicle Identification Number
                nbErr += GetVehicleIdentificationNumber(channel, canIdLen);

                // Show supported IDS
                nbErr += GetSupportedMask(channel);

                // make requests to every supported services

                nbErr += displayRequestCurrentPowertrainDiagnosticData_22_DID(channel, canIdLen);

                // array of confirmed DTC, allocated and filled by the request "emission-related DTC with confirmed status", to be used later by the request "powertrain snapshot data"
                obd_DTC_t[][] ConfirmedDTCs = new obd_DTC_t[MAX_NB_ECU][];

                nbErr += displayRequestEmissionRelatedDiagnosticTroubleCodesWithConfirmedStatus_19_42(channel, canIdLen, ConfirmedDTCs);

                // array of pending DTC, allocated and filled by the request "emission-related DTC with pending status", to be used later by the request "powertrain snapshot data"
                obd_DTC_t[][] PendingDTCs = new obd_DTC_t[MAX_NB_ECU][];

                nbErr += displayRequestEmissionRelatedDiagnosticTroubleCodesWithPendingStatus_19_42(channel, canIdLen, PendingDTCs);

                nbErr += displayRequestPowertrainFreezeFrameData_19_04(channel, canIdLen, ConfirmedDTCs, PendingDTCs);

                Console.Write("\n\nDo you wish to Clear/Reset Emission-Related Diagnostic Information of group of ALL Trouble Codes ? (y/n) ");
                char c = Console.ReadKey().KeyChar;
                if (c == 'y' || c == 'Y')
                    nbErr += displayClearResetEmissionRelatedDiagnosticInformation_14(channel, canIdLen, OBDonUDSApi.OBDONUDS_DTC_ALL_GROUPS);

                Console.Write("\n\nDo you wish to Clear/Reset Emission-Related Diagnostic Information of group of EMISSION-SYSTEM Trouble Codes ? (y/n) ");
                c = Console.ReadKey().KeyChar;
                if (c == 'y' || c == 'Y')
                    nbErr += displayClearResetEmissionRelatedDiagnosticInformation_14(channel, canIdLen, OBDonUDSApi.OBDONUDS_DTC_EMISSION_SYSTEM_GROUP);

                nbErr += displayRequestOnBoardMonitoringTestResultsForSpecificMonitoredSystems_22_MID(channel, canIdLen);

                nbErr += displayRequestControlOfOnBoardSystemTestOrComponent_31_RID(channel, canIdLen);

                nbErr += displayRequestVehicleInformation_22_ITID(channel, canIdLen);

                nbErr += displayRequestEmissionRelatedDiagnosticTroubleCodesWithPermanentStatus_19_55(channel, canIdLen);

                // array of extended DTC, allocated and filled by the request "supported DTC extended record", to be used later by the request "DTC extended data record"
                obd_DTC_t[,][] extendedDTCs = new obd_DTC_t[MAX_NB_ECU, MAX_NB_EXTENDED_DTC_RECORD][];

                nbErr += displayRequestSupportedDTCExtendedRecord_19_1A(channel, canIdLen, extendedDTCs);

                nbErr += displayRequestDTCExtendedDataRecord_19_06(channel, canIdLen, extendedDTCs);

                nbErr += displayRequestDTCsForAReadinessGroup_19_56(channel, canIdLen);
            }
            else
            {
                Console.Write("Initialization Error (0x{0:X}): ", status); printStatusDetails(status);
                nbErr++;
            }

            status = OBDonUDSApi.Uninitialize(channel);

            if (nbErr > 0)
            {
                Console.WriteLine("\n\nERROR : a total of {0} errors occured.\n", nbErr);
            }
            else
            {
                Console.WriteLine("\n\nALL Transmissions succeeded !\n");
            }

            Console.WriteLine("\nPress <Enter> to quit...");
            Console.ReadKey();

            return nbErr;
        }

        static String OK_KO(bool test)
        {
            if (test)
                return "OK";
            return "KO";
        }

        static String STATUS_OK_KO(obd_status test)
        {
            return OK_KO(OBDonUDSApi.StatusIsOk(test));
        }
        static void print_data_safe(ref obd_request_vehicle_information_response parsed_response)
        {
            // Safe
            if (parsed_response.nb_elements != 0)
            {
                byte[] vals = new byte[parsed_response.nb_elements];
                if (OBDonUDSApi.GetData(ref parsed_response, vals, (Int32)parsed_response.nb_elements))
                    for (UInt32 i = 0; i < parsed_response.nb_elements; ++i)
                        Console.Write("{0}", (char)vals[i]);
            }
        }
        static unsafe void print_data_unsafe(ref obd_request_vehicle_information_response parsed_response)
        {
            // Unsafe
            if (parsed_response.nb_elements != 0)
            {
                byte[] vals = new byte[parsed_response.nb_elements];
                if (OBDonUDSApi.GetData_unsafe(ref parsed_response, vals, (Int32)parsed_response.nb_elements))
                    for (UInt32 i = 0; i < parsed_response.nb_elements; ++i)
                        Console.Write("{0}", (char)vals[i]);
            }
        }
        static void print_data_safe(ref obd_request_current_data_response parsed_response)
        {
            // Safe
            if (parsed_response.nb_elements != 0)
            {
                obd_did_object[] vals = new obd_did_object[parsed_response.nb_elements];
                if (OBDonUDSApi.GetData(ref parsed_response, vals, (Int32)parsed_response.nb_elements))
                {
                    for (UInt32 i = 0; i < parsed_response.nb_elements; ++i)
                    {
                        Console.Write("  -> Element {0} : ", i + 1);
                        print_did_safe(i, ref vals[i]);
                    }
                }
            }
        }

        static void print_did_safe(UInt32 index, ref obd_did_object didObject)
        {
            Console.WriteLine("DID 0x{0:X}, {1} bytes ", didObject.data_identifier, didObject.size);
            Console.Write("\t\t");
            // Safe
            if (didObject.size != 0)
            {
                byte[] vals = new byte[didObject.size];
                if (OBDonUDSApi.GetData(ref didObject, vals, (Int32)didObject.size))
                    for (UInt32 i = 0; i < didObject.size; ++i)
                        Console.Write("{0:X2} ", vals[i]);
                parse_DID_details(didObject.data_identifier, index, vals[0]);
            }
            Console.WriteLine("");
        }

        static unsafe void print_data_unsafe(ref obd_request_current_data_response parsed_response)
        {
            // Unsafe
            if (parsed_response.nb_elements != 0)
            {
                obd_did_object[] vals = new obd_did_object[parsed_response.nb_elements];
                if (OBDonUDSApi.GetData_unsafe(ref parsed_response, vals, (Int32)parsed_response.nb_elements))
                {
                    for (UInt32 i = 0; i < parsed_response.nb_elements; ++i)
                    {
                        Console.Write("  -> Element {0} : ", i + 1);
                        obd_did_object o = (obd_did_object)vals[i];
                        print_did_unsafe(i, ref o);
                    }
                }
            }
        }

        static unsafe void print_did_unsafe(UInt32 index, ref obd_did_object didObject)
        {
            Console.WriteLine("DID 0x{0:X}, {1} bytes ", didObject.data_identifier, didObject.size);
            Console.Write("\t\t");
            // Unsafe
            if (didObject.size != 0)
            {
                byte[] vals = new byte[didObject.size];
                if (OBDonUDSApi.GetData_unsafe(ref didObject, vals, (Int32)didObject.size))
                {
                    for (UInt32 i = 0; i < didObject.size; ++i)
                        Console.Write("{0:X2} ", vals[i]);
                    parse_DID_details(didObject.data_identifier, index, vals[0]);
                }
            }
            Console.WriteLine("");
        }
        static int GetVehicleIdentificationNumber(cantp_handle channel, obd_msgprotocol protocol)
        {
            int nbError = 0;

            const int MAX_NB_RESPONSES = 16;

            // Request vehicle identification number (VIN) using functional addressing scheme
            obd_DID_t[] VINitid = new obd_DID_t[1] { 0xF802 };
            obd_msg msg_request = new obd_msg();
            Console.WriteLine("Request vehicle identification number (VIN)");
            obd_status status = OBDonUDSApi.RequestVehicleInformation(channel, protocol == obd_msgprotocol.OBD_MSGPROTOCOL_11BIT ? OBDonUDSApi.OBD_NAI_REQUEST_FUNCTIONAL_11B : OBDonUDSApi.OBD_NAI_REQUEST_FUNCTIONAL_29B,
                out msg_request, VINitid, 1);

            if (OBDonUDSApi.StatusIsOk(status))
            {
                // Wait for responses
                obd_msg[] msg_responses = new obd_msg[MAX_NB_RESPONSES];
                UInt32 nb_responses;
                obd_msg msg_request_confirmation = new obd_msg();
                status = OBDonUDSApi.WaitForServiceFunctional(channel, ref msg_request, MAX_NB_RESPONSES, false, msg_responses, out nb_responses, out msg_request_confirmation);
                if (OBDonUDSApi.StatusIsOk(status))
                {
                    // Parse responses
                    for (UInt32 ir = 0; ir < nb_responses; ++ir)
                    {
                        obd_request_vehicle_information_response parsed_response = new obd_request_vehicle_information_response();
                        status = OBDonUDSApi.ParseResponse_RequestVehicleInformation(ref msg_responses[ir], out parsed_response);
                        if (OBDonUDSApi.StatusIsOk(status))
                        {
                            if (parsed_response.nrc != 0)
                                Console.WriteLine("-> Negative response code 0x{0:X}", parsed_response.nrc);
                            else
                            {
                                Console.Write("-> Vehicle Identification Number: '");
                                print_data_safe(ref parsed_response);
                                Console.WriteLine("'\n");
                                Console.Write("-> Vehicle Identification Number (Unsafe mode): '");
                                print_data_unsafe(ref parsed_response);
                                Console.WriteLine("'\n");
                            }
                        }
                        else
                        {
                            Console.Write("-> Parse ERROR (0x{0:X}): ", status); printStatusDetails(status);
                            nbError++;
                        }
                        // Free message
                        OBDonUDSApi.ParsedResponseFree(ref parsed_response);
                    }
                }
                else
                {
                    Console.Write("-> Wait ERROR (0x{0:X}): ", status); printStatusDetails(status);
                    nbError++;
                }

                // Free messages
                OBDonUDSApi.MsgFree(ref msg_request_confirmation);
                for (UInt32 ir = 0; ir < nb_responses; ++ir)
                    OBDonUDSApi.MsgFree(ref msg_responses[ir]);
            }
            else
            {
                Console.Write("-> Request ERROR (0x{0:X}): ", status); printStatusDetails(status);
                nbError++;
            }
            // Free messages
            OBDonUDSApi.MsgFree(ref msg_request);

            return nbError;
        }
        static void printStatusDetails(obd_status status)
        {
            StringBuilder buffer = new StringBuilder(BUFFER_SIZE);
            if (OBDonUDSApi.StatusIsOk(OBDonUDSApi.GetErrorText(status, 0, buffer, BUFFER_SIZE)))
                Console.WriteLine("\"{0}\"", buffer);
        }
        static int GetSupportedMask(cantp_handle channel)
        {
            int nbError = 0;

            Console.WriteLine("Supported items given by API parameters");

            byte[] supportedDIDs_F4XX = new byte[DID_SUPPORTED_SIZE];
            byte[] supportedDIDs_F5XX = new byte[DID_SUPPORTED_SIZE];
            byte[] supportedDIDs_F7XX = new byte[DID_SUPPORTED_SIZE];
            byte[] supportedMIDs = new byte[DID_SUPPORTED_SIZE];
            byte[] supportedITIDs = new byte[DID_SUPPORTED_SIZE];
            byte[] supportedRIDs_E0XX = new byte[DID_SUPPORTED_SIZE];
            byte[] supportedRIDs_E1XX = new byte[DID_SUPPORTED_SIZE];

            obd_status status;

            status = OBDonUDSApi.GetValue(channel, obd_parameter.POBDONUDS_PARAMETER_SUPPORTMASK_DIDS_F4XX, supportedDIDs_F4XX, DID_SUPPORTED_SIZE);
            if (!OBDonUDSApi.StatusIsOk(status))
            {
                Console.WriteLine("-> GetValue(POBDONUDS_PARAMETER_SUPPORTMASK_DIDS_F4XX) failed: 0x{0:X}", status);
                nbError++;
            }
            status = OBDonUDSApi.GetValue(channel, obd_parameter.POBDONUDS_PARAMETER_SUPPORTMASK_DIDS_F5XX, supportedDIDs_F5XX, DID_SUPPORTED_SIZE);
            if (!OBDonUDSApi.StatusIsOk(status))
            {
                Console.WriteLine("-> GetValue(POBDONUDS_PARAMETER_SUPPORTMASK_DIDS_F5XX) failed: 0x{0:X}", status);
                nbError++;
            }
            status = OBDonUDSApi.GetValue(channel, obd_parameter.POBDONUDS_PARAMETER_SUPPORTMASK_DIDS_F7XX, supportedDIDs_F7XX, DID_SUPPORTED_SIZE);
            if (!OBDonUDSApi.StatusIsOk(status))
            {
                Console.WriteLine("-> GetValue(POBDONUDS_PARAMETER_SUPPORTMASK_DIDS_F7XX) failed: 0x{0:X}", status);
                nbError++;
            }
            status = OBDonUDSApi.GetValue(channel, obd_parameter.POBDONUDS_PARAMETER_SUPPORTMASK_MIDS, supportedMIDs, DID_SUPPORTED_SIZE);
            if (!OBDonUDSApi.StatusIsOk(status))
            {
                Console.WriteLine("-> GetValue(POBDONUDS_PARAMETER_SUPPORTMASK_MIDS) failed: 0x{0:X}", status);
                nbError++;
            }
            status = OBDonUDSApi.GetValue(channel, obd_parameter.POBDONUDS_PARAMETER_SUPPORTMASK_ITIDS, supportedITIDs, DID_SUPPORTED_SIZE);
            if (!OBDonUDSApi.StatusIsOk(status))
            {
                Console.WriteLine("-> GetValue(POBDONUDS_PARAMETER_SUPPORTMASK_ITIDS) failed: 0x{0:X}", status);
                nbError++;
            }
            status = OBDonUDSApi.GetValue(channel, obd_parameter.POBDONUDS_PARAMETER_SUPPORTMASK_RIDS_E0XX, supportedRIDs_E0XX, DID_SUPPORTED_SIZE);
            if (!OBDonUDSApi.StatusIsOk(status))
            {
                Console.WriteLine("-> GetValue(POBDONUDS_PARAMETER_SUPPORTMASK_RIDS_E0XX) failed: 0x{0:X}", status);
                nbError++;
            }
            status = OBDonUDSApi.GetValue(channel, obd_parameter.POBDONUDS_PARAMETER_SUPPORTMASK_RIDS_E1XX, supportedRIDs_E1XX, DID_SUPPORTED_SIZE);
            if (!OBDonUDSApi.StatusIsOk(status))
            {
                Console.WriteLine("-> GetValue(POBDONUDS_PARAMETER_SUPPORTMASK_RIDS_E1XX) failed: 0x{0:X}", status);
                nbError++;
            }
            for (int did = 0; did < DID_SUPPORTED_SIZE; ++did)
            {
                for (byte ecuIndex = 0; ecuIndex < MAX_NB_ECU; ++ecuIndex)
                {
                    if (((supportedDIDs_F4XX[did] >> ecuIndex) & 0x01) != 0)
                    {
                        Console.WriteLine("-> DID  0x{0:X} is supported by ECU #{1}", 0xF400 + (did + 1), ecuIndex + 1);
                    }
                    if (((supportedDIDs_F5XX[did] >> ecuIndex) & 0x01) != 0)
                    {
                        Console.WriteLine("-> DID  0x{0:X} is supported by ECU #{1}", 0xF500 + (did + 1), ecuIndex + 1);
                    }
                    if (((supportedDIDs_F7XX[did] >> ecuIndex) & 0x01) != 0)
                    {
                        Console.WriteLine("-> DID  0x{0:X} is supported by ECU #{1}", 0xF700 + (did + 1), ecuIndex + 1);
                    }
                    if (((supportedMIDs[did] >> ecuIndex) & 0x01) != 0)
                    {
                        Console.WriteLine("-> MID  0x{0:X} is supported by ECU #{1}", 0xF600 + (did + 1), ecuIndex + 1);
                    }
                    if (((supportedITIDs[did] >> ecuIndex) & 0x01) != 0)
                    {
                        Console.WriteLine("-> ITID 0x{0:X} is supported by ECU #{1}", 0xF800 + (did + 1), ecuIndex + 1);
                    }
                    if (((supportedRIDs_E0XX[did] >> ecuIndex) & 0x01) != 0)
                    {
                        Console.WriteLine("-> RID  0x{0:X} is supported by ECU #{1}", 0xE000 + (did + 1), ecuIndex + 1);
                    }
                    if (((supportedRIDs_E1XX[did] >> ecuIndex) & 0x01) != 0)
                    {
                        Console.WriteLine("-> RID  0x{0:X} is supported by ECU #{1}", 0xE100 + (did + 1), ecuIndex + 1);
                    }
                }
            }

            return nbError;
        }
        static int displayRequestCurrentPowertrainDiagnosticData_22_DID(cantp_handle channel, obd_msgprotocol protocol)
        {
            int nbErr = displayRequestCurrentPowertrainDiagnosticData_22_DID_F401_functional(channel, protocol);
            nbErr += displayRequestCurrentPowertrainDiagnosticData_22_DID_All(channel, protocol);
            return nbErr;
        }

        static int displayRequestCurrentPowertrainDiagnosticData_22_DID_All(cantp_handle channel, obd_msgprotocol protocol)
        {
            int nbError = 0;

            Console.WriteLine("\nRequest Current Powertrain Diagnostic Data physically for each ECU for all supported DID");

            // Get supported dids
            byte[] supportedDIDs_F4XX = new byte[DID_SUPPORTED_SIZE];
            byte[] supportedDIDs_F5XX = new byte[DID_SUPPORTED_SIZE];
            byte[] supportedDIDs_F7XX = new byte[DID_SUPPORTED_SIZE];

            OBDonUDSApi.GetValue(channel, obd_parameter.POBDONUDS_PARAMETER_SUPPORTMASK_DIDS_F4XX, supportedDIDs_F4XX, DID_SUPPORTED_SIZE);
            OBDonUDSApi.GetValue(channel, obd_parameter.POBDONUDS_PARAMETER_SUPPORTMASK_DIDS_F5XX, supportedDIDs_F5XX, DID_SUPPORTED_SIZE);
            OBDonUDSApi.GetValue(channel, obd_parameter.POBDONUDS_PARAMETER_SUPPORTMASK_DIDS_F7XX, supportedDIDs_F7XX, DID_SUPPORTED_SIZE);

            // Sort supported dids per ECUs
            obd_DID_t[,] ECUSupportedDIDs = new obd_DID_t[MAX_NB_ECU, DID_SUPPORTED_SIZE * 3];
            int[] ECUSupportedDIDsSize = new int[MAX_NB_ECU];
            for (obd_DID_t did = 0; did < DID_SUPPORTED_SIZE; ++did)
            {
                for (byte ecuIndex = 0; ecuIndex < MAX_NB_ECU; ++ecuIndex)
                {
                    if ((did + 1) % 0x20 != 0) // avoid DID for "requesting supported" e.g. 0xF420
                    {
                        if (((supportedDIDs_F4XX[did] >> ecuIndex) & 0x01) != 0)
                        {
                            ECUSupportedDIDs[ecuIndex, ECUSupportedDIDsSize[ecuIndex]++] = (obd_DID_t)(0xF400 + (did + 1));
                        }
                        if (((supportedDIDs_F5XX[did] >> ecuIndex) & 0x01) != 0)
                        {
                            ECUSupportedDIDs[ecuIndex, ECUSupportedDIDsSize[ecuIndex]++] = (obd_DID_t)(0xF500 + (did + 1));
                        }
                        if (((supportedDIDs_F7XX[did] >> ecuIndex) & 0x01) != 0)
                        {
                            ECUSupportedDIDs[ecuIndex, ECUSupportedDIDsSize[ecuIndex]++] = (obd_DID_t)(0xF700 + (did + 1));
                        }
                    }
                }
            }

            // Request physically each ECU on supported DIDs (6 DIDs max per request)
            for (byte ecuIndex = 0; ecuIndex < MAX_NB_ECU; ++ecuIndex)
            {
                obd_DID_t[] RequestDIDs = new obd_DID_t[MAX_DID_PER_PHYSICAL_REQUEST];
                UInt32 RequestDIDsSize = 0;
                for (int i = 0; i < ECUSupportedDIDsSize[ecuIndex]; ++i)
                {
                    if (RequestDIDsSize < MAX_DID_PER_PHYSICAL_REQUEST)
                        RequestDIDs[RequestDIDsSize++] = ECUSupportedDIDs[ecuIndex, i];

                    if (RequestDIDsSize == MAX_DID_PER_PHYSICAL_REQUEST || i == ECUSupportedDIDsSize[ecuIndex] - 1)
                    {
                        nbError += displayRequestCurrentPowertrainDiagnosticData_22_DID_Some(channel, protocol, (byte)(ecuIndex + 1), RequestDIDs, RequestDIDsSize);
                        RequestDIDsSize = 0;
                    }
                }
            }

            return nbError;
        }

        static int displayRequestCurrentPowertrainDiagnosticData_22_DID_Some(cantp_handle channel, obd_msgprotocol protocol, byte ecu, obd_DID_t[] DIDs, UInt32 DIDsLength)
        {
            int nbError = 0;
            obd_msg msg_request = new obd_msg();
            obd_status status;

            obd_netaddrinfo nai;
            nai.protocol = protocol;
            nai.source_addr = (UInt16)uds_address.PUDS_ADDRESS_ISO_15765_4_ADDR_TEST_EQUIPMENT;
            nai.target_addr = ecu;
            nai.target_type = obd_addressing.OBD_ADDRESSING_PHYSICAL;
            status = OBDonUDSApi.RequestCurrentData(channel, nai, out msg_request, DIDs, DIDsLength);
            if (OBDonUDSApi.StatusIsOk(status))
            {
                // Wait for responses
                obd_msg msg_response = new obd_msg();
                obd_msg msg_request_confirmation = new obd_msg();
                status = OBDonUDSApi.WaitForService(channel, ref msg_request, out msg_response, out msg_request_confirmation);
                if (OBDonUDSApi.StatusIsOk(status))
                {
                    // Parse response
                    obd_request_current_data_response parsed_response = new obd_request_current_data_response();
                    status = OBDonUDSApi.ParseResponse_RequestCurrentData(ref msg_response, out parsed_response);
                    if (OBDonUDSApi.StatusIsOk(status))
                    {
                        // Print parsed response
                        Console.WriteLine("-> Response from ECU #{0} : ", parsed_response.ecu_address);
                        if (parsed_response.nrc != 0)
                            Console.WriteLine("Negative response code 0x{0:X}\n", parsed_response.nrc);
                        else
                        {
                            print_data_safe(ref parsed_response);
                            Console.WriteLine("Unsafe");
                            print_data_unsafe(ref parsed_response);
                            Console.WriteLine("\n");
                        }
                    }
                    else
                    {
                        Console.Write("-> Parse DIDs: ERROR (0x{0:X}): ", status); printStatusDetails(status);
                        nbError++;
                    }
                    // Free message
                    OBDonUDSApi.ParsedResponseFree(ref parsed_response);
                }
                else
                {
                    Console.Write("-> Wait DIDs: ERROR (0x{0:X}): ", status); printStatusDetails(status);
                    nbError++;
                }

                // Free messages
                OBDonUDSApi.MsgFree(ref msg_request_confirmation);
                OBDonUDSApi.MsgFree(ref msg_response);
            }
            else
            {
                Console.Write("-> Request DIDs: ERROR (0x{0:X}): ", status); printStatusDetails(status);
                nbError++;
            }
            // Free messages
            OBDonUDSApi.MsgFree(ref msg_request);

            return nbError;
        }

        static int displayRequestCurrentPowertrainDiagnosticData_22_DID_F401_functional(cantp_handle channel, obd_msgprotocol protocol)
        {
            int nbError = 0;
            obd_DID_t[] did = new obd_DID_t[1] { 0xF401 };
            obd_msg msg_request = new obd_msg();

            Console.WriteLine("\nRequest Current Powertrain Diagnostic Data functionally for DID F401");

            obd_status status = OBDonUDSApi.RequestCurrentData(channel, protocol == obd_msgprotocol.OBD_MSGPROTOCOL_11BIT ? OBDonUDSApi.OBD_NAI_REQUEST_FUNCTIONAL_11B : OBDonUDSApi.OBD_NAI_REQUEST_FUNCTIONAL_29B,
                out msg_request, did, 1);
            if (OBDonUDSApi.StatusIsOk(status))
            {
                // Wait for responses
                obd_msg[] msg_responses = new obd_msg[MAX_NB_RESPONSES];
                UInt32 nb_responses;
                obd_msg msg_request_confirmation = new obd_msg();
                status = OBDonUDSApi.WaitForServiceFunctional(channel, ref msg_request, MAX_NB_RESPONSES, false, msg_responses, out nb_responses, out msg_request_confirmation);
                if (OBDonUDSApi.StatusIsOk(status))
                {
                    for (UInt32 ir = 0; ir < nb_responses; ++ir)
                    {
                        // Parse response
                        obd_request_current_data_response parsed_response = new obd_request_current_data_response();
                        status = OBDonUDSApi.ParseResponse_RequestCurrentData(ref msg_responses[ir], out parsed_response);
                        if (OBDonUDSApi.StatusIsOk(status))
                        {
                            // Print parsed response
                            Console.WriteLine("-> Response number {0} from ECU #{1} : ", ir + 1, parsed_response.ecu_address);
                            if (parsed_response.nrc != 0)
                                Console.WriteLine("-> Negative response code 0x{0:X}", parsed_response.nrc);
                            else
                            {
                                print_data_safe(ref parsed_response);
                                Console.WriteLine("Unsafe");
                                print_data_unsafe(ref parsed_response);
                            }
                        }
                        else
                        {
                            Console.Write("-> Parse DID: {0:X}: ERROR (0x{1:X}): ", did, status); printStatusDetails(status);
                            nbError++;
                        }

                        OBDonUDSApi.ParsedResponseFree(ref parsed_response);
                    }
                }
                else
                {
                    Console.Write("-> Wait DID: {0:X}: ERROR (0x{1:X}): ", did, status); printStatusDetails(status);
                    nbError++;
                }

                OBDonUDSApi.MsgFree(ref msg_request_confirmation);
                for (UInt32 i = 0; i < nb_responses; ++i)
                    OBDonUDSApi.MsgFree(ref msg_responses[i]);

            }
            else
            {
                Console.Write("-> Request DID: {0:X}: ERROR (0x{1:X}): ", did, status); printStatusDetails(status);
                nbError++;
            }

            OBDonUDSApi.MsgFree(ref msg_request);

            return nbError;
        }
        static void parse_DID_details(obd_DID_t dataIdentifier, UInt32 index, byte data)
        {
            // Examples of detailed parsing
            if (dataIdentifier == 0xF401 && index == 0)
            {
                parse_F401_IM_ReadinessData_Details(data);
            }
            else if (dataIdentifier == 0xF404)
            {
                parse_F404_CalculatedLoadValue_Details(data);
            }
            else if (dataIdentifier == 0xF40D)
            {
                parse_F40D_VehicleSpeedSensor_Details(data);
            }
            // TODO: parse details of other elements
        }
        static void parse_F401_IM_ReadinessData_Details(byte data)
        {
            // display Number of DTCs stored : A bits 0-6 and MIL status : A bit 7		
            Console.WriteLine("\n-> DTC_CNT: {0}d", data & 0b01111111);
            Console.WriteLine("-> MIL: {0}", (data & 0b10000000) == 0 ? "OFF" : "ON");
        }
        static void parse_F404_CalculatedLoadValue_Details(byte data)
        {
            Console.Write("\n  -> LOAD_PCT: {0:F1}%", data * 100.0 / 255);
        }

        static void parse_F40D_VehicleSpeedSensor_Details(byte data)
        {
            Console.Write("\n  -> VSS: {0:D3} km/h", data);
        }
        static int displayRequestEmissionRelatedDiagnosticTroubleCodesWithConfirmedStatus_19_42(cantp_handle channel, obd_msgprotocol protocol, obd_DTC_t[][] DTCarray)
        {
            int nbError = 0;
            obd_status status;

            Console.WriteLine("\nRequest Emission-Related Diagnostic Trouble Codes With Confirmed Status physically for each ECU");

            byte[] availableECUs = new byte[1];
            OBDonUDSApi.GetValue(channel, obd_parameter.POBDONUDS_PARAMETER_AVAILABLE_ECUS, availableECUs, 1);

            for (byte ecuIndex = 0; ecuIndex < availableECUs[0]; ++ecuIndex)
            {
                obd_netaddrinfo nai;
                nai.protocol = protocol;
                nai.source_addr = (UInt16)uds_address.PUDS_ADDRESS_ISO_15765_4_ADDR_TEST_EQUIPMENT;
                nai.target_addr = (UInt16)(ecuIndex + 1);
                nai.target_type = obd_addressing.OBD_ADDRESSING_PHYSICAL;
                obd_msg msg_request = new obd_msg();
                status = OBDonUDSApi.RequestConfirmedTroubleCodes(channel, nai, out msg_request,
                    OBDonUDSApi.OBDONUDS_EMISSION_SYSTEM_GROUP, OBDonUDSApi.OBDONUDS_DTC_STATUS_CONFIRMED, OBDonUDSApi.OBDONUDS_DTC_SEVERITY_CLASS_1);
                if (OBDonUDSApi.StatusIsOk(status))
                {
                    // Wait for response
                    obd_msg msg_response = new obd_msg();
                    obd_msg msg_request_confirmation = new obd_msg();
                    status = OBDonUDSApi.WaitForService(channel, ref msg_request, out msg_response, out msg_request_confirmation);
                    if (OBDonUDSApi.StatusIsOk(status))
                    {
                        // Parse response
                        obd_request_trouble_codes_response parsed_response = new obd_request_trouble_codes_response();
                        status = OBDonUDSApi.ParseResponse_RequestConfirmedTroubleCodes(ref msg_response, out parsed_response);
                        if (OBDonUDSApi.StatusIsOk(status))
                        {
                            // Print parsed response
                            printDTCParsed(ref parsed_response);
                            // Keep a list of DTC
                            nbError += KeepDTC(DTCarray, ecuIndex, ref parsed_response);
                        }
                        else
                        {
                            Console.Write("-> Parse : ERROR (0x{0:X}): ", status); printStatusDetails(status);
                            nbError++;
                        }
                        // Free message
                        OBDonUDSApi.ParsedResponseFree(ref parsed_response);
                    }
                    else
                    {
                        Console.Write("-> Wait ECU #{0}: ERROR (0x{1:X}): ", ecuIndex + 1, status); printStatusDetails(status);
                        nbError++;
                    }

                    // Free messages
                    OBDonUDSApi.MsgFree(ref msg_request_confirmation);
                    OBDonUDSApi.MsgFree(ref msg_response);
                }
                else
                {
                    Console.Write("-> Request : ERROR (0x{0:X}): ", status); printStatusDetails(status);
                    nbError++;
                }
                // Free messages
                OBDonUDSApi.MsgFree(ref msg_request);
            }

            return nbError;
        }
        static int displayRequestEmissionRelatedDiagnosticTroubleCodesWithPendingStatus_19_42(cantp_handle channel, obd_msgprotocol protocol, obd_DTC_t[][] DTCarray)
        {
            int nbError = 0;
            obd_status status;

            Console.WriteLine("\nRequest Emission-Related Diagnostic Trouble Codes With Pending Status physically for each ECU");

            byte[] availableECUs = new byte[1];
            OBDonUDSApi.GetValue(channel, obd_parameter.POBDONUDS_PARAMETER_AVAILABLE_ECUS, availableECUs, 1);

            for (byte ecuIndex = 0; ecuIndex < availableECUs[0]; ++ecuIndex)
            {
                obd_netaddrinfo nai;
                nai.protocol = protocol;
                nai.source_addr = (UInt16)uds_address.PUDS_ADDRESS_ISO_15765_4_ADDR_TEST_EQUIPMENT; ;
                nai.target_addr = (UInt16)(ecuIndex + 1);
                nai.target_type = obd_addressing.OBD_ADDRESSING_PHYSICAL;
                obd_msg msg_request = new obd_msg();
                status = OBDonUDSApi.RequestPendingTroubleCodes(channel, nai, out msg_request,
                    OBDonUDSApi.OBDONUDS_EMISSION_SYSTEM_GROUP, OBDonUDSApi.OBDONUDS_DTC_STATUS_PENDING, OBDonUDSApi.OBDONUDS_DTC_SEVERITY_CLASS_1);
                if (OBDonUDSApi.StatusIsOk(status))
                {
                    // Wait for response
                    obd_msg msg_response = new obd_msg();
                    obd_msg msg_request_confirmation = new obd_msg();
                    status = OBDonUDSApi.WaitForService(channel, ref msg_request, out msg_response, out msg_request_confirmation);
                    if (OBDonUDSApi.StatusIsOk(status))
                    {
                        // Parse response
                        obd_request_trouble_codes_response parsed_response = new obd_request_trouble_codes_response();
                        status = OBDonUDSApi.ParseResponse_RequestPendingTroubleCodes(ref msg_response, out parsed_response);
                        if (OBDonUDSApi.StatusIsOk(status))
                        {
                            // Print parsed response
                            printDTCParsed(ref parsed_response);
                            // Keep a list of DTC
                            nbError += KeepDTC(DTCarray, ecuIndex, ref parsed_response);
                        }
                        else
                        {
                            Console.Write("-> Parse : ERROR (0x{0:X}): ", status); printStatusDetails(status);
                            nbError++;
                        }
                        // Free message
                        OBDonUDSApi.ParsedResponseFree(ref parsed_response);
                    }
                    else
                    {
                        Console.Write("-> Wait ECU #{0}: ERROR (0x{1:X}): ", ecuIndex + 1, status); printStatusDetails(status);
                        nbError++;
                    }

                    // Free messages
                    OBDonUDSApi.MsgFree(ref msg_request_confirmation);
                    OBDonUDSApi.MsgFree(ref msg_response);
                }
                else
                {
                    Console.Write("-> Request : ERROR (0x{0:X}): ", status); printStatusDetails(status);
                    nbError++;
                }
                // Free messages
                OBDonUDSApi.MsgFree(ref msg_request);
            }

            return nbError;
        }
        static int KeepDTC(obd_DTC_t[][] DTCarray, byte ecuIndex, ref obd_request_trouble_codes_response parsed_response)
        {
            int nbError = 0;

            if (parsed_response.nb_elements != 0)
            {
                DTCarray[ecuIndex] = new obd_DTC_t[parsed_response.nb_elements];
                obd_severity_trouble_code[] vals = new obd_severity_trouble_code[parsed_response.nb_elements];
                if (OBDonUDSApi.GetData(ref parsed_response, vals, (Int32)parsed_response.nb_elements))
                    for (UInt32 i = 0; i < parsed_response.nb_elements; ++i)
                    {
                        DTCarray[ecuIndex][i] = vals[i].DTC_identifier;
                    }
            }

            return nbError;
        }

        static void printDTCParsed(ref obd_request_trouble_codes_response parsed_response)
        {
            Console.Write("-> Response from ECU #{0} : ", parsed_response.ecu_address);
            if (parsed_response.nrc != 0)
                Console.WriteLine("-> Negative response code 0x{0:X}", parsed_response.nrc);
            else
            {
                Console.Write("Report type 0x{0:X2}, Functional group identifier 0x{1:X2}, DTC status availability mask 0x{2:X2}, DTC severity mask 0x{3:X2}, DTC format identifier 0x{4:X2}",
                    parsed_response.report_type, parsed_response.functional_group_identifier, parsed_response.DTC_status_availability_mask,
                    parsed_response.DTC_severity_mask, parsed_response.DTC_format_identifier);
                if (parsed_response.nb_elements != 0)
                {
                    obd_severity_trouble_code[] vals = new obd_severity_trouble_code[parsed_response.nb_elements];
                    if (OBDonUDSApi.GetData(ref parsed_response, vals, (Int32)parsed_response.nb_elements))
                        for (UInt32 ip = 0; ip < parsed_response.nb_elements; ++ip)
                        {
                            Console.Write("\n  -> Element {0} : ", ip + 1);
                            Console.Write("DTC identifier 0x{0:X}, Status 0x{1:X2}, Severity 0x{2:X2}", vals[ip].DTC_identifier,
                               vals[ip].status_of_dtc, vals[ip].DTC_severity);
                        }
                }
                Console.WriteLine("");
            }
        }
        static int displayRequestPowertrainFreezeFrameData_19_04(cantp_handle channel, obd_msgprotocol protocol, obd_DTC_t[][] confirmedDTCs, obd_DTC_t[][] pendingDTCs)
        {
            int nbError = 0;

            Console.WriteLine("\nRequest Powertrain Freeze Frame Data physically for each ECU for each pending or confirmed DTC");

            byte[] availableECUs = new byte[1];
            OBDonUDSApi.GetValue(channel, obd_parameter.POBDONUDS_PARAMETER_AVAILABLE_ECUS, availableECUs, 1);

            for (byte ecuIndex = 0; ecuIndex < availableECUs[0]; ++ecuIndex)
            {
                nbError += displayRequestPowertrainFreezeFrameData_19_04_Some(channel, protocol, ecuIndex, confirmedDTCs);
                nbError += displayRequestPowertrainFreezeFrameData_19_04_Some(channel, protocol, ecuIndex, pendingDTCs);
            }

            return nbError;
        }

        static int displayRequestPowertrainFreezeFrameData_19_04_Some(cantp_handle channel, obd_msgprotocol protocol, byte ecuIndex, obd_DTC_t[][] DTCarray)
        {
            int nbError = 0;
            obd_status status;

            if (DTCarray[ecuIndex] == null)
                return nbError;

            for (byte DTCindex = 0; DTCindex < DTCarray[ecuIndex].Length; ++DTCindex)
            {
                for (UInt16 occurrence = 0x00; occurrence <= 0xF0; occurrence += 0xF0)
                {
                    obd_netaddrinfo nai;
                    nai.protocol = protocol;
                    nai.source_addr = (UInt16)uds_address.PUDS_ADDRESS_ISO_15765_4_ADDR_TEST_EQUIPMENT; ;
                    nai.target_addr = (UInt16)(ecuIndex + 1);
                    nai.target_type = obd_addressing.OBD_ADDRESSING_PHYSICAL;
                    obd_msg msg_request = new obd_msg();
                    status = OBDonUDSApi.RequestFreezeFrameData(channel, nai, out msg_request, DTCarray[ecuIndex][DTCindex], (byte)occurrence);
                    if (OBDonUDSApi.StatusIsOk(status))
                    {
                        // Wait for response
                        obd_msg msg_response = new obd_msg();
                        obd_msg msg_request_confirmation = new obd_msg();
                        status = OBDonUDSApi.WaitForService(channel, ref msg_request, out msg_response, out msg_request_confirmation);
                        if (OBDonUDSApi.StatusIsOk(status))
                        {
                            // Parse response
                            obd_request_freeze_frame_data_response parsed_response = new obd_request_freeze_frame_data_response();
                            status = OBDonUDSApi.ParseResponse_RequestFreezeFrameData(ref msg_response, out parsed_response);
                            if (OBDonUDSApi.StatusIsOk(status))
                            {
                                // Print parsed response
                                printSnapshot(ref parsed_response);
                            }
                            else
                            {
                                Console.Write("-> Parse : ERROR (0x{0:X}): ", status); printStatusDetails(status);
                                nbError++;
                            }
                            // Free message
                            OBDonUDSApi.ParsedResponseFree(ref parsed_response);
                        }
                        else
                        {
                            Console.Write("-> Wait failed for ECU #{0} DTC 0x{1:X} occurrence 0x{2:X} (0x{3:X}): ", ecuIndex + 1, DTCarray[ecuIndex][DTCindex], occurrence, status); printStatusDetails(status);
                            nbError++;
                        }

                        // Free messages
                        OBDonUDSApi.MsgFree(ref msg_request_confirmation);
                        OBDonUDSApi.MsgFree(ref msg_response);
                    }
                    else
                    {
                        Console.Write("-> Request : ERROR (0x{0:X}): ", status); printStatusDetails(status);
                        nbError++;
                    }
                    // Free messages
                    OBDonUDSApi.MsgFree(ref msg_request);

                }
            }

            return nbError;
        }

        static void printSnapshot(ref obd_request_freeze_frame_data_response parsed_response)
        {
            Console.Write("-> Response from ECU #{0} : ", parsed_response.ecu_address);
            if (parsed_response.nrc != 0)
            {
                Console.WriteLine("Negative response code 0x{0:X}", parsed_response.nrc);
            }
            else
            {
                Console.Write("Report type 0x{0:X2}, DTC 0x{1:X}, DTC status 0x{2:X2}, Record number 0x{3:X2}",
                    parsed_response.report_type, parsed_response.dtc_number, parsed_response.status_of_dtc,
                    parsed_response.record_number);
                if (parsed_response.nb_identifiers != 0)
                {
                    obd_did_object[] vals = new obd_did_object[parsed_response.nb_identifiers];
                    if (OBDonUDSApi.GetData(ref parsed_response, vals, parsed_response.nb_identifiers))
                        for (UInt32 id = 0; id < parsed_response.nb_identifiers; ++id)
                        {
                            Console.Write("\n  -> Identifier {0} : ", id + 1);
                            Console.Write("Data identifier 0x{0:X},", vals[id].data_identifier);
                            if (vals[id].size != 0)
                            {
                                byte[] data = new byte[vals[id].size];
                                if (OBDonUDSApi.GetData(ref vals[id], data, (Int32)vals[id].size))
                                    for (UInt32 idata = 0; idata < vals[id].size; ++idata)
                                    {
                                        Console.Write(" {0:X2}", data[idata]);
                                        // Example of detailed parsing : for DID F404
                                        if (vals[id].data_identifier == 0xF404 && vals[id].size == 1)
                                        {
                                            parse_F404_CalculatedLoadValue_Details(data[0]);
                                        }
                                        // TODO: parse details of other elements
                                    }
                            }
                        }
                }
                Console.WriteLine("");
            }
        }
        static int displayClearResetEmissionRelatedDiagnosticInformation_14(cantp_handle channel, obd_msgprotocol protocol, UInt32 GroupOfDTC)
        {
            int nbError = 0;
            Console.WriteLine("\nRequest functionally Clear/Reset Emission-Related Diagnostic Information for group of DTC 0x{0:X}", GroupOfDTC);

            obd_msg msg_request = new obd_msg();
            obd_status status = OBDonUDSApi.ClearTroubleCodes(channel, protocol == obd_msgprotocol.OBD_MSGPROTOCOL_11BIT ? OBDonUDSApi.OBD_NAI_REQUEST_FUNCTIONAL_11B : OBDonUDSApi.OBD_NAI_REQUEST_FUNCTIONAL_29B,
                out msg_request, GroupOfDTC);
            if (OBDonUDSApi.StatusIsOk(status))
            {
                // Wait for responses
                obd_msg[] msg_responses = new obd_msg[MAX_NB_RESPONSES];
                UInt32 nb_responses;
                obd_msg msg_request_confirmation = new obd_msg();
                status = OBDonUDSApi.WaitForServiceFunctional(channel, ref msg_request, MAX_NB_RESPONSES, false, msg_responses, out nb_responses, out msg_request_confirmation);
                if (OBDonUDSApi.StatusIsOk(status))
                {
                    for (UInt32 ir = 0; ir < nb_responses; ++ir)
                    {
                        // Parse response
                        obd_request_clear_trouble_codes_response parsed_response = new obd_request_clear_trouble_codes_response();
                        status = OBDonUDSApi.ParseResponse_ClearTroubleCodes(ref msg_responses[ir], out parsed_response);
                        if (OBDonUDSApi.StatusIsOk(status))
                        {
                            // Print parsed response
                            Console.Write("-> Response number {0} from ECU #{1} : ", ir + 1, parsed_response.ecu_address);
                            if (parsed_response.nrc != 0)
                            {
                                Console.WriteLine("-> Negative response code 0x{0:X}", parsed_response.nrc);
                            }
                            else
                            {
                                Console.WriteLine("-> Positive response");
                            }
                        }
                        else
                        {
                            Console.Write("-> Parse ERROR (0x{0:X}): ", status); printStatusDetails(status);
                            nbError++;
                        }

                        OBDonUDSApi.ParsedResponseFree(ref parsed_response);
                    }
                }
                else
                {
                    Console.Write("-> Wait failed (0x{0:X}): ", status); printStatusDetails(status);
                    nbError++;
                }

                OBDonUDSApi.MsgFree(ref msg_request_confirmation);
                for (UInt32 i = 0; i < nb_responses; ++i)
                    OBDonUDSApi.MsgFree(ref msg_responses[i]);

            }
            else
            {
                Console.Write("Request ERROR (0x{0:X}): ", status); printStatusDetails(status);
                nbError++;
            }

            OBDonUDSApi.MsgFree(ref msg_request);

            return nbError;

        }

        static int displayRequestOnBoardMonitoringTestResultsForSpecificMonitoredSystems_22_MID(cantp_handle channel, obd_msgprotocol protocol)
        {
            int nbError = 0;

            Console.WriteLine("\nRequest On Board Monitoring Test Results For Specific Monitored Systems physically for each ECU for all supported MID");

            // Get supported mids
            byte[] supportedMIDs = new byte[DID_SUPPORTED_SIZE];
            OBDonUDSApi.GetValue(channel, obd_parameter.POBDONUDS_PARAMETER_SUPPORTMASK_MIDS, supportedMIDs, DID_SUPPORTED_SIZE);

            // Request physically each ECU on supported MIDs (1 MIDs max per request)
            for (obd_DID_t mid = 0; mid < DID_SUPPORTED_SIZE; ++mid)
            {
                for (byte ecuIndex = 0; ecuIndex < MAX_NB_ECU; ++ecuIndex)
                {
                    if ((mid + 1) % 0x20 != 0) // avoid MID for "requesting supported" e.g. 0xF620
                    {
                        if (((supportedMIDs[mid] >> ecuIndex) & 0x01) != 0)
                        {
                            nbError += displayRequestOnBoardMonitoringTestResultsForSpecificMonitoredSystems_22_MID_one(channel, protocol, (byte)(ecuIndex + 1), (obd_DID_t)(0xF600 + (mid + 1)));
                        }
                    }
                }
            }

            return nbError;
        }

        static int displayRequestOnBoardMonitoringTestResultsForSpecificMonitoredSystems_22_MID_one(cantp_handle channel, obd_msgprotocol protocol, byte ecu, obd_DID_t MID)
        {
            int nbError = 0;
            obd_msg msg_request = new obd_msg();
            obd_status status;

            obd_netaddrinfo nai;
            nai.protocol = protocol;
            nai.source_addr = (UInt16)uds_address.PUDS_ADDRESS_ISO_15765_4_ADDR_TEST_EQUIPMENT;
            nai.target_addr = ecu;
            nai.target_type = obd_addressing.OBD_ADDRESSING_PHYSICAL;
            obd_DID_t[] mids = new obd_DID_t[1] { MID };
            status = OBDonUDSApi.RequestTestResults(channel, nai, out msg_request, mids, 1);
            if (OBDonUDSApi.StatusIsOk(status))
            {
                // Wait for response
                obd_msg msg_response = new obd_msg();
                obd_msg msg_request_confirmation = new obd_msg();
                status = OBDonUDSApi.WaitForService(channel, ref msg_request, out msg_response, out msg_request_confirmation);
                if (OBDonUDSApi.StatusIsOk(status))
                {
                    // Parse response
                    obd_request_test_results_response parsed_response;
                    status = OBDonUDSApi.ParseResponse_RequestTestResults(ref msg_response, out parsed_response);
                    if (OBDonUDSApi.StatusIsOk(status))
                    {
                        // Print parsed response
                        Console.Write("-> Response from ECU #{0} : ", parsed_response.ecu_address);
                        if (parsed_response.nrc != 0)
                            Console.WriteLine("-> Negative response code 0x{0:X}", parsed_response.nrc);
                        else
                        {
                            Console.Write("-> MID 0x{0:X} ", parsed_response.data_identifier);
                            if (parsed_response.nb_elements != 0)
                            {
                                obd_test_data_object[] vals = new obd_test_data_object[parsed_response.nb_elements];
                                if (OBDonUDSApi.GetData(ref parsed_response, vals, (Int32)parsed_response.nb_elements))
                                    for (UInt32 ip = 0; ip < parsed_response.nb_elements; ++ip)
                                    {
                                        Console.Write("\n  -> Element {0} : ", ip + 1);
                                        printTestObject(ref vals[ip]);
                                    }
                            }
                            Console.WriteLine();
                        }
                    }
                    else
                    {
                        Console.Write("-> Parse MID: ERROR (0x{0:X}): ", status); printStatusDetails(status);
                        nbError++;
                    }
                    // Free message
                    OBDonUDSApi.ParsedResponseFree(ref parsed_response);
                }
                else
                {
                    Console.Write("-> Wait MID 0x{0:X} ECU #{1}: ERROR (0x{2:X}) :", MID, ecu, status); printStatusDetails(status);
                    nbError++;
                }

                // Free messages
                OBDonUDSApi.MsgFree(ref msg_request_confirmation);
                OBDonUDSApi.MsgFree(ref msg_response);
            }
            else
            {
                Console.Write("-> Request MID: ERROR (0x{0:X}): ", status); printStatusDetails(status);
                nbError++;
            }
            // Free messages
            OBDonUDSApi.MsgFree(ref msg_request);

            return nbError;
        }
        static int displayRequestControlOfOnBoardSystemTestOrComponent_31_RID(cantp_handle channel, obd_msgprotocol protocol)
        {
            int nbError = 0;

            Console.WriteLine("\nRequest Control Of On-Board System, Test, Or Component physically for each ECU for all supported RID");

            // Get supported rids
            byte[] supportedRIDs_E0XX = new byte[DID_SUPPORTED_SIZE];
            byte[] supportedRIDs_E1XX = new byte[DID_SUPPORTED_SIZE];
            OBDonUDSApi.GetValue(channel, obd_parameter.POBDONUDS_PARAMETER_SUPPORTMASK_RIDS_E0XX, supportedRIDs_E0XX, DID_SUPPORTED_SIZE);
            OBDonUDSApi.GetValue(channel, obd_parameter.POBDONUDS_PARAMETER_SUPPORTMASK_RIDS_E1XX, supportedRIDs_E1XX, DID_SUPPORTED_SIZE);

            // Request physically each ECU on supported RIDs (1 RID max per request)
            for (obd_DID_t rid = 0; rid < DID_SUPPORTED_SIZE; ++rid)
            {
                for (byte ecuIndex = 0; ecuIndex < MAX_NB_ECU; ++ecuIndex)
                {
                    if ((rid + 1) % 0x20 != 0) // avoid RID for "requesting supported" e.g. 0xE020
                    {
                        if (((supportedRIDs_E0XX[rid] >> ecuIndex) & 0x01) != 0)
                        {
                            nbError += displayRequestControlOfOnBoardSystemTestOrComponent_31_RID_one(channel, protocol, ecuIndex, (obd_DID_t)(0xE000 + (rid + 1)));
                        }
                        if (((supportedRIDs_E1XX[rid] >> ecuIndex) & 0x01) != 0)
                        {
                            nbError += displayRequestControlOfOnBoardSystemTestOrComponent_31_RID_one(channel, protocol, ecuIndex, (obd_DID_t)(0xE100 + (rid + 1)));
                        }
                    }
                }
            }

            return nbError;
        }

        static int displayRequestControlOfOnBoardSystemTestOrComponent_31_RID_one(cantp_handle channel, obd_msgprotocol protocol, byte ecuIndex, obd_DID_t RID)
        {
            int nbError = 0;
            obd_msg msg_request = new obd_msg();
            obd_status status;

            obd_netaddrinfo nai;
            nai.protocol = protocol;
            nai.source_addr = (UInt16)uds_address.PUDS_ADDRESS_ISO_15765_4_ADDR_TEST_EQUIPMENT;
            nai.target_addr = (UInt16)(ecuIndex + 1);
            nai.target_type = obd_addressing.OBD_ADDRESSING_PHYSICAL;
            status = OBDonUDSApi.RequestControlOperation(channel, nai, out msg_request, OBDonUDSApi.OBDONUDS_ROUTINE_START, RID, null, 0);
            if (OBDonUDSApi.StatusIsOk(status))
            {
                // Wait for response
                obd_msg msg_response = new obd_msg();
                obd_msg msg_request_confirmation = new obd_msg();
                status = OBDonUDSApi.WaitForService(channel, ref msg_request, out msg_response, out msg_request_confirmation);
                if (OBDonUDSApi.StatusIsOk(status))
                {
                    // Parse response
                    obd_request_control_operation_response parsed_response;
                    status = OBDonUDSApi.ParseResponse_RequestControlOperation(ref msg_response, out parsed_response);
                    if (OBDonUDSApi.StatusIsOk(status))
                    {
                        // Print parsed response
                        Console.Write("-> Response from ECU #{0} : ", parsed_response.ecu_address);
                        if (parsed_response.nrc != 0)
                            Console.WriteLine("-> Negative response code 0x{0:X}", parsed_response.nrc);
                        else
                        {
                            Console.Write("Routine Control Type 0x{0:X2} ", parsed_response.routine_control_type);
                            Console.Write("RID 0x{0:X} ", parsed_response.routine_identifier);
                            Console.Write("Routine Info 0x{0:X2} ", parsed_response.routine_info);
                            if (parsed_response.nb_elements != 0)
                            {
                                byte[] vals = new byte[parsed_response.nb_elements];
                                if (OBDonUDSApi.GetData(ref parsed_response, vals, parsed_response.nb_elements))
                                    for (UInt32 ip = 0; ip < parsed_response.nb_elements; ++ip)
                                    {
                                        Console.Write("\n  -> Routine Status Element {0} : 0x{1:X2}", ip + 1, vals[ip]);
                                    }
                            }
                            Console.WriteLine();
                        }
                    }
                    else
                    {
                        Console.Write("-> Parse RID: ERROR (0x{0:X}): ", status); printStatusDetails(status);
                        nbError++;
                    }
                    // Free message
                    OBDonUDSApi.ParsedResponseFree(ref parsed_response);
                }
                else
                {
                    Console.Write("-> Wait RID: ERROR (0x{0:X}): ", status); printStatusDetails(status);
                    nbError++;
                }

                // Free messages
                OBDonUDSApi.MsgFree(ref msg_request_confirmation);
                OBDonUDSApi.MsgFree(ref msg_response);
            }
            else
            {
                Console.Write("-> Request RID: ERROR (0x{0:X}): ", status); printStatusDetails(status);
                nbError++;
            }
            // Free messages
            OBDonUDSApi.MsgFree(ref msg_request);

            return nbError;

        }
        static int displayRequestVehicleInformation_22_ITID(cantp_handle channel, obd_msgprotocol protocol)
        {
            int nbError = displayRequestVehicleInformation_22_ITID_F802_functional(channel, protocol);
            nbError += displayRequestVehicleInformation_22_ITID_All(channel, protocol);
            return nbError;

        }

        static int displayRequestVehicleInformation_22_ITID_F802_functional(cantp_handle channel, obd_msgprotocol protocol)
        {
            Console.WriteLine("\nRequest Vehicle Information functionally for ITID F802");

            return GetVehicleIdentificationNumber(channel, protocol);
        }

        static int displayRequestVehicleInformation_22_ITID_All(cantp_handle channel, obd_msgprotocol protocol)
        {
            int nbError = 0;

            Console.WriteLine("\nRequest Vehicle Information physically for each ECU for all supported ITID");

            // Get supported dids
            byte[] supportedITIDs = new byte[DID_SUPPORTED_SIZE];
            OBDonUDSApi.GetValue(channel, obd_parameter.POBDONUDS_PARAMETER_SUPPORTMASK_ITIDS, supportedITIDs, DID_SUPPORTED_SIZE);

            // Request physically each ECU on supported ITIDs (1 ITID max per request)
            for (obd_DID_t itid = 0; itid < DID_SUPPORTED_SIZE; ++itid)
            {
                for (byte ecuIndex = 0; ecuIndex < MAX_NB_ECU; ++ecuIndex)
                {
                    if ((itid + 1) % 0x20 != 0) // avoid ITID for "requesting supported" e.g. 0xF820
                    {
                        if (((supportedITIDs[itid] >> ecuIndex) & 0x01) != 0)
                        {
                            nbError += displayRequestVehicleInformation_22_ITID_One(channel, protocol, ecuIndex, (obd_DID_t)(0xF800 + (itid + 1)));
                        }
                    }
                }
            }

            return nbError;
        }

        static int displayRequestVehicleInformation_22_ITID_One(cantp_handle channel, obd_msgprotocol protocol, byte ecuIndex, obd_DID_t ITID)
        {
            int nbError = 0;
            obd_msg msg_request = new obd_msg();
            obd_status status;

            obd_netaddrinfo nai;
            nai.protocol = protocol;
            nai.source_addr = (UInt16)uds_address.PUDS_ADDRESS_ISO_15765_4_ADDR_TEST_EQUIPMENT;
            nai.target_addr = (UInt16)(ecuIndex + 1);
            nai.target_type = obd_addressing.OBD_ADDRESSING_PHYSICAL;
            obd_DID_t[] itids = new obd_DID_t[1] { ITID };
            status = OBDonUDSApi.RequestVehicleInformation(channel, nai, out msg_request, itids, 1);
            if (OBDonUDSApi.StatusIsOk(status))
            {
                // Wait for responses
                obd_msg msg_response = new obd_msg();
                obd_msg msg_request_confirmation = new obd_msg();
                status = OBDonUDSApi.WaitForService(channel, ref msg_request, out msg_response, out msg_request_confirmation);
                if (OBDonUDSApi.StatusIsOk(status))
                {
                    // Parse response
                    obd_request_vehicle_information_response parsed_response;
                    status = OBDonUDSApi.ParseResponse_RequestVehicleInformation(ref msg_response, out parsed_response);
                    if (OBDonUDSApi.StatusIsOk(status))
                    {
                        // Print parsed response
                        Console.Write("-> Response from ECU #{0} : ", parsed_response.ecu_address);
                        if (parsed_response.nrc != 0)
                            Console.WriteLine("-> Negative response code 0x{0:X}", parsed_response.nrc);
                        else
                        {
                            Console.Write(" ITID 0x{0:X}, {1} bytes\n\t\t", parsed_response.data_identifier, parsed_response.nb_elements);
                            if (parsed_response.nb_elements != 0)
                            {
                                byte[] vals = new byte[parsed_response.nb_elements];
                                if (OBDonUDSApi.GetData(ref parsed_response, vals, (Int32)parsed_response.nb_elements))
                                    for (UInt32 j = 0; j < parsed_response.nb_elements; ++j)
                                    {
                                        Console.Write("{0:X2} ", vals[j]);
                                    }


                                // Example of detailed parsing : for ITID F802, F804, F80A which are like strings of maximum 20 ascii characters
                                // The response may contain fill bytes 00 that we print as blank spaces
                                switch (parsed_response.data_identifier)
                                {
                                    case 0xF802:
                                    case 0xF804:
                                    case 0xF80A:
                                        {
                                            String sResult = "";
                                            for (UInt32 j = 0; j < parsed_response.nb_elements; ++j)
                                                sResult += (char)vals[j];
                                            Console.Write("\n  -> {0}: '{1}'",
                                                parsed_response.data_identifier == 0xF802 ? "VIN" : parsed_response.data_identifier == 0xF804 ? "CALID" : "ECU",
                                                sResult);
                                        }
                                        break;

                                    default:
                                        // TODO: parse details of other elements
                                        break;
                                }
                            }

                            Console.WriteLine();
                        }
                    }
                    else
                    {
                        Console.Write("-> Parse ITID: ERROR (0x{0:X}): ", status); printStatusDetails(status);
                        nbError++;
                    }
                    // Free message
                    OBDonUDSApi.ParsedResponseFree(ref parsed_response);
                }
                else
                {
                    Console.Write("-> Wait ITID 0x{0:X} ECU #{1}: ERROR (0x{2:X}): ", ITID, ecuIndex + 1, status); printStatusDetails(status);
                    nbError++;
                }

                // Free messages
                OBDonUDSApi.MsgFree(ref msg_request_confirmation);
                OBDonUDSApi.MsgFree(ref msg_response);
            }
            else
            {
                Console.Write("-> Request ITIDs: ERROR (0x{0:X}): ", status); printStatusDetails(status);
                nbError++;
            }
            // Free messages
            OBDonUDSApi.MsgFree(ref msg_request);

            return nbError;

        }
        static int displayRequestEmissionRelatedDiagnosticTroubleCodesWithPermanentStatus_19_55(cantp_handle channel, obd_msgprotocol protocol)
        {
            int nbError = 0;
            obd_status status;

            Console.WriteLine("\nRequest Emission-Related Diagnostic Trouble Codes With Permanent Status physically for each ECU");

            byte[] availableECUs = new byte[1];
            OBDonUDSApi.GetValue(channel, obd_parameter.POBDONUDS_PARAMETER_AVAILABLE_ECUS, availableECUs, 1);


            for (byte ecuIndex = 0; ecuIndex < availableECUs[0]; ++ecuIndex)
            {
                obd_netaddrinfo nai;
                nai.protocol = protocol;
                nai.source_addr = (UInt16)uds_address.PUDS_ADDRESS_ISO_15765_4_ADDR_TEST_EQUIPMENT;
                nai.target_addr = (UInt16)(ecuIndex + 1);
                nai.target_type = obd_addressing.OBD_ADDRESSING_PHYSICAL;
                obd_msg msg_request = new obd_msg();
                status = OBDonUDSApi.RequestPermanentTroubleCodes(channel, nai, out msg_request, OBDonUDSApi.OBDONUDS_EMISSION_SYSTEM_GROUP);
                if (OBDonUDSApi.StatusIsOk(status))
                {
                    // Wait for response
                    obd_msg msg_response = new obd_msg();
                    obd_msg msg_request_confirmation = new obd_msg();
                    status = OBDonUDSApi.WaitForService(channel, ref msg_request, out msg_response, out msg_request_confirmation);
                    if (OBDonUDSApi.StatusIsOk(status))
                    {
                        // Parse response
                        obd_request_permanent_trouble_codes_response parsed_response;
                        status = OBDonUDSApi.ParseResponse_RequestPermanentTroubleCodes(ref msg_response, out parsed_response);
                        if (OBDonUDSApi.StatusIsOk(status))
                        {
                            // Print parsed response
                            Console.Write("-> Response from ECU #{0} : ", parsed_response.ecu_address);
                            if (parsed_response.nrc != 0)
                                Console.WriteLine("-> Negative response code 0x{0:X}", parsed_response.nrc);
                            else
                            {
                                Console.Write("-> Report type 0x{0:X2}, Functional group identifier 0x{1:X2}, DTC status availability mask 0x{2:X2}, DTC format identifier 0x{3:X2}",
                                    parsed_response.report_type, parsed_response.functional_group_identifier, parsed_response.DTC_status_availability_mask,
                                    parsed_response.DTC_format_identifier);
                                if (parsed_response.nb_elements != 0)
                                {
                                    obd_trouble_code[] vals = new obd_trouble_code[parsed_response.nb_elements];
                                    if (OBDonUDSApi.GetData(ref parsed_response, vals, (Int32)parsed_response.nb_elements))
                                        for (UInt32 ip = 0; ip < parsed_response.nb_elements; ++ip)
                                        {
                                            Console.Write("\n  -> Element {0} : ", ip + 1);
                                            printDTC(ref vals[ip]);
                                        }
                                }
                                Console.WriteLine();
                            }
                        }
                        else
                        {
                            Console.Write("-> Parse : ERROR (0x{0:X}): ", status); printStatusDetails(status);
                            nbError++;
                        }
                        // Free message
                        OBDonUDSApi.ParsedResponseFree(ref parsed_response);
                    }
                    else
                    {
                        Console.Write("-> Wait ECU #{0}: ERROR (0x{0:X}): ", ecuIndex + 1, status); printStatusDetails(status);
                        nbError++;
                    }

                    // Free messages
                    OBDonUDSApi.MsgFree(ref msg_request_confirmation);
                    OBDonUDSApi.MsgFree(ref msg_response);
                }
                else
                {
                    Console.Write("-> Request : ERROR (0x{0:X}): ", status); printStatusDetails(status);
                    nbError++;
                }
                // Free messages
                OBDonUDSApi.MsgFree(ref msg_request);
            }

            return nbError;
        }
        static void printTestObject(ref obd_test_data_object obj)
        {
            Console.Write("TID 0x{0:X2} Unit/Scaling 0x{1:X2} Test Value 0x{2:X2} 0x{3:X2} Min.Limit 0x{4:X2} 0x{5:X2} Max.Limit 0x{6:X2} 0x{7:X2} ",
                obj.test_identifier,
                obj.unit_and_scaling,
                obj.test_value[0], obj.test_value[1],
                obj.min_test_limit[0], obj.min_test_limit[1],
                obj.max_test_limit[0], obj.max_test_limit[1]);

            // Examples of detailed parsing
            switch (obj.unit_and_scaling)
            {
                case 0x10:
                    Console.Write("\n\tTest Value {0} ms, Min.Limit {1} ms, Max.Limit {2} ms",
                        makeUint16(obj.test_value[0], obj.test_value[1]) * 1.0,
                        makeUint16(obj.min_test_limit[0], obj.min_test_limit[1]) * 1.0,
                        makeUint16(obj.max_test_limit[0], obj.max_test_limit[1]) * 1.0);
                    break;
                case 0x0A:
                    Console.Write("\n\tTest Value {0} mV, Min.Limit {1} mV, Max.Limit {2} mV",
                        makeUint16(obj.test_value[0], obj.test_value[1]) * 0.1220703125,
                        makeUint16(obj.min_test_limit[0], obj.min_test_limit[1]) * 0.1220703125,
                        makeUint16(obj.max_test_limit[0], obj.max_test_limit[1]) * 0.1220703125);
                    break;
                case 0x24:
                    Console.Write("\n\tTest Value {0}, Min.Limit {1}, Max.Limit {2}",
                        makeUint16(obj.test_value[0], obj.test_value[1]),
                        makeUint16(obj.min_test_limit[0], obj.min_test_limit[1]),
                        makeUint16(obj.max_test_limit[0], obj.max_test_limit[1]));
                    break;
                default:
                    // TODO: parse details of other elements
                    break;
            }
        }

        static void printDTC(ref obd_trouble_code obj)
        {
            Console.Write("DTC identifier 0x{0:X}, Status of DTC 0x{1:X2}", obj.DTC_identifier, obj.status_of_dtc);
        }
        static UInt16 makeUint16(byte highByte, byte lowByte)
        {
            return (UInt16)(((highByte << 8) & 0xFF00) | lowByte);
        }
        static int displayRequestSupportedDTCExtendedRecord_19_1A(cantp_handle channel, obd_msgprotocol protocol, obd_DTC_t[,][] DTCarray)
        {
            int nbError = 0;

            Console.WriteLine("\nRequest Supported DTC Extended Record physically for each ECU");

            byte[] availableECUs = new byte[1];
            OBDonUDSApi.GetValue(channel, obd_parameter.POBDONUDS_PARAMETER_AVAILABLE_ECUS, availableECUs, 1);

            for (byte ecuIndex = 0; ecuIndex < availableECUs[0]; ++ecuIndex)
            {
                for (byte extendedDataRecordIndex = 0; extendedDataRecordIndex < MAX_NB_EXTENDED_DTC_RECORD; ++extendedDataRecordIndex)
                {
                    nbError += displayRequestSupportedDTCExtendedRecord_19_1A_one(channel, protocol, DTCarray, ecuIndex, extendedDataRecordIndex);
                }
            }

            return nbError;
        }
        static int displayRequestSupportedDTCExtendedRecord_19_1A_one(cantp_handle channel, obd_msgprotocol protocol, obd_DTC_t[,][] DTCarray, byte ecuIndex, byte extendedDataRecordIndex)
        {
            int nbError = 0;
            obd_status status;

            obd_msg msg_request = new obd_msg();
            obd_netaddrinfo nai;
            nai.protocol = protocol;
            nai.source_addr = (UInt16)uds_address.PUDS_ADDRESS_ISO_15765_4_ADDR_TEST_EQUIPMENT; ;
            nai.target_addr = (UInt16)(ecuIndex + 1);
            nai.target_type = obd_addressing.OBD_ADDRESSING_PHYSICAL;
            status = OBDonUDSApi.RequestSupportedDTCExtended(channel, nai, out msg_request, (byte)(0x90 + extendedDataRecordIndex));
            if (OBDonUDSApi.StatusIsOk(status))
            {
                // Wait for response
                obd_msg msg_response = new obd_msg();
                obd_msg msg_request_confirmation = new obd_msg();
                status = OBDonUDSApi.WaitForService(channel, ref msg_request, out msg_response, out msg_request_confirmation);
                if (OBDonUDSApi.StatusIsOk(status))
                {
                    // Parse response
                    obd_request_supported_dtc_extended_response parsed_response = new obd_request_supported_dtc_extended_response();
                    status = OBDonUDSApi.ParseResponse_RequestSupportedDTCExtended(ref msg_response, out parsed_response);
                    if (OBDonUDSApi.StatusIsOk(status))
                    {
                        // Print parsed response
                        Console.Write("-> Response from ECU #{0} : ", parsed_response.ecu_address);
                        if (parsed_response.nrc != 0)
                            Console.WriteLine("-> Negative response code 0x{0:X}", parsed_response.nrc);
                        else
                        {
                            Console.Write("-> Report type 0x{0:X2}, DTC status availability mask 0x{1:X2}, extended DTC data record number 0x{2:X2}",
                                parsed_response.report_type, parsed_response.DTC_status_availability_mask,
                                parsed_response.DTC_extended_data_record_number);
                            if (parsed_response.nb_elements != 0)
                            {
                                obd_trouble_code[] vals = new obd_trouble_code[parsed_response.nb_elements];
                                if (OBDonUDSApi.GetData(ref parsed_response, vals, (Int32)parsed_response.nb_elements))
                                    for (UInt32 ip = 0; ip < parsed_response.nb_elements; ++ip)
                                    {
                                        Console.Write("\n  -> Element {0} : ", ip + 1);
                                        printDTC(ref vals[ip]);
                                    }
                            }
                            Console.WriteLine("");
                            // Keep a list of DTC
                            nbError += KeepDTCextended(DTCarray, ecuIndex, extendedDataRecordIndex, ref parsed_response);
                        }
                    }
                    else
                    {
                        Console.Write("-> Parse : ERROR (0x{0:X}): ", status); printStatusDetails(status);
                        nbError++;
                    }
                    // Free message
                    OBDonUDSApi.ParsedResponseFree(ref parsed_response);
                }
                else
                {
                    Console.Write("-> Wait failed for ECU #{0} record 0x{1:X} (0x{2:X}): ", ecuIndex + 1, extendedDataRecordIndex + 0x90, status); printStatusDetails(status);
                    nbError++;
                }

                // Free messages
                OBDonUDSApi.MsgFree(ref msg_request_confirmation);
                OBDonUDSApi.MsgFree(ref msg_response);
            }
            else
            {
                Console.Write("-> Request : ERROR (0x{0:X}): ", status); printStatusDetails(status);
                nbError++;
            }
            // Free messages
            OBDonUDSApi.MsgFree(ref msg_request);

            return nbError;
        }

        static int KeepDTCextended(obd_DTC_t[,][] DTCarray, byte ecuIndex, byte extendedDataRecordIndex,
            ref obd_request_supported_dtc_extended_response parsed_response)
        {
            int nbError = 0;

            if (parsed_response.nb_elements != 0)
            {
                DTCarray[ecuIndex, extendedDataRecordIndex] = new obd_DTC_t[parsed_response.nb_elements];
                obd_trouble_code[] vals = new obd_trouble_code[parsed_response.nb_elements];
                if (OBDonUDSApi.GetData(ref parsed_response, vals, (Int32)parsed_response.nb_elements))
                    for (UInt32 i = 0; i < parsed_response.nb_elements; ++i)
                    {
                        DTCarray[ecuIndex, extendedDataRecordIndex][i] = vals[i].DTC_identifier;
                    }
            }

            return nbError;
        }

        static int displayRequestDTCExtendedDataRecord_19_06(cantp_handle channel, obd_msgprotocol protocol, obd_DTC_t[,][] DTCarray)
        {
            int nbError = 0;

            Console.WriteLine("\nRequest DTC Extended Data Record physically for each ECU");

            for (byte ecuIndex = 0; ecuIndex < MAX_NB_ECU; ++ecuIndex)
            {
                for (byte extendedDataRecordIndex = 0; extendedDataRecordIndex < MAX_NB_EXTENDED_DTC_RECORD; ++extendedDataRecordIndex)
                {
                    nbError += displayRequestDTCExtendedDataRecord_19_06_one(channel, protocol, DTCarray, ecuIndex, extendedDataRecordIndex);
                }
            }

            return nbError;
        }

        static int displayRequestDTCExtendedDataRecord_19_06_one(cantp_handle channel, obd_msgprotocol protocol, obd_DTC_t[,][] DTCarray,
            byte ecuIndex, byte extendedDataRecordIndex)
        {
            int nbError = 0;
            obd_status status;

            obd_netaddrinfo nai;
            nai.protocol = protocol;
            nai.source_addr = (UInt16)uds_address.PUDS_ADDRESS_ISO_15765_4_ADDR_TEST_EQUIPMENT; ;
            nai.target_addr = (UInt16)(ecuIndex + 1);
            nai.target_type = obd_addressing.OBD_ADDRESSING_PHYSICAL;

            if (DTCarray[ecuIndex, extendedDataRecordIndex] != null)
            {
                for (int indexDTC = 0; indexDTC < DTCarray[ecuIndex, extendedDataRecordIndex].Length; ++indexDTC)
                {
                    obd_msg msg_request = new obd_msg();
                    status = OBDonUDSApi.RequestDTCExtended(channel, nai, out msg_request, DTCarray[ecuIndex, extendedDataRecordIndex][indexDTC], (byte)(0x90 + extendedDataRecordIndex));
                    if (OBDonUDSApi.StatusIsOk(status))
                    {
                        // Wait for response
                        obd_msg msg_response = new obd_msg();
                        obd_msg msg_request_confirmation = new obd_msg();
                        status = OBDonUDSApi.WaitForService(channel, ref msg_request, out msg_response, out msg_request_confirmation);
                        if (OBDonUDSApi.StatusIsOk(status))
                        {
                            // Parse response
                            obd_request_dtc_extended_response parsed_response = new obd_request_dtc_extended_response();
                            status = OBDonUDSApi.ParseResponse_RequestDTCExtended(ref msg_response, out parsed_response);
                            if (OBDonUDSApi.StatusIsOk(status))
                            {
                                // Print parsed response
                                Console.Write("-> Response from ECU #{0} : ", parsed_response.ecu_address);
                                if (parsed_response.nrc != 0)
                                    Console.WriteLine("-> Negative response code 0x{0:X}", parsed_response.nrc);
                                else
                                {
                                    Console.WriteLine("-> Report type 0x{0:X2}, DTC identifier 0x{1:X}, DTC status 0x{2:X2}, DTC extended data record number 0x{3:X2}, {4} bytes",
                                        parsed_response.report_type, parsed_response.DTC_identifier, parsed_response.status_of_dtc,
                                        parsed_response.DTC_extended_data_record_number, parsed_response.nb_elements);
                                    Console.Write("\t\t");
                                    if (parsed_response.nb_elements != 0)
                                    {
                                        byte[] vals = new byte[parsed_response.nb_elements];
                                        if (OBDonUDSApi.GetData(ref parsed_response, vals, (Int32)parsed_response.nb_elements))
                                            for (UInt32 ip = 0; ip < parsed_response.nb_elements; ++ip)
                                            {
                                                Console.Write("{0:X2} ", vals[ip]);
                                            }
                                        Console.WriteLine("");
                                        // Example of detailed parsing : for extended data record number 0x92 "DTC based test result"
                                        if (parsed_response.DTC_extended_data_record_number == 0x92)
                                        {
                                            int j = 0;
                                            byte numberOfTestResults = vals[j++];
                                            if (parsed_response.nb_elements >= 1U + (8U * numberOfTestResults))
                                            {
                                                for (byte indexTest = 0; indexTest < numberOfTestResults; ++indexTest)
                                                {
                                                    UInt16 unit_and_scaling = makeUint16(vals[j], vals[j + 1]);
                                                    // try to use similar function - cannot work if unit_and_scaling first byte is not 0 - TODO: parse details in other case
                                                    if (vals[j] == 0x00)
                                                    {
                                                        obd_test_data_object obj = new obd_test_data_object();
                                                        obj.unit_and_scaling = (byte)unit_and_scaling;
                                                        j += 2;
                                                        obj.test_value = new byte[2];
                                                        obj.test_value[0] = vals[j++];
                                                        obj.test_value[1] = vals[j++];
                                                        obj.min_test_limit = new byte[2];
                                                        obj.min_test_limit[0] = vals[j++];
                                                        obj.min_test_limit[1] = vals[j++];
                                                        obj.max_test_limit = new byte[2];
                                                        obj.max_test_limit[0] = vals[j++];
                                                        obj.max_test_limit[1] = vals[j++];
                                                        Console.Write("  -> ");
                                                        printTestObject(ref obj);
                                                        Console.WriteLine("");
                                                    }
                                                }
                                            }
                                        }
                                        // TODO: parse details of other elements
                                    }
                                    Console.WriteLine("");
                                }
                            }
                            else
                            {
                                Console.Write("-> Parse : ERROR (0x{0:X}): ", status); printStatusDetails(status);
                                nbError++;
                            }
                            // Free message
                            OBDonUDSApi.ParsedResponseFree(ref parsed_response);
                        }
                        else
                        {
                            Console.Write("-> Wait ECU #{0} record 0x{1:X} DTC 0x{2:X}: ERROR (0x{3:X}): ", ecuIndex + 1, extendedDataRecordIndex + 0x90,
                                DTCarray[ecuIndex, extendedDataRecordIndex][indexDTC], status);
                            printStatusDetails(status);
                            nbError++;
                        }

                        // Free messages
                        OBDonUDSApi.MsgFree(ref msg_request_confirmation);
                        OBDonUDSApi.MsgFree(ref msg_response);
                    }
                    else
                    {
                        Console.Write("-> Request : ERROR (0x{0:X}): ", status); printStatusDetails(status);
                        nbError++;
                    }
                    // Free messages
                    OBDonUDSApi.MsgFree(ref msg_request);
                }
            }

            return nbError;
        }
        static int displayRequestDTCsForAReadinessGroup_19_56(cantp_handle channel, obd_msgprotocol protocol)
        {
            int nbError = 0;

            Console.WriteLine("\nRequest DTCs For first Readiness Groups physically for each ECU");

            byte[] availableECUs = new byte[1];
            OBDonUDSApi.GetValue(channel, obd_parameter.POBDONUDS_PARAMETER_AVAILABLE_ECUS, availableECUs, 1);

            for (byte ecuIndex = 0; ecuIndex < availableECUs[0]; ++ecuIndex)
            {
                for (byte readinessGroup = 0x00; readinessGroup <= 0x14; ++readinessGroup)
                {
                    nbError += displayRequestDTCsForAReadinessGroup_19_56_one(channel, protocol, ecuIndex, readinessGroup);
                }
            }

            return nbError;
        }

        static int displayRequestDTCsForAReadinessGroup_19_56_one(cantp_handle channel, obd_msgprotocol protocol, byte ecuIndex, byte readinessGroup)
        {
            int nbError = 0;

            obd_status status;
            obd_msg msg_request = new obd_msg();
            obd_netaddrinfo nai;
            nai.protocol = protocol;
            nai.source_addr = (UInt16)uds_address.PUDS_ADDRESS_ISO_15765_4_ADDR_TEST_EQUIPMENT; ;
            nai.target_addr = (UInt16)(ecuIndex + 1);
            nai.target_type = obd_addressing.OBD_ADDRESSING_PHYSICAL;

            status = OBDonUDSApi.RequestDTCForAReadinessGroup(channel, nai, out msg_request, OBDonUDSApi.OBDONUDS_EMISSION_SYSTEM_GROUP, readinessGroup);
            if (OBDonUDSApi.StatusIsOk(status))
            {
                // Wait for responses
                obd_msg msg_response = new obd_msg();
                obd_msg msg_request_confirmation = new obd_msg();
                status = OBDonUDSApi.WaitForService(channel, ref msg_request, out msg_response, out msg_request_confirmation);
                if (OBDonUDSApi.StatusIsOk(status))
                {
                    // Parse response
                    obd_request_dtc_for_a_readiness_group_response parsed_response = new obd_request_dtc_for_a_readiness_group_response();
                    status = OBDonUDSApi.ParseResponse_RequestDTCForAReadinessGroup(ref msg_response, out parsed_response);
                    if (OBDonUDSApi.StatusIsOk(status))
                    {
                        // Print parsed response
                        Console.Write("-> Response from ECU #{0} : ", parsed_response.ecu_address);
                        if (parsed_response.nrc != 0)
                            Console.WriteLine("-> Negative response code 0x{0:X}", parsed_response.nrc);
                        else
                        {
                            Console.Write("-> Report type 0x{0:X2}, Functional group identifier 0x{1:X2}, DTC status availability mask 0x{2:X2}, DTC format identifier 0x{3:X2}, readiness group identifier 0x{4:X2}",
                                parsed_response.report_type, parsed_response.functional_group_identifier, parsed_response.DTC_status_availability_mask,
                                parsed_response.DTC_format_identifier, parsed_response.readiness_group_identifier);
                            if (parsed_response.nb_elements != 0)
                            {
                                obd_trouble_code[] vals = new obd_trouble_code[parsed_response.nb_elements];
                                if (OBDonUDSApi.GetData(ref parsed_response, vals, (Int32)parsed_response.nb_elements))
                                    for (UInt32 ip = 0; ip < parsed_response.nb_elements; ++ip)
                                    {
                                        Console.Write("\n  -> Element {0} : ", ip + 1);
                                        printDTC(ref vals[ip]);
                                    }
                            }
                            Console.WriteLine("");
                        }
                    }
                    else
                    {
                        Console.Write("-> Parse : ERROR (0x{0:X}): ", status); printStatusDetails(status);
                        nbError++;
                    }
                    // Free message
                    OBDonUDSApi.ParsedResponseFree(ref parsed_response);
                }
                else
                {
                    Console.Write("-> Wait failed for ECU #{0} Group 0x{1:X2} (0x{2:X}): ", ecuIndex + 1, readinessGroup, status); printStatusDetails(status);
                    nbError++;
                }

                // Free messages
                OBDonUDSApi.MsgFree(ref msg_request_confirmation);
                OBDonUDSApi.MsgFree(ref msg_response);
            }
            else
            {
                Console.Write("-> Request : ERROR (0x{0:X}): ", status); printStatusDetails(status);
                nbError++;
            }
            // Free messages
            OBDonUDSApi.MsgFree(ref msg_request);

            return nbError;
        }
    }
}
