//  PCAN-OBDonUDS.cs
//
//  ~~~~~~~~~~~~
//
//  PCAN-OBDonUDS API
//
//  ~~~~~~~~~~~~
//
//  ------------------------------------------------------------------
//	Last changed by:	$Author: $
//  Last changed date:	$Date:   $
//
//  Language: C#
//  ------------------------------------------------------------------
//
//  Copyright (C) 2023  PEAK-System Technik GmbH, Darmstadt
//  more Info at http://www.peak-system.com
//

// To accept unsafe functions, define UNSAFE
//#define UNSAFE

using System;
using System.Text;
using System.Runtime.InteropServices;

using Peak.Can.Uds;
using Peak.Can.IsoTp;

namespace Peak.Can.OBDonUDS
{
    #region Types definitions   
    // DID value on 2 bytes
    using obd_DID_t = UInt16;
    // DTC value on 4 bytes 
    using obd_DTC_t = UInt32;
    // RID value on 2 bytes
    using obd_RID_t = UInt16;
    #endregion

    #region Enums definition for OBDonUDS API

    /// <summary>
    /// Address of ECUs
    /// </summary>
    public enum obd_ecu : UInt16
    {
        POBD_ECU_1 = uds_address.PUDS_ADDRESS_ISO_15765_4_ADDR_ECU_1,  // ECU #1
        POBD_ECU_2 = uds_address.PUDS_ADDRESS_ISO_15765_4_ADDR_ECU_2,  // ECU #2
        POBD_ECU_3 = uds_address.PUDS_ADDRESS_ISO_15765_4_ADDR_ECU_3,  // ECU #3
        POBD_ECU_4 = uds_address.PUDS_ADDRESS_ISO_15765_4_ADDR_ECU_4,  // ECU #4
        POBD_ECU_5 = uds_address.PUDS_ADDRESS_ISO_15765_4_ADDR_ECU_5,  // ECU #5
        POBD_ECU_6 = uds_address.PUDS_ADDRESS_ISO_15765_4_ADDR_ECU_6,  // ECU #6
        POBD_ECU_7 = uds_address.PUDS_ADDRESS_ISO_15765_4_ADDR_ECU_7,  // ECU #7
        POBD_ECU_8 = uds_address.PUDS_ADDRESS_ISO_15765_4_ADDR_ECU_8,  // ECU #8
    }

    /// <summary>
    /// Services IDs
    /// </summary>
    public enum obd_service : byte
    {
        POBD_SERVICE_14 = 0x14,
        POBD_SERVICE_19 = 0x19,
        POBD_SERVICE_22 = 0x22,
        POBD_SERVICE_31 = 0x31,
    }


    /// <summary>
    /// Sub-functions of services
    /// </summary>
    public enum obd_sub_function : byte
    {
        POBD_SUB_FUNCTION_04 = 0x04,
        POBD_SUB_FUNCTION_42 = 0x42,
        POBD_SUB_FUNCTION_55 = 0x55,
        POBD_SUB_FUNCTION_1A = 0x1A,
        POBD_SUB_FUNCTION_06 = 0x06,
        POBD_SUB_FUNCTION_56 = 0x56,
    }

    /// <summary>
    /// POBDonUDS error codes (used in obd_status)
    /// </summary>
    public enum obd_errstatus : byte
    {
        POBDONUDS_ERRSTATUS_UNSUPPORTED_ECUS = 1,
        POBDONUDS_ERRSTATUS_INVALID_REQUEST_ADDRESSING,
        POBDONUDS_ERRSTATUS_INVALID_SI,
        POBDONUDS_ERRSTATUS_INVALID_DATA_LENGTH,
        POBDONUDS_ERRSTATUS_NETWORK_ERROR,
        POBDONUDS_ERRSTATUS_NOT_PARSABLE,
        POBDONUDS_CAUTION_NRC_NOT_NULL,
    }

    /// <summary>
    /// POBDonUDS status codes.
    ///
    /// Bits information:
    ///   32|  28|  24|  20|  16|  12|   8|   4|   0|
    ///     |    |    |    |    |    |    |    |    |
    ///      0000 0000 0000 0000 0000 0000 0000 0000
    ///     |    |    |    |    |         [0000 0000] => PCAN-ISO-TP API errors
    ///     |    |    |    |    |  [0 0000]           => CAN Bus status
    ///     |    |    |    | [00 000]                 => Networking message status
    ///     |    |    [0000 00]                       => PCAN-ISO-TP API extra information
    ///     |  [0 0000]                               => API Status
    ///     | [0]                                     => UDS Status flag
    ///     |[0]                                      => OBDonUDS status flag
    ///     [0]                                       => PCANBasic error flag
    /// </summary>
    public enum obd_status : UInt32
    {
        /// <summary>
        /// No error
        /// </summary>
        POBDONUDS_STATUS_OK = uds_status.PUDS_STATUS_OK,
        /// <summary>
        /// Not Initialized.
        /// </summary>
        POBDONUDS_STATUS_NOT_INITIALIZED = uds_status.PUDS_STATUS_NOT_INITIALIZED,
        /// <summary>
        /// Already Initialized.
        /// </summary>
        POBDONUDS_STATUS_ALREADY_INITIALIZED = uds_status.PUDS_STATUS_ALREADY_INITIALIZED,
        /// <summary>
        /// Could not obtain memory.
        /// </summary>
        POBDONUDS_STATUS_NO_MEMORY = uds_status.PUDS_STATUS_NO_MEMORY,
        /// <summary>
        /// Wrong message parameters.
        /// </summary>
        POBDONUDS_STATUS_PARAM_INVALID_TYPE = uds_status.PUDS_STATUS_PARAM_INVALID_TYPE,
        /// <summary>
        ///  Wrong message parameters.
        /// </summary>
        POBDONUDS_STATUS_PARAM_INVALID_VALUE = uds_status.PUDS_STATUS_PARAM_INVALID_VALUE,
        /// <summary>
        /// Unknown/generic error.
        /// </summary>
        POBDONUDS_STATUS_UNKNOWN = uds_status.PUDS_STATUS_UNKNOWN,
        /// <summary>
        /// Invalid cantp_handle.	
        /// </summary>
        POBDONUDS_STATUS_HANDLE_INVALID = uds_status.PUDS_STATUS_HANDLE_INVALID,

        // OBDonUDS status

        /// <summary>
        /// OBDonUDS error flag.
        /// </summary>
        POBDONUDS_STATUS_FLAG_OBDONUDS_ERROR = 0x40 << uds_status_offset.PCANTP_STATUS_OFFSET_UDS,
        /// <summary>
        /// Filter OBDonUDS error.
        /// </summary>
        POBDONUDS_STATUS_MASK_OBDONUDS_ERROR = 0x5F << uds_status_offset.PCANTP_STATUS_OFFSET_UDS,
        /// <summary>
        /// No OBDonUDS ECU found
        /// </summary>
        POBDONUDS_STATUS_UNSUPPORTED_ECUS = POBDONUDS_STATUS_FLAG_OBDONUDS_ERROR | (obd_errstatus.POBDONUDS_ERRSTATUS_UNSUPPORTED_ECUS << uds_status_offset.PCANTP_STATUS_OFFSET_UDS),
        /// <summary>
        /// Invalid request addressing mode 
        /// </summary>
        POBDONUDS_STATUS_INVALID_REQUEST_ADDRESSING = POBDONUDS_STATUS_FLAG_OBDONUDS_ERROR | (obd_errstatus.POBDONUDS_ERRSTATUS_INVALID_REQUEST_ADDRESSING << uds_status_offset.PCANTP_STATUS_OFFSET_UDS),
        /// <summary>
        /// Invalid Service ID found while parsing (null pointer or bad value)
        /// </summary>
        POBDONUDS_STATUS_INVALID_SI = POBDONUDS_STATUS_FLAG_OBDONUDS_ERROR | (obd_errstatus.POBDONUDS_ERRSTATUS_INVALID_SI << uds_status_offset.PCANTP_STATUS_OFFSET_UDS),
        /// <summary>
        /// Invalid length found while parsing
        /// </summary>
        POBDONUDS_STATUS_INVALID_DATA_LENGTH = POBDONUDS_STATUS_FLAG_OBDONUDS_ERROR | (obd_errstatus.POBDONUDS_ERRSTATUS_INVALID_DATA_LENGTH << uds_status_offset.PCANTP_STATUS_OFFSET_UDS),
        /// <summary>
        /// Invalid Network result found while parsing (null pointer or value indicating an error)
        /// </summary>
        POBDONUDS_STATUS_NETWORK_ERROR = POBDONUDS_STATUS_FLAG_OBDONUDS_ERROR | (obd_errstatus.POBDONUDS_ERRSTATUS_NETWORK_ERROR << uds_status_offset.PCANTP_STATUS_OFFSET_UDS),
        /// <summary>
        /// The requested DID/MID/RID/ITID corresponds to a request of type "get supported identifiers", the response is not parsable
        /// </summary>
        POBDONUDS_STATUS_NOT_PARSABLE = POBDONUDS_STATUS_FLAG_OBDONUDS_ERROR | (obd_errstatus.POBDONUDS_ERRSTATUS_NOT_PARSABLE << uds_status_offset.PCANTP_STATUS_OFFSET_UDS),
        /// <summary>
        /// NRC not null found while parsing (Caution indicating that the response is a negative response)
        /// </summary>
        POBDONUDS_STATUS_CAUTION_NRC_NOT_NULL = POBDONUDS_STATUS_FLAG_OBDONUDS_ERROR | (obd_errstatus.POBDONUDS_CAUTION_NRC_NOT_NULL << uds_status_offset.PCANTP_STATUS_OFFSET_UDS),
    }

    /// <summary>
    /// POBDonUDS parameters to be read or set
    /// </summary>
    public enum obd_parameter : UInt32
    {
        /// <summary>
        ///  (R) char(byte) array describing the api version as a null-terminated string
        /// </summary>
        POBDONUDS_PARAMETER_API_VERSION = 0x301,
        /// <summary>
        /// (R) 1 byte data describing the number of OBDonUDS ECUs detected
        /// </summary>
        POBDONUDS_PARAMETER_AVAILABLE_ECUS = 0x302,
        /// <summary>
        /// (R) 256 bytes describing supported DIDs F4XX for Service 22: Request Current Powertrain Diagnostic Data
        /// </summary>
        POBDONUDS_PARAMETER_SUPPORTMASK_DIDS_F4XX = 0x303,
        /// <summary>
        /// (R) 256 bytes describing supported DIDs F5XX for Service 22: Request Current Powertrain Diagnostic Data
        /// </summary>
        POBDONUDS_PARAMETER_SUPPORTMASK_DIDS_F5XX = 0x304,
        /// <summary>
        /// (R) 256 bytes describing supported DIDs F7XX for Service 22: Request Current Powertrain Diagnostic Data
        /// </summary>
        POBDONUDS_PARAMETER_SUPPORTMASK_DIDS_F7XX = 0x305,
        /// <summary>
        /// (R) 256 bytes describing supported MIDs F6XX (DIDs) for Service 22: Request On-board Monitoring Test Results for Specific Monitored Systems
        /// </summary>
        POBDONUDS_PARAMETER_SUPPORTMASK_MIDS = 0x306,
        /// <summary>
        /// (R) 256 bytes describing supported RIDs or TIDs E0XX for Service 31: Request Control of On-Board System, Test, or Component
        /// </summary>
        POBDONUDS_PARAMETER_SUPPORTMASK_RIDS_E0XX = 0x307,
        /// <summary>
        /// (R) 256 bytes describing supported RIDs or TIDs E1XX for Service 31: Request Control of On-Board System, Test, or Component
        /// </summary>
        POBDONUDS_PARAMETER_SUPPORTMASK_RIDS_E1XX = 0x308,
        /// <summary>
        /// (R) 256 bytes describing supported InfoTypes(ITIDs) F8XX for Service 22: Request Vehicle Information
        /// </summary>
        POBDONUDS_PARAMETER_SUPPORTMASK_ITIDS = 0x309,
        /// <summary>
        /// (R/W) 1 byte data describing the debug mode  (use POBDONUDS_DEBUG_LVL_ values)
        /// </summary>
        POBDONUDS_PARAMETER_DEBUG = 0x30A,
        /// <summary>
        /// (R) obd_baudrate data describing baudrate value(see values OBD_BAUDRATE_xxx)
        /// </summary>
        POBDONUDS_PARAMETER_BAUDRATE = 0x30B,
        /// <summary>
        /// (R) obd_msprotocol data describing CAN identifier length (see values OBD_MSGPROTOCOL_xxx)
        /// </summary>
        POBDONUDS_PARAMETER_CAN_ID_LENGTH = 0x30C
    }


    /// <summary>
    /// POBDonUDS baud rates
    /// </summary>
    public enum obd_baudrate : UInt32
    {
        /// <summary>
        /// Non legislated-OBD baudrate
        /// </summary>
        OBD_BAUDRATE_NON_LEGISLATED = 0,
        /// <summary>
        /// Baudrate 250 kBit/s
        /// </summary>
        OBD_BAUDRATE_250K = cantp_baudrate.PCANTP_BAUDRATE_250K,
        /// <summary>
        /// Baudrate 500 kBit/s
        /// </summary>
        OBD_BAUDRATE_500K = cantp_baudrate.PCANTP_BAUDRATE_500K,
        /// <summary>
        /// Autodetect baudrate mechanism
        /// </summary>
        OBD_BAUDRATE_AUTODETECT = 0xFFFFFFFFU
    }


    /// <summary>
    /// POBDonUDS message protocols
    /// </summary>
    public enum obd_msgprotocol : UInt32
    {
        /// <summary>
        /// Invalid/undefined value
        /// </summary>
        OBD_MSGPROTOCOL_UNKNOWN = cantp_isotp_format.PCANTP_ISOTP_FORMAT_NONE,
        /// <summary>
        /// 11bit CAN Identifier
        /// </summary>
        OBD_MSGPROTOCOL_11BIT = cantp_isotp_format.PCANTP_ISOTP_FORMAT_NORMAL,
        /// <summary>
        /// 29bit CAN Identifier
        /// </summary>
        OBD_MSGPROTOCOL_29BIT = cantp_isotp_format.PCANTP_ISOTP_FORMAT_FIXED_NORMAL,
    }


    /// <summary>
    /// POBDonUDS addressing modes
    /// </summary>
    public enum obd_addressing : UInt32
    {
        OBD_ADDRESSING_UNKNOWN = cantp_isotp_addressing.PCANTP_ISOTP_ADDRESSING_UNKNOWN,
        OBD_ADDRESSING_PHYSICAL = cantp_isotp_addressing.PCANTP_ISOTP_ADDRESSING_PHYSICAL,
        OBD_ADDRESSING_FUNCTIONAL = cantp_isotp_addressing.PCANTP_ISOTP_ADDRESSING_FUNCTIONAL
    }


    /// <summary>
    /// Reserved enumeration of internal parsed responses types
    /// </summary>
    public enum obd_datatype : byte
    {
        OBD_DATATYPE_UNKNOWN = 0,
        OBD_DATATYPE_REQUEST_CURRENT_DATA_RESPONSE,
        OBD_DATATYPE_REQUEST_FREEZE_FRAME_DATA_RESPONSE,
        OBD_DATATYPE_REQUEST_CONTROL_OPERATION_RESPONSE,
        OBD_DATATYPE_REQUEST_VEHICLE_INFORMATION_RESPONSE,
        OBD_DATATYPE_REQUEST_TROUBLECODES_RESPONSE,
        OBD_DATATYPE_REQUEST_PERMANENT_TROUBLECODES_RESPONSE,
        OBD_DATATYPE_REQUEST_READINESS_GROUP_TROUBLECODES_RESPONSE,
        OBD_DATATYPE_REQUEST_SUPPORTED_DTC_EXTENDED_RESPONSE,
        OBD_DATATYPE_REQUEST_CLEAR_TROUBLE_CODES_RESPONSE,
        OBD_DATATYPE_REQUEST_TEST_RESULTS_RESPONSE,
        OBD_DATATYPE_REQUEST_DTC_EXTENDED_RESPONSE
    }

    #endregion

    #region Structures
    ////////////////////////////////////////////////////////////
    // Message definitions
    ////////////////////////////////////////////////////////////

    /// <summary>
    /// Represents a OBDonUDS Network Addressing Information
    /// </summary>
    [StructLayout(LayoutKind.Sequential, Pack = 8)]
    public struct obd_netaddrinfo
    {
        /// <summary>
        /// Communication protocol
        /// </summary>
        [MarshalAs(UnmanagedType.U4)]
        public obd_msgprotocol protocol;
        /// <summary>
        /// ISO-TP target type (physical or functional)
        /// </summary>
        [MarshalAs(UnmanagedType.U4)]
        public obd_addressing target_type;
        /// <summary>
        /// Source address (by default should be PUDS_ISO_15765_4_ADDR_TEST_EQUIPMENT for requests)
        /// </summary>
        public UInt16 source_addr;
        /// <summary>
        /// Target address (see POBD_ECUxx)
        /// </summary>
        public UInt16 target_addr;

        public obd_netaddrinfo(obd_msgprotocol protocol, obd_addressing adressing, UInt16 source, UInt16 target)
        {
            this.protocol = protocol;
            this.target_type = adressing;
            this.source_addr = source;
            this.target_addr = target;
        }
    }

    /// <summary>
    /// Provides accessors to the corresponding data in the cantp_msg
    /// </summary>
    [StructLayout(LayoutKind.Sequential, Pack = 8)]
    public struct obd_msgaccess
    {
        /// <summary>
        /// Pointer to the Service ID in message's data.
        /// Use it in safe mode with Marshal.ReadByte or unsafe mode with IntPtr.toPointer.
        /// See special C# functions
        /// </summary>
        public IntPtr service_id;
        /// <summary>
        /// Pointer to the Negative Response Code (see uds_nrc enumeration) in message's data (NULL on positive response).
        /// Use it in safe mode with Marshal.ReadByte or unsafe mode with IntPtr.toPointer.
        /// See special C# functions
        /// </summary>
        public IntPtr nrc;
        /// <summary>
        /// Pointer to the ISOTP network result.
        /// Use it in safe mode with Marshal.ReadByte or unsafe mode with IntPtr.toPointer.
        /// See special C# functions
        /// </summary>
        public IntPtr network_result;
    }

    /// <summary>
    /// The OBDonUDS message
    /// </summary>
    [StructLayout(LayoutKind.Sequential, Pack = 8)]
    public struct obd_msg
    {
        public obd_netaddrinfo nai;
        public obd_msgaccess links;
        public uds_msg msg;
    }

    /// <summary>
    /// All services :
    /// Internal reserved structure for parsed responses
    /// </summary>
    [StructLayout(LayoutKind.Sequential, Pack = 8)]
    public struct obd_response_generic
    {
        [MarshalAs(UnmanagedType.U1)]
        public obd_datatype reserved_object_type;
        public UInt32 reserved_object_size;
        public UInt16 ecu_address;
        public byte nrc;
    }

    // service 22-DID

    /// <summary>
    /// Part of a parsed response to service 22-DID
    /// </summary>
    [StructLayout(LayoutKind.Sequential, Pack = 8)]
    public struct obd_did_object
    {
        /// <summary>
        /// DID
        /// </summary>
        [MarshalAs(UnmanagedType.U2)]
        public obd_DID_t data_identifier;
        /// <summary>
        /// size (number of bytes) of data
        /// </summary>
        public UInt32 size;
        /// <summary>
        /// array of bytes representing data dedicated to data_identifier
        /// Use it in safe mode with Marshal.Copy or unsafe mode with IntPtr.toPointer.
        /// See special C# functions
        /// </summary>
        public IntPtr data;
    }

    /// <summary>
    /// Parsed response to service 22-DID
    /// </summary>
    [StructLayout(LayoutKind.Sequential, Pack = 8)]
    public struct obd_request_current_data_response
    {
        /// <summary>
        /// reserved
        /// </summary>
        [MarshalAs(UnmanagedType.U1)]
        public obd_datatype reserved_object_type;
        /// <summary>
        /// reserved
        /// </summary>
        public UInt32 reserved_object_size;
        /// <summary>
        ///  ECU address
        /// </summary>
        public UInt16 ecu_address;
        /// <summary>
        /// NRC (0 if no NRC)
        /// </summary>
        public byte nrc;
        /// <summary>
        /// number of objects in elements
        /// </summary>
        public UInt32 nb_elements;
        /// <summary>
        /// array of obd_did_object objects
        /// Use it in safe mode with Marshal.PtrToStructure or unsafe mode with IntPtr.toPointer.
        /// See special C# functions
        /// </summary>
        public IntPtr elements;
    }

    // service 19-04

    /// <summary>
    /// Parsed response to service service 19-04
    /// </summary>
    [StructLayout(LayoutKind.Sequential, Pack = 8)]
    public struct obd_request_freeze_frame_data_response
    {
        /// <summary>
        /// reserved
        /// </summary>
        [MarshalAs(UnmanagedType.U1)]
        public obd_datatype reserved_object_type;
        /// <summary>
        /// reserved
        /// </summary>
        public UInt32 reserved_object_size;
        /// <summary>
        ///  ECU address
        /// </summary>
        public UInt16 ecu_address;
        /// <summary>
        /// NRC (0 if no NRC)
        /// </summary>
        public byte nrc;
        /// <summary>
        /// reportType
        /// </summary>
        public byte report_type;
        /// <summary>
        /// DTC
        /// </summary>
        [MarshalAs(UnmanagedType.U4)]
        public obd_DTC_t dtc_number;
        /// <summary>
        /// statusOfDTC
        /// </summary>
        public byte status_of_dtc;
        /// <summary>
        /// DTCSnapshotRecordNumber
        /// </summary>
        public byte record_number;
        /// <summary>
        /// DTCSnapshotRecordNumberOfIdentifiers, number of objects in identifiers
        /// </summary>
        public byte nb_identifiers;
        /// <summary>
        /// array of obd_did_object objects (DTCSnapshotRecordDataRecord)
        /// Use it in safe mode with Marshal.PtrToStructure or unsafe mode with IntPtr.toPointer.
        /// See special C# functions
        /// </summary>
        public IntPtr identifiers;
    }

    // service 19-42

    /// <summary>
    /// Part of a parsed response to service 19-42
    /// </summary>
    [StructLayout(LayoutKind.Sequential, Pack = 8)]
    public struct obd_severity_trouble_code
    {
        /// <summary>
        /// DTCSeverity
        /// </summary>
        public byte DTC_severity;
        /// <summary>
        /// DTC
        /// </summary>
        [MarshalAs(UnmanagedType.U4)]
        public obd_DTC_t DTC_identifier;
        /// <summary>
        /// statusOfDTC
        /// </summary>
        public byte status_of_dtc;
    }

    /// <summary>
    /// Parsed response to service 19-42
    /// </summary>
    [StructLayout(LayoutKind.Sequential, Pack = 8)]
    public struct obd_request_trouble_codes_response
    {
        /// <summary>
        /// reserved
        /// </summary>
        [MarshalAs(UnmanagedType.U1)]
        public obd_datatype reserved_object_type;
        /// <summary>
        /// reserved
        /// </summary>
        public UInt32 reserved_object_size;
        /// <summary>
        ///  ECU address
        /// </summary>
        public UInt16 ecu_address;
        /// <summary>
        /// NRC (0 if no NRC)
        /// </summary>
        public byte nrc;
        /// <summary>
        /// DTCReportType
        /// </summary>
        public byte report_type;
        /// <summary>
        /// FunctionalGroupIdentifier
        /// </summary>
        public byte functional_group_identifier;
        /// <summary>
        ///  DTCStatusAvailabilityMask
        /// </summary>
        public byte DTC_status_availability_mask;
        /// <summary>
        /// DTCSeverityAvailabilityMask
        /// </summary>
        public byte DTC_severity_mask;
        /// <summary>
        /// DTCFormatIdentifier
        /// </summary>
        public byte DTC_format_identifier;
        /// <summary>
        /// Number of objects in elements
        /// </summary>
        public UInt32 nb_elements;
        /// <summary>
        ///  Array of obd_severity_trouble_code objects (DTCAndSeverityRecord)
        /// Use it in safe mode with Marshal.PtrToStructure or unsafe mode with IntPtr.toPointer.
        /// See special C# functions
        /// </summary>
        public IntPtr elements;
    }

    // service 14

    /// <summary>
    /// Parsed response to service service 14
    /// </summary>
    [StructLayout(LayoutKind.Sequential, Pack = 8)]
    public struct obd_request_clear_trouble_codes_response
    {
        /// <summary>
        /// reserved
        /// </summary>
        [MarshalAs(UnmanagedType.U1)]
        public obd_datatype reserved_object_type;
        /// <summary>
        /// reserved
        /// </summary>
        public UInt32 reserved_object_size;
        /// <summary>
        ///  ECU address
        /// </summary>
        public UInt16 ecu_address;
        /// <summary>
        /// NRC (0 if no NRC)
        /// </summary>
        public byte nrc;
    }

    // service 22-MID

    /// <summary>
    /// Part of a parsed response to service 22-MID
    /// </summary> 
    [StructLayout(LayoutKind.Sequential, Pack = 8)]
    public struct obd_test_data_object
    {
        /// <summary>
        ///  TID
        /// </summary>
        public byte test_identifier;
        /// <summary>
        /// Unit and scaling ID
        /// </summary>
        public byte unit_and_scaling;
        /// <summary>
        /// Test Value (2 bytes)
        /// </summary>
        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 2)]
        public byte[] test_value;
        /// <summary>
        /// Min. Test Limit (2 bytes)
        /// </summary>
        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 2)]
        public byte[] min_test_limit;
        /// <summary>
        /// Max. Test Limit (2 bytes)
        /// </summary>
        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 2)]
        public byte[] max_test_limit;
    }

    /// <summary>
    /// Parsed response to service 22-MID
    /// </summary>
    [StructLayout(LayoutKind.Sequential, Pack = 8)]
    public struct obd_request_test_results_response
    {
        /// <summary>
        /// reserved
        /// </summary>
        [MarshalAs(UnmanagedType.U1)]
        public obd_datatype reserved_object_type;
        /// <summary>
        /// reserved
        /// </summary>
        public UInt32 reserved_object_size;
        /// <summary>
        ///  ECU address
        /// </summary>
        public UInt16 ecu_address;
        /// <summary>
        /// NRC (0 if no NRC)
        /// </summary>
        public byte nrc;
        /// <summary>
        /// DID
        /// </summary>
        [MarshalAs(UnmanagedType.U2)]
        public obd_DID_t data_identifier;
        /// <summary>
        /// Number of objects in elements
        /// </summary>
        public UInt32 nb_elements;
        /// <summary>
        /// Array of obd_test_data_object objects (data record of supported MID)
        /// Use it in safe mode with Marshal.PtrToStructure or unsafe mode with IntPtr.toPointer.
        /// See special C# functions
        /// </summary>
        public IntPtr elements;
    }

    // service 31

    /// <summary>
    /// Parsed response to service service 31
    /// </summary>
    [StructLayout(LayoutKind.Sequential, Pack = 8)]
    public struct obd_request_control_operation_response
    {
        /// <summary>
        /// reserved
        /// </summary>
        [MarshalAs(UnmanagedType.U1)]
        public obd_datatype reserved_object_type;
        /// <summary>
        /// reserved
        /// </summary>
        public UInt32 reserved_object_size;
        /// <summary>
        ///  ECU address
        /// </summary>
        public UInt16 ecu_address;
        /// <summary>
        /// NRC (0 if no NRC)
        /// </summary>
        public byte nrc;
        /// <summary>
        /// Routine Control Type
        /// </summary>
        public byte routine_control_type;
        /// <summary>
        /// RID
        /// </summary>
        [MarshalAs(UnmanagedType.U2)]
        public obd_RID_t routine_identifier;
        /// <summary>
        /// routineInfo
        /// </summary>
        public byte routine_info;
        /// <summary>
        /// Number of bytes in elements
        /// </summary>
        public byte nb_elements;
        /// <summary>
        /// Array of bytes (routineStatusRecord)
        /// Use it in safe mode with Marshal.Copy or unsafe mode with IntPtr.toPointer.
        /// See special C# functions
        /// </summary>
        public IntPtr elements;
    }

    // service 22-ITID

    /// <summary>
    /// Parsed response to service service 22-ITID
    /// </summary>
    [StructLayout(LayoutKind.Sequential, Pack = 8)]
    public struct obd_request_vehicle_information_response
    {
        /// <summary>
        /// reserved
        /// </summary>
        [MarshalAs(UnmanagedType.U1)]
        public obd_datatype reserved_object_type;
        /// <summary>
        /// reserved
        /// </summary>
        public UInt32 reserved_object_size;
        /// <summary>
        ///  ECU address
        /// </summary>
        public UInt16 ecu_address;
        /// <summary>
        /// NRC (0 if no NRC)
        /// </summary>
        public byte nrc;
        /// <summary>
        /// DID (ITID) (DID in data record of InfoType)
        /// </summary>
        [MarshalAs(UnmanagedType.U2)]
        public obd_DID_t data_identifier;
        /// <summary>
        /// Number of bytes in elements
        /// </summary>
        public UInt32 nb_elements;
        /// <summary>
        /// Array of bytes (data in data record of InfoType)
        /// Use it in safe mode with Marshal.Copy or unsafe mode with IntPtr.toPointer.
        /// See special C# functions
        /// </summary>
        public IntPtr elements;
    }

    // service 19-55

    /// <summary>
    /// Part of a parsed response to service 19-55
    /// </summary>
    [StructLayout(LayoutKind.Sequential, Pack = 8)]
    public struct obd_trouble_code
    {
        /// <summary>
        /// DTC in DTCAndStatusRecord
        /// </summary>
        [MarshalAs(UnmanagedType.U4)]
        public obd_DTC_t DTC_identifier;
        /// <summary>
        /// statusOfDTC
        /// </summary>
        public byte status_of_dtc;
    }

    /// <summary>
    /// Parsed response to service 19-55
    /// </summary>
    [StructLayout(LayoutKind.Sequential, Pack = 8)]
    public struct obd_request_permanent_trouble_codes_response
    {
        /// <summary>
        /// reserved
        /// </summary>
        [MarshalAs(UnmanagedType.U1)]
        public obd_datatype reserved_object_type;
        /// <summary>
        /// reserved
        /// </summary>
        public UInt32 reserved_object_size;
        /// <summary>
        ///  ECU address
        /// </summary>
        public UInt16 ecu_address;
        /// <summary>
        /// NRC (0 if no NRC)
        /// </summary>
        public byte nrc;
        /// <summary>
        /// DTCReportType
        /// </summary>
        public byte report_type;
        /// <summary>
        /// FunctionalGroupIdentifier
        /// </summary>
        public byte functional_group_identifier;
        /// <summary>
        /// DTCStatusAvailabilityMask
        /// </summary>
        public byte DTC_status_availability_mask;
        /// <summary>
        /// DTCFormatIdentifier
        /// </summary>
        public byte DTC_format_identifier;
        /// <summary>
        ///  Number of objects in elements
        /// </summary>
        public UInt32 nb_elements;
        /// <summary>
        /// Array of obd_trouble_code objects (DTCAndStatusRecord)
        /// Use it in safe mode with Marshal.PtrToStructure or unsafe mode with IntPtr.toPointer.
        /// See special C# functions
        /// </summary>
        public IntPtr elements;
    }

    // service 19-1A

    /// <summary>
    /// Parsed response to service service 19-1A
    /// </summary>
    [StructLayout(LayoutKind.Sequential, Pack = 8)]
    public struct obd_request_supported_dtc_extended_response
    {
        /// <summary>
        /// reserved
        /// </summary>
        [MarshalAs(UnmanagedType.U1)]
        public obd_datatype reserved_object_type;
        /// <summary>
        /// reserved
        /// </summary>
        public UInt32 reserved_object_size;
        /// <summary>
        ///  ECU address
        /// </summary>
        public UInt16 ecu_address;
        /// <summary>
        /// NRC (0 if no NRC)
        /// </summary>
        public byte nrc;
        /// <summary>
        /// DTCReportType
        /// </summary>
        public byte report_type;
        /// <summary>
        /// DTCStatusAvailabilityMask
        /// </summary>
        public byte DTC_status_availability_mask;
        /// <summary>
        /// DTCExtendedDataRecordNumber
        /// </summary>
        public byte DTC_extended_data_record_number;
        /// <summary>
        /// Number of objects in elements
        /// </summary>
        public UInt32 nb_elements;
        /// <summary>
        /// Array of obd_trouble_code objects (DTCAndStatusRecord)
        /// Use it in safe mode with Marshal.PtrToStructure or unsafe mode with IntPtr.toPointer.
        /// See special C# functions
        /// </summary>
        public IntPtr elements;
    }

    // service 19-06

    /// <summary>
    /// Parsed response to service service 19-06
    /// </summary>
    [StructLayout(LayoutKind.Sequential, Pack = 8)]
    public struct obd_request_dtc_extended_response
    {
        /// <summary>
        /// reserved
        /// </summary>
        [MarshalAs(UnmanagedType.U1)]
        public obd_datatype reserved_object_type;
        /// <summary>
        /// reserved
        /// </summary>
        public UInt32 reserved_object_size;
        /// <summary>
        ///  ECU address
        /// </summary>
        public UInt16 ecu_address;
        /// <summary>
        /// NRC (0 if no NRC)
        /// </summary>
        public byte nrc;
        /// <summary>
        /// DTCReportType
        /// </summary>
        public byte report_type;
        /// <summary>
        /// DTC in DTCAndStatusRecord
        /// </summary>
        [MarshalAs(UnmanagedType.U4)]
        public obd_DTC_t DTC_identifier;
        /// <summary>
        /// statusOfDTC
        /// </summary>
        public byte status_of_dtc;
        /// <summary>
        /// DTCExtendedDataRecordNumber
        /// </summary>
        public byte DTC_extended_data_record_number;
        /// <summary>
        /// Number of bytes in elements
        /// </summary>
        public UInt32 nb_elements;
        /// <summary>
        /// Array of bytes (DTCExtendedDataRecord)
        /// Use it in safe mode with Marshal.Copy or unsafe mode with IntPtr.toPointer.
        /// See special C# functions
        /// </summary>
        public IntPtr elements;
    }

    // service 19-56

    /// <summary>
    /// Parsed response to service service 19-56
    /// </summary>
    [StructLayout(LayoutKind.Sequential, Pack = 8)]
    public struct obd_request_dtc_for_a_readiness_group_response
    {
        /// <summary>
        /// reserved
        /// </summary>
        [MarshalAs(UnmanagedType.U1)]
        public obd_datatype reserved_object_type;
        /// <summary>
        /// reserved
        /// </summary>
        public UInt32 reserved_object_size;
        /// <summary>
        ///  ECU address
        /// </summary>
        public UInt16 ecu_address;
        /// <summary>
        /// NRC (0 if no NRC)
        /// </summary>
        public byte nrc;
        /// <summary>
        /// DTCReportType
        /// </summary>
        public byte report_type;
        /// <summary>
        /// FunctionalGroupIdentifier
        /// </summary>
        public byte functional_group_identifier;
        /// <summary>
        /// DTCStatusAvailabilityMask
        /// </summary>
        public byte DTC_status_availability_mask;
        /// <summary>
        /// DTCFormatIdentifier
        /// </summary>
        public byte DTC_format_identifier;
        /// <summary>
        /// DTCReadinessGroupIdentifier
        /// </summary>
        public byte readiness_group_identifier;
        /// <summary>
        /// Number of objects in elements
        /// </summary>
        public UInt32 nb_elements;
        /// <summary>
        /// Array of obd_trouble_code objects (DTCAndStatusRecord)
        /// Use it in safe mode with Marshal.PtrToStructure or unsafe mode with IntPtr.toPointer.
        /// See special C# functions
        /// </summary>
        public IntPtr elements;
    }

    #endregion

    #region PCAN OBDonUDS Api
    public static class OBDonUDSApi
    {
        #region Constants definitions

        /// <summary>
        /// A constant NAI for functional 11bit requests
        /// </summary>      
        public static readonly obd_netaddrinfo OBD_NAI_REQUEST_FUNCTIONAL_11B = new obd_netaddrinfo(obd_msgprotocol.OBD_MSGPROTOCOL_11BIT,
            obd_addressing.OBD_ADDRESSING_FUNCTIONAL,
            (UInt16)uds_address.PUDS_ADDRESS_ISO_15765_4_ADDR_TEST_EQUIPMENT,
            (UInt16)uds_address.PUDS_ADDRESS_ISO_15765_4_ADDR_OBD_FUNCTIONAL);

        /// <summary>
        /// A constant NAI for functional 29bit requests
        /// </summary>
        public static readonly obd_netaddrinfo OBD_NAI_REQUEST_FUNCTIONAL_29B = new obd_netaddrinfo(obd_msgprotocol.OBD_MSGPROTOCOL_29BIT,
            obd_addressing.OBD_ADDRESSING_FUNCTIONAL,
            (UInt16)uds_address.PUDS_ADDRESS_ISO_15765_4_ADDR_TEST_EQUIPMENT,
            (UInt16)uds_address.PUDS_ADDRESS_ISO_15765_4_ADDR_OBD_FUNCTIONAL);


        public const byte OBDONUDS_EMISSION_SYSTEM_GROUP = 0x33;
        public const byte OBDONUDS_DTC_SEVERITY_CLASS_1 = 0x02;
        public const byte OBDONUDS_DTC_STATUS_CONFIRMED = 0x08;
        public const byte OBDONUDS_DTC_STATUS_PENDING = 0x04;
        public const byte OBDONUDS_ROUTINE_START = 0x01;
        public const byte OBDONUDS_DTC_SNAPSHOT_RECORD_FIRST = 0x00;
        public const byte OBDONUDS_DTC_SNAPSHOT_RECORD_LAST = 0xF0;
        public const UInt32 OBDONUDS_DTC_EMISSION_SYSTEM_GROUP = 0xFFFF33;
        public const UInt32 OBDONUDS_DTC_ALL_GROUPS = 0xFFFFFF;
        #endregion

        #region POBDONUDS parameter values
        /// <summary>
        /// Disable debug messages (default)
        /// </summary>
        public const byte POBDONUDS_DEBUG_LVL_NONE = 0x00;
        /// <summary>
        /// Enable debug messages (only errors)
        /// </summary>
        public const byte POBDONUDS_DEBUG_LVL_ERROR = 0xF1;
        /// <summary>
        /// Enable debug messages (only warnings, errors)
        /// </summary>
        public const byte POBDONUDS_DEBUG_LVL_WARNING = 0xF2;
        /// <summary>
        /// Enable debug messages (only informations, warnings, errors)
        /// </summary>
        public const byte POBDONUDS_DEBUG_LVL_INFORMATION = 0xF3;
        /// <summary>
        /// Enable debug messages (only notices, informations, warnings, errors)
        /// </summary>
        public const byte POBDONUDS_DEBUG_LVL_NOTICE = 0xF4;
        /// <summary>
        /// Enable debug messages (only debug, notices, informations, warnings, errors)
        /// </summary>
        public const byte POBDONUDS_DEBUG_LVL_DEBUG = 0xF5;
        /// <summary>
        /// Enable all debug messages
        /// </summary>
        public const byte POBDONUDS_DEBUG_LVL_TRACE = 0xF6;
        #endregion

        #region PCAN-OBDonUDS API: Core function declarations

        /// <summary>
        /// Initializes an OBDonUDS-Client based on a PCAN-ISO-TP channel
        /// </summary>
        /// <remarks>Only one OBDonUDS-Client can be initialized per PCAN-ISO-TP-Channel</remarks>
        /// <param name="channel">The PCAN-ISO-TP channel to be used as OBDonUDS client</param>
        /// <param name="baudrate">The CAN Hardware speed</param>
        /// <param name="hw_type">NON PLUG&PLAY: The type of hardware and operation mode</param>
        /// <param name="io_port">NON PLUG&PLAY: The I/O address for the parallel port</param>
        /// <param name="interupt">NON PLUG&PLAY: Interrupt number of the parallel port</param>
        /// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success</returns>
        [DllImport("PCAN-OBDonUDS.dll", EntryPoint = "OBDonUDS_Initialize")]
        public static extern obd_status Initialize(
            [MarshalAs(UnmanagedType.U4)]
            cantp_handle channel,
            [MarshalAs(UnmanagedType.U4)]
            cantp_baudrate baudrate,
            [MarshalAs(UnmanagedType.U4)]
            cantp_hwtype hw_type,
            UInt32 io_port,
            UInt16 interrupt);

        /// <summary>
        /// Initializes an OBDonUDS-Client based on a PCAN-ISO-TP channel
        /// </summary>
        /// <remarks>Only one OBDonUDS-Client can be initialized per PCAN-ISO-TP-Channel</remarks>
        /// <param name="channel">The PCAN-ISO-TP channel to be used as OBDonUDS client</param>
        /// <param name="baudrate">The CAN Hardware speed</param>
        /// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success</returns>
        public static obd_status Initialize(
            [MarshalAs(UnmanagedType.U4)]
            cantp_handle channel,
            [MarshalAs(UnmanagedType.U4)]
            cantp_baudrate baudrate)
        {
            return Initialize(channel, baudrate, (cantp_hwtype)0, 0, 0);
        }

        /// <summary>
        /// Initializes an OBDonUDS-Client based on a PCAN-ISO-TP channel, using auto-detection of baudrate
        /// </summary>
        /// <remarks>Only one OBDonUDS-Client can be initialized per PCAN-ISO-TP-Channel</remarks>
        /// <param name="channel">The PCAN-ISO-TP channel to be used as OBDonUDS client</param>
        /// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success</returns>
        public static obd_status Initialize(
            [MarshalAs(UnmanagedType.U4)]
            cantp_handle channel)
        {
            return Initialize(channel, (cantp_baudrate)obd_baudrate.OBD_BAUDRATE_AUTODETECT, (cantp_hwtype)0, 0, 0);
        }

        /// <summary>
        /// Tests a channel to detect ECUs responding in OBDonEDS-mode.
        /// </summary>
        /// <remarks>This function must be called on a closed channel and leaves the channel closed</remarks>
        /// <param name="channel">The PCAN-ISO-TP channel to be used as OBDonUDS client</param>
        /// <param name="baudrate">The CAN Hardware speed</param>
        /// <param name="hw_type">NON PLUG&PLAY: The type of hardware and operation mode</param>
        /// <param name="io_port">NON PLUG&PLAY: The I/O address for the parallel port</param>
        /// <param name="Interupt">NON PLUG&PLAY: Interrupt number of the parallel port</param>
        /// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success</returns>
        [DllImport("PCAN-OBDonUDS.dll", EntryPoint = "OBDonUDS_FindOBDonEDS")]
        public static extern obd_status FindOBDonEDS(cantp_handle channel,
            cantp_baudrate baudrate,
            cantp_hwtype hw_type,
            UInt32 io_port,
            UInt16 interrupt);

        /// <summary>
        /// Tests a channel to detect ECUs responding in OBDonEDS-mode.
        /// </summary>
        /// <remarks>This function must be called on a closed channel and leaves the channel closed</remarks>
        /// <param name="channel">The PCAN-ISO-TP channel to be used as OBDonUDS client</param>
        /// <param name="baudrate">The CAN Hardware speed</param>
        /// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success</returns>
        public static obd_status FindOBDonEDS(cantp_handle channel,
            cantp_baudrate baudrate)
        {
            return FindOBDonEDS(channel, baudrate, (cantp_hwtype)0, 0, 0);
        }

        /// <summary>
        /// Tests a channel to detect ECUs responding in OBDonEDS-mode, using auto-detection of baudrate
        /// </summary>
        /// <remarks>This function must be called on a closed channel and leaves the channel closed</remarks>
        /// <param name="channel">The PCAN-ISO-TP channel to be used as OBDonUDS client</param>
        /// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success</returns>
        public static obd_status FindOBDonEDS(cantp_handle channel)
        {
            return FindOBDonEDS(channel, (cantp_baudrate)obd_baudrate.OBD_BAUDRATE_AUTODETECT, (cantp_hwtype)0, 0, 0);
        }

        /// <summary>
        /// Uninitializes an OBDonUDS-Client initialized before
        /// </summary>
        /// <param name="channel">A PCANTP channel handle representing a OBDonUDS-Client</param>
        /// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success</returns>
        [DllImport("PCAN-OBDonUDS.dll", EntryPoint = "OBDonUDS_Uninitialize")]
        public static extern obd_status Uninitialize(
            [MarshalAs(UnmanagedType.U4)]
            cantp_handle channel);

        /// <summary>
        /// Resets the receive and transmit queues of an OBDonUDS-Client 
        /// </summary>
        /// <param name="channel">A PCANTP channel handle representing a OBDonUDS-Client</param>
        /// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success</returns>
        [DllImport("PCAN-OBDonUDS.dll", EntryPoint = "OBDonUDS_Reset")]
        public static extern obd_status Reset(
            [MarshalAs(UnmanagedType.U4)]
            cantp_handle channel);

        /// <summary>
        /// Gets the initialization status of an OBDonUDS CAN-Channel
        /// </summary>
        /// <param name="channel">A PCANTP channel handle representing a OBDonUDS-Client</param>
        /// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success</returns>
        [DllImport("PCAN-OBDonUDS.dll", EntryPoint = "OBDonUDS_GetStatus")]
        public static extern obd_status GetStatus(
            [MarshalAs(UnmanagedType.U4)]
            cantp_handle channel);

        /// <summary>
        /// Checks if a status matches an expected result 
        /// </summary>
        /// <param name="status">The status to analyze.</param>
        /// <param name="status_expected">The expected status (default is POBDONUDS_STATUS_OK).</param>
        /// <param name="strict_mode">Enable strict mode (default is false). Strict mode ensures that bus or extra information are the same.</param>
        /// <returns>Returns true if the status matches expected parameter.</returns>
        [return: MarshalAs(UnmanagedType.I1)]
        [DllImport("PCAN-OBDonUDS.dll", EntryPoint = "OBDonUDS_StatusIsOk")]
        public static extern bool StatusIsOk(
            [MarshalAs(UnmanagedType.U4)]
            obd_status status,
            [MarshalAs(UnmanagedType.U4)]
            obd_status status_expected,
            [MarshalAs(UnmanagedType.I1)]
            bool strict_mode);

        /// <summary>
        /// Checks if a status matches an expected result in a non-strict mode.
        /// Strict mode ensures that bus or extra information are the same.
        /// </summary>
        /// <param name="status">The status to analyze.</param>
        /// <param name="status_expected">The expected status (default is POBDONUDS_STATUS_OK).</param>
        /// <returns>Returns true if the status matches expected parameter.</returns>
        public static bool StatusIsOk(
            obd_status status,
            obd_status status_expected)
        {
            return StatusIsOk(status, status_expected, false);
        }

        /// <summary>
        /// Checks if a status matches POBDONUDS_STATUS_OK in a non-strict mode.
        /// Strict mode ensures that bus or extra information are the same.
        /// </summary>
        /// <param name="status">The status to analyze.</param>
        /// <returns>Returns true if the status matches POBDONUDS_STATUS_OK.</returns>
        public static bool StatusIsOk(
            obd_status status)
        {
            return StatusIsOk(status, obd_status.POBDONUDS_STATUS_OK, false);
        }

        /// <summary>
        /// Returns a descriptive text of a given obd_status error
        /// code, in any desired language
        /// </summary>
        /// <remarks>The current languages available for translation are:
        /// Neutral (0x00), English (0x09), and French (0x0C)</remarks>
        /// <param name="error_code">A obd_status error code</param>
        /// <param name="language">Indicates a 'Primary language ID'</param>
        /// <param name="buffer">Buffer for a null terminated char array</param>
        /// <param name="buffer_size">Buffer size</param>
        /// <returns>A obd_status error code</returns>
        [DllImport("PCAN-OBDonUDS.dll", EntryPoint = "OBDonUDS_GetErrorText")]
        public static extern obd_status GetErrorText(
            [MarshalAs(UnmanagedType.U4)]
            obd_status error_code,
            UInt16 language,
            StringBuilder buffer,
            UInt32 buffer_size);

        /// <summary>
        /// Retrieves an OBDonUDS-Client parameter value
        /// </summary>
        /// <param name="channel">A PCANTP channel handle representing a OBDonUDS client</param>
        /// <param name="parameter">The parameter to get</param>
        /// <param name="buffer">Pointer to the parameter value</param>
        /// <param name="buffer_size">Size in bytes of the pointed value</param>
        /// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success</returns>
        [DllImport("PCAN-OBDonUDS.dll", EntryPoint = "OBDonUDS_GetValue")]
        public static extern obd_status GetValue(
            [MarshalAs(UnmanagedType.U4)]
            cantp_handle channel,
            [MarshalAs(UnmanagedType.U4)]
            obd_parameter parameter,
            IntPtr buffer,
            UInt32 buffer_size);

        /// <summary>
        /// Retrieves an OBDonUDS-Client parameter value
        /// </summary>
        /// <param name="channel">A PCANTP channel handle representing a OBDonUDS client</param>
        /// <param name="parameter">The parameter to get</param>
        /// <param name="buffer">Buffer for the parameter value</param>
        /// <param name="buffer_size">Size in bytes of the buffer</param>
        /// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success</returns>
        [DllImport("PCAN-OBDonUDS.dll", EntryPoint = "OBDonUDS_GetValue")]
        public static extern obd_status GetValue(
            [MarshalAs(UnmanagedType.U4)]
            cantp_handle channel,
            [MarshalAs(UnmanagedType.U4)]
            obd_parameter parameter,
            StringBuilder buffer,
            UInt32 buffer_size);

        /// <summary>
        /// Retrieves an OBDonUDS-Client parameter value
        /// </summary>
        /// <param name="channel">A PCANTP channel handle representing a OBDonUDS client</param>
        /// <param name="parameter">The parameter to get</param>
        /// <param name="buffer">Buffer for the parameter value</param>
        /// <param name="buffer_size">Size in bytes of the buffer</param>
        /// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success</returns>
        [DllImport("PCAN-OBDonUDS.dll", EntryPoint = "OBDonUDS_GetValue")]
        public static extern obd_status GetValue(
            [MarshalAs(UnmanagedType.U4)]
            cantp_handle channel,
            [MarshalAs(UnmanagedType.U4)]
            obd_parameter parameter,
            out UInt32 buffer,
            UInt32 buffer_size);

        /// <summary>
        /// Retrieves an OBDonUDS-Client parameter baudrate value
        /// </summary>
        /// <param name="channel">A PCANTP channel handle representing a OBDonUDS client</param>
        /// <param name="parameter">The parameter to get</param>
        /// <param name="buffer">Buffer for the baudrate parameter value</param>
        /// <param name="buffer_size">Size in bytes of the buffer</param>
        /// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success</returns>
        [DllImport("PCAN-OBDonUDS.dll", EntryPoint = "OBDonUDS_GetValue")]
        public static extern obd_status GetValue(
            [MarshalAs(UnmanagedType.U4)]
            cantp_handle channel,
            [MarshalAs(UnmanagedType.U4)]
            obd_parameter parameter,
            out obd_baudrate buffer,
            UInt32 buffer_size);

        /// <summary>
        /// Retrieves an OBDonUDS-Client parameter protocol value
        /// </summary>
        /// <param name="channel">A PCANTP channel handle representing a OBDonUDS client</param>
        /// <param name="parameter">The parameter to get</param>
        /// <param name="buffer">Buffer for the protocol parameter value</param>
        /// <param name="buffer_size">Size in bytes of the buffer</param>
        /// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success</returns>
        [DllImport("PCAN-OBDonUDS.dll", EntryPoint = "OBDonUDS_GetValue")]
        public static extern obd_status GetValue(
            [MarshalAs(UnmanagedType.U4)]
            cantp_handle channel,
            [MarshalAs(UnmanagedType.U4)]
            obd_parameter parameter,
            out obd_msgprotocol buffer,
            UInt32 buffer_size);

        /// <summary>
        /// Retrieves an OBDonUDS-Client parameter value
        /// </summary>
        /// <param name="channel">A PCANTP channel handle representing a OBDonUDS client</param>
        /// <param name="parameter">The parameter to get</param>
        /// <param name="buffer">Buffer for the parameter value</param>
        /// <param name="buffer_size">Size in bytes of the buffer</param>
        /// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success</returns>
        [DllImport("PCAN-OBDonUDS.dll", EntryPoint = "OBDonUDS_GetValue")]
        public static extern obd_status GetValue(
            [MarshalAs(UnmanagedType.U4)]
            cantp_handle channel,
            [MarshalAs(UnmanagedType.U4)]
            obd_parameter parameter,
            [MarshalAs(UnmanagedType.LPArray)]
            [Out] Byte[] buffer,
            UInt32 buffer_size);

        /// <summary>
        /// Configures or sets an OBDonUDS-Client parameter value
        /// </summary>
        /// <param name="channel">A PCANTP channel handle representing an OBDonUDS channel</param>
        /// <param name="parameter">The parameter to set</param>
        /// <param name="buffer">Pointer to the parameter value</param>
        /// <param name="buffer_size">Size in bytes of the pointed value</param>
        /// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success</returns>
        [DllImport("PCAN-OBDonUDS.dll", EntryPoint = "OBDonUDS_SetValue")]
        public static extern obd_status SetValue(
            [MarshalAs(UnmanagedType.U4)]
            cantp_handle channel,
            [MarshalAs(UnmanagedType.U4)]
            obd_parameter parameter,
            IntPtr buffer,
            UInt32 buffer_size);

        /// <summary>
        /// Configures or sets an OBDonUDS-Client parameter value
        /// </summary>
        /// <param name="channel">A PCANTP channel handle representing an OBDonUDS channel</param>
        /// <param name="parameter">The parameter to set</param>
        /// <param name="buffer">Buffer for the parameter value</param>
        /// <param name="buffer_size">Size in bytes of the buffer</param>
        /// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success</returns>
        [DllImport("PCAN-OBDonUDS.dll", EntryPoint = "OBDonUDS_SetValue")]
        public static extern obd_status SetValue(
            [MarshalAs(UnmanagedType.U4)]
            cantp_handle channel,
            [MarshalAs(UnmanagedType.U4)]
            obd_parameter parameter,
            ref UInt32 buffer,
            UInt32 buffer_size);

        /// <summary>
        /// Configures or sets an OBDonUDS-Client parameter value
        /// </summary>
        /// <param name="channel">A PCANTP channel handle representing an OBDonUDS channel</param>
        /// <param name="parameter">The parameter to set</param>
        /// <param name="buffer">Buffer for the parameter value</param>
        /// <param name="buffer_size">Size in bytes of the buffer</param>
        /// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success</returns>
        [DllImport("PCAN-OBDonUDS.dll", EntryPoint = "OBDonUDS_SetValue")]
        public static extern obd_status SetValue(
            [MarshalAs(UnmanagedType.U4)]
            cantp_handle channel,
            [MarshalAs(UnmanagedType.U4)]
            obd_parameter parameter,
            [MarshalAs(UnmanagedType.LPStr, SizeParamIndex = 3)]
            String buffer,
            UInt32 buffer_size);

        /// <summary>
        /// Configures or sets an OBDonUDS-Client parameter value
        /// </summary>
        /// <param name="channel">A PCANTP channel handle representing an OBDonUDS channel</param>
        /// <param name="parameter">The parameter to set</param>
        /// <param name="buffer">Buffer for the parameter value</param>
        /// <param name="buffer_size">Size in bytes of the buffer</param>
        /// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success</returns>
        [DllImport("PCAN-OBDonUDS.dll", EntryPoint = "OBDonUDS_SetValue")]
        public static extern obd_status SetValue(
            [MarshalAs(UnmanagedType.U4)]
            cantp_handle channel,
            [MarshalAs(UnmanagedType.U4)]
            obd_parameter parameter,
            [MarshalAs(UnmanagedType.LPArray, SizeParamIndex = 3)]
            Byte[] buffer,
            UInt32 buffer_size);

        /// <summary>
        /// Deallocates a OBDonUDS message
        /// </summary>
        /// <param name="msg_buffer">An allocated obd_msg structure buffer</param>
        /// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success</returns>
        [DllImport("PCAN-OBDonUDS.dll", EntryPoint = "OBDonUDS_MsgFree")]
        public static extern obd_status MsgFree(
            ref obd_msg msg_buffer);

        /// <summary>
        /// Handles the communication workflow for a OBDonUDS service expecting a single response.
        /// </summary>
        /// <param name="channel">A PCANTP channel handle representing a POBDonUDS channel</param>
        /// <param name="msg_request">A sent obd_msg message used as a reference to manage the OBDonUDS service</param>
        /// <param name="out_msg_response">[out] A obd_msg structure buffer to store the POBDonUDS response</param>
        /// <param name="out_msg_request_confirmation">[out] A obd_msg structure buffer to store the POBDonUDS request confirmation</param>
        /// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success</returns>
        [DllImport("PCAN-OBDonUDS.dll", EntryPoint = "OBDonUDS_WaitForService")]
        public static extern obd_status WaitForService(
            [MarshalAs(UnmanagedType.U4)]
            cantp_handle channel,
            [In] ref obd_msg msg_request,
            out obd_msg out_msg_response,
            out obd_msg out_msg_request_confirmation);

        /// <summary>
        /// Handles the communication workflow for a OBDonUDS service expecting multiple responses.
        /// </summary>
        /// <param name="channel">A PCANTP channel handle representing a POBDonUDS channel</param>
        /// <param name="msg_request">A sent obd_msg message used as a reference to manage the OBDonUDS service</param>
        /// <param name="max_msg_count">Length of the buffer array (max. messages that can be received)</param>
        /// <param name="wait_until_timeout">if <code>FALSE</code> the function is interrupted if out_msg_count reaches max_msg_count.</param>
        /// <param name="out_msg_responses">[out] A buffer to store the responses. It must be an array of "max_msg_count" entries (it must have at least
        /// a size of max_msg_count * sizeof(obd_msg) bytes) </param>
        /// <param name="out_msg_count">[out] Actual number of messages read</param>
        /// <param name="out_msg_request_confirmation">[out] A obd_msg structure buffer to store the POBDonUDS request confirmation</param>
        /// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success,
        /// PUDS_ERROR_OVERFLOW indicates success but buffer was too small to hold all responses.</returns>
        [DllImport("PCAN-OBDonUDS.dll", EntryPoint = "OBDonUDS_WaitForServiceFunctional")]
        public static extern obd_status WaitForServiceFunctional(
            [MarshalAs(UnmanagedType.U4)]
            cantp_handle channel,
            [In] ref obd_msg msg_request,
            UInt32 max_msg_count,
            [MarshalAs(UnmanagedType.U1)]
            bool wait_until_timeout,
            [MarshalAs(UnmanagedType.LPArray, SizeParamIndex = 2)]
            [Out] obd_msg[] out_msg_responses,
            out UInt32 out_msg_count,
            out obd_msg out_msg_request_confirmation);

        /// <summary>
        /// Deallocates a parsed response (see structures obd_request_XXX_response)
        /// </summary>
        /// <remarks>For all services, this function is to use after calls to functions OBDonUDS_ParseXXXX</remarks>
        /// <param name="response">The parsed response to free</param>
        /// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success.</returns>
        [DllImport("PCAN-OBDonUDS.dll", EntryPoint = "OBDonUDS_ParsedResponseFree")]
        public static extern obd_status ParsedResponseFree(
            ref obd_response_generic response);

        /// <summary>
        /// Deallocates a parsed response (see structures obd_request_XXX_response)
        /// </summary>
        /// <remarks>For all services, this function is to use after calls to functions OBDonUDS_ParseXXXX</remarks>
        /// <param name="response">The parsed response to free</param>
        /// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success.</returns>
        [DllImport("PCAN-OBDonUDS.dll", EntryPoint = "OBDonUDS_ParsedResponseFree")]
        public static extern obd_status ParsedResponseFree(
            ref obd_request_vehicle_information_response response);
        /// <summary>
        /// Deallocates a parsed response (see structures obd_request_XXX_response)
        /// </summary>
        /// <remarks>For all services, this function is to use after calls to functions OBDonUDS_ParseXXXX</remarks>
        /// <param name="response">The parsed response to free</param>
        /// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success.</returns>
        [DllImport("PCAN-OBDonUDS.dll", EntryPoint = "OBDonUDS_ParsedResponseFree")]
        public static extern obd_status ParsedResponseFree(
            ref obd_request_clear_trouble_codes_response response);
        /// <summary>
        /// Deallocates a parsed response (see structures obd_request_XXX_response)
        /// </summary>
        /// <remarks>For all services, this function is to use after calls to functions OBDonUDS_ParseXXXX</remarks>
        /// <param name="response">The parsed response to free</param>
        /// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success.</returns>
        [DllImport("PCAN-OBDonUDS.dll", EntryPoint = "OBDonUDS_ParsedResponseFree")]
        public static extern obd_status ParsedResponseFree(
            ref obd_request_control_operation_response response);
        /// <summary>
        /// Deallocates a parsed response (see structures obd_request_XXX_response)
        /// </summary>
        /// <remarks>For all services, this function is to use after calls to functions OBDonUDS_ParseXXXX</remarks>
        /// <param name="response">The parsed response to free</param>
        /// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success.</returns>
        [DllImport("PCAN-OBDonUDS.dll", EntryPoint = "OBDonUDS_ParsedResponseFree")]
        public static extern obd_status ParsedResponseFree(
            ref obd_request_current_data_response response);
        /// <summary>
        /// Deallocates a parsed response (see structures obd_request_XXX_response)
        /// </summary>
        /// <remarks>For all services, this function is to use after calls to functions OBDonUDS_ParseXXXX</remarks>
        /// <param name="response">The parsed response to free</param>
        /// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success.</returns>
        [DllImport("PCAN-OBDonUDS.dll", EntryPoint = "OBDonUDS_ParsedResponseFree")]
        public static extern obd_status ParsedResponseFree(
            ref obd_request_dtc_extended_response response);
        /// <summary>
        /// Deallocates a parsed response (see structures obd_request_XXX_response)
        /// </summary>
        /// <remarks>For all services, this function is to use after calls to functions OBDonUDS_ParseXXXX</remarks>
        /// <param name="response">The parsed response to free</param>
        /// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success.</returns>
        [DllImport("PCAN-OBDonUDS.dll", EntryPoint = "OBDonUDS_ParsedResponseFree")]
        public static extern obd_status ParsedResponseFree(
            ref obd_request_dtc_for_a_readiness_group_response response);
        /// <summary>
        /// Deallocates a parsed response (see structures obd_request_XXX_response)
        /// </summary>
        /// <remarks>For all services, this function is to use after calls to functions OBDonUDS_ParseXXXX</remarks>
        /// <param name="response">The parsed response to free</param>
        /// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success.</returns>
        [DllImport("PCAN-OBDonUDS.dll", EntryPoint = "OBDonUDS_ParsedResponseFree")]
        public static extern obd_status ParsedResponseFree(
            ref obd_request_freeze_frame_data_response response);
        /// <summary>
        /// Deallocates a parsed response (see structures obd_request_XXX_response)
        /// </summary>
        /// <remarks>For all services, this function is to use after calls to functions OBDonUDS_ParseXXXX</remarks>
        /// <param name="response">The parsed response to free</param>
        /// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success.</returns>
        [DllImport("PCAN-OBDonUDS.dll", EntryPoint = "OBDonUDS_ParsedResponseFree")]
        public static extern obd_status ParsedResponseFree(
            ref obd_request_permanent_trouble_codes_response response);
        /// <summary>
        /// Deallocates a parsed response (see structures obd_request_XXX_response)
        /// </summary>
        /// <remarks>For all services, this function is to use after calls to functions OBDonUDS_ParseXXXX</remarks>
        /// <param name="response">The parsed response to free</param>
        /// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success.</returns>
        [DllImport("PCAN-OBDonUDS.dll", EntryPoint = "OBDonUDS_ParsedResponseFree")]
        public static extern obd_status ParsedResponseFree(
            ref obd_request_supported_dtc_extended_response response);
        /// <summary>
        /// Deallocates a parsed response (see structures obd_request_XXX_response)
        /// </summary>
        /// <remarks>For all services, this function is to use after calls to functions OBDonUDS_ParseXXXX</remarks>
        /// <param name="response">The parsed response to free</param>
        /// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success.</returns>
        [DllImport("PCAN-OBDonUDS.dll", EntryPoint = "OBDonUDS_ParsedResponseFree")]
        public static extern obd_status ParsedResponseFree(
            ref obd_request_test_results_response response);
        /// <summary>
        /// Deallocates a parsed response (see structures obd_request_XXX_response)
        /// </summary>
        /// <remarks>For all services, this function is to use after calls to functions OBDonUDS_ParseXXXX</remarks>
        /// <param name="response">The parsed response to free</param>
        /// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success.</returns>
        [DllImport("PCAN-OBDonUDS.dll", EntryPoint = "OBDonUDS_ParsedResponseFree")]
        public static extern obd_status ParsedResponseFree(
            ref obd_request_trouble_codes_response response);

        #region service 22-DID

        /// <summary>
        /// Send a request to service 22-DID (Request Current Powertrain Diagnostic Data)
        /// </summary>
        /// <param name="channel">A PCANTP channel handle representing a POBDonUDS channel</param>
        /// <param name="nai">Network Addressing Information</param>
        /// <param name="out_msg_request">[out] Request added to the Tx Queue, to be later used with OBDonUDS_WaitForServiceXXX</param>
        /// <param name="data_identifier">Array of data_identifier_length DIDs</param>
        /// <param name="data_identifier_length">Length (number of DIDs) of the data_identifier array</param>
        /// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success.</returns>
        [DllImport("PCAN-OBDonUDS.dll", EntryPoint = "OBDonUDS_RequestCurrentData")]
        public static extern obd_status RequestCurrentData(
            [MarshalAs(UnmanagedType.U4)]
            cantp_handle channel,
            obd_netaddrinfo nai,
            out obd_msg out_msg_request,
            [MarshalAs(UnmanagedType.LPArray, ArraySubType = UnmanagedType.U2, SizeParamIndex = 4)]
            obd_DID_t[] data_identifier,
            UInt32 data_identifier_length);

        /// <summary>
        /// Parse a response to requested service 22-DID (Request Current Powertrain Diagnostic Data)
        /// </summary>
        /// <remarks>The result must be freed afterwards by calling OBDonUDS_ParsedResponseFree</remarks>
        /// <param name="msg_response">Response to parse</param>
        /// <param name="out_buffer">[out] Parsed result</param>
        /// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success.</returns>
        [DllImport("PCAN-OBDonUDS.dll", EntryPoint = "OBDonUDS_ParseResponse_RequestCurrentData")]
        public static extern obd_status ParseResponse_RequestCurrentData(
            [In] ref obd_msg msg_response,
            out obd_request_current_data_response out_buffer);

        #endregion

        #region service 19-04

        /// <summary>
        /// Send a request to service 19-04 (Request Powertrain Freeze Frame Data)
        /// </summary>
        /// <param name="channel">A PCANTP channel handle representing a POBDonUDS channel</param>
        /// <param name="nai">Network Addressing Information</param>
        /// <param name="out_msg_request">[out] Request added to the Tx Queue, to be later used with OBDonUDS_WaitForServiceXXX</param>
        /// <param name="DTC_Number">DTCMaskRecord</param>
        /// <param name="record_number">DTCSnapshotRecordNumber (see OBDONUDS_DTC_SNAPSHOT_RECORD_xxx)</param>
        /// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success.</returns>
        [DllImport("PCAN-OBDonUDS.dll", EntryPoint = "OBDonUDS_RequestFreezeFrameData")]
        public static extern obd_status RequestFreezeFrameData(
             [MarshalAs(UnmanagedType.U4)]
            cantp_handle channel,
            obd_netaddrinfo nai,
            out obd_msg out_msg_request,
            [MarshalAs(UnmanagedType.U4)]
            obd_DTC_t DTC_Number,
            byte record_number);

        /// <summary>
        /// Parse a response to requested service 19-04 (Request Powertrain Freeze Frame Data)
        /// </summary>
        /// <remarks>The result must be freed afterwards by calling OBDonUDS_ParsedResponseFree</remarks>
        /// <param name="msg_response">Response to parse</param>
        /// <param name="out_buffer">[out] Parsed result</param>
        /// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success.</returns>
        [DllImport("PCAN-OBDonUDS.dll", EntryPoint = "OBDonUDS_ParseResponse_RequestFreezeFrameData")]
        public static extern obd_status ParseResponse_RequestFreezeFrameData(
            [In] ref obd_msg msg_response,
            out obd_request_freeze_frame_data_response out_buffer);

        #endregion

        #region service 19-42

        /// <summary>
        /// Send a request to service 19-42 (Request Emission-Related Diagnostic Trouble Codes with Confirmed Status)
        /// </summary>
        /// <param name="channel">A PCANTP channel handle representing a POBDonUDS channel</param>
        /// <param name="nai">Network Addressing Information</param>
        /// <param name="out_msg_request">[out] Request added to the Tx Queue, to be later used with OBDonUDS_WaitForServiceXXX</param>
        /// <param name="functional_group_identifier">FunctionalGroupIdentifier (see OBDONUDS_EMISSION_SYSTEM_GROUP)</param>
        /// <param name="DTC_status_mask">DTCStatusMask (see OBDONUDS_DTC_STATUS_CONFIRMED)</param>
        /// <param name="DTC_severity_mask">DTCSeverityMask (see OBDONUDS_DTC_SEVERITY_CLASS_1)</param>
        /// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success.</returns>
        [DllImport("PCAN-OBDonUDS.dll", EntryPoint = "OBDonUDS_RequestConfirmedTroubleCodes")]
        public static extern obd_status RequestConfirmedTroubleCodes(
            [MarshalAs(UnmanagedType.U4)]
            cantp_handle channel,
            obd_netaddrinfo nai,
            out obd_msg out_msg_request,
            byte functional_group_identifier,
            byte DTC_status_mask,
            byte DTC_severity_mask);

        /// <summary>
        /// Parse a response to requested service 19-42 (Request Emission-Related Diagnostic Trouble Codes with Confirmed Status)
        /// </summary>
        /// <remarks>The result must be freed afterwards by calling OBDonUDS_ParsedResponseFree</remarks>
        /// <param name="msg_response">Response to parse</param>
        /// <param name="out_buffer">[out] Parsed result</param>
        /// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success.</returns>
        [DllImport("PCAN-OBDonUDS.dll", EntryPoint = "OBDonUDS_ParseResponse_RequestConfirmedTroubleCodes")]
        public static extern obd_status ParseResponse_RequestConfirmedTroubleCodes(
            [In] ref obd_msg msg_response,
            out obd_request_trouble_codes_response out_buffer);

        #endregion

        #region service 14

        /// <summary>
        /// Send a request to service 14 (Clear/reset Emission-related Diagnostic Information)
        /// </summary>
        /// <param name="channel">A PCANTP channel handle representing a POBDonUDS channel</param>
        /// <param name="nai">Network Addressing Information</param>
        /// <param name="out_msg_request">[out] Request added to the Tx Queue, to be later used with OBDonUDS_WaitForServiceXXX</param>
        /// <param name="group_of_DTC">GroupOfDTC</param>
        /// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success.</returns>
        [DllImport("PCAN-OBDonUDS.dll", EntryPoint = "OBDonUDS_ClearTroubleCodes")]
        public static extern obd_status ClearTroubleCodes(
            [MarshalAs(UnmanagedType.U4)]
            cantp_handle channel,
            obd_netaddrinfo nai,
            out obd_msg out_msg_request,
            [MarshalAs(UnmanagedType.U4)]
            obd_DTC_t group_of_DTC);

        /// <summary>
        /// Parse a response to requested service 14 (Clear/reset Emission-related Diagnostic Information)
        /// </summary>
        /// <remarks>The result must be freed afterwards by calling OBDonUDS_ParsedResponseFree</remarks>
        /// <param name="msg_response">Response to parse</param>
        /// <param name="out_buffer">[out] Parsed result</param>
        /// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success.</returns>
        [DllImport("PCAN-OBDonUDS.dll", EntryPoint = "OBDonUDS_ParseResponse_ClearTroubleCodes")]
        public static extern obd_status ParseResponse_ClearTroubleCodes(
            [In] ref obd_msg msg_response,
            out obd_request_clear_trouble_codes_response out_buffer);

        #endregion
        
        #region service 22-MID

        /// <summary>
        /// Send a request to service 22-MID (Request On-board Monitoring Test Results for Specific Monitored Systems)
        /// </summary>
        /// <param name="channel">A PCANTP channel handle representing a POBDonUDS channel</param>
        /// <param name="nai">Network Addressing Information</param>
        /// <param name="out_msg_request">[out] Request added to the Tx Queue, to be later used with OBDonUDS_WaitForServiceXXX</param>
        /// <param name="data_identifier">Array of data_identifier_length DIDs (MIDs)</param>
        /// <param name="data_identifier_length">Length (number of MIDs) of data_identifier (may be 1)</param>
        /// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success.</returns>
        [DllImport("PCAN-OBDonUDS.dll", EntryPoint = "OBDonUDS_RequestTestResults")]
        public static extern obd_status RequestTestResults(
           [MarshalAs(UnmanagedType.U4)]
            cantp_handle channel,
            obd_netaddrinfo nai,
            out obd_msg out_msg_request,
            [MarshalAs(UnmanagedType.LPArray, ArraySubType = UnmanagedType.U2, SizeParamIndex = 4)]
            obd_DID_t[] data_identifier,
            UInt32 data_identifier_length);

        /// <summary>
        /// Parse a response to requested service 22-MID (Request On-board Monitoring Test Results for Specific Monitored Systems)
        /// </summary>
        /// <remarks>The result must be freed afterwards by calling OBDonUDS_ParsedResponseFree</remarks>
        /// <param name="msg_response">Response to parse</param>
        /// <param name="out_buffer">[out] Parsed result</param>
        /// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success.</returns>
        [DllImport("PCAN-OBDonUDS.dll", EntryPoint = "OBDonUDS_ParseResponse_RequestTestResults")]
        public static extern obd_status ParseResponse_RequestTestResults(
            [In] ref obd_msg msg_response,
            out obd_request_test_results_response out_buffer);

        #endregion
        
        #region service 19-42

        /// <summary>
        /// Send a request to service 19-42 (Request Emission-related Diagnostic Trouble Codes with Pending Status)
        /// </summary>
        /// <param name="channel">A PCANTP channel handle representing a POBDonUDS channel</param>
        /// <param name="nai">Network Addressing Information</param>
        /// <param name="out_msg_request">[out] Request added to the Tx Queue, to be later used with OBDonUDS_WaitForServiceXXX</param>
        /// <param name="functional_group_identifier">FunctionalGroupIdentifier (see OBDONUDS_EMISSION_SYSTEM_GROUP)</param>
        /// <param name="DTC_status_mask">DTCStatusMask (see OBDONUDS_DTC_STATUS_PENDING)</param>
        /// <param name="DTC_severity_mask">DTCSeverityMask (see OBDONUDS_DTC_SEVERITY_CLASS_1)</param>
        /// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success.</returns>
        [DllImport("PCAN-OBDonUDS.dll", EntryPoint = "OBDonUDS_RequestPendingTroubleCodes")]
        public static extern obd_status RequestPendingTroubleCodes(
            [MarshalAs(UnmanagedType.U4)]
            cantp_handle channel,
            obd_netaddrinfo nai,
           out obd_msg out_msg_request,
            byte functional_group_identifier,
            byte DTC_status_mask,
            byte DTC_severity_mask);

        /// <summary>
        /// Parse a response to requested service 19-42 (Request Emission-related Diagnostic Trouble Codes with Pending Status)
        /// </summary>
        /// <remarks>The result must be freed afterwards by calling OBDonUDS_ParsedResponseFree</remarks>
        /// <param name="msg_response">Response to parse</param>
        /// <param name="out_buffer">[out] Parsed result</param>
        /// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success.</returns>
        [DllImport("PCAN-OBDonUDS.dll", EntryPoint = "OBDonUDS_ParseResponse_RequestPendingTroubleCodes")]
        public static extern obd_status ParseResponse_RequestPendingTroubleCodes(
            [In] ref obd_msg msg_response,
            out obd_request_trouble_codes_response out_buffer);
        #endregion
       
        #region service 31

        /// <summary>
        /// Send a request to service 31 (Request Control of On-Board System, Test, or Component)
        /// </summary>
        /// <param name="channel">A PCANTP channel handle representing a POBDonUDS channel</param>
        /// <param name="nai">Network Addressing Information</param>
        /// <param name="out_msg_request">[out] Request added to the Tx Queue, to be later used with OBDonUDS_WaitForServiceXXX</param>
        /// <param name="routine_control_type">Routine Control Type (see OBDONUDS_ROUTINE_START)</param>
        /// <param name="routine_identifier">RID</param>
        /// <param name="routine_control_options">RoutineControlOptionRecord, array of routine_control_options_len routineControlOption (byte) (may be NULL)</param>
        /// <param name="routine_control_options_len">Length of routine_control_options (may be 0)</param>
        /// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success.</returns>
        [DllImport("PCAN-OBDonUDS.dll", EntryPoint = "OBDonUDS_RequestControlOperation")]
        public static extern obd_status RequestControlOperation(
            [MarshalAs(UnmanagedType.U4)]
            cantp_handle channel,
            obd_netaddrinfo nai,
            out obd_msg out_msg_request,
            byte routine_control_type,
            obd_RID_t routine_identifier,
            [MarshalAs(UnmanagedType.LPArray, SizeParamIndex = 6)]
            byte[] routine_control_options,
            byte routine_control_options_len);

        /// <summary>
        /// Parse a response to requested service 31 (Request Control of On-Board System, Test, or Component)
        /// </summary>
        /// <remarks>The result must be freed afterwards by calling OBDonUDS_ParsedResponseFree</remarks>
        /// <param name="msg_response">Response to parse</param>
        /// <param name="out_buffer">[out] Parsed result</param>
        /// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success.</returns>
        [DllImport("PCAN-OBDonUDS.dll", EntryPoint = "OBDonUDS_ParseResponse_RequestControlOperation")]
        public static extern obd_status ParseResponse_RequestControlOperation(
            ref obd_msg msg_response,
            out obd_request_control_operation_response out_buffer);

        #endregion
       
        #region service 22-ITID

        /// <summary>
        /// Send a request to service 22 DID (Request Vehicle Information)
        /// </summary>
        /// <param name="channel">A PCANTP channel handle representing a POBDonUDS channel</param>
        /// <param name="nai">Network Addressing Information</param>
        /// <param name="out_msg_request">[out] Request added to the Tx Queue, to be later used with OBDonUDS_WaitForServiceXXX</param>
        /// <param name="data_identifier">Array of data_identifier_length DIDs (ITIDs)</param>
        /// <param name="data_identifier_length">Length (number of ITIDs) of data_identifier (may be 1)</param>
        /// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success.</returns>
        [DllImport("PCAN-OBDonUDS.dll", EntryPoint = "OBDonUDS_RequestVehicleInformation")]
        public static extern obd_status RequestVehicleInformation(
         [MarshalAs(UnmanagedType.U4)]
        cantp_handle channel,
            obd_netaddrinfo nai,
            out obd_msg out_msg_request,
            [MarshalAs(UnmanagedType.LPArray, ArraySubType = UnmanagedType.U2, SizeParamIndex = 4)]
            obd_DID_t[] data_identifier,
            UInt32 data_identifier_length);

        /// <summary>
        /// Parse a response to requested service 22 DID (Request Vehicle Information)
        /// </summary>
        /// <remarks>The result must be freed afterwards by calling OBDonUDS_ParsedResponseFree</remarks>
        /// <param name="msg_response">Response to parse</param>
        /// <param name="out_buffer">[out] Parsed result</param>
        /// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success.</returns>
        [DllImport("PCAN-OBDonUDS.dll", EntryPoint = "OBDonUDS_ParseResponse_RequestVehicleInformation")]
        public static extern obd_status ParseResponse_RequestVehicleInformation(
            [In] ref obd_msg msg_response,
            out obd_request_vehicle_information_response out_buffer);

        #endregion

        #region service 19-55

        /// <summary>
        /// Send a request to service 19-55 (Request Emission-related Diagnostic Trouble Codes with Permanent Status)
        /// </summary>
        /// <param name="channel">A PCANTP channel handle representing a POBDonUDS channel</param>
        /// <param name="nai">Network Addressing Information</param>
        /// <param name="out_msg_request">[out] Request added to the Tx Queue, to be later used with OBDonUDS_WaitForServiceXXX</param>
        /// <param name="functional_group_identifier">FunctionalGroupIdentifier (see OBDONUDS_EMISSION_SYSTEM_GROUP)</param>
        /// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success.</returns>
        [DllImport("PCAN-OBDonUDS.dll", EntryPoint = "OBDonUDS_RequestPermanentTroubleCodes")]
        public static extern obd_status RequestPermanentTroubleCodes(
            [MarshalAs(UnmanagedType.U4)]
            cantp_handle channel,
            obd_netaddrinfo nai,
            out obd_msg out_msg_request,
            byte functional_group_identifier);

        /// <summary>
        /// Parse a response to requested service 19-55 (Request Emission-related Diagnostic Trouble Codes with Permanent Status)
        /// </summary>
        /// <remarks>The result must be freed afterwards by calling OBDonUDS_ParsedResponseFree</remarks>
        /// <param name="msg_response">Response to parse</param>
        /// <param name="out_buffer">[out] Parsed result</param>
        /// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success.</returns>
        [DllImport("PCAN-OBDonUDS.dll", EntryPoint = "OBDonUDS_ParseResponse_RequestPermanentTroubleCodes")]
        public static extern obd_status ParseResponse_RequestPermanentTroubleCodes(
            [In] ref obd_msg msg_response,
            out obd_request_permanent_trouble_codes_response out_buffer);

        #endregion

        #region service 19 1A

        /// <summary>
        /// Send a request to service 19-1A (Request Supported DTCExtendedRecord Information)
        /// </summary>
        /// <param name="channel">A PCANTP channel handle representing a POBDonUDS channel</param>
        /// <param name="nai">Network Addressing Information</param>
        /// <param name="out_msg_request">[out] Request added to the Tx Queue, to be later used with OBDonUDS_WaitForServiceXXX</param>
        /// <param name="DTC_extended_DTC_data_record">DTCExtendedDTCDataRecord</param>
        /// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success.</returns>
        [DllImport("PCAN-OBDonUDS.dll", EntryPoint = "OBDonUDS_RequestSupportedDTCExtended")]
        public static extern obd_status RequestSupportedDTCExtended(
             [MarshalAs(UnmanagedType.U4)]
            cantp_handle channel,
            obd_netaddrinfo nai,
            out obd_msg out_msg_request,
            byte DTC_extended_DTC_data_record);

        /// <summary>
        /// Parse a response to requested service 19-1A (Request Supported DTCExtendedRecord Information)
        /// </summary>
        /// <remarks>The result must be freed afterwards by calling OBDonUDS_ParsedResponseFree</remarks>
        /// <param name="msg_response">Response to parse</param>
        /// <param name="out_buffer">[out] Parsed result</param>
        /// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success.</returns>
        [DllImport("PCAN-OBDonUDS.dll", EntryPoint = "OBDonUDS_ParseResponse_RequestSupportedDTCExtended")]
        public static extern obd_status ParseResponse_RequestSupportedDTCExtended(
            [In] ref obd_msg msg_response,
            out obd_request_supported_dtc_extended_response out_buffer);

        #endregion

        #region service 19 06

        /// <summary>
        /// Send a request to service 19-06 (Request DTCExtendedDataRecord)
        /// </summary>
        /// <param name="channel">A PCANTP channel handle representing a POBDonUDS channel</param>
        /// <param name="nai">Network Addressing Information</param>
        /// <param name="out_msg_request">[out] Request added to the Tx Queue, to be later used with OBDonUDS_WaitForServiceXXX</param>
        /// <param name="DTC_mask">DTCMaskRecord</param>
        /// <param name="DTC_extended_data_record_number">DTCExtDataRecordNumber</param>
        /// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success.</returns>
        [DllImport("PCAN-OBDonUDS.dll", EntryPoint = "OBDonUDS_RequestDTCExtended")]
        public static extern obd_status RequestDTCExtended(
            [MarshalAs(UnmanagedType.U4)]
            cantp_handle channel,
            obd_netaddrinfo nai,
            out obd_msg out_msg_request,
            [MarshalAs(UnmanagedType.U4)]
            obd_DTC_t DTC_mask,
            byte DTC_extended_data_record_number);

        /// <summary>
        /// Parse a response to requested service 19-06 (Request DTCExtendedDataRecord)
        /// </summary>
        /// <remarks>The result must be freed afterwards by calling OBDonUDS_ParsedResponseFree</remarks>
        /// <param name="msg_response">Response to parse</param>
        /// <param name="out_buffer">[out] Parsed result</param>
        /// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success.</returns>
        [DllImport("PCAN-OBDonUDS.dll", EntryPoint = "OBDonUDS_ParseResponse_RequestDTCExtended")]
        public static extern obd_status ParseResponse_RequestDTCExtended(
            [In] ref obd_msg msg_response,
            out obd_request_dtc_extended_response out_buffer);

        #endregion

        #region service 19 56

        /// <summary>
        /// Send a request to service 19-56 (Request DTCs for a ReadinessGroup)
        /// </summary>
        /// <param name="channel">A PCANTP channel handle representing a POBDonUDS channel</param>
        /// <param name="nai">Network Addressing Information</param>
        /// <param name="out_msg_request">[out] Request added to the Tx Queue, to be later used with OBDonUDS_WaitForServiceXXX</param>
        /// <param name="functional_group_identifier">FunctionalGroupIdentifier (see OBDONUDS_EMISSION_SYSTEM_GROUP)</param>
        /// <param name="readiness_group_identifier">ReadinessGroupIdentifier</param>
        /// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success.</returns>
        [DllImport("PCAN-OBDonUDS.dll", EntryPoint = "OBDonUDS_RequestDTCForAReadinessGroup")]
        public static extern obd_status RequestDTCForAReadinessGroup(
              [MarshalAs(UnmanagedType.U4)]
          cantp_handle channel,
            obd_netaddrinfo nai,
            out obd_msg out_msg_request,
            byte functional_group_identifier,
            byte readiness_group_identifier);

        /// <summary>
        /// Parse a response to requested service 19-56 (Request DTCs for a ReadinessGroup)
        /// </summary>
        /// <remarks>The result must be freed afterwards by calling OBDonUDS_ParsedResponseFree</remarks>
        /// <param name="msg_response">Response to parse</param>
        /// <param name="out_buffer">[out] Parsed result</param>
        /// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success.</returns>
        [DllImport("PCAN-OBDonUDS.dll", EntryPoint = "OBDonUDS_ParseResponse_RequestDTCForAReadinessGroup")]
        public static extern obd_status ParseResponse_RequestDTCForAReadinessGroup(
            [In] ref obd_msg msg_response,
            out obd_request_dtc_for_a_readiness_group_response out_buffer);

        #endregion

        #endregion

        #region special C# functions, examples of how to use the structures IntPtr fields  in safe mode, with Marshaling operations

        /// <summary>
        /// Get the byte pointed by IntPtr
        /// </summary>
        /// <param name="ptr">the IntPtr pointer to get byte value from</param>
        /// <returns>the byte value, or 0 if ptr is null</returns>
        public static Byte GetByte(IntPtr ptr)
        {
            if (ptr == IntPtr.Zero)
                return 0;

            return Marshal.ReadByte(ptr);
        }

        /// <summary>
        /// Get byte values of a obd_request_vehicle_information_response object
        /// </summary>
        /// <param name="parsed_response">object to retrieve values from</param>
        /// <param name="vals">allocated buffer where to store values</param>
        /// <param name="nb">number of values to retrieve</param>
        /// <returns>true if ok, false if not ok</returns>
        public static bool GetData(ref obd_request_vehicle_information_response parsed_response, byte[] vals, Int32 nb)
        {
            if (parsed_response.elements != IntPtr.Zero && parsed_response.nb_elements != 0 && nb <= parsed_response.nb_elements && vals.Length >= nb)
            {
                Marshal.Copy(parsed_response.elements, vals, 0, nb);
                return true;
            }
            return false;
        }

        /// <summary>
        /// Get obd_did_object values of a obd_request_current_data_response object
        /// </summary>
        /// <param name="parsed_response">object to retrieve values from</param>
        /// <param name="vals">allocated buffer where to store values</param>
        /// <param name="nb">number of values to retrieve</param>
        /// <returns>true if ok, false if not ok</returns>
        public static bool GetData(ref obd_request_current_data_response parsed_response, obd_did_object[] vals, Int32 nb)
        {
            if (parsed_response.elements != IntPtr.Zero && parsed_response.nb_elements != 0 && nb <= parsed_response.nb_elements && vals.Length >= nb)
            {
                int offset = 0;
                for (UInt32 i = 0; i < parsed_response.nb_elements; ++i)
                {
                    vals[i] = (obd_did_object)Marshal.PtrToStructure(parsed_response.elements + offset, typeof(obd_did_object));
                    offset += System.Runtime.InteropServices.Marshal.SizeOf(vals[i]);
                }
                return true;
            }
            return false;
        }

        /// <summary>
        /// Get byte values of a obd_did_object object
        /// </summary>
        /// <param name="didObject">object to retrieve values from</param>
        /// <param name="vals">allocated buffer where to store values</param>
        /// <param name="nb">number of values to retrieve</param>
        /// <returns>true if ok, false if not ok</returns>
        public static bool GetData(ref obd_did_object didObject, byte[] vals, Int32 nb)
        {
            if (didObject.data != IntPtr.Zero && didObject.size != 0 && nb <= didObject.size && vals.Length >= nb)
            {
                Marshal.Copy(didObject.data, vals, 0, nb);
                return true;
            }
            return false;
        }

        /// <summary>
        /// Get obd_severity_trouble_code values of a obd_request_trouble_codes_response object
        /// </summary>
        /// <param name="parsed_response">object to retrieve values from</param>
        /// <param name="vals">allocated buffer where to store values</param>
        /// <param name="nb">number of values to retrieve</param>
        /// <returns>true if ok, false if not ok</returns>
        public static bool GetData(ref obd_request_trouble_codes_response parsed_response, obd_severity_trouble_code[] vals, Int32 nb)
        {
            if (parsed_response.elements != IntPtr.Zero && parsed_response.nb_elements != 0 && nb <= parsed_response.nb_elements && vals.Length >= nb)
            {
                int offset = 0;
                for (UInt32 i = 0; i < parsed_response.nb_elements; ++i)
                {
                    vals[i] = (obd_severity_trouble_code)Marshal.PtrToStructure(parsed_response.elements + offset, typeof(obd_severity_trouble_code));
                    offset += System.Runtime.InteropServices.Marshal.SizeOf(vals[i]);
                }
                return true;
            }
            return false;
        }

        /// <summary>
        /// Get obd_did_object values of a obd_request_freeze_frame_data_response object
        /// </summary>
        /// <param name="parsed_response">object to retrieve values from</param>
        /// <param name="vals">allocated buffer where to store values</param>
        /// <param name="nb">number of values to retrieve</param>
        /// <returns>true if ok, false if not ok</returns>
        public static bool GetData(ref obd_request_freeze_frame_data_response parsed_response, obd_did_object[] vals, Int32 nb)
        {
            if (parsed_response.identifiers != IntPtr.Zero && parsed_response.nb_identifiers != 0 && nb <= parsed_response.nb_identifiers && vals.Length >= nb)
            {
                int offset = 0;
                for (UInt32 i = 0; i < parsed_response.nb_identifiers; ++i)
                {
                    vals[i] = (obd_did_object)Marshal.PtrToStructure(parsed_response.identifiers + offset, typeof(obd_did_object));
                    offset += System.Runtime.InteropServices.Marshal.SizeOf(vals[i]);
                }
                return true;
            }
            return false;
        }

        /// <summary>
        /// Get obd_test_data_object values of a obd_request_test_results_response object
        /// </summary>
        /// <param name="parsed_response">object to retrieve values from</param>
        /// <param name="vals">allocated buffer where to store values</param>
        /// <param name="nb">number of values to retrieve</param>
        /// <returns>true if ok, false if not ok</returns>
        public static bool GetData(ref obd_request_test_results_response parsed_response, obd_test_data_object[] vals, Int32 nb)
        {
            if (parsed_response.elements != IntPtr.Zero && parsed_response.nb_elements != 0 && nb <= parsed_response.nb_elements && vals.Length >= nb)
            {
                int offset = 0;
                for (UInt32 i = 0; i < parsed_response.nb_elements; ++i)
                {
                    vals[i] = (obd_test_data_object)Marshal.PtrToStructure(parsed_response.elements + offset, typeof(obd_test_data_object));
                    offset += System.Runtime.InteropServices.Marshal.SizeOf(vals[i]);
                }
                return true;
            }
            return false;
        }

        /// <summary>
        /// Get byte values of a obd_request_control_operation_response object
        /// </summary>
        /// <param name="parsed_response">object to retrieve values from</param>
        /// <param name="vals">allocated buffer where to store values</param>
        /// <param name="nb">number of values to retrieve</param>
        /// <returns>true if ok, false if not ok</returns>
        public static bool GetData(ref obd_request_control_operation_response parsed_response, byte[] vals, Int32 nb)
        {
            if (parsed_response.elements != IntPtr.Zero && parsed_response.nb_elements != 0 && nb <= parsed_response.nb_elements && vals.Length >= nb)
            {
                Marshal.Copy(parsed_response.elements, vals, 0, nb);
                return true;
            }
            return false;
        }

        /// <summary>
        /// Get obd_trouble_code values of a obd_request_permanent_trouble_codes_response object
        /// </summary>
        /// <param name="parsed_response">object to retrieve values from</param>
        /// <param name="vals">allocated buffer where to store values</param>
        /// <param name="nb">number of values to retrieve</param>
        /// <returns>true if ok, false if not ok</returns>
        public static bool GetData(ref obd_request_permanent_trouble_codes_response parsed_response, obd_trouble_code[] vals, Int32 nb)
        {
            if (parsed_response.elements != IntPtr.Zero && parsed_response.nb_elements != 0 && nb <= parsed_response.nb_elements && vals.Length >= nb)
            {
                int offset = 0;
                for (UInt32 i = 0; i < parsed_response.nb_elements; ++i)
                {
                    vals[i] = (obd_trouble_code)Marshal.PtrToStructure(parsed_response.elements + offset, typeof(obd_trouble_code));
                    offset += System.Runtime.InteropServices.Marshal.SizeOf(vals[i]);
                }
                return true;
            }
            return false;
        }

        /// <summary>
        /// Get obd_trouble_code values of a obd_request_supported_dtc_extended_response object
        /// </summary>
        /// <param name="parsed_response">object to retrieve values from</param>
        /// <param name="vals">allocated buffer where to store values</param>
        /// <param name="nb">number of values to retrieve</param>
        /// <returns>true if ok, false if not ok</returns>
        public static bool GetData(ref obd_request_supported_dtc_extended_response parsed_response, obd_trouble_code[] vals, Int32 nb)
        {
            if (parsed_response.elements != IntPtr.Zero && parsed_response.nb_elements != 0 && nb <= parsed_response.nb_elements && vals.Length >= nb)
            {
                int offset = 0;
                for (UInt32 i = 0; i < parsed_response.nb_elements; ++i)
                {
                    vals[i] = (obd_trouble_code)Marshal.PtrToStructure(parsed_response.elements + offset, typeof(obd_trouble_code));
                    offset += System.Runtime.InteropServices.Marshal.SizeOf(vals[i]);
                }
                return true;
            }
            return false;
        }

        /// <summary>
        /// Get byte values of a obd_request_dtc_extended_response object
        /// </summary>
        /// <param name="parsed_response">object to retrieve values from</param>
        /// <param name="vals">allocated buffer where to store values</param>
        /// <param name="nb">number of values to retrieve</param>
        /// <returns>true if ok, false if not ok</returns>
        public static bool GetData(ref obd_request_dtc_extended_response parsed_response, byte[] vals, Int32 nb)
        {
            if (parsed_response.elements != IntPtr.Zero && parsed_response.nb_elements != 0 && nb <= parsed_response.nb_elements && vals.Length >= nb)
            {
                Marshal.Copy(parsed_response.elements, vals, 0, nb);
                return true;
            }
            return false;
        }

        /// <summary>
        /// Get obd_trouble_code values of a obd_request_dtc_for_a_readiness_group_response object
        /// </summary>
        /// <param name="parsed_response">object to retrieve values from</param>
        /// <param name="vals">allocated buffer where to store values</param>
        /// <param name="nb">number of values to retrieve</param>
        /// <returns>true if ok, false if not ok</returns>
        public static bool GetData(ref obd_request_dtc_for_a_readiness_group_response parsed_response, obd_trouble_code[] vals, Int32 nb)
        {
            if (parsed_response.elements != IntPtr.Zero && parsed_response.nb_elements != 0 && nb <= parsed_response.nb_elements && vals.Length >= nb)
            {
                int offset = 0;
                for (UInt32 i = 0; i < parsed_response.nb_elements; ++i)
                {
                    vals[i] = (obd_trouble_code)Marshal.PtrToStructure(parsed_response.elements + offset, typeof(obd_trouble_code));
                    offset += System.Runtime.InteropServices.Marshal.SizeOf(vals[i]);
                }
                return true;
            }
            return false;
        }
        #endregion

        #region special C# functions, examples of how to use the structures IntPtr fields  in unsafe mode

#if (UNSAFE)

        /// <summary>
        /// Get the byte pointed by IntPtr
        /// </summary>
        /// <param name="ptr">the IntPtr pointer to get byte value from</param>
        /// <returns>the byte value, or 0 if ptr is null</returns>
        public unsafe static Byte GetByte_unsafe(IntPtr ptr)
        {
            if (ptr == IntPtr.Zero)
                return 0;

            Byte* pt = (Byte*)ptr.ToPointer();
            return *pt;
        }

        /// <summary>
        /// Get byte values of a obd_request_vehicle_information_response object
        /// </summary>
        /// <param name="parsed_response">object to retrieve values from</param>
        /// <param name="vals">allocated buffer where to store values</param>
        /// <param name="nb">number of values to retrieve</param>
        /// <returns>true if ok, false if not ok</returns>
        public unsafe static bool GetData_unsafe(ref obd_request_vehicle_information_response parsed_response, byte[] vals, Int32 nb)
        {
            if (parsed_response.elements != IntPtr.Zero)
            {
                byte* don = (byte*)parsed_response.elements.ToPointer();
                if (don != null && parsed_response.nb_elements != 0 && nb <= parsed_response.nb_elements && vals.Length >= nb)
                {
                    for (Int32 j = 0; j < nb; j++)
                        vals[j] = don[j];
                    return true;
                }
            }
            return false;
        }

        /// <summary>
        /// Get obd_did_object values of a obd_request_current_data_response object
        /// </summary>
        /// <param name="parsed_response">object to retrieve values from</param>
        /// <param name="vals">allocated buffer where to store values</param>
        /// <param name="nb">number of values to retrieve</param>
        /// <returns>true if ok, false if not ok</returns>
        public unsafe static bool GetData_unsafe(ref obd_request_current_data_response parsed_response, obd_did_object[] vals, Int32 nb)
        {
            if (parsed_response.elements != IntPtr.Zero)
            {
                obd_did_object* don = (obd_did_object*)parsed_response.elements.ToPointer();
                if (don != null && parsed_response.nb_elements != 0 && nb <= parsed_response.nb_elements && vals.Length >= nb)
                {
                    for (Int32 j = 0; j < nb; j++)
                        vals[j] = don[j];
                    return true;
                }
            }
            return false;
        }

        /// <summary>
        /// Get byte values of a obd_did_object object
        /// </summary>
        /// <param name="didObject">object to retrieve values from</param>
        /// <param name="vals">allocated buffer where to store values</param>
        /// <param name="nb"></param>
        /// <returns>number of values to retrieve</returns>
        public unsafe static bool GetData_unsafe(ref obd_did_object didObject, byte[] vals, Int32 nb)
        {
            if (didObject.data != IntPtr.Zero)
            {
                byte* don = (byte*)didObject.data.ToPointer();
                if (don != null && didObject.size != 0 && nb <= didObject.size && vals.Length >= nb)
                {
                    for (Int32 j = 0; j < nb; j++)
                        vals[j] = don[j];
                    return true;
                }
            }
            return false;
        }

#endif
        #endregion
    }
    #endregion
}