//  PCAN-OBDonUDS.h
//
//  ~~~~~~~~~~~~
//
//  PCAN-OBDonUDS API (SAE J1979-2_202104)
//
//  ~~~~~~~~~~~~
//
//  ------------------------------------------------------------------
//	Last changed by:	$Author: $
//  Last changed date:	$Date:   $
//
//  Language: ANSI-C
//  ------------------------------------------------------------------
//
//  Copyright (C) 2023  PEAK-System Technik GmbH, Darmstadt
//  more Info at http://www.peak-system.com
//
#ifndef __PCANOBDONUDS_H__
#define __PCANOBDONUDS_H__

#include "PCAN-UDS_2013.h"

////////////////////////////////////////////////////////////
// Enums definition for OBDonUDS API
////////////////////////////////////////////////////////////

// Address of ECUs
typedef enum _obd_ecu {
	POBD_ECU_1 = PUDS_ISO_15765_4_ADDR_ECU_1,  // ECU #1
	POBD_ECU_2 = PUDS_ISO_15765_4_ADDR_ECU_2,  // ECU #2
	POBD_ECU_3 = PUDS_ISO_15765_4_ADDR_ECU_3,  // ECU #3
	POBD_ECU_4 = PUDS_ISO_15765_4_ADDR_ECU_4,  // ECU #4
	POBD_ECU_5 = PUDS_ISO_15765_4_ADDR_ECU_5,  // ECU #5
	POBD_ECU_6 = PUDS_ISO_15765_4_ADDR_ECU_6,  // ECU #6
	POBD_ECU_7 = PUDS_ISO_15765_4_ADDR_ECU_7,  // ECU #7
	POBD_ECU_8 = PUDS_ISO_15765_4_ADDR_ECU_8,  // ECU #8
} obd_ecu;

// Services IDs
typedef enum _obd_service {
	POBD_SERVICE_14 = 0x14,
	POBD_SERVICE_19 = 0x19,
	POBD_SERVICE_22 = 0x22,
	POBD_SERVICE_31 = 0x31,
} obd_service;

// Sub-functions of services
typedef enum _obd_sub_function {
	POBD_SUB_FUNCTION_04 = 0x04,
	POBD_SUB_FUNCTION_42 = 0x42,
	POBD_SUB_FUNCTION_55 = 0x55,
	POBD_SUB_FUNCTION_1A = 0x1A,
	POBD_SUB_FUNCTION_06 = 0x06,
	POBD_SUB_FUNCTION_56 = 0x56,
} obd_sub_function;

// POBDonUDS error codes (used in obd_status)
typedef enum _obd_errstatus {
	POBDONUDS_ERRSTATUS_UNSUPPORTED_ECUS = 1,
	POBDONUDS_ERRSTATUS_INVALID_REQUEST_ADDRESSING,
	POBDONUDS_ERRSTATUS_INVALID_SI,
	POBDONUDS_ERRSTATUS_INVALID_DATA_LENGTH,
	POBDONUDS_ERRSTATUS_NETWORK_ERROR,
	POBDONUDS_ERRSTATUS_NOT_PARSABLE,
	POBDONUDS_CAUTION_NRC_NOT_NULL,
} obd_errstatus;

// POBDonUDS status codes.
//
// Bits information:
//   32|  28|  24|  20|  16|  12|   8|   4|   0|
//     |    |    |    |    |    |    |    |    |
//      0000 0000 0000 0000 0000 0000 0000 0000
//     |    |    |    |    |         [0000 0000] => PCAN-ISO-TP API errors
//     |    |    |    |    |  [0 0000]           => CAN Bus status
//     |    |    |    | [00 000]                 => Networking message status
//     |    |    [0000 00]                       => PCAN-ISO-TP API extra information
//     |  [0 0000]                               => API Status
//     | [0]                                     => UDS Status flag
//     |[0]                                      => OBDonUDS status flag
//     [0]                                       => PCANBasic error flag
typedef enum _obd_status {
	POBDONUDS_STATUS_OK = PUDS_STATUS_OK,          // No error
	POBDONUDS_STATUS_NOT_INITIALIZED = PUDS_STATUS_NOT_INITIALIZED,                           // Not Initialized.
	POBDONUDS_STATUS_ALREADY_INITIALIZED = PUDS_STATUS_ALREADY_INITIALIZED,                   // Already Initialized.
	POBDONUDS_STATUS_NO_MEMORY = PUDS_STATUS_NO_MEMORY,                                       // Could not obtain memory.
	POBDONUDS_STATUS_PARAM_INVALID_TYPE = PUDS_STATUS_PARAM_INVALID_TYPE,                     // Wrong message parameters.
	POBDONUDS_STATUS_PARAM_INVALID_VALUE = PUDS_STATUS_PARAM_INVALID_VALUE,                   // Wrong message parameters.
	POBDONUDS_STATUS_UNKNOWN = PUDS_STATUS_UNKNOWN,                                           // Unknown/generic error.
	POBDONUDS_STATUS_HANDLE_INVALID = PUDS_STATUS_HANDLE_INVALID,                             // Invalid cantp_handle.	
	// OBDonUDS status
	POBDONUDS_STATUS_FLAG_OBDONUDS_ERROR = 0x40 << PCANTP_STATUS_OFFSET_UDS, // OBDonUDS error flag.
	POBDONUDS_STATUS_MASK_OBDONUDS_ERROR = 0x5F << PCANTP_STATUS_OFFSET_UDS, // Filter OBDonUDS error.
	POBDONUDS_STATUS_UNSUPPORTED_ECUS = POBDONUDS_STATUS_FLAG_OBDONUDS_ERROR | (POBDONUDS_ERRSTATUS_UNSUPPORTED_ECUS << PCANTP_STATUS_OFFSET_UDS), // No OBDonUDS ECU found
	POBDONUDS_STATUS_INVALID_REQUEST_ADDRESSING = POBDONUDS_STATUS_FLAG_OBDONUDS_ERROR | (POBDONUDS_ERRSTATUS_INVALID_REQUEST_ADDRESSING << PCANTP_STATUS_OFFSET_UDS), // Invalid request addressing mode
	POBDONUDS_STATUS_INVALID_SI = POBDONUDS_STATUS_FLAG_OBDONUDS_ERROR | (POBDONUDS_ERRSTATUS_INVALID_SI << PCANTP_STATUS_OFFSET_UDS), // Invalid Service ID found while parsing (null pointer or bad value)
	POBDONUDS_STATUS_INVALID_DATA_LENGTH = POBDONUDS_STATUS_FLAG_OBDONUDS_ERROR | (POBDONUDS_ERRSTATUS_INVALID_DATA_LENGTH << PCANTP_STATUS_OFFSET_UDS), // Invalid length found while parsing
	POBDONUDS_STATUS_NETWORK_ERROR = POBDONUDS_STATUS_FLAG_OBDONUDS_ERROR | (POBDONUDS_ERRSTATUS_NETWORK_ERROR << PCANTP_STATUS_OFFSET_UDS), // Invalid Network result found while parsing (null pointer or value indicating an error)
	POBDONUDS_STATUS_NOT_PARSABLE = POBDONUDS_STATUS_FLAG_OBDONUDS_ERROR | (POBDONUDS_ERRSTATUS_NOT_PARSABLE << PCANTP_STATUS_OFFSET_UDS), // The requested DID/MID/RID/ITID corresponds to a request of type "get supported identifiers", the response is not parsable
	POBDONUDS_STATUS_CAUTION_NRC_NOT_NULL = POBDONUDS_STATUS_FLAG_OBDONUDS_ERROR | (POBDONUDS_CAUTION_NRC_NOT_NULL << PCANTP_STATUS_OFFSET_UDS), // NRC not null found while parsing (Caution indicating that the response is a negative response)
} obd_status;

// POBDonUDS parameters to be read or set
typedef enum _obd_parameter {
	POBDONUDS_PARAMETER_API_VERSION = 0x301,				// (R) char(byte) array describing the api version as a null-terminated string
	POBDONUDS_PARAMETER_AVAILABLE_ECUS = 0x302,				// (R) 1 byte data describing the number of OBDonUDS ECUs detected
	POBDONUDS_PARAMETER_SUPPORTMASK_DIDS_F4XX = 0x303,      // (R) 256 bytes describing supported DIDs F4XX for Service 22: Request Current Powertrain Diagnostic Data
	POBDONUDS_PARAMETER_SUPPORTMASK_DIDS_F5XX = 0x304,      // (R) 256 bytes describing supported DIDs F5XX for Service 22: Request Current Powertrain Diagnostic Data
	POBDONUDS_PARAMETER_SUPPORTMASK_DIDS_F7XX = 0x305,      // (R) 256 bytes describing supported DIDs F7XX for Service 22: Request Current Powertrain Diagnostic Data
	POBDONUDS_PARAMETER_SUPPORTMASK_MIDS = 0x306,           // (R) 256 bytes describing supported MIDs F6XX (DIDs) for Service 22: Request On-board Monitoring Test Results for Specific Monitored Systems
	POBDONUDS_PARAMETER_SUPPORTMASK_RIDS_E0XX = 0x307,      // (R) 256 bytes describing supported RIDs or TIDs E0XX for Service 31: Request Control of On-Board System, Test, or Component
	POBDONUDS_PARAMETER_SUPPORTMASK_RIDS_E1XX = 0x308,      // (R) 256 bytes describing supported RIDs or TIDs E1XX for Service 31: Request Control of On-Board System, Test, or Component
	POBDONUDS_PARAMETER_SUPPORTMASK_ITIDS = 0x309,			// (R) 256 bytes describing supported InfoTypes(ITIDs) F8XX for Service 22: Request Vehicle Information
	POBDONUDS_PARAMETER_DEBUG = 0x30A,						// (R/W) 1 byte data describing the debug mode  (use POBDONUDS_DEBUG_LVL_ values)
	POBDONUDS_PARAMETER_BAUDRATE = 0x30B,					// (R) obd_baudrate data describing baudrate value(see values OBD_BAUDRATE_xxx)
	POBDONUDS_PARAMETER_CAN_ID_LENGTH = 0x30C				// (R) obd_msprotocol data describing CAN identifier length (see values OBD_MSGPROTOCOL_xxx)
} obd_parameter;

// POBDonUDS baud rates
typedef enum _obd_baudrate {
	OBD_BAUDRATE_NON_LEGISLATED = 0,			// Non legislated-OBD baudrate
	OBD_BAUDRATE_250K = PCANTP_BAUDRATE_250K,	// Baudrate 250 kBit/s
	OBD_BAUDRATE_500K = PCANTP_BAUDRATE_500K,	// Baudrate 500 kBit/s
	OBD_BAUDRATE_AUTODETECT = 0xFFFFFFFFU		// Autodetect baudrate mechanism
} obd_baudrate;

// POBDonUDS message protocols
typedef enum _obd_msgprotocol {
	OBD_MSGPROTOCOL_UNKNOWN = PCANTP_ISOTP_FORMAT_NONE,			// Invalid/undefined value
	OBD_MSGPROTOCOL_11BIT = PCANTP_ISOTP_FORMAT_NORMAL,			// 11bit CAN Identifier
	OBD_MSGPROTOCOL_29BIT = PCANTP_ISOTP_FORMAT_FIXED_NORMAL,	// 29bit CAN Identifier
} obd_msgprotocol;

// POBDonUDS addressing modes
typedef enum _obd_addressing {
	OBD_ADDRESSING_UNKNOWN = PCANTP_ISOTP_ADDRESSING_UNKNOWN,
	OBD_ADDRESSING_PHYSICAL = PCANTP_ISOTP_ADDRESSING_PHYSICAL,
	OBD_ADDRESSING_FUNCTIONAL = PCANTP_ISOTP_ADDRESSING_FUNCTIONAL
} obd_addressing;

// Reserved enumeration of internal parsed responses types
typedef enum _obd_datatype {
	OBD_DATATYPE_UNKNOWN = 0,
	OBD_DATATYPE_REQUEST_CURRENT_DATA_RESPONSE,
	OBD_DATATYPE_REQUEST_FREEZE_FRAME_DATA_RESPONSE,
	OBD_DATATYPE_REQUEST_CONTROL_OPERATION_RESPONSE,
	OBD_DATATYPE_REQUEST_VEHICLE_INFORMATION_RESPONSE,
	OBD_DATATYPE_REQUEST_TROUBLECODES_RESPONSE,
	OBD_DATATYPE_REQUEST_PERMANENT_TROUBLECODES_RESPONSE,
	OBD_DATATYPE_REQUEST_READINESS_GROUP_TROUBLECODES_RESPONSE,
	OBD_DATATYPE_REQUEST_SUPPORTED_DTC_EXTENDED_RESPONSE,
	OBD_DATATYPE_REQUEST_CLEAR_TROUBLE_CODES_RESPONSE,
	OBD_DATATYPE_REQUEST_TEST_RESULTS_RESPONSE,
	OBD_DATATYPE_REQUEST_DTC_EXTENDED_RESPONSE

} obd_datatype;

////////////////////////////////////////////////////////////
// Types definitions
////////////////////////////////////////////////////////////
// DID value on 2 bytes
typedef uint16_t obd_DID_t;
// DTC value on 4 bytes 
typedef uint32_t obd_DTC_t;
// RID value on 2 bytes
typedef uint16_t obd_RID_t;

////////////////////////////////////////////////////////////
// Message definitions
////////////////////////////////////////////////////////////

#pragma pack(push, 8)

// Represents a OBDonUDS Network Addressing Information
typedef struct _obd_netaddrinfo {
	obd_msgprotocol protocol;	// Communication protocol
	obd_addressing target_type;	// ISO-TP target type (physical or functional)
	uint16_t source_addr;		// Source address (by default should be PUDS_ISO_15765_4_ADDR_TEST_EQUIPMENT for requests)
	uint16_t target_addr;		// Target address (see POBD_ECUxx)
} obd_netaddrinfo;

// A constant NAI for functional 11bit requests
const obd_netaddrinfo OBD_NAI_REQUEST_FUNCTIONAL_11B = {
	OBD_MSGPROTOCOL_11BIT,
	OBD_ADDRESSING_FUNCTIONAL,
	PUDS_ISO_15765_4_ADDR_TEST_EQUIPMENT,
	PUDS_ISO_15765_4_ADDR_OBD_FUNCTIONAL
};

// A constant NAI for functional 29bit requests
const obd_netaddrinfo OBD_NAI_REQUEST_FUNCTIONAL_29B = {
	OBD_MSGPROTOCOL_29BIT,
	OBD_ADDRESSING_FUNCTIONAL,
	PUDS_ISO_15765_4_ADDR_TEST_EQUIPMENT,
	PUDS_ISO_15765_4_ADDR_OBD_FUNCTIONAL
};

// Provides accessors to the corresponding data in the cantp_msg
typedef struct _obd_msgaccess {
	uint8_t* service_id;				// Pointer to the Service ID in message's data.
	uint8_t* nrc;						// Pointer to the Negative Response Code (see uds_nrc enumeration) in message's data (NULL on positive response).
	cantp_netstatus* network_result;	// Pointer to the ISOTP network result.
} obd_msgaccess;

// The OBDonUDS message
typedef struct _obd_msg {
	obd_netaddrinfo nai;
	obd_msgaccess links;
	uds_msg msg;
} obd_msg;

// All services :
// Internal reserved structure for parsed responses
typedef struct _obd_response_generic {
	obd_datatype reserved_object_type;
	uint32_t reserved_object_size;
	uint16_t ecu_address;
	uint8_t nrc;
} obd_response_generic;

// service 22-DID

// Part of a parsed response to service 22-DID
typedef struct _obd_did_object {
	obd_DID_t data_identifier;								// DID
	uint32_t size;											// size (number of bytes) of data
	uint8_t* data;											// array of bytes representing data dedicated to data_identifier
} obd_did_object;

// Parsed response to service 22-DID
typedef struct _obd_request_current_data_response {
	obd_datatype reserved_object_type;						// reserved
	uint32_t reserved_object_size;							// reserved
	uint16_t ecu_address;									// ECU address
	uint8_t nrc;											// NRC (0 if no NRC)
	uint32_t nb_elements;									// number of objects in elements
	obd_did_object* elements;								// array of objects
} obd_request_current_data_response;

// service 19-04

// Parsed response to service service 19-04
typedef struct _obd_request_freeze_frame_data_response {
	obd_datatype reserved_object_type;						// reserved
	uint32_t reserved_object_size;							// reserved
	uint16_t ecu_address;									// ECU address
	uint8_t nrc;											// NRC (0 if no NRC)
	uint8_t report_type;									// reportType
	obd_DTC_t dtc_number;									// DTC
	uint8_t status_of_dtc;									// statusOfDTC
	uint8_t record_number;									// DTCSnapshotRecordNumber
	uint8_t nb_identifiers;									// DTCSnapshotRecordNumberOfIdentifiers, number of objects in identifiers
	obd_did_object* identifiers;						    // array of objects (DTCSnapshotRecordDataRecord)
} obd_request_freeze_frame_data_response;

// service 19-42

// Part of a parsed response to service 19-42
typedef struct _obd_severity_trouble_code {
	uint8_t DTC_severity;									// DTCSeverity
	obd_DTC_t DTC_identifier;								// DTC
	uint8_t status_of_dtc;									// statusOfDTC
} obd_severity_trouble_code;

// Parsed response to service 19-42
typedef struct _obd_request_trouble_codes_response {
	obd_datatype reserved_object_type;						// reserved
	uint32_t reserved_object_size;							// reserved
	uint16_t ecu_address;									// ECU address
	uint8_t nrc;											// NRC (0 if no NRC)
	uint8_t report_type;									// DTCReportType
	uint8_t functional_group_identifier;					// FunctionalGroupIdentifier
	uint8_t DTC_status_availability_mask;					// DTCStatusAvailabilityMask
	uint8_t DTC_severity_mask;								// DTCSeverityAvailabilityMask
	uint8_t DTC_format_identifier;							// DTCFormatIdentifier
	uint32_t nb_elements;									// Number of objects in elements
	obd_severity_trouble_code* elements;					// Array of objects (DTCAndSeverityRecord)
} obd_request_trouble_codes_response;

// service 14

// Parsed response to service service 14
typedef struct _obd_request_clear_trouble_codes_response {
	obd_datatype reserved_object_type;						// reserved
	uint32_t reserved_object_size;							// reserved
	uint16_t ecu_address;									// ECU address
	uint8_t nrc;											// NRC (0 if no NRC)
} obd_request_clear_trouble_codes_response;

// service 22-MID

// Part of a parsed response to service 22-MID
typedef struct _obd_test_data_object {
	uint8_t test_identifier;								// TID
	uint8_t unit_and_scaling;								// Unit and scaling ID
	uint8_t test_value[2];									// Test Value
	uint8_t min_test_limit[2];								// Min. Test Limit
	uint8_t max_test_limit[2];								// Max. Test Limit
} obd_test_data_object;

// Parsed response to service 22-MID
typedef struct _obd_request_test_results_response {
	obd_datatype reserved_object_type;						// reserved
	uint32_t reserved_object_size;							// reserved
	uint16_t ecu_address;									// ECU address
	uint8_t nrc;											// NRC (0 if no NRC)
	obd_DID_t data_identifier;								// DID
	uint32_t nb_elements;									// Number of objects in elements
	obd_test_data_object* elements;							// Array of objects (data record of supported MID)
} obd_request_test_results_response;

// service 31

// Parsed response to service service 31
typedef struct _obd_request_control_operation_response {
	obd_datatype reserved_object_type;						// reserved
	uint32_t reserved_object_size;							// reserved
	uint16_t ecu_address;									// ECU address
	uint8_t nrc;											// NRC (0 if no NRC)
	uint8_t routine_control_type;							// Routine Control Type
	obd_RID_t routine_identifier;							// RID
	uint8_t routine_info;									// routineInfo
	uint8_t nb_elements;									// Number of bytes in elements
	uint8_t* elements;										// Array of bytes (routineStatusRecord)
} obd_request_control_operation_response;

// service 22-ITID

// Parsed response to service service 22-ITID
typedef struct _obd_request_vehicle_information_response {
	obd_datatype reserved_object_type;						// reserved
	uint32_t reserved_object_size;							// reserved
	uint16_t ecu_address;									// ECU address
	uint8_t nrc;											// NRC (0 if no NRC)
	obd_DID_t data_identifier;								// DID (ITID) (DID in data record of InfoType)
	uint32_t nb_elements;									// Number of bytes in elements
	uint8_t* elements;										// Array of bytes (data in data record of InfoType)
} obd_request_vehicle_information_response;

// service 19-55

// Part of a parsed response to service 19-55
typedef struct _obd_trouble_code {
	obd_DTC_t DTC_identifier;								// DTC in DTCAndStatusRecord
	uint8_t status_of_dtc;									// statusOfDTC
} obd_trouble_code;

// Parsed response to service 19-55
typedef struct _obd_request_permanent_trouble_codes_response {
	obd_datatype reserved_object_type;						// reserved
	uint32_t reserved_object_size;							// reserved
	uint16_t ecu_address;									// ECU address
	uint8_t nrc;											// NRC (0 if no NRC)
	uint8_t report_type;									// DTCReportType
	uint8_t functional_group_identifier;					// FunctionalGroupIdentifier
	uint8_t DTC_status_availability_mask;					// DTCStatusAvailabilityMask
	uint8_t DTC_format_identifier;							// DTCFormatIdentifier
	uint32_t nb_elements;									// Number of objects in elements
	obd_trouble_code* elements;								// Array of objects (DTCAndStatusRecord)
} obd_request_permanent_trouble_codes_response;

// service 19-1A

// Parsed response to service service 19-1A
typedef struct _obd_request_supported_dtc_extended_response {
	obd_datatype reserved_object_type;						// reserved
	uint32_t reserved_object_size;							// reserved
	uint16_t ecu_address;									// ECU address
	uint8_t nrc;											// NRC (0 if no NRC)
	uint8_t report_type;									// DTCReportType
	uint8_t DTC_status_availability_mask;					// DTCStatusAvailabilityMask
	uint8_t DTC_extended_data_record_number;				// DTCExtendedDataRecordNumber
	uint32_t nb_elements;									// Number of objects in elements
	obd_trouble_code* elements;								// Array of objects (DTCAndStatusRecord)
} obd_request_supported_dtc_extended_response;

// service 19-06

// Parsed response to service service 19-06
typedef struct _obd_request_dtc_extended_response {
	obd_datatype reserved_object_type;						// reserved
	uint32_t reserved_object_size;							// reserved
	uint16_t ecu_address;									// ECU address
	uint8_t nrc;											// NRC (0 if no NRC)
	uint8_t report_type;									// DTCReportType
	obd_DTC_t DTC_identifier;								// DTC in DTCAndStatusRecord
	uint8_t status_of_dtc;									// statusOfDTC
	uint8_t DTC_extended_data_record_number;				// DTCExtendedDataRecordNumber
	uint32_t nb_elements;									// Number of bytes in elements
	uint8_t* elements;										// Array of bytes (DTCExtendedDataRecord)
} obd_request_dtc_extended_response;

// service 19-56

// parsed response to service service 19-56
typedef struct _obd_request_dtc_for_a_readiness_group_response {
	obd_datatype reserved_object_type;						// reserved
	uint32_t reserved_object_size;							// reserved
	uint16_t ecu_address;									// ECU address
	uint8_t nrc;											// NRC (0 if no NRC)
	uint8_t report_type;									// DTCReportType
	uint8_t functional_group_identifier;					// FunctionalGroupIdentifier
	uint8_t DTC_status_availability_mask;					// DTCStatusAvailabilityMask
	uint8_t DTC_format_identifier;							// DTCFormatIdentifier
	uint8_t readiness_group_identifier;						// DTCReadinessGroupIdentifier
	uint32_t nb_elements;									// Number of objects in elements
	obd_trouble_code* elements;								// Array of objects (DTCAndStatusRecord)
} obd_request_dtc_for_a_readiness_group_response;

#pragma pack(pop)

////////////////////////////////////////////////////////////
// Constants definitions
////////////////////////////////////////////////////////////

#define OBDONUDS_EMISSION_SYSTEM_GROUP 0x33
#define OBDONUDS_DTC_SEVERITY_CLASS_1 0x02
#define OBDONUDS_DTC_STATUS_CONFIRMED 0x08
#define OBDONUDS_DTC_STATUS_PENDING 0x04
#define OBDONUDS_ROUTINE_START 0x01
#define OBDONUDS_DTC_SNAPSHOT_RECORD_FIRST 0x00
#define OBDONUDS_DTC_SNAPSHOT_RECORD_LAST 0xF0
#define OBDONUDS_DTC_EMISSION_SYSTEM_GROUP 0xFFFF33
#define OBDONUDS_DTC_ALL_GROUPS 0xFFFFFF


// POBDONUDS parameter values
#define POBDONUDS_DEBUG_LVL_NONE         0x00    // Disable debug messages (default)
#define POBDONUDS_DEBUG_LVL_ERROR        0xF1    // Enable debug messages (only errors)
#define POBDONUDS_DEBUG_LVL_WARNING      0xF2    // Enable debug messages (only warnings, errors)
#define POBDONUDS_DEBUG_LVL_INFORMATION  0xF3    // Enable debug messages (only informations, warnings, errors)
#define POBDONUDS_DEBUG_LVL_NOTICE       0xF4    // Enable debug messages (only notices, informations, warnings, errors)
#define POBDONUDS_DEBUG_LVL_DEBUG        0xF5    // Enable debug messages (only debug, notices, informations, warnings, errors)
#define POBDONUDS_DEBUG_LVL_TRACE        0xF6    // Enable all debug messages

////////////////////////////////////////////////////////////
// PCAN-OBDonUDS API: Core function declarations
////////////////////////////////////////////////////////////


#ifdef __cplusplus

// extra operator for obd_status
inline obd_status  operator|   (obd_status a, obd_status b) { return static_cast<obd_status>(static_cast<int>(a) | static_cast<int>(b)); }
inline obd_status  operator&   (obd_status a, obd_status b) { return static_cast<obd_status>(static_cast<int>(a) & static_cast<int>(b)); }
inline obd_status& operator&=  (obd_status& a, obd_status b) { return (a = (a & b)); }
inline obd_status  operator~   (obd_status a) { return static_cast<obd_status>(~static_cast<int>(a)); }

extern "C" {
#define _DEF_ARG =0
#define _DEF_ARG_BAUDRATE =static_cast<cantp_baudrate>(OBD_BAUDRATE_AUTODETECT)
#define _DEF_ARG_NULL =nullptr
#ifndef _DEF_ARG_HW
#define _DEF_ARG_HW =static_cast<cantp_hwtype>(0)
#endif
#define _DEF_ARG_OBDONUDS_OK =POBDONUDS_STATUS_OK
#else
#define _DEF_ARG
#define _DEF_ARG_BAUDRATE
#define _DEF_ARG_NULL
#define _DEF_ARG_HW
#define _DEF_ARG_OBDONUDS_OK
#endif

/// <summary>
/// Initializes an OBDonUDS-Client based on a PCAN-ISO-TP channel
/// </summary>
/// <remarks>Only one OBDonUDS-Client can be initialized per PCAN-ISO-TP-Channel</remarks>
/// <param name="channel">The PCAN-ISO-TP channel to be used as OBDonUDS client</param>
/// <param name="baudrate">The CAN Hardware speed</param>
/// <param name="hw_type">NON PLUG&PLAY: The type of hardware and operation mode</param>
/// <param name="io_port">NON PLUG&PLAY: The I/O address for the parallel port</param>
/// <param name="interupt">NON PLUG&PLAY: Interrupt number of the parallel port</param>
/// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success</returns>
obd_status __stdcall OBDonUDS_Initialize(
	cantp_handle channel,
	cantp_baudrate baudrate _DEF_ARG_BAUDRATE,
	cantp_hwtype hw_type _DEF_ARG_HW,
	uint32_t io_port _DEF_ARG,
	uint16_t interrupt _DEF_ARG);

/// <summary>
/// Tests a channel to detect ECUs responding in OBDonEDS-mode.
/// </summary>
/// <remarks>This function must be called on a closed channel and leaves the channel closed</remarks>
/// <param name="channel">The PCAN-ISO-TP channel to be used as OBDonUDS client</param>
/// <param name="baudrate">The CAN Hardware speed</param>
/// <param name="hw_type">NON PLUG&PLAY: The type of hardware and operation mode</param>
/// <param name="io_port">NON PLUG&PLAY: The I/O address for the parallel port</param>
/// <param name="Interupt">NON PLUG&PLAY: Interrupt number of the parallel port</param>
/// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success</returns>
obd_status __stdcall OBDonUDS_FindOBDonEDS(cantp_handle channel,
	cantp_baudrate baudrate _DEF_ARG_BAUDRATE,
	cantp_hwtype hw_type _DEF_ARG_HW,
	uint32_t io_port _DEF_ARG,
	uint16_t interrupt _DEF_ARG);

/// <summary>
/// Uninitializes an OBDonUDS-Client initialized before
/// </summary>
/// <param name="channel">A PCANTP channel handle representing a OBDonUDS-Client</param>
/// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success</returns>
obd_status __stdcall OBDonUDS_Uninitialize(
	cantp_handle channel);

/// <summary>
/// Resets the receive and transmit queues of an OBDonUDS-Client 
/// </summary>
/// <param name="channel">A PCANTP channel handle representing a OBDonUDS-Client</param>
/// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success</returns>
obd_status __stdcall OBDonUDS_Reset(
	cantp_handle channel);

/// <summary>
/// Gets the initialization status of an OBDonUDS CAN-Channel
/// </summary>
/// <param name="channel">A PCANTP channel handle representing a OBDonUDS-Client</param>
/// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success</returns>
obd_status __stdcall OBDonUDS_GetStatus(
	cantp_handle channel);

/// <summary>
/// Checks if a status matches an expected result (default is POBDONUDS_STATUS_OK).
/// </summary>
/// <param name="status">The status to analyze.</param>
/// <param name="status_expected">The expected status (default is POBDONUDS_STATUS_OK).</param>
/// <param name="strict_mode">Enable strict mode (default is false). Strict mode ensures that bus or extra information are the same.</param>
/// <returns>Returns true if the status matches expected parameter.</returns>
bool __stdcall OBDonUDS_StatusIsOk(
	const obd_status status,
	const obd_status status_expected _DEF_ARG_OBDONUDS_OK,
	bool strict_mode _DEF_ARG);

/// <summary>
/// Returns a descriptive text of a given obd_status error
/// code, in any desired language
/// </summary>
/// <remarks>The current languages available for translation are:
/// Neutral (0x00), English (0x09), and French (0x0C)</remarks>
/// <param name="error_code">A obd_status error code</param>
/// <param name="language">Indicates a 'Primary language ID'</param>
/// <param name="buffer">Buffer for a null terminated char array</param>
/// <param name="buffer_size">Buffer size</param>
/// <returns>A obd_status error code</returns>
obd_status __stdcall OBDonUDS_GetErrorText(
	obd_status error_code,
	uint16_t language,
	char* buffer,
	uint32_t buffer_size);

/// <summary>
/// Retrieves an OBDonUDS-Client parameter value
/// </summary>
/// <param name="channel">A PCANTP channel handle representing a OBDonUDS client</param>
/// <param name="parameter">The parameter to get</param>
/// <param name="buffer">Buffer for the parameter value</param>
/// <param name="buffer_size">Size in bytes of the buffer</param>
/// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success</returns>
obd_status __stdcall OBDonUDS_GetValue(
	cantp_handle channel,
	obd_parameter parameter,
	void* buffer,
	uint32_t buffer_size);

/// <summary>
/// Configures or sets an OBDonUDS-Client parameter value
/// </summary>
/// <param name="channel">A PCANTP channel handle representing an OBDonUDS channel</param>
/// <param name="parameter">The parameter to set</param>
/// <param name="buffer">Buffer for the parameter value</param>
/// <param name="buffer_size">Size in bytes of the buffer</param>
/// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success</returns>
obd_status __stdcall OBDonUDS_SetValue(
	cantp_handle channel,
	obd_parameter parameter,
	void* buffer,
	uint32_t buffer_size);

/// <summary>
/// Deallocates a OBDonUDS message
/// </summary>
/// <param name="msg_buffer">An allocated obd_msg structure buffer</param>
/// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success</returns>
obd_status __stdcall OBDonUDS_MsgFree(
	obd_msg* msg_buffer);


/// <summary>
/// Handles the communication workflow for a OBDonUDS service expecting a single response.
/// </summary>
/// <param name="channel">A PCANTP channel handle representing a POBDonUDS channel</param>
/// <param name="msg_request">A sent obd_msg message used as a reference to manage the OBDonUDS service</param>
/// <param name="out_msg_response">[out] A obd_msg structure buffer to store the POBDonUDS response</param>
/// <param name="out_msg_request_confirmation">[out] A obd_msg structure buffer to store the POBDonUDS request confirmation</param>
/// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success</returns>
obd_status __stdcall OBDonUDS_WaitForService(
	cantp_handle channel,
	obd_msg* msg_request,
	obd_msg* out_msg_response,
	obd_msg* out_msg_request_confirmation);

/// <summary>
/// Handles the communication workflow for a OBDonUDS service expecting multiple responses.
/// </summary>
/// <param name="channel">A PCANTP channel handle representing a POBDonUDS channel</param>
/// <param name="msg_request">A sent obd_msg message used as a reference to manage the OBDonUDS service</param>
/// <param name="max_msg_count">Length of the buffer array (max. messages that can be received)</param>
/// <param name="wait_until_timeout">if <code>FALSE</code> the function is interrupted if out_msg_count reaches max_msg_count.</param>
/// <param name="out_msg_responses">[out] A buffer to store the responses. It must be an array of "max_msg_count" entries (it must have at least
/// a size of max_msg_count * sizeof(obd_msg) bytes) </param>
/// <param name="out_msg_count">[out] Actual number of messages read</param>
/// <param name="out_msg_request_confirmation">[out] A obd_msg structure buffer to store the POBDonUDS request confirmation</param>
/// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success,
/// PUDS_ERROR_OVERFLOW indicates success but buffer was too small to hold all responses.</returns>
obd_status __stdcall OBDonUDS_WaitForServiceFunctional(
	cantp_handle channel,
	obd_msg* msg_request,
	uint32_t max_msg_count,
	bool wait_until_timeout,
	obd_msg* out_msg_responses,
	uint32_t* out_msg_count,
	obd_msg* out_msg_request_confirmation);

/// <summary>
/// Deallocates a parsed response (see structures obd_request_XXX_response)
/// </summary>
/// <remarks>For all services, this function is to use after calls to functions OBDonUDS_ParseXXXX</remarks>
/// <param name="response">The parsed response to free</param>
/// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success.</returns>
obd_status __stdcall  OBDonUDS_ParsedResponseFree(obd_response_generic* response);

// service 22-DID

/// <summary>
/// Send a request to service 22-DID (Request Current Powertrain Diagnostic Data)
/// </summary>
/// <param name="channel">A PCANTP channel handle representing a POBDonUDS channel</param>
/// <param name="nai">Network Addressing Information</param>
/// <param name="out_msg_request">[out] Request added to the Tx Queue, to be later used with OBDonUDS_WaitForServiceXXX</param>
/// <param name="data_identifier">Array of data_identifier_length DIDs</param>
/// <param name="data_identifier_length">Length (number of DIDs) of the data_identifier array</param>
/// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success.</returns>
obd_status __stdcall OBDonUDS_RequestCurrentData(
	cantp_handle channel,
	obd_netaddrinfo nai,
	obd_msg* out_msg_request,
	obd_DID_t* data_identifier,
	uint32_t data_identifier_length);

/// <summary>
/// Parse a response to requested service 22-DID (Request Current Powertrain Diagnostic Data)
/// </summary>
/// <remarks>The result must be freed afterwards by calling OBDonUDS_ParsedResponseFree</remarks>
/// <param name="msg_response">Response to parse</param>
/// <param name="out_buffer">[out] Parsed result</param>
/// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success.</returns>
obd_status __stdcall OBDonUDS_ParseResponse_RequestCurrentData(
	obd_msg* msg_response,
	obd_request_current_data_response* out_buffer);

// service 19-04

/// <summary>
/// Send a request to service 19-04 (Request Powertrain Freeze Frame Data)
/// </summary>
/// <param name="channel">A PCANTP channel handle representing a POBDonUDS channel</param>
/// <param name="nai">Network Addressing Information</param>
/// <param name="out_msg_request">[out] Request added to the Tx Queue, to be later used with OBDonUDS_WaitForServiceXXX</param>
/// <param name="DTC_Number">DTCMaskRecord</param>
/// <param name="record_number">DTCSnapshotRecordNumber (see OBDONUDS_DTC_SNAPSHOT_RECORD_xxx)</param>
/// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success.</returns>
obd_status __stdcall OBDonUDS_RequestFreezeFrameData(
	cantp_handle channel,
	obd_netaddrinfo nai,
	obd_msg* out_msg_request,
	obd_DTC_t DTC_Number,
	uint8_t record_number);

/// <summary>
/// Parse a response to requested service 19-04 (Request Powertrain Freeze Frame Data)
/// </summary>
/// <remarks>The result must be freed afterwards by calling OBDonUDS_ParsedResponseFree</remarks>
/// <param name="msg_response">Response to parse</param>
/// <param name="out_buffer">[out] Parsed result</param>
/// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success.</returns>
obd_status __stdcall OBDonUDS_ParseResponse_RequestFreezeFrameData(
	obd_msg* msg_response,
	obd_request_freeze_frame_data_response* out_buffer);


// service 19-42

/// <summary>
/// Send a request to service 19-42 (Request Emission-Related Diagnostic Trouble Codes with Confirmed Status)
/// </summary>
/// <param name="channel">A PCANTP channel handle representing a POBDonUDS channel</param>
/// <param name="nai">Network Addressing Information</param>
/// <param name="out_msg_request">[out] Request added to the Tx Queue, to be later used with OBDonUDS_WaitForServiceXXX</param>
/// <param name="functional_group_identifier">FunctionalGroupIdentifier (see OBDONUDS_EMISSION_SYSTEM_GROUP)</param>
/// <param name="DTC_status_mask">DTCStatusMask (see OBDONUDS_DTC_STATUS_CONFIRMED)</param>
/// <param name="DTC_severity_mask">DTCSeverityMask (see OBDONUDS_DTC_SEVERITY_CLASS_1)</param>
/// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success.</returns>
obd_status __stdcall OBDonUDS_RequestConfirmedTroubleCodes(
	cantp_handle channel,
	obd_netaddrinfo nai,
	obd_msg* out_msg_request,
	uint8_t functional_group_identifier,
	uint8_t DTC_status_mask,
	uint8_t DTC_severity_mask);

/// <summary>
/// Parse a response to requested service 19-42 (Request Emission-Related Diagnostic Trouble Codes with Confirmed Status)
/// </summary>
/// <remarks>The result must be freed afterwards by calling OBDonUDS_ParsedResponseFree</remarks>
/// <param name="msg_response">Response to parse</param>
/// <param name="out_buffer">[out] Parsed result</param>
/// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success.</returns>
obd_status __stdcall OBDonUDS_ParseResponse_RequestConfirmedTroubleCodes(
	obd_msg* msg_response,
	obd_request_trouble_codes_response* out_buffer);


// service 14

/// <summary>
/// Send a request to service 14 (Clear/reset Emission-related Diagnostic Information)
/// </summary>
/// <param name="channel">A PCANTP channel handle representing a POBDonUDS channel</param>
/// <param name="nai">Network Addressing Information</param>
/// <param name="out_msg_request">[out] Request added to the Tx Queue, to be later used with OBDonUDS_WaitForServiceXXX</param>
/// <param name="group_of_DTC">GroupOfDTC</param>
/// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success.</returns>
obd_status __stdcall OBDonUDS_ClearTroubleCodes(
	cantp_handle channel,
	obd_netaddrinfo nai,
	obd_msg* out_msg_request,
	obd_DTC_t group_of_DTC);

/// <summary>
/// Parse a response to requested service 14 (Clear/reset Emission-related Diagnostic Information)
/// </summary>
/// <remarks>The result must be freed afterwards by calling OBDonUDS_ParsedResponseFree</remarks>
/// <param name="msg_response">Response to parse</param>
/// <param name="out_buffer">[out] Parsed result</param>
/// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success.</returns>
obd_status __stdcall OBDonUDS_ParseResponse_ClearTroubleCodes(
	obd_msg* msg_response,
	obd_request_clear_trouble_codes_response* out_buffer);


// service 22-MID

/// <summary>
/// Send a request to service 22-MID (Request On-board Monitoring Test Results for Specific Monitored Systems)
/// </summary>
/// <param name="channel">A PCANTP channel handle representing a POBDonUDS channel</param>
/// <param name="nai">Network Addressing Information</param>
/// <param name="out_msg_request">[out] Request added to the Tx Queue, to be later used with OBDonUDS_WaitForServiceXXX</param>
/// <param name="data_identifier">Array of data_identifier_length DIDs (MIDs)</param>
/// <param name="data_identifier_length">Length (number of MIDs) of data_identifier (may be 1)</param>
/// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success.</returns>
obd_status __stdcall OBDonUDS_RequestTestResults(
	cantp_handle channel,
	obd_netaddrinfo nai,
	obd_msg* out_msg_request,
	obd_DID_t* data_identifier,
	uint32_t data_identifier_length);

/// <summary>
/// Parse a response to requested service 22-MID (Request On-board Monitoring Test Results for Specific Monitored Systems)
/// </summary>
/// <remarks>The result must be freed afterwards by calling OBDonUDS_ParsedResponseFree</remarks>
/// <param name="msg_response">Response to parse</param>
/// <param name="out_buffer">[out] Parsed result</param>
/// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success.</returns>
obd_status __stdcall OBDonUDS_ParseResponse_RequestTestResults(
	obd_msg* msg_response,
	obd_request_test_results_response* out_buffer);

// service 19-42

/// <summary>
/// Send a request to service 19-42 (Request Emission-related Diagnostic Trouble Codes with Pending Status)
/// </summary>
/// <param name="channel">A PCANTP channel handle representing a POBDonUDS channel</param>
/// <param name="nai">Network Addressing Information</param>
/// <param name="out_msg_request">[out] Request added to the Tx Queue, to be later used with OBDonUDS_WaitForServiceXXX</param>
/// <param name="functional_group_identifier">FunctionalGroupIdentifier (see OBDONUDS_EMISSION_SYSTEM_GROUP)</param>
/// <param name="DTC_status_mask">DTCStatusMask (see OBDONUDS_DTC_STATUS_PENDING)</param>
/// <param name="DTC_severity_mask">DTCSeverityMask (see OBDONUDS_DTC_SEVERITY_CLASS_1)</param>
/// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success.</returns>
obd_status __stdcall OBDonUDS_RequestPendingTroubleCodes(
	cantp_handle channel,
	obd_netaddrinfo nai,
	obd_msg* out_msg_request,
	uint8_t functional_group_identifier,
	uint8_t DTC_status_mask,
	uint8_t DTC_severity_mask);

/// <summary>
/// Parse a response to requested service 19-42 (Request Emission-related Diagnostic Trouble Codes with Pending Status)
/// </summary>
/// <remarks>The result must be freed afterwards by calling OBDonUDS_ParsedResponseFree</remarks>
/// <param name="msg_response">Response to parse</param>
/// <param name="out_buffer">[out] Parsed result</param>
/// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success.</returns>
obd_status __stdcall OBDonUDS_ParseResponse_RequestPendingTroubleCodes(
	obd_msg* msg_response,
	obd_request_trouble_codes_response* out_buffer);

// service 31

/// <summary>
/// Send a request to service 31 (Request Control of On-Board System, Test, or Component)
/// </summary>
/// <param name="channel">A PCANTP channel handle representing a POBDonUDS channel</param>
/// <param name="nai">Network Addressing Information</param>
/// <param name="out_msg_request">[out] Request added to the Tx Queue, to be later used with OBDonUDS_WaitForServiceXXX</param>
/// <param name="routine_control_type">Routine Control Type (see OBDONUDS_ROUTINE_START)</param>
/// <param name="routine_identifier">RID</param>
/// <param name="routine_control_options">RoutineControlOptionRecord, array of routine_control_options_len routineControlOption (byte) (may be NULL)</param>
/// <param name="routine_control_options_len">Length of routine_control_options (may be 0)</param>
/// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success.</returns>
obd_status __stdcall OBDonUDS_RequestControlOperation(
	cantp_handle channel,
	obd_netaddrinfo nai,
	obd_msg* out_msg_request,		
	uint8_t routine_control_type,
	obd_RID_t routine_identifier,
	uint8_t* routine_control_options,
	uint8_t routine_control_options_len); 

/// <summary>
/// Parse a response to requested service 31 (Request Control of On-Board System, Test, or Component)
/// </summary>
/// <remarks>The result must be freed afterwards by calling OBDonUDS_ParsedResponseFree</remarks>
/// <param name="msg_response">Response to parse</param>
/// <param name="out_buffer">[out] Parsed result</param>
/// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success.</returns>
obd_status __stdcall OBDonUDS_ParseResponse_RequestControlOperation(
	obd_msg* msg_response,
	obd_request_control_operation_response* out_buffer);

// service 22-ITID

/// <summary>
/// Send a request to service 22 DID (Request Vehicle Information)
/// </summary>
/// <param name="channel">A PCANTP channel handle representing a POBDonUDS channel</param>
/// <param name="nai">Network Addressing Information</param>
/// <param name="out_msg_request">[out] Request added to the Tx Queue, to be later used with OBDonUDS_WaitForServiceXXX</param>
/// <param name="data_identifier">Array of data_identifier_length DIDs (ITIDs)</param>
/// <param name="data_identifier_length">Length (number of ITIDs) of data_identifier (may be 1)</param>
/// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success.</returns>
obd_status __stdcall OBDonUDS_RequestVehicleInformation(
	cantp_handle channel,
	obd_netaddrinfo nai,
	obd_msg* out_msg_request,
	obd_DID_t* data_identifier,
	uint32_t data_identifier_length);

/// <summary>
/// Parse a response to requested service 22 DID (Request Vehicle Information)
/// </summary>
/// <remarks>The result must be freed afterwards by calling OBDonUDS_ParsedResponseFree</remarks>
/// <param name="msg_response">Response to parse</param>
/// <param name="out_buffer">[out] Parsed result</param>
/// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success.</returns>
obd_status __stdcall OBDonUDS_ParseResponse_RequestVehicleInformation(
	obd_msg* msg_response,
	obd_request_vehicle_information_response* out_buffer);


// service 19-55

/// <summary>
/// Send a request to service 19-55 (Request Emission-related Diagnostic Trouble Codes with Permanent Status)
/// </summary>
/// <param name="channel">A PCANTP channel handle representing a POBDonUDS channel</param>
/// <param name="nai">Network Addressing Information</param>
/// <param name="out_msg_request">[out] Request added to the Tx Queue, to be later used with OBDonUDS_WaitForServiceXXX</param>
/// <param name="functional_group_identifier">FunctionalGroupIdentifier (see OBDONUDS_EMISSION_SYSTEM_GROUP)</param>
/// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success.</returns>
obd_status __stdcall OBDonUDS_RequestPermanentTroubleCodes(
	cantp_handle channel,
	obd_netaddrinfo nai,
	obd_msg* out_msg_request,
	uint8_t functional_group_identifier);

/// <summary>
/// Parse a response to requested service 19-55 (Request Emission-related Diagnostic Trouble Codes with Permanent Status)
/// </summary>
/// <remarks>The result must be freed afterwards by calling OBDonUDS_ParsedResponseFree</remarks>
/// <param name="msg_response">Response to parse</param>
/// <param name="out_buffer">[out] Parsed result</param>
/// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success.</returns>
obd_status __stdcall OBDonUDS_ParseResponse_RequestPermanentTroubleCodes(
	obd_msg* msg_response,
	obd_request_permanent_trouble_codes_response* out_buffer);

// service 19 1A

/// <summary>
/// Send a request to service 19-1A (Request Supported DTCExtendedRecord Information)
/// </summary>
/// <param name="channel">A PCANTP channel handle representing a POBDonUDS channel</param>
/// <param name="nai">Network Addressing Information</param>
/// <param name="out_msg_request">[out] Request added to the Tx Queue, to be later used with OBDonUDS_WaitForServiceXXX</param>
/// <param name="DTC_extended_DTC_data_record">DTCExtendedDTCDataRecord</param>
/// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success.</returns>
obd_status __stdcall OBDonUDS_RequestSupportedDTCExtended(
	cantp_handle channel,
	obd_netaddrinfo nai,
	obd_msg* out_msg_request,
	uint8_t DTC_extended_DTC_data_record);

/// <summary>
/// Parse a response to requested service 19-1A (Request Supported DTCExtendedRecord Information)
/// </summary>
/// <remarks>The result must be freed afterwards by calling OBDonUDS_ParsedResponseFree</remarks>
/// <param name="msg_response">Response to parse</param>
/// <param name="out_buffer">[out] Parsed result</param>
/// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success.</returns>
obd_status __stdcall OBDonUDS_ParseResponse_RequestSupportedDTCExtended(
	obd_msg* msg_response,
	obd_request_supported_dtc_extended_response* out_buffer);

// service 19 06

/// <summary>
/// Send a request to service 19-06 (Request DTCExtendedDataRecord)
/// </summary>
/// <param name="channel">A PCANTP channel handle representing a POBDonUDS channel</param>
/// <param name="nai">Network Addressing Information</param>
/// <param name="out_msg_request">[out] Request added to the Tx Queue, to be later used with OBDonUDS_WaitForServiceXXX</param>
/// <param name="DTC_mask">DTCMaskRecord</param>
/// <param name="DTC_extended_data_record_number">DTCExtDataRecordNumber</param>
/// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success.</returns>
obd_status __stdcall OBDonUDS_RequestDTCExtended(
	cantp_handle channel,
	obd_netaddrinfo nai,
	obd_msg* out_msg_request,
	obd_DTC_t DTC_mask,
	uint8_t DTC_extended_data_record_number);

/// <summary>
/// Parse a response to requested service 19-06 (Request DTCExtendedDataRecord)
/// </summary>
/// <remarks>The result must be freed afterwards by calling OBDonUDS_ParsedResponseFree</remarks>
/// <param name="msg_response">Response to parse</param>
/// <param name="out_buffer">[out] Parsed result</param>
/// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success.</returns>
obd_status __stdcall OBDonUDS_ParseResponse_RequestDTCExtended(
	obd_msg* msg_response,
	obd_request_dtc_extended_response* out_buffer);

// service 19 56

/// <summary>
/// Send a request to service 19-56 (Request DTCs for a ReadinessGroup)
/// </summary>
/// <param name="channel">A PCANTP channel handle representing a POBDonUDS channel</param>
/// <param name="nai">Network Addressing Information</param>
/// <param name="out_msg_request">[out] Request added to the Tx Queue, to be later used with OBDonUDS_WaitForServiceXXX</param>
/// <param name="functional_group_identifier">FunctionalGroupIdentifier (see OBDONUDS_EMISSION_SYSTEM_GROUP)</param>
/// <param name="readiness_group_identifier">ReadinessGroupIdentifier</param>
/// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success.</returns>
obd_status __stdcall OBDonUDS_RequestDTCForAReadinessGroup(
	cantp_handle channel,
	obd_netaddrinfo nai,
	obd_msg* out_msg_request,
	uint8_t functional_group_identifier,
	uint8_t readiness_group_identifier);

/// <summary>
/// Parse a response to requested service 19-56 (Request DTCs for a ReadinessGroup)
/// </summary>
/// <remarks>The result must be freed afterwards by calling OBDonUDS_ParsedResponseFree</remarks>
/// <param name="msg_response">Response to parse</param>
/// <param name="out_buffer">[out] Parsed result</param>
/// <returns>A obd_status code. POBDONUDS_STATUS_OK is returned on success.</returns>
obd_status __stdcall OBDonUDS_ParseResponse_RequestDTCForAReadinessGroup(
	obd_msg* msg_response,
	obd_request_dtc_for_a_readiness_group_response* out_buffer);

#ifdef __cplusplus
}
#endif

#endif