#include <stdio.h>

#include "lib/PCAN-OBDonUDS.h"

#define OK_KO(test) (test)?"OK":"KO"
#define STATUS_OK_KO(test) OK_KO(OBDonUDS_StatusIsOk(test, POBDONUDS_STATUS_OK, false))

#define BUFFER_SIZE 256
#define DID_SUPPORTED_SIZE 256
#define DID_VALUE_SIZE 5

int main()
{
	int result = -1;
	// Get API version
	char buffer[BUFFER_SIZE];
	memset(buffer, 0, BUFFER_SIZE);
	obd_status status = OBDonUDS_GetValue(PCANTP_HANDLE_NONEBUS, POBDONUDS_PARAMETER_API_VERSION, buffer, sizeof(buffer));
	printf("Get API version (%s): %s\n", STATUS_OK_KO(status), buffer);

	// Configure logs to get general debug information
	uint8_t debugValue = POBDONUDS_DEBUG_LVL_DEBUG;
	status = OBDonUDS_SetValue(PCANTP_HANDLE_NONEBUS, POBDONUDS_PARAMETER_DEBUG, &debugValue, sizeof(debugValue));
	printf("Configure general logs (%s)\n", STATUS_OK_KO(status));

	// Initialize
	cantp_handle channel = PCANTP_HANDLE_USBBUS1; // TODO: modify the value according to your available PCAN devices.
	printf("Channel : 0x%02x, Channel status : 0x%02x\n", channel, OBDonUDS_GetStatus(channel));
	printf("Connecting with automatic detection of baudrate...\n");
	status = OBDonUDS_Initialize(channel, (cantp_baudrate)OBD_BAUDRATE_AUTODETECT, (cantp_hwtype)(0), 0, 0);
	printf("Initialize (%s)\n", STATUS_OK_KO(status));

	if (OBDonUDS_StatusIsOk(status, POBDONUDS_STATUS_OK, false))
	{
		result = 1;
		obd_baudrate baudrate = (obd_baudrate)0;
		status = OBDonUDS_GetValue(channel, POBDONUDS_PARAMETER_BAUDRATE, &baudrate, sizeof(baudrate));
		printf("-> Baudrate (%s): %s\n", STATUS_OK_KO(status), (baudrate == OBD_BAUDRATE_500K) ? "500 kbit/s" : (baudrate == OBD_BAUDRATE_250K) ? "250kbit/s" : "unknown");

		obd_msgprotocol canIdLen = (obd_msgprotocol)0;
		status = OBDonUDS_GetValue(channel, POBDONUDS_PARAMETER_CAN_ID_LENGTH, &canIdLen, sizeof(canIdLen));
		printf("-> Can Id length (%s): %s\n", STATUS_OK_KO(status), (canIdLen == OBD_MSGPROTOCOL_11BIT) ? "11" : (canIdLen == OBD_MSGPROTOCOL_29BIT) ? "29" : "unknown");

		// Print supported DIDs F4XX for service $22 - Request current Data
		uint8_t supportedDIDs_0xF4XX[DID_SUPPORTED_SIZE];
		memset(supportedDIDs_0xF4XX, 0, sizeof(supportedDIDs_0xF4XX));
		status = OBDonUDS_GetValue(channel, POBDONUDS_PARAMETER_SUPPORTMASK_DIDS_F4XX, supportedDIDs_0xF4XX, sizeof(supportedDIDs_0xF4XX));
		printf("Get supported DIDs 0xF4XX (%s):\n", STATUS_OK_KO(status));
		if (OBDonUDS_StatusIsOk(status, POBDONUDS_STATUS_OK, false))
		{
			result = 2;
			for (size_t did = 0; did < DID_SUPPORTED_SIZE; ++did)
			{
				for (uint8_t j = 0; j < 8; ++j)
				{
					if ((supportedDIDs_0xF4XX[did] >> j) & 0x01)
					{
						printf("-> DID 0x%04zX is supported by ECU #%d\n", 0xF400 + (did + 1), j + 1);
					}
				}
			}
		}
		else {
			printf("-> failed status: 0x%08X\n", status);
		}

		// Request current data values of 5 DID using physical address scheme (point to point)
		obd_DID_t DIDs[DID_VALUE_SIZE] = { 0xF415, 0xF401, 0xF405, 0xF403, 0xF40C }; // TODO: change the DIDs according to those supported
		obd_ecu myECU = POBD_ECU_1; // TODO: change the ECU if wanted
		obd_netaddrinfo nai;
		nai.protocol = canIdLen;
		nai.target_type = OBD_ADDRESSING_PHYSICAL;
		nai.source_addr = PUDS_ISO_15765_4_ADDR_TEST_EQUIPMENT;
		nai.target_addr = myECU;
		obd_msg msg_request;
		memset(&msg_request, 0, sizeof(msg_request));
		status = OBDonUDS_RequestCurrentData(channel, nai, &msg_request, DIDs, DID_VALUE_SIZE);
		printf("Request Current Powertrain diagnostic data (%s)\n", STATUS_OK_KO(status));

		if (OBDonUDS_StatusIsOk(status, POBDONUDS_STATUS_OK, false))
		{
			result = 3;
			// Wait for responses
			obd_msg msg_response;
			memset(&msg_response, 0, sizeof(msg_response));
			obd_msg msg_request_confirmation;
			memset(&msg_request_confirmation, 0, sizeof(msg_request_confirmation));
			status = OBDonUDS_WaitForService(channel, &msg_request, &msg_response, &msg_request_confirmation);
			printf("Wait for service physical (%s):\n", STATUS_OK_KO(status));
			if (OBDonUDS_StatusIsOk(status, POBDONUDS_STATUS_OK, false))
			{
				result = 4;
				// Print raw (unparsed) response
				printf("-> Raw response : \n\t");
				if (msg_response.links.network_result != NULL && (*msg_response.links.network_result) == PCANTP_NETSTATUS_OK &&
					msg_response.msg.msg.msgdata.any != NULL && msg_response.msg.msg.msgdata.any->data != NULL)
				{
					printf("[");
					for (uint32_t j = 0; j < msg_response.msg.msg.msgdata.any->length; ++j)
					{
						printf("%02x ", msg_response.msg.msg.msgdata.any->data[j]);
					}
					printf("]\n");
				}
				else {
					printf("N/A, an error occured, check content of the internal UDS/ISOTP message...\n");
				}

				// Parse response
				obd_request_current_data_response parsed_response;
				memset(&parsed_response, 0, sizeof(parsed_response));
				status = OBDonUDS_ParseResponse_RequestCurrentData(&msg_response, &parsed_response);
				printf("Parse response (%s):\n", STATUS_OK_KO(status));
				if (OBDonUDS_StatusIsOk(status, POBDONUDS_STATUS_OK, false))
				{
					result = 0;
					// Print parsed response
					if (parsed_response.nrc != 0)
						printf("-> Negative response code 0x%02X\n", parsed_response.nrc);
					else
					{
						for (uint32_t i = 0; i < parsed_response.nb_elements; ++i)
						{
							printf("-> Element #%d : DID 0x%04X, %02d bytes: [", i + 1, parsed_response.elements[i].data_identifier, parsed_response.elements[i].size);
							for (uint32_t j = 0; j < parsed_response.elements[i].size; ++j)
							{
								printf("%02x ", parsed_response.elements[i].data[j]);
							}
							printf("]\n");
						}
					}
				}
				else {
					printf("-> failed status: 0x%08X\n", status);
				}
				// Free message
				OBDonUDS_ParsedResponseFree((obd_response_generic*)&parsed_response);
			}
			else {
				printf("-> failed status: 0x%08X\n", status);
			}
			// Free messages
			OBDonUDS_MsgFree(&msg_request_confirmation);
			OBDonUDS_MsgFree(&msg_response);
		}
		else {
			printf("-> failed status: 0x%08X\n", status);
		}
		// Free messages
		OBDonUDS_MsgFree(&msg_request);
	}
	else {
		printf("-> failed status: 0x%08X\n", status);
	}


	status = OBDonUDS_Uninitialize(channel);
	printf("Uninitialize (%s)\n", STATUS_OK_KO(status));
	return result;
}