#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "lib/PCAN-OBDonUDS.h"

#define OK_KO(test) (test)?"OK":"KO"
#define STATUS_OK_KO(test) OK_KO(OBDonUDS_StatusIsOk(test, POBDONUDS_STATUS_OK, false))

#define BUFFER_SIZE 256
#define MAX_NB_RESPONSES 16
#define DID_SUPPORTED_SIZE 256
#define MAX_NB_ECU 8
#define MAX_DID_PER_PHYSICAL_REQUEST 6
#define MAX_NB_EXTENDED_DTC_RECORD 16

typedef struct {
	size_t size;
	obd_DTC_t* array;
} DTC_array_t;

static void printStatusDetails(obd_status status);
static int GetVehicleIdentificationNumber(cantp_handle channel, obd_msgprotocol protocol);
static int GetSupportedMask(cantp_handle channel);
static int displayRequestCurrentPowertrainDiagnosticData_22_DID(cantp_handle channel, obd_msgprotocol protocol);
static int displayRequestCurrentPowertrainDiagnosticData_22_DID_F401_functional(cantp_handle channel, obd_msgprotocol protocol);
static int displayRequestCurrentPowertrainDiagnosticData_22_DID_All(cantp_handle channel, obd_msgprotocol protocol);
static int displayRequestCurrentPowertrainDiagnosticData_22_DID_Some(cantp_handle channel, obd_msgprotocol protocol, uint8_t ecu, obd_DID_t* DIDs, uint32_t DIDsLength);
static void parse_F401_IM_ReadinessData_Details(uint8_t data);
static void parse_F404_CalculatedLoadValue_Details(uint8_t data);
static void parse_F40D_VehicleSpeedSensor_Details(uint8_t data);
static int displayRequestOnBoardMonitoringTestResultsForSpecificMonitoredSystems_22_MID(cantp_handle channel, obd_msgprotocol protocol);
static int displayRequestOnBoardMonitoringTestResultsForSpecificMonitoredSystems_22_MID_one(cantp_handle channel, obd_msgprotocol protocol, uint8_t ecu, obd_DID_t MID);
static void printTestObject(obd_test_data_object* pobject);
static uint16_t makeUint16(uint8_t highByte, uint8_t lowByte);
static int displayRequestEmissionRelatedDiagnosticTroubleCodesWithConfirmedStatus_19_42(cantp_handle channel, obd_msgprotocol protocol, DTC_array_t DTCarray[MAX_NB_ECU]);
static void printDTC(obd_trouble_code* object);
static void printDTCParsed(obd_request_trouble_codes_response* parsed_response);
static int KeepDTC(DTC_array_t DTCarray[MAX_NB_ECU], uint8_t ecuIndex, obd_request_trouble_codes_response* parsed_response);
static int displayClearResetEmissionRelatedDiagnosticInformation_14(cantp_handle channel, obd_msgprotocol protocol, uint32_t GroupOfDTC);
static int displayRequestPowertrainFreezeFrameData_19_04(cantp_handle channel, obd_msgprotocol protocol, DTC_array_t confirmedDTCs[MAX_NB_ECU], DTC_array_t pendingDTCs[MAX_NB_ECU]);
static int displayRequestPowertrainFreezeFrameData_19_04_Some(cantp_handle channel, obd_msgprotocol protocol, uint8_t ecuIndex, DTC_array_t DTCarray[MAX_NB_ECU]);
static void printSnapshot(obd_request_freeze_frame_data_response* parsed_response);
static int displayRequestEmissionRelatedDiagnosticTroubleCodesWithPendingStatus_19_42(cantp_handle channel, obd_msgprotocol protocol, DTC_array_t DTCarray[MAX_NB_ECU]);
static int displayRequestControlOfOnBoardSystemTestOrComponent_31_RID(cantp_handle channel, obd_msgprotocol protocol);
static int displayRequestControlOfOnBoardSystemTestOrComponent_31_RID_one(cantp_handle channel, obd_msgprotocol protocol, uint8_t ecuIndex, obd_DID_t RID);
static int displayRequestVehicleInformation_22_ITID(cantp_handle channel, obd_msgprotocol protocol);
static int displayRequestVehicleInformation_22_ITID_F802_functional(cantp_handle channel, obd_msgprotocol protocol);
static int displayRequestVehicleInformation_22_ITID_All(cantp_handle channel, obd_msgprotocol protocol);
static int displayRequestVehicleInformation_22_ITID_One(cantp_handle channel, obd_msgprotocol protocol, uint8_t ecuIndex, obd_DID_t ITID);
static int displayRequestEmissionRelatedDiagnosticTroubleCodesWithPermanentStatus_19_55(cantp_handle channel, obd_msgprotocol protocol);
static int displayRequestSupportedDTCExtendedRecord_19_1A(cantp_handle channel, obd_msgprotocol protocol, DTC_array_t DTCarray[MAX_NB_ECU][MAX_NB_EXTENDED_DTC_RECORD]);
static int displayRequestSupportedDTCExtendedRecord_19_1A_one(cantp_handle channel, obd_msgprotocol protocol, DTC_array_t DTCarray[MAX_NB_ECU][MAX_NB_EXTENDED_DTC_RECORD],
	uint8_t ecuIndex, uint8_t extendedDataRecordIndex);
static int KeepDTCextended(DTC_array_t DTCarray[MAX_NB_ECU][MAX_NB_EXTENDED_DTC_RECORD], uint8_t ecuIndex, uint8_t extendedDataRecordIndex,
	obd_request_supported_dtc_extended_response* parsed_response);
static int displayRequestDTCExtendedDataRecord_19_06(cantp_handle channel, obd_msgprotocol protocol, DTC_array_t DTCarray[MAX_NB_ECU][MAX_NB_EXTENDED_DTC_RECORD]);
static int displayRequestDTCExtendedDataRecord_19_06_one(cantp_handle channel, obd_msgprotocol protocol, DTC_array_t DTCarray[MAX_NB_ECU][MAX_NB_EXTENDED_DTC_RECORD],
	uint8_t ecuIndex, uint8_t extendedDataRecordIndex);
static int displayRequestDTCsForAReadinessGroup_19_56(cantp_handle channel, obd_msgprotocol protocol);
static int displayRequestDTCsForAReadinessGroup_19_56_one(cantp_handle channel, obd_msgprotocol protocol, uint8_t ecuIndex, uint8_t readinessGroup);

void printStatusDetails(obd_status status)
{
	char buffer[256];
	memset(buffer, 0, sizeof(buffer));
	if (OBDonUDS_StatusIsOk(OBDonUDS_GetErrorText(status, 0, buffer, sizeof(buffer)), POBDONUDS_STATUS_OK, false))
		printf("\"%s\"\n", buffer);
}

int GetVehicleIdentificationNumber(cantp_handle channel, obd_msgprotocol protocol)
{
	int nbError = 0;

	// Request vehicle identification number (VIN) using functional addressing scheme
	obd_DID_t VINitid = 0xF802;
	obd_msg msg_request;
	memset(&msg_request, 0, sizeof(msg_request));
	printf("Request vehicle identification number (VIN)\n");
	obd_status status = OBDonUDS_RequestVehicleInformation(channel, protocol == OBD_MSGPROTOCOL_11BIT ? OBD_NAI_REQUEST_FUNCTIONAL_11B : OBD_NAI_REQUEST_FUNCTIONAL_29B,
		&msg_request, &VINitid, 1);
	if (OBDonUDS_StatusIsOk(status, POBDONUDS_STATUS_OK, false))
	{
		// Wait for responses
		obd_msg msg_responses[MAX_NB_RESPONSES];
		memset(msg_responses, 0, sizeof(msg_responses));
		uint32_t nb_responses;
		obd_msg msg_request_confirmation;
		memset(&msg_request_confirmation, 0, sizeof(msg_request_confirmation));
		status = OBDonUDS_WaitForServiceFunctional(channel, &msg_request, MAX_NB_RESPONSES, false, msg_responses, &nb_responses, &msg_request_confirmation);
		if (OBDonUDS_StatusIsOk(status, POBDONUDS_STATUS_OK, false))
		{
			// Parse responses
			for (uint32_t ir = 0; ir < nb_responses; ++ir)
			{
				obd_request_vehicle_information_response parsed_response;
				memset(&parsed_response, 0, sizeof(parsed_response));
				status = OBDonUDS_ParseResponse_RequestVehicleInformation(&(msg_responses[ir]), &parsed_response);
				if (OBDonUDS_StatusIsOk(status, POBDONUDS_STATUS_OK, false))
				{
					if (parsed_response.nrc != 0)
						printf("-> Negative response code 0x%02X\n", parsed_response.nrc);
					else
					{
						printf("-> Vehicle Identification Number: '");
						for (size_t i = 0; i < parsed_response.nb_elements; ++i)
							printf("%c", parsed_response.elements[i]);
						printf("'\n\n");
					}
				}
				else
				{
					printf("-> Parse ERROR (0x%X): ", status); printStatusDetails(status);
					nbError++;
				}

				// Free message
				OBDonUDS_ParsedResponseFree((obd_response_generic*)&parsed_response);
			}
		}
		else
		{
			printf("-> Wait ERROR (0x%X): ", status); printStatusDetails(status);
			nbError++;
		}
		// Free messages
		OBDonUDS_MsgFree(&msg_request_confirmation);
		for (uint32_t ir = 0; ir < nb_responses; ++ir)
			OBDonUDS_MsgFree(&(msg_responses[ir]));
	}
	else
	{
		printf("-> Request ERROR (0x%X): ", status); printStatusDetails(status);
		nbError++;
	}

	// Free messages
	OBDonUDS_MsgFree(&msg_request);

	return nbError;
}

int GetSupportedMask(cantp_handle channel)
{
	int nbError = 0;

	printf("Supported items given by API parameters\n");
	
	uint8_t supportedDIDs_F4XX[DID_SUPPORTED_SIZE];
	memset(supportedDIDs_F4XX, 0, sizeof(supportedDIDs_F4XX));
	uint8_t supportedDIDs_F5XX[DID_SUPPORTED_SIZE];
	memset(supportedDIDs_F5XX, 0, sizeof(supportedDIDs_F5XX));
	uint8_t supportedDIDs_F7XX[DID_SUPPORTED_SIZE];
	memset(supportedDIDs_F7XX, 0, sizeof(supportedDIDs_F7XX));
	uint8_t supportedMIDs[DID_SUPPORTED_SIZE];
	memset(supportedMIDs, 0, sizeof(supportedMIDs));
	uint8_t supportedITIDs[DID_SUPPORTED_SIZE];
	memset(supportedITIDs, 0, sizeof(supportedITIDs));
	uint8_t supportedRIDs_E0XX[DID_SUPPORTED_SIZE];
	memset(supportedRIDs_E0XX, 0, sizeof(supportedRIDs_E0XX));
	uint8_t supportedRIDs_E1XX[DID_SUPPORTED_SIZE];
	memset(supportedRIDs_E1XX, 0, sizeof(supportedRIDs_E1XX));

	obd_status status;
	
	status = OBDonUDS_GetValue(channel, POBDONUDS_PARAMETER_SUPPORTMASK_DIDS_F4XX, supportedDIDs_F4XX, sizeof(supportedDIDs_F4XX));
	if (!OBDonUDS_StatusIsOk(status, POBDONUDS_STATUS_OK, false))
	{
		printf("-> GetValue(POBDONUDS_PARAMETER_SUPPORTMASK_DIDS_F4XX) failed: 0x%08X\n", status);
		nbError++;
	}
	status = OBDonUDS_GetValue(channel, POBDONUDS_PARAMETER_SUPPORTMASK_DIDS_F5XX, supportedDIDs_F5XX, sizeof(supportedDIDs_F5XX));
	if (!OBDonUDS_StatusIsOk(status, POBDONUDS_STATUS_OK, false))
	{
		printf("-> GetValue(POBDONUDS_PARAMETER_SUPPORTMASK_DIDS_F5XX) failed: 0x%08X\n", status);
		nbError++;
	}
	status = OBDonUDS_GetValue(channel, POBDONUDS_PARAMETER_SUPPORTMASK_DIDS_F7XX, supportedDIDs_F7XX, sizeof(supportedDIDs_F7XX));
	if (!OBDonUDS_StatusIsOk(status, POBDONUDS_STATUS_OK, false))
	{
		printf("-> GetValue(POBDONUDS_PARAMETER_SUPPORTMASK_DIDS_F7XX) failed: 0x%08X\n", status);
		nbError++;
	}
	status = OBDonUDS_GetValue(channel, POBDONUDS_PARAMETER_SUPPORTMASK_MIDS, supportedMIDs, sizeof(supportedMIDs));
	if (!OBDonUDS_StatusIsOk(status, POBDONUDS_STATUS_OK, false))
	{
		printf("-> GetValue(POBDONUDS_PARAMETER_SUPPORTMASK_MIDS) failed: 0x%08X\n", status);
		nbError++;
	}
	status = OBDonUDS_GetValue(channel, POBDONUDS_PARAMETER_SUPPORTMASK_ITIDS, supportedITIDs, sizeof(supportedITIDs));
	if (!OBDonUDS_StatusIsOk(status, POBDONUDS_STATUS_OK, false))
	{
		printf("-> GetValue(POBDONUDS_PARAMETER_SUPPORTMASK_ITIDS) failed: 0x%08X\n", status);
		nbError++;
	}
	status = OBDonUDS_GetValue(channel, POBDONUDS_PARAMETER_SUPPORTMASK_RIDS_E0XX, supportedRIDs_E0XX, sizeof(supportedRIDs_E0XX));
	if (!OBDonUDS_StatusIsOk(status, POBDONUDS_STATUS_OK, false))
	{
		printf("-> GetValue(POBDONUDS_PARAMETER_SUPPORTMASK_RIDS_E0XX) failed: 0x%08X\n", status);
		nbError++;
	}
	status = OBDonUDS_GetValue(channel, POBDONUDS_PARAMETER_SUPPORTMASK_RIDS_E1XX, supportedRIDs_E1XX, sizeof(supportedRIDs_E1XX));
	if (!OBDonUDS_StatusIsOk(status, POBDONUDS_STATUS_OK, false))
	{
		printf("-> GetValue(POBDONUDS_PARAMETER_SUPPORTMASK_RIDS_E1XX) failed: 0x%08X\n", status);
		nbError++;
	}
	for (size_t did = 0; did < DID_SUPPORTED_SIZE; ++did)
	{
		for (uint8_t ecuIndex = 0; ecuIndex < MAX_NB_ECU; ++ecuIndex)
		{
			if ((supportedDIDs_F4XX[did] >> ecuIndex) & 0x01)
			{
				printf("-> DID  0x%04zX is supported by ECU #%d\n", 0xF400 + (did + 1), ecuIndex + 1);
			}
			if ((supportedDIDs_F5XX[did] >> ecuIndex) & 0x01)
			{
				printf("-> DID  0x%04zX is supported by ECU #%d\n", 0xF500 + (did + 1), ecuIndex + 1);
			}
			if ((supportedDIDs_F7XX[did] >> ecuIndex) & 0x01)
			{
				printf("-> DID  0x%04zX is supported by ECU #%d\n", 0xF700 + (did + 1), ecuIndex + 1);
			}
			if ((supportedMIDs[did] >> ecuIndex) & 0x01)
			{
				printf("-> MID  0x%04zX is supported by ECU #%d\n", 0xF600 + (did + 1), ecuIndex + 1);
			}
			if ((supportedITIDs[did] >> ecuIndex) & 0x01)
			{
				printf("-> ITID 0x%04zX is supported by ECU #%d\n", 0xF800 + (did + 1), ecuIndex + 1);
			}
			if ((supportedRIDs_E0XX[did] >> ecuIndex) & 0x01)
			{
				printf("-> RID  0x%04zX is supported by ECU #%d\n", 0xE000 + (did + 1), ecuIndex + 1);
			}
			if ((supportedRIDs_E1XX[did] >> ecuIndex) & 0x01)
			{
				printf("-> RID  0x%04zX is supported by ECU #%d\n", 0xE100 + (did + 1), ecuIndex + 1);
			}
		}
	}

	return nbError;
}

int displayRequestCurrentPowertrainDiagnosticData_22_DID(cantp_handle channel, obd_msgprotocol protocol)
{
	int nbErr = displayRequestCurrentPowertrainDiagnosticData_22_DID_F401_functional(channel, protocol);
	nbErr += displayRequestCurrentPowertrainDiagnosticData_22_DID_All(channel, protocol);
	return nbErr;
}

int displayRequestCurrentPowertrainDiagnosticData_22_DID_All(cantp_handle channel, obd_msgprotocol protocol)
{
	int nbError = 0;

	printf("\nRequest Current Powertrain Diagnostic Data physically for each ECU for all supported DID\n");


	// Get supported dids
	uint8_t supportedDIDs_F4XX[DID_SUPPORTED_SIZE];
	uint8_t supportedDIDs_F5XX[DID_SUPPORTED_SIZE];
	uint8_t supportedDIDs_F7XX[DID_SUPPORTED_SIZE];

	OBDonUDS_GetValue(channel, POBDONUDS_PARAMETER_SUPPORTMASK_DIDS_F4XX, supportedDIDs_F4XX, sizeof(supportedDIDs_F4XX));
	OBDonUDS_GetValue(channel, POBDONUDS_PARAMETER_SUPPORTMASK_DIDS_F5XX, supportedDIDs_F5XX, sizeof(supportedDIDs_F5XX));
	OBDonUDS_GetValue(channel, POBDONUDS_PARAMETER_SUPPORTMASK_DIDS_F7XX, supportedDIDs_F7XX, sizeof(supportedDIDs_F7XX));

	// Sort supported dids per ECUs
	obd_DID_t ECUSupportedDIDs[MAX_NB_ECU][DID_SUPPORTED_SIZE * 3];
	memset(ECUSupportedDIDs, 0, sizeof(ECUSupportedDIDs));
	size_t ECUSupportedDIDsSize[MAX_NB_ECU];
	memset(ECUSupportedDIDsSize, 0, sizeof(ECUSupportedDIDsSize));
	for (obd_DID_t did = 0; did < DID_SUPPORTED_SIZE; ++did)
	{
		for (uint8_t ecuIndex = 0; ecuIndex < MAX_NB_ECU; ++ecuIndex)
		{
			if ((did + 1) % 0x20 != 0) // avoid DID for "requesting supported" e.g. 0xF420
			{
				if ((supportedDIDs_F4XX[did] >> ecuIndex) & 0x01)
				{
					ECUSupportedDIDs[ecuIndex][ECUSupportedDIDsSize[ecuIndex]++] = 0xF400 + (did + 1);
				}
				if ((supportedDIDs_F5XX[did] >> ecuIndex) & 0x01)
				{
					ECUSupportedDIDs[ecuIndex][ECUSupportedDIDsSize[ecuIndex]++] = 0xF500 + (did + 1);
				}
				if ((supportedDIDs_F7XX[did] >> ecuIndex) & 0x01)
				{
					ECUSupportedDIDs[ecuIndex][ECUSupportedDIDsSize[ecuIndex]++] = 0xF700 + (did + 1);
				}
			}
		}
	}

	// Request physically each ECU on supported DIDs (6 DIDs max per request)
	for (uint8_t ecuIndex = 0; ecuIndex < MAX_NB_ECU; ++ecuIndex)
	{
		obd_DID_t RequestDIDs[MAX_DID_PER_PHYSICAL_REQUEST];
		memset(RequestDIDs, 0, sizeof(RequestDIDs));
		uint32_t RequestDIDsSize = 0;
		for (size_t i = 0; i < ECUSupportedDIDsSize[ecuIndex]; ++i)
		{
			if (RequestDIDsSize < MAX_DID_PER_PHYSICAL_REQUEST)
				RequestDIDs[RequestDIDsSize++] = ECUSupportedDIDs[ecuIndex][i];

			if (RequestDIDsSize == MAX_DID_PER_PHYSICAL_REQUEST || i == ECUSupportedDIDsSize[ecuIndex] - 1)
			{
				nbError += displayRequestCurrentPowertrainDiagnosticData_22_DID_Some(channel, protocol, ecuIndex + 1, RequestDIDs, RequestDIDsSize);
				RequestDIDsSize = 0;
			}
		}
	}

	return nbError;
}

int displayRequestCurrentPowertrainDiagnosticData_22_DID_Some(cantp_handle channel, obd_msgprotocol protocol, uint8_t ecu, obd_DID_t* DIDs, uint32_t DIDsLength)
{
	int nbError = 0;
	obd_msg msg_request;
	obd_status status;

	obd_netaddrinfo nai;
	nai.protocol = protocol;
	nai.source_addr = PUDS_ISO_15765_4_ADDR_TEST_EQUIPMENT;
	nai.target_addr = ecu;
	nai.target_type = OBD_ADDRESSING_PHYSICAL;
	memset(&msg_request, 0, sizeof(msg_request));
	status = OBDonUDS_RequestCurrentData(channel, nai, &msg_request, DIDs, DIDsLength);
	if (OBDonUDS_StatusIsOk(status, POBDONUDS_STATUS_OK, false))
	{
		// Wait for responses
		obd_msg msg_response;
		memset(&msg_response, 0, sizeof(msg_response));
		obd_msg msg_request_confirmation;
		memset(&msg_request_confirmation, 0, sizeof(msg_request_confirmation));
		status = OBDonUDS_WaitForService(channel, &msg_request, &msg_response, &msg_request_confirmation);
		if (OBDonUDS_StatusIsOk(status, POBDONUDS_STATUS_OK, false))
		{
			// Parse response
			obd_request_current_data_response parsed_response;
			memset(&parsed_response, 0, sizeof(parsed_response));
			status = OBDonUDS_ParseResponse_RequestCurrentData(&msg_response, &parsed_response);
			if (OBDonUDS_StatusIsOk(status, POBDONUDS_STATUS_OK, false))
			{
				// Print parsed response
				printf("-> Response from ECU #%d : ", parsed_response.ecu_address);
				if (parsed_response.nrc != 0)
					printf("Negative response code 0x%02X\n", parsed_response.nrc);
				else
				{
					for (uint32_t ip = 0; ip < parsed_response.nb_elements; ++ip)
					{
						printf("\n  -> Element %d : DID 0x%X, %d bytes\n\t\t", ip + 1, parsed_response.elements[ip].data_identifier, parsed_response.elements[ip].size);
						for (uint32_t j = 0; j < parsed_response.elements[ip].size; ++j)
						{
							printf("%02X ", parsed_response.elements[ip].data[j]);
						}
						// Example of detailed parsing : for DID F404
						if (parsed_response.elements[ip].data_identifier == 0xF404 && parsed_response.elements[ip].size == 1)
						{
							printf("\n  -> ");
							parse_F404_CalculatedLoadValue_Details(parsed_response.elements[ip].data[0]);
						}
						// Example of detailed parsing : for DID F40D
						if (parsed_response.elements[ip].data_identifier == 0xF40D && parsed_response.elements[ip].size == 1)
						{
							printf("\n  -> ");
							parse_F40D_VehicleSpeedSensor_Details(parsed_response.elements[ip].data[0]);
						}
						// TODO: parse details of other elements
					}
					printf("\n");
				}
			}
			else
			{
				printf("-> Parse DIDs: ERROR (0x%X): ", status); printStatusDetails(status);
				nbError++;
			}
			// Free message
			OBDonUDS_ParsedResponseFree((obd_response_generic*)&parsed_response);
		}
		else
		{
			printf("-> Wait DIDs: ERROR (0x%X): ", status); printStatusDetails(status);
			nbError++;
		}

		// Free messages
		OBDonUDS_MsgFree(&msg_request_confirmation);
		OBDonUDS_MsgFree(&msg_response);
	}
	else
	{
		printf("-> Request DIDs: ERROR (0x%X): ", status); printStatusDetails(status);
		nbError++;
	}
	// Free messages
	OBDonUDS_MsgFree(&msg_request);

	return nbError;
}

int displayRequestCurrentPowertrainDiagnosticData_22_DID_F401_functional(cantp_handle channel, obd_msgprotocol protocol)
{
	int nbError = 0;
	obd_DID_t did = 0xF401;
	obd_msg msg_request;
	memset(&msg_request, 0, sizeof(msg_request));

	printf("\nRequest Current Powertrain Diagnostic Data functionally for DID F401\n");


	obd_status status = OBDonUDS_RequestCurrentData(channel, protocol == OBD_MSGPROTOCOL_11BIT ? OBD_NAI_REQUEST_FUNCTIONAL_11B : OBD_NAI_REQUEST_FUNCTIONAL_29B, 
		&msg_request, &did, 1);
	if (OBDonUDS_StatusIsOk(status, POBDONUDS_STATUS_OK, false))
	{
		// Wait for responses
		obd_msg msg_responses[MAX_NB_RESPONSES];
		memset(msg_responses, 0, sizeof(msg_responses));
		uint32_t nb_responses;
		obd_msg msg_request_confirmation;
		memset(&msg_request_confirmation, 0, sizeof(msg_request_confirmation));
		status = OBDonUDS_WaitForServiceFunctional(channel, &msg_request, MAX_NB_RESPONSES, false, msg_responses, &nb_responses, &msg_request_confirmation);
		if (OBDonUDS_StatusIsOk(status, POBDONUDS_STATUS_OK, false))
		{
			for (uint32_t ir = 0; ir < nb_responses; ++ir)
			{
				// Parse response
				obd_request_current_data_response parsed_response;
				memset(&parsed_response, 0, sizeof(parsed_response));
				status = OBDonUDS_ParseResponse_RequestCurrentData(&msg_responses[ir], &parsed_response);
				if (OBDonUDS_StatusIsOk(status, POBDONUDS_STATUS_OK, false))
				{
					// Print parsed response
					printf("-> Response number %d from ECU #%d : ", ir + 1, parsed_response.ecu_address);
					if (parsed_response.nrc != 0)
						printf("-> Negative response code 0x%02X\n", parsed_response.nrc);
					else
					{
						for (uint32_t ip = 0; ip < parsed_response.nb_elements; ++ip)
						{
							printf("\n  -> Element %d : DID 0x%X, %d bytes\n\t\t", ip + 1, parsed_response.elements[ip].data_identifier, parsed_response.elements[ip].size);
							for (uint32_t j = 0; j < parsed_response.elements[ip].size; ++j)
							{
								printf("%02X ", parsed_response.elements[ip].data[j]);
							}
							printf("\n");
						}

						// Now display Number of DTCs stored : A bits 0-6 and MIL status : A bit 7					
						if (parsed_response.nb_elements >= 1 && parsed_response.elements[0].size >= 1)// Check A is present (should be present !)
						{
							parse_F401_IM_ReadinessData_Details(parsed_response.elements[0].data[0]);
						}
						else
						{
							printf("-> Parsed DID: %02X: ERROR (A is missing): ", did);
							nbError++;
						}
					}
				}
				else
				{
					printf("-> Parse DID: %02X: ERROR (0x%X): ", did, status); printStatusDetails(status);
					nbError++;
				}

				OBDonUDS_ParsedResponseFree((obd_response_generic*)&parsed_response);
			}
		}
		else
		{
			printf("-> Wait DID: %02X: ERROR (0x%X): ", did, status); printStatusDetails(status);
			nbError++;
		}

		OBDonUDS_MsgFree(&msg_request_confirmation);
		for (uint32_t i = 0; i < nb_responses; ++i)
			OBDonUDS_MsgFree(&(msg_responses[i]));

	}
	else
	{
		printf("-> Request DID: %02X: ERROR (0x%X): ", did, status); printStatusDetails(status);
		nbError++;
	}

	OBDonUDS_MsgFree(&msg_request);

	return nbError;
}

void parse_F401_IM_ReadinessData_Details(uint8_t data)
{
	printf("-> DTC_CNT: %dd\n", data & 0b01111111);
	printf("-> MIL: %s\n", (data & 0b10000000) == 0 ? "OFF" : "ON");
}

void parse_F404_CalculatedLoadValue_Details(uint8_t data)
{
	printf("LOAD_PCT: %03.1f%%", data * 100.0 / 255);
}

void parse_F40D_VehicleSpeedSensor_Details(uint8_t data)
{
	printf("VSS: %03d km/h", data);
}

int displayRequestOnBoardMonitoringTestResultsForSpecificMonitoredSystems_22_MID(cantp_handle channel, obd_msgprotocol protocol)
{
	int nbError = 0;

	printf("\nRequest On Board Monitoring Test Results For Specific Monitored Systems physically for each ECU for all supported MID\n");

	// Get supported mids
	uint8_t supportedMIDs[DID_SUPPORTED_SIZE];
	OBDonUDS_GetValue(channel, POBDONUDS_PARAMETER_SUPPORTMASK_MIDS, supportedMIDs, sizeof(supportedMIDs));

	// Request physically each ECU on supported MIDs (1 MIDs max per request)
	for (obd_DID_t mid = 0; mid < DID_SUPPORTED_SIZE; ++mid)
	{
		for (uint8_t ecuIndex = 0; ecuIndex < MAX_NB_ECU; ++ecuIndex)
		{
			if ((mid + 1) % 0x20 != 0) // avoid MID for "requesting supported" e.g. 0xF620
			{
				if ((supportedMIDs[mid] >> ecuIndex) & 0x01)
				{
					nbError += displayRequestOnBoardMonitoringTestResultsForSpecificMonitoredSystems_22_MID_one(channel, protocol, ecuIndex + 1, 0xF600 + (mid + 1));
				}
			}
		}
	}

	return nbError;
}

int displayRequestOnBoardMonitoringTestResultsForSpecificMonitoredSystems_22_MID_one(cantp_handle channel, obd_msgprotocol protocol, uint8_t ecu, obd_DID_t MID)
{
	int nbError = 0;
	obd_msg msg_request;
	obd_status status;

	obd_netaddrinfo nai;
	nai.protocol = protocol;
	nai.source_addr = PUDS_ISO_15765_4_ADDR_TEST_EQUIPMENT;
	nai.target_addr = ecu;
	nai.target_type = OBD_ADDRESSING_PHYSICAL;
	memset(&msg_request, 0, sizeof(msg_request));
	status = OBDonUDS_RequestTestResults(channel, nai, &msg_request, &MID, 1);
	if (OBDonUDS_StatusIsOk(status, POBDONUDS_STATUS_OK, false))
	{
		// Wait for response
		obd_msg msg_response;
		memset(&msg_response, 0, sizeof(msg_response));
		obd_msg msg_request_confirmation;
		memset(&msg_request_confirmation, 0, sizeof(msg_request_confirmation));
		status = OBDonUDS_WaitForService(channel, &msg_request, &msg_response, &msg_request_confirmation);
		if (OBDonUDS_StatusIsOk(status, POBDONUDS_STATUS_OK, false))
		{
			// Parse response
			obd_request_test_results_response parsed_response;
			memset(&parsed_response, 0, sizeof(parsed_response));
			status = OBDonUDS_ParseResponse_RequestTestResults(&msg_response, &parsed_response);
			if (OBDonUDS_StatusIsOk(status, POBDONUDS_STATUS_OK, false))
			{
				// Print parsed response
				printf("-> Response from ECU #%d : ", parsed_response.ecu_address);
				if (parsed_response.nrc != 0)
					printf("-> Negative response code 0x%02X\n", parsed_response.nrc);
				else
				{
					printf("-> MID 0x%X ", parsed_response.data_identifier);
					for (uint32_t ip = 0; ip < parsed_response.nb_elements; ++ip)
					{
						printf("\n  -> Element %d : ", ip + 1);
						printTestObject(&(parsed_response.elements[ip]));
					}
					printf("\n");
				}
			}
			else
			{
				printf("-> Parse MID: ERROR (0x%X): ", status); printStatusDetails(status);
				nbError++;
			}
			// Free message
			OBDonUDS_ParsedResponseFree((obd_response_generic*)&parsed_response);
		}
		else
		{
			printf("-> Wait MID 0x%X ECU #%d: ERROR (0x%X) :", MID, ecu, status); printStatusDetails(status);
			nbError++;
		}

		// Free messages
		OBDonUDS_MsgFree(&msg_request_confirmation);
		OBDonUDS_MsgFree(&msg_response);
	}
	else
	{
		printf("-> Request MID: ERROR (0x%X): ", status); printStatusDetails(status);
		nbError++;
	}
	// Free messages
	OBDonUDS_MsgFree(&msg_request);

	return nbError;
}

void printTestObject(obd_test_data_object* object)
{
	if (object == NULL)
		return;

	printf("TID 0x%02X Unit/Scaling 0x%02X Test Value 0x%02X 0x%02X Min.Limit 0x%02X 0x%02X Max.Limit 0x%02X 0x%02X ",
		object->test_identifier,
		object->unit_and_scaling,
		object->test_value[0], object->test_value[1],
		object->min_test_limit[0], object->min_test_limit[1],
		object->max_test_limit[0], object->max_test_limit[1]);

	// Examples of detailed parsing
	switch (object->unit_and_scaling)
	{
	case 0x10:
		printf("\n\tTest Value %f ms, Min.Limit %f ms, Max.Limit %f ms",
			makeUint16(object->test_value[0], object->test_value[1]) * 1.0,
			makeUint16(object->min_test_limit[0], object->min_test_limit[1]) * 1.0,
			makeUint16(object->max_test_limit[0], object->max_test_limit[1]) * 1.0);
		break;
	case 0x0A:
		printf("\n\tTest Value %f mV, Min.Limit %f mV, Max.Limit %f mV",
			makeUint16(object->test_value[0], object->test_value[1]) * 0.1220703125,
			makeUint16(object->min_test_limit[0], object->min_test_limit[1]) * 0.1220703125,
			makeUint16(object->max_test_limit[0], object->max_test_limit[1]) * 0.1220703125);
		break;
	case 0x24:
		printf("\n\tTest Value %u, Min.Limit %u, Max.Limit %u",
			makeUint16(object->test_value[0], object->test_value[1]),
			makeUint16(object->min_test_limit[0], object->min_test_limit[1]),
			makeUint16(object->max_test_limit[0], object->max_test_limit[1]));
		break;
	default:
		// TODO: parse details of other elements
		break;
	}
}

uint16_t makeUint16(uint8_t highByte, uint8_t lowByte)
{
	return ((highByte << 8) & 0xFF00) | lowByte;
}

int displayRequestEmissionRelatedDiagnosticTroubleCodesWithConfirmedStatus_19_42(cantp_handle channel, obd_msgprotocol protocol, DTC_array_t DTCarray[MAX_NB_ECU])
{
	int nbError = 0;
	obd_status status;

	printf("\nRequest Emission-Related Diagnostic Trouble Codes With Confirmed Status physically for each ECU\n");

	uint8_t availableECUs = 0;
	OBDonUDS_GetValue(channel, POBDONUDS_PARAMETER_AVAILABLE_ECUS, &availableECUs, sizeof(availableECUs));

	for (uint8_t ecuIndex = 0; ecuIndex < availableECUs; ++ecuIndex)
	{
		obd_netaddrinfo nai;
		nai.protocol = protocol;
		nai.source_addr = PUDS_ISO_15765_4_ADDR_TEST_EQUIPMENT;
		nai.target_addr = ecuIndex + 1;
		nai.target_type = OBD_ADDRESSING_PHYSICAL;
		obd_msg msg_request;
		memset(&msg_request, 0, sizeof(msg_request));
		status = OBDonUDS_RequestConfirmedTroubleCodes(channel, nai, &msg_request, OBDONUDS_EMISSION_SYSTEM_GROUP, OBDONUDS_DTC_STATUS_CONFIRMED, OBDONUDS_DTC_SEVERITY_CLASS_1);
		if (OBDonUDS_StatusIsOk(status, POBDONUDS_STATUS_OK, false))
		{
			// Wait for response
			obd_msg msg_response;
			memset(&msg_response, 0, sizeof(msg_response));
			obd_msg msg_request_confirmation;
			memset(&msg_request_confirmation, 0, sizeof(msg_request_confirmation));
			status = OBDonUDS_WaitForService(channel, &msg_request, &msg_response, &msg_request_confirmation);
			if (OBDonUDS_StatusIsOk(status, POBDONUDS_STATUS_OK, false))
			{
				// Parse response
				obd_request_trouble_codes_response parsed_response;
				memset(&parsed_response, 0, sizeof(parsed_response));
				status = OBDonUDS_ParseResponse_RequestConfirmedTroubleCodes(&msg_response, &parsed_response);
				if (OBDonUDS_StatusIsOk(status, POBDONUDS_STATUS_OK, false))
				{
					// Print parsed response
					printDTCParsed(&parsed_response);
					// Keep a list of DTC
					nbError += KeepDTC(DTCarray, ecuIndex, &parsed_response);
				}
				else
				{
					printf("-> Parse : ERROR (0x%X): ", status); printStatusDetails(status);
					nbError++;
				}
				// Free message
				OBDonUDS_ParsedResponseFree((obd_response_generic*)&parsed_response);
			}
			else
			{
				printf("-> Wait ECU #%d: ERROR (0x%X): ", ecuIndex + 1, status); printStatusDetails(status);
				nbError++;
			}

			// Free messages
			OBDonUDS_MsgFree(&msg_request_confirmation);
			OBDonUDS_MsgFree(&msg_response);
		}
		else
		{
			printf("-> Request : ERROR (0x%X): ", status); printStatusDetails(status);
			nbError++;
		}
		// Free messages
		OBDonUDS_MsgFree(&msg_request);
	}

	return nbError;
}


int displayClearResetEmissionRelatedDiagnosticInformation_14(cantp_handle channel, obd_msgprotocol protocol, uint32_t GroupOfDTC)
{
	int nbError = 0;
	printf("\nRequest functionally Clear/Reset Emission-Related Diagnostic Information for group of DTC 0x%04X\n", GroupOfDTC);

	obd_msg msg_request;
	memset(&msg_request, 0, sizeof(msg_request));
	obd_status status = OBDonUDS_ClearTroubleCodes(channel, protocol == OBD_MSGPROTOCOL_11BIT ? OBD_NAI_REQUEST_FUNCTIONAL_11B : OBD_NAI_REQUEST_FUNCTIONAL_29B, 
		&msg_request, GroupOfDTC);
	if (OBDonUDS_StatusIsOk(status, POBDONUDS_STATUS_OK, false))
	{
		// Wait for responses
		obd_msg msg_responses[MAX_NB_RESPONSES];
		memset(msg_responses, 0, sizeof(msg_responses));
		uint32_t nb_responses;
		obd_msg msg_request_confirmation;
		memset(&msg_request_confirmation, 0, sizeof(msg_request_confirmation));
		status = OBDonUDS_WaitForServiceFunctional(channel, &msg_request, MAX_NB_RESPONSES, false, msg_responses, &nb_responses, &msg_request_confirmation);
		if (OBDonUDS_StatusIsOk(status, POBDONUDS_STATUS_OK, false))
		{
			for (uint32_t ir = 0; ir < nb_responses; ++ir)
			{
				// Parse response
				obd_request_clear_trouble_codes_response parsed_response;
				memset(&parsed_response, 0, sizeof(parsed_response));
				status = OBDonUDS_ParseResponse_ClearTroubleCodes(&msg_responses[ir], &parsed_response);
				if (OBDonUDS_StatusIsOk(status, POBDONUDS_STATUS_OK, false))
				{
					// Print parsed response
					printf("-> Response number %d from ECU #%d : ", ir + 1, parsed_response.ecu_address);
					if (parsed_response.nrc != 0)
					{
						printf("-> Negative response code 0x%02X\n", parsed_response.nrc);
					}
					else
					{
						printf("-> Positive response\n");
					}
				}
				else
				{
					printf("-> Parse ERROR (0x%X): ", status); printStatusDetails(status);
					nbError++;
				}

				OBDonUDS_ParsedResponseFree((obd_response_generic*)&parsed_response);
			}
		}
		else
		{
			printf("-> Wait failed (0x%08X): ", status); printStatusDetails(status);
			nbError++;
		}

		OBDonUDS_MsgFree(&msg_request_confirmation);
		for (uint32_t i = 0; i < nb_responses; ++i)
			OBDonUDS_MsgFree(&(msg_responses[i]));

	}
	else
	{
		printf("Request ERROR (0x%X): ", status); printStatusDetails(status);
		nbError++;
	}

	OBDonUDS_MsgFree(&msg_request);

	return nbError;

}

int displayRequestPowertrainFreezeFrameData_19_04(cantp_handle channel, obd_msgprotocol protocol, DTC_array_t confirmedDTCs[MAX_NB_ECU], DTC_array_t pendingDTCs[MAX_NB_ECU])
{
	int nbError = 0;

	printf("\nRequest Powertrain Freeze Frame Data physically for each ECU for each pending or confirmed DTC\n");

	uint8_t availableECUs = 0;
	OBDonUDS_GetValue(channel, POBDONUDS_PARAMETER_AVAILABLE_ECUS, &availableECUs, sizeof(availableECUs));

	for (uint8_t ecuIndex = 0; ecuIndex < availableECUs; ++ecuIndex)
	{
		nbError += displayRequestPowertrainFreezeFrameData_19_04_Some(channel, protocol, ecuIndex, confirmedDTCs);
		nbError += displayRequestPowertrainFreezeFrameData_19_04_Some(channel, protocol, ecuIndex, pendingDTCs);
	}

	return nbError;
}

int displayRequestPowertrainFreezeFrameData_19_04_Some(cantp_handle channel, obd_msgprotocol protocol, uint8_t ecuIndex, DTC_array_t DTCarray[MAX_NB_ECU])
{
	int nbError = 0;
	obd_status status;

	if (DTCarray == NULL || DTCarray[ecuIndex].array == NULL)
		return nbError;

	for (uint8_t DTCindex = 0; DTCindex < DTCarray[ecuIndex].size; ++DTCindex)
	{
		for (uint16_t occurrence = 0x00; occurrence <= 0xF0; occurrence += 0xF0)
		{
			obd_netaddrinfo nai;
			nai.protocol = protocol;
			nai.source_addr = PUDS_ISO_15765_4_ADDR_TEST_EQUIPMENT;
			nai.target_addr = ecuIndex + 1;
			nai.target_type = OBD_ADDRESSING_PHYSICAL;
			obd_msg msg_request;
			memset(&msg_request, 0, sizeof(msg_request));
			status = OBDonUDS_RequestFreezeFrameData(channel, nai, &msg_request, DTCarray[ecuIndex].array[DTCindex], (uint8_t)occurrence);
			if (OBDonUDS_StatusIsOk(status, POBDONUDS_STATUS_OK, false))
			{
				// Wait for response
				obd_msg msg_response;
				memset(&msg_response, 0, sizeof(msg_response));
				obd_msg msg_request_confirmation;
				memset(&msg_request_confirmation, 0, sizeof(msg_request_confirmation));
				status = OBDonUDS_WaitForService(channel, &msg_request, &msg_response, &msg_request_confirmation);
				if (OBDonUDS_StatusIsOk(status, POBDONUDS_STATUS_OK, false))
				{
					// Parse response
					obd_request_freeze_frame_data_response parsed_response;
					memset(&parsed_response, 0, sizeof(parsed_response));
					status = OBDonUDS_ParseResponse_RequestFreezeFrameData(&msg_response, &parsed_response);
					if (OBDonUDS_StatusIsOk(status, POBDONUDS_STATUS_OK, false))
					{
						// Print parsed response
						printSnapshot(&parsed_response);
					}
					else
					{
						printf("-> Parse : ERROR (0x%X): ", status); printStatusDetails(status);
						nbError++;
					}
					// Free message
					OBDonUDS_ParsedResponseFree((obd_response_generic*)&parsed_response);
				}
				else
				{
					printf("-> Wait failed for ECU #%d DTC 0x%06X occurrence 0x%02X (0x%08X): ", ecuIndex + 1, DTCarray[ecuIndex].array[DTCindex], occurrence, status); printStatusDetails(status);
					nbError++;
				}

				// Free messages
				OBDonUDS_MsgFree(&msg_request_confirmation);
				OBDonUDS_MsgFree(&msg_response);
			}
			else
			{
				printf("-> Request : ERROR (0x%X): ", status); printStatusDetails(status);
				nbError++;
			}
			// Free messages
			OBDonUDS_MsgFree(&msg_request);

		}
	}

	return nbError;
}

static void printSnapshot(obd_request_freeze_frame_data_response* parsed_response)
{
	if (parsed_response == NULL)
		return;

	printf("-> Response from ECU #%d : ", parsed_response->ecu_address);
	if (parsed_response->nrc != 0)
	{
		printf("Negative response code 0x%02X\n", parsed_response->nrc);
	}
	else
	{
		printf("Report type 0x%02X, DTC 0x%06X, DTC status 0x%02X, Record number 0x%02X",
			parsed_response->report_type, parsed_response->dtc_number, parsed_response->status_of_dtc,
			parsed_response->record_number);
		for (uint32_t id = 0; id < parsed_response->nb_identifiers; ++id)
		{
			printf("\n  -> Identifier %d : ", id + 1);
			printf("Data identifier 0x%04X,", parsed_response->identifiers[id].data_identifier);
			for (uint32_t idata = 0; idata < parsed_response->identifiers[id].size; ++idata)
			{
				printf(" %02X", parsed_response->identifiers[id].data[idata]);
				// Example of detailed parsing : for DID F404
				if (parsed_response->identifiers[id].data_identifier == 0xF404 && parsed_response->identifiers[id].size == 1)
				{
					printf("\n  -> ");
					parse_F404_CalculatedLoadValue_Details(parsed_response->identifiers[id].data[0]);
				}
				// TODO: parse details of other elements
			}
		}
		printf("\n");
	}
}

int displayRequestEmissionRelatedDiagnosticTroubleCodesWithPendingStatus_19_42(cantp_handle channel, obd_msgprotocol protocol, DTC_array_t DTCarray[MAX_NB_ECU])
{
	int nbError = 0;
	obd_status status;

	printf("\nRequest Emission-Related Diagnostic Trouble Codes With Pending Status physically for each ECU\n");

	uint8_t availableECUs = 0;
	OBDonUDS_GetValue(channel, POBDONUDS_PARAMETER_AVAILABLE_ECUS, &availableECUs, sizeof(availableECUs));

	for (uint8_t ecuIndex = 0; ecuIndex < availableECUs; ++ecuIndex)
	{
		obd_netaddrinfo nai;
		nai.protocol = protocol;
		nai.source_addr = PUDS_ISO_15765_4_ADDR_TEST_EQUIPMENT;
		nai.target_addr = ecuIndex + 1;
		nai.target_type = OBD_ADDRESSING_PHYSICAL;
		obd_msg msg_request;
		memset(&msg_request, 0, sizeof(msg_request));
		status = OBDonUDS_RequestPendingTroubleCodes(channel, nai, &msg_request, OBDONUDS_EMISSION_SYSTEM_GROUP, OBDONUDS_DTC_STATUS_PENDING, OBDONUDS_DTC_SEVERITY_CLASS_1);
		if (OBDonUDS_StatusIsOk(status, POBDONUDS_STATUS_OK, false))
		{
			// Wait for response
			obd_msg msg_response;
			memset(&msg_response, 0, sizeof(msg_response));
			obd_msg msg_request_confirmation;
			memset(&msg_request_confirmation, 0, sizeof(msg_request_confirmation));
			status = OBDonUDS_WaitForService(channel, &msg_request, &msg_response, &msg_request_confirmation);
			if (OBDonUDS_StatusIsOk(status, POBDONUDS_STATUS_OK, false))
			{
				// Parse response
				obd_request_trouble_codes_response parsed_response;
				memset(&parsed_response, 0, sizeof(parsed_response));
				status = OBDonUDS_ParseResponse_RequestPendingTroubleCodes(&msg_response, &parsed_response);
				if (OBDonUDS_StatusIsOk(status, POBDONUDS_STATUS_OK, false))
				{
					// Print parsed response
					printDTCParsed(&parsed_response);
					// Keep a list of DTC
					nbError += KeepDTC(DTCarray, ecuIndex, &parsed_response);
				}
				else
				{
					printf("-> Parse : ERROR (0x%X): ", status); printStatusDetails(status);
					nbError++;
				}
				// Free message
				OBDonUDS_ParsedResponseFree((obd_response_generic*)&parsed_response);
			}
			else
			{
				printf("-> Wait ECU #%d: ERROR (0x%X): ", ecuIndex + 1, status); printStatusDetails(status);
				nbError++;
			}

			// Free messages
			OBDonUDS_MsgFree(&msg_request_confirmation);
			OBDonUDS_MsgFree(&msg_response);
		}
		else
		{
			printf("-> Request : ERROR (0x%X): ", status); printStatusDetails(status);
			nbError++;
		}
		// Free messages
		OBDonUDS_MsgFree(&msg_request);
	}

	return nbError;
}

int displayRequestControlOfOnBoardSystemTestOrComponent_31_RID(cantp_handle channel, obd_msgprotocol protocol)
{
	int nbError = 0;

	printf("\nRequest Control Of On-Board System, Test, Or Component physically for each ECU for all supported RID\n");

	// Get supported rids
	uint8_t supportedRIDs_E0XX[DID_SUPPORTED_SIZE];
	uint8_t supportedRIDs_E1XX[DID_SUPPORTED_SIZE];
	OBDonUDS_GetValue(channel, POBDONUDS_PARAMETER_SUPPORTMASK_RIDS_E0XX, supportedRIDs_E0XX, sizeof(supportedRIDs_E0XX));
	OBDonUDS_GetValue(channel, POBDONUDS_PARAMETER_SUPPORTMASK_RIDS_E1XX, supportedRIDs_E1XX, sizeof(supportedRIDs_E1XX));

	// Request physically each ECU on supported RIDs (1 RID max per request)
	for (obd_DID_t rid = 0; rid < DID_SUPPORTED_SIZE; ++rid)
	{
		for (uint8_t ecuIndex = 0; ecuIndex < MAX_NB_ECU; ++ecuIndex)
		{
			if ((rid + 1) % 0x20 != 0) // avoid RID for "requesting supported" e.g. 0xE020
			{
				if ((supportedRIDs_E0XX[rid] >> ecuIndex) & 0x01)
				{
					nbError += displayRequestControlOfOnBoardSystemTestOrComponent_31_RID_one(channel, protocol, ecuIndex, 0xE000 + (rid + 1));
				}
				if ((supportedRIDs_E1XX[rid] >> ecuIndex) & 0x01)
				{
					nbError += displayRequestControlOfOnBoardSystemTestOrComponent_31_RID_one(channel, protocol, ecuIndex, 0xE100 + (rid + 1));
				}
			}
		}
	}

	return nbError;
}

int displayRequestControlOfOnBoardSystemTestOrComponent_31_RID_one(cantp_handle channel, obd_msgprotocol protocol, uint8_t ecuIndex, obd_DID_t RID)
{
	int nbError = 0;
	obd_msg msg_request;
	obd_status status;

	obd_netaddrinfo nai;
	nai.protocol = protocol;
	nai.source_addr = PUDS_ISO_15765_4_ADDR_TEST_EQUIPMENT;
	nai.target_addr = ecuIndex + 1;
	nai.target_type = OBD_ADDRESSING_PHYSICAL;
	memset(&msg_request, 0, sizeof(msg_request));
	status = OBDonUDS_RequestControlOperation(channel, nai, &msg_request, OBDONUDS_ROUTINE_START, RID, NULL, 0);
	if (OBDonUDS_StatusIsOk(status, POBDONUDS_STATUS_OK, false))
	{
		// Wait for response
		obd_msg msg_response;
		memset(&msg_response, 0, sizeof(msg_response));
		obd_msg msg_request_confirmation;
		memset(&msg_request_confirmation, 0, sizeof(msg_request_confirmation));
		status = OBDonUDS_WaitForService(channel, &msg_request, &msg_response, &msg_request_confirmation);
		if (OBDonUDS_StatusIsOk(status, POBDONUDS_STATUS_OK, false))
		{
			// Parse response
			obd_request_control_operation_response parsed_response;
			memset(&parsed_response, 0, sizeof(parsed_response));
			status = OBDonUDS_ParseResponse_RequestControlOperation(&msg_response, &parsed_response);
			if (OBDonUDS_StatusIsOk(status, POBDONUDS_STATUS_OK, false))
			{
				// Print parsed response
				printf("-> Response from ECU #%d : ", parsed_response.ecu_address);
				if (parsed_response.nrc != 0)
					printf("-> Negative response code 0x%02X\n", parsed_response.nrc);
				else
				{
					printf("Routine Control Type 0x%02X ", parsed_response.routine_control_type);
					printf("RID 0x%X ", parsed_response.routine_identifier);
					printf("Routine Info 0x%02X ", parsed_response.routine_info);
					for (uint32_t ip = 0; ip < parsed_response.nb_elements; ++ip)
					{
						printf("\n  -> Routine Status Element %d : 0x%02X", ip + 1, parsed_response.elements[ip]);
					}
					printf("\n");
				}
			}
			else
			{
				printf("-> Parse RID: ERROR (0x%X): ", status); printStatusDetails(status);
				nbError++;
			}
			// Free message
			OBDonUDS_ParsedResponseFree((obd_response_generic*)&parsed_response);
		}
		else
		{
			printf("-> Wait RID: ERROR (0x%X): ", status); printStatusDetails(status);
			nbError++;
		}

		// Free messages
		OBDonUDS_MsgFree(&msg_request_confirmation);
		OBDonUDS_MsgFree(&msg_response);
	}
	else
	{
		printf("-> Request RID: ERROR (0x%X): ", status); printStatusDetails(status);
		nbError++;
	}
	// Free messages
	OBDonUDS_MsgFree(&msg_request);

	return nbError;

}

int displayRequestVehicleInformation_22_ITID(cantp_handle channel, obd_msgprotocol protocol)
{
	int nbError = displayRequestVehicleInformation_22_ITID_F802_functional(channel, protocol);
	nbError += displayRequestVehicleInformation_22_ITID_All(channel, protocol);
	return nbError;

}

int displayRequestVehicleInformation_22_ITID_F802_functional(cantp_handle channel, obd_msgprotocol protocol)
{
	printf("\nRequest Vehicle Information functionally for ITID F802\n");

	return GetVehicleIdentificationNumber(channel, protocol);
}

int displayRequestVehicleInformation_22_ITID_All(cantp_handle channel, obd_msgprotocol protocol)
{
	int nbError = 0;

	printf("\nRequest Vehicle Information physically for each ECU for all supported ITID\n");

	// Get supported dids
	uint8_t supportedITIDs[DID_SUPPORTED_SIZE];
	OBDonUDS_GetValue(channel, POBDONUDS_PARAMETER_SUPPORTMASK_ITIDS, supportedITIDs, sizeof(supportedITIDs));

	// Request physically each ECU on supported ITIDs (1 ITID max per request)
	for (obd_DID_t itid = 0; itid < DID_SUPPORTED_SIZE; ++itid)
	{
		for (uint8_t ecuIndex = 0; ecuIndex < MAX_NB_ECU; ++ecuIndex)
		{
			if ((itid + 1) % 0x20 != 0) // avoid ITID for "requesting supported" e.g. 0xF820
			{
				if ((supportedITIDs[itid] >> ecuIndex) & 0x01)
				{
					nbError += displayRequestVehicleInformation_22_ITID_One(channel, protocol, ecuIndex, 0xF800 + (itid + 1));
				}
			}
		}
	}

	return nbError;
}

int displayRequestVehicleInformation_22_ITID_One(cantp_handle channel, obd_msgprotocol protocol, uint8_t ecuIndex, obd_DID_t ITID)
{
	int nbError = 0;
	obd_msg msg_request;
	obd_status status;

	obd_netaddrinfo nai;
	nai.protocol = protocol;
	nai.source_addr = PUDS_ISO_15765_4_ADDR_TEST_EQUIPMENT;
	nai.target_addr = ecuIndex + 1;
	nai.target_type = OBD_ADDRESSING_PHYSICAL;
	memset(&msg_request, 0, sizeof(msg_request));
	status = OBDonUDS_RequestVehicleInformation(channel, nai, &msg_request, &ITID, 1);
	if (OBDonUDS_StatusIsOk(status, POBDONUDS_STATUS_OK, false))
	{
		// Wait for responses
		obd_msg msg_response;
		memset(&msg_response, 0, sizeof(msg_response));
		obd_msg msg_request_confirmation;
		memset(&msg_request_confirmation, 0, sizeof(msg_request_confirmation));
		status = OBDonUDS_WaitForService(channel, &msg_request, &msg_response, &msg_request_confirmation);
		if (OBDonUDS_StatusIsOk(status, POBDONUDS_STATUS_OK, false))
		{
			// Parse response
			obd_request_vehicle_information_response parsed_response;
			memset(&parsed_response, 0, sizeof(parsed_response));
			status = OBDonUDS_ParseResponse_RequestVehicleInformation(&msg_response, &parsed_response);
			if (OBDonUDS_StatusIsOk(status, POBDONUDS_STATUS_OK, false))
			{
				// Print parsed response
				printf("-> Response from ECU #%d : ", parsed_response.ecu_address);
				if (parsed_response.nrc != 0)
					printf("-> Negative response code 0x%02X\n", parsed_response.nrc);
				else
				{
					printf(" ITID 0x%X, %d bytes\n\t\t", parsed_response.data_identifier, parsed_response.nb_elements);

					for (uint32_t j = 0; j < parsed_response.nb_elements; ++j)
					{
						printf("%02X ", parsed_response.elements[j]);
					}

					// Example of detailed parsing : for ITID F802, F804, F80A which are like strings of maximum 20 ascii characters
					switch (parsed_response.data_identifier)
					{
					case 0xF802:
					{
						char resultString[BUFFER_SIZE];
						memset(resultString, 0, BUFFER_SIZE);
						memcpy_s(resultString, BUFFER_SIZE, (const char*)parsed_response.elements, parsed_response.nb_elements);
						printf("\n  -> VIN: '%s'", resultString);
					}
					break;
					case 0xF804:
					{
						// The response may contain fill bytes 00 that we print as blank spaces
						printf("\n  -> CALID: '");
						size_t i = 0;
						char resultString[BUFFER_SIZE];
						while (i < parsed_response.nb_elements)
						{
							memset(resultString, 0, BUFFER_SIZE);
							strcpy_s(resultString, BUFFER_SIZE, (const char*)&parsed_response.elements[i]);
							printf("%s ", resultString);
							i += strlen(resultString) + 1;
						}
						printf("'");
					}
					break;
					case 0xF80A:
					{
						char resultString[BUFFER_SIZE];
						memset(resultString, 0, BUFFER_SIZE);
						memcpy_s(resultString, BUFFER_SIZE, (const char*)parsed_response.elements, parsed_response.nb_elements);
						printf("\n  -> ECU: '%s'", resultString);
					}
					break;
					default:
						// TODO: parse details of other elements
						break;
					}

					printf("\n");
				}
			}
			else
			{
				printf("-> Parse ITID: ERROR (0x%X): ", status); printStatusDetails(status);
				nbError++;
			}
			// Free message
			OBDonUDS_ParsedResponseFree((obd_response_generic*)&parsed_response);
		}
		else
		{
			printf("-> Wait ITID 0x%X ECU #%d: ERROR (0x%X): ", ITID, ecuIndex + 1, status); printStatusDetails(status);
			nbError++;
		}

		// Free messages
		OBDonUDS_MsgFree(&msg_request_confirmation);
		OBDonUDS_MsgFree(&msg_response);
	}
	else
	{
		printf("-> Request ITIDs: ERROR (0x%X): ", status); printStatusDetails(status);
		nbError++;
	}
	// Free messages
	OBDonUDS_MsgFree(&msg_request);

	return nbError;

}

int displayRequestEmissionRelatedDiagnosticTroubleCodesWithPermanentStatus_19_55(cantp_handle channel, obd_msgprotocol protocol)
{
	int nbError = 0;
	obd_status status;

	printf("\nRequest Emission-Related Diagnostic Trouble Codes With Permanent Status physically for each ECU\n");

	uint8_t availableECUs = 0;
	OBDonUDS_GetValue(channel, POBDONUDS_PARAMETER_AVAILABLE_ECUS, &availableECUs, sizeof(availableECUs));

	for (uint8_t ecuIndex = 0; ecuIndex < availableECUs; ++ecuIndex)
	{
		obd_netaddrinfo nai;
		nai.protocol = protocol;
		nai.source_addr = PUDS_ISO_15765_4_ADDR_TEST_EQUIPMENT;
		nai.target_addr = ecuIndex + 1;
		nai.target_type = OBD_ADDRESSING_PHYSICAL;
		obd_msg msg_request;
		memset(&msg_request, 0, sizeof(msg_request));
		status = OBDonUDS_RequestPermanentTroubleCodes(channel, nai, &msg_request, OBDONUDS_EMISSION_SYSTEM_GROUP);
		if (OBDonUDS_StatusIsOk(status, POBDONUDS_STATUS_OK, false))
		{
			// Wait for response
			obd_msg msg_response;
			memset(&msg_response, 0, sizeof(msg_response));
			obd_msg msg_request_confirmation;
			memset(&msg_request_confirmation, 0, sizeof(msg_request_confirmation));
			status = OBDonUDS_WaitForService(channel, &msg_request, &msg_response, &msg_request_confirmation);
			if (OBDonUDS_StatusIsOk(status, POBDONUDS_STATUS_OK, false))
			{
				// Parse response
				obd_request_permanent_trouble_codes_response parsed_response;
				memset(&parsed_response, 0, sizeof(parsed_response));
				status = OBDonUDS_ParseResponse_RequestPermanentTroubleCodes(&msg_response, &parsed_response);
				if (OBDonUDS_StatusIsOk(status, POBDONUDS_STATUS_OK, false))
				{
					// Print parsed response
					printf("-> Response from ECU #%d : ", parsed_response.ecu_address);
					if (parsed_response.nrc != 0)
						printf("-> Negative response code 0x%02X\n", parsed_response.nrc);
					else
					{
						printf("-> Report type 0x%02X, Functional group identifier 0x%02X, DTC status availability mask 0x%02X, DTC format identifier 0x%02X",
							parsed_response.report_type, parsed_response.functional_group_identifier, parsed_response.DTC_status_availability_mask,
							parsed_response.DTC_format_identifier);
						for (uint32_t ip = 0; ip < parsed_response.nb_elements; ++ip)
						{
							printf("\n  -> Element %d : ", ip + 1);
							printDTC(&(parsed_response.elements[ip]));
						}
						printf("\n");
					}
				}
				else
				{
					printf("-> Parse : ERROR (0x%X): ", status); printStatusDetails(status);
					nbError++;
				}
				// Free message
				OBDonUDS_ParsedResponseFree((obd_response_generic*)&parsed_response);
			}
			else
			{
				printf("-> Wait ECU #%d: ERROR (0x%X): ", ecuIndex + 1, status); printStatusDetails(status);
				nbError++;
			}

			// Free messages
			OBDonUDS_MsgFree(&msg_request_confirmation);
			OBDonUDS_MsgFree(&msg_response);
		}
		else
		{
			printf("-> Request : ERROR (0x%X): ", status); printStatusDetails(status);
			nbError++;
		}
		// Free messages
		OBDonUDS_MsgFree(&msg_request);
	}

	return nbError;
}

void printDTC(obd_trouble_code* object)
{
	if (object == NULL)
		return;

	printf("DTC identifier 0x%06X, Status of DTC 0x%02X", object->DTC_identifier, object->status_of_dtc);
}

void printDTCParsed(obd_request_trouble_codes_response* parsed_response)
{
	if (parsed_response == NULL)
		return;

	printf("-> Response from ECU #%d : ", parsed_response->ecu_address);
	if (parsed_response->nrc != 0)
		printf("-> Negative response code 0x%02X\n", parsed_response->nrc);
	else
	{
		printf("Report type 0x%02X, Functional group identifier 0x%02X, DTC status availability mask 0x%02X, DTC severity mask 0x%02X, DTC format identifier 0x%02X",
			parsed_response->report_type, parsed_response->functional_group_identifier, parsed_response->DTC_status_availability_mask,
			parsed_response->DTC_severity_mask, parsed_response->DTC_format_identifier);
		for (uint32_t ip = 0; ip < parsed_response->nb_elements; ++ip)
		{
			printf("\n  -> Element %d : ", ip + 1);
			printf("DTC identifier 0x%06X, Status 0x%02X, Severity 0x%02X", parsed_response->elements[ip].DTC_identifier,
				parsed_response->elements[ip].status_of_dtc, parsed_response->elements[ip].DTC_severity);
		}
		printf("\n");
	}
}

int KeepDTC(DTC_array_t DTCarray[MAX_NB_ECU], uint8_t ecuIndex, obd_request_trouble_codes_response* parsed_response)
{
	int nbError = 0;

	if (parsed_response == NULL || DTCarray == NULL)
		return nbError;

	if (parsed_response->nb_elements != 0)
	{
#ifdef __cplusplus
		DTCarray[ecuIndex].array = new obd_DTC_t[parsed_response->nb_elements];
#else
		DTCarray[ecuIndex].array = (obd_DTC_t*)malloc(parsed_response->nb_elements * sizeof(obd_DTC_t));
#endif
		if (DTCarray[ecuIndex].array == NULL)
		{
			printf(" Out of memory !!!\n");
			nbError++;
		}
		else
		{
			DTCarray[ecuIndex].size = parsed_response->nb_elements;
			for (uint32_t i = 0; i < parsed_response->nb_elements; ++i)
			{
				DTCarray[ecuIndex].array[i] = parsed_response->elements[i].DTC_identifier;
			}
		}
	}

	return nbError;
}

int displayRequestSupportedDTCExtendedRecord_19_1A(cantp_handle channel, obd_msgprotocol protocol, DTC_array_t DTCarray[MAX_NB_ECU][MAX_NB_EXTENDED_DTC_RECORD])
{
	int nbError = 0;

	printf("\nRequest Supported DTC Extended Record physically for each ECU\n");

	uint8_t availableECUs = 0;
	OBDonUDS_GetValue(channel, POBDONUDS_PARAMETER_AVAILABLE_ECUS, &availableECUs, sizeof(availableECUs));

	for (uint8_t ecuIndex = 0; ecuIndex < availableECUs; ++ecuIndex)
	{
		for (uint8_t extendedDataRecordIndex = 0; extendedDataRecordIndex < MAX_NB_EXTENDED_DTC_RECORD; ++extendedDataRecordIndex)
		{
			nbError += displayRequestSupportedDTCExtendedRecord_19_1A_one(channel, protocol, DTCarray, ecuIndex, extendedDataRecordIndex);
		}
	}

	return nbError;
}

int displayRequestSupportedDTCExtendedRecord_19_1A_one(cantp_handle channel, obd_msgprotocol protocol, DTC_array_t DTCarray[MAX_NB_ECU][MAX_NB_EXTENDED_DTC_RECORD],
	uint8_t ecuIndex, uint8_t extendedDataRecordIndex)
{
	int nbError = 0;
	obd_status status;

	obd_msg msg_request;
	memset(&msg_request, 0, sizeof(msg_request));
	obd_netaddrinfo nai;
	nai.protocol = protocol;
	nai.source_addr = PUDS_ISO_15765_4_ADDR_TEST_EQUIPMENT;
	nai.target_addr = ecuIndex + 1;
	nai.target_type = OBD_ADDRESSING_PHYSICAL;
	status = OBDonUDS_RequestSupportedDTCExtended(channel, nai, &msg_request, 0x90 + extendedDataRecordIndex);
	if (OBDonUDS_StatusIsOk(status, POBDONUDS_STATUS_OK, false))
	{
		// Wait for response
		obd_msg msg_response;
		memset(&msg_response, 0, sizeof(msg_response));
		obd_msg msg_request_confirmation;
		memset(&msg_request_confirmation, 0, sizeof(msg_request_confirmation));
		status = OBDonUDS_WaitForService(channel, &msg_request, &msg_response, &msg_request_confirmation);
		if (OBDonUDS_StatusIsOk(status, POBDONUDS_STATUS_OK, false))
		{
			// Parse response
			obd_request_supported_dtc_extended_response parsed_response;
			memset(&parsed_response, 0, sizeof(parsed_response));
			status = OBDonUDS_ParseResponse_RequestSupportedDTCExtended(&msg_response, &parsed_response);
			if (OBDonUDS_StatusIsOk(status, POBDONUDS_STATUS_OK, false))
			{
				// Print parsed response
				printf("-> Response from ECU #%d : ", parsed_response.ecu_address);
				if (parsed_response.nrc != 0)
					printf("-> Negative response code 0x%02X\n", parsed_response.nrc);
				else
				{
					printf("-> Report type 0x%02X, DTC status availability mask 0x%02X, extended DTC data record number 0x%02X",
						parsed_response.report_type, parsed_response.DTC_status_availability_mask,
						parsed_response.DTC_extended_data_record_number);
					for (uint32_t ip = 0; ip < parsed_response.nb_elements; ++ip)
					{
						printf("\n  -> Element %d : ", ip + 1);
						printDTC(&(parsed_response.elements[ip]));
					}
					printf("\n");
					// Keep a list of DTC
					nbError += KeepDTCextended(DTCarray, ecuIndex, extendedDataRecordIndex, &parsed_response);
				}
			}
			else
			{
				printf("-> Parse : ERROR (0x%X): ", status); printStatusDetails(status);
				nbError++;
			}
			// Free message
			OBDonUDS_ParsedResponseFree((obd_response_generic*)&parsed_response);
		}
		else
		{
			printf("-> Wait failed for ECU #%d record 0x%02X (0x%08X): ", ecuIndex + 1, extendedDataRecordIndex + 0x90, status); printStatusDetails(status);
			nbError++;
		}

		// Free messages
		OBDonUDS_MsgFree(&msg_request_confirmation);
		OBDonUDS_MsgFree(&msg_response);
	}
	else
	{
		printf("-> Request : ERROR (0x%X): ", status); printStatusDetails(status);
		nbError++;
	}
	// Free messages
	OBDonUDS_MsgFree(&msg_request);

	return nbError;
}

int KeepDTCextended(DTC_array_t DTCarray[MAX_NB_ECU][MAX_NB_EXTENDED_DTC_RECORD], uint8_t ecuIndex, uint8_t extendedDataRecordIndex,
	obd_request_supported_dtc_extended_response* parsed_response)
{
	int nbError = 0;

	if (parsed_response == NULL || DTCarray == NULL)
		return nbError;

	if (parsed_response->nb_elements != 0)
	{
#ifdef __cplusplus
		DTCarray[ecuIndex][extendedDataRecordIndex].array = new obd_DTC_t[parsed_response->nb_elements];
#else
		DTCarray[ecuIndex][extendedDataRecordIndex].array = (obd_DTC_t*)malloc(parsed_response->nb_elements * sizeof(obd_DTC_t));
#endif
		if (DTCarray[ecuIndex][extendedDataRecordIndex].array == NULL)
		{
			printf(" Out of memory !!!\n");
			nbError++;
		}
		else
		{
			DTCarray[ecuIndex][extendedDataRecordIndex].size = parsed_response->nb_elements;
			for (uint32_t i = 0; i < parsed_response->nb_elements; ++i)
			{
				DTCarray[ecuIndex][extendedDataRecordIndex].array[i] = parsed_response->elements[i].DTC_identifier;
			}
		}
	}

	return nbError;
}

int displayRequestDTCExtendedDataRecord_19_06(cantp_handle channel, obd_msgprotocol protocol, DTC_array_t DTCarray[MAX_NB_ECU][MAX_NB_EXTENDED_DTC_RECORD])
{
	int nbError = 0;

	printf("\nRequest DTC Extended Data Record physically for each ECU\n");

	for (uint8_t ecuIndex = 0; ecuIndex < MAX_NB_ECU; ++ecuIndex)
	{
		for (uint8_t extendedDataRecordIndex = 0; extendedDataRecordIndex < MAX_NB_EXTENDED_DTC_RECORD; ++extendedDataRecordIndex)
		{
			nbError += displayRequestDTCExtendedDataRecord_19_06_one(channel, protocol, DTCarray, ecuIndex, extendedDataRecordIndex);
		}
	}

	return nbError;
}

int displayRequestDTCExtendedDataRecord_19_06_one(cantp_handle channel, obd_msgprotocol protocol, DTC_array_t DTCarray[MAX_NB_ECU][MAX_NB_EXTENDED_DTC_RECORD],
	uint8_t ecuIndex, uint8_t extendedDataRecordIndex)
{
	int nbError = 0;
	obd_status status;

	obd_netaddrinfo nai;
	nai.protocol = protocol;
	nai.source_addr = PUDS_ISO_15765_4_ADDR_TEST_EQUIPMENT;
	nai.target_addr = ecuIndex + 1;
	nai.target_type = OBD_ADDRESSING_PHYSICAL;

	for (size_t indexDTC = 0; indexDTC < DTCarray[ecuIndex][extendedDataRecordIndex].size; ++indexDTC)
	{
		obd_msg msg_request;
		memset(&msg_request, 0, sizeof(msg_request));
		status = OBDonUDS_RequestDTCExtended(channel, nai, &msg_request, DTCarray[ecuIndex][extendedDataRecordIndex].array[indexDTC], 0x90 + extendedDataRecordIndex);
		if (OBDonUDS_StatusIsOk(status, POBDONUDS_STATUS_OK, false))
		{
			// Wait for response
			obd_msg msg_response;
			memset(&msg_response, 0, sizeof(msg_response));
			obd_msg msg_request_confirmation;
			memset(&msg_request_confirmation, 0, sizeof(msg_request_confirmation));
			status = OBDonUDS_WaitForService(channel, &msg_request, &msg_response, &msg_request_confirmation);
			if (OBDonUDS_StatusIsOk(status, POBDONUDS_STATUS_OK, false))
			{
				// Parse response
				obd_request_dtc_extended_response parsed_response;
				memset(&parsed_response, 0, sizeof(parsed_response));
				status = OBDonUDS_ParseResponse_RequestDTCExtended(&msg_response, &parsed_response);
				if (OBDonUDS_StatusIsOk(status, POBDONUDS_STATUS_OK, false))
				{
					// Print parsed response
					printf("-> Response from ECU #%d : ", parsed_response.ecu_address);
					if (parsed_response.nrc != 0)
						printf("-> Negative response code 0x%02X\n", parsed_response.nrc);
					else
					{
						printf("-> Report type 0x%02X, DTC identifier 0x%06X, DTC status 0x%02X, DTC extended data record number 0x%02X, %d bytes\n\t\t",
							parsed_response.report_type, parsed_response.DTC_identifier, parsed_response.status_of_dtc,
							parsed_response.DTC_extended_data_record_number, parsed_response.nb_elements);
						for (uint32_t ip = 0; ip < parsed_response.nb_elements; ++ip)
						{
							printf("%02X ", parsed_response.elements[ip]);
						}
						printf("\n");
						// Example of detailed parsing : for extended data record number 0x92 "DTC based test result"
						if (parsed_response.DTC_extended_data_record_number == 0x92)
						{
							if (parsed_response.nb_elements >= 1)
							{
								size_t j = 0;
								uint8_t numberOfTestResults = parsed_response.elements[j++];
								if (parsed_response.nb_elements >= 1U + (8U * numberOfTestResults))
								{
									for (uint8_t indexTest = 0; indexTest < numberOfTestResults; ++indexTest)
									{
										uint16_t unit_and_scaling = makeUint16(parsed_response.elements[j], parsed_response.elements[j + 1]);
										// try to use similar function - cannot work if unit_and_scaling first byte is not 0 - TODO: parse details in other case
										if (parsed_response.elements[j] == 0x00)
										{
											obd_test_data_object object;
											memset(&object, 0, sizeof(object));
											object.unit_and_scaling = (uint8_t)unit_and_scaling;
											j += 2;
											object.test_value[0] = parsed_response.elements[j++];
											object.test_value[1] = parsed_response.elements[j++];
											object.min_test_limit[0] = parsed_response.elements[j++];
											object.min_test_limit[1] = parsed_response.elements[j++];
											object.max_test_limit[0] = parsed_response.elements[j++];
											object.max_test_limit[1] = parsed_response.elements[j++];
											printf("  -> ");
											printTestObject(&object);
											printf("\n");
										}
									}
								}
							}
						}
						// TODO: parse details of other elements
					}
				}
				else
				{
					printf("-> Parse : ERROR (0x%X): ", status); printStatusDetails(status);
					nbError++;
				}
				// Free message
				OBDonUDS_ParsedResponseFree((obd_response_generic*)&parsed_response);
			}
			else
			{
				printf("-> Wait ECU #%d record 0x%02X DTC 0x%06x: ERROR (0x%X): ", ecuIndex + 1, extendedDataRecordIndex + 0x90,
					DTCarray[ecuIndex][extendedDataRecordIndex].array[indexDTC], status);
				printStatusDetails(status);
				nbError++;
			}

			// Free messages
			OBDonUDS_MsgFree(&msg_request_confirmation);
			OBDonUDS_MsgFree(&msg_response);
		}
		else
		{
			printf("-> Request : ERROR (0x%X): ", status); printStatusDetails(status);
			nbError++;
		}
		// Free messages
		OBDonUDS_MsgFree(&msg_request);
	}

	return nbError;
}

int displayRequestDTCsForAReadinessGroup_19_56(cantp_handle channel, obd_msgprotocol protocol)
{
	int nbError = 0;

	printf("\nRequest DTCs For first Readiness Groups physically for each ECU\n");

	uint8_t availableECUs = 0;
	OBDonUDS_GetValue(channel, POBDONUDS_PARAMETER_AVAILABLE_ECUS, &availableECUs, sizeof(availableECUs));

	for (uint8_t ecuIndex = 0; ecuIndex < availableECUs; ++ecuIndex)
	{
		for (uint8_t readinessGroup = 0x00; readinessGroup <= 0x14; ++readinessGroup)
		{
			nbError += displayRequestDTCsForAReadinessGroup_19_56_one(channel, protocol, ecuIndex, readinessGroup);
		}
	}

	return nbError;
}

int displayRequestDTCsForAReadinessGroup_19_56_one(cantp_handle channel, obd_msgprotocol protocol, uint8_t ecuIndex, uint8_t readinessGroup)
{
	int nbError = 0;

	obd_status status;
	obd_msg msg_request;
	memset(&msg_request, 0, sizeof(msg_request));
	obd_netaddrinfo nai;
	nai.protocol = protocol;
	nai.source_addr = PUDS_ISO_15765_4_ADDR_TEST_EQUIPMENT;
	nai.target_addr = ecuIndex + 1;
	nai.target_type = OBD_ADDRESSING_PHYSICAL;

	status = OBDonUDS_RequestDTCForAReadinessGroup(channel, nai, &msg_request, OBDONUDS_EMISSION_SYSTEM_GROUP, readinessGroup);
	if (OBDonUDS_StatusIsOk(status, POBDONUDS_STATUS_OK, false))
	{
		// Wait for responses
		obd_msg msg_response;
		memset(&msg_response, 0, sizeof(msg_response));
		obd_msg msg_request_confirmation;
		memset(&msg_request_confirmation, 0, sizeof(msg_request_confirmation));
		status = OBDonUDS_WaitForService(channel, &msg_request, &msg_response, &msg_request_confirmation);
		if (OBDonUDS_StatusIsOk(status, POBDONUDS_STATUS_OK, false))
		{
			// Parse response
			obd_request_dtc_for_a_readiness_group_response parsed_response;
			memset(&parsed_response, 0, sizeof(parsed_response));
			status = OBDonUDS_ParseResponse_RequestDTCForAReadinessGroup(&msg_response, &parsed_response);
			if (OBDonUDS_StatusIsOk(status, POBDONUDS_STATUS_OK, false))
			{
				// Print parsed response
				printf("-> Response from ECU #%d : ", parsed_response.ecu_address);
				if (parsed_response.nrc != 0)
					printf("-> Negative response code 0x%02X\n", parsed_response.nrc);
				else
				{
					printf("-> Report type 0x%02X, Functional group identifier 0x%02X, DTC status availability mask 0x%02X, DTC format identifier 0x%02X, readiness group identifier 0x%02X",
						parsed_response.report_type, parsed_response.functional_group_identifier, parsed_response.DTC_status_availability_mask,
						parsed_response.DTC_format_identifier, parsed_response.readiness_group_identifier);
					for (uint32_t ip = 0; ip < parsed_response.nb_elements; ++ip)
					{
						printf("\n  -> Element %d : ", ip + 1);
						printDTC(&(parsed_response.elements[ip]));
					}
					printf("\n");
				}
			}
			else
			{
				printf("-> Parse : ERROR (0x%X): ", status); printStatusDetails(status);
				nbError++;
			}
			// Free message
			OBDonUDS_ParsedResponseFree((obd_response_generic*)&parsed_response);
		}
		else
		{
			printf("-> Wait failed for ECU #%d Group 0x%02X (0x%08X): ", ecuIndex + 1, readinessGroup, status); printStatusDetails(status);
			nbError++;
		}

		// Free messages
		OBDonUDS_MsgFree(&msg_request_confirmation);
		OBDonUDS_MsgFree(&msg_response);
	}
	else
	{
		printf("-> Request : ERROR (0x%X): ", status); printStatusDetails(status);
		nbError++;
	}
	// Free messages
	OBDonUDS_MsgFree(&msg_request);

	return nbError;
}

int main()
{
	obd_status status = POBDONUDS_STATUS_OK;
	cantp_handle channel = PCANTP_HANDLE_NONEBUS;
	int nbErr = 0;
	char c = 0;
	uint8_t debugValue = POBDONUDS_DEBUG_LVL_DEBUG;

	// Get API version
	char buffer[BUFFER_SIZE];
	memset(buffer, 0, BUFFER_SIZE);
	status = OBDonUDS_GetValue(PCANTP_HANDLE_NONEBUS, POBDONUDS_PARAMETER_API_VERSION, buffer, sizeof(buffer));
	printf("Get API version (%s): %s\n", STATUS_OK_KO(status), buffer);

#if 0
	// Configure logs to get general debug information
	status = OBDonUDS_SetValue(PCANTP_HANDLE_NONEBUS, POBDONUDS_PARAMETER_DEBUG, &debugValue, sizeof(debugValue));
	printf("Configure general logs (%s)\n", STATUS_OK_KO(status));
#endif

	// Initialize
	channel = PCANTP_HANDLE_USBBUS1; // TODO: modify the value according to your available PCAN devices.
	printf("Channel : 0x%02x, Channel status : 0x%02x\n", channel, OBDonUDS_GetStatus(channel));
	printf("Connecting with automatic detection of baudrate...\n");
	status = OBDonUDS_Initialize(channel, (cantp_baudrate)OBD_BAUDRATE_AUTODETECT, (cantp_hwtype)(0), 0, 0);
	printf("Initialize (%s)\n", STATUS_OK_KO(status));

	if (OBDonUDS_StatusIsOk(status, POBDONUDS_STATUS_OK, false))
	{
		obd_baudrate baudrate = (obd_baudrate)0;
		status = OBDonUDS_GetValue(channel, POBDONUDS_PARAMETER_BAUDRATE, &baudrate, sizeof(baudrate));
		printf("-> Baudrate (%s): %s\n", STATUS_OK_KO(status), (baudrate == OBD_BAUDRATE_500K) ? "500 kbit/s" : (baudrate == OBD_BAUDRATE_250K) ? "250kbit/s" : "unknown");

		obd_msgprotocol canIdLen = (obd_msgprotocol)0;
		status = OBDonUDS_GetValue(channel, POBDONUDS_PARAMETER_CAN_ID_LENGTH, &canIdLen, sizeof(canIdLen));
		printf("-> Can Id length (%s): %s\n", STATUS_OK_KO(status), (canIdLen == OBD_MSGPROTOCOL_11BIT) ? "11" : (canIdLen == OBD_MSGPROTOCOL_29BIT) ? "29" : "unknown");

#if 0
		// Configure logs to get channel debug information
		status = OBDonUDS_SetValue(channel, POBDONUDS_PARAMETER_DEBUG, &debugValue, sizeof(debugValue));
		printf("Configure channel logs %s\n", STATUS_OK_KO(status));
#endif

		// Request and print Vehicle Identification Number
		nbErr += GetVehicleIdentificationNumber(channel, canIdLen);

		// Show supported IDS
		nbErr += GetSupportedMask(channel);

		// make requests to every supported services

		nbErr += displayRequestCurrentPowertrainDiagnosticData_22_DID(channel, canIdLen);

		// array of confirmed DTC, allocated and filled by the request "emission-related DTC with confirmed status", to be used later by the request "powertrain snapshot data"
		DTC_array_t ConfirmedDTCs[MAX_NB_ECU];
		memset(ConfirmedDTCs, 0, sizeof(ConfirmedDTCs));

		nbErr += displayRequestEmissionRelatedDiagnosticTroubleCodesWithConfirmedStatus_19_42(channel, canIdLen, ConfirmedDTCs);

		// array of pending DTC, allocated and filled by the request "emission-related DTC with pending status", to be used later by the request "powertrain snapshot data"
		DTC_array_t PendingDTCs[MAX_NB_ECU];
		memset(PendingDTCs, 0, sizeof(PendingDTCs));

		nbErr += displayRequestEmissionRelatedDiagnosticTroubleCodesWithPendingStatus_19_42(channel, canIdLen, PendingDTCs);

		nbErr += displayRequestPowertrainFreezeFrameData_19_04(channel, canIdLen, ConfirmedDTCs, PendingDTCs);

		printf("\n\nDo you wish to Clear/Reset Emission-Related Diagnostic Information of group of ALL Trouble Codes ? (y/n) ");
		scanf_s("%c", &c, 1);
		if (c == 'y' || c == 'Y')
			nbErr += displayClearResetEmissionRelatedDiagnosticInformation_14(channel, canIdLen, OBDONUDS_DTC_ALL_GROUPS);
		while ((c = getchar()) != '\n' && c != EOF)
			;/* discard */

		printf("\n\nDo you wish to Clear/Reset Emission-Related Diagnostic Information of group of EMISSION-SYSTEM Trouble Codes ? (y/n) ");
		scanf_s("%c", &c, 1);
		if (c == 'y' || c == 'Y')
			nbErr += displayClearResetEmissionRelatedDiagnosticInformation_14(channel, canIdLen, OBDONUDS_DTC_EMISSION_SYSTEM_GROUP);
		while ((c = getchar()) != '\n' && c != EOF)
			;/* discard */

		nbErr += displayRequestOnBoardMonitoringTestResultsForSpecificMonitoredSystems_22_MID(channel, canIdLen);

		nbErr += displayRequestControlOfOnBoardSystemTestOrComponent_31_RID(channel, canIdLen);

		nbErr += displayRequestVehicleInformation_22_ITID(channel, canIdLen);

		nbErr += displayRequestEmissionRelatedDiagnosticTroubleCodesWithPermanentStatus_19_55(channel, canIdLen);

		// array of extended DTC, allocated and filled by the request "supported DTC extended record", to be used later by the request "DTC extended data record"
		DTC_array_t extendedDTCs[MAX_NB_ECU][MAX_NB_EXTENDED_DTC_RECORD];
		memset(extendedDTCs, 0, sizeof(extendedDTCs));

		nbErr += displayRequestSupportedDTCExtendedRecord_19_1A(channel, canIdLen, extendedDTCs);

		nbErr += displayRequestDTCExtendedDataRecord_19_06(channel, canIdLen, extendedDTCs);

		nbErr += displayRequestDTCsForAReadinessGroup_19_56(channel, canIdLen);

		// free memory
		for (uint8_t ecu = 0; ecu < MAX_NB_ECU; ++ecu)
		{
			if (PendingDTCs[ecu].array != NULL)
			{
#ifdef __cplusplus
				delete[]PendingDTCs[ecu].array;
#else
				free(PendingDTCs[ecu].array);
#endif
				PendingDTCs[ecu].array = NULL;
				PendingDTCs[ecu].size = 0;
			}
			if (ConfirmedDTCs[ecu].array != NULL)
			{
#ifdef __cplusplus
				delete[]ConfirmedDTCs[ecu].array;
#else
				free(ConfirmedDTCs[ecu].array);
#endif
				ConfirmedDTCs[ecu].array = NULL;
				ConfirmedDTCs[ecu].size = 0;
			}
			for (uint8_t extendedDTCrecord = 0; extendedDTCrecord < MAX_NB_EXTENDED_DTC_RECORD; ++extendedDTCrecord)
			{
				if (extendedDTCs[ecu][extendedDTCrecord].array != NULL)
				{
#ifdef __cplusplus
					delete[]extendedDTCs[ecu][extendedDTCrecord].array;
#else
					free(extendedDTCs[ecu][extendedDTCrecord].array);
#endif
					extendedDTCs[ecu][extendedDTCrecord].array = NULL;
					extendedDTCs[ecu][extendedDTCrecord].size = 0;
				}
			}
		}
	}
	else
	{
		printf("Initialization Error (0x%X): ", status); printStatusDetails(status);
		nbErr++;
	}

	status = OBDonUDS_Uninitialize(channel);

	if (nbErr > 0)
	{
		printf("\n\nERROR : a total of %d errors occured.\n\n", nbErr);
	}
	else
	{
		printf("\n\nALL Transmissions succeeded !\n\n");
	}

	printf("\nPress <Enter> to quit...\n");
	getchar();
	return nbErr;
}