#include <stdio.h>

// OBDII
#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#include "lib/PCAN-OBDII.h"
// end of OBDII

#include "lib/PCAN-OBDonUDS.h"

#define OK_KO(test) (test)?"OK":"KO"
#define STATUS_OK_KO(test) OK_KO(OBDonUDS_StatusIsOk(test, POBDONUDS_STATUS_OK, false))
#define STATUS_OK_KO_OBDII(test) OK_KO(test == POBDII_ERROR_OK)

#define BUFFER_SIZE 256
#define MAX_NB_RESPONSES 16

static void printOBDonUDSvehicleIdentificationNumber(cantp_handle channel, obd_msgprotocol canIdLen);


void printOBDonUDSvehicleIdentificationNumber(cantp_handle channel, obd_msgprotocol canIdLen)
{
	// Request vehicle identification number (VIN) using functional addressing scheme
	obd_DID_t VINitid = 0xF802;
	obd_msg msg_request;
	memset(&msg_request, 0, sizeof(msg_request));
	obd_status status = OBDonUDS_RequestVehicleInformation(channel, canIdLen == OBD_MSGPROTOCOL_11BIT ? OBD_NAI_REQUEST_FUNCTIONAL_11B : OBD_NAI_REQUEST_FUNCTIONAL_29B,
		&msg_request, &VINitid, 1);
	printf("Request VIN (%s)\n", STATUS_OK_KO(status));
	if (OBDonUDS_StatusIsOk(status, POBDONUDS_STATUS_OK, false))
	{
		// Wait for responses
		obd_msg msg_responses[MAX_NB_RESPONSES];
		memset(&msg_responses, 0, sizeof(msg_responses));
		uint32_t nb_responses;
		obd_msg msg_request_confirmation;
		memset(&msg_request_confirmation, 0, sizeof(msg_request_confirmation));
		status = OBDonUDS_WaitForServiceFunctional(channel, &msg_request, MAX_NB_RESPONSES, false, msg_responses, &nb_responses, &msg_request_confirmation);
		printf("Wait for service functional (%s)\n", STATUS_OK_KO(status));
		if (OBDonUDS_StatusIsOk(status, POBDONUDS_STATUS_OK, false))
		{
			printf("->  Received %d responses\n", nb_responses);
			// Parse responses
			for (uint32_t ir = 0; ir < nb_responses; ++ir)
			{
				obd_request_vehicle_information_response parsed_response;
				memset(&parsed_response, 0, sizeof(parsed_response));
				status = OBDonUDS_ParseResponse_RequestVehicleInformation(&(msg_responses[ir]), &parsed_response);
				printf("Parse response #%d (%s):\n", ir + 1, STATUS_OK_KO(status));
				if (OBDonUDS_StatusIsOk(status, POBDONUDS_STATUS_OK, false))
				{
					if (parsed_response.nrc != 0)
					{
						printf("-> Negative response code: 0x%02X\n", parsed_response.nrc);
					}
					else
					{
						printf("-> Vehicle Identification Number: ");
						for (uint32_t i = 0; i < parsed_response.nb_elements; ++i)
							printf("%c", parsed_response.elements[i]);
						printf("\n");
					}
				}
				else {
					printf("-> failed status: 0x%08X\n", status);
				}
				// Free message
				OBDonUDS_ParsedResponseFree((obd_response_generic*)&parsed_response);
			}
		}
		else {
			printf("-> failed status: 0x%08X\n", status);
		}
		// Free messages
		OBDonUDS_MsgFree(&msg_request_confirmation);
		for (uint32_t ir = 0; ir < nb_responses; ++ir)
			OBDonUDS_MsgFree(&(msg_responses[ir]));
	}
	else {
		printf("-> failed status: 0x%08X\n", status);
	}
	// Free messages
	OBDonUDS_MsgFree(&msg_request);
}

int main()
{
	int result = -1;
	// Get API version
	char buffer[BUFFER_SIZE];
	memset(buffer, 0, BUFFER_SIZE);
	obd_status status = OBDonUDS_GetValue(PCANTP_HANDLE_NONEBUS, POBDONUDS_PARAMETER_API_VERSION, buffer, sizeof(buffer));
	printf("Get API version (%s): %s\n", STATUS_OK_KO(status), buffer);

	cantp_handle channel = PCANTP_HANDLE_USBBUS1; // TODO: modify the value according to your available PCAN devices.
	printf("Channel : 0x%02x, Channel status : 0x%02x\n", channel, OBDonUDS_GetStatus(channel));
	printf("Trying to find OBDonEDS (OBDII) ECUs\n");
	status = OBDonUDS_FindOBDonEDS(channel, (cantp_baudrate)OBD_BAUDRATE_AUTODETECT, (cantp_hwtype)(0), 0, 0);
	printf("FindOBDonEDS (%s)\n", STATUS_OK_KO(status));
	if (OBDonUDS_StatusIsOk(status, POBDONUDS_STATUS_OK, false))
	{
		result = 1;
		printf("-> Found OBDonEDS (OBDII) ECUs\n");
		printf("Trying to initialize communication with OBDonEDS (OBDII) API\n");
		TPOBDIIStatus obdIIstatus = OBDII_Initialize((TPOBDIICANHandle)channel, POBDII_BAUDRATE_AUTODETECT, 0, 0, 0);
		printf("OBDII_Initialize (%s)\n", STATUS_OK_KO_OBDII(obdIIstatus));
		if (obdIIstatus == POBDII_ERROR_OK)
		{
			result = 0;
			// Request Vehicule Identification Number with OBDonEDS (OBDII) API
			TPOBDIIInfoData dataIT;
			obdIIstatus = OBDII_RequestVehicleInformation(channel, 0x02, &dataIT, 1);
			printf("OBDII_RequestVehicleInformation (%s)\n", STATUS_OK_KO_OBDII(obdIIstatus));
			if (obdIIstatus == POBDII_ERROR_OK)
				printf("Vehicule Identification Number: '%s'\n\n", dataIT.TEXT);
			OBDII_Uninitialize(channel);
		}
	}
	else
	{
		if (OBDonUDS_StatusIsOk(status, POBDONUDS_STATUS_UNSUPPORTED_ECUS, false))
		{
			result = 2;
			printf("-> Did not find OBDonEDS (OBDII) ECUs\n");
			printf("Trying to initialize communication with OBDonUDS API\n");
			status = OBDonUDS_Initialize(channel, (cantp_baudrate)OBD_BAUDRATE_AUTODETECT, (cantp_hwtype)(0), 0, 0);
			printf("Initialize (%s)\n", STATUS_OK_KO(status));
			if (OBDonUDS_StatusIsOk(status, POBDONUDS_STATUS_OK, false))
			{
				result = 0;
				printf("-> Initialization with OBDonUDS API succeeded : almost one ECU found\n");
				uint8_t numberOfECU = 0;
				status = OBDonUDS_GetValue(channel, POBDONUDS_PARAMETER_AVAILABLE_ECUS, &numberOfECU, sizeof(numberOfECU));
				printf("-> Number of OBDonUDS ECU detected (%s): %d\n", STATUS_OK_KO(status), numberOfECU);
				obd_msgprotocol canIdLen = (obd_msgprotocol)0;
				status = OBDonUDS_GetValue(channel, POBDONUDS_PARAMETER_CAN_ID_LENGTH, &canIdLen, sizeof(canIdLen));
				printf("-> Can Id length (%s): %s\n", STATUS_OK_KO(status), (canIdLen == OBD_MSGPROTOCOL_11BIT) ? "11" : (canIdLen == OBD_MSGPROTOCOL_29BIT) ? "29" : "unknown");
				// Request Vehicule Identification Number with OBDonUDS API
				printOBDonUDSvehicleIdentificationNumber(channel, canIdLen);
			}
			else
			{
				result = 3;
				if (OBDonUDS_StatusIsOk(status, POBDONUDS_STATUS_UNSUPPORTED_ECUS, false))
				{
					printf("-> Did not find ONDonUDS ECUs\n");
				}
				else
				{
					printf("-> Unattended status 0x%08X, check channel\n", status);
				}
			}
		}
		else
		{
			printf("-> Unattended status 0x%08X, check channel\n", status);
			result = 4;
		}
	}
	return result;
}