///////////////////////////////////////////////////////////////////////////////
//  PEAK_RP1210.h
//
//    RP1210C Definitions for PCAN Hardware, under Windows Vista, 7, 8, and 10. 
//
// 
//    Principle:
//    ~~~~~~~~~~
//    The PEAKRP32 implements the Recommended Practice described in the 
//	  document "RP1210C(T)_bal.pdf" from TMC, for ECU reprogramming and communication.
//	  It works as a Bridge between software which implement the RP1210C specification
//	  and the PCAN-Hardware from PEAK.
//
//    Features:
//    ~~~~~~~~~
//	   - Supported Protocols: CAN, and J1939
//     - Complete support of the Recommended Practice, Version C. 
//     - Support multiple Client connections from different processes.
//
//  ------------------------------------------------------------------
//     Author: K.Wagner
//	   Last change: 28.01.2020 Wagner
//
//     Language: ANSI C
//  ------------------------------------------------------------------
//
//     ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
//     Copyright (C) 1995-2020 by PEAK-System Technik GmbH, Darmstadt/Germany
//     More Info at http://www.peak-system.com 
//     ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
///////////////////////////////////////////////////////////////////////////////
#ifndef __PEAKRP1210H__
#define __PEAKRP1210H__

#ifndef TRUE
#define TRUE 0x01
#define FALSE 0x00
#endif

////////////////////////////////////////////////////////////////////////////
// RP1210C Constants
///////////////////////////////////////////////////////////////////////////

// Maximum values
//
#define MAX_ERR_MSG_LENGTH							80   // Default maximum count of characters for a char buffer wihtin the RP1210_GetErrorMsg function.
#define MAX_J1939_DATA_LENGTH						1785 // Maximum length of data bytes within a J1939 Message (7 * 255)
#define MAX_J1939_MESSAGE_LENGTH					1796 // Maximum size a J1939 message can be
#define MAX_ISO15765_MESSAGE_LENGTH					4108 // Maximum size an ISO15765 message can be (+1)

// Initialization File information
//
#define MODULE_NAME									"PEAKRP32"
#define INI_NAME									"PEAKRP32.ini"
#define DLL_NAME									"PEAKRP32.dll"

// On-Vehicle protocols 
//
#define CAN_PRO_ID									10
#define J1939_PRO_ID								20
#define ISO15765_PRO_ID								30
#define CAN_STRING									"CAN"
#define J1939_STRING								"J1939"
#define ISO15765_STRING								"ISO15765"

// Configuration parameters
//
#define NULL_WINDOW									0 // Windows 3.1 is no longer supported.

#define BLOCKING_IO									1 // For Blocking calls to send/read.
#define NON_BLOCKING_IO								0 // For Non-Blocking calls to send/read.
#define BLOCK_INFINITE								0 // For Non-Blocking calls to send/read.

#define ECHO_OFF									0 // Echo Mode
#define ECHO_ON										1 // Echo Mode

#define RECEIVE_ON									1 // Set Message Receive
#define RECEIVE_OFF									0 // Set Message Receive

#define FILTER_INCLUSIVE							0x00 // Filter Mode 
#define FILTER_EXCLUSIVE							0x01 // Filter Mode

#define STANDARD_CAN								0 // Setting of CAN filters
#define EXTENDED_CAN								1 // Setting of CAN filters
#define STANDARD_CAN_ISO15765_EXTENDED				2 // 11-bit with ISO15765 extended address
#define EXTENDED_CAN_ISO15765_EXTENDED				3 // 29-bit with ISO15765 extended address
#define STANDARD_MIXED_CAN_ISO15765					4 // 11-bit identifier with mixed addressing

#define FILTER_PGN									0x01 // Setting of J1939 filters
#define FILTER_PRIORITY								0x02 // Setting of J1939 filters
#define FILTER_SOURCE								0x04 // Setting of J1939 filters
#define FILTER_DESTINATION							0x08 // Setting of J1939 filters

#define ISO15765_ACTUAL_MESSAGE                     0 // ISO-15765 large message
#define ISO15765_CONFIRM                            1 // ISO-15765 message transmission result
#define ISO15765_FF_INDICATION                      2 // ISO-15765 first frame reception indication
#define ISO15765_RX_ERROR_INDICATION                3 // ISO-15765 message reception error indication

#define BUSOFF_RESET_ON                             1 // Set Hardware Auto-Reset
#define BUSOFF_RESET_OFF							0 // Set Hardware Auto-Reset

#define CHANGE_BAUD_NOW								0 // Change Baud
#define MSG_FIRST_CHANGE_BAUD						1 // Change Baud
#define RP1210_BAUD_9600							0 // Change Baud
#define RP1210_BAUD_19200							1 // Change Baud
#define RP1210_BAUD_38400							2 // Change Baud
#define RP1210_BAUD_57600							3 // Change Baud
#define RP1210_BAUD_125k							4 // Change Baud
#define RP1210_BAUD_250k							5 // Change Baud
#define RP1210_BAUD_500k							6 // Change Baud
#define RP1210_BAUD_1000k							7 // Change Baud

// Periodic senden configuration
//
#define ADD_LIST									1 // Add a message to the list.
#define VIEW_B_LIST									2 // View an entry in the list.
#define DESTROY_LIST								3 // Remove all entries in the list.
#define REMOVE_ENTRY								4 // Remove a specific entry from the list.
#define LIST_LENGTH									5 // Returns number of items in list.

// Function Return Status
//
#define BLOCK_UNTIL_DONE							0 // A function doesn't return until completes. 
#define RETURN_BEFORE_COMPLETION					2 // A function return immediately, even if it is not completed. 

///////////////////////////////////////////////////////////////////////////
// RP1210_SendCommand Defines
///////////////////////////////////////////////////////////////////////////
#define RPCMD_RESET_DEVICE							0
#define RPCMD_SET_ALL_FILTERS_STATES_TO_PASS		3
#define RPCMD_SET_MESSAGE_FILTERING_FOR_J1939		4
#define RPCMD_SET_MESSAGE_FILTERING_FOR_CAN			5
#define RPCMD_SET_MESSAGE_FILTERING_FOR_ISO15765    9
#define RPCMD_ECHO_TRANSMITTED_MESSAGES				16
#define RPCMD_SET_ALL_FILTERS_STATES_TO_DISCARD		17
#define RPCMD_SET_MESSAGE_RECEIVE					18
#define RPCMD_PROTECT_J1939_ADDRESS					19
#define RPCMD_SET_BROADCAST_FOR_CAN					21
#define RPCMD_SET_BROADCAST_FOR_J1939				22
#define RPCMD_SET_J1939_FILTER_TYPE					25
#define RPCMD_SET_CAN_FILTER_TYPE					26
#define RPCMD_SET_J1939_INTERPACKET_TIME			27
#define RPCMD_SET_MAX_ERROR_MSG_SIZE				28
#define RPCMD_DISALLOW_FURTHER_CONNECTIONS			29
#define RPCMD_SET_BLOCK_TIMEOUT						215
#define RPCMD_RELEASE_J1939_ADDRESS					31
#define RPCMD_SET_ISO15765_FILTER_TYPE              32
#define RPCMD_SET_BROADCAST_FOR_ISO15765            33
#define RPCMD_SET_ISO15765_FLOW_CONTROL             34
#define RPCMD_CLEAR_ISO15765_FLOW_CONTROL           35
#define RPCMD_FLUSH_TX_RX_BUFFERS					39
#define RPCMD_GET_PROTOCOL_CONNECTION_SPEED			45
#define RPCMD_SET_J1939_BAUD						37
#define RPCMD_SET_ISO15765_BAUD                     38
#define RPCMD_SET_CAN_BAUD							47

// Propriertary commands
//
#define RPCMD_SET_AUTORESET_ON_BUSOFF				256

///////////////////////////////////////////////////////////////////////////
// RP1210 Return Definitions
///////////////////////////////////////////////////////////////////////////
// Standard Errors
//
#define NO_ERRORS									0
#define ERR_DLL_NOT_INITIALIZED						128 
#define ERR_INVALID_CLIENT_ID						129 
#define ERR_CLIENT_ALREADY_CONNECTED				130 
#define ERR_CLIENT_AREA_FULL						131 
#define ERR_FREE_MEMORY								132 
#define ERR_NOT_ENOUGH_MEMORY						133 
#define ERR_INVALID_DEVICE							134 
#define ERR_DEVICE_IN_USE							135 
#define ERR_INVALID_PROTOCOL						136
#define ERR_TX_QUEUE_FULL							137
#define ERR_TX_QUEUE_CORRUPT						138
#define ERR_RX_QUEUE_FULL							139
#define ERR_RX_QUEUE_CORRUPT						140
#define ERR_MESSAGE_TOO_LONG						141
#define ERR_HARDWARE_NOT_RESPONDING					142 
#define ERR_COMMAND_NOT_SUPPORTED					143
#define ERR_INVALID_COMMAND							144
#define ERR_TXMESSAGE_STATUS						145
#define ERR_ADDRESS_CLAIM_FAILED					146
#define ERR_CANNOT_SET_PRIORITY						147
#define ERR_CLIENT_DISCONNECTED						148
#define ERR_CONNECT_NOT_ALLOWED						149
#define ERR_CHANGE_MODE_FAILED						150
#define ERR_BUS_OFF									151
#define ERR_COULD_NOT_TX_ADDRESS_CLAIMED			152
#define ERR_ADDRESS_LOST							153
#define ERR_CODE_NOT_FOUND							154
#define ERR_BLOCK_NOT_ALLOWED						155
#define ERR_MULTIPLE_CLIENTS_CONNECTED				156
#define ERR_ADDRESS_NEVER_CLAIMED					157
#define ERR_WINDOW_HANDLE_REQUIRED					158
#define ERR_MESSAGE_NOT_SENT						159
#define ERR_MAX_NOTIFY_EXCEEDED						160
#define ERR_MAX_FILTERS_EXCEEDED					161
#define ERR_HARDWARE_STATUS_CHANGE					162

#define ERR_INI_FILE_NOT_IN_WIN_DIR					202
#define ERR_INI_SECTION_NOT_FOUND					204
#define ERR_INI_KEY_NOT_FOUND						205
#define ERR_INVALID_KEY_STRING						206
#define ERR_DEVICE_NOT_SUPPORTED					207
#define ERR_INVALID_PORT_PARAM						208
#define ERR_COMMAND_TIMED_OUT						213

#define ERR_OS_NOT_SUPPORTED						220
#define ERR_COMMAND_QUEUE_IS_FULL					222
#define ERR_CANNOT_SET_CAN_BAUDRATE					224
#define ERR_CANNOT_CLAIM_BROADCAST_ADDRESS			225
#define ERR_OUT_OF_ADDRESS_RESOURCES				226
#define ERR_ADDRESS_RELEASE_FAILED					227
#define ERR_COMM_DEVICE_IN_USE						230

#define ERR_DATA_LINK_CONFLICT						441
#define ERR_ADAPTER_NOT_RESPONDING					453
#define ERR_CAN_BAUD_SET_NONSTANDARD				454
#define ERR_MULTIPLE_CONNECTIONS_NOT_ALLOWED_NOW	455
#define ERR_J1708_BAUD_SET_NONSTANDARD				456
#define ERR_J1939_BAUD_SET_NONSTANDARD				457
#define ERR_ISO15765_BAUD_SET_NONSTANDARD			458

#define ERR_INVALID_IOCTL_ID						600
#define ERR_NULL_PARAMETER							601
#define ERR_HARDWARE_NOT_SUPPORTED					602

// Custom Errors
//
#define ERR_INVALID_PARAMETER						193	// Function parameter values are invalid.
#define ERR_INVALID_LICENCE							194	// There are problems with the licence.

///////////////////////////////////////////////////////////////////////////
// RP1210A J1939 Protocol Defines
///////////////////////////////////////////////////////////////////////////
#define J1939_NULL_ADDR								0xFE //NULL address
#define J1939_GLOBAL_ADDR							0xFF // Global Address
#define J1939_NULL_BYTE								0xFF // NULL Byte
#define J1939_PACKS_UNLIMITED						0xFF // RTS and CTS unlimited value
#define J1939_CM_RTS								0x10 // Control-Byte "Request to Send"
#define J1939_CM_CTS								0x11 // Control-Byte "Clear to Send"
#define J1939_CM_EOM_ACK							0x13 // Control-Byte "End of Message Acknowledge"
#define J1939_CM_CA									0xFF // Control-Byte "Connection Abort"
#define J1939_CM_BAM								0x20 // Control-Byte "Broadcast Announce Message".

#define J1939_PGN_CONN_MGMENT						0xEC00 // PGN for TP.CM
#define J1939_PGN_DATA_TRANSFER						0xEB00 // PGN for TP.DT
#define J1939_PGN_REQUEST							0xEA00 // PGN for Message "Request".
#define J1939_PGN_ADDR_CLAIM						0xEE00 // PGN for Address Protection Management.
#define J1939_PGN_PROPRIETARY_A						0xEF00 // PGN for Proprietary_A - DataPage 0

///////////////////////////////////////////////////////////////////////////
// RP1210C ISO15765 Protocol Defines
///////////////////////////////////////////////////////////////////////////
#define N_OK					                    0x0 // 
#define N_TIMEOUT_A				                    0x1
#define N_TIMEOUT_Bs			                    0x2
#define N_TIMEOUT_Cr			                    0x3
#define N_WRONG_SN				                    0x4
#define N_INVALID_FS			                    0x5
#define N_UNEXP_PDU				                    0x6
#define N_WFT_OVRN				                    0x7
#define N_BUFFER_OVFLW			                    0x8
#define N_ERROR					                    0x9

///////////////////////////////////////////////////////////////////////////
// RP1210C  Function prototypes
///////////////////////////////////////////////////////////////////////////
#ifdef __cplusplus
extern "C" {
#endif

///////////////////////////////////////////////////////////////////////////
// RP1210_ClientConnect()
//   Registers and connects a Client to a PCAN-Hardware.
//   The Client does not receive any messages until a filter is set, 
//	 using the RP1210_SendCommand function.
//
// REMARKS:
//	 - The hWnd parameter can be zero. 
//
// Return Value:
//	 - When success, the identifier of the client is returned (between 0 
//	   and 127), otherwise an error code.
//
// Possible errors:	
//   ERR_INVALID_PARAMETER, ERR_INVALID_DEVICE, ERR_CLIENT_AREA_FULL, 
//	 ERR_HARDWARE_NOT_RESPONDING, ERR_DEVICE_IN_USE, ERR_INVALID_PROTOCOL,
//	 ERR_DLL_NOT_INITIALIZED
//
short WINAPI RP1210_ClientConnect(
    HWND hwndClient,								// Window handle of the Client (information purposes)
    short nDeviceID,								// Device which will be connected
    char *fpchProtocol,								// Name of protocol to be used
    long lTxBufferSize,								// Size of the Send buffer/Queue
    long lRcvBufferSize,							// Size of the receive buffer/Queue
    short nIsAppPacketizingIncomingMsgs				// J1939 Self Packetizing flag
);


///////////////////////////////////////////////////////////////////////////
// RP1210_ClientDisconnect()
//   Disconnects a Client from a PCAN-Hardware.
//
// Return Value:
//	 - When success, NO_ERRORS is returned, otherwise an error code.
//
// Possible errors:	
//   ERR_INVALID_CLIENT_ID, ERR_FREE_MEMORY
//
short WINAPI RP1210_ClientDisconnect(
    short nClientID									// Handle/ID of the Client
);


///////////////////////////////////////////////////////////////////////////
// RP1210_SendMessage()
//   Sends a CAN message to the connected PCAN-Hardware.
//
// REMARKS:
//	 - nNotifyStatusOnTx will not used (muss be 0). 
//   - nBlockOnSend will not used. Resoltves to NON_BLOCKING_IO
//	 - Buffer must be allocated by the client application.
//
// Return Value:
//	 - When success, NO_ERRORS is returned, otherwise an error code.
//
// Possible errors:	
//   ERR_INVALID_CLIENT_ID, ERR_INVALID_PARAMETER, ERR_CLIENT_DISCONNECTED,
//   ERR_BUS_OFF, ERR_TX_QUEUE_FULL, ERR_NOT_ENOUGH_MEMORY 
//
short WINAPI RP1210_SendMessage(
    short nClientID,								// Identifier of the Client
    char *fpchClientMessage,						// Buffer contained the message to send
    short nMessageSize,								// Size in bytes of the messsage buffer
    short nNotifyStatusOnTx,						// Win 3.1 flag for message Transmit-notifying 
    short nBlockOnSend								// If the function block until the message is sent
);

///////////////////////////////////////////////////////////////////////////
// RP1210_ReadMessage()
//   Reads a CAN message from the connected PCAN-Hardware.
//
// REMARKS:
//	 - Buffer must be allocated by the client application.
//
// Return Value:
//	 - When success, the number of bytes read including the four bytes 
//	   for timestamp is returned. Zero (0) will  be returned if no message 
//	   is present. If an error occurred, then a value, corresponding to the 
//	   additive inverse of the error code, is returned (for example, -193 is 
//     returned if the error corresponds to code ERR_INVALID_PARAMETER).
//
// Possible errors:	
//   ERR_INVALID_CLIENT_ID, ERR_INVALID_PARAMETER, ERR_CLIENT_DISCONNECTED, 
//	 ERR_MESSAGE_TOO_LONG
//
short WINAPI RP1210_ReadMessage(
    short nClientID,								// Identifier of the Client
    char *fpchAPIMessage,							// Buffer to place the read message
    short nBufferSize,								// Maximum size of the message buffer
    short nBlockOnRead								// If the function block until a message is received 
);


///////////////////////////////////////////////////////////////////////////
// RP1210_SendCommand()
//   Send a command to the PCAN driver associated with the current client 
//	 connection.
//
// REMARKS:
//	 - Buffer must be allocated by the client application.
//
// Return Value:
//	 - When success, NO_ERRORS is returned, otherwise an error code.
//
// Possible errors:	
//   ERR_INVALID_CLIENT_ID, ERR_MULTIPLE_CLIENTS_CONNECTED, ERR_INVALID_COMMAND, 
//   ERR_COMMAND_NOT_SUPPORTED
//
short WINAPI RP1210_SendCommand(
    short nCommandNumber,							// The command number to be send
    short nClientID,								// Identifier of the Client
    char *fpchClientCommand,						// Buffer with data related to the command
    short nMessageSize								// Size in bytes of the command buffer
);

///////////////////////////////////////////////////////////////////////////
// RP1210_ReadVersion()
//   Reads the version Information of the PCANRP dll. 
//
// REMARKS:
//	 - Buffers must be allocated by the client application.
//	 - Buffers must support at least 2 bytes.
//
// Return Value:
//	 - None.
//
// Possible errors:	
//   None
//
void WINAPI RP1210_ReadVersion(
    char *fpchDLLMajorVersion,						// Buffer to put the DLL major Version
    char *fpchDLLMinorVersion,						// Buffer to put the DLL minor Version
    char *fpchAPIMajorVersion,						// Buffer to put the API major Version
    char *fpchAPIMinorVersion						// Buffer to put the API minor Version
);

///////////////////////////////////////////////////////////////////////////
// RP1210_GetErrorMsg()
//   Retrieves an explanation text for a given error code.
//
// REMARKS:
//	 - Buffer must be allocated by the client application.
//   - Buffer must support at least 80 bytes.
//
// Return Value:
//	 - When success, NO_ERRORS is returned, otherwise an error code.
//
// Possible errors:	
//   ERR_CODE_NOT_FOUND, ERR_INVALID_PARAMETER
//
short WINAPI RP1210_GetErrorMsg(
    short ErrorCode,								// The numerical value of the Error 
    char *fpchDescription							// Buffer to place the description of the error
);

///////////////////////////////////////////////////////////////////////////
// RP1210_GetHardwareStatus()
//   Retrieves the hardware interface status and whether the PCAN-device is 
//   connected or not.
//
// REMARKS:
//	 - Buffer must be allocated by the client application.
//   - Buffer must be 16 bytes length.
//
// Return Value:
//	 - When success, NO_ERRORS is returned, otherwise an error code.
//
// Possible errors:	
//   ERR_CODE_NOT_FOUND, ERR_INVALID_PARAMETER
//
short WINAPI RP1210_GetHardwareStatus(
    short nClientID,								// Identifier of the Client
    char *fpchClientInfo,							// Buffer to place the Status information
    short nInfoSize,								// Always set to 16 bytes.
    short nBlockOnRequest							// Flag to block until the status change
);

///////////////////////////////////////////////////////////////////////////
// RP1210_GetLastErrorMsg()
//		Gets more information about an Error Code. 
//
// REMARKS:
//	 - Buffer must be allocated by the client application.
//   - Buffer must support at least 80 bytes.
//
// Return Value:
//	 - When success, NO_ERRORS is returned, otherwise an error code. 
//
// Possible errors:
//   ERR_CODE_NOT_FOUND, ERR_INVALID_PARAMETER

short WINAPI RP1210_GetLastErrorMsg(
	short ErrorCode,
	int *SubErrorCode,
	char *fpchDescription,
	short nClientID
);

///////////////////////////////////////////////////////////////////////////
// RP1210_ReadDetailedVersion()
//   Reads detailed information about the PEAK RP1210C Api and/or firmware 
//   of the device to which a client is connected. 
//
// REMARKS:
//	 - Buffers must be allocated by the client application.
//	 - Buffers must be 17 bytes in length.
//
// Return Value:
//	 - When success, NO_ERRORS is returned, otherwise an error code.
//
// Possible errors:	
//   ERR_HARDWARE_NOT_RESPONDING, ERR_INVALID_CLIENT_ID
//

short WINAPI RP1210_ReadDetailedVersion(
	short nClientID,
	char *fpchAPIVersionInfo,
	char *fpchDLLVersionInfo,
	char *fpchFWVersionInfo
);

///////////////////////////////////////////////////////////////////////////
// RP1210_Ioctl()
//   Reads and/or writes hardware and software configuration and 
//   status parameters
//
// REMARKS:
//	 - NOT SUPPORTED
//
// Return Value:
//	 - ERR_INVALID_COMMAND
//
// Possible errors:	
//   ERR_INVALID_COMMAND
//
short WINAPI RP1210_Ioctl(
	short nClientID,
	long nIoctlID,
	void *pInput,
	void *pOutput
);

#ifdef __cplusplus
}
#endif
//
#endif