// PUDS_2013.pas
//
// ~~~~~~~~~~~~
//
// PCAN-UDS API 2013 (ISO 14229-1:2013)
//
// ~~~~~~~~~~~~
//
// ------------------------------------------------------------------
// Author : Fabrice Vergnaud
// Last changed by:     $Author: Fabrice $
// Last changed date:   $Date: 2025-01-24 11:02:27 +0100 (Fri, 24 Jan 2025) $
//
// Language: Delphi OO
// ------------------------------------------------------------------
//
// Copyright (C) 2020  PEAK-System Technik GmbH, Darmstadt
// more Info at http://www.peak-system.com
//

// To use PCAN-UDS_2006 and PCAN-UDS_2013 together: do define PUDS_API_COMPATIBILITY_ISO_2006"
// {$DEFINE PUDS_API_COMPATIBILITY_ISO_2006}

unit PUDS_2013;

interface

uses
  PCANBasic,
  PCANTP_2016;

/// //////////////////////////////////////////////////////
// Enums definition for UDS API
/// //////////////////////////////////////////////////////

const
  // Defines constants used by the next enum: uds_status
  PCANTP_STATUS_OFFSET_BUS = 8;
  PCANTP_STATUS_OFFSET_NET = (PCANTP_STATUS_OFFSET_BUS + 5);
  PCANTP_STATUS_OFFSET_INFO = (PCANTP_STATUS_OFFSET_NET + 5);
  PCANTP_STATUS_OFFSET_UDS = (PCANTP_STATUS_OFFSET_INFO + 6);

type

  /// <summary>
  /// Represents PUDS error codes (used in uds_status)
  /// </summary>
{$Z4}
  uds_errstatus = (PUDS_ERRSTATUS_SERVICE_NO_MESSAGE = 1,
    PUDS_ERRSTATUS_SERVICE_TIMEOUT_CONFIRMATION,
    PUDS_ERRSTATUS_SERVICE_TIMEOUT_RESPONSE, PUDS_ERRSTATUS_RESET,
    PUDS_ERRSTATUS_ERROR_WAIT_FOR_P3_TIMING,
    PUDS_ERRSTATUS_SERVICE_ALREADY_PENDING, PUDS_ERRSTATUS_SERVICE_TX_ERROR,
    PUDS_ERRSTATUS_SERVICE_RX_ERROR, PUDS_ERRSTATUS_SERVICE_RX_OVERFLOW,
    PUDS_ERRSTATUS_MESSAGE_BUFFER_ALREADY_USED);

  /// <summary>
  /// Represents the PCANTP & UDS error and status codes.
  ///
  /// Bits information:
  /// 32|  28|  24|  20|  16|  12|   8|   4|   0|
  /// |    |    |    |    |    |    |    |    |
  /// 0000 0000 0000 0000 0000 0000 0000 0000
  /// |    |    |    |    |         [0000 0000] => PCAN-ISO-TP API errors
  /// |    |    |    |    |  [0 0000]           => CAN Bus status
  /// |    |    |    | [00 000]                 => Networking message status
  /// |    |    [0000 00]                       => PCAN-ISO-TP API extra information
  /// |  [0 0000]                               => API Status
  /// | [0]                                     => UDS Status
  /// |[0]                                      => Reserved
  /// [0]                                         => PCANBasic error flag
  /// </summary>
{$Z4}
  uds_status = (
    /// <summary>
    /// No error
    /// </summary>
    PUDS_STATUS_OK = UInt32(PCANTP_STATUS_OK),
    /// <summary>
    /// Not Initialized.
    /// </summary>
    PUDS_STATUS_NOT_INITIALIZED = UInt32(PCANTP_STATUS_NOT_INITIALIZED),
    /// <summary>
    /// Already Initialized.
    /// </summary>
    PUDS_STATUS_ALREADY_INITIALIZED = UInt32(PCANTP_STATUS_ALREADY_INITIALIZED),
    /// <summary>
    /// Could not obtain memory.
    /// </summary>
    PUDS_STATUS_NO_MEMORY = UInt32(PCANTP_STATUS_NO_MEMORY),
    /// <summary>
    /// Input buffer overflow.
    /// </summary>
    PUDS_STATUS_OVERFLOW = UInt32(PCANTP_STATUS_OVERFLOW),
    /// <summary>
    /// No message available.
    /// </summary>
    PUDS_STATUS_NO_MESSAGE = UInt32(PCANTP_STATUS_NO_MESSAGE),
    /// <summary>
    /// Wrong message parameters.
    /// </summary>
    PUDS_STATUS_PARAM_INVALID_TYPE = UInt32(PCANTP_STATUS_PARAM_INVALID_TYPE),
    /// <summary>
    /// Wrong message parameters.
    /// </summary>
    PUDS_STATUS_PARAM_INVALID_VALUE = UInt32(PCANTP_STATUS_PARAM_INVALID_VALUE),
    /// <summary>
    /// Mapping not initialized.
    /// </summary>
    PUDS_STATUS_MAPPING_NOT_INITIALIZED = UInt32
    (PCANTP_STATUS_MAPPING_NOT_INITIALIZED),
    /// <summary>
    /// Mapping parameters are invalid.
    /// </summary>
    PUDS_STATUS_MAPPING_INVALID = UInt32(PCANTP_STATUS_MAPPING_INVALID),
    /// <summary>
    /// Mapping already defined.
    /// </summary>
    PUDS_STATUS_MAPPING_ALREADY_INITIALIZED = UInt32
    (PCANTP_STATUS_MAPPING_ALREADY_INITIALIZED),
    /// <summary>
    /// Buffer is too small.
    /// </summary>
    PUDS_STATUS_PARAM_BUFFER_TOO_SMALL = UInt32
    (PCANTP_STATUS_PARAM_BUFFER_TOO_SMALL),
    /// <summary>
    /// Tx queue is full.
    /// </summary>
    PUDS_STATUS_QUEUE_TX_FULL = UInt32(PCANTP_STATUS_QUEUE_TX_FULL),
    /// <summary>
    /// Failed to get an access to the internal lock.
    /// </summary>
    PUDS_STATUS_LOCK_TIMEOUT = UInt32(PCANTP_STATUS_LOCK_TIMEOUT),
    /// <summary>
    /// Invalid cantp_handle.
    /// </summary>
    PUDS_STATUS_HANDLE_INVALID = UInt32(PCANTP_STATUS_HANDLE_INVALID),
    /// <summary>
    /// Unknown/generic error.
    /// </summary>
    PUDS_STATUS_UNKNOWN = UInt32(PCANTP_STATUS_UNKNOWN),

    // Bus status flags (bits [8..11])

    /// <summary>
    /// Channel is in BUS - LIGHT error state.
    /// </summary>
    PUDS_STATUS_FLAG_BUS_LIGHT = UInt32(PCANTP_STATUS_FLAG_BUS_LIGHT),
    /// <summary>
    /// Channel is in BUS - HEAVY error state.
    /// </summary>
    PUDS_STATUS_FLAG_BUS_HEAVY = UInt32(PCANTP_STATUS_FLAG_BUS_HEAVY),
    /// <summary>
    /// Channel is in BUS - HEAVY error state.
    /// </summary>
    PUDS_STATUS_FLAG_BUS_WARNING = UInt32(PCANTP_STATUS_FLAG_BUS_WARNING),
    /// <summary>
    /// Channel is error passive state.
    /// </summary>
    PUDS_STATUS_FLAG_BUS_PASSIVE = UInt32(PCANTP_STATUS_FLAG_BUS_PASSIVE),
    /// <summary>
    /// Channel is in BUS - OFF error state.
    /// </summary>
    PUDS_STATUS_FLAG_BUS_OFF = UInt32(PCANTP_STATUS_FLAG_BUS_OFF),
    /// <summary>
    /// Mask for all bus errors.
    /// </summary>
    PUDS_STATUS_FLAG_BUS_ANY = UInt32(PCANTP_STATUS_FLAG_BUS_ANY),
    /// <summary>
    /// This flag states if one of the following network errors occurred with the fetched message.
    /// </summary>
    PUDS_STATUS_FLAG_NETWORK_RESULT = UInt32(PCANTP_STATUS_FLAG_NETWORK_RESULT),

    // Network status (bits [13..17])

    /// <summary>
    /// Timeout occurred between 2 frames transmission (sender and receiver side).
    /// </summary>
    PUDS_STATUS_NETWORK_TIMEOUT_A = UInt32(PCANTP_STATUS_NETWORK_TIMEOUT_A),
    /// <summary>
    /// Sender side timeout while waiting for flow control frame.
    /// </summary>
    PUDS_STATUS_NETWORK_TIMEOUT_Bs = UInt32(PCANTP_STATUS_NETWORK_TIMEOUT_Bs),
    /// <summary>
    /// Receiver side timeout while waiting for consecutive frame.
    /// </summary>
    PUDS_STATUS_NETWORK_TIMEOUT_Cr = UInt32(PCANTP_STATUS_NETWORK_TIMEOUT_Cr),
    /// <summary>
    /// Unexpected sequence number.
    /// </summary>
    PUDS_STATUS_NETWORK_WRONG_SN = UInt32(PCANTP_STATUS_NETWORK_WRONG_SN),
    /// <summary>
    /// Invalid or unknown FlowStatus.
    /// </summary>
    PUDS_STATUS_NETWORK_INVALID_FS = UInt32(PCANTP_STATUS_NETWORK_INVALID_FS),
    /// <summary>
    /// Unexpected protocol data unit.
    /// </summary>
    PUDS_STATUS_NETWORK_UNEXP_PDU = UInt32(PCANTP_STATUS_NETWORK_UNEXP_PDU),
    /// <summary>
    /// Reception of flow control WAIT frame that exceeds the maximum counter defined by PUDS_PARAMETER_WFT_MAX.
    /// </summary>
    PUDS_STATUS_NETWORK_WFT_OVRN = UInt32(PCANTP_STATUS_NETWORK_WFT_OVRN),
    /// <summary>
    /// Buffer on the receiver side cannot store the data length (server side only).
    /// </summary>
    PUDS_STATUS_NETWORK_BUFFER_OVFLW = UInt32
    (PCANTP_STATUS_NETWORK_BUFFER_OVFLW),
    /// <summary>
    /// General error.
    /// </summary>
    PUDS_STATUS_NETWORK_ERROR = UInt32(PCANTP_STATUS_NETWORK_ERROR),
    /// <summary>
    /// Message was invalid and ignored.
    /// </summary>
    PUDS_STATUS_NETWORK_IGNORED = UInt32(PCANTP_STATUS_NETWORK_IGNORED),
    /// <summary>
    /// Receiver side timeout while transmitting.
    /// </summary>
    PUDS_STATUS_NETWORK_TIMEOUT_Ar = UInt32(PCANTP_STATUS_NETWORK_TIMEOUT_Ar),
    /// <summary>
    /// Sender side timeout while transmitting.
    /// </summary>
    PUDS_STATUS_NETWORK_TIMEOUT_As = UInt32(PCANTP_STATUS_NETWORK_TIMEOUT_As),

    // Extra information flags

    /// <summary>
    /// Input was modified by the API.
    /// </summary>
    PUDS_STATUS_CAUTION_INPUT_MODIFIED = UInt32
    (PCANTP_STATUS_CAUTION_INPUT_MODIFIED),
    /// <summary>
    /// DLC value of the input was modified by the API.
    /// </summary>
    PUDS_STATUS_CAUTION_DLC_MODIFIED = UInt32
    (PCANTP_STATUS_CAUTION_DLC_MODIFIED),
    /// <summary>
    /// Data Length value of the input was modified by the API.
    /// </summary>
    PUDS_STATUS_CAUTION_DATA_LENGTH_MODIFIED = UInt32
    (PCANTP_STATUS_CAUTION_DATA_LENGTH_MODIFIED),
    /// <summary>
    /// FD flags of the input was modified by the API.
    /// </summary>
    PUDS_STATUS_CAUTION_FD_FLAG_MODIFIED = UInt32
    (PCANTP_STATUS_CAUTION_FD_FLAG_MODIFIED),
    /// <summary>
    /// Receive queue is full.
    /// </summary>
    PUDS_STATUS_CAUTION_RX_QUEUE_FULL = UInt32
    (PCANTP_STATUS_CAUTION_RX_QUEUE_FULL),
    /// <summary>
    /// Buffer is used by another thread or API.
    /// </summary>
    PUDS_STATUS_CAUTION_BUFFER_IN_USE = UInt32
    (PCANTP_STATUS_CAUTION_BUFFER_IN_USE),

    // Lower API status code: see also PCANTP_STATUS_xx macros

    /// <summary>
    /// PCAN error flag, remove flag to get a usable PCAN error/status code (cf. PCANBasic API).
    /// </summary>
    PUDS_STATUS_FLAG_PCAN_STATUS = UInt32(PCANTP_STATUS_FLAG_PCAN_STATUS),

    // Masks to merge/retrieve different status by type in a uds_status

    /// <summary>
    /// Filter general error.
    /// </summary>
    PUDS_STATUS_MASK_ERROR = UInt32(PCANTP_STATUS_MASK_ERROR),
    /// <summary>
    /// Filter bus error.
    /// </summary>
    PUDS_STATUS_MASK_BUS = UInt32(PCANTP_STATUS_MASK_BUS),
    /// <summary>
    /// Filter network error.
    /// </summary>
    PUDS_STATUS_MASK_ISOTP_NET = UInt32(PCANTP_STATUS_MASK_ISOTP_NET),
    /// <summary>
    /// Filter extra information.
    /// </summary>
    PUDS_STATUS_MASK_INFO = UInt32(PCANTP_STATUS_MASK_INFO),
    /// <summary>
    /// Filter PCAN error (encapsulated PCAN-Basic status).
    /// </summary>
    PUDS_STATUS_MASK_PCAN = UInt32(PCANTP_STATUS_MASK_PCAN),

    // UDS service status.

    /// <summary>
    /// UDS error flag.
    /// </summary>
    PUDS_STATUS_FLAG_UDS_ERROR = $20 Shl PCANTP_STATUS_OFFSET_UDS,
    /// <summary>
    /// Filter UDS error.
    /// </summary>
    PUDS_STATUS_MASK_UDS_ERROR = $3F shl PCANTP_STATUS_OFFSET_UDS,
    /// <summary>
    /// UDS No message avaiable.
    /// </summary>
    PUDS_STATUS_SERVICE_NO_MESSAGE = PUDS_STATUS_FLAG_UDS_ERROR Or
    (UInt32(PUDS_ERRSTATUS_SERVICE_NO_MESSAGE) shl PCANTP_STATUS_OFFSET_UDS),
    /// <summary>
    /// Timeout while waiting message confirmation (loopback).
    /// </summary>
    PUDS_STATUS_SERVICE_TIMEOUT_CONFIRMATION = PUDS_STATUS_FLAG_UDS_ERROR Or
    (UInt32(PUDS_ERRSTATUS_SERVICE_TIMEOUT_CONFIRMATION)
    shl PCANTP_STATUS_OFFSET_UDS),
    /// <summary>
    /// Timeout while waiting request message response.
    /// </summary>
    PUDS_STATUS_SERVICE_TIMEOUT_RESPONSE = PUDS_STATUS_FLAG_UDS_ERROR Or
    (UInt32(PUDS_ERRSTATUS_SERVICE_TIMEOUT_RESPONSE)
    shl PCANTP_STATUS_OFFSET_UDS),
    /// <summary>
    /// UDS reset error.
    /// </summary>
    PUDS_STATUS_RESET = PUDS_STATUS_FLAG_UDS_ERROR Or
    (UInt32(PUDS_ERRSTATUS_RESET) shl PCANTP_STATUS_OFFSET_UDS),
    /// <summary>
    /// UDS wait for P3 timing error.
    /// </summary>
    PUDS_STATUS_ERROR_WAIT_FOR_P3_TIMING = PUDS_STATUS_FLAG_UDS_ERROR Or
    (UInt32(PUDS_ERRSTATUS_ERROR_WAIT_FOR_P3_TIMING)
    shl PCANTP_STATUS_OFFSET_UDS),
    /// <summary>
    /// A message with the same service identifier is already pending in the reception queue,
    /// user must read response for his previous request before or clear the reception queues with Reset_2013.
    /// </summary>
    PUDS_STATUS_SERVICE_ALREADY_PENDING = PUDS_STATUS_FLAG_UDS_ERROR Or
    (UInt32(PUDS_ERRSTATUS_SERVICE_ALREADY_PENDING)
    shl PCANTP_STATUS_OFFSET_UDS),
    /// <summary>
    /// An error occurred during the transmission of the UDS request message.
    /// </summary>
    PUDS_STATUS_SERVICE_TX_ERROR = PUDS_STATUS_FLAG_UDS_ERROR Or
    (UInt32(PUDS_ERRSTATUS_SERVICE_TX_ERROR) shl PCANTP_STATUS_OFFSET_UDS),
    /// <summary>
    /// An error occurred during the reception of the UDS response message.
    /// </summary>
    PUDS_STATUS_SERVICE_RX_ERROR = PUDS_STATUS_FLAG_UDS_ERROR Or
    (UInt32(PUDS_ERRSTATUS_SERVICE_RX_ERROR) shl PCANTP_STATUS_OFFSET_UDS),
    /// <summary>
    /// Service received more messages than input buffer expected.
    /// </summary>
    PUDS_STATUS_SERVICE_RX_OVERFLOW = PUDS_STATUS_FLAG_UDS_ERROR Or
    (UInt32(PUDS_ERRSTATUS_SERVICE_RX_OVERFLOW) shl PCANTP_STATUS_OFFSET_UDS),
    /// <summary>
    /// Given message buffer was already used, user must release buffer with MsgFree_2013 before reusing it.
    /// </summary>
    PUDS_STATUS_MESSAGE_BUFFER_ALREADY_USED = PUDS_STATUS_FLAG_UDS_ERROR Or
    (UInt32(PUDS_ERRSTATUS_MESSAGE_BUFFER_ALREADY_USED)
    shl PCANTP_STATUS_OFFSET_UDS));

  /// <summary>
  /// List of parameters handled by PCAN-UDS
  /// Note: PCAN-ISO-TP and PCAN-Basic parameters (PCANTP_PARAMETER_xxx, PCAN_PARAM_xxx) are compatible via casting.
  /// </summary>
{$Z4}
  uds_parameter = (
    /// <summary>
    /// (R/ ) uint8_t[]: PCAN-UDS API version parameter
    /// </summary>
    PUDS_PARAMETER_API_VERSION = $201,
    /// <summary>
    /// (R/W) uint8_t: data describing the debug mode  (use PUDS_DEBUG_LVL_ values)
    /// </summary>
    PUDS_PARAMETER_DEBUG = $203,
    /// <summary>
    /// (R/W) uintptr_t: data is pointer to a HANDLE created by CreateEvent function
    /// </summary>
    PUDS_PARAMETER_RECEIVE_EVENT = $204,
    /// <summary>
    /// (R/W) uint16_t: ISO-TP physical address
    /// </summary>
    PUDS_PARAMETER_SERVER_ADDRESS = $207,
    /// <summary>
    /// (R/W) uds_sessioninfo: ECU Session information
    /// </summary>
    PUDS_PARAMETER_SESSION_INFO = $209,
    /// <summary>
    /// (R/W) uint32_t: max time in milliseconds to wait to receive the request loopback
    /// </summary>
    PUDS_PARAMETER_TIMEOUT_REQUEST = $20A,
    /// <summary>
    /// (R/W) uint32_t: max time in milliseconds to wait to receive the message response indication
    /// </summary>
    PUDS_PARAMETER_TIMEOUT_RESPONSE = $20B,
    /// <summary>
    /// (R/W) bool: Automatic tester present. Default value: true
    /// </summary>
    PUDS_PARAMETER_AUTOMATIC_TESTER_PRESENT = $20C,
    /// <summary>
    /// (R/W) bool: Use no response flag for automatic tester present. Default value: true
    /// </summary>
    PUDS_PARAMETER_USE_NO_RESPONSE_AUTOMATIC_TESTER_PRESENT = $213,
    /// <summary>
    /// (R/W) bool: Wait for P3 timing. Default value: true (ISO-14229-2_2013 §8.3 Minimum time between client request messages, p.36)
    /// </summary>
    PUDS_PARAMETER_AUTO_P3_TIMING_MANAGEMENT = $20D,
    /// <summary>
    /// (R/ ) uint16_t[size]: List of pysical addresses to listen to.
    /// NOTE: for the parameter PUDS_PARAMETER_LISTENED_ADDRESSES the size of the array must
    /// be specified in the "buffer_size" parameter of the "UDS_GetValue_2013" function
    /// </summary>
    PUDS_PARAMETER_LISTENED_ADDRESSES = $210,
    /// <summary>
    /// ( /W) uint16_t: Add a listening address to the list of physical addresses to listen to
    /// </summary>
    PUDS_PARAMETER_ADD_LISTENED_ADDRESS = $211,
    /// <summary>
    /// ( /W) uint16_t: Remove a listening address from the list of physical addresses to listen to
    /// </summary>
    PUDS_PARAMETER_REMOVE_LISTENED_ADDRESS = $212,

    /// <summary>
    /// (R/ ) uint8_t: data describing the condition of a channel.
    /// </summary>
    PUDS_PARAMETER_CHANNEL_CONDITION = UInt32
    (PCANTP_PARAMETER_CHANNEL_CONDITION),
    /// <summary>
    /// (R/W) uint8_t: data stating the default DLC to use when transmitting messages with CAN FD
    /// </summary>
    PUDS_PARAMETER_CAN_TX_DL = UInt32(PCANTP_PARAMETER_CAN_TX_DL),
    /// <summary>
    /// (R/W) uint8_t: data stating if CAN frame DLC uses padding or not
    /// </summary>
    PUDS_PARAMETER_CAN_DATA_PADDING = UInt32(PCANTP_PARAMETER_CAN_DATA_PADDING),
    /// <summary>
    /// (R/W) uint8_t: data stating the value used for CAN data padding
    /// </summary>
    PUDS_PARAMETER_CAN_PADDING_VALUE = UInt32
    (PCANTP_PARAMETER_CAN_PADDING_VALUE),
    /// <summary>
    /// (R/W) uint8_t: data stating the default priority value for normal fixed, mixed and enhanced addressing (default=6)
    /// </summary>
    PUDS_PARAMETER_J1939_PRIORITY = UInt32(PCANTP_PARAMETER_J1939_PRIORITY),
    /// <summary>
    /// (R/W) uint8_t: data describing the block size parameter (BS)
    /// </summary>
    PUDS_PARAMETER_BLOCK_SIZE = UInt32(PCANTP_PARAMETER_BLOCK_SIZE),
    /// <summary>
    /// (R/W) uint8_t: data describing the seperation time parameter (STmin)
    /// </summary>
    PUDS_PARAMETER_SEPARATION_TIME = UInt32(PCANTP_PARAMETER_SEPARATION_TIME),
    /// <summary>
    /// (R/W) uint8_t[4]: data describing the Wait Frame Transmissions parameter.
    /// </summary>
    PUDS_PARAMETER_WFT_MAX = UInt32(PCANTP_PARAMETER_WFT_MAX),
    /// <summary>
    /// (R/W) uint8_t: data to set predefined ISO values for timeouts (see PCANTP_ISO_TIMEOUTS_*).
    /// </summary>
    PUDS_PARAMETER_ISO_TIMEOUTS = UInt32(PCANTP_PARAMETER_ISO_TIMEOUTS),
    /// <summary>
    /// ( /W) uint8_t: data stating to clear Rx/Tx queues and CAN controller (channel is unitialized and re-initialized but settings and mappings are kept)
    /// </summary>
    PUDS_PARAMETER_RESET_HARD = UInt32(PCANTP_PARAMETER_RESET_HARD),
    PUDS_PARAMETER_HARDWARE_NAME = UInt32(PCAN_HARDWARE_NAME),
    PUDS_PARAMETER_DEVICE_NUMBER = UInt32(PCAN_DEVICE_NUMBER),
    PUDS_PARAMETER_CONTROLLER_NUMBER = UInt32(PCAN_CONTROLLER_NUMBER),
    PUDS_PARAMETER_CHANNEL_FEATURES = UInt32(PCAN_CHANNEL_FEATURES));

  /// <summary>
  /// Represents type and flags for a usd_msg
  /// </summary>
{$Z4}
  uds_msgtype = (
    /// <summary>
    /// Unacknowledge Segmented Data Transfert (ISO-TP message)
    /// </summary>
    PUDS_MSGTYPE_USDT = 0,
    /// <summary>
    /// Unacknowledge Unsegmented Data Transfert (msg_physical will use a single CAN/CAN-FD frame without ISO-TP protocol control information)
    /// </summary>
    PUDS_MSGTYPE_UUDT = 1,
    /// <summary>
    /// ECU(s) shall not reply to the request on positive response.
    /// </summary>
    PUDS_MSGTYPE_FLAG_NO_POSITIVE_RESPONSE = 2,
    /// <summary>
    /// Message is a loopback
    /// </summary>
    PUDS_MSGTYPE_FLAG_LOOPBACK = 4,
    /// <summary>
    /// Mask to get the type (USDT or UUDT)
    /// </summary>
    PUDS_MSGTYPE_MASK_TYPE = $1);

  /// <summary>
  /// Represents ISO-TP network addressing information supported in UDS
  /// </summary>
{$Z4}
  uds_msgprotocol = (
    /// <summary>
    /// Non ISO-TP frame (Unacknowledge Unsegmented Data Transfer)
    /// </summary>
    PUDS_MSGPROTOCOL_NONE = UInt32($0),
    /// <summary>
    /// ISO-TP 11 bits Extended addressing (mapping required)
    /// </summary>
    PUDS_MSGPROTOCOL_ISO_15765_2_11B_EXTENDED = UInt32($07),
    /// <summary>
    /// ISO-TP 11 bits Normal addressing (mapping required)
    /// </summary>
    PUDS_MSGPROTOCOL_ISO_15765_2_11B_NORMAL = UInt32($01),
    /// <summary>
    /// ISO-TP 11 bits Mixed addressing (mapping required)
    /// </summary>
    PUDS_MSGPROTOCOL_ISO_15765_2_11B_REMOTE = UInt32($02),
    /// <summary>
    /// ISO-TP 29 bits Extended addressing (mapping required)
    /// </summary>
    PUDS_MSGPROTOCOL_ISO_15765_2_29B_EXTENDED = UInt32($08),
    /// <summary>
    /// ISO-TP 29 bits Fixed Normal addressing
    /// </summary>
    PUDS_MSGPROTOCOL_ISO_15765_2_29B_FIXED_NORMAL = UInt32($03),
    /// <summary>
    /// ISO-TP 29 bits Normal addressing (mapping required)
    /// </summary>
    PUDS_MSGPROTOCOL_ISO_15765_2_29B_NORMAL = UInt32($06),
    /// <summary>
    /// ISO-TP 29 bits Mixed addressing
    /// </summary>
    PUDS_MSGPROTOCOL_ISO_15765_2_29B_REMOTE = UInt32($04),
    /// <summary>
    /// ISO-TP Enhanced addressing
    /// </summary>
    PUDS_MSGPROTOCOL_ISO_15765_3_29B_ENHANCED = UInt32($05));

  /// <summary>
  /// Represents UDS negative response codes (see ISO 14229-1:2013 §A.1 Negative response codes p.325)
  /// </summary>
{$Z1}
  uds_nrc = (
    /// <summary>
    /// Positive Response
    /// </summary>
    PUDS_NRC_PR = $00,
    /// <summary>
    /// General Reject
    /// </summary>
    PUDS_NRC_GR = $10,
    /// <summary>
    /// Service Not Supported
    /// </summary>
    PUDS_NRC_SNS = $11,
    /// <summary>
    /// Sub Function Not Supported
    /// </summary>
    PUDS_NRC_SFNS = $12,
    /// <summary>
    /// Incorrect Message Length Or Invalid Format
    /// </summary>
    PUDS_NRC_IMLOIF = $13,
    /// <summary>
    /// Response Too Long
    /// </summary>
    PUDS_NRC_RTL = $14,
    /// <summary>
    /// Busy Repeat Request
    /// </summary>
    PUDS_NRC_BRR = $21,
    /// <summary>
    /// Conditions Not Correct
    /// </summary>
    PUDS_NRC_CNC = $22,
    /// <summary>
    /// Request Sequence Error
    /// </summary>
    PUDS_NRC_RSE = $24,
    /// <summary>
    /// No Response From Subnet Component
    /// </summary>
    PUDS_NRC_NRFSC = $25,
    /// <summary>
    /// Failure Prevents Execution Of Requested Action
    /// </summary>
    PUDS_NRC_FPEORA = $26,
    /// <summary>
    /// Request Out Of Range
    /// </summary>
    PUDS_NRC_ROOR = $31,
    /// <summary>
    /// Security Access Denied
    /// </summary>
    PUDS_NRC_SAD = $33,
    /// <summary>
    /// Authentication Required
    /// </summary>
    PUDS_NRC_AR = $34,
    /// <summary>
    /// Invalid Key
    /// </summary>
    PUDS_NRC_IK = $35,
    /// <summary>
    /// Exceeded Number Of Attempts
    /// </summary>
    PUDS_NRC_ENOA = $36,
    /// <summary>
    /// Required Time Delay Not Expired
    /// </summary>
    PUDS_NRC_RTDNE = $37,
    /// <summary>
    /// Secure Data Transmission Required
    /// </summary>
    PUDS_NRC_SDTR = $38,
    /// <summary>
    /// Secure Data Transmission Not Allowed
    /// </summary>
    PUDS_NRC_SDTNA = $39,
    /// <summary>
    /// Secure Data Verification Failed
    /// </summary>
    PUDS_NRC_SDTF = $3A,
    /// <summary>
    /// Certificate Verification Failed Invalid Time Period
    /// </summary>
    PUDS_NRC_CVFITP = $50,
    /// <summary>
    /// Certificate Verification Failed Invalid SIGnature
    /// </summary>
    PUDS_NRC_CVFISIG = $51,
    /// <summary>
    /// Certificate Verification Failed Invalid Chain of Trust
    /// </summary>
    PUDS_NRC_CVFICOT = $52,
    /// <summary>
    /// Certificate Verification Failed Invalid Type
    /// </summary>
    PUDS_NRC_CVFIT = $53,
    /// <summary>
    /// Certificate Verification Failed Invalid Format
    /// </summary>
    PUDS_NRC_CVFIF = $54,
    /// <summary>
    /// Certificate Verification Failed Invalid Content
    /// </summary>
    PUDS_NRC_CVFIC = $55,
    /// <summary>
    /// Certificate Verification Failed Invalid SCoPe
    /// </summary>
    PUDS_NRC_CVFISCP = $56,
    /// <summary>
    /// Certificate Verification Failed Invalid CERTificate(revoked)
    /// </summary>
    PUDS_NRC_CVFICERT = $57,
    /// <summary>
    /// Ownership Verification Failed
    /// </summary>
    PUDS_NRC_OVF = $58,
    /// <summary>
    /// Challenge Calculation Failed
    /// </summary>
    PUDS_NRC_CCF = $59,
    /// <summary>
    /// Setting Access Rights Failed
    /// </summary>
    PUDS_NRC_SARF = $5A,
    /// <summary>
    /// Session Key Creation / Derivation Failed
    /// </summary>
    PUDS_NRC_SKCDF = $5B,
    /// <summary>
    /// Configuration Data Usage Failed
    /// </summary>
    PUDS_NRC_CDUF = $5C,
    /// <summary>
    /// DeAuthentication Failed
    /// </summary>
    PUDS_NRC_DAF = $5D,
    /// <summary>
    /// Upload Download Not Accepted
    /// </summary>
    PUDS_NRC_UDNA = $70,
    /// <summary>
    /// Transfer Data Suspended
    /// </summary>
    PUDS_NRC_TDS = $71,
    /// <summary>
    /// General Programming Failure
    /// </summary>
    PUDS_NRC_GPF = $72,
    /// <summary>
    /// Wrong Block Sequence Counter
    /// </summary>
    PUDS_NRC_WBSC = $73,
    /// <summary>
    /// Request Correctly Received - Response Pending
    /// </summary>
    PUDS_NRC_RCRRP = $78,
    /// <summary>
    /// Sub Function Not Supported In Active Session
    /// </summary>
    PUDS_NRC_SFNSIAS = $7E,
    /// <summary>
    /// Service Not Supported In Active Session
    /// </summary>
    PUDS_NRC_SNSIAS = $7F,
    /// <summary>
    /// RPM Too High
    /// </summary>
    PUDS_NRC_RPMTH = $81,
    /// <summary>
    /// RPM Too Low
    /// </summary>
    PUDS_NRC_RPMTL = $82,
    /// <summary>
    /// Engine Is Running
    /// </summary>
    PUDS_NRC_EIR = $83,
    /// <summary>
    /// Engine Is Not Running
    /// </summary>
    PUDS_NRC_EINR = $84,
    /// <summary>
    /// Engine Run Time Too Low
    /// </summary>
    PUDS_NRC_ERTTL = $85,
    /// <summary>
    /// TEMPerature Too High
    /// </summary>
    PUDS_NRC_TEMPTH = $86,
    /// <summary>
    /// TEMPerature Too Low
    /// </summary>
    PUDS_NRC_TEMPTL = $87,
    /// <summary>
    /// Vehicle Speed Too High
    /// </summary>
    PUDS_NRC_VSTH = $88,
    /// <summary>
    /// Vehicle Speed Too Low
    /// </summary>
    PUDS_NRC_VSTL = $89,
    /// <summary>
    /// Throttle / Pedal Too High
    /// </summary>
    PUDS_NRC_TPTH = $8A,
    /// <summary>
    /// Throttle / Pedal Too Low
    /// </summary>
    PUDS_NRC_TPTL = $8B,
    /// <summary>
    /// Transmission Range Not In Neutral
    /// </summary>
    PUDS_NRC_TRNIN = $8C,
    /// <summary>
    /// Transmission Range Not In Gear
    /// </summary>
    PUDS_NRC_TRNIG = $8D,
    /// <summary>
    /// Brake Switch(es) Not Closed(brake pedal not pressed or not applied)
    /// </summary>
    PUDS_NRC_BSNC = $8F,
    /// <summary>
    /// Shifter Lever Not In Park
    /// </summary>
    PUDS_NRC_SLNIP = $90,
    /// <summary>
    /// Torque Converter Clutch Locked
    /// </summary>
    PUDS_NRC_TCCL = $91,
    /// <summary>
    /// Voltage Too High
    /// </summary>
    PUDS_NRC_VTH = $92,
    /// <summary>
    /// Voltage Too Low
    /// </summary>
    PUDS_NRC_VTL = $93,
    /// <summary>
    /// Resource Temporarily Not Available
    /// </summary>
    PUDS_NRC_RTNA = $94);

  /// <summary>
  /// PUDS ISO_15765_4 11 bit CAN ID definitions
  /// </summary>
{$Z4}
  uds_can_id = (
    /// <summary>
    /// CAN ID for functionally addressed request messages sent by external test equipment
    /// </summary>
    PUDS_CAN_ID_ISO_15765_4_FUNCTIONAL_REQUEST = $7DF,
    /// <summary>
    /// physical request CAN ID from external test equipment to ECU #1
    /// </summary>
    PUDS_CAN_ID_ISO_15765_4_PHYSICAL_REQUEST_1 = $7E0,
    /// <summary>
    /// physical response CAN ID from ECU #1 to external test equipment
    /// </summary>
    PUDS_CAN_ID_ISO_15765_4_PHYSICAL_RESPONSE_1 = $7E8,
    /// <summary>
    /// physical request CAN ID from external test equipment to ECU #2
    /// </summary>
    PUDS_CAN_ID_ISO_15765_4_PHYSICAL_REQUEST_2 = $7E1,
    /// <summary>
    /// physical response CAN ID from ECU #2 to external test equipment
    /// </summary>
    PUDS_CAN_ID_ISO_15765_4_PHYSICAL_RESPONSE_2 = $7E9,
    /// <summary>
    /// physical request CAN ID from external test equipment to ECU #3
    /// </summary>
    PUDS_CAN_ID_ISO_15765_4_PHYSICAL_REQUEST_3 = $7E2,
    /// <summary>
    /// physical response CAN ID from ECU #3 to external test equipment
    /// </summary>
    PUDS_CAN_ID_ISO_15765_4_PHYSICAL_RESPONSE_3 = $7EA,
    /// <summary>
    /// physical request CAN ID from external test equipment to ECU #4
    /// </summary>
    PUDS_CAN_ID_ISO_15765_4_PHYSICAL_REQUEST_4 = $7E3,
    /// <summary>
    /// physical response CAN ID from ECU #4 to external test equipment
    /// </summary>
    PUDS_CAN_ID_ISO_15765_4_PHYSICAL_RESPONSE_4 = $7EB,
    /// <summary>
    /// physical request CAN ID from external test equipment to ECU #5
    /// </summary>
    PUDS_CAN_ID_ISO_15765_4_PHYSICAL_REQUEST_5 = $7E4,
    /// <summary>
    /// physical response CAN ID from ECU #5 to external test equipment
    /// </summary>
    PUDS_CAN_ID_ISO_15765_4_PHYSICAL_RESPONSE_5 = $7EC,
    /// <summary>
    /// physical request CAN ID from external test equipment to ECU #6
    /// </summary>
    PUDS_CAN_ID_ISO_15765_4_PHYSICAL_REQUEST_6 = $7E5,
    /// <summary>
    /// physical response CAN ID from ECU #6 to external test equipment
    /// </summary>
    PUDS_CAN_ID_ISO_15765_4_PHYSICAL_RESPONSE_6 = $7ED,
    /// <summary>
    /// physical request CAN ID from external test equipment to ECU #7
    /// </summary>
    PUDS_CAN_ID_ISO_15765_4_PHYSICAL_REQUEST_7 = $7E6,
    /// <summary>
    /// physical response CAN ID from ECU #7 to external test equipment
    /// </summary>
    PUDS_CAN_ID_ISO_15765_4_PHYSICAL_RESPONSE_7 = $7EE,
    /// <summary>
    /// physical request CAN ID from external test equipment to ECU #8
    /// </summary>
    PUDS_CAN_ID_ISO_15765_4_PHYSICAL_REQUEST_8 = $7E7,
    /// <summary>
    /// physical response CAN ID from ECU #8 to external test equipment
    /// </summary>
    PUDS_CAN_ID_ISO_15765_4_PHYSICAL_RESPONSE_8 = $7EF);

  /// <summary>
  /// PUDS ISO_15765_4 address definitions
  /// </summary>
{$Z2}
  uds_address = (
    /// <summary>
    /// external test equipment
    /// </summary>
    PUDS_ADDRESS_ISO_15765_4_ADDR_TEST_EQUIPMENT = $F1,
    /// <summary>
    /// OBD funtional system
    /// </summary>
    PUDS_ADDRESS_ISO_15765_4_ADDR_OBD_FUNCTIONAL = $33,
    /// <summary>
    /// ECU 1
    /// </summary>
    PUDS_ADDRESS_ISO_15765_4_ADDR_ECU_1 = $1,
    /// <summary>
    /// ECU 2
    /// </summary>
    PUDS_ADDRESS_ISO_15765_4_ADDR_ECU_2 = $2,
    /// <summary>
    /// ECU 3
    /// </summary>
    PUDS_ADDRESS_ISO_15765_4_ADDR_ECU_3 = $3,
    /// <summary>
    /// ECU 4
    /// </summary>
    PUDS_ADDRESS_ISO_15765_4_ADDR_ECU_4 = $4,
    /// <summary>
    /// ECU 5
    /// </summary>
    PUDS_ADDRESS_ISO_15765_4_ADDR_ECU_5 = $5,
    /// <summary>
    /// ECU 6
    /// </summary>
    PUDS_ADDRESS_ISO_15765_4_ADDR_ECU_6 = $6,
    /// <summary>
    /// ECU 7
    /// </summary>
    PUDS_ADDRESS_ISO_15765_4_ADDR_ECU_7 = $7,
    /// <summary>
    /// ECU 8
    /// </summary>
    PUDS_ADDRESS_ISO_15765_4_ADDR_ECU_8 = $8);

  /// <summary>
  /// PUDS Service ids defined in ISO 14229-1:2013
  /// </summary>
{$Z1}
  uds_service = (
    /// <summary>
    /// see ISO 14229-1:2013
    /// </summary>
    PUDS_SERVICE_SI_DiagnosticSessionControl = $10,
    /// <summary>
    /// see ISO 14229-1:2013
    /// </summary>
    PUDS_SERVICE_SI_ECUReset = $11,
    /// <summary>
    /// see ISO 14229-1:2013
    /// </summary>
    PUDS_SERVICE_SI_SecurityAccess = $27,
    /// <summary>
    /// see ISO 14229-1:2013
    /// </summary>
    PUDS_SERVICE_SI_CommunicationControl = $28,
    /// <summary>
    /// see ISO 14229-1:2013
    /// </summary>
    PUDS_SERVICE_SI_TesterPresent = $3E,
    /// <summary>
    /// see ISO 14229-1:2013
    /// </summary>
    PUDS_SERVICE_SI_AccessTimingParameter = $83,
    /// <summary>
    /// see ISO 14229-1:2013
    /// </summary>
    PUDS_SERVICE_SI_SecuredDataTransmission = $84,
    /// <summary>
    /// see ISO 14229-1:2013
    /// </summary>
    PUDS_SERVICE_SI_ControlDTCSetting = $85,
    /// <summary>
    /// see ISO 14229-1:2013
    /// </summary>
    PUDS_SERVICE_SI_ResponseOnEvent = $86,
    /// <summary>
    /// see ISO 14229-1:2013
    /// </summary>
    PUDS_SERVICE_SI_LinkControl = $87,
    /// <summary>
    /// see ISO 14229-1:2013
    /// </summary>
    PUDS_SERVICE_SI_ReadDataByIdentifier = $22,
    /// <summary>
    /// see ISO 14229-1:2013
    /// </summary>
    PUDS_SERVICE_SI_ReadMemoryByAddress = $23,
    /// <summary>
    /// see ISO 14229-1:2013
    /// </summary>
    PUDS_SERVICE_SI_ReadScalingDataByIdentifier = $24,
    /// <summary>
    /// see ISO 14229-1:2013
    /// </summary>
    PUDS_SERVICE_SI_ReadDataByPeriodicIdentifier = $2A,
    /// <summary>
    /// see ISO 14229-1:2013
    /// </summary>
    PUDS_SERVICE_SI_DynamicallyDefineDataIdentifier = $2C,
    /// <summary>
    /// see ISO 14229-1:2013
    /// </summary>
    PUDS_SERVICE_SI_WriteDataByIdentifier = $2E,
    /// <summary>
    /// see ISO 14229-1:2013
    /// </summary>
    PUDS_SERVICE_SI_WriteMemoryByAddress = $3D,
    /// <summary>
    /// see ISO 14229-1:2013
    /// </summary>
    PUDS_SERVICE_SI_ClearDiagnosticInformation = $14,
    /// <summary>
    /// see ISO 14229-1:2013
    /// </summary>
    PUDS_SERVICE_SI_ReadDTCInformation = $19,
    /// <summary>
    /// see ISO 14229-1:2013
    /// </summary>
    PUDS_SERVICE_SI_InputOutputControlByIdentifier = $2F,
    /// <summary>
    /// see ISO 14229-1:2013
    /// </summary>
    PUDS_SERVICE_SI_RoutineControl = $31,
    /// <summary>
    /// see ISO 14229-1:2013
    /// </summary>
    PUDS_SERVICE_SI_RequestDownload = $34,
    /// <summary>
    /// see ISO 14229-1:2013
    /// </summary>
    PUDS_SERVICE_SI_RequestUpload = $35,
    /// <summary>
    /// see ISO 14229-1:2013
    /// </summary>
    PUDS_SERVICE_SI_TransferData = $36,
    /// <summary>
    /// see ISO 14229-1:2013
    /// </summary>
    PUDS_SERVICE_SI_RequestTransferExit = $37,
    /// <summary>
    /// see ISO 14229-1:2013
    /// </summary>
    PUDS_SERVICE_SI_RequestFileTransfer = $38,
    /// <summary>
    /// see ISO 14229-1:2020
    /// </summary>
    PUDS_SERVICE_SI_Authentication = $29,
    /// <summary>
    /// negative response
    /// </summary>
    PUDS_SERVICE_NR_SI = $7F);

  /// //////////////////////////////////////////////////////
  // UDS Services Enum declarations
  /// //////////////////////////////////////////////////////

  // ISO-14229-1:2013 §9.2.2.2 p.39
{$Z1}
  uds_svc_param_dsc = (
    /// <summary>
    /// Default Session
    /// </summary>
    PUDS_SVC_PARAM_DSC_DS = $1,
    /// <summary>
    /// ECU Programming Session
    /// </summary>
    PUDS_SVC_PARAM_DSC_ECUPS = $2,
    /// <summary>
    /// ECU Extended Diagnostic Session
    /// </summary>
    PUDS_SVC_PARAM_DSC_ECUEDS = $3,
    /// <summary>
    /// Safety System Diagnostic Session
    /// </summary>
    PUDS_SVC_PARAM_DSC_SSDS = $4);

  // ISO-14229-1:2013 §9.3.2.2 p.43
{$Z1}
  uds_svc_param_er = (
    /// <summary>
    /// Hard Reset
    /// </summary>
    PUDS_SVC_PARAM_ER_HR = $1,
    /// <summary>
    /// Key Off on Reset
    /// </summary>
    PUDS_SVC_PARAM_ER_KOFFONR = $2,
    /// <summary>
    /// Soft Reset
    /// </summary>
    PUDS_SVC_PARAM_ER_SR = $3,
    /// <summary>
    /// Enable Rapid Power Shutdown
    /// </summary>
    PUDS_SVC_PARAM_ER_ERPSD = $4,
    /// <summary>
    /// Disable Rapid Power Shutdown
    /// </summary>
    PUDS_SVC_PARAM_ER_DRPSD = $5);

  // ISO-14229-1:2013 §9.5.2.2 p.54
{$Z1}
  uds_svc_param_cc = (
    /// <summary>
    /// Enable Rx and Tx
    /// </summary>
    PUDS_SVC_PARAM_CC_ERXTX = $0,
    /// <summary>
    /// Enable Rx and Disable Tx
    /// </summary>
    PUDS_SVC_PARAM_CC_ERXDTX = $1,
    /// <summary>
    /// Disable Rx and Enable Tx
    /// </summary>
    PUDS_SVC_PARAM_CC_DRXETX = $2,
    /// <summary>
    /// Disable Rx and Tx
    /// </summary>
    PUDS_SVC_PARAM_CC_DRXTX = $3,
    /// <summary>
    /// Enable Rx And Disable Tx With Enhanced Address Information
    /// </summary>
    PUDS_SVC_PARAM_CC_ERXDTXWEAI = $4,
    /// <summary>
    /// Enable Rx And Tx With Enhanced Address Information
    /// </summary>
    PUDS_SVC_PARAM_CC_ERXTXWEAI = $5);

  // ISO-14229-1:2013 §9.6.2.2 p.59
{$Z1}
  uds_svc_param_tp = (
    /// <summary>
    /// Zero SubFunction
    /// </summary>
    PUDS_SVC_PARAM_TP_ZSUBF = $0);


  // ISO-14229-1:2013 §9.9.2.2 p.72
{$Z1}
  uds_svc_param_cdtcs = (
    /// <summary>
    /// The server(s) shall resume the setting of diagnostic trouble codes
    /// </summary>
    PUDS_SVC_PARAM_CDTCS_ON = $1,
    /// <summary>
    /// The server(s) shall stop the setting of diagnostic trouble codes
    /// </summary>
    PUDS_SVC_PARAM_CDTCS_OFF = $2);

  // ISO-14229-1:2013 §9.10.2.2.1 p.78
{$Z1}
  uds_svc_param_roe = (
    /// <summary>
    /// Stop Response On Event
    /// </summary>
    PUDS_SVC_PARAM_ROE_STPROE = $0,
    /// <summary>
    /// On DTC Status Change
    /// </summary>
    PUDS_SVC_PARAM_ROE_ONDTCS = $1,
    /// <summary>
    /// On Timer Interrupt
    /// </summary>
    PUDS_SVC_PARAM_ROE_OTI = $2,
    /// <summary>
    /// On Change Of Data Identifier
    /// </summary>
    PUDS_SVC_PARAM_ROE_OCODID = $3,
    /// <summary>
    /// Report Activated Events
    /// </summary>
    PUDS_SVC_PARAM_ROE_RAE = $4,
    /// <summary>
    /// Start Response On Event
    /// </summary>
    PUDS_SVC_PARAM_ROE_STRTROE = $5,
    /// <summary>
    /// Clear Response On Event
    /// </summary>
    PUDS_SVC_PARAM_ROE_CLRROE = $6,
    /// <summary>
    /// On Comparison Of Values
    /// </summary>
    PUDS_SVC_PARAM_ROE_OCOV = $7,
    /// <summary>
    /// Report Most Recent Dtc On Status Change (ISO 14229-1:2020 10.9.2.2 p.121)
    /// </summary>
    PUDS_SVC_PARAM_ROE_RMRDOSC = $8,
    /// <summary>
    /// Report Dtc Record Information On Dtc Status Change (ISO 14229-1:2020 10.9.2.2 p.121)
    /// </summary>
    PUDS_SVC_PARAM_ROE_RDRIODSC = $9);

{$Z1}
  uds_svc_param_roe_recommended_service_id = (
    /// <summary>
    /// Recommended service (first byte of service to respond to record)
    /// </summary>
    PUDS_SVC_PARAM_ROE_STRT_SI_RDBI = Byte
    (PUDS_SERVICE_SI_ReadDataByIdentifier),
    /// <summary>
    /// Recommended service (first byte of service to respond to record)
    /// </summary>
    PUDS_SVC_PARAM_ROE_STRT_SI_RDTCI = Byte(PUDS_SERVICE_SI_ReadDTCInformation),
    /// <summary>
    /// Recommended service (first byte of service to respond to record)
    /// </summary>
    PUDS_SVC_PARAM_ROE_STRT_SI_RC = Byte(PUDS_SERVICE_SI_RoutineControl),
    /// <summary>
    /// Recommended service (first byte of service to respond to record)
    /// </summary>
    PUDS_SVC_PARAM_ROE_STRT_SI_IOCBI = Byte
    (PUDS_SERVICE_SI_InputOutputControlByIdentifier));

  // ISO-14229-1:2013 §9.11.2.2 p.101
{$Z1}
  uds_svc_param_lc = (
    /// <summary>
    /// Verify Baudrate Transition With Fixed Baudrate
    /// </summary>
    PUDS_SVC_PARAM_LC_VBTWFBR = $1,
    /// <summary>
    /// Verify Baudrate Transition With Specific Baudrate
    /// </summary>
    PUDS_SVC_PARAM_LC_VBTWSBR = $2,
    /// <summary>
    /// Transition Baudrate
    /// </summary>
    PUDS_SVC_PARAM_LC_TB = $3);

{$Z1}
  uds_svc_param_lc_baudrate_identifier = (
    /// <summary>
    /// standard PC baud rate of 9.6 KBaud
    /// </summary>
    PUDS_SVC_PARAM_LC_BAUDRATE_PC_9600 = $1,
    /// <summary>
    /// standard PC baud rate of 19.2 KBaud
    /// </summary>
    PUDS_SVC_PARAM_LC_BAUDRATE_PC_19200 = $2,
    /// <summary>
    /// standard PC baud rate of 38.4 KBaud
    /// </summary>
    PUDS_SVC_PARAM_LC_BAUDRATE_PC_38400 = $3,
    /// <summary>
    /// standard PC baud rate of 57.6 KBaud
    /// </summary>
    PUDS_SVC_PARAM_LC_BAUDRATE_PC_57600 = $4,
    /// <summary>
    /// standard PC baud rate of 115.2 KBaud
    /// </summary>
    PUDS_SVC_PARAM_LC_BAUDRATE_PC_115200 = $5,
    /// <summary>
    /// standard CAN baud rate of 125 KBaud
    /// </summary>
    PUDS_SVC_PARAM_LC_BAUDRATE_CAN_125K = $10,
    /// <summary>
    /// standard CAN baud rate of 250 KBaud
    /// </summary>
    PUDS_SVC_PARAM_LC_BAUDRATE_CAN_250K = $11,
    /// <summary>
    /// standard CAN baud rate of 500 KBaud
    /// </summary>
    PUDS_SVC_PARAM_LC_BAUDRATE_CAN_500K = $12,
    /// <summary>
    /// standard CAN baud rate of 1 MBaud
    /// </summary>
    PUDS_SVC_PARAM_LC_BAUDRATE_CAN_1M = $13,
    /// <summary>
    /// Programming setup
    /// </summary>
    PUDS_SVC_PARAM_LC_BAUDRATE_PROGSU = $20);

  // ISO-14229-1:2013 §C.1 p337
{$Z2}
  uds_svc_param_di = (
    /// <summary>
    /// bootSoftwareIdentificationDataIdentifier
    /// </summary>
    PUDS_SVC_PARAM_DI_BSIDID = $F180,
    /// <summary>
    /// applicationSoftwareIdentificationDataIdentifier
    /// </summary>
    PUDS_SVC_PARAM_DI_ASIDID = $F181,
    /// <summary>
    /// applicationDataIdentificationDataIdentifier
    /// </summary>
    PUDS_SVC_PARAM_DI_ADIDID = $F182,
    /// <summary>
    /// bootSoftwareIdentificationDataIdentifier
    /// </summary>
    PUDS_SVC_PARAM_DI_BSFPDID = $F183,
    /// <summary>
    /// applicationSoftwareFingerprintDataIdentifier
    /// </summary>
    PUDS_SVC_PARAM_DI_ASFPDID = $F184,
    /// <summary>
    /// applicationDataFingerprintDataIdentifier
    /// </summary>
    PUDS_SVC_PARAM_DI_ADFPDID = $F185,
    /// <summary>
    /// activeDiagnosticSessionDataIdentifier
    /// </summary>
    PUDS_SVC_PARAM_DI_ADSDID = $F186,
    /// <summary>
    /// vehicleManufacturerSparePartNumberDataIdentifier
    /// </summary>
    PUDS_SVC_PARAM_DI_VMSPNDID = $F187,
    /// <summary>
    /// vehicleManufacturerECUSoftwareNumberDataIdentifier
    /// </summary>
    PUDS_SVC_PARAM_DI_VMECUSNDID = $F188,
    /// <summary>
    /// vehicleManufacturerECUSoftwareVersionNumberDataIdentifier
    /// </summary>
    PUDS_SVC_PARAM_DI_VMECUSVNDID = $F189,
    /// <summary>
    /// systemSupplierIdentifierDataIdentifier
    /// </summary>
    PUDS_SVC_PARAM_DI_SSIDDID = $F18A,
    /// <summary>
    /// ECUManufacturingDateDataIdentifier
    /// </summary>
    PUDS_SVC_PARAM_DI_ECUMDDID = $F18B,
    /// <summary>
    /// ECUSerialNumberDataIdentifier
    /// </summary>
    PUDS_SVC_PARAM_DI_ECUSNDID = $F18C,
    /// <summary>
    /// supportedFunctionalUnitsDataIdentifier
    /// </summary>
    PUDS_SVC_PARAM_DI_SFUDID = $F18D,
    /// <summary>
    /// vehicleManufacturerKitAssemblyPartNumberDataIdentifier
    /// </summary>
    PUDS_SVC_PARAM_DI_VMKAPNDID = $F18E,
    /// <summary>
    /// VINDataIdentifier
    /// </summary>
    PUDS_SVC_PARAM_DI_VINDID = $F190,
    /// <summary>
    /// vehicleManufacturerECUHardwareNumberDataIdentifier
    /// </summary>
    PUDS_SVC_PARAM_DI_VMECUHNDID = $F191,
    /// <summary>
    /// systemSupplierECUHardwareNumberDataIdentifier
    /// </summary>
    PUDS_SVC_PARAM_DI_SSECUHWNDID = $F192,
    /// <summary>
    /// systemSupplierECUHardwareVersionNumberDataIdentifier
    /// </summary>
    PUDS_SVC_PARAM_DI_SSECUHWVNDID = $F193,
    /// <summary>
    /// systemSupplierECUSoftwareNumberDataIdentifier
    /// </summary>
    PUDS_SVC_PARAM_DI_SSECUSWNDID = $F194,
    /// <summary>
    /// systemSupplierECUSoftwareVersionNumberDataIdentifier
    /// </summary>
    PUDS_SVC_PARAM_DI_SSECUSWVNDID = $F195,
    /// <summary>
    /// exhaustRegulationOrTypeApprovalNumberDataIdentifier
    /// </summary>
    PUDS_SVC_PARAM_DI_EROTANDID = $F196,
    /// <summary>
    /// systemNameOrEngineTypeDataIdentifier
    /// </summary>
    PUDS_SVC_PARAM_DI_SNOETDID = $F197,
    /// <summary>
    /// repairShopCodeOrTesterSerialNumberDataIdentifier
    /// </summary>
    PUDS_SVC_PARAM_DI_RSCOTSNDID = $F198,
    /// <summary>
    /// programmingDateDataIdentifier
    /// </summary>
    PUDS_SVC_PARAM_DI_PDDID = $F199,
    /// <summary>
    /// calibrationRepairShopCodeOrCalibrationEquipmentSerialNumberDataIdentifier
    /// </summary>
    PUDS_SVC_PARAM_DI_CRSCOCESNDID = $F19A,
    /// <summary>
    /// calibrationDateDataIdentifier
    /// </summary>
    PUDS_SVC_PARAM_DI_CDDID = $F19B,
    /// <summary>
    /// calibrationEquipmentSoftwareNumberDataIdentifier
    /// </summary>
    PUDS_SVC_PARAM_DI_CESWNDID = $F19C,
    /// <summary>
    /// ECUInstallationDateDataIdentifier
    /// </summary>
    PUDS_SVC_PARAM_DI_EIDDID = $F19D,
    /// <summary>
    /// ODXFileDataIdentifier
    /// </summary>
    PUDS_SVC_PARAM_DI_ODXFDID = $F19E,
    /// <summary>
    /// entityDataIdentifier
    /// </summary>
    PUDS_SVC_PARAM_DI_EDID = $F19F);

  // ISO-14229-1:2013 §C.4 p.351
{$Z1}
  uds_svc_param_rdbpi = (
    /// <summary>
    /// Send At Slow Rate
    /// </summary>
    PUDS_SVC_PARAM_RDBPI_SASR = $1,
    /// <summary>
    /// Send At Medium Rate
    /// </summary>
    PUDS_SVC_PARAM_RDBPI_SAMR = $2,
    /// <summary>
    /// Send At Fast Rate
    /// </summary>
    PUDS_SVC_PARAM_RDBPI_SAFR = $3,
    /// <summary>
    /// Stop Sending
    /// </summary>
    PUDS_SVC_PARAM_RDBPI_SS = $4);

  // ISO-14229-1:2013 §10.6.2.2 p.144
{$Z1}
  uds_svc_param_dddi = (
    /// <summary>
    /// Define By Identifier
    /// </summary>
    PUDS_SVC_PARAM_DDDI_DBID = $1,
    /// <summary>
    /// Define By Memory Address
    /// </summary>
    PUDS_SVC_PARAM_DDDI_DBMA = $2,
    /// <summary>
    /// Clear Dynamically Defined Data Identifier
    /// </summary>
    PUDS_SVC_PARAM_DDDI_CDDDI = $3);

  // ISO-14229-1:2013 §11.3.2.2 p.194
{$Z1}
  uds_svc_param_rdtci = (
    /// <summary>
    /// report Number Of DTC By Status Mask
    /// </summary>
    PUDS_SVC_PARAM_RDTCI_RNODTCBSM = $1,
    /// <summary>
    /// report DTC By Status Mask
    /// </summary>
    PUDS_SVC_PARAM_RDTCI_RDTCBSM = $2,
    /// <summary>
    /// report DTC Snapshot Identification
    /// </summary>
    PUDS_SVC_PARAM_RDTCI_RDTCSSI = $3,
    /// <summary>
    /// report DTC Snapshot Record By DTC Number
    /// </summary>
    PUDS_SVC_PARAM_RDTCI_RDTCSSBDTC = $4,
    /// <summary>
    /// report DTC Snapshot Record By Record Number
    /// </summary>
    PUDS_SVC_PARAM_RDTCI_RDTCSSBRN = $5,
    /// <summary>
    /// report DTC Extended Data Record By DTC Number
    /// </summary>
    PUDS_SVC_PARAM_RDTCI_RDTCEDRBDN = $6,
    /// <summary>
    /// report Number Of DTC By Severity Mask Record
    /// </summary>
    PUDS_SVC_PARAM_RDTCI_RNODTCBSMR = $7,
    /// <summary>
    /// report DTC By Severity Mask Record
    /// </summary>
    PUDS_SVC_PARAM_RDTCI_RDTCBSMR = $8,
    /// <summary>
    /// report Severity Information Of DTC
    /// </summary>
    PUDS_SVC_PARAM_RDTCI_RSIODTC = $9,
    /// <summary>
    /// report Supported DTC
    /// </summary>
    PUDS_SVC_PARAM_RDTCI_RSUPDTC = $A,
    /// <summary>
    /// report First Test Failed DTC
    /// </summary>
    PUDS_SVC_PARAM_RDTCI_RFTFDTC = $B,
    /// <summary>
    /// report First Confirmed DTC
    /// </summary>
    PUDS_SVC_PARAM_RDTCI_RFCDTC = $C,
    /// <summary>
    /// report Most Recent Test Failed DTC
    /// </summary>
    PUDS_SVC_PARAM_RDTCI_RMRTFDTC = $D,
    /// <summary>
    /// report Most Recent Confirmed DTC
    /// </summary>
    PUDS_SVC_PARAM_RDTCI_RMRCDTC = $E,
    /// <summary>
    /// report Mirror Memory DTC By Status Mask
    /// </summary>
    PUDS_SVC_PARAM_RDTCI_RMMDTCBSM = $F,
    /// <summary>
    /// report Mirror Memory DTC Extended Data Record By DTC Number
    /// </summary>
    PUDS_SVC_PARAM_RDTCI_RMMDEDRBDN = $10,
    /// <summary>
    /// report Number Of Mirror MemoryDTC By Status Mask
    /// </summary>
    PUDS_SVC_PARAM_RDTCI_RNOMMDTCBSM = $11,
    /// <summary>
    /// report Number Of Emissions Related OBD DTC By Status Mask
    /// </summary>
    PUDS_SVC_PARAM_RDTCI_RNOOBDDTCBSM = $12,
    /// <summary>
    /// report Emissions Related OBD DTC By Status Mask
    /// </summary>
    PUDS_SVC_PARAM_RDTCI_ROBDDTCBSM = $13,
    /// <summary>
    /// report DTC Ext Data Record By Record Number
    /// </summary>
    PUDS_SVC_PARAM_RDTCI_RDTCEDBR = $16,
    /// <summary>
    /// report User Def Memory DTC By Status Mask
    /// </summary>
    PUDS_SVC_PARAM_RDTCI_RUDMDTCBSM = $17,
    /// <summary>
    /// report User Def Memory DTC Snapshot Record By DTC Number
    /// </summary>
    PUDS_SVC_PARAM_RDTCI_RUDMDTCSSBDTC = $18,
    /// <summary>
    /// report User Def Memory DTC Ext Data Record By DTC Number
    /// </summary>
    PUDS_SVC_PARAM_RDTCI_RUDMDTCEDRBDN = $19,
    /// <summary>
    /// report report DTC Extended Data Record Identification (ISO_14229-1 2020)
    /// </summary>
    PUDS_SVC_PARAM_RDTCI_RDTCEDI = $1A,
    /// <summary>
    /// report WWHOBD DTC By Mask Record
    /// </summary>
    PUDS_SVC_PARAM_RDTCI_RWWHOBDDTCBMR = $42,
    /// <summary>
    /// report WWHOBD DTC With Permanent Status
    /// </summary>
    PUDS_SVC_PARAM_RDTCI_RWWHOBDDTCWPS = $55,
    /// <summary>
    /// report DTC Information By DTC Readiness Group Identifier (ISO_14229-1 2020)
    /// </summary>
    PUDS_SVC_PARAM_RDTCI_RDTCBRGI = $56,
    // Reminder: following parameters were not defined as they are NOT in ISO-15765-3 :
    /// <summary>
    /// report DTC Fault Detection Counter
    /// </summary>
    PUDS_SVC_PARAM_RDTCI_RDTCFDC = $14,
    /// <summary>
    /// report DTC With Permanent Status
    /// </summary>
    PUDS_SVC_PARAM_RDTCI_RDTCWPS = $15);


  // DTCSeverityMask (DTCSVM): ISO-14229-1:2013 §D.3 p.366
{$Z1}
  uds_svc_param_rdtci_dtcsvm = (
    /// <summary>
    /// DTC severity bit definitions: no SeverityAvailable
    /// </summary>
    PUDS_SVC_PARAM_RDTCI_DTCSVM_NSA = $0,
    /// <summary>
    /// DTC severity bit definitions: maintenance Only
    /// </summary>
    PUDS_SVC_PARAM_RDTCI_DTCSVM_MO = $20,
    /// <summary>
    /// DTC severity bit definitions: check At Next Halt
    /// </summary>
    PUDS_SVC_PARAM_RDTCI_DTCSVM_CHKANH = $40,
    /// <summary>
    /// DTC severity bit definitions: check Immediately
    /// </summary>
    PUDS_SVC_PARAM_RDTCI_DTCSVM_CHKI = $80);

  // ISO-14229-1:2013 §E.1 p.374
{$Z1}
  uds_svc_param_iocbi = (
    /// <summary>
    /// inputOutputControlParameter: returnControlToECU (0 controlState bytes in request)
    /// </summary>
    PUDS_SVC_PARAM_IOCBI_RCTECU = $0,
    /// <summary>
    /// inputOutputControlParameter: resetToDefault (0 controlState bytes in request)
    /// </summary>
    PUDS_SVC_PARAM_IOCBI_RTD = $1,
    /// <summary>
    /// inputOutputControlParameter: freezeCurrentState (0 controlState bytes in request)
    /// </summary>
    PUDS_SVC_PARAM_IOCBI_FCS = $2,
    /// <summary>
    /// inputOutputControlParameter: shortTermAdjustment
    /// </summary>
    PUDS_SVC_PARAM_IOCBI_STA = $3);

  // ISO-14229-1:2013 §13.2.2.2 p.262
{$Z1}
  uds_svc_param_rc = (
    /// <summary>
    /// Start Routine
    /// </summary>
    PUDS_SVC_PARAM_RC_STR = $1,
    /// <summary>
    /// Stop Routine
    /// </summary>
    PUDS_SVC_PARAM_RC_STPR = $2,
    /// <summary>
    /// Request Routine Results
    /// </summary>
    PUDS_SVC_PARAM_RC_RRR = $3);

  // routineIdentifier: ISO-14229-1:2013 §F.1 p.375
{$Z2}
  uds_svc_param_rc_rid = (
    /// <summary>
    /// routineIdentifier: DeployLoopRoutineID
    /// </summary>
    PUDS_SVC_PARAM_RC_RID_DLRI_ = $E200,
    /// <summary>
    /// routineIdentifier: eraseMemory
    /// </summary>
    PUDS_SVC_PARAM_RC_RID_EM_ = $FF00,
    /// <summary>
    /// routineIdentifier: checkProgrammingDependencies
    /// </summary>
    PUDS_SVC_PARAM_RC_RID_CPD_ = $FF01,
    /// <summary>
    /// routineIdentifier: eraseMirrorMemoryDTCs
    /// </summary>
    PUDS_SVC_PARAM_RC_RID_EMMDTC_ = $FF02);

  // See ISO-14229-1:2013 §9.7.2.2 p.62 table 74
{$Z1}
  uds_svc_param_atp = (
    /// <summary>
    /// Read Extended Timing Parameter Set
    /// </summary>
    PUDS_SVC_PARAM_ATP_RETPS = $1,
    /// <summary>
    /// Set Timing Parameters To Default Values
    /// </summary>
    PUDS_SVC_PARAM_ATP_STPTDV = $2,
    /// <summary>
    /// Read Currently Active Timing Parameters
    /// </summary>
    PUDS_SVC_PARAM_ATP_RCATP = $3,
    /// <summary>
    /// Set Timing Parameters To Given Values
    /// </summary>
    PUDS_SVC_PARAM_ATP_STPTGV = $4);

  // See ISO-14229-1:2013 Annex G p.376 table G.1
{$Z1}
  uds_svc_param_rft_moop = (
    /// <summary>
    /// Add File
    /// </summary>
    PUDS_SVC_PARAM_RFT_MOOP_ADDFILE = $1,
    /// <summary>
    /// Delete File
    /// </summary>
    PUDS_SVC_PARAM_RFT_MOOP_DELFILE = $2,
    /// <summary>
    /// Replace File
    /// </summary>
    PUDS_SVC_PARAM_RFT_MOOP_REPLFILE = $3,
    /// <summary>
    /// Read File
    /// </summary>
    PUDS_SVC_PARAM_RFT_MOOP_RDFILE = $4,
    /// <summary>
    /// Read Dir
    /// </summary>
    PUDS_SVC_PARAM_RFT_MOOP_RDDIR = $5,
    /// <summary>
    /// Resume File (ISO-14229-1:2020 Annex G p.447 table G.1)
    /// </summary>
    PUDS_SVC_PARAM_RFT_MOOP_RSFILE = $6);

  // Represents the subfunction parameter for UDS service Authentication (see ISO 14229-1:2020 §10.6.5.2 Table 74 Request message SubFunction parameter definition p.76)
{$Z1}
  uds_svc_authentication_subfunction = (
    /// <summary>
    /// DeAuthenticate
    /// </summary>
    PUDS_SVC_PARAM_AT_DA = $00,
    /// <summary>
    /// VerifyCertificateUnidirectional
    /// </summary>
    PUDS_SVC_PARAM_AT_VCU = $01,
    /// <summary>
    /// VerifyCertificateBidirectional
    /// </summary>
    PUDS_SVC_PARAM_AT_VCB = $02,
    /// <summary>
    /// ProofOfOwnership
    /// </summary>
    PUDS_SVC_PARAM_AT_POWN = $03,
    /// <summary>
    /// TransmitCertificate
    /// </summary>
    PUDS_SVC_PARAM_AT_TC = $04,
    /// <summary>
    /// RequestChallengeForAuthentication
    /// </summary>
    PUDS_SVC_PARAM_AT_RCFA = $05,
    /// <summary>
    /// VerifyProofOfOwnershipUnidirectional
    /// </summary>
    PUDS_SVC_PARAM_AT_VPOWNU = $06,
    /// <summary>
    /// VerifyProofOfOwnershipBidirectional
    /// </summary>
    PUDS_SVC_PARAM_AT_VPOWNB = $07,
    /// <summary>
    /// AuthenticationConfiguration
    /// </summary>
    PUDS_SVC_PARAM_AT_AC = $08);

  // Represents the return parameter for UDS service Authentication (see ISO 14229-1:2020 §B.5 AuthenticationReturnParameter definitions p.403)
{$Z1}
  uds_svc_authentication_return_parameter = (
    /// <summary>
    /// Request Accepted
    /// </summary>
    PUDS_SVC_PARAM_AT_RV_RA = $00,
    /// <summary>
    /// General Reject
    /// </summary>
    PUDS_SVC_PARAM_AT_RV_GR = $01,
    /// <summary>
    /// Authentication Configuration APCE
    /// </summary>
    PUDS_SVC_PARAM_AT_RV_ACAPCE = $02,
    /// <summary>
    /// Authentication Configuration ACR with Asymmetric Cryptography
    /// </summary>
    PUDS_SVC_PARAM_AT_RV_ACACRAC = $03,
    /// <summary>
    /// Authentication Configuration ACR with Symmetric Cryptography
    /// </summary>
    PUDS_SVC_PARAM_AT_RV_ACACRSC = $04,
    /// <summary>
    /// DeAuthentication Successful
    /// </summary>
    PUDS_SVC_PARAM_AT_RV_DAS = $10,
    /// <summary>
    /// Certificate Verified, Ownership Verification Necessary
    /// </summary>
    PUDS_SVC_PARAM_AT_RV_CVOVN = $11,
    /// <summary>
    /// Ownership Verified, Authentication Complete
    /// </summary>
    PUDS_SVC_PARAM_AT_RV_OVAC = $12,
    /// <summary>
    /// Certificate Verified
    /// </summary>
    PUDS_SVC_PARAM_AT_RV_CV = $13);

  // Pointers

  Puds_msg = ^uds_msg;
  Pcantp_timestamp = ^cantp_timestamp;
  Puds_mapping = ^uds_mapping;
  PUInt16 = ^UInt16;
  Puds_svc_param_di = ^uds_svc_param_di;

  /// //////////////////////////////////////////////////////
  // Message definitions
  /// //////////////////////////////////////////////////////

  /// <summary>
  /// Represents a UDS Network Addressing Information
  /// </summary>
  uds_netaddrinfo = record
    /// <summary>
    /// communication protocol
    /// </summary>
    protocol: uds_msgprotocol;
    /// <summary>
    /// ISO-TP target type
    /// </summary>
    target_type: cantp_isotp_addressing;
    /// <summary>
    /// source address
    /// </summary>
    source_addr: UInt16;
    /// <summary>
    /// target address
    /// </summary>
    target_addr: UInt16;
    /// <summary>
    /// extension address
    /// </summary>
    extension_addr: Byte;
  End;

  /// <summary>
  /// Represents the diagnostic session's information of a server
  /// </summary>
  uds_sessioninfo = record
    /// <summary>
    /// Network address information
    /// </summary>
    nai: uds_netaddrinfo;
    /// <summary>
    /// Types and flags of the CAN/CAN-FD frames
    /// </summary>
    can_msg_type: cantp_can_msgtype;
    /// <summary>
    /// Activated Diagnostic Session (see PUDS_SVC_PARAM_DSC_xxx values)
    /// </summary>
    session_type: Byte;
    /// <summary>
    /// Default P2Can_Server_Max timing for the activated session (resolution: 1ms)
    /// </summary>
    timeout_p2can_server_max: UInt16;
    /// <summary>
    /// Enhanced P2Can_Server_Max timing for the activated session (resolution: 10ms)
    /// </summary>
    timeout_enhanced_p2can_server_max: UInt16;
    /// <summary>
    /// Time between 2 TesterPresents
    /// </summary>
    s3_client_ms: UInt16;
  End;

  /// <summary>
  /// Represents the configuration of a PUDS message
  /// </summary>
  uds_msgconfig = record
    /// <summary>
    /// structure specific flags
    /// </summary>
    typem: uds_msgtype;
    /// <summary>
    /// Network Addressing Information
    /// </summary>
    nai: uds_netaddrinfo;
    /// <summary>
    /// (optional) CAN ID (for configuration use either nai or m_can_id)
    /// </summary>
    can_id: UInt32;
    /// <summary>
    /// optional flags for the CAN layer (29 bits CAN-ID, FD, BRS)
    /// </summary>
    can_msgtype: cantp_can_msgtype;
    /// <summary>
    /// Default CAN DLC value to use with segmented messages
    /// </summary>
    can_tx_dlc: Byte;
  End;

  /// <summary>
  /// Represents a mapping between an UDS Network Addressing Information and a CAN ID.
  /// </summary>
  uds_mapping = record
    /// <summary>
    /// Mapping's unique ID
    /// </summary>
    uid: Pointer;
    /// <summary>
    /// CAN ID mapped to the Network Address Information
    /// </summary>
    can_id: UInt32;
    /// <summary>
    /// CAN ID used for the flow control frame (formerly 'can_id_resp')
    /// </summary>
    can_id_flow_ctrl: UInt32;
    /// <summary>
    /// CAN frame msgtype (only PCANTP_CAN_MSGTYPE_STANDARD or PCANTP_CAN_MSGTYPE_EXTENDED is mandatory)
    /// </summary>
    can_msgtype: cantp_can_msgtype;
    /// <summary>
    /// Default CAN DLC value to use with segmented messages
    /// </summary>
    can_tx_dlc: Byte;
    /// <summary>
    /// Network Addressing Information
    /// </summary>
    nai: uds_netaddrinfo;
  End;

  /// <summary>
  /// Provides accessors to the corresponding data in the cantp_msg
  /// </summary>
  uds_msgaccess = record
    /// <summary>
    /// Pointer to the Service ID in message's data.
    /// Use it with Marshal.ReadByte, Marshal.WriteByte
    /// See special C# functions
    /// </summary>
    service_id: ^Byte;
    /// <summary>
    /// Pointer to the first parameter in message's data.
    /// Use it with Marshal.ReadByte, Marshal.WriteByte
    /// See special VB.Net functions
    /// </summary>
    param: ^Byte;
    /// <summary>
    /// Pointer to the Negative Response Code (see uds_nrc enumeration) in message's data (NULL on positive response).
    /// Use it with Marshal.ReadByte, Marshal.WriteByte
    /// See special VB.Net functions
    /// </summary>
    nrc: ^Byte;
  End;

  /// <summary>
  /// Represents the content of a UDS message.
  /// </summary>
  uds_msg = record
    /// <summary>
    /// structure specific flags
    /// </summary>
    typem: uds_msgtype;
    /// <summary>
    /// quick accessors to the cantp_msg data
    /// </summary>
    links: uds_msgaccess;
    /// <summary>
    /// the PCANTP message encapsulating the UDS data
    /// </summary>
    msg: cantp_msg;
  End;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}

  TUDSApi_2013 = class
{$ELSE}
  TUDSApi = class
{$ENDIF}
  public
  class var
  const

    /// <summary>
    /// Mapping does not require a Flow Control frame.
    /// </summary>
    PUDS_MAPPING_FLOW_CTRL_NONE = UInt32(-1);

{$IF Not Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
    /// <summary>
    /// server wants more time
    /// </summary>
    PUDS_NRC_EXTENDED_TIMING = $78;
    /// <summary>
    /// positive response offset
    /// </summary>
    PUDS_SI_POSITIVE_RESPONSE = $40;

    /// ///////////////////////
    // PUDS parameter values
    /// ///////////////////////

    /// <summary>
    /// Default maximum timeout in milliseconds for UDS transmit confirmation
    /// </summary>
    PUDS_TIMEOUT_REQUEST = 10000;
    /// <summary>
    /// Default maximum timeout in milliseconds for UDS response reception
    /// </summary>
    PUDS_TIMEOUT_RESPONSE = 10000;
    /// <summary>
    /// Flag stating that the address is defined as a ISO-15765-3 address
    /// </summary>
    PUDS_SERVER_ADDR_FLAG_ENHANCED_ISO_15765_3 = $1000;
    /// <summary>
    /// Mask used for the ISO-15765-3 enhanced addresses
    /// </summary>
    PUDS_SERVER_ADDR_MASK_ENHANCED_ISO_15765_3 = $07FF;
    /// <summary>
    /// The Channel is illegal or not available
    /// </summary>
    PUDS_CHANNEL_UNAVAILABLE = $00;
    /// <summary>
    /// The Channel is available
    /// </summary>
    PUDS_CHANNEL_AVAILABLE = $01;
    /// <summary>
    /// The Channel is valid, and is being used
    /// </summary>
    PUDS_CHANNEL_OCCUPIED = $02;
    /// <summary>
    /// Uses CAN frame data optimization
    /// </summary>
    PUDS_CAN_DATA_PADDING_NONE = $00;
    /// <summary>
    /// Uses CAN frame data padding (default, i.e. CAN DLC = 8)
    /// </summary>
    PUDS_CAN_DATA_PADDING_ON = $01;
    /// <summary>
    /// Default value used if CAN data padding is enabled
    /// </summary>
    PUDS_CAN_DATA_PADDING_VALUE = $55;
{$ENDIF}
    /// <summary>
    /// Default server performance requirement in ms (See ISO_14229-2_2013 §7.2 table 4)
    /// </summary>
    PUDS_P2CAN_SERVER_MAX_DEFAULT = 50;
    /// <summary>
    /// Enhanced server performance requirement in ms (See ISO_14229-2_2013 §7.2 table 4)
    /// </summary>
    PUDS_P2CAN_ENHANCED_SERVER_MAX_DEFAULT = 5000;
    /// <summary>
    /// Default P3 timing parameter in ms (See ISO_14229-2_2013 §7.2 table 4)
    /// </summary>
    PUDS_P3CAN_DEFAULT = PUDS_P2CAN_SERVER_MAX_DEFAULT;
    /// <summary>
    /// Recommended S3 client timeout in ms (See ISO_14229-2_2013 §7.5 table 5)
    /// </summary>
    PUDS_S3_CLIENT_TIMEOUT_RECOMMENDED = 2000;

    /// <summary>
    /// Disable debug messages (default)
    /// </summary>
    PUDS_DEBUG_LVL_NONE = $00;
    /// <summary>
    /// Enable debug messages (only errors)
    /// </summary>
    PUDS_DEBUG_LVL_ERROR = $F1;
    /// <summary>
    /// Enable debug messages (only warnings, errors)
    /// </summary>
    PUDS_DEBUG_LVL_WARNING = $F2;
    /// <summary>
    /// Enable debug messages (only informations, warnings, errors)
    /// </summary>
    PUDS_DEBUG_LVL_INFORMATION = $F3;
    /// <summary>
    /// Enable debug messages (only notices, informations, warnings, errors)
    /// </summary>
    PUDS_DEBUG_LVL_NOTICE = $F4;
    /// <summary>
    /// Enable debug messages (only debug, notices, informations, warnings, errors)
    /// </summary>
    PUDS_DEBUG_LVL_DEBUG = $F5;
    /// <summary>
    /// Enable all debug messages
    /// </summary>
    PUDS_DEBUG_LVL_TRACE = $F6;

    /// <summary>
    /// Option that can be used as channel identifier in Svc* functions: only prepare uds_msg structure and do not send it
    /// </summary>
    PUDS_ONLY_PREPARE_REQUEST: cantp_handle = PCANTP_HANDLE_NONEBUS;

    /// /////////////////////////////////////////
    // PCAN-UDS API: Core function declarations
    /// /////////////////////////////////////////

    /// <summary>
    /// Initializes a PUDS channel based on a PCANTP channel handle (without CAN-FD support)
    /// </summary>
    /// <remarks>Only one PUDS channel can be initialized per CAN-Channel</remarks>
    /// <param name="channel">A PCANTP channel handle</param>
    /// <param name="baudrate">The CAN Hardware speed</param>
    /// <param name="hw_type">NON PLUG-N-PLAY: The type of hardware and operation mode</param>
    /// <param name="io_port">NON PLUG-N-PLAY: The I/O address for the parallel port</param>
    /// <param name="interrupt">NON PLUG-N-PLAY: Interrupt number of the parallel port</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function Initialize_2013(channel: cantp_handle;
      baudrate: cantp_baudrate; hw_type: cantp_hwtype; io_port: UInt32;
      interrupt: UInt16): uds_status; overload;

    /// <summary>
    /// Initializes a PUDS channel based on a PCANTP channel handle (without CAN-FD support)
    /// </summary>
    /// <remarks>Only one PUDS channel can be initialized per CAN-Channel</remarks>
    /// <param name="channel">A PCANTP channel handle</param>
    /// <param name="baudrate">The CAN Hardware speed</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function Initialize_2013(channel: cantp_handle;
      baudrate: cantp_baudrate): uds_status; overload;

    /// <summary>
    /// Initializes a PUDS channel based on a PCANTP channel handle (including CAN-FD support)
    /// </summary>
    /// <param name="channel">The handle of a FD capable PCANTP Channel</param>
    /// <param name="bitrate_fd">The speed for the communication (FD bit rate string)</param>
    /// <remarks>Only one PUDS channel can be initialized per CAN-Channel.
    /// See PCAN_BR_* values
    /// * Parameter and values must be separated by '='
    /// * Couples of parameter/value must be separated by ','
    /// * Following parameter must be filled out: f_clock, data_brp, data_sjw, data_tseg1, data_tseg2,
    /// nom_brp, nom_sjw, nom_tseg1, nom_tseg2.
    /// * Following parameters are optional (not used yet): data_ssp_offset, nom_samp
    /// </remarks>
    /// <example>f_clock_mhz=80,nom_brp=0,nom_tseg1=13,nom_tseg2=0,nom_sjw=0,data_brp=0,
    /// data_tseg1=13,data_tseg2=0,data_sjw=0</example>
    /// <returns>A uds_status error code</returns>
    class function InitializeFD_2013(channel: cantp_handle;
      const bitrate_fd: cantp_bitrate): uds_status;

    /// <summary>
    /// Uninitializes a PUDS channel
    /// </summary>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function Uninitialize_2013(channel: cantp_handle): uds_status;

    /// <summary>
    /// Resets the receive and transmit queues of a PUDS channel
    /// </summary>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function Reset_2013(channel: cantp_handle): uds_status;

    /// <summary>
    /// Gets information about the internal BUS status of a PUDS channel
    /// </summary>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function GetCanBusStatus_2013(channel: cantp_handle): uds_status;

    /// <summary>
    /// Reads a PUDS message from the receive queue of a PUDS channel
    /// </summary>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="out_msg_buffer">[out]A uds_msg structure buffer to store the PUDS message</param>
    /// <param name="in_msg_request">(Optional) If NULL the first available message is fetched.
    /// Otherwise in_msg_request must represent a sent PUDS request.
    /// To look for the request confirmation, in_msg_request->type should not have the loopback flag;
    /// otherwise a response from the target ECU will be searched.</param>
    /// <param name="out_timestamp">A cantp_timestamp structure buffer to get
    /// the reception time of the message. If this value is not desired, this parameter
    /// should be passed as NULL</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function Read_2013(channel: cantp_handle; var out_msg_buffer: uds_msg;
      in_msg_request: Puds_msg; out_timestamp: Pcantp_timestamp)
      : uds_status; overload;

    /// <summary>
    /// Reads a PUDS message from the receive queue of a PUDS channel
    /// </summary>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="out_msg_buffer">[out]A uds_msg structure buffer to store the PUDS message</param>
    /// <param name="in_msg_request">(Optional) If NULL the first available message is fetched.
    /// Otherwise in_msg_request must represent a sent PUDS request.
    /// To look for the request confirmation, in_msg_request->type should not have the loopback flag;
    /// otherwise a response from the target ECU will be searched.</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function Read_2013(channel: cantp_handle; var out_msg_buffer: uds_msg;
      in_msg_request: Puds_msg): uds_status; overload;

    /// <summary>
    /// Reads a PUDS message from the receive queue of a PUDS channel
    /// </summary>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="out_msg_buffer">[out]A uds_msg structure buffer to store the PUDS message</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function Read_2013(channel: cantp_handle; var out_msg_buffer: uds_msg)
      : uds_status; overload;

    /// <summary>
    /// Transmits a PUDS message
    /// </summary>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="msg_buffer">A uds_msg buffer with the message to be sent</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function Write_2013(channel: cantp_handle; msg_buffer: Puds_msg)
      : uds_status;

    /// <summary>
    /// Adds a user-defined UDS mapping (relation between a CAN ID and a UDS Network Address Information)
    /// </summary>
    /// <remark>
    /// Defining a mapping enables ISO-TP communication with opened Addressing Formats
    /// (like PCANTP_ISOTP_FORMAT_NORMAL or PCANTP_ISOTP_FORMAT_EXTENDED).
    /// </remark>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="mapping">Mapping to be added</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function AddMapping_2013(channel: cantp_handle; mapping: Puds_mapping)
      : uds_status;

    /// <summary>
    /// Removes all user-defined PUDS mappings corresponding to a CAN ID
    /// </summary>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="can_id">The mapped CAN ID to search for</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function RemoveMappingByCanId_2013(channel: cantp_handle;
      can_id: UInt32): uds_status;

    /// <summary>
    /// Removes a user-defined PUDS mapping
    /// </summary>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="mapping">The mapping to remove</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function RemoveMapping_2013(channel: cantp_handle;
      mapping: uds_mapping): uds_status;

    /// <summary>
    /// Retrieves a mapping matching the given CAN identifier and message type (11bits, 29 bits, FD, etc.)
    /// </summary>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="buffer">[out]Buffer to store the searched mapping</param>
    /// <param name="can_id">The mapped CAN ID to look for</param>
    /// <param name="can_msgtype">The CAN message type to look for (11bits, 29 bits, FD, etc.)</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success,
    /// PUDS_STATUS_MAPPING_NOT_INITIALIZED if no mapping was found.</returns>
    class function GetMapping_2013(channel: cantp_handle;
      var buffer: uds_mapping; can_id: UInt32; can_msgtype: cantp_can_msgtype)
      : uds_status;

    /// <summary>
    /// Retrieves all the mappings defined for a PUDS channel
    /// </summary>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="buffer">[out]Buffer of mappings</param>
    /// <param name="buffer_length">The number of uds_mapping elements the buffer can store.</param>
    /// <param name="count">[out]The actual number of elements copied in the buffer.</param>
    class function GetMappings_2013(channel: cantp_handle; buffer: Puds_mapping;
      buffer_length: UInt16; count: PUInt16): uds_status;

    /// <summary>
    /// Adds a "PASS" filter on a CAN ID
    /// </summary>
    /// <remark>
    /// CAN and CAN FD frames matching this CAN ID will be fetchable by the UDS API with UDS_Read_2013 function.
    /// By default all frames are ignored and are available in lower APIs.
    /// </remark>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="can_id">CAN identifier to listen to</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function AddCanIdFilter_2013(channel: cantp_handle; can_id: UInt32)
      : uds_status;

    /// <summary>
    /// Remove a "PASS" filter on a CAN ID
    /// </summary>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="can_id">CAN identifier to remove</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function RemoveCanIdFilter_2013(channel: cantp_handle; can_id: UInt32)
      : uds_status;

    /// <summary>
    /// Retrieves a PUDS channel value
    /// </summary>
    /// <remark>
    /// * Parameter PUDS_PARAM_SERVER_ADDRESS uses 2 bytes data to describe
    /// the physical address of the equipment, but the first byte is needed only
    /// for ISO-15765-3 Enhanced diagnostics 29 bit CAN ID where addresses
    /// are 11 bits long.
    /// * Parameter PUDS_PARAM_SERVER_FILTER uses 2 bytes data to describe
    /// a functional address, but the first byte is needed only
    /// for ISO-15765-3 Enhanced diagnostics 29 bit CAN ID where addresses
    /// are 11 bits long; the Most Significant Bit is used to define filter
    /// status (see PUDS_SERVER_FILTER_LISTEN).
    /// </remark>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="parameter">The parameter to get</param>
    /// <param name="buffer">Buffer for the parameter value</param>
    /// <param name="buffer_size">Size in bytes of the buffer</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function GetValue_2013(channel: cantp_handle;
      parameter: uds_parameter; buffer: Pointer; buffer_size: UInt32)
      : uds_status; overload;

    /// <summary>
    /// Retrieves a PUDS channel value
    /// </summary>
    /// <remark>
    /// * Parameter PUDS_PARAM_SERVER_ADDRESS uses 2 bytes data to describe
    /// the physical address of the equipment, but the first byte is needed only
    /// for ISO-15765-3 Enhanced diagnostics 29 bit CAN ID where addresses
    /// are 11 bits long.
    /// * Parameter PUDS_PARAM_SERVER_FILTER uses 2 bytes data to describe
    /// a functional address, but the first byte is needed only
    /// for ISO-15765-3 Enhanced diagnostics 29 bit CAN ID where addresses
    /// are 11 bits long; the Most Significant Bit is used to define filter
    /// status (see PUDS_SERVER_FILTER_LISTEN).
    /// </remark>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="parameter">The parameter to get</param>
    /// <param name="buffer">Buffer for the parameter value</param>
    /// <param name="buffer_size">Size in bytes of the buffer</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function GetValue_2013(channel: cantp_handle;
      parameter: uds_parameter; buffer: PAnsiChar; buffer_size: UInt32)
      : uds_status; overload;

    /// <summary>
    /// Retrieves a PUDS channel value
    /// </summary>
    /// <remark>
    /// * Parameter PUDS_PARAM_SERVER_ADDRESS uses 2 bytes data to describe
    /// the physical address of the equipment, but the first byte is needed only
    /// for ISO-15765-3 Enhanced diagnostics 29 bit CAN ID where addresses
    /// are 11 bits long.
    /// * Parameter PUDS_PARAM_SERVER_FILTER uses 2 bytes data to describe
    /// a functional address, but the first byte is needed only
    /// for ISO-15765-3 Enhanced diagnostics 29 bit CAN ID where addresses
    /// are 11 bits long; the Most Significant Bit is used to define filter
    /// status (see PUDS_SERVER_FILTER_LISTEN).
    /// </remark>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="parameter">The parameter to get</param>
    /// <param name="buffer">Buffer for the parameter value</param>
    /// <param name="buffer_size">Size in bytes of the buffer</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function GetValue_2013(channel: cantp_handle;
      parameter: uds_parameter; buffer: PLongWord; buffer_size: UInt32)
      : uds_status; overload;

    /// <summary>
    /// Retrieves a PUDS channel value
    /// </summary>
    /// <remark>
    /// * Parameter PUDS_PARAM_SERVER_ADDRESS uses 2 bytes data to describe
    /// the physical address of the equipment, but the first byte is needed only
    /// for ISO-15765-3 Enhanced diagnostics 29 bit CAN ID where addresses
    /// are 11 bits long.
    /// * Parameter PUDS_PARAM_SERVER_FILTER uses 2 bytes data to describe
    /// a functional address, but the first byte is needed only
    /// for ISO-15765-3 Enhanced diagnostics 29 bit CAN ID where addresses
    /// are 11 bits long; the Most Significant Bit is used to define filter
    /// status (see PUDS_SERVER_FILTER_LISTEN).
    /// </remark>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="parameter">The parameter to get</param>
    /// <param name="buffer">Buffer for the parameter value</param>
    /// <param name="buffer_size">Size in bytes of the buffer</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function GetValue_2013(channel: cantp_handle;
      parameter: uds_parameter; buffer: PByte; buffer_size: UInt32)
      : uds_status; overload;

    /// <summary>
    /// Configures or sets a PUDS channel value
    /// </summary>
    /// <remark>
    /// * Parameter PUDS_PARAM_SERVER_ADDRESS uses 2 bytes data to describe
    /// the physical address of the equipment, but the first byte is needed only
    /// for ISO-15765-3 Enhanced diagnostics 29 bit CAN ID where addresses
    /// are 11 bits long.
    /// * Parameter PUDS_PARAM_SERVER_FILTER uses 2 bytes data to describe
    /// a functional address, but the first byte is needed only
    /// for ISO-15765-3 Enhanced diagnostics 29 bit CAN ID where addresses
    /// are 11 bits long; the Most Significant Bit is used to define filter
    /// status.
    /// </remark>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="parameter">The parameter to set</param>
    /// <param name="buffer">Buffer with the value to be set</param>
    /// <param name="buffer_size">Size in bytes of the buffer</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function SetValue_2013(channel: cantp_handle;
      parameter: uds_parameter; buffer: Pointer; buffer_size: UInt32)
      : uds_status; overload;

    /// <summary>
    /// Configures or sets a PUDS channel value
    /// </summary>
    /// <remark>
    /// * Parameter PUDS_PARAM_SERVER_ADDRESS uses 2 bytes data to describe
    /// the physical address of the equipment, but the first byte is needed only
    /// for ISO-15765-3 Enhanced diagnostics 29 bit CAN ID where addresses
    /// are 11 bits long.
    /// * Parameter PUDS_PARAM_SERVER_FILTER uses 2 bytes data to describe
    /// a functional address, but the first byte is needed only
    /// for ISO-15765-3 Enhanced diagnostics 29 bit CAN ID where addresses
    /// are 11 bits long; the Most Significant Bit is used to define filter
    /// status.
    /// </remark>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="parameter">The parameter to set</param>
    /// <param name="buffer">Buffer with the value to be set</param>
    /// <param name="buffer_size">Size in bytes of the buffer</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function SetValue_2013(channel: cantp_handle;
      parameter: uds_parameter; buffer: PLongWord; buffer_size: UInt32)
      : uds_status; overload;

    /// <summary>
    /// Configures or sets a PUDS channel value
    /// </summary>
    /// <remark>
    /// * Parameter PUDS_PARAM_SERVER_ADDRESS uses 2 bytes data to describe
    /// the physical address of the equipment, but the first byte is needed only
    /// for ISO-15765-3 Enhanced diagnostics 29 bit CAN ID where addresses
    /// are 11 bits long.
    /// * Parameter PUDS_PARAM_SERVER_FILTER uses 2 bytes data to describe
    /// a functional address, but the first byte is needed only
    /// for ISO-15765-3 Enhanced diagnostics 29 bit CAN ID where addresses
    /// are 11 bits long; the Most Significant Bit is used to define filter
    /// status.
    /// </remark>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="parameter">The parameter to set</param>
    /// <param name="buffer">Buffer with the value to be set</param>
    /// <param name="buffer_size">Size in bytes of the buffer</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function SetValue_2013(channel: cantp_handle;
      parameter: uds_parameter; buffer: PAnsiChar; buffer_size: UInt32)
      : uds_status; overload;

    /// <summary>
    /// Configures or sets a PUDS channel value
    /// </summary>
    /// <remark>
    /// * Parameter PUDS_PARAM_SERVER_ADDRESS uses 2 bytes data to describe
    /// the physical address of the equipment, but the first byte is needed only
    /// for ISO-15765-3 Enhanced diagnostics 29 bit CAN ID where addresses
    /// are 11 bits long.
    /// * Parameter PUDS_PARAM_SERVER_FILTER uses 2 bytes data to describe
    /// a functional address, but the first byte is needed only
    /// for ISO-15765-3 Enhanced diagnostics 29 bit CAN ID where addresses
    /// are 11 bits long; the Most Significant Bit is used to define filter
    /// status.
    /// </remark>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="parameter">The parameter to set</param>
    /// <param name="buffer">Buffer with the value to be set</param>
    /// <param name="buffer_size">Size in bytes of the buffer</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function SetValue_2013(channel: cantp_handle;
      parameter: uds_parameter; buffer: PByte; buffer_size: UInt32)
      : uds_status; overload;

    /// <summary>
    /// Returns a descriptive text of a given uds_status error
    /// code, in any desired language
    /// </summary>
    /// <remarks>The current languages available for translation are:
    /// Neutral (0x00), German (0x07), English (0x09), Spanish (0x0A),
    /// Italian (0x10) and French (0x0C)</remarks>
    /// <param name="error_code">A uds_status error code</param>
    /// <param name="language">Indicates a 'Primary language ID'</param>
    /// <param name="buffer">Buffer for a null terminated char array</param>
    /// <param name="buffer_size">Buffer size</param>
    /// <returns>A uds_status error code</returns>
    class function GetErrorText_2013(error_code: uds_status; language: UInt16;
      buffer: PAnsiChar; buffer_size: UInt32): uds_status;

    /// <summary>
    /// Gets the session information known by the API
    /// </summary>
    /// <remark>
    /// session_info must be initialized a network address information associated to an ECU.
    /// Note that the session's information within the API may be different to the actual session of the corresponding ECU.
    /// </remark>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="session_info">[in|out] The session is filled if an ECU session, matching session_info->nai, exists</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function GetSessionInformation_2013(channel: cantp_handle;
      var session_info: uds_sessioninfo): uds_status;

    /// <summary>
    /// Checks if a status matches an expected result (default is PUDS_STATUS_OK).
    /// </summary>
    /// <param name="status">The status to analyze.</param>
    /// <param name="status_expected">The expected status (default is PUDS_STATUS_OK).</param>
    /// <param name="strict_mode">Enable strict mode (default is false). Strict mode ensures that bus or extra information are the same.</param>
    /// <returns>Returns true if the status matches expected parameter.</returns>
    class function StatusIsOk_2013(status: uds_status;
      status_expected: uds_status; strict_mode: Boolean): Boolean; overload;

    /// <summary>
    /// Checks if a status matches an expected result in a non-strict mode (default is PUDS_STATUS_OK).
    /// Strict mode ensures that bus or extra information are the same.
    /// </summary>
    /// <param name="status">The status to analyze.</param>
    /// <param name="status_expected">The expected status (default is PUDS_STATUS_OK).</param>
    /// <returns>Returns true if the status matches expected parameter.</returns>
    class function StatusIsOk_2013(status: uds_status;
      status_expected: uds_status): Boolean; overload;

    /// <summary>
    /// Checks if a status matches PUDS_STATUS_OK in a non-strict mode.
    /// Strict mode ensures that bus or extra information are the same.
    /// </summary>
    /// <param name="status">The status to analyze.</param>
    /// <returns>Returns true if the status matches PUDS_STATUS_OK.</returns>
    class function StatusIsOk_2013(status: uds_status): Boolean; overload;

    /// //////////////////////////////////////////////////////
    // PCAN-UDS API: PUDS Message initialization function declarations
    /// //////////////////////////////////////////////////////

    /// <summary>
    /// Allocates a PUDS message based on the given configuration
    /// </summary>
    /// <param name="msg_buffer">A uds_msg structure buffer (it will be freed if required)</param>
    /// <param name="msg_configuration">Configuration of the PUDS message to allocate</param>
    /// <param name="msg_data_length">Length of the message's data</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function MsgAlloc_2013(var msg_buffer: uds_msg;
      msg_configuration: uds_msgconfig; msg_data_length: UInt32): uds_status;

    /// <summary>
    /// Deallocates a PUDS message
    /// </summary>
    /// <param name="msg_buffer">An allocated uds_msg structure buffer</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function MsgFree_2013(var msg_buffer: uds_msg): uds_status;

    /// <summary>
    /// Copies a PUDS message to another buffer.
    /// </summary>
    /// <param name="msg_buffer_dst">A uds_msg structure buffer to store the copied message.</param>
    /// <param name="msg_buffer_src">The uds_msg structure buffer to copy.</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function MsgCopy_2013(var msg_buffer_dst: uds_msg;
      const msg_buffer_src: Puds_msg): uds_status;

    /// <summary>
    /// Moves a PUDS message to another buffer (and cleans the original message structure).
    /// </summary>
    /// <param name="msg_buffer_dst">A uds_msg structure buffer to store the message.</param>
    /// <param name="msg_buffer_src">The uds_msg structure buffer used as the source (will be cleaned).</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function MsgMove_2013(var msg_buffer_dst: uds_msg;
      var msg_buffer_src: uds_msg): uds_status;

    /// //////////////////////////////////////////////////////
    // PCAN-UDS API: Utility function declarations
    /// //////////////////////////////////////////////////////

    /// <summary>
    /// Waits for a message (a response or a transmit confirmation) based on a UDS request
    /// </summary>
    /// <remarks>
    /// Warning: The order of the parameters has changed in PCAN-UDS 2.0 API.
    /// </remarks>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="msg_request">A sent uds_msg message used as a reference to find the waited message</param>
    /// <param name="is_waiting_for_tx">States if the message to wait for is a transmit confirmation</param>
    /// <param name="timeout">Maximum time to wait (in milliseconds) for a message indication corresponding to the message request</param>
    /// <param name="timeout_enhanced">Maximum time to wait for a message indication if the server requests more time</param>
    /// <param name="out_msg_response">A uds_msg structure buffer to store the PUDS response</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function WaitForSingleMessage_2013(channel: cantp_handle;
      msg_request: Puds_msg; is_waiting_for_tx: Boolean; timeout: UInt32;
      timeout_enhanced: UInt32; var out_msg_response: uds_msg): uds_status;

    /// <summary>
    /// Waits for multiple responses (from a functional request for instance) based on a PUDS message request.
    /// </summary>
    /// <remarks>
    /// Warning: The order of the parameters has changed in PCAN-UDS 2.0 API.
    /// </remarks>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="msg_request">A sent uds_msg message used as a reference to find the waited message</param>
    /// <param name="timeout">Maximum time to wait (in milliseconds) for a message indication corresponding to the message request.</param>
    /// <param name="timeout_enhanced">Maximum time to wait for a message indication if the server requested more time</param>
    /// <param name="wait_until_timeout">if <code>FALSE</code> the function is interrupted if out_msg_count reaches max_msg_count.</param>
    /// <param name="max_msg_count">Length of the buffer array (max. messages that can be received)</param>
    /// <param name="out_msg_responses">Buffer must be an array of 'max_msg_count' entries (must have at least
    /// a size of max_msg_count * sizeof(uds_msg) bytes</param>
    /// <param name="out_msg_count">Actual number of messages read</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success,
    /// PUDS_ERROR_OVERFLOW indicates success but buffer was too small to hold all responses.</returns>
    class function WaitForFunctionalResponses_2013(channel: cantp_handle;
      msg_request: Puds_msg; timeout: UInt32; timeout_enhanced: UInt32;
      wait_until_timeout: Boolean; max_msg_count: UInt32;
      out_msg_responses: Puds_msg; var out_msg_count: UInt32): uds_status;

    /// <summary>
    /// Handles the communication workflow for a UDS service expecting a single response.
    /// </summary>
    /// <remark>
    /// 1) Warning: The order of the parameters has changed in PCAN-UDS 2.0 API.
    /// 2) The function waits for a transmit confirmation then for a message response.
    /// Even if the SuppressPositiveResponseMessage flag is set, the function will still wait
    /// for an eventual Negative Response.
    /// </remark>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="msg_request">A sent uds_msg message used as a reference to manage the UDS service</param>
    /// <param name="out_msg_response">A uds_msg structure buffer to store the PUDS response</param>
    /// <param name="out_msg_request_confirmation">A uds_msg structure buffer to store the PUDS request confirmation</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function WaitForService_2013(channel: cantp_handle;
      msg_request: Puds_msg; var out_msg_response: uds_msg;
      out_msg_request_confirmation: Puds_msg): uds_status; overload;

    /// <summary>
    /// Handles the communication workflow for a UDS service expecting multiple responses.
    /// <remark>
    /// 1) Warning: The order of the parameters has changed in PCAN-UDS 2.0 API.
    /// 2) The function waits for a transmit confirmation then for N message responses.
    /// Even if the SuppressPositiveResponseMessage flag is set, the function will still wait
    /// for eventual Negative Responses.
    /// </remark>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="msg_request">sent uds_msg message</param>
    /// <param name="max_msg_count">Length of the buffer array (max. messages that can be received)</param>
    /// <param name="wait_until_timeout">if <code>FALSE</code> the function is interrupted if out_msg_count reaches max_msg_count.</param>
    /// <param name="out_msg_responses">Buffer must be an array of 'max_msg_count' entries (must have at least
    /// a size of max_msg_count * sizeof(uds_msg) bytes</param>
    /// <param name="out_msg_count">Actual number of messages read</param>
    /// <param name="out_msg_request_confirmation">A uds_msg structure buffer to store the PUDS request confirmation</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success,
    /// PUDS_ERROR_OVERFLOW indicates success but buffer was too small to hold all responses.</returns>
    /// </summary>
    class function WaitForServiceFunctional_2013(channel: cantp_handle;
      msg_request: Puds_msg; max_msg_count: UInt32; wait_until_timeout: Boolean;
      out_msg_responses: Puds_msg; var out_msg_count: UInt32;
      out_msg_request_confirmation: Puds_msg): uds_status overload;

    /// ////////////////////////////////////////////////
    // PCAN-UDS API: UDS Service function declarations
    /// ////////////////////////////////////////////////

    /// <summary>
    /// The DiagnosticSessionControl service is used to enable different diagnostic sessions in the server.
    /// </summary>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="request_config">Request configuration</param>
    /// <param name="out_msg_request">(out) request message created and sent by the function</param>
    /// <param name="session_type">Subfunction parameter: type of the session (see PUDS_SVC_PARAM_DSC_xxx)</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function SvcDiagnosticSessionControl_2013(channel: cantp_handle;
      request_config: uds_msgconfig; var out_msg_request: uds_msg;
      session_type: uds_svc_param_dsc): uds_status;

    /// <summary>
    /// The ECUReset service is used by the client to request a server reset.
    /// </summary>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="request_config">Request configuration</param>
    /// <param name="out_msg_request">(out) request message created and sent by the function</param>
    /// <param name="reset_type">Subfunction parameter: type of Reset (see PUDS_SVC_PARAM_ER_xxx)</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function SvcECUReset_2013(channel: cantp_handle;
      request_config: uds_msgconfig; var out_msg_request: uds_msg;
      reset_type: uds_svc_param_er): uds_status;

    // ISO-14229-1:2013 §9.4.2.2 p.49

{$IF Not Defined(PUDS_API_COMPATIBILITY_ISO_2006)}

    /// <summary>
    /// Request Seed
    /// </summary>
  const
    PUDS_SVC_PARAM_SA_RSD_1: Byte = $1;

    /// <summary>
    /// Request Seed
    /// </summary>
  const
    PUDS_SVC_PARAM_SA_RSD_3: Byte = $3;

    /// <summary>
    /// Request Seed
    /// </summary>
  const
    PUDS_SVC_PARAM_SA_RSD_5: Byte = $5;

    /// <summary>
    /// Request Seed (odd numbers)
    /// </summary>
  const
    PUDS_SVC_PARAM_SA_RSD_MIN: Byte = $7;

    /// <summary>
    /// Request Seed (odd numbers)
    /// </summary>
  const
    PUDS_SVC_PARAM_SA_RSD_MAX: Byte = $5F;

    /// <summary>
    /// Send Key
    /// </summary>
  const
    PUDS_SVC_PARAM_SA_SK_2: Byte = $2;

    /// <summary>
    /// Send Key
    /// </summary>
  const
    PUDS_SVC_PARAM_SA_SK_4: Byte = $4;

    /// <summary>
    /// Send Key
    /// </summary>
  const
    PUDS_SVC_PARAM_SA_SK_6: Byte = $6;

    /// <summary>
    /// Send Key (even numbers)
    /// </summary>
  const
    PUDS_SVC_PARAM_SA_SK_MIN: Byte = $8;

    /// <summary>
    /// Send Key (even numbers)
    /// </summary>
  const
    PUDS_SVC_PARAM_SA_SK_MAX: Byte = $60;
{$ENDIF}
    /// <summary>
    /// SecurityAccess service provides a means to access data and/or diagnostic services which have
    /// restricted access for security, emissions or safety reasons.
    /// </summary>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="request_config">Request configuration</param>
    /// <param name="out_msg_request">(out) request message created and sent by the function</param>
    /// <param name="security_access_type">Subfunction parameter: type of SecurityAccess (see PUDS_SVC_PARAM_SA_xxx)</param>
    /// <param name="security_access_data">If Requesting Seed, buffer is the optional data to transmit to a server (like identification).
    /// If Sending Key, data holds the value generated by the security algorithm corresponding to a specific "seed" value</param>
    /// <param name="security_access_data_size">Size in bytes of the buffer</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function SvcSecurityAccess_2013(channel: cantp_handle;
      request_config: uds_msgconfig; var out_msg_request: uds_msg;
      security_access_type: Byte; security_access_data: PByte;
      security_access_data_size: UInt32): uds_status; overload;

    /// <summary>
    /// SecurityAccess service provides a means to access data and/or diagnostic services which have
    /// restricted access for security, emissions or safety reasons.
    /// </summary>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="request_config">Request configuration</param>
    /// <param name="out_msg_request">(out) request message created and sent by the function</param>
    /// <param name="security_access_type">Subfunction parameter: type of SecurityAccess (see PUDS_SVC_PARAM_SA_xxx)</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function SvcSecurityAccess_2013(channel: cantp_handle;
      request_config: uds_msgconfig; var out_msg_request: uds_msg;
      security_access_type: Byte): uds_status; overload;

    // ISO-14229-1:2013 §9.5.2.2 p.54

{$IF Not Defined(PUDS_API_COMPATIBILITY_ISO_2006)}

    /// <summary>
    /// Application (01b)
    /// </summary>
  const
    PUDS_SVC_PARAM_CC_FLAG_APPL: Byte = $1;

    /// <summary>
    /// NetworkManagement (10b)
    /// </summary>
  const
    PUDS_SVC_PARAM_CC_FLAG_NWM: Byte = $2;

    /// <summary>
    /// Disable/Enable specified communicationType (see Flags APPL/NMW)
    /// in the receiving node and all connected networks
    /// </summary>
  const
    PUDS_SVC_PARAM_CC_FLAG_DESCTIRNCN: Byte = $0;

    /// <summary>
    /// Disable/Enable network which request is received on
    /// </summary>
  const
    PUDS_SVC_PARAM_CC_FLAG_DENWRIRO: Byte = $F0;

    /// <summary>
    /// Disable/Enable specific network identified by network number (minimum value)
    /// </summary>
  const
    PUDS_SVC_PARAM_CC_FLAG_DESNIBNN_MIN: Byte = $10;

    /// <summary>
    /// Disable/Enable specific network identified by network number (maximum value)
    /// </summary>
  const
    PUDS_SVC_PARAM_CC_FLAG_DESNIBNN_MAX: Byte = $E0;

    /// <summary>
    /// Mask for DESNIBNN bits
    /// </summary>
  const
    PUDS_SVC_PARAM_CC_FLAG_DESNIBNN_MASK: Byte = $F0;

{$ENDIF}
    /// <summary>
    /// CommunicationControl service's purpose is to switch on/off the transmission
    /// and/or the reception of certain messages of (a) server(s).
    /// </summary>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="request_config">Request configuration</param>
    /// <param name="out_msg_request">(out) request message created and sent by the function</param>
    /// <param name="control_type">Subfunction parameter: type of CommunicationControl (see PUDS_SVC_PARAM_CC_xxx)</param>
    /// <param name="communication_type">a bit-code value to reference the kind of communication to be controlled,
    /// See PUDS_SVC_PARAM_CC_FLAG_xxx flags and ISO_14229-1:2013 §B.1 p.333 for bit-encoding</param>
    /// <param name="node_identification_number">Identify a node on a sub-network (only used with
    /// PUDS_SVC_PARAM_CC_ERXDTXWEAI or PUDS_SVC_PARAM_CC_ERXTXWEAI control type)</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function SvcCommunicationControl_2013(channel: cantp_handle;
      request_config: uds_msgconfig; var out_msg_request: uds_msg;
      control_type: uds_svc_param_cc; communication_type: Byte;
      node_identification_number: UInt16): uds_status; overload;

    /// <summary>
    /// CommunicationControl service's purpose is to switch on/off the transmission
    /// and/or the reception of certain messages of (a) server(s).
    /// </summary>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="request_config">Request configuration</param>
    /// <param name="out_msg_request">(out) request message created and sent by the function</param>
    /// <param name="control_type">Subfunction parameter: type of CommunicationControl (see PUDS_SVC_PARAM_CC_xxx)</param>
    /// <param name="communication_type">a bit-code value to reference the kind of communication to be controlled,
    /// See PUDS_SVC_PARAM_CC_FLAG_xxx flags and ISO_14229-1:2013 §B.1 p.333 for bit-encoding</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function SvcCommunicationControl_2013(channel: cantp_handle;
      request_config: uds_msgconfig; var out_msg_request: uds_msg;
      control_type: uds_svc_param_cc; communication_type: Byte)
      : uds_status; overload;

    /// <summary>
    /// TesterPresent service indicates to a server (or servers) that a client is still connected
    /// to the vehicle and that certain diagnostic services and/or communications
    /// that have been previously activated are to remain active.
    /// </summary>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="request_config">Request configuration</param>
    /// <param name="out_msg_request">(out) request message created and sent by the function</param>
    /// <param name="testerpresent_type">No Subfunction parameter by default (PUDS_SVC_PARAM_TP_ZSUBF)</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function SvcTesterPresent_2013(channel: cantp_handle;
      request_config: uds_msgconfig; var out_msg_request: uds_msg;
      testerpresent_type: uds_svc_param_tp): uds_status; overload;

    /// <summary>
    /// TesterPresent service indicates to a server (or servers) that a client is still connected
    /// to the vehicle and that certain diagnostic services and/or communications
    /// that have been previously activated are to remain active.
    /// No Subfunction parameter (PUDS_SVC_PARAM_TP_ZSUBF)
    /// </summary>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="request_config">Request configuration</param>
    /// <param name="out_msg_request">(out) request message created and sent by the function</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function SvcTesterPresent_2013(channel: cantp_handle;
      request_config: uds_msgconfig; var out_msg_request: uds_msg)
      : uds_status; overload;

    // ISO-14229-1:2013 §9.8 p.66

    /// <summary>
    /// SecuredDataTransmission(2013) service's purpose is to transmit data that is protected
    /// against attacks from third parties, which could endanger data security.
    /// </summary>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="request_config">Request configuration (PUDS_MSGTYPE_FLAG_NO_POSITIVE_RESPONSE is ignored)</param>
    /// <param name="out_msg_request">(out) request message created and sent by the function</param>
    /// <param name="security_data_request_record">buffer containing the data as processed by the Security Sub-Layer (See ISO-15764)</param>
    /// <param name="security_data_request_record_size">Size in bytes of the buffer</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function SvcSecuredDataTransmission_2013(channel: cantp_handle;
      request_config: uds_msgconfig; var out_msg_request: uds_msg;
      security_data_request_record: PByte;
      security_data_request_record_size: UInt32): uds_status;

    // ISO-14229-1:2020 §16.2 p.358
    /// <summary>
    /// The messsage is a request message.
    /// </summary>
  const
    PUDS_SVC_PARAM_APAR_REQUEST_MSG_FLAG: Byte = $1;

    /// <summary>
    /// A pre - established key is used.
    /// </summary>
  const
    PUDS_SVC_PARAM_APAR_PRE_ESTABLISHED_KEY_FLAG: Byte = $8;

    /// <summary>
    /// Message is encrypted.
    /// </summary>
  const
    PUDS_SVC_PARAM_APAR_ENCRYPTED_MSG_FLAG: Byte = $10;

    /// <summary>
    /// Message is signed.
    /// </summary>
  const
    PUDS_SVC_PARAM_APAR_SIGNED_MSG_FLAG: Byte = $20;

    /// <summary>
    /// Signature on the response is requested.
    /// </summary>
  const
    PUDS_SVC_PARAM_APAR_REQUEST_RESPONSE_SIGNATURE_FLAG: Byte = $40;

    /// <summary>
    /// SecuredDataTransmission(2020) service's purpose is to transmit data that is protected
    /// against attacks from third parties, which could endanger data security.
    /// </summary>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="request_config">Request configuration (PUDS_MSGTYPE_FLAG_NO_POSITIVE_RESPONSE is ignored)</param>
    /// <param name="out_msg_request">(out) request message created and sent by the function</param>
    /// <param name="administrative_parameter">Security features used in the message (see PUDS_SVC_PARAM_APAR* definitions)</param>
    /// <param name="signature_encryption_calculation">Signature or encryption algorithm identifier</param>
    /// <param name="anti_replay_counter">Anti-replay counter value</param>
    /// <param name="internal_service_identifier">Internal message service request identifier</param>
    /// <param name="service_specific_parameters">Buffer that contains internal message service request data</param>
    /// <param name="service_specific_parameters_size">Internal message service request data size (in bytes)</param>
    /// <param name="signature_mac">Buffer that contains signature used to verify the message</param>
    /// <param name="signature_size">Size in bytes of the signature</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function SvcSecuredDataTransmission_2020(channel: cantp_handle;
      request_config: uds_msgconfig; var out_msg_request: uds_msg;
      administrative_parameter: UInt16; signature_encryption_calculation: Byte;
      anti_replay_counter: UInt16; internal_service_identifier: Byte;
      service_specific_parameters: PByte;
      service_specific_parameters_size: UInt32; signature_mac: PByte;
      signature_size: UInt16): uds_status;

    /// <summary>
    /// ControlDTCSetting service shall be used by a client to stop or resume the setting of
    /// diagnostic trouble codes (DTCs) in the server(s).
    /// </summary>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="request_config">Request configuration</param>
    /// <param name="out_msg_request">(out) request message created and sent by the function</param>
    /// <param name="dtc_setting_type">Subfunction parameter (see PUDS_SVC_PARAM_CDTCS_xxx)</param>
    /// <param name="dtc_setting_control_option_record">This parameter record is user-optional and transmits data to a server when controlling the DTC setting.
    /// It can contain a list of DTCs to be turned on or off.</param>
    /// <param name="dtc_setting_control_option_record_size">Size in bytes of the buffer</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function SvcControlDTCSetting_2013(channel: cantp_handle;
      request_config: uds_msgconfig; var out_msg_request: uds_msg;
      dtc_setting_type: uds_svc_param_cdtcs;
      dtc_setting_control_option_record: PByte;
      dtc_setting_control_option_record_size: UInt32): uds_status; overload;

    /// <summary>
    /// ControlDTCSetting service shall be used by a client to stop or resume the setting of
    /// diagnostic trouble codes (DTCs) in the server(s).
    /// </summary>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="request_config">Request configuration</param>
    /// <param name="out_msg_request">(out) request message created and sent by the function</param>
    /// <param name="dtc_setting_type">Subfunction parameter (see PUDS_SVC_PARAM_CDTCS_xxx)</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function SvcControlDTCSetting_2013(channel: cantp_handle;
      request_config: uds_msgconfig; var out_msg_request: uds_msg;
      dtc_setting_type: uds_svc_param_cdtcs): uds_status; overload;

    // ISO-14229-1:2013 §9.10.2.2.1 p.78
{$IF Not Defined(PUDS_API_COMPATIBILITY_ISO_2006)}

    /// <summary>
    /// Expected size of event type record for ROE_STPROE
    /// </summary>
  const
    PUDS_SVC_PARAM_ROE_STPROE_LEN: Byte = 0;

    /// <summary>
    /// Expected size of event type record for ROE_ONDTCS
    /// </summary>
  const
    PUDS_SVC_PARAM_ROE_ONDTCS_LEN: Byte = 1;

    /// <summary>
    /// Expected size of event type record for ROE_OTI
    /// </summary>
  const
    PUDS_SVC_PARAM_ROE_OTI_LEN: Byte = 1;

    /// <summary>
    /// Expected size of event type record for ROE_OCODID
    /// </summary>
  const
    PUDS_SVC_PARAM_ROE_OCODID_LEN: Byte = 2;

    /// <summary>
    /// Expected size of event type record for ROE_RAE
    /// </summary>
  const
    PUDS_SVC_PARAM_ROE_RAE_LEN: Byte = 0;

    /// <summary>
    /// Expected size of event type record for ROE_STRTROE
    /// </summary>
  const
    PUDS_SVC_PARAM_ROE_STRTROE_LEN: Byte = 0;

    /// <summary>
    /// Expected size of event type record for ROE_CLRROE
    /// </summary>
  const
    PUDS_SVC_PARAM_ROE_CLRROE_LEN: Byte = 0;

    /// <summary>
    /// Expected size of event type record for ROE_OCOV
    /// </summary>
  const
    PUDS_SVC_PARAM_ROE_OCOV_LEN: Byte = 10;

    /// <summary>
    /// Expected size of event type record for ROE_RMRDOSC
    /// </summary>
  const
    PUDS_SVC_PARAM_ROE_RMRDOSC_LEN: Byte = 1;

{$ENDIF}
{$IF Not Defined(PUDS_API_COMPATIBILITY_ISO_2006)}

    /// <summary>
    /// Infinite Time To Response (eventWindowTime parameter)
    /// </summary>
  const
    PUDS_SVC_PARAM_ROE_EWT_ITTR: Byte = $2;
{$ENDIF}
    /// <summary>
    /// Short event window time (eventWindowTime parameter)
    /// </summary>
  const
    PUDS_SVC_PARAM_ROE_EWT_SEWT: Byte = $3;
    /// <summary>
    /// Medium event window time (eventWindowTime parameter)
    /// </summary>
  const
    PUDS_SVC_PARAM_ROE_EWT_MEWT: Byte = $4;
    /// <summary>
    /// Long event window time (eventWindowTime parameter)
    /// </summary>
  const
    PUDS_SVC_PARAM_ROE_EWT_LEWT: Byte = $5;
    /// <summary>
    /// Power window time (eventWindowTime parameter)
    /// </summary>
  const
    PUDS_SVC_PARAM_ROE_EWT_PWT: Byte = $6;
    /// <summary>
    /// Ignition window time (eventWindowTime parameter)
    /// </summary>
  const
    PUDS_SVC_PARAM_ROE_EWT_IWT: Byte = $7;
    /// <summary>
    /// Manufacturer trigger event window time (eventWindowTime parameter)
    /// </summary>
  const
    PUDS_SVC_PARAM_ROE_EWT_MTEWT: Byte = $8;

    /// <summary>
    /// Slow rate (onTimerInterrupt parameter)
    /// </summary>
  const
    PUDS_SVC_PARAM_ROE_OTI_SLOW_RATE: Byte = $1;
    /// <summary>
    /// Medium rate (onTimerInterrupt parameter)
    /// </summary>
  const
    PUDS_SVC_PARAM_ROE_OTI_MEDIUM_RATE: Byte = $2;
    /// <summary>
    /// Fast rate (onTimerInterrupt parameter)
    /// </summary>
  const
    PUDS_SVC_PARAM_ROE_OTI_FAST_RATE: Byte = $3;

    /// <summary>
    /// The ResponseOnEvent service requests a server to
    /// start or stop transmission of responses on a specified event.
    /// </summary>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="request_config">Request configuration</param>
    /// <param name="out_msg_request">(out) request message created and sent by the function</param>
    /// <param name="event_type">Subfunction parameter: event type (see PUDS_SVC_PARAM_ROE_xxx)</param>
    /// <param name="store_event">Storage State (TRUE = Store Event, FALSE = Do Not Store Event)</param>
    /// <param name="event_window_time">Specify a window for the event logic to be active in the server (see PUDS_SVC_PARAM_ROE_EWT_ITTR)</param>
    /// <param name="event_type_record">Additional parameters for the specified event type</param>
    /// <param name="event_type_record_size">Size in bytes of the event type record (see PUDS_SVC_PARAM_ROE_xxx_LEN)</param>
    /// <param name="service_to_respond_to_record">Service parameters, with first byte as service Id (see PUDS_SVC_PARAM_ROE_STRT_SI_xxx)</param>
    /// <param name="service_to_respond_to_record_size">Size in bytes of the service to respond to record</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function SvcResponseOnEvent_2013(channel: cantp_handle;
      request_config: uds_msgconfig; var out_msg_request: uds_msg;
      event_type: uds_svc_param_roe; store_event: Boolean;
      event_window_time: Byte; event_type_record: PByte;
      event_type_record_size: UInt32; service_to_respond_to_record: PByte;
      service_to_respond_to_record_size: UInt32): uds_status; overload;

    /// <summary>
    /// The ResponseOnEvent service requests a server to
    /// start or stop transmission of responses on a specified event.
    /// </summary>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="request_config">Request configuration</param>
    /// <param name="out_msg_request">(out) request message created and sent by the function</param>
    /// <param name="event_type">Subfunction parameter: event type (see PUDS_SVC_PARAM_ROE_xxx)</param>
    /// <param name="store_event">Storage State (TRUE = Store Event, FALSE = Do Not Store Event)</param>
    /// <param name="event_window_time">Specify a window for the event logic to be active in the server (see PUDS_SVC_PARAM_ROE_EWT_ITTR)</param>
    /// <param name="event_type_record">Additional parameters for the specified event type</param>
    /// <param name="event_type_record_size">Size in bytes of the event type record (see PUDS_SVC_PARAM_ROE_xxx_LEN)</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function SvcResponseOnEvent_2013(channel: cantp_handle;
      request_config: uds_msgconfig; var out_msg_request: uds_msg;
      event_type: uds_svc_param_roe; store_event: Boolean;
      event_window_time: Byte; event_type_record: PByte;
      event_type_record_size: UInt32): uds_status; overload;

    /// <summary>
    /// The ResponseOnEvent service requests a server to
    /// start or stop transmission of responses on a specified event.
    /// </summary>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="request_config">Request configuration</param>
    /// <param name="out_msg_request">(out) request message created and sent by the function</param>
    /// <param name="event_type">Subfunction parameter: event type (see PUDS_SVC_PARAM_ROE_xxx)</param>
    /// <param name="store_event">Storage State (TRUE = Store Event, FALSE = Do Not Store Event)</param>
    /// <param name="event_window_time">Specify a window for the event logic to be active in the server (see PUDS_SVC_PARAM_ROE_EWT_ITTR)</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function SvcResponseOnEvent_2013(channel: cantp_handle;
      request_config: uds_msgconfig; var out_msg_request: uds_msg;
      event_type: uds_svc_param_roe; store_event: Boolean;
      event_window_time: Byte): uds_status; overload;

    /// <summary>
    /// The LinkControl service is used to control the communication link baud rate
    /// between the client and the server(s) for the exchange of diagnostic data.
    /// </summary>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="request_config">Request configuration</param>
    /// <param name="out_msg_request">(out) request message created and sent by the function</param>
    /// <param name="link_control_type">Subfunction parameter: Link Control Type (see PUDS_SVC_PARAM_LC_xxx)</param>
    /// <param name="baudrate_identifier">defined baud rate identifier (see PUDS_SVC_PARAM_LC_BAUDRATE_xxx)</param>
    /// <param name="link_baudrate">used only with PUDS_SVC_PARAM_LC_VBTWSBR parameter:
    /// a three-byte value baud rate (baudrate High, Middle and Low bytes).</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success </returns>
    class function SvcLinkControl_2013(channel: cantp_handle;
      request_config: uds_msgconfig; var out_msg_request: uds_msg;
      link_control_type: uds_svc_param_lc;
      baudrate_identifier: uds_svc_param_lc_baudrate_identifier;
      link_baudrate: UInt32): uds_status; overload;

    /// <summary>
    /// The LinkControl service is used to control the communication link baud rate
    /// between the client and the server(s) for the exchange of diagnostic data.
    /// Use only if link_control_type parameter is not PUDS_SVC_PARAM_LC_VBTWSBR
    /// </summary>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="request_config">Request configuration</param>
    /// <param name="out_msg_request">(out) request message created and sent by the function</param>
    /// <param name="link_control_type">Subfunction parameter: Link Control Type (see PUDS_SVC_PARAM_LC_xxx)</param>
    /// <param name="baudrate_identifier">defined baud rate identifier (see PUDS_SVC_PARAM_LC_BAUDRATE_xxx)</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success </returns>
    class function SvcLinkControl_2013(channel: cantp_handle;
      request_config: uds_msgconfig; var out_msg_request: uds_msg;
      link_control_type: uds_svc_param_lc;
      baudrate_identifier: uds_svc_param_lc_baudrate_identifier)
      : uds_status; overload;

    // ISO-14229-1:2013 §10.2 p.106
    /// <summary>
    /// The ReadDataByIdentifier service allows the client to request data record values
    /// from the server identified by one or more dataIdentifiers.
    /// </summary>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="request_config">Request configuration (PUDS_MSGTYPE_FLAG_NO_POSITIVE_RESPONSE is ignored)</param>
    /// <param name="out_msg_request">(out) request message created and sent by the function</param>
    /// <param name="data_identifier">buffer containing a list of two-byte Data Identifiers (see PUDS_SVC_PARAM_DI_xxx)</param>
    /// <param name="data_identifier_length">Number of elements in the buffer (size in uint16_t of the buffer)</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function SvcReadDataByIdentifier_2013(channel: cantp_handle;
      request_config: uds_msgconfig; var out_msg_request: uds_msg;
      data_identifier: Puds_svc_param_di; data_identifier_length: UInt32)
      : uds_status;

    // ISO-14229-1:2013 §10.3 p.113
    /// <summary>
    /// The ReadMemoryByAddress service allows the client to request memory data from the server
    /// via a provided starting address and to specify the size of memory to be read.
    /// </summary>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="request_config">Request configuration (PUDS_MSGTYPE_FLAG_NO_POSITIVE_RESPONSE is ignored)</param>
    /// <param name="out_msg_request">(out) request message created and sent by the function</param>
    /// <param name="memory_address_buffer">starting address buffer of server memory from which data is to be retrieved</param>
    /// <param name="memory_address_size">Size in bytes of the memory_address_buffer (max.: 0xF)</param>
    /// <param name="memory_size_buffer">number of bytes to be read starting at the address specified by memory_address_buffer</param>
    /// <param name="memory_size_size">Size in bytes of the memory_size_buffer (max.: 0xF)</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function SvcReadMemoryByAddress_2013(channel: cantp_handle;
      request_config: uds_msgconfig; var out_msg_request: uds_msg;
      memory_address_buffer: PByte; memory_address_size: Byte;
      memory_size_buffer: PByte; memory_size_size: Byte): uds_status;

    // ISO-14229-1:2013 §10.4 p.119
    /// <summary>
    /// The ReadScalingDataByIdentifier service allows the client to request
    /// scaling data record information from the server identified by a dataIdentifier.
    /// </summary>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="request_config">Request configuration (PUDS_MSGTYPE_FLAG_NO_POSITIVE_RESPONSE is ignored)</param>
    /// <param name="out_msg_request">(out) request message created and sent by the function</param>
    /// <param name="data_identifier">a two-byte Data Identifier (see PUDS_SVC_PARAM_DI_xxx)</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function SvcReadScalingDataByIdentifier_2013(channel: cantp_handle;
      request_config: uds_msgconfig; var out_msg_request: uds_msg;
      data_identifier: uds_svc_param_di): uds_status;

    /// <summary>
    /// The ReadDataByPeriodicIdentifier service allows the client to request the periodic transmission
    /// of data record values from the server identified by one or more periodicDataIdentifiers.
    /// </summary>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="request_config">Request configuration (PUDS_MSGTYPE_FLAG_NO_POSITIVE_RESPONSE is ignored)</param>
    /// <param name="out_msg_request">(out) request message created and sent by the function</param>
    /// <param name="transmission_mode">transmission rate code (see PUDS_SVC_PARAM_RDBPI_xxx)</param>
    /// <param name="periodic_data_identifier">buffer containing a list of Periodic Data Identifiers</param>
    /// <param name="periodic_data_identifier_size">Number of elements in the buffer (size in bytes of the buffer)</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function SvcReadDataByPeriodicIdentifier_2013(channel: cantp_handle;
      request_config: uds_msgconfig; var out_msg_request: uds_msg;
      transmission_mode: uds_svc_param_rdbpi; periodic_data_identifier: PByte;
      periodic_data_identifier_size: UInt32): uds_status;

    /// <summary>
    /// The DynamicallyDefineDataIdentifier service allows the client to dynamically define
    /// in a server a data identifier that can be read via the ReadDataByIdentifier service at a later time.
    /// The Define By Identifier subfunction specifies that definition of the dynamic data
    /// identifier shall occur via a data identifier reference.
    /// </summary>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="request_config">Request configuration</param>
    /// <param name="out_msg_request">(out) request message created and sent by the function</param>
    /// <param name="dynamically_defined_data_identifier">a two-byte Data Identifier (see PUDS_SVC_PARAM_DI_xxx)</param>
    /// <param name="source_data_identifier">buffer containing the sources of information to be included into the dynamic data record</param>
    /// <param name="memory_size">buffer containing the total numbers of bytes from the source data record address</param>
    /// <param name="position_in_source_data_record">buffer containing the starting byte positions of the excerpt of the source data record</param>
    /// <param name="number_of_elements">Number of elements in SourceDataIdentifier/position_in_source_data_record/memory_size triplet.</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function SvcDynamicallyDefineDataIdentifierDBID_2013
      (channel: cantp_handle; request_config: uds_msgconfig;
      var out_msg_request: uds_msg; dynamically_defined_data_identifier
      : uds_svc_param_di; source_data_identifier: PUInt16; memory_size: PByte;
      position_in_source_data_record: PByte; number_of_elements: UInt32)
      : uds_status;

    /// <summary>
    /// The DynamicallyDefineDataIdentifier service allows the client to dynamically define
    /// in a server a data identifier that can be read via the ReadDataByIdentifier service at a later time.
    /// The Define By Memory Address subfunction specifies that definition of the dynamic data
    /// identifier shall occur via an address reference.
    /// </summary>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="request_config">Request configuration</param>
    /// <param name="out_msg_request">(out) request message created and sent by the function</param>
    /// <param name="dynamically_defined_data_identifier">a two-byte Data Identifier (see PUDS_SVC_PARAM_DI_xxx)</param>
    /// <param name="memory_address_size">Size in bytes of the memory address items in the memory_address_buffer (max.: 0xF)</param>
    /// <param name="memory_size_size">Size in bytes of the memory size items in the memory_size_buffer (max.: 0xF)</param>
    /// <param name="memory_address_buffer">buffer containing the memory address buffer,
    /// must be an array of 'number_of_elements' items whose size is 'memory_address_size'
    /// (size is 'number_of_elements * memory_address_size' bytes)</param>
    /// <param name="memory_size_buffer">buffer containing the memory size buffer,
    /// must be an array of 'number_of_elements' items whose size is 'memory_size_size'
    /// (size is 'number_of_elements * memory_size_size' bytes)</param>
    /// <param name="number_of_elements">Number of elements in memory_address_buffer/memory_size_buffer couple.</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function SvcDynamicallyDefineDataIdentifierDBMA_2013
      (channel: cantp_handle; request_config: uds_msgconfig;
      var out_msg_request: uds_msg; dynamically_defined_data_identifier
      : uds_svc_param_di; memory_address_size: Byte; memory_size_size: Byte;
      memory_address_buffer: PByte; memory_size_buffer: PByte;
      number_of_elements: UInt32): uds_status;

    /// <summary>
    /// The Clear Dynamically Defined Data Identifier subfunction shall be used to clear
    /// the specified dynamic data identifier.
    /// </summary>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="request_config">Request configuration</param>
    /// <param name="out_msg_request">(out) request message created and sent by the function</param>
    /// <param name="dynamically_defined_data_identifier">a two-byte Data Identifier (see PUDS_SVC_PARAM_DI_xxx)</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function SvcDynamicallyDefineDataIdentifierCDDDI_2013
      (channel: cantp_handle; request_config: uds_msgconfig;
      var out_msg_request: uds_msg; dynamically_defined_data_identifier
      : uds_svc_param_di): uds_status;

    /// <summary>
    /// The Clear All Dynamically Defined Data Identifier function shall be used to clear
    /// all dynamic data identifier declared in the server.
    /// </summary>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="request_config">Request configuration</param>
    /// <param name="out_msg_request">(out) request message created and sent by the function</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function SvcDynamicallyDefineDataIdentifierClearAllDDDI_2013
      (channel: cantp_handle; request_config: uds_msgconfig;
      var out_msg_request: uds_msg): uds_status;

    // ISO-14229-1:2013 §10.7 p.162
    /// <summary>
    /// The WriteDataByIdentifier service allows the client to write information into the server at an internal location
    /// specified by the provided data identifier.
    /// </summary>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="request_config">Request configuration (PUDS_MSGTYPE_FLAG_NO_POSITIVE_RESPONSE is ignored)</param>
    /// <param name="out_msg_request">(out) request message created and sent by the function</param>
    /// <param name="data_identifier">a two-byte Data Identifier (see PUDS_SVC_PARAM_DI_xxx)</param>
    /// <param name="data_record">buffer containing the data to write</param>
    /// <param name="data_record_size">Size in bytes of the buffer</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function SvcWriteDataByIdentifier_2013(channel: cantp_handle;
      request_config: uds_msgconfig; var out_msg_request: uds_msg;
      data_identifier: uds_svc_param_di; data_record: PByte;
      data_record_size: UInt32): uds_status;

    // ISO-14229-1:2013 §10.8 p.167
    /// <summary>
    /// The WriteMemoryByAddress service allows the client to write
    /// information into the server at one or more contiguous memory locations.
    /// </summary>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="request_config">Request configuration (PUDS_MSGTYPE_FLAG_NO_POSITIVE_RESPONSE is ignored)</param>
    /// <param name="out_msg_request">(out) request message created and sent by the function</param>
    /// <param name="memory_address_buffer">Starting address buffer of server memory to which data is to be written</param>
    /// <param name="memory_address_size">Size in bytes of the memory_address_buffer (max.: 0xF)</param>
    /// <param name="memory_size_buffer">number of bytes to be written starting at the address specified by memory_address_buffer</param>
    /// <param name="memory_size_size">Size in bytes of the memory_size_buffer (max.: 0xF)</param>
    /// <param name="data_record">buffer containing the data to write</param>
    /// <param name="data_record_size">Size in bytes of the buffer</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function SvcWriteMemoryByAddress_2013(channel: cantp_handle;
      request_config: uds_msgconfig; var out_msg_request: uds_msg;
      memory_address_buffer: PByte; memory_address_size: Byte;
      memory_size_buffer: PByte; memory_size_size: Byte; data_record: PByte;
      data_record_size: UInt32): uds_status;

    // ISO-14229-1:2013 §11.2 p.175

{$IF Not Defined(PUDS_API_COMPATIBILITY_ISO_2006)}

    /// <summary>
    /// Emissions-related systems group of DTCs
    /// </summary>
  const
    PUDS_SVC_PARAM_CDI_ERS: UInt32 = $0;

    /// <summary>
    /// All Groups of DTCs
    /// </summary>
  const
    PUDS_SVC_PARAM_CDI_AGDTC: UInt32 = $FFFFFF;
{$ENDIF}
    /// <summary>
    /// The ClearDiagnosticInformation service is used by the client to clear diagnostic information
    /// in one server's or multiple servers' memory.
    /// </summary>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="request_config">Request configuration (PUDS_MSGTYPE_FLAG_NO_POSITIVE_RESPONSE is ignored)</param>
    /// <param name="out_msg_request">(out) request message created and sent by the function</param>
    /// <param name="group_of_dtc">a three-byte value indicating the group of DTCs (e.g. powertrain, body, chassis)
    /// or the particular DTC to be cleared (see PUDS_SVC_PARAM_CDI_xxx)</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function SvcClearDiagnosticInformation_2013(channel: cantp_handle;
      request_config: uds_msgconfig; var out_msg_request: uds_msg;
      group_of_dtc: UInt32): uds_status;

    /// <summary>
    /// The ClearDiagnosticInformation service is used by the client to clear diagnostic information
    /// in one server's or multiple servers' memory.
    /// </summary>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="request_config">Request configuration (PUDS_MSGTYPE_FLAG_NO_POSITIVE_RESPONSE is ignored)</param>
    /// <param name="out_msg_request">(out) request message created and sent by the function</param>
    /// <param name="group_of_dtc">a three-byte value indicating the group of DTCs (e.g. powertrain, body, chassis)
    /// or the particular DTC to be cleared (see PUDS_SVC_PARAM_CDI_xxx)</param>
    /// <param name="memory_selection">User defined DTC memory</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function SvcClearDiagnosticInformation_2020(channel: cantp_handle;
      request_config: uds_msgconfig; var out_msg_request: uds_msg;
      group_of_dtc: UInt32; memory_selection: Byte): uds_status;

    /// <summary>
    /// This service allows a client to read the status of server-resident Diagnostic Trouble Code (DTC) information.
    /// Only reportNumberOfDTCByStatusMask, reportDTCByStatusMask, reportMirrorMemoryDTCByStatusMask,
    /// reportNumberOfMirrorMemoryDTCByStatusMask, reportNumberOfEmissionsRelatedOBDDTCByStatusMask,
    /// reportEmissionsRelatedOBDDTCByStatusMask Sub-functions are allowed.
    /// </summary>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="request_config">Request configuration</param>
    /// <param name="out_msg_request">(out) request message created and sent by the function</param>
    /// <param name="PUDS_SVC_PARAM_RDTCI_Type">Subfunction parameter: ReadDTCInformation type, use one of the following:
    /// PUDS_SVC_PARAM_RDTCI_RNODTCBSM, PUDS_SVC_PARAM_RDTCI_RDTCBSM,
    /// PUDS_SVC_PARAM_RDTCI_RMMDTCBSM, PUDS_SVC_PARAM_RDTCI_RNOMMDTCBSM,
    /// PUDS_SVC_PARAM_RDTCI_RNOOBDDTCBSM, PUDS_SVC_PARAM_RDTCI_ROBDDTCBSM</param>
    /// <param name="dtc_status_mask">Contains eight DTC status bit.</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function SvcReadDTCInformation_2013(channel: cantp_handle;
      request_config: uds_msgconfig; var out_msg_request: uds_msg;
      PUDS_SVC_PARAM_RDTCI_Type: uds_svc_param_rdtci; dtc_status_mask: Byte)
      : uds_status;

    /// <summary>
    /// This service allows a client to read the status of server-resident Diagnostic Trouble Code (DTC) information.
    /// The sub-function reportDTCSnapshotRecordByDTCNumber (PUDS_SVC_PARAM_RDTCI_RDTCSSBDTC) is implicit.
    /// </summary>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="request_config">Request configuration</param>
    /// <param name="out_msg_request">(out) request message created and sent by the function</param>
    /// <param name="dtc_mask">a unique identification number (three byte value) for a specific diagnostic trouble code</param>
    /// <param name="dtc_snapshot_record_number">the number of the specific DTCSnapshot data records</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function SvcReadDTCInformationRDTCSSBDTC_2013(channel: cantp_handle;
      request_config: uds_msgconfig; var out_msg_request: uds_msg;
      dtc_mask: UInt32; dtc_snapshot_record_number: Byte): uds_status;

    /// <summary>
    /// This service allows a client to read the status of server-resident Diagnostic Trouble Code (DTC) information.
    /// The sub-function reportDTCSnapshotByRecordNumber (PUDS_SVC_PARAM_RDTCI_RDTCSSBRN) is implicit.
    /// </summary>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="request_config">Request configuration</param>
    /// <param name="out_msg_request">(out) request message created and sent by the function</param>
    /// <param name="dtc_snapshot_record_number">the number of the specific DTCSnapshot data records</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function SvcReadDTCInformationRDTCSSBRN_2013(channel: cantp_handle;
      request_config: uds_msgconfig; var out_msg_request: uds_msg;
      dtc_snapshot_record_number: Byte): uds_status;

    /// <summary>
    /// This service allows a client to read the status of server-resident Diagnostic Trouble Code (DTC) information.
    /// Only reportDTCExtendedDataRecordByDTCNumber and reportMirrorMemoryDTCExtendedDataRecordByDTCNumber Sub-functions are allowed.
    /// </summary>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="request_config">Request configuration</param>
    /// <param name="out_msg_request">(out) request message created and sent by the function</param>
    /// <param name="PUDS_SVC_PARAM_RDTCI_Type">Subfunction parameter: ReadDTCInformation type, use one of the following:
    /// PUDS_SVC_PARAM_RDTCI_RDTCEDRBDN, PUDS_SVC_PARAM_RDTCI_RMMDEDRBDN</param>
    /// <param name="dtc_mask">a unique identification number (three byte value) for a specific diagnostic trouble code</param>
    /// <param name="dtc_extended_data_record_number">the number of the specific DTCExtendedData record requested.</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function SvcReadDTCInformationReportExtended_2013
      (channel: cantp_handle; request_config: uds_msgconfig;
      var out_msg_request: uds_msg;
      PUDS_SVC_PARAM_RDTCI_Type: uds_svc_param_rdtci; dtc_mask: UInt32;
      dtc_extended_data_record_number: Byte): uds_status;

    /// <summary>
    /// This service allows a client to read the status of server-resident Diagnostic Trouble Code (DTC) information.
    /// Only reportNumberOfDTCBySeverityMaskRecord and reportDTCSeverityInformation Sub-functions are allowed.
    /// </summary>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="request_config">Request configuration</param>
    /// <param name="out_msg_request">(out) request message created and sent by the function</param>
    /// <param name="PUDS_SVC_PARAM_RDTCI_Type">Subfunction parameter: ReadDTCInformation type, use one of the following:
    /// PUDS_SVC_PARAM_RDTCI_RNODTCBSMR, PUDS_SVC_PARAM_RDTCI_RDTCBSMR</param>
    /// <param name="dtc_severity_mask">a mask of eight (8) DTC severity bits (see PUDS_SVC_PARAM_RDTCI_DTCSVM_xxx)</param>
    /// <param name="dtc_status_mask">a mask of eight (8) DTC status bits</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function SvcReadDTCInformationReportSeverity_2013
      (channel: cantp_handle; request_config: uds_msgconfig;
      var out_msg_request: uds_msg;
      PUDS_SVC_PARAM_RDTCI_Type: uds_svc_param_rdtci; dtc_severity_mask: Byte;
      dtc_status_mask: Byte): uds_status;

    /// <summary>
    /// This service allows a client to read the status of server-resident Diagnostic Trouble Code (DTC) information.
    /// The sub-function reportSeverityInformationOfDTC (PUDS_SVC_PARAM_RDTCI_RSIODTC) is implicit.
    /// </summary>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="request_config">Request configuration</param>
    /// <param name="out_msg_request">(out) request message created and sent by the function</param>
    /// <param name="dtc_mask">a unique identification number for a specific diagnostic trouble code</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function SvcReadDTCInformationRSIODTC_2013(channel: cantp_handle;
      request_config: uds_msgconfig; var out_msg_request: uds_msg;
      dtc_mask: UInt32): uds_status;

    /// <summary>
    /// This service allows a client to read the status of server-resident Diagnostic Trouble Code _2013(DTC) information.
    /// Only reportSupportedDTC, reportFirstTestFailedDTC, reportFirstConfirmedDTC, reportMostRecentTestFailedDTC,
    /// reportMostRecentConfirmedDTC, reportDTCFaultDetectionCounter, reportDTCWithPermanentStatus,
    /// and reportDTCSnapshotIdentification Sub-functions are allowed.
    /// </summary>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="request_config">Request configuration</param>
    /// <param name="out_msg_request">(out) request message created and sent by the function</param>
    /// <param name="PUDS_SVC_PARAM_RDTCI_Type">Subfunction parameter: ReadDTCInformation type, use one of the following:
    /// PUDS_SVC_PARAM_RDTCI_RFTFDTC, PUDS_SVC_PARAM_RDTCI_RFCDTC,
    /// PUDS_SVC_PARAM_RDTCI_RMRTFDTC, PUDS_SVC_PARAM_RDTCI_RMRCDTC,
    /// PUDS_SVC_PARAM_RDTCI_RSUPDTC, PUDS_SVC_PARAM_RDTCI_RDTCWPS,
    /// PUDS_SVC_PARAM_RDTCI_RDTCSSI, PUDS_SVC_PARAM_RDTCI_RDTCFDC</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function SvcReadDTCInformationNoParam_2013(channel: cantp_handle;
      request_config: uds_msgconfig; var out_msg_request: uds_msg;
      PUDS_SVC_PARAM_RDTCI_Type: uds_svc_param_rdtci): uds_status;

    /// <summary>
    /// This service allows a client to read the status of server-resident Diagnostic Trouble Code (DTC) information.
    /// The sub-function reportDTCExtDataRecordByRecordNumber (PUDS_SVC_PARAM_RDTCI_RDTCEDBR) is implicit.
    /// </summary>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="request_config">Request configuration</param>
    /// <param name="out_msg_request">(out) request message created and sent by the function</param>
    /// <param name="dtc_extended_data_record_number">DTC extended data record number</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function SvcReadDTCInformationRDTCEDBR_2013(channel: cantp_handle;
      request_config: uds_msgconfig; var out_msg_request: uds_msg;
      dtc_extended_data_record_number: Byte): uds_status;

    /// <summary>
    /// This service allows a client to read the status of server-resident Diagnostic Trouble Code (DTC) information.
    /// The sub-function reportUserDefMemoryDTCByStatusMask (PUDS_SVC_PARAM_RDTCI_RUDMDTCBSM) is implicit.
    /// </summary>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="request_config">Request configuration</param>
    /// <param name="out_msg_request">(out) request message created and sent by the function</param>
    /// <param name="dtc_status_mask">a mask of eight (8) DTC status bits</param>
    /// <param name="memory_selection">Memory selection</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function SvcReadDTCInformationRUDMDTCBSM_2013(channel: cantp_handle;
      request_config: uds_msgconfig; var out_msg_request: uds_msg;
      dtc_status_mask: Byte; memory_selection: Byte): uds_status;

    /// <summary>
    /// This service allows a client to read the status of server-resident Diagnostic Trouble Code (DTC) information.
    /// The sub-function reportUserDefMemoryDTCSnapshotRecordByDTCNumber (PUDS_SVC_PARAM_RDTCI_RUDMDTCSSBDTC) is implicit.
    /// </summary>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="request_config">Request configuration</param>
    /// <param name="out_msg_request">(out) request message created and sent by the function</param>
    /// <param name="dtc_mask">a unique identification number (three byte value) for a specific diagnostic trouble code</param>
    /// <param name="user_def_dtc_snapshot_record_number">User DTC snapshot record number</param>
    /// <param name="memory_selection">Memory selection</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function SvcReadDTCInformationRUDMDTCSSBDTC_2013
      (channel: cantp_handle; request_config: uds_msgconfig;
      var out_msg_request: uds_msg; dtc_mask: UInt32;
      user_def_dtc_snapshot_record_number: Byte; memory_selection: Byte)
      : uds_status;

    /// <summary>
    /// This service allows a client to read the status of server-resident Diagnostic Trouble Code (DTC) information.
    /// The sub-function reportUserDefMemoryDTCExtDataRecordByDTCNumber (PUDS_SVC_PARAM_RDTCI_RUDMDTCEDRBDN) is implicit.
    /// </summary>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="request_config">Request configuration</param>
    /// <param name="out_msg_request">(out) request message created and sent by the function</param>
    /// <param name="dtc_mask">a unique identification number (three byte value) for a specific diagnostic trouble code</param>
    /// <param name="dtc_extended_data_record_number">DTC extened data record number</param>
    /// <param name="memory_selection">Memory selection</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function SvcReadDTCInformationRUDMDTCEDRBDN_2013
      (channel: cantp_handle; request_config: uds_msgconfig;
      var out_msg_request: uds_msg; dtc_mask: UInt32;
      dtc_extended_data_record_number: Byte; memory_selection: Byte)
      : uds_status;

    /// <summary>
    /// ISO_14229-1 2020
    /// This service allows a client to read the status of server-resident Diagnostic Trouble Code (DTC) information.
    /// The sub-function reportSupportedDTCExtDataRecord (PUDS_SVC_PARAM_RDTCI_RDTCEDI) is implicit.
    /// </summary>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="request_config">Request configuration</param>
    /// <param name="out_msg_request">(out) request message created and sent by the function</param>
    /// <param name="dtc_extended_data_record_number">DTC extended data record number</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function SvcReadDTCInformationRDTCEDI_2020(channel: cantp_handle;
      request_config: uds_msgconfig; var out_msg_request: uds_msg;
      dtc_extended_data_record_number: Byte): uds_status;

    /// <summary>
    /// This service allows a client to read the status of server-resident Diagnostic Trouble Code (DTC) information.
    /// The sub-function reportWWHOBDDTCByMaskRecord (PUDS_SVC_PARAM_RDTCI_RWWHOBDDTCBMR) is implicit.
    /// </summary>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="request_config">Request configuration</param>
    /// <param name="out_msg_request">(out) request message created and sent by the function</param>
    /// <param name="functional_group_identifier">Functional group identifier</param>
    /// <param name="dtc_status_mask">a mask of eight (8) DTC status bits</param>
    /// <param name="dtc_severity_mask">a mask of eight (8) DTC severity bits (see PUDS_SVC_PARAM_RDTCI_DTCSVM_xxx)</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function SvcReadDTCInformationRWWHOBDDTCBMR_2013
      (channel: cantp_handle; request_config: uds_msgconfig;
      var out_msg_request: uds_msg; functional_group_identifier: Byte;
      dtc_status_mask: Byte; dtc_severity_mask: Byte): uds_status;

    /// <summary>
    /// This service allows a client to read the status of server-resident Diagnostic Trouble Code (DTC) information.
    /// The sub-function reportWWHOBDDTCWithPermanentStatus (PUDS_SVC_PARAM_RDTCI_RWWHOBDDTCWPS ) is implicit.
    /// </summary>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="request_config">Request configuration</param>
    /// <param name="out_msg_request">(out) request message created and sent by the function</param>
    /// <param name="functional_group_identifier">Functional group identifier</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function SvcReadDTCInformationRWWHOBDDTCWPS_2013
      (channel: cantp_handle; request_config: uds_msgconfig;
      var out_msg_request: uds_msg; functional_group_identifier: Byte)
      : uds_status;

    /// <summary>
    /// ISO_14229-1 2020
    /// This service allows a client to read the status of server-resident Diagnostic Trouble Code (DTC) information.
    /// The sub-function reportDTCInformationByDTCReadinessGroupIdentifier (PUDS_SVC_PARAM_RDTCI_RDTCBRGI ) is implicit.
    /// </summary>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="request_config">Request configuration</param>
    /// <param name="out_msg_request">(out) request message created and sent by the function</param>
    /// <param name="functional_group_identifier">Functional group identifier</param>
    /// <param name="dtc_readiness_group_identifier">DTC readiness group identifier</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function SvcReadDTCInformationRDTCBRGI_2020(channel: cantp_handle;
      request_config: uds_msgconfig; var out_msg_request: uds_msg;
      functional_group_identifier: Byte; dtc_readiness_group_identifier: Byte)
      : uds_status;

    /// <summary>
    /// The InputOutputControlByIdentifier service is used by the client to substitute a value for an input signal,
    /// internal server function and/or control an output (actuator) of an electronic system.
    /// </summary>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="request_config">Request configuration (PUDS_MSGTYPE_FLAG_NO_POSITIVE_RESPONSE is ignored)</param>
    /// <param name="out_msg_request">(out) request message created and sent by the function</param>
    /// <param name="data_identifier">a two-byte Data Identifier (see PUDS_SVC_PARAM_DI_xxx)</param>
    /// <param name="control_option_record">First byte can be used as either an inputOutputControlParameter
    /// that describes how the server shall control its inputs or outputs (see PUDS_SVC_PARAM_IOCBI_xxx),
    /// or as an additional controlState byte</param>
    /// <param name="control_option_record_size">Size in bytes of the control_option_record buffer</param>
    /// <param name="control_enable_mask_record">The control_enable_mask_record shall only be supported when
    /// the inputOutputControlParameter is used (see control_option_record) and the dataIdentifier to be controlled consists
    /// of more than one parameter (i.e. the dataIdentifier is bit-mapped or packeted by definition).
    /// There shall be one bit in the control_enable_mask_record corresponding to each individual parameter
    /// defined within the dataIdentifier.</param>
    /// <param name="control_enable_mask_record_size">Size in bytes of the control_enable_mask_record buffer</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function SvcInputOutputControlByIdentifier_2013(channel: cantp_handle;
      request_config: uds_msgconfig; var out_msg_request: uds_msg;
      data_identifier: uds_svc_param_di; control_option_record: PByte;
      control_option_record_size: UInt32; control_enable_mask_record: PByte;
      control_enable_mask_record_size: UInt32): uds_status; overload;

    /// <summary>
    /// The InputOutputControlByIdentifier service is used by the client to substitute a value for an input signal,
    /// internal server function and/or control an output (actuator) of an electronic system.
    /// </summary>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="request_config">Request configuration (PUDS_MSGTYPE_FLAG_NO_POSITIVE_RESPONSE is ignored)</param>
    /// <param name="out_msg_request">(out) request message created and sent by the function</param>
    /// <param name="data_identifier">a two-byte Data Identifier (see PUDS_SVC_PARAM_DI_xxx)</param>
    /// <param name="control_option_record">First byte can be used as either an inputOutputControlParameter
    /// that describes how the server shall control its inputs or outputs (see PUDS_SVC_PARAM_IOCBI_xxx),
    /// or as an additional controlState byte</param>
    /// <param name="control_option_record_size">Size in bytes of the control_option_record buffer</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function SvcInputOutputControlByIdentifier_2013(channel: cantp_handle;
      request_config: uds_msgconfig; var out_msg_request: uds_msg;
      data_identifier: uds_svc_param_di; control_option_record: PByte;
      control_option_record_size: UInt32): uds_status; overload;

    /// <summary>
    /// The RoutineControl service is used by the client to start/stop a routine,
    /// and request routine results.
    /// </summary>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="request_config">Request configuration</param>
    /// <param name="out_msg_request">(out) request message created and sent by the function</param>
    /// <param name="routine_control_type">Subfunction parameter: RoutineControl type (see PUDS_SVC_PARAM_RC_xxx)</param>
    /// <param name="routine_identifier">Server Local Routine Identifier (see PUDS_SVC_PARAM_RC_RID_xxx)</param>
    /// <param name="routine_control_option_record">buffer containing the Routine Control Options (only with start and stop routine sub-functions)</param>
    /// <param name="routine_control_option_record_size">Size in bytes of the buffer</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function SvcRoutineControl_2013(channel: cantp_handle;
      request_config: uds_msgconfig; var out_msg_request: uds_msg;
      routine_control_type: uds_svc_param_rc;
      routine_identifier: uds_svc_param_rc_rid;
      routine_control_option_record: PByte;
      routine_control_option_record_size: UInt32): uds_status; overload;

    /// <summary>
    /// The RoutineControl service is used by the client to start/stop a routine,
    /// and request routine results.
    /// Use only with not start nor stop routine sub-functions.
    /// </summary>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="request_config">Request configuration</param>
    /// <param name="out_msg_request">(out) request message created and sent by the function</param>
    /// <param name="routine_control_type">Subfunction parameter: RoutineControl type (see PUDS_SVC_PARAM_RC_xxx)</param>
    /// <param name="routine_identifier">Server Local Routine Identifier (see PUDS_SVC_PARAM_RC_RID_xxx)</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function SvcRoutineControl_2013(channel: cantp_handle;
      request_config: uds_msgconfig; var out_msg_request: uds_msg;
      routine_control_type: uds_svc_param_rc;
      routine_identifier: uds_svc_param_rc_rid): uds_status; overload;

    // ISO-14229-1:2013 §14.2 p.270
    /// <summary>
    /// The requestDownload service is used by the client to initiate a data transfer
    /// from the client to the server (download).
    /// </summary>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="request_config">Request configuration (PUDS_MSGTYPE_FLAG_NO_POSITIVE_RESPONSE is ignored)</param>
    /// <param name="out_msg_request">(out) request message created and sent by the function</param>
    /// <param name="compression_method">A nibble-value that specifies the "compressionMethod",
    /// The value 0x0 specifies that no compressionMethod is used.</param>
    /// <param name="encrypting_method">A nibble-value that specifies the "encryptingMethod",
    /// The value 0x0 specifies that no encryptingMethod is used.</param>
    /// <param name="memory_address_buffer">starting address of server memory to which data is to be written</param>
    /// <param name="memory_address_size">Size in bytes of the memory_address_buffer buffer (max.: 0xF)</param>
    /// <param name="memory_size_buffer">used by the server to compare the uncompressed memory size with
    /// the total amount of data transferred during the TransferData service</param>
    /// <param name="memory_size_size">Size in bytes of the memory_size_buffer buffer (max.: 0xF)</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function SvcRequestDownload_2013(channel: cantp_handle;
      request_config: uds_msgconfig; var out_msg_request: uds_msg;
      compression_method: Byte; encrypting_method: Byte;
      memory_address_buffer: PByte; memory_address_size: Byte;
      memory_size_buffer: PByte; memory_size_size: Byte): uds_status;

    // ISO-14229-1:2013 §14.3 p.275
    /// <summary>
    /// The requestUpload service is used by the client to initiate a data transfer
    /// from the server to the client (upload).
    /// </summary>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="request_config">Request configuration (PUDS_MSGTYPE_FLAG_NO_POSITIVE_RESPONSE is ignored)</param>
    /// <param name="out_msg_request">(out) request message created and sent by the function</param>
    /// <param name="compression_method">A nibble-value that specifies the "compressionMethod",
    /// The value 0x0 specifies that no compressionMethod is used.</param>
    /// <param name="encrypting_method">A nibble-value that specifies the "encryptingMethod",
    /// The value 0x0 specifies that no encryptingMethod is used.</param>
    /// <param name="memory_address_buffer">starting address of server memory from which data is to be retrieved</param>
    /// <param name="memory_address_size">Size in bytes of the memory_address_buffer buffer (max.: 0xF)</param>
    /// <param name="memory_size_buffer">used by the server to compare the uncompressed memory size with
    /// the total amount of data transferred during the TransferData service</param>
    /// <param name="memory_size_size">Size in bytes of the memory_size_buffer buffer (max.: 0xF)</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function SvcRequestUpload_2013(channel: cantp_handle;
      request_config: uds_msgconfig; var out_msg_request: uds_msg;
      compression_method: Byte; encrypting_method: Byte;
      memory_address_buffer: PByte; memory_address_size: Byte;
      memory_size_buffer: PByte; memory_size_size: Byte): uds_status;

    // ISO-14229-1:2013 §14.4 p.280
    /// <summary>
    /// The TransferData service is used by the client to transfer data either from the client
    /// to the server (download) or from the server to the client (upload).
    /// </summary>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="request_config">Request configuration (PUDS_MSGTYPE_FLAG_NO_POSITIVE_RESPONSE is ignored)</param>
    /// <param name="out_msg_request">(out) request message created and sent by the function</param>
    /// <param name="block_sequence_counter">The blockSequenceCounter parameter value starts at 01 hex
    /// with the first TransferData request that follows the RequestDownload (34 hex)
    /// or RequestUpload (35 hex) service. Its value is incremented by 1 for each subsequent
    /// TransferData request. At the value of FF hex, the blockSequenceCounter rolls over
    /// and starts at 00 hex with the next TransferData request message.</param>
    /// <param name="transfer_request_parameter_record">buffer containing the required transfer parameters</param>
    /// <param name="transfer_request_parameter_record_size">Size in bytes of the buffer</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function SvcTransferData_2013(channel: cantp_handle;
      request_config: uds_msgconfig; var out_msg_request: uds_msg;
      block_sequence_counter: Byte; transfer_request_parameter_record: PByte;
      transfer_request_parameter_record_size: UInt32): uds_status; overload;

    /// <summary>
    /// The TransferData service is used by the client to transfer data either from the client
    /// to the server (download) or from the server to the client (upload).
    /// </summary>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="request_config">Request configuration (PUDS_MSGTYPE_FLAG_NO_POSITIVE_RESPONSE is ignored)</param>
    /// <param name="out_msg_request">(out) request message created and sent by the function</param>
    /// <param name="block_sequence_counter">The blockSequenceCounter parameter value starts at 01 hex
    /// with the first TransferData request that follows the RequestDownload (34 hex)
    /// or RequestUpload (35 hex) service. Its value is incremented by 1 for each subsequent
    /// TransferData request. At the value of FF hex, the blockSequenceCounter rolls over
    /// and starts at 00 hex with the next TransferData request message.</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function SvcTransferData_2013(channel: cantp_handle;
      request_config: uds_msgconfig; var out_msg_request: uds_msg;
      block_sequence_counter: Byte): uds_status; overload;

    // ISO-14229-1:2013 §14.5 p.285
    /// <summary>
    /// The RequestTransferExit service is used by the client to terminate a data
    /// transfer between client and server (upload or download).
    /// </summary>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="request_config">Request configuration (PUDS_MSGTYPE_FLAG_NO_POSITIVE_RESPONSE is ignored)</param>
    /// <param name="out_msg_request">(out) request message created and sent by the function</param>
    /// <param name="transfer_request_parameter_record">buffer containing the required transfer parameters</param>
    /// <param name="transfer_request_parameter_record_size">Size in bytes of the buffer</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function SvcRequestTransferExit_2013(channel: cantp_handle;
      request_config: uds_msgconfig; var out_msg_request: uds_msg;
      transfer_request_parameter_record: PByte;
      transfer_request_parameter_record_size: UInt32): uds_status; overload;

    /// <summary>
    /// The RequestTransferExit service is used by the client to terminate a data
    /// transfer between client and server (upload or download).
    /// </summary>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="request_config">Request configuration (PUDS_MSGTYPE_FLAG_NO_POSITIVE_RESPONSE is ignored)</param>
    /// <param name="out_msg_request">(out) request message created and sent by the function</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function SvcRequestTransferExit_2013(channel: cantp_handle;
      request_config: uds_msgconfig; var out_msg_request: uds_msg): uds_status; overload;

    // See ISO-14229-1:2013 §9.7.2.2 p.62 table 74

    /// <summary>
    /// AccessTimingParameter service.
    /// </summary>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="request_config">Request configuration</param>
    /// <param name="out_msg_request">(out) request message created and sent by the function</param>
    /// <param name="access_type">Access type, see PUDS_SVC_PARAM_ATP_* values</param>
    /// <param name="request_record">Timing parameter request record</param>
    /// <param name="request_record_size">Size in byte of the request record</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function SvcAccessTimingParameter_2013(channel: cantp_handle;
      request_config: uds_msgconfig; var out_msg_request: uds_msg;
      access_type: uds_svc_param_atp; request_record: PByte;
      request_record_size: UInt32): uds_status;

    // See ISO-14229-1:2013 Annex G p.376 table G.1

    /// <summary>
    /// RequestFileTransfer service.
    /// </summary>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="request_config">Request configuration (PUDS_MSGTYPE_FLAG_NO_POSITIVE_RESPONSE is ignored)</param>
    /// <param name="out_msg_request">(out) request message created and sent by the function</param>
    /// <param name="mode_of_operation">Mode of operation (add, delete, replace, read), see PUDS_SVC_PARAM_RFT_MOOP_* values</param>
    /// <param name="file_path_and_name_size">Size in bytes of file_path_and_name buffer</param>
    /// <param name="file_path_and_name">File path and name string</param>
    /// <param name="compression_method">A nibble-value that specifies the "compressionMethod", the value 0x0 specifies that no compressionMethod is used.</param>
    /// <param name="encrypting_method">A nibble-value that specifies the "encryptingMethod", the value 0x0 specifies that no encryptingMethod is used.</param>
    /// <param name="file_size_parameter_size">Size in byte of file_size_uncompressed and file_size_compressed parameters</param>
    /// <param name="file_size_uncompressed">Uncompressed file size</param>
    /// <param name="file_size_compressed">Compressed file size</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function SvcRequestFileTransfer_2013(channel: cantp_handle;
      request_config: uds_msgconfig; var out_msg_request: uds_msg;
      mode_of_operation: uds_svc_param_rft_moop;
      file_path_and_name_size: UInt16; file_path_and_name: PByte;
      compression_method: Byte; encrypting_method: Byte;
      file_size_parameter_size: Byte; file_size_uncompressed: PByte;
      file_size_compressed: PByte): uds_status; overload;

    /// <summary>
    /// RequestFileTransfer service.
    /// </summary>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="request_config">Request configuration (PUDS_MSGTYPE_FLAG_NO_POSITIVE_RESPONSE is ignored)</param>
    /// <param name="out_msg_request">(out) request message created and sent by the function</param>
    /// <param name="mode_of_operation">Mode of operation (add, delete, replace, read), see PUDS_SVC_PARAM_RFT_MOOP_* values</param>
    /// <param name="file_path_and_name_size">Size in bytes of file_path_and_name buffer</param>
    /// <param name="file_path_and_name">File path and name string</param>
    /// <param name="compression_method">A nibble-value that specifies the "compressionMethod", the value 0x0 specifies that no compressionMethod is used.</param>
    /// <param name="encrypting_method">A nibble-value that specifies the "encryptingMethod", the value 0x0 specifies that no encryptingMethod is used.</param>
    /// <param name="file_size_parameter_size">Size in byte of file_size_uncompressed and file_size_compressed parameters</param>
    /// <param name="file_size_uncompressed">Uncompressed file size</param>
    /// <param name="file_size_compressed">Compressed file size</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function SvcRequestFileTransfer_2013(channel: cantp_handle;
      request_config: uds_msgconfig; var out_msg_request: uds_msg;
      mode_of_operation: uds_svc_param_rft_moop;
      file_path_and_name_size: UInt16; file_path_and_name: PAnsiChar;
      compression_method: Byte; encrypting_method: Byte;
      file_size_parameter_size: Byte; file_size_uncompressed: PByte;
      file_size_compressed: PByte): uds_status; overload;

    /// <summary>
    /// RequestFileTransfer service.
    /// Use with mode of operation which does not require file compressed or uncompressed sizes
    /// </summary>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="request_config">Request configuration (PUDS_MSGTYPE_FLAG_NO_POSITIVE_RESPONSE is ignored)</param>
    /// <param name="out_msg_request">(out) request message created and sent by the function</param>
    /// <param name="mode_of_operation">Mode of operation (add, delete, replace, read), see PUDS_SVC_PARAM_RFT_MOOP_* values</param>
    /// <param name="file_path_and_name_size">Size in bytes of file_path_and_name buffer</param>
    /// <param name="file_path_and_name">File path and name string</param>
    /// <param name="compression_method">A nibble-value that specifies the "compressionMethod", the value 0x0 specifies that no compressionMethod is used.</param>
    /// <param name="encrypting_method">A nibble-value that specifies the "encryptingMethod", the value 0x0 specifies that no encryptingMethod is used.</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function SvcRequestFileTransfer_2013(channel: cantp_handle;
      request_config: uds_msgconfig; var out_msg_request: uds_msg;
      mode_of_operation: uds_svc_param_rft_moop;
      file_path_and_name_size: UInt16; file_path_and_name: PByte;
      compression_method: Byte; encrypting_method: Byte): uds_status; overload;

    /// <summary>
    /// RequestFileTransfer service.
    /// Use with mode of operation which does not require file compressed or uncompressed sizes
    /// </summary>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="request_config">Request configuration (PUDS_MSGTYPE_FLAG_NO_POSITIVE_RESPONSE is ignored)</param>
    /// <param name="out_msg_request">(out) request message created and sent by the function</param>
    /// <param name="mode_of_operation">Mode of operation (add, delete, replace, read), see PUDS_SVC_PARAM_RFT_MOOP_* values</param>
    /// <param name="file_path_and_name_size">Size in bytes of file_path_and_name buffer</param>
    /// <param name="file_path_and_name">File path and name string</param>
    /// <param name="compression_method">A nibble-value that specifies the "compressionMethod", the value 0x0 specifies that no compressionMethod is used.</param>
    /// <param name="encrypting_method">A nibble-value that specifies the "encryptingMethod", the value 0x0 specifies that no encryptingMethod is used.</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function SvcRequestFileTransfer_2013(channel: cantp_handle;
      request_config: uds_msgconfig; var out_msg_request: uds_msg;
      mode_of_operation: uds_svc_param_rft_moop;
      file_path_and_name_size: UInt16; file_path_and_name: PAnsiChar;
      compression_method: Byte; encrypting_method: Byte): uds_status; overload;

    /// <summary>
    /// RequestFileTransfer service.
    /// Use with mode of operation which does not require file compressed or uncompressed sizes, nor compression or encrypting methods
    /// </summary>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="request_config">Request configuration (PUDS_MSGTYPE_FLAG_NO_POSITIVE_RESPONSE is ignored)</param>
    /// <param name="out_msg_request">(out) request message created and sent by the function</param>
    /// <param name="mode_of_operation">Mode of operation (add, delete, replace, read), see PUDS_SVC_PARAM_RFT_MOOP_* values</param>
    /// <param name="file_path_and_name_size">Size in bytes of file_path_and_name buffer</param>
    /// <param name="file_path_and_name">File path and name string</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function SvcRequestFileTransfer_2013(channel: cantp_handle;
      request_config: uds_msgconfig; var out_msg_request: uds_msg;
      mode_of_operation: uds_svc_param_rft_moop;
      file_path_and_name_size: UInt16; file_path_and_name: PByte)
      : uds_status; overload;

    /// <summary>
    /// RequestFileTransfer service.
    /// Use with mode of operation which does not require file compressed or uncompressed sizes, nor compression or encrypting methods
    /// </summary>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="request_config">Request configuration (PUDS_MSGTYPE_FLAG_NO_POSITIVE_RESPONSE is ignored)</param>
    /// <param name="out_msg_request">(out) request message created and sent by the function</param>
    /// <param name="mode_of_operation">Mode of operation (add, delete, replace, read), see PUDS_SVC_PARAM_RFT_MOOP_* values</param>
    /// <param name="file_path_and_name_size">Size in bytes of file_path_and_name buffer</param>
    /// <param name="file_path_and_name">File path and name string</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function SvcRequestFileTransfer_2013(channel: cantp_handle;
      request_config: uds_msgconfig; var out_msg_request: uds_msg;
      mode_of_operation: uds_svc_param_rft_moop;
      file_path_and_name_size: UInt16; file_path_and_name: PAnsiChar)
      : uds_status; overload;

    /// <summary>
    /// Sends Authentication service request with deAuthenticate subfunction.
    /// </summary>
    /// <remarks>
    ///  API provides uds_svc_authentication_subfunction and uds_svc_authentication_return_parameter
    ///  enumerations to help user to decode Authentication service responses.
    /// </remarks>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="request_config">Request configuration</param>
    /// <param name="out_msg_request">(out) request message created and sent by the function</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function SvcAuthenticationDA_2020(channel: cantp_handle;
      request_config: uds_msgconfig; var out_msg_request: uds_msg): uds_status;

    /// <summary>
    /// Sends Authentication service request with verifyCertificateUnidirectional subfunction.
    /// </summary>
    /// <remarks>
    ///  API provides uds_svc_authentication_subfunction and uds_svc_authentication_return_parameter
    ///  enumerations to help user to decode Authentication service responses.
    /// </remarks>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="request_config">Request configuration</param>
    /// <param name="out_msg_request">(out) request message created and sent by the function</param>
    /// <param name="communication_configuration">Configuration information about communication</param>
    /// <param name="certificate_client">Buffer containing the certificate of the client</param>
    /// <param name="certificate_client_size">Size in bytes of the certificate buffer</param>
    /// <param name="challenge_client">Buffer containing the challenge of the client</param>
    /// <param name="challenge_client_size">Size in bytes of the challenge buffer</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function SvcAuthenticationVCU_2020(channel: cantp_handle;
      request_config: uds_msgconfig; var out_msg_request: uds_msg;
      communication_configuration: Byte; certificate_client: PByte;
      certificate_client_size: UInt16; challenge_client: PByte;
      challenge_client_size: UInt16): uds_status; overload;

    /// <summary>
    /// Sends Authentication service request with verifyCertificateUnidirectional subfunction,
    /// without challenge buffer.
    /// </summary>
    /// <remarks>
    ///  API provides uds_svc_authentication_subfunction and uds_svc_authentication_return_parameter
    ///  enumerations to help user to decode Authentication service responses.
    /// </remarks>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="request_config">Request configuration</param>
    /// <param name="out_msg_request">(out) request message created and sent by the function</param>
    /// <param name="communication_configuration">Configuration information about communication</param>
    /// <param name="certificate_client">Buffer containing the certificate of the client</param>
    /// <param name="certificate_client_size">Size in bytes of the certificate buffer</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function SvcAuthenticationVCU_2020(channel: cantp_handle;
      request_config: uds_msgconfig; var out_msg_request: uds_msg;
      communication_configuration: Byte; certificate_client: PByte;
      certificate_client_size: UInt16): uds_status; overload;

    /// <summary>
    /// Sends Authentication service request with verifyCertificateBidirectional subfunction.
    /// </summary>
    /// <remarks>
    ///  API provides uds_svc_authentication_subfunction and uds_svc_authentication_return_parameter
    ///  enumerations to help user to decode Authentication service responses.
    /// </remarks>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="request_config">Request configuration</param>
    /// <param name="out_msg_request">(out) request message created and sent by the function</param>
    /// <param name="communication_configuration">Configuration information about communication</param>
    /// <param name="certificate_client">Buffer containing the certificate of the client</param>
    /// <param name="certificate_client_size">Size in bytes of the certificate buffer</param>
    /// <param name="challenge_client">Buffer containing the challenge of the client</param>
    /// <param name="challenge_client_size">Size in bytes of the challenge buffer</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function SvcAuthenticationVCB_2020(channel: cantp_handle;
      request_config: uds_msgconfig; var out_msg_request: uds_msg;
      communication_configuration: Byte; certificate_client: PByte;
      certificate_client_size: UInt16; challenge_client: PByte;
      challenge_client_size: UInt16): uds_status;

    /// <summary>
    /// Sends Authentication service request with proofOfOwnership subfunction.
    /// </summary>
    /// <remarks>
    ///  API provides uds_svc_authentication_subfunction and uds_svc_authentication_return_parameter
    ///  enumerations to help user to decode Authentication service responses.
    /// </remarks>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="request_config">Request configuration</param>
    /// <param name="out_msg_request">(out) request message created and sent by the function</param>
    /// <param name="proof_of_ownership_client">Buffer containing the proof of ownership of the client</param>
    /// <param name="proof_of_ownership_client_size">Size in bytes of the proof of ownership buffer</param>
    /// <param name="ephemeral_public_key_client">Buffer containing the ephemeral public key of the client</param>
    /// <param name="ephemeral_public_key_client_size">Size in bytes of the ephemeral public key buffer</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function SvcAuthenticationPOWN_2020(channel: cantp_handle;
      request_config: uds_msgconfig; var out_msg_request: uds_msg;
      proof_of_ownership_client: PByte; proof_of_ownership_client_size: UInt16;
      ephemeral_public_key_client: PByte;
      ephemeral_public_key_client_size: UInt16): uds_status; overload;

    /// <summary>
    /// Sends Authentication service request with proofOfOwnership subfunction,
    /// without ephemeral public key.
    /// </summary>
    /// <remarks>
    ///  API provides uds_svc_authentication_subfunction and uds_svc_authentication_return_parameter
    ///  enumerations to help user to decode Authentication service responses.
    /// </remarks>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="request_config">Request configuration</param>
    /// <param name="out_msg_request">(out) request message created and sent by the function</param>
    /// <param name="proof_of_ownership_client">Buffer containing the proof of ownership of the client</param>
    /// <param name="proof_of_ownership_client_size">Size in bytes of the proof of ownership buffer</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function SvcAuthenticationPOWN_2020(channel: cantp_handle;
      request_config: uds_msgconfig; var out_msg_request: uds_msg;
      proof_of_ownership_client: PByte; proof_of_ownership_client_size: UInt16)
      : uds_status; overload;

    /// <summary>
    /// Sends Authentication service request with requestChallengeForAuthentication subfunction.
    /// </summary>
    /// <remarks>
    ///  API provides uds_svc_authentication_subfunction and uds_svc_authentication_return_parameter
    ///  enumerations to help user to decode Authentication service responses.
    /// </remarks>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="request_config">Request configuration</param>
    /// <param name="out_msg_request">(out) request message created and sent by the function</param>
    /// <param name="communication_configuration">Configuration information about communication</param>
    /// <param name="algorithm_indicator">Buffer of 16 bytes containing the algorithm indicator</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function SvcAuthenticationRCFA_2020(channel: cantp_handle;
      request_config: uds_msgconfig; var out_msg_request: uds_msg;
      communication_configuration: Byte; algorithm_indicator: PByte)
      : uds_status;

    /// <summary>
    /// Sends Authentication service request with verifyProofOfOwnershipUnidirectional subfunction.
    /// </summary>
    /// <remarks>
    ///  API provides uds_svc_authentication_subfunction and uds_svc_authentication_return_parameter
    ///  enumerations to help user to decode Authentication service responses.
    /// </remarks>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="request_config">Request configuration</param>
    /// <param name="out_msg_request">(out) request message created and sent by the function</param>
    /// <param name="algorithm_indicator">Buffer of 16 bytes containing the algorithm indicator</param>
    /// <param name="proof_of_ownership_client">Buffer containing the proof of ownership of the client</param>
    /// <param name="proof_of_ownership_client_size">Size in bytes of the proof of ownership buffer</param>
    /// <param name="challenge_client">Buffer containing the challenge of the client</param>
    /// <param name="challenge_client_size">Size in bytes of the challenge buffer</param>
    /// <param name="additional_parameter">Buffer containing additional parameters</param>
    /// <param name="additional_parameter_size">Size in bytes of the additional parameter buffer</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function SvcAuthenticationVPOWNU_2020(channel: cantp_handle;
      request_config: uds_msgconfig; var out_msg_request: uds_msg;
      algorithm_indicator: PByte; proof_of_ownership_client: PByte;
      proof_of_ownership_client_size: UInt16; challenge_client: PByte;
      challenge_client_size: UInt16; additional_parameter: PByte;
      additional_parameter_size: UInt16): uds_status; overload;

    /// <summary>
    /// Sends Authentication service request with verifyProofOfOwnershipUnidirectional subfunction,
    /// without additional parameter
    /// </summary>
    /// <remarks>
    ///  API provides uds_svc_authentication_subfunction and uds_svc_authentication_return_parameter
    ///  enumerations to help user to decode Authentication service responses.
    /// </remarks>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="request_config">Request configuration</param>
    /// <param name="out_msg_request">(out) request message created and sent by the function</param>
    /// <param name="algorithm_indicator">Buffer of 16 bytes containing the algorithm indicator</param>
    /// <param name="proof_of_ownership_client">Buffer containing the proof of ownership of the client</param>
    /// <param name="proof_of_ownership_client_size">Size in bytes of the proof of ownership buffer</param>
    /// <param name="challenge_client">Buffer containing the challenge of the client</param>
    /// <param name="challenge_client_size">Size in bytes of the challenge buffer</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function SvcAuthenticationVPOWNU_2020(channel: cantp_handle;
      request_config: uds_msgconfig; var out_msg_request: uds_msg;
      algorithm_indicator: PByte; proof_of_ownership_client: PByte;
      proof_of_ownership_client_size: UInt16; challenge_client: PByte;
      challenge_client_size: UInt16): uds_status; overload;

    /// <summary>
    /// Sends Authentication service request with verifyProofOfOwnershipUnidirectional subfunction,
    /// without additional parameter, without challenge buffer
    /// </summary>
    /// <remarks>
    ///  API provides uds_svc_authentication_subfunction and uds_svc_authentication_return_parameter
    ///  enumerations to help user to decode Authentication service responses.
    /// </remarks>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="request_config">Request configuration</param>
    /// <param name="out_msg_request">(out) request message created and sent by the function</param>
    /// <param name="algorithm_indicator">Buffer of 16 bytes containing the algorithm indicator</param>
    /// <param name="proof_of_ownership_client">Buffer containing the proof of ownership of the client</param>
    /// <param name="proof_of_ownership_client_size">Size in bytes of the proof of ownership buffer</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function SvcAuthenticationVPOWNU_2020(channel: cantp_handle;
      request_config: uds_msgconfig; var out_msg_request: uds_msg;
      algorithm_indicator: PByte; proof_of_ownership_client: PByte;
      proof_of_ownership_client_size: UInt16): uds_status; overload;

    /// <summary>
    /// Sends Authentication service request with verifyProofOfOwnershipBidirectional subfunction.
    /// </summary>
    /// <remarks>
    ///  API provides uds_svc_authentication_subfunction and uds_svc_authentication_return_parameter
    ///  enumerations to help user to decode Authentication service responses.
    /// </remarks>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="request_config">Request configuration</param>
    /// <param name="out_msg_request">(out) request message created and sent by the function</param>
    /// <param name="algorithm_indicator">Buffer of 16 bytes containing the algorithm indicator</param>
    /// <param name="proof_of_ownership_client">Buffer containing the proof of ownership of the client</param>
    /// <param name="proof_of_ownership_client_size">Size in bytes of the proof of ownership buffer</param>
    /// <param name="challenge_client">Buffer containing the challenge of the client</param>
    /// <param name="challenge_client_size">Size in bytes of the challenge buffer</param>
    /// <param name="additional_parameter">Buffer containing additional parameters</param>
    /// <param name="additional_parameter_size">Size in bytes of the additional parameter buffer</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function SvcAuthenticationVPOWNB_2020(channel: cantp_handle;
      request_config: uds_msgconfig; var out_msg_request: uds_msg;
      algorithm_indicator: PByte; proof_of_ownership_client: PByte;
      proof_of_ownership_client_size: UInt16; challenge_client: PByte;
      challenge_client_size: UInt16; additional_parameter: PByte;
      additional_parameter_size: UInt16): uds_status; overload;

    /// <summary>
    /// Sends Authentication service request with verifyProofOfOwnershipBidirectional subfunction,
    /// without additional parameters
    /// </summary>
    /// <remarks>
    ///  API provides uds_svc_authentication_subfunction and uds_svc_authentication_return_parameter
    ///  enumerations to help user to decode Authentication service responses.
    /// </remarks>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="request_config">Request configuration</param>
    /// <param name="out_msg_request">(out) request message created and sent by the function</param>
    /// <param name="algorithm_indicator">Buffer of 16 bytes containing the algorithm indicator</param>
    /// <param name="proof_of_ownership_client">Buffer containing the proof of ownership of the client</param>
    /// <param name="proof_of_ownership_client_size">Size in bytes of the proof of ownership buffer</param>
    /// <param name="challenge_client">Buffer containing the challenge of the client</param>
    /// <param name="challenge_client_size">Size in bytes of the challenge buffer</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function SvcAuthenticationVPOWNB_2020(channel: cantp_handle;
      request_config: uds_msgconfig; var out_msg_request: uds_msg;
      algorithm_indicator: PByte; proof_of_ownership_client: PByte;
      proof_of_ownership_client_size: UInt16; challenge_client: PByte;
      challenge_client_size: UInt16): uds_status; overload;

    /// <summary>
    /// Sends Authentication service request with authenticationConfiguration subfunction.
    /// </summary>
    /// <remarks>
    ///  API provides uds_svc_authentication_subfunction and uds_svc_authentication_return_parameter
    ///  enumerations to help user to decode Authentication service responses.
    /// </remarks>
    /// <param name="channel">A PCANTP channel handle representing a PUDS channel</param>
    /// <param name="request_config">Request configuration</param>
    /// <param name="out_msg_request">(out) request message created and sent by the function</param>
    /// <returns>A uds_status code. PUDS_STATUS_OK is returned on success</returns>
    class function SvcAuthenticationAC_2020(channel: cantp_handle;
      request_config: uds_msgconfig; var out_msg_request: uds_msg): uds_status;

  end;

implementation

uses SysUtils;

const
  DLL_Name = 'PCAN-UDS.DLL';

function UDS_Initialize_2013(channel: cantp_handle; baudrate: cantp_baudrate;
  hw_type: cantp_hwtype; io_port: UInt32; interrupt: UInt16): uds_status;
  stdcall; external DLL_Name;
function UDS_InitializeFD_2013(channel: cantp_handle;
  const bitrate_fd: cantp_bitrate): uds_status; stdcall; external DLL_Name;
function UDS_Uninitialize_2013(channel: cantp_handle): uds_status; stdcall;
  external DLL_Name;
function UDS_Reset_2013(channel: cantp_handle): uds_status; stdcall;
  external DLL_Name;
function UDS_GetCanBusStatus_2013(channel: cantp_handle): uds_status; stdcall;
  external DLL_Name;
function UDS_Read_2013(channel: cantp_handle; var out_msg_buffer: uds_msg;
  in_msg_request: Puds_msg; out_timestamp: Pcantp_timestamp): uds_status;
  stdcall; external DLL_Name;
function UDS_Write_2013(channel: cantp_handle; msg_buffer: Puds_msg)
  : uds_status; stdcall; external DLL_Name;
function UDS_AddMapping_2013(channel: cantp_handle; mapping: Puds_mapping)
  : uds_status; stdcall; external DLL_Name;
function UDS_RemoveMappingByCanId_2013(channel: cantp_handle; can_id: UInt32)
  : uds_status; stdcall; external DLL_Name;
function UDS_RemoveMapping_2013(channel: cantp_handle; mapping: uds_mapping)
  : uds_status; stdcall; external DLL_Name;
function UDS_GetMapping_2013(channel: cantp_handle; var buffer: uds_mapping;
  can_id: UInt32; can_msgtype: cantp_can_msgtype): uds_status; stdcall;
  external DLL_Name;
function UDS_GetMappings_2013(channel: cantp_handle; buffer: Puds_mapping;
  buffer_length: UInt16; count: PUInt16): uds_status; stdcall;
  external DLL_Name;
function UDS_AddCanIdFilter_2013(channel: cantp_handle; can_id: UInt32)
  : uds_status; stdcall; external DLL_Name;
function UDS_RemoveCanIdFilter_2013(channel: cantp_handle; can_id: UInt32)
  : uds_status; stdcall; external DLL_Name;
function UDS_GetValue_2013(channel: cantp_handle; parameter: uds_parameter;
  buffer: Pointer; buffer_size: UInt32): uds_status; stdcall; external DLL_Name;
function UDS_SetValue_2013(channel: cantp_handle; parameter: uds_parameter;
  buffer: Pointer; buffer_size: UInt32): uds_status; stdcall; external DLL_Name;
function UDS_GetErrorText_2013(error_code: uds_status; language: UInt16;
  buffer: Pointer; buffer_size: UInt32): uds_status; stdcall; external DLL_Name;
function UDS_GetSessionInformation_2013(channel: cantp_handle;
  var session_info: uds_sessioninfo): uds_status; stdcall; external DLL_Name;
function UDS_StatusIsOk_2013(status: uds_status; status_expected: uds_status;
  strict_mode: Boolean): Boolean; stdcall; external DLL_Name;
function UDS_MsgAlloc_2013(var msg_buffer: uds_msg;
  msg_configuration: uds_msgconfig; msg_data_length: UInt32): uds_status;
  stdcall; external DLL_Name;
function UDS_MsgFree_2013(var msg_buffer: uds_msg): uds_status; stdcall;
  external DLL_Name;
function UDS_MsgCopy_2013(var msg_buffer_dst: uds_msg;
  const msg_buffer_src: Puds_msg): uds_status; stdcall; external DLL_Name;
function UDS_MsgMove_2013(var msg_buffer_dst: uds_msg;
  var msg_buffer_src: uds_msg): uds_status; stdcall; external DLL_Name;

function UDS_WaitForSingleMessage_2013(channel: cantp_handle;
  msg_request: Puds_msg; is_waiting_for_tx: Boolean; timeout: UInt32;
  timeout_enhanced: UInt32; var out_msg_response: uds_msg): uds_status; stdcall;
  external DLL_Name;
function UDS_WaitForFunctionalResponses_2013(channel: cantp_handle;
  msg_request: Puds_msg; timeout: UInt32; timeout_enhanced: UInt32;
  wait_until_timeout: Boolean; max_msg_count: UInt32;
  out_msg_responses: Puds_msg; var out_msg_count: UInt32): uds_status; stdcall;
  external DLL_Name;
function UDS_WaitForService_2013(channel: cantp_handle; msg_request: Puds_msg;
  var out_msg_response: uds_msg; out_msg_request_confirmation: Puds_msg)
  : uds_status; stdcall; external DLL_Name;
function UDS_WaitForServiceFunctional_2013(channel: cantp_handle;
  msg_request: Puds_msg; max_msg_count: UInt32; wait_until_timeout: Boolean;
  out_msg_responses: Puds_msg; var out_msg_count: UInt32;
  out_msg_request_confirmation: Puds_msg): uds_status; stdcall;
  external DLL_Name;

function UDS_SvcDiagnosticSessionControl_2013(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  session_type: uds_svc_param_dsc): uds_status; stdcall; external DLL_Name;
function UDS_SvcECUReset_2013(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  reset_type: uds_svc_param_er): uds_status; stdcall; external DLL_Name;
function UDS_SvcSecurityAccess_2013(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  security_access_type: Byte; security_access_data: PByte;
  security_access_data_size: UInt32): uds_status; stdcall; external DLL_Name;
function UDS_SvcCommunicationControl_2013(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  control_type: uds_svc_param_cc; communication_type: Byte;
  node_identification_number: UInt16): uds_status; stdcall; external DLL_Name;
function UDS_SvcTesterPresent_2013(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  testerpresent_type: uds_svc_param_tp): uds_status; stdcall; external DLL_Name;
function UDS_SvcSecuredDataTransmission_2013(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  security_data_request_record: PByte;
  security_data_request_record_size: UInt32): uds_status; stdcall;
  external DLL_Name;
function UDS_SvcSecuredDataTransmission_2020(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  administrative_parameter: UInt16; signature_encryption_calculation: Byte;
  anti_replay_counter: UInt16; internal_service_identifier: Byte;
  service_specific_parameters: PByte; service_specific_parameters_size: UInt32;
  signature_mac: PByte; signature_size: UInt16): uds_status; stdcall;
  external DLL_Name;
function UDS_SvcControlDTCSetting_2013(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  dtc_setting_type: uds_svc_param_cdtcs;
  dtc_setting_control_option_record: PByte;
  dtc_setting_control_option_record_size: UInt32): uds_status; stdcall;
  external DLL_Name;
function UDS_SvcResponseOnEvent_2013(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  event_type: uds_svc_param_roe; store_event: Boolean; event_window_time: Byte;
  event_type_record: PByte; event_type_record_size: UInt32;
  service_to_respond_to_record: PByte;
  service_to_respond_to_record_size: UInt32): uds_status; stdcall;
  external DLL_Name;
function UDS_SvcLinkControl_2013(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  link_control_type: uds_svc_param_lc;
  baudrate_identifier: uds_svc_param_lc_baudrate_identifier;
  link_baudrate: UInt32): uds_status; stdcall; external DLL_Name;
function UDS_SvcReadDataByIdentifier_2013(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  data_identifier: Puds_svc_param_di; data_identifier_length: UInt32)
  : uds_status; stdcall; external DLL_Name;
function UDS_SvcReadMemoryByAddress_2013(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  memory_address_buffer: PByte; memory_address_size: Byte;
  memory_size_buffer: PByte; memory_size_size: Byte): uds_status; stdcall;
  external DLL_Name;
function UDS_SvcReadScalingDataByIdentifier_2013(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  data_identifier: uds_svc_param_di): uds_status; stdcall; external DLL_Name;
function UDS_SvcReadDataByPeriodicIdentifier_2013(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  transmission_mode: uds_svc_param_rdbpi; periodic_data_identifier: PByte;
  periodic_data_identifier_size: UInt32): uds_status; stdcall;
  external DLL_Name;
function UDS_SvcDynamicallyDefineDataIdentifierDBID_2013(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  dynamically_defined_data_identifier: uds_svc_param_di;
  source_data_identifier: PUInt16; memory_size: PByte;
  position_in_source_data_record: PByte; number_of_elements: UInt32)
  : uds_status; stdcall; external DLL_Name;
function UDS_SvcDynamicallyDefineDataIdentifierDBMA_2013(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  dynamically_defined_data_identifier: uds_svc_param_di;
  memory_address_size: Byte; memory_size_size: Byte;
  memory_address_buffer: PByte; memory_size_buffer: PByte;
  number_of_elements: UInt32): uds_status; stdcall; external DLL_Name;
function UDS_SvcDynamicallyDefineDataIdentifierCDDDI_2013(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  dynamically_defined_data_identifier: uds_svc_param_di): uds_status; stdcall;
  external DLL_Name;
function UDS_SvcDynamicallyDefineDataIdentifierClearAllDDDI_2013
  (channel: cantp_handle; request_config: uds_msgconfig;
  var out_msg_request: uds_msg): uds_status; stdcall; external DLL_Name;
function UDS_SvcWriteDataByIdentifier_2013(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  data_identifier: uds_svc_param_di; data_record: PByte;
  data_record_size: UInt32): uds_status; stdcall; external DLL_Name;
function UDS_SvcWriteMemoryByAddress_2013(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  memory_address_buffer: PByte; memory_address_size: Byte;
  memory_size_buffer: PByte; memory_size_size: Byte; data_record: PByte;
  data_record_size: UInt32): uds_status; stdcall; external DLL_Name;
function UDS_SvcClearDiagnosticInformation_2013(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  group_of_dtc: UInt32): uds_status; stdcall; external DLL_Name;
function UDS_SvcClearDiagnosticInformation_2020(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  group_of_dtc: UInt32; memory_selection: Byte): uds_status; stdcall;
  external DLL_Name;
function UDS_SvcReadDTCInformation_2013(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  PUDS_SVC_PARAM_RDTCI_Type: uds_svc_param_rdtci; dtc_status_mask: Byte)
  : uds_status; stdcall; external DLL_Name;
function UDS_SvcReadDTCInformationRDTCSSBDTC_2013(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg; dtc_mask: UInt32;
  dtc_snapshot_record_number: Byte): uds_status; stdcall; external DLL_Name;
function UDS_SvcReadDTCInformationRDTCSSBRN_2013(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  dtc_snapshot_record_number: Byte): uds_status; stdcall; external DLL_Name;
function UDS_SvcReadDTCInformationReportExtended_2013(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  PUDS_SVC_PARAM_RDTCI_Type: uds_svc_param_rdtci; dtc_mask: UInt32;
  dtc_extended_data_record_number: Byte): uds_status; stdcall;
  external DLL_Name;
function UDS_SvcReadDTCInformationReportSeverity_2013(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  PUDS_SVC_PARAM_RDTCI_Type: uds_svc_param_rdtci; dtc_severity_mask: Byte;
  dtc_status_mask: Byte): uds_status; stdcall; external DLL_Name;
function UDS_SvcReadDTCInformationRSIODTC_2013(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg; dtc_mask: UInt32)
  : uds_status; stdcall; external DLL_Name;
function UDS_SvcReadDTCInformationNoParam_2013(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  PUDS_SVC_PARAM_RDTCI_Type: uds_svc_param_rdtci): uds_status; stdcall;
  external DLL_Name;
function UDS_SvcReadDTCInformationRDTCEDBR_2013(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  dtc_extended_data_record_number: Byte): uds_status; stdcall;
  external DLL_Name;
function UDS_SvcReadDTCInformationRUDMDTCBSM_2013(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  dtc_status_mask: Byte; memory_selection: Byte): uds_status; stdcall;
  external DLL_Name;
function UDS_SvcReadDTCInformationRUDMDTCSSBDTC_2013(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg; dtc_mask: UInt32;
  user_def_dtc_snapshot_record_number: Byte; memory_selection: Byte)
  : uds_status; stdcall; external DLL_Name;
function UDS_SvcReadDTCInformationRUDMDTCEDRBDN_2013(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg; dtc_mask: UInt32;
  dtc_extended_data_record_number: Byte; memory_selection: Byte): uds_status;
  stdcall; external DLL_Name;
function UDS_SvcReadDTCInformationRDTCEDI_2020(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  dtc_extended_data_record_number: Byte): uds_status; stdcall;
  external DLL_Name;
function UDS_SvcReadDTCInformationRWWHOBDDTCBMR_2013(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  functional_group_identifier: Byte; dtc_status_mask: Byte;
  dtc_severity_mask: Byte): uds_status; stdcall; external DLL_Name;
function UDS_SvcReadDTCInformationRWWHOBDDTCWPS_2013(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  functional_group_identifier: Byte): uds_status; stdcall; external DLL_Name;
function UDS_SvcReadDTCInformationRDTCBRGI_2020(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  functional_group_identifier: Byte; dtc_readiness_group_identifier: Byte)
  : uds_status; stdcall; external DLL_Name;
function UDS_SvcInputOutputControlByIdentifier_2013(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  data_identifier: uds_svc_param_di; control_option_record: PByte;
  control_option_record_size: UInt32; control_enable_mask_record: PByte;
  control_enable_mask_record_size: UInt32): uds_status; stdcall;
  external DLL_Name;
function UDS_SvcRoutineControl_2013(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  routine_control_type: uds_svc_param_rc;
  routine_identifier: uds_svc_param_rc_rid;
  routine_control_option_record: PByte;
  routine_control_option_record_size: UInt32): uds_status; stdcall;
  external DLL_Name;
function UDS_SvcRequestDownload_2013(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  compression_method: Byte; encrypting_method: Byte;
  memory_address_buffer: PByte; memory_address_size: Byte;
  memory_size_buffer: PByte; memory_size_size: Byte): uds_status; stdcall;
  external DLL_Name;
function UDS_SvcRequestUpload_2013(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  compression_method: Byte; encrypting_method: Byte;
  memory_address_buffer: PByte; memory_address_size: Byte;
  memory_size_buffer: PByte; memory_size_size: Byte): uds_status; stdcall;
  external DLL_Name;
function UDS_SvcTransferData_2013(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  block_sequence_counter: Byte; transfer_request_parameter_record: PByte;
  transfer_request_parameter_record_size: UInt32): uds_status; stdcall;
  external DLL_Name;
function UDS_SvcRequestTransferExit_2013(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  transfer_request_parameter_record: PByte;
  transfer_request_parameter_record_size: UInt32): uds_status; stdcall;
  external DLL_Name;
function UDS_SvcAccessTimingParameter_2013(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  access_type: uds_svc_param_atp; request_record: PByte;
  request_record_size: UInt32): uds_status; stdcall; external DLL_Name;
function UDS_SvcRequestFileTransfer_2013(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  mode_of_operation: uds_svc_param_rft_moop; file_path_and_name_size: UInt16;
  file_path_and_name: Pointer; compression_method: Byte;
  encrypting_method: Byte; file_size_parameter_size: Byte;
  file_size_uncompressed: PByte; file_size_compressed: PByte): uds_status;
  stdcall; external DLL_Name;
function UDS_SvcAuthenticationDA_2020(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg): uds_status;
  stdcall; external DLL_Name;
function UDS_SvcAuthenticationVCU_2020(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  communication_configuration: Byte; certificate_client: PByte;
  certificate_client_size: UInt16; challenge_client: PByte;
  challenge_client_size: UInt16): uds_status; stdcall; external DLL_Name;
function UDS_SvcAuthenticationVCB_2020(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  communication_configuration: Byte; certificate_client: PByte;
  certificate_client_size: UInt16; challenge_client: PByte;
  challenge_client_size: UInt16): uds_status; stdcall; external DLL_Name;
function UDS_SvcAuthenticationPOWN_2020(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  proof_of_ownership_client: PByte; proof_of_ownership_client_size: UInt16;
  ephemeral_public_key_client: PByte; ephemeral_public_key_client_size: UInt16)
  : uds_status; stdcall; external DLL_Name;
function UDS_SvcAuthenticationRCFA_2020(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  communication_configuration: Byte; algorithm_indicator: PByte): uds_status;
  stdcall; external DLL_Name;
function UDS_SvcAuthenticationVPOWNU_2020(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  algorithm_indicator: PByte; proof_of_ownership_client: PByte;
  proof_of_ownership_client_size: UInt16; challenge_client: PByte;
  challenge_client_size: UInt16; additional_parameter: PByte;
  additional_parameter_size: UInt16): uds_status; stdcall; external DLL_Name;
function UDS_SvcAuthenticationVPOWNB_2020(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  algorithm_indicator: PByte; proof_of_ownership_client: PByte;
  proof_of_ownership_client_size: UInt16; challenge_client: PByte;
  challenge_client_size: UInt16; additional_parameter: PByte;
  additional_parameter_size: UInt16): uds_status; stdcall; external DLL_Name;
function UDS_SvcAuthenticationAC_2020(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg): uds_status;
  stdcall; external DLL_Name;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  Initialize_2013(channel: cantp_handle; baudrate: cantp_baudrate;
  hw_type: cantp_hwtype; io_port: UInt32; interrupt: UInt16): uds_status;
begin
  result := UDS_Initialize_2013(channel, baudrate, hw_type, io_port, interrupt);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  Initialize_2013(channel: cantp_handle; baudrate: cantp_baudrate): uds_status;
begin
  result := UDS_Initialize_2013(channel, baudrate, cantp_hwtype(0), 0, 0);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  InitializeFD_2013(channel: cantp_handle; const bitrate_fd: cantp_bitrate)
  : uds_status;
begin
  result := UDS_InitializeFD_2013(channel, bitrate_fd);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  Uninitialize_2013(channel: cantp_handle): uds_status;
begin
  result := UDS_Uninitialize_2013(channel);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  Reset_2013(channel: cantp_handle): uds_status;
begin
  result := UDS_Reset_2013(channel);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  GetCanBusStatus_2013(channel: cantp_handle): uds_status;
begin
  result := UDS_GetCanBusStatus_2013(channel);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  Read_2013(channel: cantp_handle; var out_msg_buffer: uds_msg;
  in_msg_request: Puds_msg; out_timestamp: Pcantp_timestamp): uds_status;
begin
  result := UDS_Read_2013(channel, out_msg_buffer, in_msg_request,
    out_timestamp);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  Read_2013(channel: cantp_handle; var out_msg_buffer: uds_msg;
  in_msg_request: Puds_msg): uds_status;
begin
  result := UDS_Read_2013(channel, out_msg_buffer, in_msg_request, nil);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  Read_2013(channel: cantp_handle; var out_msg_buffer: uds_msg): uds_status;
begin
  result := UDS_Read_2013(channel, out_msg_buffer, nil, nil);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  Write_2013(channel: cantp_handle; msg_buffer: Puds_msg): uds_status;
begin
  result := UDS_Write_2013(channel, msg_buffer);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  AddMapping_2013(channel: cantp_handle; mapping: Puds_mapping): uds_status;
begin
  result := UDS_AddMapping_2013(channel, mapping);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  RemoveMappingByCanId_2013(channel: cantp_handle; can_id: UInt32): uds_status;
begin
  result := UDS_RemoveMappingByCanId_2013(channel, can_id);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  RemoveMapping_2013(channel: cantp_handle; mapping: uds_mapping): uds_status;
begin
  result := UDS_RemoveMapping_2013(channel, mapping);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  GetMapping_2013(channel: cantp_handle; var buffer: uds_mapping;
  can_id: UInt32; can_msgtype: cantp_can_msgtype): uds_status;
begin
  result := UDS_GetMapping_2013(channel, buffer, can_id, can_msgtype);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  GetMappings_2013(channel: cantp_handle; buffer: Puds_mapping;
  buffer_length: UInt16; count: PUInt16): uds_status;
begin
  result := UDS_GetMappings_2013(channel, buffer, buffer_length, count);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  AddCanIdFilter_2013(channel: cantp_handle; can_id: UInt32): uds_status;
begin
  result := UDS_AddCanIdFilter_2013(channel, can_id);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  RemoveCanIdFilter_2013(channel: cantp_handle; can_id: UInt32): uds_status;
begin
  result := UDS_RemoveCanIdFilter_2013(channel, can_id);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  GetValue_2013(channel: cantp_handle; parameter: uds_parameter;
  buffer: Pointer; buffer_size: UInt32): uds_status;
begin
  result := UDS_GetValue_2013(channel, parameter, buffer, buffer_size);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  GetValue_2013(channel: cantp_handle; parameter: uds_parameter;
  buffer: PAnsiChar; buffer_size: UInt32): uds_status;
begin
  result := UDS_GetValue_2013(channel, parameter, buffer, buffer_size);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  GetValue_2013(channel: cantp_handle; parameter: uds_parameter;
  buffer: PLongWord; buffer_size: UInt32): uds_status;
begin
  result := UDS_GetValue_2013(channel, parameter, buffer, buffer_size);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  GetValue_2013(channel: cantp_handle; parameter: uds_parameter; buffer: PByte;
  buffer_size: UInt32): uds_status;
begin
  result := UDS_GetValue_2013(channel, parameter, buffer, buffer_size);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  SetValue_2013(channel: cantp_handle; parameter: uds_parameter;
  buffer: Pointer; buffer_size: UInt32): uds_status;
begin
  result := UDS_SetValue_2013(channel, parameter, buffer, buffer_size);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  SetValue_2013(channel: cantp_handle; parameter: uds_parameter;
  buffer: PLongWord; buffer_size: UInt32): uds_status;
begin
  result := UDS_SetValue_2013(channel, parameter, buffer, buffer_size);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  SetValue_2013(channel: cantp_handle; parameter: uds_parameter;
  buffer: PAnsiChar; buffer_size: UInt32): uds_status;
begin
  result := UDS_SetValue_2013(channel, parameter, buffer, buffer_size);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  SetValue_2013(channel: cantp_handle; parameter: uds_parameter; buffer: PByte;
  buffer_size: UInt32): uds_status;
begin
  result := UDS_SetValue_2013(channel, parameter, buffer, buffer_size);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  GetErrorText_2013(error_code: uds_status; language: UInt16; buffer: PAnsiChar;
  buffer_size: UInt32): uds_status;
begin
  result := UDS_GetErrorText_2013(error_code, language, buffer, buffer_size);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  GetSessionInformation_2013(channel: cantp_handle;
  var session_info: uds_sessioninfo): uds_status;
begin
  result := UDS_GetSessionInformation_2013(channel, session_info);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  StatusIsOk_2013(status: uds_status; status_expected: uds_status;
  strict_mode: Boolean): Boolean;
begin
  result := UDS_StatusIsOk_2013(status, status_expected, strict_mode);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  StatusIsOk_2013(status: uds_status; status_expected: uds_status): Boolean;
begin
  result := UDS_StatusIsOk_2013(status, status_expected, False);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  StatusIsOk_2013(status: uds_status): Boolean;
begin
  result := UDS_StatusIsOk_2013(status, uds_status.PUDS_STATUS_OK, False);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  MsgAlloc_2013(var msg_buffer: uds_msg; msg_configuration: uds_msgconfig;
  msg_data_length: UInt32): uds_status;
begin
  result := UDS_MsgAlloc_2013(msg_buffer, msg_configuration, msg_data_length);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  MsgFree_2013(var msg_buffer: uds_msg): uds_status;
begin
  result := UDS_MsgFree_2013(msg_buffer);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  MsgCopy_2013(var msg_buffer_dst: uds_msg; const msg_buffer_src: Puds_msg)
  : uds_status;
begin
  result := UDS_MsgCopy_2013(msg_buffer_dst, msg_buffer_src);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  MsgMove_2013(var msg_buffer_dst: uds_msg; var msg_buffer_src: uds_msg)
  : uds_status;
begin
  result := UDS_MsgMove_2013(msg_buffer_dst, msg_buffer_src);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  WaitForSingleMessage_2013(channel: cantp_handle; msg_request: Puds_msg;
  is_waiting_for_tx: Boolean; timeout: UInt32; timeout_enhanced: UInt32;
  var out_msg_response: uds_msg): uds_status;
begin
  result := UDS_WaitForSingleMessage_2013(channel, msg_request,
    is_waiting_for_tx, timeout, timeout_enhanced, out_msg_response);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  WaitForFunctionalResponses_2013(channel: cantp_handle; msg_request: Puds_msg;
  timeout: UInt32; timeout_enhanced: UInt32; wait_until_timeout: Boolean;
  max_msg_count: UInt32; out_msg_responses: Puds_msg; var out_msg_count: UInt32)
  : uds_status;
begin
  result := UDS_WaitForFunctionalResponses_2013(channel, msg_request, timeout,
    timeout_enhanced, wait_until_timeout, max_msg_count, out_msg_responses,
    out_msg_count);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  WaitForService_2013(channel: cantp_handle; msg_request: Puds_msg;
  var out_msg_response: uds_msg; out_msg_request_confirmation: Puds_msg)
  : uds_status;
begin
  result := UDS_WaitForService_2013(channel, msg_request, out_msg_response,
    out_msg_request_confirmation);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  WaitForServiceFunctional_2013(channel: cantp_handle; msg_request: Puds_msg;
  max_msg_count: UInt32; wait_until_timeout: Boolean;
  out_msg_responses: Puds_msg; var out_msg_count: UInt32;
  out_msg_request_confirmation: Puds_msg): uds_status;
begin
  result := UDS_WaitForServiceFunctional_2013(channel, msg_request,
    max_msg_count, wait_until_timeout, out_msg_responses, out_msg_count,
    out_msg_request_confirmation);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  SvcDiagnosticSessionControl_2013(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  session_type: uds_svc_param_dsc): uds_status;
begin
  result := UDS_SvcDiagnosticSessionControl_2013(channel, request_config,
    out_msg_request, session_type);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  SvcECUReset_2013(channel: cantp_handle; request_config: uds_msgconfig;
  var out_msg_request: uds_msg; reset_type: uds_svc_param_er): uds_status;
begin
  result := UDS_SvcECUReset_2013(channel, request_config, out_msg_request,
    reset_type);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  SvcSecurityAccess_2013(channel: cantp_handle; request_config: uds_msgconfig;
  var out_msg_request: uds_msg; security_access_type: Byte;
  security_access_data: PByte; security_access_data_size: UInt32): uds_status;
begin
  result := UDS_SvcSecurityAccess_2013(channel, request_config, out_msg_request,
    security_access_type, security_access_data, security_access_data_size);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  SvcSecurityAccess_2013(channel: cantp_handle; request_config: uds_msgconfig;
  var out_msg_request: uds_msg; security_access_type: Byte): uds_status;
begin
  result := UDS_SvcSecurityAccess_2013(channel, request_config, out_msg_request,
    security_access_type, nil, 0);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  SvcCommunicationControl_2013(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  control_type: uds_svc_param_cc; communication_type: Byte;
  node_identification_number: UInt16): uds_status;
begin
  result := UDS_SvcCommunicationControl_2013(channel, request_config,
    out_msg_request, control_type, communication_type,
    node_identification_number);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  SvcCommunicationControl_2013(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  control_type: uds_svc_param_cc; communication_type: Byte): uds_status;
begin
  result := UDS_SvcCommunicationControl_2013(channel, request_config,
    out_msg_request, control_type, communication_type, 0);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  SvcTesterPresent_2013(channel: cantp_handle; request_config: uds_msgconfig;
  var out_msg_request: uds_msg; testerpresent_type: uds_svc_param_tp)
  : uds_status;
begin
  result := UDS_SvcTesterPresent_2013(channel, request_config, out_msg_request,
    testerpresent_type);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  SvcTesterPresent_2013(channel: cantp_handle; request_config: uds_msgconfig;
  var out_msg_request: uds_msg): uds_status;
begin
  result := UDS_SvcTesterPresent_2013(channel, request_config, out_msg_request,
    uds_svc_param_tp.PUDS_SVC_PARAM_TP_ZSUBF);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  SvcSecuredDataTransmission_2013(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  security_data_request_record: PByte;
  security_data_request_record_size: UInt32): uds_status;
begin
  result := UDS_SvcSecuredDataTransmission_2013(channel, request_config,
    out_msg_request, security_data_request_record,
    security_data_request_record_size);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  SvcSecuredDataTransmission_2020(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  administrative_parameter: UInt16; signature_encryption_calculation: Byte;
  anti_replay_counter: UInt16; internal_service_identifier: Byte;
  service_specific_parameters: PByte; service_specific_parameters_size: UInt32;
  signature_mac: PByte; signature_size: UInt16): uds_status;
begin
  result := UDS_SvcSecuredDataTransmission_2020(channel, request_config,
    out_msg_request, administrative_parameter, signature_encryption_calculation,
    anti_replay_counter, internal_service_identifier,
    service_specific_parameters, service_specific_parameters_size,
    signature_mac, signature_size);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  SvcControlDTCSetting_2013(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  dtc_setting_type: uds_svc_param_cdtcs;
  dtc_setting_control_option_record: PByte;
  dtc_setting_control_option_record_size: UInt32): uds_status;
begin
  result := UDS_SvcControlDTCSetting_2013(channel, request_config,
    out_msg_request, dtc_setting_type, dtc_setting_control_option_record,
    dtc_setting_control_option_record_size);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  SvcControlDTCSetting_2013(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  dtc_setting_type: uds_svc_param_cdtcs): uds_status;
begin
  result := UDS_SvcControlDTCSetting_2013(channel, request_config,
    out_msg_request, dtc_setting_type, nil, 0);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  SvcResponseOnEvent_2013(channel: cantp_handle; request_config: uds_msgconfig;
  var out_msg_request: uds_msg; event_type: uds_svc_param_roe;
  store_event: Boolean; event_window_time: Byte; event_type_record: PByte;
  event_type_record_size: UInt32; service_to_respond_to_record: PByte;
  service_to_respond_to_record_size: UInt32): uds_status;
begin
  result := UDS_SvcResponseOnEvent_2013(channel, request_config,
    out_msg_request, event_type, store_event, event_window_time,
    event_type_record, event_type_record_size, service_to_respond_to_record,
    service_to_respond_to_record_size);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  SvcResponseOnEvent_2013(channel: cantp_handle; request_config: uds_msgconfig;
  var out_msg_request: uds_msg; event_type: uds_svc_param_roe;
  store_event: Boolean; event_window_time: Byte; event_type_record: PByte;
  event_type_record_size: UInt32): uds_status;
begin
  result := UDS_SvcResponseOnEvent_2013(channel, request_config,
    out_msg_request, event_type, store_event, event_window_time,
    event_type_record, event_type_record_size, nil, 0);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  SvcResponseOnEvent_2013(channel: cantp_handle; request_config: uds_msgconfig;
  var out_msg_request: uds_msg; event_type: uds_svc_param_roe;
  store_event: Boolean; event_window_time: Byte): uds_status;
begin
  result := UDS_SvcResponseOnEvent_2013(channel, request_config,
    out_msg_request, event_type, store_event, event_window_time,
    nil, 0, nil, 0);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  SvcLinkControl_2013(channel: cantp_handle; request_config: uds_msgconfig;
  var out_msg_request: uds_msg; link_control_type: uds_svc_param_lc;
  baudrate_identifier: uds_svc_param_lc_baudrate_identifier;
  link_baudrate: UInt32): uds_status;
begin
  result := UDS_SvcLinkControl_2013(channel, request_config, out_msg_request,
    link_control_type, baudrate_identifier, link_baudrate);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  SvcLinkControl_2013(channel: cantp_handle; request_config: uds_msgconfig;
  var out_msg_request: uds_msg; link_control_type: uds_svc_param_lc;
  baudrate_identifier: uds_svc_param_lc_baudrate_identifier): uds_status;
begin
  result := UDS_SvcLinkControl_2013(channel, request_config, out_msg_request,
    link_control_type, baudrate_identifier, 0);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  SvcReadDataByIdentifier_2013(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  data_identifier: Puds_svc_param_di; data_identifier_length: UInt32)
  : uds_status;
begin
  result := UDS_SvcReadDataByIdentifier_2013(channel, request_config,
    out_msg_request, data_identifier, data_identifier_length);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  SvcReadMemoryByAddress_2013(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  memory_address_buffer: PByte; memory_address_size: Byte;
  memory_size_buffer: PByte; memory_size_size: Byte): uds_status;
begin
  result := UDS_SvcReadMemoryByAddress_2013(channel, request_config,
    out_msg_request, memory_address_buffer, memory_address_size,
    memory_size_buffer, memory_size_size);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  SvcReadScalingDataByIdentifier_2013(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  data_identifier: uds_svc_param_di): uds_status;
begin
  result := UDS_SvcReadScalingDataByIdentifier_2013(channel, request_config,
    out_msg_request, data_identifier);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  SvcReadDataByPeriodicIdentifier_2013(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  transmission_mode: uds_svc_param_rdbpi; periodic_data_identifier: PByte;
  periodic_data_identifier_size: UInt32): uds_status;
begin
  result := UDS_SvcReadDataByPeriodicIdentifier_2013(channel, request_config,
    out_msg_request, transmission_mode, periodic_data_identifier,
    periodic_data_identifier_size);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  SvcDynamicallyDefineDataIdentifierDBID_2013(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  dynamically_defined_data_identifier: uds_svc_param_di;
  source_data_identifier: PUInt16; memory_size: PByte;
  position_in_source_data_record: PByte; number_of_elements: UInt32)
  : uds_status;
begin
  result := UDS_SvcDynamicallyDefineDataIdentifierDBID_2013(channel,
    request_config, out_msg_request, dynamically_defined_data_identifier,
    source_data_identifier, memory_size, position_in_source_data_record,
    number_of_elements);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  SvcDynamicallyDefineDataIdentifierDBMA_2013(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  dynamically_defined_data_identifier: uds_svc_param_di;
  memory_address_size: Byte; memory_size_size: Byte;
  memory_address_buffer: PByte; memory_size_buffer: PByte;
  number_of_elements: UInt32): uds_status;
begin
  result := UDS_SvcDynamicallyDefineDataIdentifierDBMA_2013(channel,
    request_config, out_msg_request, dynamically_defined_data_identifier,
    memory_address_size, memory_size_size, memory_address_buffer,
    memory_size_buffer, number_of_elements);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  SvcDynamicallyDefineDataIdentifierCDDDI_2013(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  dynamically_defined_data_identifier: uds_svc_param_di): uds_status;
begin
  result := UDS_SvcDynamicallyDefineDataIdentifierCDDDI_2013(channel,
    request_config, out_msg_request, dynamically_defined_data_identifier);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  SvcDynamicallyDefineDataIdentifierClearAllDDDI_2013(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg): uds_status;
begin
  result := UDS_SvcDynamicallyDefineDataIdentifierClearAllDDDI_2013(channel,
    request_config, out_msg_request);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}

class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  SvcWriteDataByIdentifier_2013(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  data_identifier: uds_svc_param_di; data_record: PByte;
  data_record_size: UInt32): uds_status;
begin
  result := UDS_SvcWriteDataByIdentifier_2013(channel, request_config,
    out_msg_request, data_identifier, data_record, data_record_size);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  SvcWriteMemoryByAddress_2013(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  memory_address_buffer: PByte; memory_address_size: Byte;
  memory_size_buffer: PByte; memory_size_size: Byte; data_record: PByte;
  data_record_size: UInt32): uds_status;
begin
  result := UDS_SvcWriteMemoryByAddress_2013(channel, request_config,
    out_msg_request, memory_address_buffer, memory_address_size,
    memory_size_buffer, memory_size_size, data_record, data_record_size);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  SvcClearDiagnosticInformation_2013(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  group_of_dtc: UInt32): uds_status;
begin
  result := UDS_SvcClearDiagnosticInformation_2013(channel, request_config,
    out_msg_request, group_of_dtc);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  SvcClearDiagnosticInformation_2020(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  group_of_dtc: UInt32; memory_selection: Byte): uds_status;
begin
  result := UDS_SvcClearDiagnosticInformation_2020(channel, request_config,
    out_msg_request, group_of_dtc, memory_selection);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  SvcReadDTCInformation_2013(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  PUDS_SVC_PARAM_RDTCI_Type: uds_svc_param_rdtci; dtc_status_mask: Byte)
  : uds_status;
begin
  result := UDS_SvcReadDTCInformation_2013(channel, request_config,
    out_msg_request, PUDS_SVC_PARAM_RDTCI_Type, dtc_status_mask);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  SvcReadDTCInformationRDTCSSBDTC_2013(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg; dtc_mask: UInt32;
  dtc_snapshot_record_number: Byte): uds_status;
begin
  result := UDS_SvcReadDTCInformationRDTCSSBDTC_2013(channel, request_config,
    out_msg_request, dtc_mask, dtc_snapshot_record_number);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  SvcReadDTCInformationRDTCSSBRN_2013(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  dtc_snapshot_record_number: Byte): uds_status;
begin
  result := UDS_SvcReadDTCInformationRDTCSSBRN_2013(channel, request_config,
    out_msg_request, dtc_snapshot_record_number);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  SvcReadDTCInformationReportExtended_2013(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  PUDS_SVC_PARAM_RDTCI_Type: uds_svc_param_rdtci; dtc_mask: UInt32;
  dtc_extended_data_record_number: Byte): uds_status;
begin
  result := UDS_SvcReadDTCInformationReportExtended_2013(channel,
    request_config, out_msg_request, PUDS_SVC_PARAM_RDTCI_Type, dtc_mask,
    dtc_extended_data_record_number);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  SvcReadDTCInformationReportSeverity_2013(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  PUDS_SVC_PARAM_RDTCI_Type: uds_svc_param_rdtci; dtc_severity_mask: Byte;
  dtc_status_mask: Byte): uds_status;
begin
  result := UDS_SvcReadDTCInformationReportSeverity_2013(channel,
    request_config, out_msg_request, PUDS_SVC_PARAM_RDTCI_Type,
    dtc_severity_mask, dtc_status_mask);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  SvcReadDTCInformationRSIODTC_2013(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg; dtc_mask: UInt32)
  : uds_status;
begin
  result := UDS_SvcReadDTCInformationRSIODTC_2013(channel, request_config,
    out_msg_request, dtc_mask);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  SvcReadDTCInformationNoParam_2013(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  PUDS_SVC_PARAM_RDTCI_Type: uds_svc_param_rdtci): uds_status;
begin
  result := UDS_SvcReadDTCInformationNoParam_2013(channel, request_config,
    out_msg_request, PUDS_SVC_PARAM_RDTCI_Type);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  SvcReadDTCInformationRDTCEDBR_2013(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  dtc_extended_data_record_number: Byte): uds_status;
begin
  result := UDS_SvcReadDTCInformationRDTCEDBR_2013(channel, request_config,
    out_msg_request, dtc_extended_data_record_number);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  SvcReadDTCInformationRUDMDTCBSM_2013(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  dtc_status_mask: Byte; memory_selection: Byte): uds_status;
begin
  result := UDS_SvcReadDTCInformationRUDMDTCBSM_2013(channel, request_config,
    out_msg_request, dtc_status_mask, memory_selection);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  SvcReadDTCInformationRUDMDTCSSBDTC_2013(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg; dtc_mask: UInt32;
  user_def_dtc_snapshot_record_number: Byte; memory_selection: Byte)
  : uds_status;
begin
  result := UDS_SvcReadDTCInformationRUDMDTCSSBDTC_2013(channel, request_config,
    out_msg_request, dtc_mask, user_def_dtc_snapshot_record_number,
    memory_selection);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  SvcReadDTCInformationRUDMDTCEDRBDN_2013(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg; dtc_mask: UInt32;
  dtc_extended_data_record_number: Byte; memory_selection: Byte): uds_status;
begin
  result := UDS_SvcReadDTCInformationRUDMDTCEDRBDN_2013(channel, request_config,
    out_msg_request, dtc_mask, dtc_extended_data_record_number,
    memory_selection);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  SvcReadDTCInformationRDTCEDI_2020(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  dtc_extended_data_record_number: Byte): uds_status;
begin
  result := UDS_SvcReadDTCInformationRDTCEDI_2020(channel, request_config,
    out_msg_request, dtc_extended_data_record_number);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  SvcReadDTCInformationRWWHOBDDTCBMR_2013(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  functional_group_identifier: Byte; dtc_status_mask: Byte;
  dtc_severity_mask: Byte): uds_status;
begin
  result := UDS_SvcReadDTCInformationRWWHOBDDTCBMR_2013(channel, request_config,
    out_msg_request, functional_group_identifier, dtc_status_mask,
    dtc_severity_mask);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  SvcReadDTCInformationRWWHOBDDTCWPS_2013(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  functional_group_identifier: Byte): uds_status;
begin
  result := UDS_SvcReadDTCInformationRWWHOBDDTCWPS_2013(channel, request_config,
    out_msg_request, functional_group_identifier);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  SvcReadDTCInformationRDTCBRGI_2020(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  functional_group_identifier: Byte; dtc_readiness_group_identifier: Byte)
  : uds_status;
begin
  result := UDS_SvcReadDTCInformationRDTCBRGI_2020(channel, request_config,
    out_msg_request, functional_group_identifier,
    dtc_readiness_group_identifier);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  SvcInputOutputControlByIdentifier_2013(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  data_identifier: uds_svc_param_di; control_option_record: PByte;
  control_option_record_size: UInt32; control_enable_mask_record: PByte;
  control_enable_mask_record_size: UInt32): uds_status;
begin
  result := UDS_SvcInputOutputControlByIdentifier_2013(channel, request_config,
    out_msg_request, data_identifier, control_option_record,
    control_option_record_size, control_enable_mask_record,
    control_enable_mask_record_size);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  SvcInputOutputControlByIdentifier_2013(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  data_identifier: uds_svc_param_di; control_option_record: PByte;
  control_option_record_size: UInt32): uds_status;
begin
  result := UDS_SvcInputOutputControlByIdentifier_2013(channel, request_config,
    out_msg_request, data_identifier, control_option_record,
    control_option_record_size, nil, 0);
end;


{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  SvcRoutineControl_2013(channel: cantp_handle; request_config: uds_msgconfig;
  var out_msg_request: uds_msg; routine_control_type: uds_svc_param_rc;
  routine_identifier: uds_svc_param_rc_rid;
  routine_control_option_record: PByte;
  routine_control_option_record_size: UInt32): uds_status;
begin
  result := UDS_SvcRoutineControl_2013(channel, request_config, out_msg_request,
    routine_control_type, routine_identifier, routine_control_option_record,
    routine_control_option_record_size);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  SvcRoutineControl_2013(channel: cantp_handle; request_config: uds_msgconfig;
  var out_msg_request: uds_msg; routine_control_type: uds_svc_param_rc;
  routine_identifier: uds_svc_param_rc_rid): uds_status;
begin
  result := UDS_SvcRoutineControl_2013(channel, request_config, out_msg_request,
    routine_control_type, routine_identifier, nil, 0);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  SvcRequestDownload_2013(channel: cantp_handle; request_config: uds_msgconfig;
  var out_msg_request: uds_msg; compression_method: Byte;
  encrypting_method: Byte; memory_address_buffer: PByte;
  memory_address_size: Byte; memory_size_buffer: PByte; memory_size_size: Byte)
  : uds_status;
begin
  result := UDS_SvcRequestDownload_2013(channel, request_config,
    out_msg_request, compression_method, encrypting_method,
    memory_address_buffer, memory_address_size, memory_size_buffer,
    memory_size_size);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  SvcRequestUpload_2013(channel: cantp_handle; request_config: uds_msgconfig;
  var out_msg_request: uds_msg; compression_method: Byte;
  encrypting_method: Byte; memory_address_buffer: PByte;
  memory_address_size: Byte; memory_size_buffer: PByte; memory_size_size: Byte)
  : uds_status;
begin
  result := UDS_SvcRequestUpload_2013(channel, request_config, out_msg_request,
    compression_method, encrypting_method, memory_address_buffer,
    memory_address_size, memory_size_buffer, memory_size_size);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  SvcTransferData_2013(channel: cantp_handle; request_config: uds_msgconfig;
  var out_msg_request: uds_msg; block_sequence_counter: Byte;
  transfer_request_parameter_record: PByte;
  transfer_request_parameter_record_size: UInt32): uds_status;
begin
  result := UDS_SvcTransferData_2013(channel, request_config, out_msg_request,
    block_sequence_counter, transfer_request_parameter_record,
    transfer_request_parameter_record_size);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  SvcTransferData_2013(channel: cantp_handle; request_config: uds_msgconfig;
  var out_msg_request: uds_msg; block_sequence_counter: Byte): uds_status;
begin
  result := UDS_SvcTransferData_2013(channel, request_config, out_msg_request,
    block_sequence_counter, nil, 0);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  SvcRequestTransferExit_2013(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  transfer_request_parameter_record: PByte;
  transfer_request_parameter_record_size: UInt32): uds_status;
begin
  result := UDS_SvcRequestTransferExit_2013(channel, request_config,
    out_msg_request, transfer_request_parameter_record,
    transfer_request_parameter_record_size);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  SvcRequestTransferExit_2013(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg
  ): uds_status;
begin
  result := UDS_SvcRequestTransferExit_2013(channel, request_config,
    out_msg_request, nil, 0);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  SvcAccessTimingParameter_2013(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  access_type: uds_svc_param_atp; request_record: PByte;
  request_record_size: UInt32): uds_status;
begin
  result := UDS_SvcAccessTimingParameter_2013(channel, request_config,
    out_msg_request, access_type, request_record, request_record_size);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  SvcRequestFileTransfer_2013(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  mode_of_operation: uds_svc_param_rft_moop; file_path_and_name_size: UInt16;
  file_path_and_name: PByte; compression_method: Byte; encrypting_method: Byte;
  file_size_parameter_size: Byte; file_size_uncompressed: PByte;
  file_size_compressed: PByte): uds_status;
begin
  result := UDS_SvcRequestFileTransfer_2013(channel, request_config,
    out_msg_request, mode_of_operation, file_path_and_name_size,
    file_path_and_name, compression_method, encrypting_method,
    file_size_parameter_size, file_size_uncompressed, file_size_compressed);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  SvcRequestFileTransfer_2013(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  mode_of_operation: uds_svc_param_rft_moop; file_path_and_name_size: UInt16;
  file_path_and_name: PAnsiChar; compression_method: Byte;
  encrypting_method: Byte; file_size_parameter_size: Byte;
  file_size_uncompressed: PByte; file_size_compressed: PByte): uds_status;
begin
  result := UDS_SvcRequestFileTransfer_2013(channel, request_config,
    out_msg_request, mode_of_operation, file_path_and_name_size,
    file_path_and_name, compression_method, encrypting_method,
    file_size_parameter_size, file_size_uncompressed, file_size_compressed);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  SvcRequestFileTransfer_2013(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  mode_of_operation: uds_svc_param_rft_moop; file_path_and_name_size: UInt16;
  file_path_and_name: PByte; compression_method: Byte; encrypting_method: Byte)
  : uds_status;
begin
  result := UDS_SvcRequestFileTransfer_2013(channel, request_config,
    out_msg_request, mode_of_operation, file_path_and_name_size,
    file_path_and_name, compression_method, encrypting_method, 0, nil, nil);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  SvcRequestFileTransfer_2013(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  mode_of_operation: uds_svc_param_rft_moop; file_path_and_name_size: UInt16;
  file_path_and_name: PAnsiChar; compression_method: Byte;
  encrypting_method: Byte): uds_status;
begin
  result := UDS_SvcRequestFileTransfer_2013(channel, request_config,
    out_msg_request, mode_of_operation, file_path_and_name_size,
    file_path_and_name, compression_method, encrypting_method, 0, nil, nil);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  SvcRequestFileTransfer_2013(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  mode_of_operation: uds_svc_param_rft_moop; file_path_and_name_size: UInt16;
  file_path_and_name: PByte): uds_status;
begin
  result := UDS_SvcRequestFileTransfer_2013(channel, request_config,
    out_msg_request, mode_of_operation, file_path_and_name_size,
    file_path_and_name, 0, 0, 0, nil, nil);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  SvcRequestFileTransfer_2013(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  mode_of_operation: uds_svc_param_rft_moop; file_path_and_name_size: UInt16;
  file_path_and_name: PAnsiChar): uds_status;
begin
  result := UDS_SvcRequestFileTransfer_2013(channel, request_config,
    out_msg_request, mode_of_operation, file_path_and_name_size,
    file_path_and_name, 0, 0, 0, nil, nil);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  SvcAuthenticationDA_2020(channel: cantp_handle; request_config: uds_msgconfig;
  var out_msg_request: uds_msg): uds_status;
begin
  result := UDS_SvcAuthenticationDA_2020(channel, request_config,
    out_msg_request);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  SvcAuthenticationVCU_2020(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  communication_configuration: Byte; certificate_client: PByte;
  certificate_client_size: UInt16; challenge_client: PByte;
  challenge_client_size: UInt16): uds_status;
begin
  result := UDS_SvcAuthenticationVCU_2020(channel, request_config,
    out_msg_request, communication_configuration, certificate_client,
    certificate_client_size, challenge_client, challenge_client_size);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  SvcAuthenticationVCU_2020(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  communication_configuration: Byte; certificate_client: PByte;
  certificate_client_size: UInt16): uds_status;
begin
  result := UDS_SvcAuthenticationVCU_2020(channel, request_config,
    out_msg_request, communication_configuration, certificate_client,
    certificate_client_size, nil, 0);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  SvcAuthenticationVCB_2020(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  communication_configuration: Byte; certificate_client: PByte;
  certificate_client_size: UInt16; challenge_client: PByte;
  challenge_client_size: UInt16): uds_status;
begin
  result := UDS_SvcAuthenticationVCB_2020(channel, request_config,
    out_msg_request, communication_configuration, certificate_client,
    certificate_client_size, challenge_client, challenge_client_size);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  SvcAuthenticationPOWN_2020(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  proof_of_ownership_client: PByte; proof_of_ownership_client_size: UInt16;
  ephemeral_public_key_client: PByte; ephemeral_public_key_client_size: UInt16)
  : uds_status;
begin
  result := UDS_SvcAuthenticationPOWN_2020(channel, request_config,
    out_msg_request, proof_of_ownership_client, proof_of_ownership_client_size,
    ephemeral_public_key_client, ephemeral_public_key_client_size);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  SvcAuthenticationPOWN_2020(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  proof_of_ownership_client: PByte; proof_of_ownership_client_size: UInt16)
  : uds_status;
begin
  result := UDS_SvcAuthenticationPOWN_2020(channel, request_config,
    out_msg_request, proof_of_ownership_client,
    proof_of_ownership_client_size, nil, 0);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  SvcAuthenticationRCFA_2020(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  communication_configuration: Byte; algorithm_indicator: PByte): uds_status;
begin
  result := UDS_SvcAuthenticationRCFA_2020(channel, request_config,
    out_msg_request, communication_configuration, algorithm_indicator);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  SvcAuthenticationVPOWNU_2020(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  algorithm_indicator: PByte; proof_of_ownership_client: PByte;
  proof_of_ownership_client_size: UInt16; challenge_client: PByte;
  challenge_client_size: UInt16; additional_parameter: PByte;
  additional_parameter_size: UInt16): uds_status;
begin
  result := UDS_SvcAuthenticationVPOWNU_2020(channel, request_config,
    out_msg_request, algorithm_indicator, proof_of_ownership_client,
    proof_of_ownership_client_size, challenge_client, challenge_client_size,
    additional_parameter, additional_parameter_size);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  SvcAuthenticationVPOWNU_2020(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  algorithm_indicator: PByte; proof_of_ownership_client: PByte;
  proof_of_ownership_client_size: UInt16; challenge_client: PByte;
  challenge_client_size: UInt16): uds_status;
begin
  result := UDS_SvcAuthenticationVPOWNU_2020(channel, request_config,
    out_msg_request, algorithm_indicator, proof_of_ownership_client,
    proof_of_ownership_client_size, challenge_client,
    challenge_client_size, nil, 0);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  SvcAuthenticationVPOWNU_2020(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  algorithm_indicator: PByte; proof_of_ownership_client: PByte;
  proof_of_ownership_client_size: UInt16): uds_status;
begin
  result := UDS_SvcAuthenticationVPOWNU_2020(channel, request_config,
    out_msg_request, algorithm_indicator, proof_of_ownership_client,
    proof_of_ownership_client_size, nil, 0, nil, 0);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  SvcAuthenticationVPOWNB_2020(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  algorithm_indicator: PByte; proof_of_ownership_client: PByte;
  proof_of_ownership_client_size: UInt16; challenge_client: PByte;
  challenge_client_size: UInt16; additional_parameter: PByte;
  additional_parameter_size: UInt16): uds_status;
begin
  result := UDS_SvcAuthenticationVPOWNB_2020(channel, request_config,
    out_msg_request, algorithm_indicator, proof_of_ownership_client,
    proof_of_ownership_client_size, challenge_client, challenge_client_size,
    additional_parameter, additional_parameter_size);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  SvcAuthenticationVPOWNB_2020(channel: cantp_handle;
  request_config: uds_msgconfig; var out_msg_request: uds_msg;
  algorithm_indicator: PByte; proof_of_ownership_client: PByte;
  proof_of_ownership_client_size: UInt16; challenge_client: PByte;
  challenge_client_size: UInt16): uds_status;
begin
  result := UDS_SvcAuthenticationVPOWNB_2020(channel, request_config,
    out_msg_request, algorithm_indicator, proof_of_ownership_client,
    proof_of_ownership_client_size, challenge_client,
    challenge_client_size, nil, 0);
end;

{$IF Defined(PUDS_API_COMPATIBILITY_ISO_2006)}
class function TUDSApi_2013.
{$ELSE}
class function TUDSApi.
{$ENDIF}
  SvcAuthenticationAC_2020(channel: cantp_handle; request_config: uds_msgconfig;
  var out_msg_request: uds_msg): uds_status;
begin
  result := UDS_SvcAuthenticationAC_2020(channel, request_config,
    out_msg_request);
end;

end.
