﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

using Peak.Can.IsoTp;
using Peak.Can.Uds;
using System.Runtime.InteropServices;

namespace server_response_ecu_reset
{
    class Program
    {
        /// <summary>Entry point of the program, start a small server wich only support ECUReset service</summary>
        static void Main(string[] args)
        {
            uds_status status;
            cantp_handle tp_handle;
            UInt32 server_address;
            const int BUFFER_SIZE = 256;
            StringBuilder buffer = new StringBuilder(BUFFER_SIZE);
            uds_msgconfig config_physical = new uds_msgconfig();
            bool stop;
            bool wait_result;
            uds_status read_status;
            uds_msg request_msg = new uds_msg();
            uds_msg response_msg = new uds_msg();

            // Initialize variables
            tp_handle = cantp_handle.PCANTP_HANDLE_USBBUS2; // TODO: modify the value according to your available PCAN devices.
            server_address = (UInt32)uds_address.PUDS_ADDRESS_ISO_15765_4_ADDR_ECU_1;

            // Print version informations
            status = UDSApi.GetValue_2013(cantp_handle.PCANTP_HANDLE_NONEBUS, uds_parameter.PUDS_PARAMETER_API_VERSION, buffer, BUFFER_SIZE);
            Console.WriteLine("PCAN-UDS API Version - {0}: {1}", buffer, STATUS_OK_KO(status));

            // Initialize server
            status = UDSApi.Initialize_2013(tp_handle, cantp_baudrate.PCANTP_BAUDRATE_500K);
            Console.WriteLine("Initialize channel: {0}", STATUS_OK_KO(status));

            // Set server address parameter
            status = UDSApi.SetValue_2013(tp_handle, uds_parameter.PUDS_PARAMETER_SERVER_ADDRESS, ref server_address, sizeof(UInt32));
            Console.WriteLine("Set server address: {0}", STATUS_OK_KO(status));

            // Set a receive event
            System.Threading.AutoResetEvent receive_event = new System.Threading.AutoResetEvent(false);
            if (IntPtr.Size == 4)
            {
                UInt32 tmp_buffer = Convert.ToUInt32(receive_event.SafeWaitHandle.DangerousGetHandle().ToInt32());
                status = UDSApi.SetValue_2013(tp_handle, uds_parameter.PUDS_PARAMETER_RECEIVE_EVENT, ref tmp_buffer, sizeof(UInt32));
            }
            else if (IntPtr.Size == 8)
            {
                Int64 tmp_buffer = receive_event.SafeWaitHandle.DangerousGetHandle().ToInt64();
                byte[] byte_array = BitConverter.GetBytes(tmp_buffer);
                status = UDSApi.SetValue_2013(tp_handle, uds_parameter.PUDS_PARAMETER_RECEIVE_EVENT, byte_array, sizeof(UInt64));
            }
            Console.WriteLine("Set receive event parameter: {0}", STATUS_OK_KO(status));

            // Initialize a physical configuration
            config_physical.can_id = (UInt32)(0xFFFFFFFF);
            config_physical.can_msgtype = cantp_can_msgtype.PCANTP_CAN_MSGTYPE_STANDARD;
            config_physical.nai.protocol = uds_msgprotocol.PUDS_MSGPROTOCOL_ISO_15765_2_11B_NORMAL;
            config_physical.nai.target_type = cantp_isotp_addressing.PCANTP_ISOTP_ADDRESSING_PHYSICAL;
            config_physical.type = uds_msgtype.PUDS_MSGTYPE_USDT;
            config_physical.nai.extension_addr = 0;

            // Read while user do not press Q
            Console.WriteLine("Start listening, press Q to quit.");
            stop = false;
            do
            {

                // Wait a receive event on receiver
                //	note: timeout is used to check keyboard hit.
                wait_result = receive_event.WaitOne(1000);

                // If we get a receive event
                if (wait_result)
                {
                    do
                    {
                        // Read first available message (no filtering based on message's type is set):
                        read_status = UDSApi.Read_2013(tp_handle, out request_msg);
                        Console.WriteLine("Try to read a message: {0}", STATUS_OK_KO(read_status));
                        if (UDSApi.StatusIsOk_2013(read_status))
                        {
                            cantp_msgdata req_any = new cantp_msgdata();
                            if (request_msg.msg.Msgdata != IntPtr.Zero)
                                req_any = request_msg.msg.Msgdata_any_Copy;

                            // We receive a request, check its length and if it is not a loopback message
                            if (req_any.length >= 1 && (req_any.flags & cantp_msgflag.PCANTP_MSGFLAG_LOOPBACK) == 0)
                            {
                                byte req_srv_id;
                                UDSApi.GetDataServiceId_2013(ref request_msg, out req_srv_id);
                                // This is a valid request, switch services
                                switch (req_srv_id)
                                {
                                    case (byte)uds_service.PUDS_SERVICE_SI_ECUReset:

                                        // Allocate response message
                                        Byte param0;
                                        UDSApi.GetDataParameter_2013(ref request_msg, 0, out param0);

                                        if (param0 == (Byte)UDSApi.uds_svc_param_er.PUDS_SVC_PARAM_ER_ERPSD)
                                        {
                                            status = UDSApi.MsgAlloc_2013(out response_msg, config_physical, 3);
                                            UDSApi.SetDataParameter_2013(ref response_msg, 1, (Byte)0x66); // Power down time
                                        }
                                        else
                                        {
                                            status = UDSApi.MsgAlloc_2013(out response_msg, config_physical, 2);
                                        }
                                        Console.WriteLine("Prepare response message for ECUReset service: {0}", STATUS_OK_KO(status));

                                        if (UDSApi.StatusIsOk_2013(status))
                                        {
                                            // Fill parameters
                                            cantp_netaddrinfo net_addr_info = request_msg.msg.Msgdata_isotp_Copy.netaddrinfo;
                                            net_addr_info.target_addr = net_addr_info.source_addr;
                                            net_addr_info.source_addr = (UInt16)server_address;
                                            CanTpApi.setNetaddrinfo_2016(ref response_msg.msg, ref net_addr_info);
                                            UDSApi.SetDataServiceId_2013(ref response_msg, (Byte)uds_service.PUDS_SERVICE_SI_ECUReset + UDSApi.PUDS_SI_POSITIVE_RESPONSE);
                                            UDSApi.SetDataParameter_2013(ref response_msg, 0, param0);

                                            // Write response message
                                            status = UDSApi.Write_2013(tp_handle, ref response_msg);
                                            Console.WriteLine("Write response message for ECUReset service: {0}", STATUS_OK_KO(status));
                                        }

                                        // Free response message (and clean memory in order to reallocate later)
                                        status = UDSApi.MsgFree_2013(ref response_msg);
                                        Console.WriteLine("Free response message: {0}", STATUS_OK_KO(status));
                                        break;
                                    default:
                                        Console.WriteLine("Unknown service (0x{0:X})", req_srv_id);
                                        break;
                                }
                            }
                        }

                        // Free request message (and clean memory in order to reallocate later)
                        status = UDSApi.MsgFree_2013(ref request_msg);
                        Console.WriteLine("Free request message: {0}", STATUS_OK_KO(status));
                    } while (!UDSApi.StatusIsOk_2013(read_status, uds_status.PUDS_STATUS_NO_MESSAGE));
                }

                // Quit when user press Q
                if (Console.KeyAvailable)
                {
                    char keyboard_res = Console.ReadKey().KeyChar;
                    if (keyboard_res == 'Q' || keyboard_res == 'q')
                    {
                        stop = true;
                    }
                }
            } while (stop == false);

            // Close receive event
            if (IntPtr.Size == 4)
            {
                UInt32 tmp_buffer = 0;
                status = UDSApi.SetValue_2013(tp_handle, uds_parameter.PUDS_PARAMETER_RECEIVE_EVENT, ref tmp_buffer, sizeof(UInt32));
            }
            else if (IntPtr.Size == 8)
            {
                Int64 tmp_buffer = 0;
                byte[] byte_array = BitConverter.GetBytes(tmp_buffer);
                status = UDSApi.SetValue_2013(tp_handle, uds_parameter.PUDS_PARAMETER_RECEIVE_EVENT, byte_array, sizeof(UInt64));
            }
            Console.WriteLine("Stop receive event: {0}", STATUS_OK_KO(status));
            receive_event.Close();
            Console.WriteLine("Close receive event: {0}", STATUS_OK_KO(status));

            // Close server
            status = UDSApi.Uninitialize_2013(tp_handle);
            Console.WriteLine("Uninitialize channel: {0}", STATUS_OK_KO(status));

            // Exit
            Console.WriteLine("Press any key to continue...");
            Console.In.Read();
        }


        static String OK_KO(bool test)
        {
            if (test)
                return "OK";
            return "KO";
        }
        static String STATUS_OK_KO(uds_status test)
        {
            return OK_KO(UDSApi.StatusIsOk_2013(test));
        }
    }
}
