﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

using Peak.Can.IsoTp;
using Peak.Can.Uds;
using System.Runtime.InteropServices;

namespace client_request_ecu_reset
{
    class Program
    {
        static bool FORCE_CUSTOM_SESSION = true;

        /// <summary>Helper: check if two UDS session info structure are the same</summary>
        /// <param name="session_1">Session to compare</param>
        /// <param name="session_2">Session to compare</param>
        /// <returns>Same session info (true) or not (false)</returns>
        static bool is_same_uds_sessioninfo(ref uds_sessioninfo session_1, ref uds_sessioninfo session_2)
        {
            bool res = false;
            if (session_1.nai.protocol == session_2.nai.protocol
                && session_1.nai.target_type == session_2.nai.target_type
                && session_1.nai.source_addr == session_2.nai.source_addr
                && session_1.nai.target_addr == session_2.nai.target_addr
                && session_1.nai.extension_addr == session_2.nai.extension_addr
                && session_1.can_msg_type == session_2.can_msg_type
                && session_1.session_type == session_2.session_type
                && session_1.timeout_p2can_server_max == session_2.timeout_p2can_server_max
                && session_1.timeout_enhanced_p2can_server_max == session_2.timeout_enhanced_p2can_server_max
                && session_1.s3_client_ms == session_2.s3_client_ms)
            {
                res = true;
            }
            return res;
        }

        /// <summary>Entry point of the program, start a UDS channel, ask ECureset service and set new session info</summary>
        static void Main(string[] args)
        {

            const int BUFFER_SIZE = 256;
            StringBuilder buffer = new StringBuilder(BUFFER_SIZE);
            uds_status status;
            cantp_handle tp_handle;
            uds_msgconfig config_physical = new uds_msgconfig();
            uds_msg msg_request = new uds_msg();
            uds_msg request_confirmation = new uds_msg();
            uds_msg response = new uds_msg();
            uds_sessioninfo new_sessioninfo = new uds_sessioninfo();
            uds_sessioninfo read_sessioninfo = new uds_sessioninfo();

            // Initialize variables
            tp_handle = cantp_handle.PCANTP_HANDLE_USBBUS1; // TODO: modify the value according to your available PCAN devices.

            // Print version informations
            status = UDSApi.GetValue_2013(cantp_handle.PCANTP_HANDLE_NONEBUS, uds_parameter.PUDS_PARAMETER_API_VERSION, buffer, BUFFER_SIZE);
            Console.WriteLine("PCAN-UDS API Version - {0}: {1}", buffer, STATUS_OK_KO(status));

            // Initialize client
            status = UDSApi.Initialize_2013(tp_handle, cantp_baudrate.PCANTP_BAUDRATE_500K);
            Console.WriteLine("Initialize channel: {0}", STATUS_OK_KO(status));

            // Initialize a physical configuration
            config_physical.can_id = (UInt32)uds_can_id.PUDS_CAN_ID_ISO_15765_4_PHYSICAL_REQUEST_1;
            config_physical.can_msgtype = cantp_can_msgtype.PCANTP_CAN_MSGTYPE_STANDARD;
            config_physical.nai.protocol = uds_msgprotocol.PUDS_MSGPROTOCOL_ISO_15765_2_11B_NORMAL;
            config_physical.nai.target_type = cantp_isotp_addressing.PCANTP_ISOTP_ADDRESSING_PHYSICAL;
            config_physical.type = uds_msgtype.PUDS_MSGTYPE_USDT;
            config_physical.nai.source_addr = (UInt16)uds_address.PUDS_ADDRESS_ISO_15765_4_ADDR_TEST_EQUIPMENT;
            config_physical.nai.target_addr = (UInt16)uds_address.PUDS_ADDRESS_ISO_15765_4_ADDR_ECU_1;
            config_physical.nai.extension_addr = 0;

            // Execute ECUReset and wait response
            status = UDSApi.SvcECUReset_2013(tp_handle, config_physical, out msg_request, UDSApi.uds_svc_param_er.PUDS_SVC_PARAM_ER_HR);
            Console.WriteLine("Execute ECUReset service: {0}", STATUS_OK_KO(status));
            status = UDSApi.WaitForService_2013(tp_handle, ref msg_request, out response, out request_confirmation);
            Console.WriteLine("Wait for service: {0}", STATUS_OK_KO(status));

            if (FORCE_CUSTOM_SESSION)
            {
                // Some custom ECU may be in a non default session after reset
                // to inform the UDS API, it is possible to override the current session information:

                // Initialize new session info
                new_sessioninfo.can_msg_type = cantp_can_msgtype.PCANTP_CAN_MSGTYPE_STANDARD;
                new_sessioninfo.session_type = 0x05;
                new_sessioninfo.timeout_p2can_server_max = 1;
                new_sessioninfo.timeout_enhanced_p2can_server_max = 2;
                new_sessioninfo.s3_client_ms = 8;
                new_sessioninfo.nai = config_physical.nai;

                // Set new session info
                int session_size = Marshal.SizeOf(new_sessioninfo);
                IntPtr session_ptr = Marshal.AllocHGlobal(session_size);
                Marshal.StructureToPtr(new_sessioninfo, session_ptr, false);
                status = UDSApi.SetValue_2013(tp_handle, uds_parameter.PUDS_PARAMETER_SESSION_INFO, session_ptr, (UInt32)session_size);
                Console.WriteLine("Set new session info: {0}", STATUS_OK_KO(status));
                Marshal.FreeHGlobal(session_ptr);

                // Get session info parameter and check if the new value is added
                read_sessioninfo = new uds_sessioninfo();
                read_sessioninfo.nai = config_physical.nai;
                status = UDSApi.GetSessionInformation_2013(tp_handle, out read_sessioninfo);
                Console.WriteLine("Get session info: {0}", STATUS_OK_KO(status));
                Console.WriteLine("Check added session info: {0}", OK_KO(is_same_uds_sessioninfo(ref read_sessioninfo, ref new_sessioninfo) == true));
            }

            // Free message structures
            status = UDSApi.MsgFree_2013(ref msg_request);
            Console.WriteLine("Free request message: {0}", STATUS_OK_KO(status));
            status = UDSApi.MsgFree_2013(ref request_confirmation);
            Console.WriteLine("Free request confirmation: {0}", STATUS_OK_KO(status));
            status = UDSApi.MsgFree_2013(ref response);
            Console.WriteLine("Free response message: {0}", STATUS_OK_KO(status));

            // Close client
            status = UDSApi.Uninitialize_2013(tp_handle);
            Console.WriteLine("Uninitialize channel: {0}", STATUS_OK_KO(status));

            // Exit
            Console.WriteLine("Press any key to continue...");
            Console.In.Read();
        }

        static String OK_KO(bool test)
        {
            if (test)
                return "OK";
            return "KO";
        }
        static String STATUS_OK_KO(uds_status test)
        {
            return OK_KO(UDSApi.StatusIsOk_2013(test));
        }
    }
}
