﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

using Peak.Can.IsoTp;
using Peak.Can.Uds;
using System.Runtime.InteropServices;

namespace client_request_ecu_reset_uudt
{
    class Program
    {
        /// <summary>Entry point of the program, start a UDS channel, ask ECureset service in UUDT mode (using 0x123 and 0x124 can id)</summary>
        static void Main(string[] args)
        {
            const int BUFFER_SIZE = 256;
            StringBuilder buffer = new StringBuilder(BUFFER_SIZE);
            uds_status status;
            cantp_handle tp_handle;
            uds_msgconfig config_physical = new uds_msgconfig();
            uds_msg msg_request = new uds_msg();
            uds_msg request_confirmation = new uds_msg();
            uds_msg response = new uds_msg();
            UInt32 timeout_request;
            UInt32 timeout_response;
            bool do_read;

            // Initialize variables
            tp_handle = cantp_handle.PCANTP_HANDLE_USBBUS1; // TODO: modify the value according to your available PCAN devices.

            // Print version informations
            status = UDSApi.GetValue_2013(cantp_handle.PCANTP_HANDLE_NONEBUS, uds_parameter.PUDS_PARAMETER_API_VERSION, buffer, BUFFER_SIZE);
            Console.WriteLine("PCAN-UDS API Version - {0}: {1}", buffer, STATUS_OK_KO(status));

            // Initialize channel
            status = UDSApi.Initialize_2013(tp_handle, cantp_baudrate.PCANTP_BAUDRATE_500K);
            Console.WriteLine("Initialize channel: {0}", STATUS_OK_KO(status));

            // Get timeout values
            status = UDSApi.GetValue_2013(tp_handle, uds_parameter.PUDS_PARAMETER_TIMEOUT_REQUEST, out timeout_request, sizeof(UInt32));
            Console.WriteLine("Get request timeout value ({0}ms): {1}", timeout_request, STATUS_OK_KO(status));
            status = UDSApi.GetValue_2013(tp_handle, uds_parameter.PUDS_PARAMETER_TIMEOUT_RESPONSE, out timeout_response, sizeof(UInt32));
            Console.WriteLine("Get response timeout value ({0}ms): {1}", timeout_response, STATUS_OK_KO(status));

            // Add filter on 0x123 and 0x124 can id
            status = UDSApi.AddCanIdFilter_2013(tp_handle, 0x123);
            Console.WriteLine("Add can id filter (0x123): {0}", STATUS_OK_KO(status));
            status = UDSApi.AddCanIdFilter_2013(tp_handle, 0x124);
            Console.WriteLine("Add can id filter (0x124): {0}", STATUS_OK_KO(status));

            // Initialize the request configuration: UUDT message using 0x123 can id.
            config_physical.can_id = 0x123;
            config_physical.can_msgtype = cantp_can_msgtype.PCANTP_CAN_MSGTYPE_STANDARD;
            config_physical.nai.protocol = uds_msgprotocol.PUDS_MSGPROTOCOL_ISO_15765_2_11B_NORMAL;
            config_physical.nai.target_type = cantp_isotp_addressing.PCANTP_ISOTP_ADDRESSING_PHYSICAL;
            config_physical.type = uds_msgtype.PUDS_MSGTYPE_UUDT;

            // Execute ECUReset
            status = UDSApi.SvcECUReset_2013(tp_handle, config_physical, out msg_request, UDSApi.uds_svc_param_er.PUDS_SVC_PARAM_ER_HR);
            Console.WriteLine("Execute ECUReset service: {0}", STATUS_OK_KO(status));

            // Wait request confirmation (loopback)
            status = UDSApi.WaitForSingleMessage_2013(tp_handle, ref msg_request, true, timeout_request, UDSApi.PUDS_P2CAN_ENHANCED_SERVER_MAX_DEFAULT, out request_confirmation);
            Console.WriteLine("Wait request confirmation: {0}", STATUS_OK_KO(status));

            // Read the first available UUDT message (server should reply with can id 0x124)
            do
            {
                status = UDSApi.Read_2013(tp_handle, out response);
                do_read = UDSApi.StatusIsOk_2013(status, uds_status.PUDS_STATUS_NO_MESSAGE, false);
            } while (do_read);

            if (UDSApi.StatusIsOk_2013(status, uds_status.PUDS_STATUS_OK, false))
            {
                Byte val;
                Console.Write("Read response can ID=0x{0:X4} [", response.msg.can_info.can_id);
                for (int i = 0; i < response.msg.Msgdata_any_Copy.length; i++)
                {
                    CanTpApi.getData_2016(ref response.msg, i, out val);
                    Console.Write(" 0x{0:X2}", (int)val);
                }
                Console.WriteLine(" ]");
            }
            else
            {
                Console.WriteLine("Failed to read message: {0}", STATUS_OK_KO(status));
            }

            // Free message structures
            status = UDSApi.MsgFree_2013(ref msg_request);
            Console.WriteLine("Free request message: {0}", STATUS_OK_KO(status));
            status = UDSApi.MsgFree_2013(ref request_confirmation);
            Console.WriteLine("Free request confirmation: {0}", STATUS_OK_KO(status));
            status = UDSApi.MsgFree_2013(ref response);
            Console.WriteLine("Free response message: {0}", STATUS_OK_KO(status));

            // Close client
            status = UDSApi.Uninitialize_2013(tp_handle);
            Console.WriteLine("Uninitialize channel: {0}", STATUS_OK_KO(status));

            // Exit
            Console.WriteLine("Press any key to continue...");
            Console.In.Read();
        }


        static String OK_KO(bool test)
        {
            if (test)
                return "OK";
            return "KO";
        }
        static String STATUS_OK_KO(uds_status test)
        {
            return OK_KO(UDSApi.StatusIsOk_2013(test));
        }
    }
}
