﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Runtime.InteropServices;

using Peak.Can.IsoTp;
using Peak.Can.Uds;
using System.Threading;

namespace client_all_request
{
    class Program
    {
        // Console handling
        static readonly Boolean USE_GETCH = false;
        static void CLEAR_CONSOLE()
        {
            if (USE_GETCH) Console.Clear();
        }

        // A global counter to keep track of the number of failed tests (see display_uds_msg function)
        static int g_nbErr = 0;

        /// <summary> A simple function that waits for user input</summary>
        static void waitGetch()
        {
            if (USE_GETCH)
            {
                Console.WriteLine();
                Console.Write(" Press <Enter> to continue...");
                Console.In.Read();
                CLEAR_CONSOLE();
            }
        }

        /// <summary>A function that displays UDS Request and Response messages (and count error if no response)</summary>
        /// <param name="request">Request message</param>
        /// <param name="response">Received response message</param>
        /// <param name="no_response_expected">if no response is expected, do not increment error counter</param>
        static void display_uds_msg(ref uds_msg request, ref uds_msg response, bool no_response_expected)
        {
            display_uds_msg_request(ref request);

            display_uds_msg_response(ref response, no_response_expected);
        }

        static void display_uds_msg_response(ref uds_msg response, bool no_response_expected)
        {
            if (response.msg.Msgdata != IntPtr.Zero)
            {
                cantp_msgdata_isotp resp_isotp = new cantp_msgdata_isotp();
                if (response.msg.Msgdata != IntPtr.Zero)
                    resp_isotp = response.msg.Msgdata_isotp_Copy;

                Console.WriteLine(); Console.WriteLine("UDS RESPONSE from 0x{0:X4} (to 0x{1:X4}, with extension address 0x{2:X2}) - result: {3} - {4}",
                    resp_isotp.netaddrinfo.source_addr, resp_isotp.netaddrinfo.target_addr,
                    resp_isotp.netaddrinfo.extension_addr, (int)resp_isotp.netstatus,
                    (resp_isotp.netstatus != cantp_netstatus.PCANTP_NETSTATUS_OK) ? "ERROR !!!" : "OK !");
                // display data
                Console.Write("-> Length: {0}, Data= ", resp_isotp.length);
                for (int i = 0; i < resp_isotp.length; i++)
                {
                    Byte val = 0;
                    CanTpApi.getData_2016(ref response.msg, i, out val);
                    Console.Write("{0:X2} ", val);
                }
                Console.WriteLine("");
            }
            else if (!no_response_expected)
            {
                Console.WriteLine(); Console.WriteLine(" /!\\ ERROR: NO UDS RESPONSE !!");
                g_nbErr++;
            }
        }

        static void display_uds_msg_request(ref uds_msg request, bool no_response_expected)
        {
            display_uds_msg_request(ref request);

            if (!no_response_expected)
            {
                Console.WriteLine(); Console.WriteLine(" /!\\ ERROR: NO UDS RESPONSE !!");
                g_nbErr++;
            }
        }

        static void display_uds_msg_request(ref uds_msg request)
        {
            cantp_msgdata_isotp req_isotp = new cantp_msgdata_isotp();
            if (request.msg.Msgdata != IntPtr.Zero)
                req_isotp = request.msg.Msgdata_isotp_Copy;

            Console.WriteLine(); Console.WriteLine("UDS request from 0x{0:X4} (to 0x{1:X4}, with extension address 0x{2:X2}) - result: {3} - {4}",
                req_isotp.netaddrinfo.source_addr, req_isotp.netaddrinfo.target_addr,
                req_isotp.netaddrinfo.extension_addr, (int)req_isotp.netstatus,
                (req_isotp.netstatus != cantp_netstatus.PCANTP_NETSTATUS_OK) ? "ERROR !!!" : "OK !");
            // display data
            Console.Write("-> Length: {0}, Data= ",
                req_isotp.length);
            for (int i = 0; i < req_isotp.length; i++)
            {
                Byte val = 0;
                CanTpApi.getData_2016(ref request.msg, i, out val);
                Console.Write("{0:X2} ", val);
            }
            Console.WriteLine("");
        }

        /// <summary>Helper: Inverts the bytes of a 32 bits numeric value</summary>
        /// <param name="v">Value to revert</param>
        /// <returns>Reverted value</returns>
        static UInt32 Reverse32(UInt32 v)
        {

            return (v & 0x000000FF) << 24 | (v & 0x0000FF00) << 8 | (v & 0x00FF0000) >> 8 | ((v & 0xFF000000) >> 24) & 0x000000FF;
        }

        static void UInt32ToBytes(UInt32 dw_buffer, Byte[] byte_buffer)
        {
            byte_buffer[0] = (Byte)(dw_buffer & 0x000000FF);
            byte_buffer[1] = (Byte)((dw_buffer & 0x0000FF00) >> 8);
            byte_buffer[2] = (Byte)((dw_buffer & 0x00FF0000) >> 16);
            byte_buffer[3] = (Byte)((dw_buffer & 0xFF000000) >> 24);
        }

        /// <summary>Call UDS Service DiagnosticSessionControl</summary>
        /// <param name="channel">cantp channel handle</param>
        /// <param name="config">Configuration of the request message (type, network address information...)</param>
        static void testDiagnosticSessionControl(cantp_handle channel, uds_msgconfig config)
        {
            uds_status status;
            uds_sessioninfo session_info = new uds_sessioninfo();
            uds_msg request = new uds_msg();
            uds_msg response = new uds_msg();
            uds_msg confirmation = new uds_msg();

            CLEAR_CONSOLE();
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("*** UDS Service: DiagnosticSessionControl ***");

            // Read default session information Server is not yet known (status will be PUDS_ERROR_NOT_INITIALIZED)
            // yet the API will still set session info to default values
            session_info.nai = config.nai;
            int session_size = Marshal.SizeOf(session_info);
            IntPtr session_ptr = Marshal.AllocHGlobal(session_size);
            Marshal.StructureToPtr(session_info, session_ptr, true);
            status = UDSApi.GetValue_2013(channel, uds_parameter.PUDS_PARAMETER_SESSION_INFO, session_ptr, (UInt32)session_size);
            session_info = (uds_sessioninfo)Marshal.PtrToStructure(session_ptr, typeof(uds_sessioninfo));
            Marshal.FreeHGlobal(session_ptr);
            Console.WriteLine(" Diagnostic Session Information: {0}, 0x{1:X4} => {2} = [{3:X4}; {4:X4}]", (int)status, session_info.nai.target_addr,
                session_info.session_type, session_info.timeout_p2can_server_max, session_info.timeout_enhanced_p2can_server_max);
            waitGetch();

            // Set Diagnostic session to DEFAULT (to get session information)
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("Setting a DEFAULT Diagnostic Session:");
            status = UDSApi.SvcDiagnosticSessionControl_2013(channel, config, out request, UDSApi.uds_svc_param_dsc.PUDS_SVC_PARAM_DSC_DS);
            if (UDSApi.StatusIsOk_2013(status))
                status = UDSApi.WaitForService_2013(channel, ref request, out response, out confirmation);
            Console.WriteLine(" UDS_SvcDiagnosticSessionControl_2013: {0}", (int)status);
            if (UDSApi.StatusIsOk_2013(status))
                display_uds_msg(ref confirmation, ref response, false);
            else
                display_uds_msg_request(ref request, false);
            status = UDSApi.MsgFree_2013(ref request);
            Console.WriteLine(" Free request message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref response);
            Console.WriteLine(" Free response message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref confirmation);
            Console.WriteLine(" Free confirmation message: {0}", (int)status);

            // Read current session information
            session_info = new uds_sessioninfo();
            session_info.nai = config.nai;
            session_ptr = Marshal.AllocHGlobal(session_size);
            Marshal.StructureToPtr(session_info, session_ptr, true);
            status = UDSApi.GetValue_2013(channel, uds_parameter.PUDS_PARAMETER_SESSION_INFO, session_ptr, (UInt32)session_size);
            session_info = (uds_sessioninfo)Marshal.PtrToStructure(session_ptr, typeof(uds_sessioninfo));
            Marshal.FreeHGlobal(session_ptr);
            Console.WriteLine(" Diagnostic Session Information: {0}, 0x{1:X4} => {2} = [{3:X4}; {4:X4}]", (int)status, session_info.nai.target_addr,
                session_info.session_type, session_info.timeout_p2can_server_max, session_info.timeout_enhanced_p2can_server_max);
            waitGetch();

            // Set Diagnostic session to PROGRAMMING
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("Setting a ECUPS Diagnostic Session:");
            status = UDSApi.SvcDiagnosticSessionControl_2013(channel, config, out request, UDSApi.uds_svc_param_dsc.PUDS_SVC_PARAM_DSC_ECUPS);
            if (UDSApi.StatusIsOk_2013(status))
                status = UDSApi.WaitForService_2013(channel, ref request, out response, out confirmation);
            Console.WriteLine(" UDS_SvcDiagnosticSessionControl_2013: {0}", (int)status);
            if (UDSApi.StatusIsOk_2013(status))
                display_uds_msg(ref confirmation, ref response, false);
            else
                display_uds_msg_request(ref request, false);
            status = UDSApi.MsgFree_2013(ref request);
            Console.WriteLine(" Free request message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref response);
            Console.WriteLine(" Free response message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref confirmation);
            Console.WriteLine(" Free confirmation message: {0}", (int)status);

            // Read current session information
            session_info = new uds_sessioninfo();
            session_info.nai = config.nai;
            session_ptr = Marshal.AllocHGlobal(session_size);
            Marshal.StructureToPtr(session_info, session_ptr, true);
            status = UDSApi.GetValue_2013(channel, uds_parameter.PUDS_PARAMETER_SESSION_INFO, session_ptr, (UInt32)session_size);
            session_info = (uds_sessioninfo)Marshal.PtrToStructure(session_ptr, typeof(uds_sessioninfo));
            Marshal.FreeHGlobal(session_ptr);
            Console.WriteLine(" Diagnostic Session Information: {0}, 0x{1:X4} => {2} = [{3:X4}; {4:X4}]", (int)status, session_info.nai.target_addr,
                session_info.session_type, session_info.timeout_p2can_server_max, session_info.timeout_enhanced_p2can_server_max);
            Console.WriteLine(" Assert that Auto TesterPresent Frame is sent...");
            Thread.Sleep(2000);
            Console.WriteLine(" Should transmit an Auto TesterPresent Frame");
            Thread.Sleep(2000);
            Console.WriteLine(" Should transmit an Auto TesterPresent Frame");
            waitGetch();

            // Set Diagnostic session back to DEFAULT
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("Setting a DEFAULT Diagnostic Session:");
            status = UDSApi.SvcDiagnosticSessionControl_2013(channel, config, out request, UDSApi.uds_svc_param_dsc.PUDS_SVC_PARAM_DSC_DS);
            if (UDSApi.StatusIsOk_2013(status))
                status = UDSApi.WaitForService_2013(channel, ref request, out response, out confirmation);
            Console.WriteLine(" UDS_SvcDiagnosticSessionControl_2013: {0}", (int)status);
            if (UDSApi.StatusIsOk_2013(status))
                display_uds_msg(ref confirmation, ref response, false);
            else
                display_uds_msg_request(ref request, false);
            status = UDSApi.MsgFree_2013(ref request);
            Console.WriteLine(" Free request message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref response);
            Console.WriteLine(" Free response message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref confirmation);
            Console.WriteLine(" Free confirmation message: {0}", (int)status);
            Console.WriteLine(" Assert that NO Auto TesterPresent Frame is sent...");
            Thread.Sleep(2000);
            Console.WriteLine(" Should NOT transmit an Auto TesterPresent Frame");
            Thread.Sleep(2000);
            Console.WriteLine(" Should NOT transmit an Auto TesterPresent Frame");
            waitGetch();
        }

        /// <summary>Call UDS Service ECUReset</summary>
        /// <param name="channel">cantp channel handle</param>
        /// <param name="config">Configuration of the request message (type, network address information...)</param>
        static void testECUReset(cantp_handle channel, uds_msgconfig config)
        {
            uds_status status;
            uds_msg request = new uds_msg();
            uds_msg response = new uds_msg();
            uds_msg confirmation = new uds_msg();

            CLEAR_CONSOLE();
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("*** UDS Service: ECUReset ***");

            // Sends a physical ECUReset message
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("Sends a physical ECUReset message: ");
            status = UDSApi.SvcECUReset_2013(channel, config, out request, UDSApi.uds_svc_param_er.PUDS_SVC_PARAM_ER_SR);
            if (UDSApi.StatusIsOk_2013(status))
                status = UDSApi.WaitForService_2013(channel, ref request, out response, out confirmation);
            Console.WriteLine(" UDS_SvcECUReset_2013: {0}", (int)status);
            if (UDSApi.StatusIsOk_2013(status))
                display_uds_msg(ref confirmation, ref response, false);
            else
                display_uds_msg_request(ref request, false);

            // Free messages
            status = UDSApi.MsgFree_2013(ref request);
            Console.WriteLine(" Free request message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref response);
            Console.WriteLine(" Free response message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref confirmation);
            Console.WriteLine(" Free confirmation message: {0}", (int)status);
            waitGetch();
        }

        /// <summary>Call UDS Service SecurityAccess</summary>
        /// <param name="channel">cantp channel handle</param>
        /// <param name="config">Configuration of the request message (type, network address information...)</param>
        static void testSecurityAccess(cantp_handle channel, uds_msgconfig config)
        {
            uds_status status;
            uds_msg request = new uds_msg();
            uds_msg response = new uds_msg();
            uds_msg confirmation = new uds_msg();
            UInt32 dw_buffer;
            Byte[] security_access_data = new Byte[4];
            UInt32 value_little_endian;

            CLEAR_CONSOLE();
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("*** UDS Service: SecurityAccess ***");

            // Sends a physical SecurityAccess message
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("Sends a physical SecurityAccess message: ");
            value_little_endian = 0xF0A1B2C3;
            dw_buffer = Reverse32(value_little_endian); // use helper function to set MSB as 1st byte in the buffer (Win32 uses little endian format)
            UInt32ToBytes(dw_buffer, security_access_data);
            status = UDSApi.SvcSecurityAccess_2013(channel, config, out request, UDSApi.PUDS_SVC_PARAM_SA_RSD_1, security_access_data, 4);
            if (UDSApi.StatusIsOk_2013(status))
                status = UDSApi.WaitForService_2013(channel, ref request, out response, out confirmation);
            Console.WriteLine(" UDS_SvcSecurityAccess_2013: {0}", (int)status);
            if (UDSApi.StatusIsOk_2013(status))
                display_uds_msg(ref confirmation, ref response, false);
            else
                display_uds_msg_request(ref request, false);

            // Free messages
            status = UDSApi.MsgFree_2013(ref request);
            Console.WriteLine(" Free request message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref response);
            Console.WriteLine(" Free response message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref confirmation);
            Console.WriteLine(" Free confirmation message: {0}", (int)status);
            waitGetch();
        }

        /// <summary>Call UDS Service CommunicationControl</summary>
        /// <param name="channel">cantp channel handle</param>
        /// <param name="config">Configuration of the request message (type, network address information...)</param>
        static void testCommunicationControl(cantp_handle channel, uds_msgconfig config)
        {
            uds_status status;
            uds_msg request = new uds_msg();
            uds_msg response = new uds_msg();
            uds_msg confirmation = new uds_msg();

            CLEAR_CONSOLE();
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("*** UDS Service: CommunicationControl ***");

            // Sends a physical CommunicationControl message
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("Sends a physical CommunicationControl message: ");
            status = UDSApi.SvcCommunicationControl_2013(channel, config, out request,
                UDSApi.uds_svc_param_cc.PUDS_SVC_PARAM_CC_ERXTX, UDSApi.PUDS_SVC_PARAM_CC_FLAG_APPL | UDSApi.PUDS_SVC_PARAM_CC_FLAG_NWM | UDSApi.PUDS_SVC_PARAM_CC_FLAG_DENWRIRO);
            if (UDSApi.StatusIsOk_2013(status))
                status = UDSApi.WaitForService_2013(channel, ref request, out response, out confirmation);
            Console.WriteLine(" UDS_SvcCommunicationControl_2013: {0}", (int)status);
            if (UDSApi.StatusIsOk_2013(status))
                display_uds_msg(ref confirmation, ref response, false);
            else
                display_uds_msg_request(ref request, false);

            // Free messages
            status = UDSApi.MsgFree_2013(ref request);
            Console.WriteLine(" Free request message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref response);
            Console.WriteLine(" Free response message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref confirmation);
            Console.WriteLine(" Free confirmation message: {0}", (int)status);
            waitGetch();
        }

        /// <summary>UDS Call Service TesterPresent</summary>
        /// <param name="channel">cantp channel handle</param>
        /// <param name="config">Configuration of the request message (type, network address information...)</param>
        static void testTesterPresent(cantp_handle channel, uds_msgconfig config)
        {
            uds_status status;
            uds_msg request = new uds_msg();
            uds_msg[] tresponse = new uds_msg[1];
            uds_msg response = new uds_msg();
            UInt32 response_count = 0;
            uds_msg confirmation = new uds_msg();

            CLEAR_CONSOLE();
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("*** UDS Service: TesterPresent ***");

            // Sends a physical TesterPresent message
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("Sends a physical TesterPresent message: ");
            status = UDSApi.SvcTesterPresent_2013(channel, config, out request, UDSApi.uds_svc_param_tp.PUDS_SVC_PARAM_TP_ZSUBF);
            if (UDSApi.StatusIsOk_2013(status))
                status = UDSApi.WaitForService_2013(channel, ref request, out response, out confirmation);
            Console.WriteLine(" UDS_SvcTesterPresent_2013: {0}", (int)status);
            if (UDSApi.StatusIsOk_2013(status))
                display_uds_msg(ref confirmation, ref response, false);
            else
                display_uds_msg_request(ref request, false);
            status = UDSApi.MsgFree_2013(ref request);
            Console.WriteLine(" Free request message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref response);
            Console.WriteLine(" Free response message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref confirmation);
            Console.WriteLine(" Free confirmation message: {0}", (int)status);
            waitGetch();

            // Sends a physical TesterPresent message with no positive response
            config.type = uds_msgtype.PUDS_MSGTYPE_FLAG_NO_POSITIVE_RESPONSE;
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("Sends a physical TesterPresent message with no positive response:");
            status = UDSApi.SvcTesterPresent_2013(channel, config, out request, UDSApi.uds_svc_param_tp.PUDS_SVC_PARAM_TP_ZSUBF);
            if (UDSApi.StatusIsOk_2013(status))
                status = UDSApi.WaitForService_2013(channel, ref request, out response, out confirmation);
            Console.WriteLine(" UDS_SvcTesterPresent_2013: {0}", (int)status);
            if (UDSApi.StatusIsOk_2013(status))
                display_uds_msg(ref confirmation, ref response, false);
            else
                display_uds_msg_request(ref request, true);
            status = UDSApi.MsgFree_2013(ref request);
            Console.WriteLine(" Free request message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref response);
            Console.WriteLine(" Free response message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref confirmation);
            Console.WriteLine(" Free confirmation message: {0}", (int)status);
            waitGetch();

            // Sends a functional TesterPresent message
            config.type = uds_msgtype.PUDS_MSGTYPE_USDT;
            config.nai.target_type = cantp_isotp_addressing.PCANTP_ISOTP_ADDRESSING_FUNCTIONAL;
            config.nai.target_addr = (UInt16)uds_address.PUDS_ADDRESS_ISO_15765_4_ADDR_OBD_FUNCTIONAL;
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("Sends a functional TesterPresent message: ");
            status = UDSApi.SvcTesterPresent_2013(channel, config, out request, UDSApi.uds_svc_param_tp.PUDS_SVC_PARAM_TP_ZSUBF);
            if (UDSApi.StatusIsOk_2013(status))
                status = UDSApi.WaitForServiceFunctional_2013(channel, ref request, 1, true, tresponse, out response_count, out confirmation);
            Console.WriteLine(" UDS_SvcTesterPresent_2013: {0}", (int)status);
            if (UDSApi.StatusIsOk_2013(status))
                display_uds_msg(ref confirmation, ref tresponse[0], false);
            else
                display_uds_msg_request(ref request, false);
            status = UDSApi.MsgFree_2013(ref request);
            Console.WriteLine(" Free request message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref tresponse[0]);
            Console.WriteLine(" Free response message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref confirmation);
            Console.WriteLine(" Free confirmation message: {0}", (int)status);
            waitGetch();

            // Sends a functional TesterPresent message with no positive response
            config.type = uds_msgtype.PUDS_MSGTYPE_FLAG_NO_POSITIVE_RESPONSE;
            config.nai.target_type = cantp_isotp_addressing.PCANTP_ISOTP_ADDRESSING_FUNCTIONAL;
            config.nai.target_addr = (UInt16)uds_address.PUDS_ADDRESS_ISO_15765_4_ADDR_OBD_FUNCTIONAL;
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("Sends a functional TesterPresent message with no positive response:");
            status = UDSApi.SvcTesterPresent_2013(channel, config, out request, UDSApi.uds_svc_param_tp.PUDS_SVC_PARAM_TP_ZSUBF);
            if (UDSApi.StatusIsOk_2013(status))
                status = UDSApi.WaitForServiceFunctional_2013(channel, ref request, 1, true, tresponse, out response_count, out confirmation);
            Console.WriteLine(" UDS_SvcTesterPresent_2013: {0}", (int)status);
            if (UDSApi.StatusIsOk_2013(status) && response_count != 0)
                display_uds_msg(ref confirmation, ref tresponse[0], false);
            else
                display_uds_msg_request(ref request, true);
            status = UDSApi.MsgFree_2013(ref request);
            Console.WriteLine(" Free request message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref tresponse[0]);
            Console.WriteLine(" Free response message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref confirmation);
            Console.WriteLine(" Free confirmation message: {0}", (int)status);
            waitGetch();
        }

        /// <summary>Call UDS Service SecuredDataTransmission</summary>
        /// <param name="channel">cantp channel handle</param>
        /// <param name="config">Configuration of the request message (type, network address information...)</param>
        static void testSecuredDataTransmission(cantp_handle channel, uds_msgconfig config)
        {
            uds_status status;
            uds_msg request = new uds_msg();
            uds_msg response = new uds_msg();
            uds_msg ecureset_request = new uds_msg();
            uds_msg confirmation = new uds_msg();
            UInt32 dw_buffer;
            Byte[] security_data_request_record = new Byte[4];
            UInt32 value_little_endian;
            UInt16 administrative_parameter;
            Byte signature_encryption_calculation;
            UInt16 anti_replay_counter;
            Byte internal_service_identifier;
            Byte[] service_specific_parameters = new Byte[4];
            UInt32 service_specific_parameters_size;
            Byte[] signature_mac = new Byte[6];
            UInt16 signature_size;

            CLEAR_CONSOLE();
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("*** UDS Service: SecuredDataTransmission ***");

            // Sends a physical SecuredDataTransmission/2013 message
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("Sends a physical SecuredDataTransmission/2013 message: ");
            value_little_endian = 0xF0A1B2C3;
            dw_buffer = Reverse32(value_little_endian); // use helper function to set MSB as 1st byte in the buffer (Win32 uses little endian format)
            UInt32ToBytes(dw_buffer, security_data_request_record);
            status = UDSApi.SvcSecuredDataTransmission_2013(channel, config, out request, security_data_request_record, 4);
            if (UDSApi.StatusIsOk_2013(status))
                status = UDSApi.WaitForService_2013(channel, ref request, out response, out confirmation);
            Console.WriteLine(" UDS_SvcSecuredDataTransmission_2013: {0}", (int)status);
            if (UDSApi.StatusIsOk_2013(status))
                display_uds_msg(ref confirmation, ref response, false);
            else
                display_uds_msg_request(ref request, false);

            // Free messages
            status = UDSApi.MsgFree_2013(ref request);
            Console.WriteLine(" Free request message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref response);
            Console.WriteLine(" Free response message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref confirmation);
            Console.WriteLine(" Free confirmation message: {0}", (int)status);
            waitGetch();

            // Sends a physical SecuredDataTransmission/2013 message prepared with PUDS_ONLY_PREPARE_REQUEST option
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("Sends a physical SecuredDataTransmission/2013 prepared with PUDS_ONLY_PREPARE_REQUEST option: ");
            status = UDSApi.SvcECUReset_2013(UDSApi.PUDS_ONLY_PREPARE_REQUEST, config, out ecureset_request, UDSApi.uds_svc_param_er.PUDS_SVC_PARAM_ER_HR);
            Console.WriteLine(" Prepare ECUReset request for SecuredDataTransmission: {0}", (int)status);
            if (UDSApi.StatusIsOk_2013(status, uds_status.PUDS_STATUS_OK, false))
            {
                byte[] ecureset_data = new byte[ecureset_request.msg.Msgdata_any_Copy.length];
                for (int i = 0; i < ecureset_request.msg.Msgdata_any_Copy.length; i++)
                {
                    Byte val = 0;
                    CanTpApi.getData_2016(ref ecureset_request.msg, i, out val);
                    ecureset_data[i] = val;
                }
                status = UDSApi.SvcSecuredDataTransmission_2013(channel, config, out request, ecureset_data, ecureset_request.msg.Msgdata_any_Copy.length);
            }
            if (UDSApi.StatusIsOk_2013(status, uds_status.PUDS_STATUS_OK, false))
                status = UDSApi.WaitForService_2013(channel, ref request, out response, out confirmation);
            Console.WriteLine(" UDS_SvcSecuredDataTransmission_2013: {0}", (int)status);
            if (UDSApi.StatusIsOk_2013(status, uds_status.PUDS_STATUS_OK, false))
                display_uds_msg(ref confirmation, ref response, false);
            else
                display_uds_msg_request(ref request, false);

            // Free messages
            status = UDSApi.MsgFree_2013(ref ecureset_request);
            Console.WriteLine(" Free prepared message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref request);
            Console.WriteLine(" Free request message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref response);
            Console.WriteLine(" Free response message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref confirmation);
            Console.WriteLine(" Free confirmation message: {0}", (int)status);
            waitGetch();

            // Sends a physical SecuredDataTransmission/2020 message
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("Sends a physical SecuredDataTransmission/2020 message: ");
            administrative_parameter = UDSApi.PUDS_SVC_PARAM_APAR_REQUEST_MSG_FLAG | UDSApi.PUDS_SVC_PARAM_APAR_REQUEST_RESPONSE_SIGNATURE_FLAG | UDSApi.PUDS_SVC_PARAM_APAR_SIGNED_MSG_FLAG;
            signature_encryption_calculation = 0x0;
            anti_replay_counter = 0x0124;
            internal_service_identifier = 0x2E;
            service_specific_parameters[0] = 0xF1;
            service_specific_parameters[1] = 0x23;
            service_specific_parameters[2] = 0xAA;
            service_specific_parameters[3] = 0x55;
            service_specific_parameters_size = 4;
            signature_mac[0] = 0xDB;
            signature_mac[1] = 0xD1;
            signature_mac[2] = 0x0E;
            signature_mac[3] = 0xDC;
            signature_mac[4] = 0x55;
            signature_mac[5] = 0xAA;
            signature_size = 0x0006;
            status = UDSApi.SvcSecuredDataTransmission_2020(channel, config, out request, administrative_parameter, signature_encryption_calculation, anti_replay_counter, internal_service_identifier, service_specific_parameters, service_specific_parameters_size, signature_mac, signature_size);
            if (UDSApi.StatusIsOk_2013(status, uds_status.PUDS_STATUS_OK, false))
                status = UDSApi.WaitForService_2013(channel, ref request, out response, out confirmation);
            Console.WriteLine(" UDS_SvcSecuredDataTransmission_2013: {0}", (int)status);
            if (UDSApi.StatusIsOk_2013(status, uds_status.PUDS_STATUS_OK, false))
                display_uds_msg(ref confirmation, ref response, false);
            else
                display_uds_msg_request(ref request, false);

            // Free messages
            status = UDSApi.MsgFree_2013(ref request);
            Console.WriteLine(" Free request message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref response);
            Console.WriteLine(" Free response message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref confirmation);
            Console.WriteLine(" Free confirmation message: {0}", (int)status);
            waitGetch();
        }

        /// <summary>Call UDS Service ControlDTCSetting</summary>
        /// <param name="channel">cantp channel handle</param>
        /// <param name="config">Configuration of the request message (type, network address information...)</param>
        static void testControlDTCSetting(cantp_handle channel, uds_msgconfig config)
        {
            uds_status status;
            uds_msg request = new uds_msg();
            uds_msg response = new uds_msg();
            uds_msg confirmation = new uds_msg();
            UInt32 dw_buffer;
            Byte[] dtc_setting_control_option_record = new Byte[4];
            UInt32 value_little_endian;

            CLEAR_CONSOLE();
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("*** UDS Service: ControlDTCSetting ***");

            // Sends a physical ControlDTCSetting message
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("Sends a physical ControlDTCSetting message: ");
            value_little_endian = 0xF1A1B2EE;
            dw_buffer = Reverse32(value_little_endian); // use helper function to set MSB as 1st byte in the buffer (Win32 uses little endian format)
            UInt32ToBytes(dw_buffer, dtc_setting_control_option_record);
            status = UDSApi.SvcControlDTCSetting_2013(channel, config, out request, UDSApi.uds_svc_param_cdtcs.PUDS_SVC_PARAM_CDTCS_OFF, dtc_setting_control_option_record, 3);
            if (UDSApi.StatusIsOk_2013(status))
                status = UDSApi.WaitForService_2013(channel, ref request, out response, out confirmation);
            Console.WriteLine(" UDS_SvcControlDTCSetting_2013: {0}", (int)status);
            if (UDSApi.StatusIsOk_2013(status))
                display_uds_msg(ref confirmation, ref response, false);
            else
                display_uds_msg_request(ref request, false);

            // Free messages
            status = UDSApi.MsgFree_2013(ref request);
            Console.WriteLine(" Free request message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref response);
            Console.WriteLine(" Free response message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref confirmation);
            Console.WriteLine(" Free confirmation message: {0}", (int)status);
            waitGetch();
        }

        /// <summary>Call UDS Service ResponseOnEvent</summary>
        /// <param name="channel">cantp channel handle</param>
        /// <param name="config">Configuration of the request message (type, network address information...)</param>
        static void testResponseOnEvent(cantp_handle channel, uds_msgconfig config)
        {
            uds_status status;
            uds_msg request = new uds_msg();
            uds_msg response = new uds_msg();
            uds_msg confirmation = new uds_msg();
            Byte[] event_type_record = new Byte[50];
            Byte[] service_to_respond_to_record = new Byte[50];

            CLEAR_CONSOLE();
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("*** UDS Service: ResponseOnEvent ***");

            // Sends a physical ResponseOnEvent message
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("Sends a physical ResponseOnEvent message: ");
            event_type_record[0] = 0x08;
            service_to_respond_to_record[0] = (Byte)uds_service.PUDS_SERVICE_SI_ReadDTCInformation;
            service_to_respond_to_record[1] = (Byte)UDSApi.uds_svc_param_rdtci.PUDS_SVC_PARAM_RDTCI_RNODTCBSM;
            service_to_respond_to_record[2] = 0x01;
            status = UDSApi.SvcResponseOnEvent_2013(channel, config, out request, UDSApi.uds_svc_param_roe.PUDS_SVC_PARAM_ROE_ONDTCS, false, 0x08, event_type_record,
                UDSApi.PUDS_SVC_PARAM_ROE_OTI_LEN, service_to_respond_to_record, 3);
            if (UDSApi.StatusIsOk_2013(status))
                status = UDSApi.WaitForService_2013(channel, ref request, out response, out confirmation);
            Console.WriteLine(" UDS_SvcResponseOnEvent_2013: {0}", (int)status);
            if (UDSApi.StatusIsOk_2013(status))
                display_uds_msg(ref confirmation, ref response, false);
            else
                display_uds_msg_request(ref request, false);

            // Free messages
            status = UDSApi.MsgFree_2013(ref request);
            Console.WriteLine(" Free request message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref response);
            Console.WriteLine(" Free response message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref confirmation);
            Console.WriteLine(" Free confirmation message: {0}", (int)status);
            waitGetch();
        }

        /// <summary>Call UDS Service LinkControl</summary>
        /// <param name="channel">cantp channel handle</param>
        /// <param name="config">Configuration of the request message (type, network address information...)</param>
        static void testLinkControl(cantp_handle channel, uds_msgconfig config)
        {
            uds_status status;
            uds_msg request = new uds_msg();
            uds_msg response = new uds_msg();
            uds_msg confirmation = new uds_msg();

            CLEAR_CONSOLE();
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("*** UDS Service: LinkControl ***");

            // Sends a physical LinkControl message
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("Sends a physical LinkControl message (Verify Fixed Baudrate): ");
            status = UDSApi.SvcLinkControl_2013(channel, config, out request, UDSApi.uds_svc_param_lc.PUDS_SVC_PARAM_LC_VBTWFBR, UDSApi.uds_svc_param_lc_baudrate_identifier.PUDS_SVC_PARAM_LC_BAUDRATE_CAN_500K, 0);
            if (UDSApi.StatusIsOk_2013(status))
                status = UDSApi.WaitForService_2013(channel, ref request, out response, out confirmation);
            Console.WriteLine(" UDS_SvcLinkControl_2013: {0}", (int)status);
            if (UDSApi.StatusIsOk_2013(status))
                display_uds_msg(ref confirmation, ref response, false);
            else
                display_uds_msg_request(ref request, false);
            status = UDSApi.MsgFree_2013(ref request);
            Console.WriteLine(" Free request message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref response);
            Console.WriteLine(" Free response message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref confirmation);
            Console.WriteLine(" Free confirmation message: {0}", (int)status);

            // Sends a physical LinkControl message
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("Sends a physical LinkControl message (Verify Specific Baudrate): ");
            status = UDSApi.SvcLinkControl_2013(channel, config, out request, UDSApi.uds_svc_param_lc.PUDS_SVC_PARAM_LC_VBTWSBR, 0, 500000); // 500K = 0x0007a120
            if (UDSApi.StatusIsOk_2013(status))
                status = UDSApi.WaitForService_2013(channel, ref request, out response, out confirmation);
            Console.WriteLine(" UDS_SvcLinkControl_2013: {0}", (int)status);
            if (UDSApi.StatusIsOk_2013(status))
                display_uds_msg(ref confirmation, ref response, false);
            else
                display_uds_msg_request(ref request, false);
            status = UDSApi.MsgFree_2013(ref request);
            Console.WriteLine(" Free request message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref response);
            Console.WriteLine(" Free response message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref confirmation);
            Console.WriteLine(" Free confirmation message: {0}", (int)status);

            // Sends a physical LinkControl message
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("Sends a physical LinkControl message (Transition): ");
            status = UDSApi.SvcLinkControl_2013(channel, config, out request, UDSApi.uds_svc_param_lc.PUDS_SVC_PARAM_LC_TB, 0, 0);
            if (UDSApi.StatusIsOk_2013(status))
                status = UDSApi.WaitForService_2013(channel, ref request, out response, out confirmation);
            Console.WriteLine(" UDS_SvcLinkControl_2013: {0}", (int)status);
            if (UDSApi.StatusIsOk_2013(status))
                display_uds_msg(ref confirmation, ref response, false);
            else
                display_uds_msg_request(ref request, false);
            status = UDSApi.MsgFree_2013(ref request);
            Console.WriteLine(" Free request message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref response);
            Console.WriteLine(" Free response message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref confirmation);
            Console.WriteLine(" Free confirmation message: {0}", (int)status);
            waitGetch();
        }

        /// <summary>Call UDS Service ReadDataByIdentifier</summary>
        /// <param name="channel">cantp channel handle</param>
        /// <param name="config">Configuration of the request message (type, network address information...)</param>
        static void testReadDataByIdentifier(cantp_handle channel, uds_msgconfig config)
        {
            uds_status status;
            uds_msg request = new uds_msg();
            uds_msg response = new uds_msg();
            uds_msg confirmation = new uds_msg();
            UDSApi.uds_svc_param_di[] buffer = { UDSApi.uds_svc_param_di.PUDS_SVC_PARAM_DI_ADSDID, UDSApi.uds_svc_param_di.PUDS_SVC_PARAM_DI_ECUMDDID };

            CLEAR_CONSOLE();
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("*** UDS Service: ReadDataByIdentifier ***");

            // Sends a physical ReadDataByIdentifier message
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("Sends a physical ReadDataByIdentifier message: ");
            status = UDSApi.SvcReadDataByIdentifier_2013(channel, config, out request, buffer, 2);
            if (UDSApi.StatusIsOk_2013(status))
                status = UDSApi.WaitForService_2013(channel, ref request, out response, out confirmation);
            Console.WriteLine(" UDS_SvcReadDataByIdentifier_2013: {0}", (int)status);
            if (UDSApi.StatusIsOk_2013(status))
                display_uds_msg(ref confirmation, ref response, false);
            else
                display_uds_msg_request(ref request, false);

            // Free messages
            status = UDSApi.MsgFree_2013(ref request);
            Console.WriteLine(" Free request message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref response);
            Console.WriteLine(" Free response message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref confirmation);
            Console.WriteLine(" Free confirmation message: {0}", (int)status);
            waitGetch();
        }

        /// <summary>Call UDS Service ReadMemoryByAddress</summary>
        /// <param name="channel">cantp channel handle</param>
        /// <param name="config">Configuration of the request message (type, network address information...)</param>
        static void testReadMemoryByAddress(cantp_handle channel, uds_msgconfig config)
        {
            uds_status status;
            uds_msg request = new uds_msg();
            uds_msg response = new uds_msg();
            uds_msg confirmation = new uds_msg();
            Byte[] memory_address_buffer = new Byte[10];
            Byte[] memory_size_buffer = new Byte[10];
            Byte memory_address_size = 10;
            Byte memory_size_size = 3;

            CLEAR_CONSOLE();
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("*** UDS Service: ReadMemoryByAddress ***");

            // Sends a physical ReadMemoryByAddress message
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("Sends a physical ReadMemoryByAddress message: ");
            for (int i = 0; i < memory_address_size; i++)
            {
                memory_address_buffer[i] = (Byte)('A' + i);
                memory_size_buffer[i] = (Byte)('1' + i);
            }
            status = UDSApi.SvcReadMemoryByAddress_2013(channel, config, out request, memory_address_buffer, memory_address_size, memory_size_buffer, memory_size_size);
            if (UDSApi.StatusIsOk_2013(status))
                status = UDSApi.WaitForService_2013(channel, ref request, out response, out confirmation);
            Console.WriteLine(" UDS_SvcReadMemoryByAddress_2013: {0}", (int)status);
            if (UDSApi.StatusIsOk_2013(status))
                display_uds_msg(ref confirmation, ref response, false);
            else
                display_uds_msg_request(ref request, false);

            // Free messages
            status = UDSApi.MsgFree_2013(ref request);
            Console.WriteLine(" Free request message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref response);
            Console.WriteLine(" Free response message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref confirmation);
            Console.WriteLine(" Free confirmation message: {0}", (int)status);
            waitGetch();
        }

        /// <summary>Call UDS Service ReadScalingDataByIdentifier</summary>
        /// <param name="channel">cantp channel handle</param>
        /// <param name="config">Configuration of the request message (type, network address information...)</param>
        static void testReadScalingDataByIdentifier(cantp_handle channel, uds_msgconfig config)
        {
            uds_status status;
            uds_msg request = new uds_msg();
            uds_msg response = new uds_msg();
            uds_msg confirmation = new uds_msg();

            CLEAR_CONSOLE();
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("*** UDS Service: ReadScalingDataByIdentifier ***");

            // Sends a physical ReadScalingDataByIdentifier message
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("Sends a physical ReadScalingDataByIdentifier message: ");
            status = UDSApi.SvcReadScalingDataByIdentifier_2013(channel, config, out request, UDSApi.uds_svc_param_di.PUDS_SVC_PARAM_DI_BSFPDID);
            if (UDSApi.StatusIsOk_2013(status))
                status = UDSApi.WaitForService_2013(channel, ref request, out response, out confirmation);
            Console.WriteLine(" UDS_SvcReadScalingDataByIdentifier_2013: {0}", (int)status);
            if (UDSApi.StatusIsOk_2013(status))
                display_uds_msg(ref confirmation, ref response, false);
            else
                display_uds_msg_request(ref request, false);

            // Free messages
            status = UDSApi.MsgFree_2013(ref request);
            Console.WriteLine(" Free request message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref response);
            Console.WriteLine(" Free response message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref confirmation);
            Console.WriteLine(" Free confirmation message: {0}", (int)status);
            waitGetch();
        }

        /// <summary>Call UDS Service ReadDataByPeriodicIdentifier</summary>
        /// <param name="channel">cantp channel handle</param>
        /// <param name="config">Configuration of the request message (type, network address information...)</param>
        static void testReadDataByPeriodicIdentifier(cantp_handle channel, uds_msgconfig config)
        {
            uds_status status;
            uds_msg request = new uds_msg();
            uds_msg response = new uds_msg();
            uds_msg confirmation = new uds_msg();
            Byte[] periodic_data_identifier = new Byte[20];
            UInt16 periodic_data_identifier_size = 10;

            CLEAR_CONSOLE();
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("*** UDS Service: ReadDataByPeriodicIdentifier ***");

            // Sends a physical ReadScalingDataByIdentifier message
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("Sends a physical ReadDataByPeriodicIdentifier message: ");
            for (int i = 0; i < periodic_data_identifier_size; i++)
                periodic_data_identifier[i] = (Byte)('A' + i);
            status = UDSApi.SvcReadDataByPeriodicIdentifier_2013(channel, config, out request, UDSApi.uds_svc_param_rdbpi.PUDS_SVC_PARAM_RDBPI_SAMR, periodic_data_identifier, periodic_data_identifier_size);
            if (UDSApi.StatusIsOk_2013(status))
                status = UDSApi.WaitForService_2013(channel, ref request, out response, out confirmation);
            Console.WriteLine(" UDS_SvcReadDataByPeriodicIdentifier_2013: {0}", (int)status);
            if (UDSApi.StatusIsOk_2013(status))
                display_uds_msg(ref confirmation, ref response, false);
            else
                display_uds_msg_request(ref request, false);

            // Free messages
            status = UDSApi.MsgFree_2013(ref request);
            Console.WriteLine(" Free request message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref response);
            Console.WriteLine(" Free response message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref confirmation);
            Console.WriteLine(" Free confirmation message: {0}", (int)status);
            waitGetch();
        }

        /// <summary>Call UDS Service DynamicallyDefineDataIdentifier</summary>
        /// <param name="channel">cantp channel handle</param>
        /// <param name="config">Configuration of the request message (type, network address information...)</param>
        static void testDynamicallyDefineDataIdentifier(cantp_handle channel, uds_msgconfig config)
        {
            uds_status status;
            uds_msg request = new uds_msg();
            uds_msg response = new uds_msg();
            uds_msg confirmation = new uds_msg();
            UInt16[] source_data_identifier = new UInt16[20];
            Byte[] memory_size = new Byte[20];
            Byte[] position_in_source_data_record = new Byte[20];
            UInt16 number_of_elements = 10;
            Byte[] memory_address_buffer = new Byte[15];
            Byte[] memory_size_buffer = new Byte[9];
            Byte memory_address_size;
            Byte memory_size_size;

            CLEAR_CONSOLE();
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("*** UDS Service: DynamicallyDefineDataIdentifier ***");

            // Sends a physical DynamicallyDefineDataIdentifierDBID message
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("Sends a physical DynamicallyDefineDataIdentifierDBID message: ");
            for (int i = 0; i < number_of_elements; i++)
            {
                source_data_identifier[i] = (UInt16)(((0xF0 + i) << 8) + ('A' + i));
                memory_size[i] = (Byte)(i + 1);
                position_in_source_data_record[i] = (Byte)(100 + i);
            }
            status = UDSApi.SvcDynamicallyDefineDataIdentifierDBID_2013(channel, config, out request, UDSApi.uds_svc_param_di.PUDS_SVC_PARAM_DI_CDDID, source_data_identifier,
                memory_size, position_in_source_data_record, number_of_elements);
            if (UDSApi.StatusIsOk_2013(status))
                status = UDSApi.WaitForService_2013(channel, ref request, out response, out confirmation);
            Console.WriteLine(" UDS_SvcDynamicallyDefineDataIdentifierDBID_2013: {0}", (int)status);
            if (UDSApi.StatusIsOk_2013(status))
                display_uds_msg(ref confirmation, ref response, false);
            else
                display_uds_msg_request(ref request, false);
            status = UDSApi.MsgFree_2013(ref request);
            Console.WriteLine(" Free request message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref response);
            Console.WriteLine(" Free response message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref confirmation);
            Console.WriteLine(" Free confirmation message: {0}", (int)status);
            waitGetch();

            // Sends a physical UDS_SvcDynamicallyDefineDataIdentifierDBMA message
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("Sends a physical UDS_SvcDynamicallyDefineDataIdentifierDBMA_2013 message: ");
            number_of_elements = 3;
            memory_address_size = 5;
            memory_size_size = 3;
            for (int j = 0; j < number_of_elements; j++)
            {
                for (int i = 0; i < memory_address_size; i++)
                    memory_address_buffer[memory_address_size * j + i] = (Byte)((10 * j) + i + 1);
                for (int i = 0; i < memory_size_size; i++)
                    memory_size_buffer[memory_size_size * j + i] = (Byte)(100 + (10 * j) + i + 1);
            }
            status = UDSApi.SvcDynamicallyDefineDataIdentifierDBMA_2013(channel, config, out request, UDSApi.uds_svc_param_di.PUDS_SVC_PARAM_DI_CESWNDID, memory_address_size,
                memory_size_size, memory_address_buffer, memory_size_buffer, number_of_elements);
            if (UDSApi.StatusIsOk_2013(status))
                status = UDSApi.WaitForService_2013(channel, ref request, out response, out confirmation);
            Console.WriteLine(" UDS_SvcDynamicallyDefineDataIdentifierDBMA_2013: {0}", (int)status);
            if (UDSApi.StatusIsOk_2013(status))
                display_uds_msg(ref confirmation, ref response, false);
            else
                display_uds_msg_request(ref request, false);
            status = UDSApi.MsgFree_2013(ref request);
            Console.WriteLine(" Free request message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref response);
            Console.WriteLine(" Free response message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref confirmation);
            Console.WriteLine(" Free confirmation message: {0}", (int)status);
            waitGetch();

            // Sends a physical UDS_SvcDynamicallyDefineDataIdentifierCDDDI message
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("Sends a physical UDS_SvcDynamicallyDefineDataIdentifierCDDDI_2013 message: ");
            status = UDSApi.SvcDynamicallyDefineDataIdentifierCDDDI_2013(channel, config, out request, UDSApi.uds_svc_param_di.PUDS_SVC_PARAM_DI_CESWNDID);
            if (UDSApi.StatusIsOk_2013(status))
                status = UDSApi.WaitForService_2013(channel, ref request, out response, out confirmation);
            Console.WriteLine(" UDS_SvcDynamicallyDefineDataIdentifierCDDDI_2013: {0}", (int)status);
            if (UDSApi.StatusIsOk_2013(status))
                display_uds_msg(ref confirmation, ref response, false);
            else
                display_uds_msg_request(ref request, false);
            status = UDSApi.MsgFree_2013(ref request);
            Console.WriteLine(" Free request message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref response);
            Console.WriteLine(" Free response message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref confirmation);
            Console.WriteLine(" Free confirmation message: {0}", (int)status);
            waitGetch();

            // Sends a physical UDS_SvcDynamicallyDefineDataIdentifierClearAllDDDI_2013 message
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("Sends a physical UDS_SvcDynamicallyDefineDataIdentifierClearAllDDDI_2013 message: ");
            status = UDSApi.SvcDynamicallyDefineDataIdentifierClearAllDDDI_2013(channel, config, out request);
            if (UDSApi.StatusIsOk_2013(status))
                status = UDSApi.WaitForService_2013(channel, ref request, out response, out confirmation);
            Console.WriteLine(" UDS_SvcDynamicallyDefineDataIdentifierClearAllDDDI_2013: {0}", (int)status);
            if (UDSApi.StatusIsOk_2013(status))
                display_uds_msg(ref confirmation, ref response, false);
            else
                display_uds_msg_request(ref request, false);
            status = UDSApi.MsgFree_2013(ref request);
            Console.WriteLine(" Free request message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref response);
            Console.WriteLine(" Free response message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref confirmation);
            Console.WriteLine(" Free confirmation message: {0}", (int)status);
            waitGetch();
        }

        /// <summary>Call UDS Service WriteDataByIdentifier</summary>
        /// <param name="channel">cantp channel handle</param>
        /// <param name="config">Configuration of the request message (type, network address information...)</param>
        static void testWriteDataByIdentifier(cantp_handle channel, uds_msgconfig config)
        {
            uds_status status;
            uds_msg request = new uds_msg();
            uds_msg response = new uds_msg();
            uds_msg confirmation = new uds_msg();
            Byte[] data_record = new Byte[10];
            UInt16 data_record_size = 10;

            CLEAR_CONSOLE();
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("*** UDS Service: WriteDataByIdentifier ***");

            // Sends a physical WriteDataByIdentifier message
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("Sends a physical WriteDataByIdentifier message: ");
            for (int i = 0; i < data_record_size; i++)
                data_record[i] = (Byte)('A' + i);
            status = UDSApi.SvcWriteDataByIdentifier_2013(channel, config, out request, UDSApi.uds_svc_param_di.PUDS_SVC_PARAM_DI_ASFPDID, data_record, data_record_size);
            if (UDSApi.StatusIsOk_2013(status))
                status = UDSApi.WaitForService_2013(channel, ref request, out response, out confirmation);
            Console.WriteLine(" UDS_SvcWriteDataByIdentifier_2013: {0}", (int)status);
            if (UDSApi.StatusIsOk_2013(status))
                display_uds_msg(ref confirmation, ref response, false);
            else
                display_uds_msg_request(ref request, false);

            // Free messages
            status = UDSApi.MsgFree_2013(ref request);
            Console.WriteLine(" Free request message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref response);
            Console.WriteLine(" Free response message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref confirmation);
            Console.WriteLine(" Free confirmation message: {0}", (int)status);
            waitGetch();
        }

        /// <summary>Call UDS Service WriteMemoryByAddress</summary>
        /// <param name="channel">cantp channel handle</param>
        /// <param name="config">Configuration of the request message (type, network address information...)</param>
        static void testWriteMemoryByAddress(cantp_handle channel, uds_msgconfig config)
        {
            uds_status status;
            uds_msg request = new uds_msg();
            uds_msg response = new uds_msg();
            uds_msg confirmation = new uds_msg();
            Byte[] data_record = new Byte[50];
            Byte[] memory_address_buffer = new Byte[50];
            Byte[] memory_size_buffer = new Byte[50];
            UInt16 data_record_size = 50;
            Byte memory_address_size = 5;
            Byte memory_size_size = 3;

            CLEAR_CONSOLE();
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("*** UDS Service: WriteMemoryByAddress ***");

            // Sends a physical WriteMemoryByAddress message
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("Sends a physical WriteMemoryByAddress message: ");
            for (int i = 0; i < data_record_size; i++)
            {
                data_record[i] = (Byte)(i + 1);
                memory_address_buffer[i] = (Byte)('A' + i);
                memory_size_buffer[i] = (Byte)(10 + i);
            }
            status = UDSApi.SvcWriteMemoryByAddress_2013(channel, config, out request, memory_address_buffer, memory_address_size, memory_size_buffer, memory_size_size,
                data_record, data_record_size);
            if (UDSApi.StatusIsOk_2013(status))
                status = UDSApi.WaitForService_2013(channel, ref request, out response, out confirmation);
            Console.WriteLine(" UDS_SvcWriteMemoryByAddress_2013: {0}", (int)status);
            if (UDSApi.StatusIsOk_2013(status))
                display_uds_msg(ref confirmation, ref response, false);
            else
                display_uds_msg_request(ref request, false);

            // Free messages
            status = UDSApi.MsgFree_2013(ref request);
            Console.WriteLine(" Free request message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref response);
            Console.WriteLine(" Free response message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref confirmation);
            Console.WriteLine(" Free confirmation message: {0}", (int)status);
            waitGetch();
        }

        /// <summary>Call UDS Service ClearDiagnosticInformation</summary>
        /// <param name="channel">cantp channel handle</param>
        /// <param name="config">Configuration of the request message (type, network address information...)</param>
        static void testClearDiagnosticInformation(cantp_handle channel, uds_msgconfig config)
        {
            uds_status status;
            uds_msg request = new uds_msg();
            uds_msg response = new uds_msg();
            uds_msg confirmation = new uds_msg();

            CLEAR_CONSOLE();
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("*** UDS Service: ClearDiagnosticInformation ***");

            // Sends a physical ClearDiagnosticInformation message
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("Sends a physical ClearDiagnosticInformation message: ");
            status = UDSApi.SvcClearDiagnosticInformation_2013(channel, config, out request, 0xF1A2B3);
            if (UDSApi.StatusIsOk_2013(status))
                status = UDSApi.WaitForService_2013(channel, ref request, out response, out confirmation);
            Console.WriteLine(" UDS_SvcClearDiagnosticInformation_2013: {0}", (int)status);
            if (UDSApi.StatusIsOk_2013(status))
                display_uds_msg(ref confirmation, ref response, false);
            else
                display_uds_msg_request(ref request, false);

            // Free messages
            status = UDSApi.MsgFree_2013(ref request);
            Console.WriteLine(" Free request message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref response);
            Console.WriteLine(" Free response message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref confirmation);
            Console.WriteLine(" Free confirmation message: {0}", (int)status);
            waitGetch();

            // Sends a physical ClearDiagnosticInformation message with memory selection parameter (2020)
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("Sends a physical ClearDiagnosticInformation message with memory selection parameter: ");
            status = UDSApi.SvcClearDiagnosticInformation_2020(channel, config, out request, 0xF1A2B3, 0x42);
            if (UDSApi.StatusIsOk_2013(status))
                status = UDSApi.WaitForService_2013(channel, ref request, out response, out confirmation);
            Console.WriteLine(" UDS_SvcClearDiagnosticInformation_2020: {0}", (int)status);
            if (UDSApi.StatusIsOk_2013(status))
                display_uds_msg(ref confirmation, ref response, false);
            else
                display_uds_msg_request(ref request, false);

            // Free messages
            status = UDSApi.MsgFree_2013(ref request);
            Console.WriteLine(" Free request message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref response);
            Console.WriteLine(" Free response message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref confirmation);
            Console.WriteLine(" Free confirmation message: {0}", (int)status);
            waitGetch();
        }

        /// <summary>Call UDS Service ReadDTCInformation</summary>
        /// <param name="channel">cantp channel handle</param>
        /// <param name="config">Configuration of the request message (type, network address information...)</param>
        static void testReadDTCInformation(cantp_handle channel, uds_msgconfig config)
        {
            uds_status status;
            uds_msg request = new uds_msg();
            uds_msg response = new uds_msg();
            uds_msg confirmation = new uds_msg();

            CLEAR_CONSOLE();
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("*** UDS Service: ReadDTCInformation ***");

            // Sends a physical ReadDTCInformation message
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("Sends a physical ReadDTCInformation message: ");
            status = UDSApi.SvcReadDTCInformation_2013(channel, config, out request, UDSApi.uds_svc_param_rdtci.PUDS_SVC_PARAM_RDTCI_RNODTCBSM, 0xF1);
            if (UDSApi.StatusIsOk_2013(status))
                status = UDSApi.WaitForService_2013(channel, ref request, out response, out confirmation);
            Console.WriteLine(" UDS_SvcReadDTCInformation_2013: {0}", (int)status);
            if (UDSApi.StatusIsOk_2013(status))
                display_uds_msg(ref confirmation, ref response, false);
            else
                display_uds_msg_request(ref request, false);
            status = UDSApi.MsgFree_2013(ref request);
            Console.WriteLine(" Free request message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref response);
            Console.WriteLine(" Free response message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref confirmation);
            Console.WriteLine(" Free confirmation message: {0}", (int)status);
            waitGetch();

            // Sends a physical ReadDTCInformationRDTCSSBDTC message
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("Sends a physical ReadDTCInformationRDTCSSBDTC message: ");
            status = UDSApi.SvcReadDTCInformationRDTCSSBDTC_2013(channel, config, out request, 0x00A1B2B3, 0x12);
            if (UDSApi.StatusIsOk_2013(status))
                status = UDSApi.WaitForService_2013(channel, ref request, out response, out confirmation);
            Console.WriteLine(" UDS_SvcReadDTCInformationRDTCSSBDTC_2013: {0}", (int)status);
            if (UDSApi.StatusIsOk_2013(status))
                display_uds_msg(ref confirmation, ref response, false);
            else
                display_uds_msg_request(ref request, false);
            status = UDSApi.MsgFree_2013(ref request);
            Console.WriteLine(" Free request message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref response);
            Console.WriteLine(" Free response message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref confirmation);
            Console.WriteLine(" Free confirmation message: {0}", (int)status);
            waitGetch();

            // Sends a physical ReadDTCInformationRDTCSSBRN message
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("Sends a physical ReadDTCInformationRDTCSSBRN message: ");
            status = UDSApi.SvcReadDTCInformationRDTCSSBRN_2013(channel, config, out request, 0x12);
            if (UDSApi.StatusIsOk_2013(status))
                status = UDSApi.WaitForService_2013(channel, ref request, out response, out confirmation);
            Console.WriteLine(" UDS_SvcReadDTCInformationRDTCSSBRN_2013: {0}", (int)status);
            if (UDSApi.StatusIsOk_2013(status))
                display_uds_msg(ref confirmation, ref response, false);
            else
                display_uds_msg_request(ref request, false);
            status = UDSApi.MsgFree_2013(ref request);
            Console.WriteLine(" Free request message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref response);
            Console.WriteLine(" Free response message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref confirmation);
            Console.WriteLine(" Free confirmation message: {0}", (int)status);
            waitGetch();

            // Sends a physical ReadDTCInformationReportExtended message
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("Sends a physical ReadDTCInformationReportExtended message: ");
            status = UDSApi.SvcReadDTCInformationReportExtended_2013(channel, config, out request, UDSApi.uds_svc_param_rdtci.PUDS_SVC_PARAM_RDTCI_RDTCEDRBDN, 0x00A1B2B3,
                0x12);
            if (UDSApi.StatusIsOk_2013(status))
                status = UDSApi.WaitForService_2013(channel, ref request, out response, out confirmation);
            Console.WriteLine(" UDS_SvcReadDTCInformationReportExtended_2013: {0}", (int)status);
            if (UDSApi.StatusIsOk_2013(status))
                display_uds_msg(ref confirmation, ref response, false);
            else
                display_uds_msg_request(ref request, false);
            status = UDSApi.MsgFree_2013(ref request);
            Console.WriteLine(" Free request message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref response);
            Console.WriteLine(" Free response message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref confirmation);
            Console.WriteLine(" Free confirmation message: {0}", (int)status);
            waitGetch();

            // Sends a physical ReadDTCInformationReportSeverity message
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("Sends a physical ReadDTCInformationReportSeverity message: ");
            status = UDSApi.SvcReadDTCInformationReportSeverity_2013(channel, config, out request, UDSApi.uds_svc_param_rdtci.PUDS_SVC_PARAM_RDTCI_RNODTCBSMR, 0xF1, 0x12);
            if (UDSApi.StatusIsOk_2013(status))
                status = UDSApi.WaitForService_2013(channel, ref request, out response, out confirmation);
            Console.WriteLine(" UDS_SvcReadDTCInformationReportSeverity_2013: {0}", (int)status);
            if (UDSApi.StatusIsOk_2013(status))
                display_uds_msg(ref confirmation, ref response, false);
            else
                display_uds_msg_request(ref request, false);
            status = UDSApi.MsgFree_2013(ref request);
            Console.WriteLine(" Free request message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref response);
            Console.WriteLine(" Free response message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref confirmation);
            Console.WriteLine(" Free confirmation message: {0}", (int)status);
            waitGetch();

            // Sends a physical ReadDTCInformationRSIODTC message
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("Sends a physical ReadDTCInformationRSIODTC message: ");
            status = UDSApi.SvcReadDTCInformationRSIODTC_2013(channel, config, out request, 0xF1A1B2B3);
            if (UDSApi.StatusIsOk_2013(status))
                status = UDSApi.WaitForService_2013(channel, ref request, out response, out confirmation);
            Console.WriteLine(" UDS_SvcReadDTCInformationRSIODTC_2013: {0}", (int)status);
            if (UDSApi.StatusIsOk_2013(status))
                display_uds_msg(ref confirmation, ref response, false);
            else
                display_uds_msg_request(ref request, false);
            status = UDSApi.MsgFree_2013(ref request);
            Console.WriteLine(" Free request message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref response);
            Console.WriteLine(" Free response message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref confirmation);
            Console.WriteLine(" Free confirmation message: {0}", (int)status);
            waitGetch();

            // Sends a physical ReadDTCInformationNoParam message
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("Sends a physical ReadDTCInformationNoParam message: ");
            status = UDSApi.SvcReadDTCInformationNoParam_2013(channel, config, out request, UDSApi.uds_svc_param_rdtci.PUDS_SVC_PARAM_RDTCI_RSUPDTC);
            if (UDSApi.StatusIsOk_2013(status))
                status = UDSApi.WaitForService_2013(channel, ref request, out response, out confirmation);
            Console.WriteLine(" UDS_SvcReadDTCInformationNoParam_2013: {0}", (int)status);
            if (UDSApi.StatusIsOk_2013(status))
                display_uds_msg(ref confirmation, ref response, false);
            else
                display_uds_msg_request(ref request, false);
            status = UDSApi.MsgFree_2013(ref request);
            Console.WriteLine(" Free request message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref response);
            Console.WriteLine(" Free response message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref confirmation);
            Console.WriteLine(" Free confirmation message: {0}", (int)status);
            waitGetch();

            // Sends a physical UDS_SvcReadDTCInformationRDTCEDBR_2013 message
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("Sends a physical UDS_SvcReadDTCInformationRDTCEDBR_2013 message: ");
            status = UDSApi.SvcReadDTCInformationRDTCEDBR_2013(channel, config, out request, 0x12);
            if (UDSApi.StatusIsOk_2013(status))
                status = UDSApi.WaitForService_2013(channel, ref request, out response, out confirmation);
            Console.WriteLine(" UDS_SvcReadDTCInformationRDTCEDBR_2013: {0}", (int)status);
            if (UDSApi.StatusIsOk_2013(status))
                display_uds_msg(ref confirmation, ref response, false);
            else
                display_uds_msg_request(ref request, false);
            status = UDSApi.MsgFree_2013(ref request);
            Console.WriteLine(" Free request message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref response);
            Console.WriteLine(" Free response message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref confirmation);
            Console.WriteLine(" Free confirmation message: {0}", (int)status);
            waitGetch();

            // Sends a physical UDS_SvcReadDTCInformationRUDMDTCBSM_2013 message
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("Sends a physical UDS_SvcReadDTCInformationRUDMDTCBSM_2013 message: ");
            status = UDSApi.SvcReadDTCInformationRUDMDTCBSM_2013(channel, config, out request, 0x12, 0x34);
            if (UDSApi.StatusIsOk_2013(status))
                status = UDSApi.WaitForService_2013(channel, ref request, out response, out confirmation);
            Console.WriteLine(" UDS_SvcReadDTCInformationRUDMDTCBSM_2013: {0}", (int)status);
            if (UDSApi.StatusIsOk_2013(status))
                display_uds_msg(ref confirmation, ref response, false);
            else
                display_uds_msg_request(ref request, false);
            status = UDSApi.MsgFree_2013(ref request);
            Console.WriteLine(" Free request message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref response);
            Console.WriteLine(" Free response message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref confirmation);
            Console.WriteLine(" Free confirmation message: {0}", (int)status);
            waitGetch();

            // Sends a physical UDS_SvcReadDTCInformationRUDMDTCSSBDTC_2013 message
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("Sends a physical UDS_SvcReadDTCInformationRUDMDTCSSBDTC_2013 message: ");
            status = UDSApi.SvcReadDTCInformationRUDMDTCSSBDTC_2013(channel, config, out request, 0x00123456, 0x78, 0x9A);
            if (UDSApi.StatusIsOk_2013(status))
                status = UDSApi.WaitForService_2013(channel, ref request, out response, out confirmation);
            Console.WriteLine(" UDS_SvcReadDTCInformationRUDMDTCSSBDTC_2013: {0}", (int)status);
            if (UDSApi.StatusIsOk_2013(status))
                display_uds_msg(ref confirmation, ref response, false);
            else
                display_uds_msg_request(ref request, false);
            status = UDSApi.MsgFree_2013(ref request);
            Console.WriteLine(" Free request message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref response);
            Console.WriteLine(" Free response message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref confirmation);
            Console.WriteLine(" Free confirmation message: {0}", (int)status);
            waitGetch();

            // Sends a physical UDS_SvcReadDTCInformationRUDMDTCEDRBDN_2013 message
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("Sends a physical UDS_SvcReadDTCInformationRUDMDTCEDRBDN_2013 message: ");
            status = UDSApi.SvcReadDTCInformationRUDMDTCEDRBDN_2013(channel, config, out request, 0x00123456, 0x78, 0x9A);
            if (UDSApi.StatusIsOk_2013(status))
                status = UDSApi.WaitForService_2013(channel, ref request, out response, out confirmation);
            Console.WriteLine(" UDS_SvcReadDTCInformationRUDMDTCEDRBDN_2013: {0}", (int)status);
            if (UDSApi.StatusIsOk_2013(status))
                display_uds_msg(ref confirmation, ref response, false);
            else
                display_uds_msg_request(ref request, false);
            status = UDSApi.MsgFree_2013(ref request);
            Console.WriteLine(" Free request message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref response);
            Console.WriteLine(" Free response message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref confirmation);
            Console.WriteLine(" Free confirmation message: {0}", (int)status);
            waitGetch();

            // Sends a physical UDS_SvcReadDTCInformationRDTCEDI_2020 message
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("Sends a physical UDS_SvcReadDTCInformationRDTCEDI_2020 message: ");
            status = UDSApi.SvcReadDTCInformationRDTCEDI_2020(channel, config, out request, 0x12);
            if (UDSApi.StatusIsOk_2013(status))
                status = UDSApi.WaitForService_2013(channel, ref request, out response, out confirmation);
            Console.WriteLine(" UDS_SvcReadDTCInformationRDTCEDI_2020: {0}", (int)status);
            if (UDSApi.StatusIsOk_2013(status))
                display_uds_msg(ref confirmation, ref response, false);
            else
                display_uds_msg_request(ref request, false);
            status = UDSApi.MsgFree_2013(ref request);
            Console.WriteLine(" Free request message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref response);
            Console.WriteLine(" Free response message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref confirmation);
            Console.WriteLine(" Free confirmation message: {0}", (int)status);
            waitGetch();

            // Sends a physical UDS_SvcReadDTCInformationRWWHOBDDTCBMR_2013 message
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("Sends a physical UDS_SvcReadDTCInformationRWWHOBDDTCBMR_2013 message: ");
            status = UDSApi.SvcReadDTCInformationRWWHOBDDTCBMR_2013(channel, config, out request, 0x12, 0x34, 0x56);
            if (UDSApi.StatusIsOk_2013(status))
                status = UDSApi.WaitForService_2013(channel, ref request, out response, out confirmation);
            Console.WriteLine(" UDS_SvcReadDTCInformationRWWHOBDDTCBMR_2013: {0}", (int)status);
            if (UDSApi.StatusIsOk_2013(status))
                display_uds_msg(ref confirmation, ref response, false);
            else
                display_uds_msg_request(ref request, false);
            status = UDSApi.MsgFree_2013(ref request);
            Console.WriteLine(" Free request message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref response);
            Console.WriteLine(" Free response message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref confirmation);
            Console.WriteLine(" Free confirmation message: {0}", (int)status);
            waitGetch();

            // Sends a physical UDS_SvcReadDTCInformationRWWHOBDDTCWPS_2013 message
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("Sends a physical UDS_SvcReadDTCInformationRWWHOBDDTCWPS_2013 message: ");
            status = UDSApi.SvcReadDTCInformationRWWHOBDDTCWPS_2013(channel, config, out request, 0x12);
            if (UDSApi.StatusIsOk_2013(status))
                status = UDSApi.WaitForService_2013(channel, ref request, out response, out confirmation);
            Console.WriteLine(" UDS_SvcReadDTCInformationRWWHOBDDTCWPS_2013: {0}", (int)status);
            if (UDSApi.StatusIsOk_2013(status))
                display_uds_msg(ref confirmation, ref response, false);
            else
                display_uds_msg_request(ref request, false);
            status = UDSApi.MsgFree_2013(ref request);
            Console.WriteLine(" Free request message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref response);
            Console.WriteLine(" Free response message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref confirmation);
            Console.WriteLine(" Free confirmation message: {0}", (int)status);
            waitGetch();

            // Sends a physical UDS_SvcReadDTCInformationRDTCBRGI_2020 message
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("Sends a physical UDS_SvcReadDTCInformationRDTCBRGI_2020 message: ");
            status = UDSApi.SvcReadDTCInformationRDTCBRGI_2020(channel, config, out request, 0x12, 0x34);
            if (UDSApi.StatusIsOk_2013(status))
                status = UDSApi.WaitForService_2013(channel, ref request, out response, out confirmation);
            Console.WriteLine(" UDS_SvcReadDTCInformationRDTCBRGI_2020: {0}", (int)status);
            if (UDSApi.StatusIsOk_2013(status))
                display_uds_msg(ref confirmation, ref response, false);
            else
                display_uds_msg_request(ref request, false);
            status = UDSApi.MsgFree_2013(ref request);
            Console.WriteLine(" Free request message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref response);
            Console.WriteLine(" Free response message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref confirmation);
            Console.WriteLine(" Free confirmation message: {0}", (int)status);
            waitGetch();
        }

        /// <summary>Call UDS Service InputOutputControlByIdentifier</summary>
        /// <param name="channel">cantp channel handle</param>
        /// <param name="config">Configuration of the request message (type, network address information...)</param>
        static void testInputOutputControlByIdentifier(cantp_handle channel, uds_msgconfig config)
        {
            uds_status status;
            uds_msg request = new uds_msg();
            uds_msg response = new uds_msg();
            uds_msg confirmation = new uds_msg();
            Byte[] control_option_record = new Byte[10];
            Byte[] control_enable_mask_record = new Byte[10];
            UInt16 control_option_record_size = 10;
            UInt16 control_enable_mask_record_size = 5;

            CLEAR_CONSOLE();
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("*** UDS Service: InputOutputControlByIdentifier ***");

            // Sends a physical InputOutputControlByIdentifier message
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("Sends a physical InputOutputControlByIdentifier message: ");
            for (int i = 0; i < control_option_record_size; i++)
            {
                control_option_record[i] = (Byte)('A' + i);
                control_enable_mask_record[i] = (Byte)(10 + i);
            }
            status = UDSApi.SvcInputOutputControlByIdentifier_2013(channel, config, out request, UDSApi.uds_svc_param_di.PUDS_SVC_PARAM_DI_SSECUSWVNDID, control_option_record,
                control_option_record_size, control_enable_mask_record, control_enable_mask_record_size);
            if (UDSApi.StatusIsOk_2013(status))
                status = UDSApi.WaitForService_2013(channel, ref request, out response, out confirmation);
            Console.WriteLine(" UDS_SvcInputOutputControlByIdentifier_2013: {0}", (int)status);
            if (UDSApi.StatusIsOk_2013(status))
                display_uds_msg(ref confirmation, ref response, false);
            else
                display_uds_msg_request(ref request, false);

            // Free messages
            status = UDSApi.MsgFree_2013(ref request);
            Console.WriteLine(" Free request message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref response);
            Console.WriteLine(" Free response message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref confirmation);
            Console.WriteLine(" Free confirmation message: {0}", (int)status);
            waitGetch();
        }

        /// <summary>Call UDS Service RoutineControl</summary>
        /// <param name="channel">cantp channel handle</param>
        /// <param name="config">Configuration of the request message (type, network address information...)</param>
        static void testRoutineControl(cantp_handle channel, uds_msgconfig config)
        {
            uds_status status;
            uds_msg request = new uds_msg();
            uds_msg response = new uds_msg();
            uds_msg confirmation = new uds_msg();
            Byte[] routine_control_option_record = new Byte[10];
            UInt16 routine_control_option_record_size = 10;

            CLEAR_CONSOLE();
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("*** UDS Service: RoutineControl ***");

            // Sends a physical RoutineControl message
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("Sends a physical RoutineControl message: ");
            for (int i = 0; i < routine_control_option_record_size; i++)
            {
                routine_control_option_record[i] = (Byte)('A' + i);
            }
            status = UDSApi.SvcRoutineControl_2013(channel, config, out request, UDSApi.uds_svc_param_rc.PUDS_SVC_PARAM_RC_RRR,
                (UDSApi.uds_svc_param_rc_rid)0xF1A2, routine_control_option_record, routine_control_option_record_size);
            if (UDSApi.StatusIsOk_2013(status))
                status = UDSApi.WaitForService_2013(channel, ref request, out response, out confirmation);
            Console.WriteLine(" UDS_SvcRoutineControl_2013: {0}", (int)status);
            if (UDSApi.StatusIsOk_2013(status))
                display_uds_msg(ref confirmation, ref response, false);
            else
                display_uds_msg_request(ref request, false);

            // Free messages
            status = UDSApi.MsgFree_2013(ref request);
            Console.WriteLine(" Free request message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref response);
            Console.WriteLine(" Free response message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref confirmation);
            Console.WriteLine(" Free confirmation message: {0}", (int)status);
            waitGetch();
        }

        /// <summary>Call UDS Service RequestDownload</summary>
        /// <param name="channel">cantp channel handle</param>
        /// <param name="config">Configuration of the request message (type, network address information...)</param>
        static void testRequestDownload(cantp_handle channel, uds_msgconfig config)
        {
            uds_status status;
            uds_msg request = new uds_msg();
            uds_msg response = new uds_msg();
            uds_msg confirmation = new uds_msg();
            Byte[] memory_address_buffer = new Byte[15];
            Byte[] memory_size_buffer = new Byte[15];
            Byte memory_address_size = 15;
            Byte memory_size_size = 8;

            CLEAR_CONSOLE();
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("*** UDS Service: RequestDownload ***");

            // Sends a physical RequestDownload message
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("Sends a physical RequestDownload message: ");
            for (int i = 0; i < memory_address_size; i++)
            {
                memory_address_buffer[i] = (Byte)('A' + i);
                memory_size_buffer[i] = (Byte)(10 + i);
            }
            status = UDSApi.SvcRequestDownload_2013(channel, config, out request, 0x01, 0x02, memory_address_buffer, memory_address_size, memory_size_buffer,
                memory_size_size);
            if (UDSApi.StatusIsOk_2013(status))
                status = UDSApi.WaitForService_2013(channel, ref request, out response, out confirmation);
            Console.WriteLine(" UDS_SvcRequestDownload_2013: {0}", (int)status);
            if (UDSApi.StatusIsOk_2013(status))
                display_uds_msg(ref confirmation, ref response, false);
            else
                display_uds_msg_request(ref request, false);

            // Free messages
            status = UDSApi.MsgFree_2013(ref request);
            Console.WriteLine(" Free request message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref response);
            Console.WriteLine(" Free response message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref confirmation);
            Console.WriteLine(" Free confirmation message: {0}", (int)status);
            waitGetch();
        }

        /// <summary>Call UDS Service RequestUpload</summary>
        /// <param name="channel">cantp channel handle</param>
        /// <param name="config">Configuration of the request message (type, network address information...)</param>
        static void testRequestUpload(cantp_handle channel, uds_msgconfig config)
        {
            uds_status status;
            uds_msg request = new uds_msg();
            uds_msg response = new uds_msg();
            uds_msg confirmation = new uds_msg();
            Byte[] memory_address_buffer = new Byte[4];
            Byte[] memory_size_buffer = new Byte[4];
            Byte memory_address_size = 4;
            Byte memory_size_size = 4;

            CLEAR_CONSOLE();
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("*** UDS Service: RequestUpload ***");

            // Sends a physical RequestUpload message
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("Sends a physical RequestUpload message: ");
            for (int i = 0; i < memory_size_size; i++)
            {
                memory_address_buffer[i] = (Byte)('A' + i);
                memory_size_buffer[i] = (Byte)(10 + i);
            }
            status = UDSApi.SvcRequestUpload_2013(channel, config, out request, 0x01, 0x02, memory_address_buffer, memory_address_size, memory_size_buffer,
                memory_size_size);
            if (UDSApi.StatusIsOk_2013(status))
                status = UDSApi.WaitForService_2013(channel, ref request, out response, out confirmation);
            Console.WriteLine(" UDS_SvcRequestUpload_2013: {0}", (int)status);
            if (UDSApi.StatusIsOk_2013(status))
                display_uds_msg(ref confirmation, ref response, false);
            else
                display_uds_msg_request(ref request, false);

            // Free messages
            status = UDSApi.MsgFree_2013(ref request);
            Console.WriteLine(" Free request message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref response);
            Console.WriteLine(" Free response message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref confirmation);
            Console.WriteLine(" Free confirmation message: {0}", (int)status);
            waitGetch();
        }

        /// <summary>Call UDS Service RequestTransferExit</summary>
        /// <param name="channel">cantp channel handle</param>
        /// <param name="config">Configuration of the request message (type, network address information...)</param>
        static void testRequestTransferExit(cantp_handle channel, uds_msgconfig config)
        {
            uds_status status;
            uds_msg request = new uds_msg();
            uds_msg response = new uds_msg();
            uds_msg confirmation = new uds_msg();
            Byte[] record = new Byte[50];
            Byte record_size = 20;

            CLEAR_CONSOLE();
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("*** UDS Service: RequestTransferExit ***");

            // Sends a physical RequestTransferExit message
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("Sends a physical RequestTransferExit message: ");
            for (int i = 0; i < record_size; i++)
            {
                record[i] = (Byte)('A' + i);
            }
            status = UDSApi.SvcRequestTransferExit_2013(channel, config, out request, record, record_size);
            if (UDSApi.StatusIsOk_2013(status))
                status = UDSApi.WaitForService_2013(channel, ref request, out response, out confirmation);
            Console.WriteLine(" UDS_SvcRequestTransferExit_2013: {0}", (int)status);
            if (UDSApi.StatusIsOk_2013(status))
                display_uds_msg(ref confirmation, ref response, false);
            else
                display_uds_msg_request(ref request, false);

            // Free messages
            status = UDSApi.MsgFree_2013(ref request);
            Console.WriteLine(" Free request message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref response);
            Console.WriteLine(" Free response message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref confirmation);
            Console.WriteLine(" Free confirmation message: {0}", (int)status);
            waitGetch();
        }

        /// <summary>Call UDS Service TransferData</summary>
        /// <param name="channel">cantp channel handle</param>
        /// <param name="config">Configuration of the request message (type, network address information...)</param>
        static void testTransferData(cantp_handle channel, uds_msgconfig config)
        {
            uds_status status;
            uds_msg request = new uds_msg();
            uds_msg response = new uds_msg();
            uds_msg confirmation = new uds_msg();
            Byte[] record = new Byte[50];
            Byte record_size = 50;

            CLEAR_CONSOLE();
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("*** UDS Service: TransferData ***");

            // Sends a physical TransferData message
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("Sends a physical TransferData message: ");
            for (int i = 0; i < record_size; i++)
            {
                record[i] = (Byte)('A' + i);
            }
            status = UDSApi.SvcTransferData_2013(channel, config, out request, 0x01, record, record_size);
            if (UDSApi.StatusIsOk_2013(status))
                status = UDSApi.WaitForService_2013(channel, ref request, out response, out confirmation);
            Console.WriteLine(" UDS_SvcTransferData_2013: {0}", (int)status);
            if (UDSApi.StatusIsOk_2013(status))
                display_uds_msg(ref confirmation, ref response, false);
            else
                display_uds_msg_request(ref request, false);

            // Free messages
            status = UDSApi.MsgFree_2013(ref request);
            Console.WriteLine(" Free request message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref response);
            Console.WriteLine(" Free response message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref confirmation);
            Console.WriteLine(" Free confirmation message: {0}", (int)status);
            waitGetch();
        }

        /// <summary>Call UDS Service TransferData with MAX_DATA length</summary>
        /// <param name="channel">cantp channel handle</param>
        /// <param name="config">Configuration of the request message (type, network address information...)</param>
        static void testTransferDataBigMessage(cantp_handle channel, uds_msgconfig config)
        {
            uds_status status;
            uds_msg request = new uds_msg();
            uds_msg response = new uds_msg();
            uds_msg confirmation = new uds_msg();
            Byte[] record = new Byte[4095];
            UInt16 record_size = 4093;

            CLEAR_CONSOLE();
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("*** UDS Service: TransferData with MAX_DATA ***");

            // Sends a physical TransferData message with the maximum data available. The goal is to show that
            // WaitForService_2013 does not return a TIMEOUT error although the transmit and receive time of all the
            // data will be longer than the default time to get a response.
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("Sends a physical TransferData message (length={0}): ", record_size);
            for (int i = 0; i < record_size; i++)
                record[i] = (Byte)('A' + i);
            status = UDSApi.SvcTransferData_2013(channel, config, out request, 0x01, record, record_size);
            if (UDSApi.StatusIsOk_2013(status))
                status = UDSApi.WaitForService_2013(channel, ref request, out response, out confirmation);
            Console.WriteLine(" UDS_SvcTransferData_2013: {0}", (int)status);
            if (UDSApi.StatusIsOk_2013(status))
                display_uds_msg(ref confirmation, ref response, false);
            else
                display_uds_msg_request(ref request, false);

            // Free messages
            status = UDSApi.MsgFree_2013(ref request);
            Console.WriteLine(" Free request message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref response);
            Console.WriteLine(" Free response message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref confirmation);
            Console.WriteLine(" Free confirmation message: {0}", (int)status);
            waitGetch();
        }


        /// <summary>Call UDS Service TransferData</summary>
        /// <param name="channel">cantp channel handle</param>
        /// <param name="config">Configuration of the request message (type, network address information...)</param>
        static void testTransferDataMultipleFunctionalMessage(cantp_handle channel, uds_msgconfig config)
        {
            uds_status status;
            uds_msg request = new uds_msg();
            uds_msg[] tresponse = new uds_msg[1];
            uds_msg confirmation = new uds_msg();
            Byte[] record = new Byte[5];
            UInt16 record_size = 5;
            UInt32 response_count = 0;


            CLEAR_CONSOLE();
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("*** UDS Service: TransferData with functional message***");

            // Initialize request message
            config.nai.target_addr = (UInt16)uds_address.PUDS_ADDRESS_ISO_15765_4_ADDR_OBD_FUNCTIONAL;
            config.nai.target_type = cantp_isotp_addressing.PCANTP_ISOTP_ADDRESSING_FUNCTIONAL;

            // Sends a functional TransferData message. The goal is to show that UDS_WaitForServiceFunctional_2013 waits long
            // enough to fetch all possible ECU responses.
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("Sends a functional TransferData message: ");
            for (int i = 0; i < record_size; i++)
            {
                record[i] = (Byte)('A' + i);
            }
            status = UDSApi.SvcTransferData_2013(channel, config, out request, 0x01, record, record_size);
            if (UDSApi.StatusIsOk_2013(status))
                status = UDSApi.WaitForServiceFunctional_2013(channel, ref request, 1, true, tresponse, out response_count, out confirmation);
            Console.WriteLine(" UDS_SvcTransferData_2013: {0}", (int)status);
            if (UDSApi.StatusIsOk_2013(status))
            {
                display_uds_msg_request(ref request, true);
                Console.WriteLine(); Console.WriteLine(" Received {0} UDS responses:", response_count);
                display_uds_msg_response(ref tresponse[0], false);
            }
            else
                display_uds_msg_request(ref request, false);

            // Free messages
            status = UDSApi.MsgFree_2013(ref request);
            Console.WriteLine(" Free request message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref tresponse[0]);
            Console.WriteLine(" Free response message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref confirmation);
            Console.WriteLine(" Free confirmation message: {0}", (int)status);
            waitGetch();
        }

        /// <summary>Sample to use event</summary>
        /// <param name="channel">cantp channel handle</param>
        /// <param name="config">Configuration of the request message (type, network address information...)</param>
        static void testUsingEvent(cantp_handle channel, uds_msgconfig config)
        {
            uds_status status;
            uds_status read_status;
            uds_msg request = new uds_msg();
            uds_msg response = new uds_msg();
            UInt32 tmp_event_handle;
            bool res;
            bool stop;

            // set event handler
            System.Threading.AutoResetEvent receive_event = new System.Threading.AutoResetEvent(false);
            tmp_event_handle = Convert.ToUInt32(receive_event.SafeWaitHandle.DangerousGetHandle().ToInt32());
            status = UDSApi.SetValue_2013(cantp_handle.PCANTP_HANDLE_USBBUS1, uds_parameter.PUDS_PARAMETER_RECEIVE_EVENT, ref tmp_event_handle, sizeof(UInt32));

            if (!UDSApi.StatusIsOk_2013(status, uds_status.PUDS_STATUS_OK, false))
            {
                Console.WriteLine("Failed to set event, aborting...");
                receive_event.Close();
                waitGetch();
                return;
            }

            CLEAR_CONSOLE();
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("*** UDS Service with Event: TesterPresent ***");

            // Sends a physical TesterPresent message
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("Sends a physical TesterPresent message: ");
            status = UDSApi.SvcTesterPresent_2013(channel, config, out request, UDSApi.uds_svc_param_tp.PUDS_SVC_PARAM_TP_ZSUBF);
            Console.WriteLine(" UDS_SvcTesterPresent_2013: {0}", (int)status);

            if (UDSApi.StatusIsOk_2013(status, uds_status.PUDS_STATUS_OK, false))
            {

                // Instead of calling WaitForService function, this sample demonstrates how event can be used. But note that
                // the use of a thread listening to notifications and making the read operations is preferred.
                stop = false;

                // wait until we receive expected response
                do
                {
                    res = receive_event.WaitOne(System.Threading.Timeout.Infinite);
                    if (res)
                    {

                        // read all messages
                        do
                        {
                            read_status = UDSApi.Read_2013(channel, out response);
                            if (UDSApi.StatusIsOk_2013(read_status, uds_status.PUDS_STATUS_OK, false))
                            {
                                // this is a simple message check (type and sender/receiver address): to filter UDS request
                                // confirmation and get first message from target, but real use-case should check that the UDS
                                // service ID matches the request
                                if (response.msg.Msgdata_isotp_Copy.netaddrinfo.msgtype == cantp_isotp_msgtype.PCANTP_ISOTP_MSGTYPE_DIAGNOSTIC
                                    && response.msg.Msgdata_isotp_Copy.netaddrinfo.source_addr == config.nai.target_addr
                                    && response.msg.Msgdata_isotp_Copy.netaddrinfo.target_addr == config.nai.source_addr)
                                {
                                    stop = true;
                                    display_uds_msg(ref request, ref response, false);
                                }
                            }

                            // Free response message
                            status = UDSApi.MsgFree_2013(ref response);
                            Console.WriteLine(" Free response message: {0}", (int)status);
                        } while (!UDSApi.StatusIsOk_2013(read_status, uds_status.PUDS_STATUS_NO_MESSAGE));
                    }
                } while (!stop);
            }
            waitGetch();

            // Free request message
            status = UDSApi.MsgFree_2013(ref request);
            Console.WriteLine(" Free request message: {0}", (int)status);

            // Uninitialize event
            tmp_event_handle = 0;
            status = UDSApi.SetValue_2013(channel, uds_parameter.PUDS_PARAMETER_RECEIVE_EVENT, ref tmp_event_handle, sizeof(UInt32));
            Console.WriteLine(" Remove receive event: {0}", (int)status);
        }



        /// <summary>Call UDS Service RequestFileTransfer</summary>
        /// <param name="channel">cantp channel handle</param>
        /// <param name="config">Configuration of the request message (type, network address information...)</param>
        static void testRequestFileTransfer(cantp_handle channel, uds_msgconfig config)
        {
            uds_status status;
            uds_msg request = new uds_msg();
            uds_msg response = new uds_msg();
            uds_msg confirmation = new uds_msg();
            Byte[] file_size_uncompressed = { 0xD, 0x00 };
            Byte[] file_size_compressed = { 0xA, 0x00 };
            Byte file_size_length = 2;
            string sfile_name = "toto.txt";
            UInt16 file_name_length = (UInt16)sfile_name.Length;


            CLEAR_CONSOLE();
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("*** UDS Service: RequestFileTransfer ***");

            // Sends a physical RequestFileTransfer message
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("Sends a physical RequestFileTransfer message: ");
            status = UDSApi.SvcRequestFileTransfer_2013(channel, config, out request, UDSApi.uds_svc_param_rft_moop.PUDS_SVC_PARAM_RFT_MOOP_REPLFILE, file_name_length,
                sfile_name, 0, 0, file_size_length, file_size_uncompressed, file_size_compressed);
            if (UDSApi.StatusIsOk_2013(status))
                status = UDSApi.WaitForService_2013(channel, ref request, out response, out confirmation);
            Console.WriteLine(" UDS_SvcRequestFileTransfer_2013: {0}", (int)status);
            if (UDSApi.StatusIsOk_2013(status))
                display_uds_msg(ref confirmation, ref response, false);
            else
                display_uds_msg_request(ref request, false);

            // Free messages
            status = UDSApi.MsgFree_2013(ref request);
            Console.WriteLine(" Free request message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref response);
            Console.WriteLine(" Free response message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref confirmation);
            Console.WriteLine(" Free confirmation message: {0}", (int)status);
            waitGetch();
        }

        /// <summary>Call UDS Service AccessTimingParameter</summary>
        /// <param name="channel">cantp channel handle</param>
        /// <param name="config">Configuration of the request message (type, network address information...)</param>
        static void testAccessTimingParameter(cantp_handle channel, uds_msgconfig config)
        {
            uds_status status;
            uds_msg request = new uds_msg();
            uds_msg response = new uds_msg();
            uds_msg confirmation = new uds_msg();
            Byte[] request_record = { 0xAB, 0xCD };
            UInt32 record_size = 2;


            CLEAR_CONSOLE();
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("*** UDS Service: AccessTimingParameter ***");

            // Sends a physical AccessTimingParameter message
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("Sends a physical AccessTimingParameter message: ");
            status = UDSApi.SvcAccessTimingParameter_2013(channel, config, out request, UDSApi.uds_svc_param_atp.PUDS_SVC_PARAM_ATP_RCATP, request_record, record_size);
            if (UDSApi.StatusIsOk_2013(status))
                status = UDSApi.WaitForService_2013(channel, ref request, out response, out confirmation);
            Console.WriteLine(" UDS_SvcAccessTimingParameter_2013: {0}", (int)status);
            if (UDSApi.StatusIsOk_2013(status))
                display_uds_msg(ref confirmation, ref response, false);
            else
                display_uds_msg_request(ref request, false);

            // Free messages
            status = UDSApi.MsgFree_2013(ref request);
            Console.WriteLine(" Free request message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref response);
            Console.WriteLine(" Free response message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref confirmation);
            Console.WriteLine(" Free confirmation message: {0}", (int)status);
            waitGetch();
        }

        /// <summary>Call UDS Service Authentication</summary>
        /// <param name="channel">cantp channel handle</param>
        /// <param name="config">Configuration of the request message (type, network address information...)</param>
        static void testAuthentication(cantp_handle channel, uds_msgconfig config)
        {
            uds_status status;
            uds_msg request = new Peak.Can.Uds.uds_msg();
            uds_msg response = new uds_msg();
            uds_msg confirmation = new uds_msg();

            Byte communication_configuration = 0;
            Byte[] certificate_client = new Byte[2] { 0x12, 0x34 };
            UInt16 certificate_client_size = 2;
            Byte[] challenge_client = new Byte[2] { 0x56, 0x78 };
            UInt16 challenge_client_size = 2;
            Byte[] proof_of_ownership_client = new Byte[2] { 0x9A, 0xBC };
            UInt16 proof_of_ownership_client_size = 2;
            Byte[] ephemeral_public_key_client = new Byte[2] { 0xDE, 0xF0 };
            UInt16 ephemeral_public_key_client_size = 2;
            Byte[] algorithm_indicator = new Byte[16] { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F };
            Byte[] additional_parameter = new Byte[2] { 0xAA, 0xBB };
            UInt16 additional_parameter_size = 2;

            CLEAR_CONSOLE();
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("*** UDS Service: Authentication ***");

            // Sends a physical Authentication/deAuthenticate message
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("Sends a physical Authentication/deAuthenticate message: ");
            status = UDSApi.SvcAuthenticationDA_2020(channel, config, out request);
            if (UDSApi.StatusIsOk_2013(status, uds_status.PUDS_STATUS_OK, false))
                status = UDSApi.WaitForService_2013(channel, ref request, out response, out confirmation);
            Console.WriteLine(" UDS_SvcAuthenticationDA_2020: {0}", (int)status);
            if (UDSApi.StatusIsOk_2013(status, uds_status.PUDS_STATUS_OK, false))
                display_uds_msg(ref confirmation, ref response, false);
            else
                display_uds_msg_request(ref request, false);

            // Free messages
            status = UDSApi.MsgFree_2013(ref request);
            Console.WriteLine(" Free request message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref response);
            Console.WriteLine(" Free response message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref confirmation);
            Console.WriteLine(" Free confirmation message: {0}", (int)status);
            waitGetch();


            // Sends a physical Authentication/verifyCertificateUnidirectional message
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("Sends a physical Authentication/verifyCertificateUnidirectional message: ");
            status = UDSApi.SvcAuthenticationVCU_2020(channel, config, out request, communication_configuration, certificate_client, certificate_client_size, challenge_client, challenge_client_size);
            if (UDSApi.StatusIsOk_2013(status, uds_status.PUDS_STATUS_OK, false))
                status = UDSApi.WaitForService_2013(channel, ref request, out response, out confirmation);
            Console.WriteLine(" UDS_SvcAuthenticationVCU_2020: {0}", (int)status);
            if (UDSApi.StatusIsOk_2013(status, uds_status.PUDS_STATUS_OK, false))
                display_uds_msg(ref confirmation, ref response, false);
            else
                display_uds_msg_request(ref request, false);

            // Free messages
            status = UDSApi.MsgFree_2013(ref request);
            Console.WriteLine(" Free request message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref response);
            Console.WriteLine(" Free response message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref confirmation);
            Console.WriteLine(" Free confirmation message: {0}", (int)status);
            waitGetch();


            // Sends a physical Authentication/verifyCertificateBidirectional message
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("Sends a physical Authentication/verifyCertificateBidirectional message: ");
            status = UDSApi.SvcAuthenticationVCB_2020(channel, config, out request, communication_configuration, certificate_client, certificate_client_size, challenge_client, challenge_client_size);
            if (UDSApi.StatusIsOk_2013(status, uds_status.PUDS_STATUS_OK, false))
                status = UDSApi.WaitForService_2013(channel, ref request, out response, out confirmation);
            Console.WriteLine(" UDS_SvcAuthenticationVCB_2020: {0}", (int)status);
            if (UDSApi.StatusIsOk_2013(status, uds_status.PUDS_STATUS_OK, false))
                display_uds_msg(ref confirmation, ref response, false);
            else
                display_uds_msg_request(ref request, false);

            // Free messages
            status = UDSApi.MsgFree_2013(ref request);
            Console.WriteLine(" Free request message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref response);
            Console.WriteLine(" Free response message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref confirmation);
            Console.WriteLine(" Free confirmation message: {0}", (int)status);
            waitGetch();


            // Sends a physical Authentication/proofOfOwnership message
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("Sends a physical Authentication/proofOfOwnership message: ");
            status = UDSApi.SvcAuthenticationPOWN_2020(channel, config, out request, proof_of_ownership_client, proof_of_ownership_client_size, ephemeral_public_key_client, ephemeral_public_key_client_size);
            if (UDSApi.StatusIsOk_2013(status, uds_status.PUDS_STATUS_OK, false))
                status = UDSApi.WaitForService_2013(channel, ref request, out response, out confirmation);
            Console.WriteLine(" UDS_SvcAuthenticationPOWN_2020: {0}", (int)status);
            if (UDSApi.StatusIsOk_2013(status, uds_status.PUDS_STATUS_OK, false))
                display_uds_msg(ref confirmation, ref response, false);
            else
                display_uds_msg_request(ref request, false);

            // Free messages
            status = UDSApi.MsgFree_2013(ref request);
            Console.WriteLine(" Free request message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref response);
            Console.WriteLine(" Free response message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref confirmation);
            Console.WriteLine(" Free confirmation message: {0}", (int)status);
            waitGetch();


            // Sends a physical Authentication/requestChallengeForAuthentication message
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("Sends a physical Authentication/requestChallengeForAuthentication message: ");
            status = UDSApi.SvcAuthenticationRCFA_2020(channel, config, out request, communication_configuration, algorithm_indicator);
            if (UDSApi.StatusIsOk_2013(status, uds_status.PUDS_STATUS_OK, false))
                status = UDSApi.WaitForService_2013(channel, ref request, out response, out confirmation);
            Console.WriteLine(" UDS_SvcAuthenticationRCFA_2020: {0}", (int)status);
            if (UDSApi.StatusIsOk_2013(status, uds_status.PUDS_STATUS_OK, false))
                display_uds_msg(ref confirmation, ref response, false);
            else
                display_uds_msg_request(ref request, false);

            // Free messages
            status = UDSApi.MsgFree_2013(ref request);
            Console.WriteLine(" Free request message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref response);
            Console.WriteLine(" Free response message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref confirmation);
            Console.WriteLine(" Free confirmation message: {0}", (int)status);
            waitGetch();


            // Sends a physical Authentication/verifyProofOfOwnershipUnidirectional message
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("Sends a physical Authentication/verifyProofOfOwnershipUnidirectional message: ");
            status = UDSApi.SvcAuthenticationVPOWNU_2020(channel, config, out request, algorithm_indicator, proof_of_ownership_client, proof_of_ownership_client_size, challenge_client, challenge_client_size, additional_parameter, additional_parameter_size);
            if (UDSApi.StatusIsOk_2013(status, uds_status.PUDS_STATUS_OK, false))
                status = UDSApi.WaitForService_2013(channel, ref request, out response, out confirmation);
            Console.WriteLine(" UDS_SvcAuthenticationVPOWNU_2020: {0}", (int)status);
            if (UDSApi.StatusIsOk_2013(status, uds_status.PUDS_STATUS_OK, false))
                display_uds_msg(ref confirmation, ref response, false);
            else
                display_uds_msg_request(ref request, false);

            // Free messages
            status = UDSApi.MsgFree_2013(ref request);
            Console.WriteLine(" Free request message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref response);
            Console.WriteLine(" Free response message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref confirmation);
            Console.WriteLine(" Free confirmation message: {0}", (int)status);
            waitGetch();


            // Sends a physical Authentication/verifyProofOfOwnershipBidirectional message
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("Sends a physical Authentication/verifyProofOfOwnershipBidirectional message: ");
            status = UDSApi.SvcAuthenticationVPOWNB_2020(channel, config, out request, algorithm_indicator, proof_of_ownership_client, proof_of_ownership_client_size, challenge_client, challenge_client_size, additional_parameter, additional_parameter_size);
            if (UDSApi.StatusIsOk_2013(status))
                status = UDSApi.WaitForService_2013(channel, ref request, out response, out confirmation);
            Console.WriteLine(" UDS_SvcAuthenticationVPOWNB_2020: {0}", (int)status);
            if (UDSApi.StatusIsOk_2013(status))
                display_uds_msg(ref confirmation, ref response, false);
            else
                display_uds_msg_request(ref request, false);

            // Free messages
            status = UDSApi.MsgFree_2013(ref request);
            Console.WriteLine(" Free request message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref response);
            Console.WriteLine(" Free response message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref confirmation);
            Console.WriteLine(" Free confirmation message: {0}", (int)status);
            waitGetch();


            // Sends a physical Authentication/authenticationConfiguration message
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("Sends a physical Authentication/authenticationConfiguration message: ");
            status = UDSApi.SvcAuthenticationAC_2020(channel, config, out request);
            if (UDSApi.StatusIsOk_2013(status))
                status = UDSApi.WaitForService_2013(channel, ref request, out response, out confirmation);
            Console.WriteLine(" UDS_SvcAuthenticationAC_2020: {0}", (int)status);
            if (UDSApi.StatusIsOk_2013(status))
                display_uds_msg(ref confirmation, ref response, false);
            else
                display_uds_msg_request(ref request, false);

            // Free messages
            status = UDSApi.MsgFree_2013(ref request);
            Console.WriteLine(" Free request message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref response);
            Console.WriteLine(" Free response message: {0}", (int)status);
            status = UDSApi.MsgFree_2013(ref confirmation);
            Console.WriteLine(" Free confirmation message: {0}", (int)status);
            waitGetch();
        }

        /// <summary>Entry point of the program, start a CAN UDS client</summary>
        static void Main(string[] args)
        {
            cantp_handle client_handle;
            uds_status status;
            UInt32 dw_buffer;
            uds_msgconfig config = new uds_msgconfig();

            // Set the PCAN-Channel to use
            client_handle = cantp_handle.PCANTP_HANDLE_USBBUS1; // TODO: modify the value according to your available PCAN devices.

            // Initializing of the UDS Communication session
            status = UDSApi.Initialize_2013(client_handle, cantp_baudrate.PCANTP_BAUDRATE_500K);
            Console.WriteLine("Initialize UDS: {0}", (int)status);

            // Define TimeOuts
            dw_buffer = CanTpApi.PCANTP_ISO_TIMEOUTS_15765_4;
            status = UDSApi.SetValue_2013(client_handle, uds_parameter.PUDS_PARAMETER_ISO_TIMEOUTS, ref dw_buffer, sizeof(UInt32));
            Console.WriteLine("Set ISO 15765-4 timeouts values: {0}", (int)status);
            waitGetch();

            // Define Network Address Information used for all the tests
            config.can_id = (UInt32)0xFFFFFFFF;
            config.can_msgtype = cantp_can_msgtype.PCANTP_CAN_MSGTYPE_STANDARD;
            config.nai.protocol = uds_msgprotocol.PUDS_MSGPROTOCOL_ISO_15765_2_11B_NORMAL;
            config.nai.target_type = cantp_isotp_addressing.PCANTP_ISOTP_ADDRESSING_PHYSICAL;
            config.type = uds_msgtype.PUDS_MSGTYPE_USDT;
            config.nai.source_addr = (UInt16)uds_address.PUDS_ADDRESS_ISO_15765_4_ADDR_TEST_EQUIPMENT;
            config.nai.target_addr = (UInt16)uds_address.PUDS_ADDRESS_ISO_15765_4_ADDR_ECU_1;

            // The following functions call UDS Services
            testDiagnosticSessionControl(client_handle, config);
            testECUReset(client_handle, config);
            testSecurityAccess(client_handle, config);
            testCommunicationControl(client_handle, config);
            testTesterPresent(client_handle, config);
            testSecuredDataTransmission(client_handle, config);
            testControlDTCSetting(client_handle, config);
            testResponseOnEvent(client_handle, config);
            testLinkControl(client_handle, config);
            testReadDataByIdentifier(client_handle, config);
            testReadMemoryByAddress(client_handle, config);
            testReadScalingDataByIdentifier(client_handle, config);
            testReadDataByPeriodicIdentifier(client_handle, config);
            testDynamicallyDefineDataIdentifier(client_handle, config);
            testWriteDataByIdentifier(client_handle, config);
            testWriteMemoryByAddress(client_handle, config);
            testClearDiagnosticInformation(client_handle, config);
            testReadDTCInformation(client_handle, config);
            testInputOutputControlByIdentifier(client_handle, config);
            testRoutineControl(client_handle, config);
            testRequestDownload(client_handle, config);
            testRequestUpload(client_handle, config);
            testTransferData(client_handle, config);
            testRequestTransferExit(client_handle, config);
            testAccessTimingParameter(client_handle, config);
            testRequestFileTransfer(client_handle, config);
            testAuthentication(client_handle, config);

            // Miscellaneous examples
            testTransferDataBigMessage(client_handle, config);
            testTransferDataMultipleFunctionalMessage(client_handle, config);
            testUsingEvent(client_handle, config);

            // Uninitialize channel
            status = UDSApi.Uninitialize_2013(client_handle);
            Console.WriteLine(" Unitialize channel: {0}", (int)status);

            // Display a small report and quit
            if (g_nbErr > 0)
            {
                Console.WriteLine(); Console.WriteLine("ERROR: {0} errors occurred.", g_nbErr);
            }
            else
            {
                Console.WriteLine(); Console.WriteLine("ALL Transmissions succeeded !");
            }
            Console.WriteLine(); Console.WriteLine(); Console.WriteLine("Press <Enter> to quit...");
            Console.In.Read();
        }
    }
}
