﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Runtime.InteropServices;

using Peak.Can.IsoTp;
using Peak.Can.Uds;
using System.Threading;

namespace server_response_ReadDataByPeriodicIdentifier
{
    class Program
    {
        const String PCAN_BITRATE = "f_clock=40000000,nom_brp=2,nom_tseg1=63,nom_tseg2=16,nom_sjw=16,data_brp=2,data_tseg1=7,data_tseg2=2,data_sjw=2";

        static String OK_KO(bool test)
        {
            if (test)
                return "OK";
            return "KO";
        }

        static String STATUS_OK_KO(uds_status test)
        {
            return OK_KO(UDSApi.StatusIsOk_2013(test));
        }

        /// <summary>
        ///  Entry point of the program, start a small server wich only support ReadDataByPeriodicIdentifier service.
        ///  This example use a specific addressing. It receives request from test equipement (0xF1 to 0xC1) in 29b fixed
        ///  normal addressing and sends responses for each periodic data identifier with 0x1F22C1F1 can identifier (UUDT).
        /// </summary>
        /// <returns>By convention, return success.</returns>
        static void Main(string[] args)
        {
            uds_status status;
            cantp_handle server_handle;
            UInt32 server_address;
            uds_msgconfig service_response_config = new uds_msgconfig();
            bool stop;
            bool wait_result;
            uds_status read_status;
            uds_msg request_msg = new uds_msg();
            uds_msg service_response_msg = new uds_msg();
            uds_msg periodic_response;
            uds_msgconfig periodic_msg_config = new uds_msgconfig();
            int keyboard_res;
            UInt32 padding_value;
            UInt32 can_tx_dl;
            Byte periodic_data_identifier;
            UInt32 timeout_value;
            UInt32 periodic_data_identifier_length;
            UInt32 i;

            // Initialize variables
            server_handle = cantp_handle.PCANTP_HANDLE_USBBUS2; // TODO: modify the value according to your available PCAN devices.
            server_address = 0xC1;

            // Initialize server
            status = UDSApi.InitializeFD_2013(server_handle, PCAN_BITRATE);
            Console.WriteLine("Initialize channel: {0}", STATUS_OK_KO(status));

            // Set server address parameter
            status = UDSApi.SetValue_2013(server_handle, uds_parameter.PUDS_PARAMETER_SERVER_ADDRESS, ref server_address, sizeof(UInt32));
            Console.WriteLine("Set server address: {0}", STATUS_OK_KO(status));

            // Set a padding value
            padding_value = 0xFF;
            status = UDSApi.SetValue_2013(server_handle, uds_parameter.PUDS_PARAMETER_CAN_PADDING_VALUE, ref padding_value, sizeof(UInt32));
            Console.WriteLine("Set padding value: {0}", STATUS_OK_KO(status));

            // Define CAN_TX_DL=15
            can_tx_dl = 15;
            status = UDSApi.SetValue_2013(server_handle, uds_parameter.PUDS_PARAMETER_CAN_TX_DL, ref can_tx_dl, sizeof(UInt32));
            Console.WriteLine("Set CAN TX DL: {0}", STATUS_OK_KO(status));

            // Set UDS timeouts
            timeout_value = 5000;
            status = UDSApi.SetValue_2013(server_handle, uds_parameter.PUDS_PARAMETER_TIMEOUT_REQUEST, ref timeout_value, sizeof(UInt32));
            Console.WriteLine("Set request timeout(ms): {0}", STATUS_OK_KO(status));
            status = UDSApi.SetValue_2013(server_handle, uds_parameter.PUDS_PARAMETER_TIMEOUT_RESPONSE, ref timeout_value, sizeof(UInt32));
            Console.WriteLine("Set response timeout(ms): {0}", STATUS_OK_KO(status));

            // Set a receive event
            System.Threading.AutoResetEvent receive_event = new System.Threading.AutoResetEvent(false);
            if (IntPtr.Size == 4)
            {
                UInt32 tmp_buffer = Convert.ToUInt32(receive_event.SafeWaitHandle.DangerousGetHandle().ToInt32());
                status = UDSApi.SetValue_2013(server_handle, uds_parameter.PUDS_PARAMETER_RECEIVE_EVENT, ref tmp_buffer, sizeof(UInt32));
            }
            else if (IntPtr.Size == 8)
            {
                Int64 tmp_buffer = receive_event.SafeWaitHandle.DangerousGetHandle().ToInt64();
                byte[] byte_array = BitConverter.GetBytes(tmp_buffer);
                status = UDSApi.SetValue_2013(server_handle, uds_parameter.PUDS_PARAMETER_RECEIVE_EVENT, byte_array, sizeof(UInt64));
            }
            Console.WriteLine("Set receive event parameter: {0}", STATUS_OK_KO(status));

            // Initialize service response configuration
            service_response_config.can_id = (UInt32)0xFFFFFFFF;
            service_response_config.can_msgtype = cantp_can_msgtype.PCANTP_CAN_MSGTYPE_EXTENDED | cantp_can_msgtype.PCANTP_CAN_MSGTYPE_FD | cantp_can_msgtype.PCANTP_CAN_MSGTYPE_BRS;
            service_response_config.nai.protocol = uds_msgprotocol.PUDS_MSGPROTOCOL_ISO_15765_2_29B_FIXED_NORMAL;
            service_response_config.nai.target_type = cantp_isotp_addressing.PCANTP_ISOTP_ADDRESSING_PHYSICAL;
            service_response_config.type = uds_msgtype.PUDS_MSGTYPE_USDT;
            service_response_config.nai.source_addr = (UInt16)server_address;
            service_response_config.nai.target_addr = (UInt16)uds_address.PUDS_ADDRESS_ISO_15765_4_ADDR_TEST_EQUIPMENT;
            service_response_config.nai.extension_addr = 0;

            // Initialize responses configuration (for each periodic data identifier contained in the request)
            periodic_msg_config.can_id = 0x1F22C1F1;
            periodic_msg_config.can_msgtype = cantp_can_msgtype.PCANTP_CAN_MSGTYPE_EXTENDED | cantp_can_msgtype.PCANTP_CAN_MSGTYPE_FD | cantp_can_msgtype.PCANTP_CAN_MSGTYPE_BRS;
            periodic_msg_config.nai.protocol = uds_msgprotocol.PUDS_MSGPROTOCOL_ISO_15765_2_29B_NORMAL;
            periodic_msg_config.nai.target_type = cantp_isotp_addressing.PCANTP_ISOTP_ADDRESSING_PHYSICAL;
            periodic_msg_config.type = uds_msgtype.PUDS_MSGTYPE_UUDT;
            periodic_msg_config.nai.source_addr = (UInt16)server_address;
            periodic_msg_config.nai.target_addr = (UInt16)uds_address.PUDS_ADDRESS_ISO_15765_4_ADDR_TEST_EQUIPMENT;
            periodic_msg_config.nai.extension_addr = 0;

            // Add a filter for 0x1F22C1F1 can id (in order to receive UUDT loopback messages)
            status = UDSApi.AddCanIdFilter_2013(server_handle, periodic_msg_config.can_id);
            Console.WriteLine("Add can identifier filter: {0}", STATUS_OK_KO(status));

            // Read while user do not press Q
            Console.WriteLine("Start listening, press Q to quit.");
            stop = false;
            do
            {

                // Wait a receive event on receiver
                //	note: timeout is used to check keyboard hit.
                wait_result = receive_event.WaitOne(1000);

                // If we get a receive event
                if (wait_result)
                {

                    do
                    {

                        // Read first available message (no filtering based on message's type is set):
                        read_status = UDSApi.Read_2013(server_handle, out request_msg);
                        Console.WriteLine("Try to read a message: {0}", STATUS_OK_KO(read_status));
                        if (UDSApi.StatusIsOk_2013(read_status))
                        {

                            // We receive a request, check its length and if it is not a loopback message
                            if (request_msg.msg.Msgdata_any_Copy.length >= 1 && (request_msg.msg.Msgdata_any_Copy.flags & cantp_msgflag.PCANTP_MSGFLAG_LOOPBACK) == 0)
                            {
                                Byte service_id;
                                UDSApi.GetDataServiceId_2013(ref request_msg, out service_id);

                                // This is a valid request, switch services
                                switch (service_id)
                                {
                                    case (Byte)uds_service.PUDS_SERVICE_SI_ReadDataByPeriodicIdentifier:

                                        // Allocates service response message
                                        status = UDSApi.MsgAlloc_2013(out service_response_msg, service_response_config, 1);
                                        if (UDSApi.StatusIsOk_2013(status))
                                        {
                                            UDSApi.SetDataServiceId_2013(ref service_response_msg, (Byte)uds_service.PUDS_SERVICE_SI_ReadDataByPeriodicIdentifier + UDSApi.PUDS_SI_POSITIVE_RESPONSE);
                                        }
                                        Console.WriteLine("Prepare response message for ReadDataByPeriodicIdentifier service: {0}", STATUS_OK_KO(status));

                                        // Write service response message
                                        status = UDSApi.Write_2013(server_handle, ref service_response_msg);
                                        Console.WriteLine("Write response message for ReadDataByPeriodicIdentifier service: {0}", STATUS_OK_KO(status));

                                        // Free response message (and clean memory in order to reallocate later)
                                        status = UDSApi.MsgFree_2013(ref service_response_msg);
                                        Console.WriteLine("Free response message: {0}", STATUS_OK_KO(status));

                                        // Sends a message for each data identifier with a specific addressing.
                                        periodic_data_identifier_length = request_msg.msg.Msgdata_any_Copy.length - 2;
                                        for (i = 0; i < periodic_data_identifier_length; i++)
                                        {

                                            // Allocates and prepares message with dummy data
                                            periodic_response = new uds_msg();
                                            status = UDSApi.MsgAlloc_2013(out periodic_response, periodic_msg_config, 5);
                                            if (UDSApi.StatusIsOk_2013(status))
                                            {
                                                UDSApi.GetDataParameter_2013(ref request_msg, (int)i + 1, out periodic_data_identifier);
                                                CanTpApi.setData_2016(ref periodic_response.msg, 0, periodic_data_identifier);
                                                CanTpApi.setData_2016(ref periodic_response.msg, 1, 0x12);
                                                CanTpApi.setData_2016(ref periodic_response.msg, 2, 0x34);
                                                CanTpApi.setData_2016(ref periodic_response.msg, 3, 0x56);
                                                CanTpApi.setData_2016(ref periodic_response.msg, 4, 0x78);

                                                Console.WriteLine("Allocates message for 0x{0:x2} periodic data identifier: {1}", periodic_data_identifier, STATUS_OK_KO(status));
                                                status = UDSApi.Write_2013(server_handle, ref periodic_response);
                                                Console.WriteLine("Write message for 0x{0:x2} periodic data identifier: {1}", periodic_data_identifier, STATUS_OK_KO(status));
                                                status = UDSApi.MsgFree_2013(ref periodic_response);
                                                Console.WriteLine("Free message for 0x{0:x2} periodic data identifier: {1}", periodic_data_identifier, STATUS_OK_KO(status));
                                            }
                                        }
                                        break;
                                    default:
                                        Console.WriteLine("Unknown service (0x{0:x2})", service_id);
                                        break;
                                }
                            }
                        }

                        // Free request message (in order to reallocate later)
                        status = UDSApi.MsgFree_2013(ref request_msg);
                        Console.WriteLine("Free request message: {0}", STATUS_OK_KO(status));
                    } while (!UDSApi.StatusIsOk_2013(read_status, uds_status.PUDS_STATUS_NO_MESSAGE, false));
                }

                // Quit when user press Q
                if (Console.KeyAvailable)
                {
                    keyboard_res = Console.ReadKey().KeyChar;
                    if (keyboard_res == 'Q' || keyboard_res == 'q')
                    {
                        stop = true;
                    }
                }
            } while (stop == false);

            // Close receive event
            if (IntPtr.Size == 4)
            {
                UInt32 tmp_buffer = 0;
                status = UDSApi.SetValue_2013(server_handle, uds_parameter.PUDS_PARAMETER_RECEIVE_EVENT, ref tmp_buffer, sizeof(UInt32));
            }
            else if (IntPtr.Size == 8)
            {
                Int64 tmp_buffer = 0;
                byte[] byte_array = BitConverter.GetBytes(tmp_buffer);
                status = UDSApi.SetValue_2013(server_handle, uds_parameter.PUDS_PARAMETER_RECEIVE_EVENT, byte_array, sizeof(UInt64));
            }
            Console.WriteLine("Stop receive event: {0}", STATUS_OK_KO(status));
            receive_event.Close();
            Console.WriteLine("Close receive event: {0}", STATUS_OK_KO(status));

            // Close server
            status = UDSApi.Uninitialize_2013(server_handle);
            Console.WriteLine("Uninitialize channel: {0}", STATUS_OK_KO(status));

            // Exit
            Console.WriteLine("Press any key to continue...");
            Console.In.Read();
        }
    }
}
