﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Runtime.InteropServices;

using Peak.Can.IsoTp;
using Peak.Can.Uds;
using System.Threading;

namespace client_request_ReadDataByPeriodicIdentifier
{
    class Program
    {
        const String PCAN_BITRATE = "f_clock=40000000,nom_brp=2,nom_tseg1=63,nom_tseg2=16,nom_sjw=16,data_brp=2,data_tseg1=7,data_tseg2=2,data_sjw=2";
        const long RESPONSES_TIMEOUT_MS = 5000;
        static String OK_KO(bool test)
        {
            if (test)
                return "OK";
            return "KO";
        }
        static String STATUS_OK_KO(uds_status test)
        {
            return OK_KO(UDSApi.StatusIsOk_2013(test));
        }


        /// <summary>
        ///  Entry point of the program, start a UDS channel, ask ReadDataByPeriodicIdentifier service.
        ///  It sends a request from 0xF1 to 0xC1 address in 29b fixed with normal addressing. Then read
        ///  responses from 0x1F22C1F1 can identifier (UUDT).
        /// </summary>
        /// <returns>By convention, return success.</returns>
        static void Main(string[] args)
        {
            uds_status status;
            cantp_handle client_handle;
            uds_msgconfig request_config;
            uds_msg msg_request;
            uds_msg request_confirmation;
            uds_msg service_response;
            UInt32 can_tx_dl;
            UInt32 timeout_value;
            uds_msg periodic_response;
            Byte response_count;
            long start_reading;
            int i;

            // Initialize variables
            client_handle = cantp_handle.PCANTP_HANDLE_USBBUS1; // TODO: modify the value according to your available PCAN devices.
            request_config = new uds_msgconfig();
            msg_request = new uds_msg();
            request_confirmation = new uds_msg();
            service_response = new uds_msg();
            periodic_response = new uds_msg();

            // Initialize client
            status = UDSApi.InitializeFD_2013(client_handle, PCAN_BITRATE);
            Console.WriteLine("Initialize channel: {0}", STATUS_OK_KO(status));

            // Define CAN_TX_DL
            can_tx_dl = 15;
            status = UDSApi.SetValue_2013(client_handle, uds_parameter.PUDS_PARAMETER_CAN_TX_DL, ref can_tx_dl, sizeof(UInt32));
            Console.WriteLine("Set CAN TX DL: {0}", STATUS_OK_KO(status));

            // Check CAN_TX_DL
            can_tx_dl = 0;
            status = UDSApi.GetValue_2013(client_handle, uds_parameter.PUDS_PARAMETER_CAN_TX_DL, out can_tx_dl, sizeof(UInt32));
            Console.WriteLine("Check new CAN TX DL value({0}): {1}", can_tx_dl, STATUS_OK_KO(status));

            // Set UDS timeouts
            timeout_value = 5000;
            status = UDSApi.SetValue_2013(client_handle, uds_parameter.PUDS_PARAMETER_TIMEOUT_REQUEST, ref timeout_value, sizeof(UInt32));
            Console.WriteLine("Set request timeout(ms): {0}", STATUS_OK_KO(status));
            status = UDSApi.SetValue_2013(client_handle, uds_parameter.PUDS_PARAMETER_TIMEOUT_RESPONSE, ref timeout_value, sizeof(UInt32));
            Console.WriteLine("Set response timeout(ms): {0}", STATUS_OK_KO(status));

            // Check timeouts
            timeout_value = 0;
            status = UDSApi.GetValue_2013(client_handle, uds_parameter.PUDS_PARAMETER_TIMEOUT_REQUEST, out timeout_value, sizeof(UInt32));
            Console.WriteLine("Get new request timeout({0}ms): {1}", timeout_value, STATUS_OK_KO(status));
            timeout_value = 0;
            status = UDSApi.GetValue_2013(client_handle, uds_parameter.PUDS_PARAMETER_TIMEOUT_RESPONSE, out timeout_value, sizeof(UInt32));
            Console.WriteLine("Get new response timeout({0}ms): {1}", timeout_value, STATUS_OK_KO(status));

            // Initialize the request configuration
            request_config.can_id = 0xFFFFFFFF;
            request_config.can_msgtype = cantp_can_msgtype.PCANTP_CAN_MSGTYPE_EXTENDED | cantp_can_msgtype.PCANTP_CAN_MSGTYPE_FD | cantp_can_msgtype.PCANTP_CAN_MSGTYPE_BRS;
            request_config.nai.protocol = uds_msgprotocol.PUDS_MSGPROTOCOL_ISO_15765_2_29B_FIXED_NORMAL;
            request_config.nai.target_type = cantp_isotp_addressing.PCANTP_ISOTP_ADDRESSING_PHYSICAL;
            request_config.type = uds_msgtype.PUDS_MSGTYPE_USDT;
            request_config.nai.source_addr = (UInt16)uds_address.PUDS_ADDRESS_ISO_15765_4_ADDR_TEST_EQUIPMENT;
            request_config.nai.target_addr = 0xC1;
            request_config.nai.extension_addr = 0;

            // Add a filter for 0x1F22C1F1 can id (in order to receive UUDT messages)
            status = UDSApi.AddCanIdFilter_2013(client_handle, 0x1F22C1F1);
            Console.WriteLine("Add can identifier filter: {0}", STATUS_OK_KO(status));

            // Execute ReadDataByPeriodicIdentifier and wait service response
            Byte[] periodic_data_identifier = { 0x12, 0x34, 0x56, 0x78 };
            status = UDSApi.SvcReadDataByPeriodicIdentifier_2013(client_handle, request_config, out msg_request, UDSApi.uds_svc_param_rdbpi.PUDS_SVC_PARAM_RDBPI_SAMR, periodic_data_identifier, 4);
            Console.WriteLine("Execute ReadDataByPeriodicIdentifier service: {0}", STATUS_OK_KO(status));
            status = UDSApi.WaitForService_2013(client_handle, ref msg_request, out service_response, out request_confirmation);
            Console.WriteLine("Wait for service (confirmation and response): {0}", STATUS_OK_KO(status));

            // Read responses for each given periodic data identifier
            response_count = 0;
            start_reading = DateTime.UtcNow.Ticks / TimeSpan.TicksPerMillisecond;
            do
            {
                status = UDSApi.Read_2013(client_handle, out periodic_response);
                if (UDSApi.StatusIsOk_2013(status, uds_status.PUDS_STATUS_OK, false) && periodic_response.msg.Msgdata_any_Copy.length > 0)
                {
                    response_count++;
                    Byte msg_data_buffer = 0;
                    CanTpApi.getData_2016(ref periodic_response.msg, 0, out msg_data_buffer);
                    Console.Write("Read response for 0x{0:x2} periodic data identifier: [", msg_data_buffer);
                    for (i = 0; i < periodic_response.msg.Msgdata_any_Copy.length; i++)
                    {
                        CanTpApi.getData_2016(ref periodic_response.msg, i, out msg_data_buffer);
                        Console.Write(" 0x{0:x2}", msg_data_buffer);
                    }
                    Console.WriteLine(" ]");
                    status = UDSApi.MsgFree_2013(ref periodic_response);
                    Console.WriteLine("Free response buffer: {0}", STATUS_OK_KO(status));
                }

                // Wait for 4 responses or timeout
            } while (response_count < 4 && (DateTime.UtcNow.Ticks / TimeSpan.TicksPerMillisecond) - start_reading < RESPONSES_TIMEOUT_MS);

            // Free message structures
            status = UDSApi.MsgFree_2013(ref msg_request);
            Console.WriteLine("Free request message: {0}", STATUS_OK_KO(status));
            status = UDSApi.MsgFree_2013(ref request_confirmation);
            Console.WriteLine("Free request confirmation: {0}", STATUS_OK_KO(status));
            status = UDSApi.MsgFree_2013(ref service_response);
            Console.WriteLine("Free service response: {0}", STATUS_OK_KO(status));

            // Close client
            status = UDSApi.Uninitialize_2013(client_handle);
            Console.WriteLine("Uninitialize channel: {0}", STATUS_OK_KO(status));

            // Exit
            Console.WriteLine("Press any key to continue...");
            Console.In.Read();
        }
    }
}
